#!/usr/bin/env python

import sys, os, stat, subprocess

from arklib.ark_cred import ark_cred
from arklib.ark_s3 import ark_s3

apply_script = """#!/bin/bash

# Puppet module path
TMP_PATH="/tmp/puppet"
MODULE_PATH="${TMP_PATH}/modules:${TMP_PATH}/shared"

function update_sources_list() {
  if [ $PUPPET_EXISTS -eq 1 ]; then
    # Add puppetlabs repository
    wget http://apt.puppetlabs.com/puppetlabs-release-precise.deb -O /tmp/puppetlabs-release-precise.deb
    sudo dpkg -i /tmp/puppetlabs-release-precise.deb
    rm /tmp/puppetlabs-release-precise.deb
    sudo apt-get update
  fi
}

function setup_puppet() {
  if [ $PUPPET_EXISTS -eq 1 ]; then
    sudo apt-get install -y puppet facter hiera
  else
    echo "puppet is installed"
  fi
}

function install_if_not_exists() {
  pkg=$1
  dpkg -s $pkg > /dev/null 2>&1
  if [ $? -eq 1 ] ; then
    sudo apt-get -y install $pkg
  else
    echo "$pkg exists"
  fi
}

function install_arklib_if_not_exists() {
  str=`sudo pip freeze 2>/dev/null | grep ^arklib`
  if [ -z "$str" ]; then
    sudo pip install -U arklib
  else
    echo "arklib exists"
  fi
}

function install_hiera() {
  echo "Installing hiera data ..."

  # Create directories if it does not exists
  sudo mkdir -p /etc/puppet
  sudo mkdir -p /var/lib/hiera

  # Delete existing data if exists
  sudo rm -f /etc/hiera.yaml
  sudo rm -f /etc/puppet/hiera.yaml
  sudo rm -fr /var/lib/hiera/*

  # Copy hiera files
  sudo cp /tmp/puppet/modules/hiera/files/hiera.yaml /etc/puppet
  sudo cp -r /tmp/puppet/modules/hiera/files/data/* /var/lib/hiera/
}

function install_tag_facts() {
  echo "Installing tag facts ..."

  # Create new facts.d directory if not exists
  sudo mkdir -p /etc/facter/facts.d

  # Remove the old one if exists
  rm -f etc/facter/facts.d/ark-ec2-tags.sh

  cat << "EOF" | sudo tee /etc/facter/facts.d/ark-ec2-tags.sh > /dev/null 2>&1
#!/bin/bash
sudo ark-ec2-tags

EOF

  sudo chmod +x /etc/facter/facts.d/ark-ec2-tags.sh
}

function install_deps() {
  echo "Checking and installing dependencies..."
  install_if_not_exists libyaml-dev
  install_if_not_exists git-core
  install_if_not_exists python-dev
  install_if_not_exists build-essential
  install_if_not_exists python-pip
  install_arklib_if_not_exists
  install_hiera
  install_tag_facts
}

function apply_manifest() {
  local system=$1

  rm -fr /tmp/puppet /tmp/puppet.log
  tar xzf /tmp/puppet.tar.gz -C /tmp > /dev/null 2>&1

  install_deps

  if [ "$system" = "system" ]; then
    echo "Applying system.pp ..."
    sudo puppet apply --debug --color=false --modulepath=${MODULE_PATH} /tmp/puppet/manifests/system.pp
  else
    role=`sudo ark-ec2-tags | grep ec2_tag_role | cut -f2 -d=`

    IFS=","
    for r in ${role}; do
      echo "Applying ${r}.pp ..."
      sudo puppet apply --debug --color=false --modulepath=${MODULE_PATH} /tmp/puppet/manifests/${r}.pp
    done
  fi

  # Clean up
  rm -fr /tmp/puppet.tar.gz /tmp/puppet
}

# # # # # # # # # #
dpkg -s puppet > /dev/null
PUPPET_EXISTS=$?

update_sources_list
setup_puppet
apply_manifest $1

"""

script_path = '/tmp/ark_apply_puppet.sh'
local_puppet_path = '/tmp/puppet.tar.gz'

def download_manifest_from_s3(url, _cred):
  s3 = ark_s3(_cred.aws_access_key, _cred.aws_secret_key)
  bucket, key = s3.parse_bucket_key(s3.trim_s3_scheme(url))
  s3.download_file(bucket, key, local_puppet_path)

def write_script():
  f = open(script_path, 'w')
  f.write(apply_script)
  f.close()
  st = os.stat(script_path)
  os.chmod(script_path, st.st_mode | stat.S_IEXEC)

def exec_script(system):
  subprocess.call(script_path + " " + system, shell=True)

def delete_script():
  os.remove(script_path)

def run_main(argv):
  if len(argv) != 2 and len(argv) != 3:
    print "Syntax: ark-puppet-apply [s3 url to the <puppet manifest>.tar.gz] [system]"
    sys.exit(1)

  try:
    _cred = ark_cred()
    download_manifest_from_s3(argv[1], _cred)
    write_script()

    if len(argv) == 3 and argv[2] == "system":
      exec_script("system")
    else:
      exec_script("")

    delete_script()
  except Exception, e:
    print e
    sys.exit(1)

def main():
  run_main(sys.argv)

# # # # # # # # # #

if __name__ == "__main__":
  main()

