try:
  import unittest2 as unittest
except ImportError:
  import unittest

from mock import MagicMock as Mock
from mock import patch
from arklib.ark_s3 import *

class ark_s3_test(unittest.TestCase):
  def setUp(self):
    self.s3 = ark_s3('access', 'secret')

  def test_aws_cred(self):
    self.failUnless(self.s3.access_key == 'access')
    self.failUnless(self.s3.secret_key == 'secret')

  def test_s3_instance(self):
    self.assertIsNotNone(self.s3.s3)

  @patch('arklib.ark_s3.Key')
  def test_get_content(self, key):
    # Setup
    self.s3.s3 = Mock()
    bucket = Mock()
    key_mock = Mock()
    key.return_value = key_mock
    self.s3.s3.get_bucket.return_value = bucket
    key_mock.get_contents_as_string.return_value = "return value"

    # Call
    result = self.s3.get_content("bucket", "key")

    # Asserts
    self.s3.s3.get_bucket.assert_called_with("bucket")
    key.assert_called_with(bucket)
    self.assertEquals(key_mock.key, "key")
    key_mock.get_contents_as_string.assert_called_with()
    self.assertEqual(result, "return value")

  def test_download_file(self):
    # Setup
    self.s3.s3 = Mock()
    bucket = Mock()
    self.s3.s3.get_bucket.return_value = bucket
    key = Mock()
    bucket.get_key.return_value = key

    # Call
    self.s3.download_file("bucket", "key", "file")

    # Asserts
    self.s3.s3.get_bucket.assert_called_with("bucket")
    bucket.get_key.assert_called_with("key")
    key.get_contents_to_filename.assert_called_with("file")

  
  def test_download_file_without_key(self):
    # Setup
    self.s3.s3 = Mock()
    bucket = Mock()
    self.s3.s3.get_bucket.return_value = bucket
    bucket.get_key.return_value = None

    # Asserts
    self.assertRaises(Exception, self.s3.download_file, "bucket", "key", "file")

  @patch('arklib.ark_s3.Key')
  def test_upload_file(self, m_key):
    # Setup
    self.s3.s3 = Mock()
    bucket = Mock()
    self.s3.s3.get_bucket.return_value = bucket
    key = Mock()
    m_key.return_value = key

    # Call
    self.s3.upload_file("bucket", "key", "file")

    # Asserts
    self.s3.s3.get_bucket.assert_called_with("bucket")
    m_key.assert_called_with(bucket)
    self.assertEquals(key.key, "key")
    key.set_contents_from_filename.assert_called_with("file")

  def test_parse_bucket_key(self):
    path = 'bucket/key/a.txt'
    bucket, key = self.s3.parse_bucket_key(path)
    self.assertEquals(bucket, 'bucket')
    self.assertEquals(key, 'key/a.txt')

  def test_is_s3_url(self):
    self.assertTrue(self.s3.is_s3_url('s3://bucket/key/a'))
    self.assertTrue(self.s3.is_s3_url('S3://bucket/key/a'))

  def test_trim_s3_scheme(self):
    self.assertEquals(self.s3.trim_s3_scheme('s3://bucket/a'), 'bucket/a')
    self.assertEquals(self.s3.trim_s3_scheme('S3://buCket/a'), 'buCket/a')
    self.assertNotEquals(self.s3.trim_s3_scheme('S4://buCket/a'), 'buCket/a')

