try:
  import unittest2 as unittest
except ImportError:
  import unittest

from mock import MagicMock as Mock
from mock import patch
from arklib.ark_route53 import *

class ark_route53_test(unittest.TestCase):
  def setUp(self):
    self.route53 = ark_route53('access', 'secret')

  def test_route53_instance(self):
    self.assertIsNotNone(self.route53.route53)

  def test_find_hosted_domain(self):
    m_route53 = Mock()
    self.route53.route53 = m_route53
    rv = {u'ListHostedZonesResponse': 
        {u'HostedZones': 
          [
            {u'ResourceRecordSetCount': u'219', 
              u'CallerReference': u'0A3E8F92-E1B5-165E-A328-3F6C5C4854C9', 
              u'Config': {}, 
              u'Id': u'/hostedzone/ABCDEFG', 
              u'Name': u'foo.domain.com.'}], 
            u'IsTruncated': u'false', 
            u'MaxItems': u'100'}}
    m_route53.get_all_hosted_zones.return_value =  rv;

    # Call
    zi = self.route53.find_hosted_zone_id('foo.domain.com')

    # Asserts
    m_route53.get_all_hosted_zones.assert_called_with()
    self.assertEquals(zi, 'ABCDEFG')

  def test_find_hosted_domain_not_found(self):
    m_route53 = Mock()
    self.route53.route53 = m_route53
    rv = {u'ListHostedZonesResponse': 
        {u'HostedZones': 
          [
            {u'ResourceRecordSetCount': u'219', 
              u'CallerReference': u'0A3E8F92-E1B5-165E-A328-3F6C5C4854C9', 
              u'Config': {}, 
              u'Id': u'/hostedzone/ABCDEFG', 
              u'Name': u'foo.domain.com.'}], 
            u'IsTruncated': u'false', 
            u'MaxItems': u'100'}}
    m_route53.get_all_hosted_zones.return_value =  rv;

    # Call
    zi = self.route53.find_hosted_zone_id('bar.domain.com')

    # Asserts
    m_route53.get_all_hosted_zones.assert_called_with()
    self.assertIsNone(zi)

  def test_find_hosted_domain_no_found(self):
    m_route53 = Mock()
    self.route53.route53 = m_route53
    rv = {u'ListHostedZonesResponse': 
          {u'HostedZones': [],
           u'IsTruncated': u'false', 
           u'MaxItems': u'100'}}
    m_route53.get_all_hosted_zones.return_value =  rv;

    # Call
    zi = self.route53.find_hosted_zone_id('bar.domain.com')

    # Asserts
    m_route53.get_all_hosted_zones.assert_called_with()
    self.assertIsNone(zi)


  def test_domain_exists_false(self):
    self.route53.route53 = Mock()
    self.route53.route53.get_all_rrsets.return_value = ""
    result = self.route53.domain_exists("domain", "rtype", "zoneId")
    self.route53.route53.get_all_rrsets.assert_called_with("zoneId");
    self.assertFalse(result)

  def test_domain_exists_true(self):
    self.route53.route53 = Mock()

    # Setup record mock
    record = Mock()
    record_xml = """
    <ResourceRecordSet>
      <Name>test.ec2.viglink.com.</Name>
      <Type>CNAME</Type>
      <TTL>15</TTL>
      <ResourceRecords>
        <ResourceRecord>
          <Value>ec2-54-224-143-216.compute-1.amazonaws.com</Value>
        </ResourceRecord>
      </ResourceRecords>
    </ResourceRecordSet>
    """
    record.to_xml.return_value = record_xml
    self.route53.route53.get_all_rrsets.return_value = [record]

    # Action 
    result = self.route53.domain_exists("test.ec2.viglink.com", "CNAME", "zoneId")

    # Asserts
    self.route53.route53.get_all_rrsets.assert_called_with("zoneId");
    self.assertTrue(result)
  
  @patch('boto.route53.record.ResourceRecordSets')
  def test_create_cname_record(self, record_set):
    # Setup
    self.route53.route53 = Mock()
    changes = Mock()
    record_set.return_value = changes
    change = Mock()
    changes.add_change.return_value = change

    # Action
    self.route53.create_cname_record("name", "value", "zoneId")

    # Asserts
    record_set.assert_called_with(self.route53.route53, "zoneId") 
    changes.add_change.assert_called_with("CREATE", "name", "CNAME")
    change.add_value.assert_called_with("value")
    changes.commit.assert_called_with()
 
  @patch('boto.route53.record.ResourceRecordSets')
  def test_delete_cname_record(self, record_set):
    # Setup
    self.route53.route53 = Mock()
    changes = Mock()
    record_set.return_value = changes
    change = Mock()
    changes.add_change.return_value = change

    # Action
    self.route53.delete_cname_record("name", "value", "zoneId")

    # Asserts
    record_set.assert_called_with(self.route53.route53, "zoneId") 
    changes.add_change.assert_called_with("DELETE", "name", "CNAME")
    change.add_value.assert_called_with("value")
    changes.commit.assert_called_with()

