#!/usr/bin/env python
"""
Routine that monitors a log file, and, if it hasn't been modified in a certain amount of time,
sends a signal to the process that called the routine.
 ... OR, if timeout is negative, kill parent when filename is created.

Gavin ran into problems using SIGALRM, so came up with this solution instead.
"""

import atexit
import os
import signal
import sys
import time
from math import fabs

def monitor_and_signal(filename, timeout, signal):
    '''timeout positive: if file doesn't change in timeout seconds, send this process {signal}.
       timeout negative: send process {signal} when filename exists.
       Returns monitoring process pid.'''

    if timeout < 0 and os.access(filename, os.F_OK):
        os.remove(filename)

    parent_pid = os.getpid()

    # Strategy is:  fork a child process, which mostly sleeps.
    pid = os.fork()
    if pid == 0:
        # We're the child process; monitor the file:
        while 1:
            time.sleep(fabs(timeout))
            try:
                info = os.stat(filename)
                if timeout < 0:  # Monitoring a 'quit' file
                    os.kill(parent_pid, signal)
                    sys.exit(0)
            except OSError:
                if timeout > 0:
                    sys.exit(0)  # Nothing to monitor any more, exit.
            if (timeout > 0) and ((time.time() - info.st_mtime) > timeout):
                try:
                    os.kill(parent_pid, signal)
                except OSError:
                    sys.exit(0)  # Parent process goes away: exit.
    return pid
 
if __name__ == '__main__':
    fname = "/tmp/pretendLog.log"
    fp = open(fname, "w")
    fp.write("Testing\n")

    monitor_and_signal(fname, 3, signal.SIGINT)

    for i in range(1,8):
        print("%d second delay (interrupt at >3)"%i)
        try:
            time.sleep(i)
        except KeyboardInterrupt:
            print("Interrupt, was sleeping %d seconds"%i)
        fp.write("%d\n"%i)
        fp.flush()
    fp.close()

