#!/usr/bin/env python

"""
``pyscripting`` -- make shell scripting with python easier.

Getting started
===============

Create your python shell script ``myscript.py``::

  #!/usr/bin/env python

  from scripting import sh

  sh.ls('-l')

Use your script::

  chmod +x myscript.py
  ./myscript.py


Calling external commands
=========================

There are three ways to call external commands.

If possible, python replacement of an external command will be used.
Replacements will not be used only in direct calling.

For all replaced commands see `Replaced commands`_.


Direct calling
--------------

Returns exit code.

::

    sh('ls', '-l')

Indirect calling
----------------

Returns exit code.

::

    sh.ls('-l')

Call and return output
----------------------

Returns stripped stdout (stderr will not be included).

Using this method, output of command will not be printed to stdout. Before
returning output, leading white spaces will be stripped.

Don't use this methos for large outputs.

::

    output = sh.get('ls', '-l')
    print('Output was: %s' % output)

Argument handling
=================

You can access arguments passed to script using ``argv`` property::

    sh.argv[0] - called script name
    sh.argv[1] - first argument

.. _Replaced commands:

Replaced commands
=================

To avoid overhead and for simplicity reasons, some external commands was
replaced by python's internal functions, which works much faster, that calling
external command.


``basename``
    Same as external ``basename``.

    Returns string.

``exit``
    Same as external ``exit``.

``find``
    Similar to external ``find`` command.

    Returns iterator of all found files.

    Example usage::

        for f in sh.find(type='f', exclude=['*.pyc']):
            print(f)

``mkdir``
    Same as external ``mkdir``.

``mkdirs``
    Same as external ``mkdir -p``.

``test``
    Similar to external ``test``.

    Returns boolean.

    Example usage::

        if sh.test('-d', '/tmp'):
            print('/tmp is directory.')


Makefile functionality
======================

Example (``myscript.py``)::

    #!/usr/bin/env python

    from scripting import sh, Makefile

    make = Makefile(sh)

    @make('/tmp/myfile.txt')
    def myrule(target):
        sh.touch(target)

    @make()
    def main(target):
        myrule()

    make.run(main)

Last line ``make.run(main)`` checks ``sys.argv`` and executes specified rule or
default if no particular rule is specified. ``myrule`` will be executed only,
if target file ``/tmp/myfile.txt`` does not exists.::

    ./myscript.py

Now call particular rule::

    ./myscript.py myrule

"""

import os
import sys
import stat
import fnmatch
import subprocess

__all__ = ['sh', 'Makefile']

commands = {
    'mkdir': os.mkdir,
    'basename': os.path.basename,
    'exit': sys.exit,
}


class Target(object):
    def __init__(self, target=None, sources=None, sh=None):
        self.target = target
        self.sources = sources
        self.sh = sh

    def __str__(self):
        return self.target

    def __call__(self, rule):
        self.rule = rule
        return self.make

    def make(self):
        if self.is_missing() or self.is_outdated():
            self.rule(self)

    def is_missing(self):
        return not self.target or not self.sh.test('-e', self.target)

    def is_outdated(self):
        if not self.sources is None:
            for src in self.sources:
                if not self.sh.sh('-c', 'test %s -ot %s' % (self.target, src)):
                    return True
        return False;


class Makefile(object):
    targets = []

    def __init__(self, sh):
        self.sh = sh

    def __call__(self, *args, **kwargs):
        kwargs.setdefault('sh', self.sh)
        target = Target(*args, **kwargs)
        self.targets.append(target)
        return target

    def run(self, rule):
        if len(sys.argv) > 1:
            for target in self.targets:
                if target.rule.__name__ == sys.argv[1]:
                    return target.make()
        else:
            rule()


class Shell(object):
    """
    >>> sh('echo', 'test')
    0
    >>> sh.echo('test')
    0
    >>> sh.get('echo', 'test')
    'test'
    >>> sh.test('-d', '/tmp')
    True
    >>> list(sh.find(type='f', exclude=['*.pyc', './.hg*']))
    ['./README.rst', './scripting.py']

    """
    path = os.environ['PATH'].split(':')
    env = {}
    last = None
    argv = sys.argv

    def __call__(self, cmd, *args, **kwargs):
        cmdargs = self._get_cmdargs(cmd, *args, **kwargs)
        cwd = os.path.abspath(os.curdir)
        self.last = subprocess.Popen(cmdargs, cwd=cwd, env=self.env)
        return self.last.wait()

    def __getattr__(self, cmd):
        def call(*args, **kwargs):
            if cmd in commands:
                return commands[cmd](*args, **kwargs)
            else:
                return self(cmd, *args, **kwargs)
        return call

    def _get_cmd(self, cmd):
        for path in self.path:
            _cmd = os.path.join(path, cmd)
            if os.path.exists(_cmd):
                return _cmd
        return False

    def _get_cmdargs(self, cmd, *args, **kwargs):
        _cmd = self._get_cmd(cmd)

        if not _cmd:
            raise Exception('command not found: %s' % cmd)

        cmdargs = [_cmd]
        if args:
            cmdargs.extend(args)
        if kwargs:
            for i in kwargs.iteritems():
                cmdargs.extend(i)
        return [str(i) for i in cmdargs]

    def get(self, cmd, *args, **kwargs):
        cmdargs = self._get_cmdargs(cmd, *args, **kwargs)
        cwd = os.path.abspath(os.curdir)
        self.last = subprocess.Popen(cmdargs, cwd=cwd, env=self.env,
                                     stdout=subprocess.PIPE)
        self.last.wait()
        return self.last.stdout.read().strip()

    def mkdirs(self, *paths, **kwargs):
        mode = kwargs.get('mode', 0777)
        for path in paths:
            try:
                os.makedirs(path, mode)
            except OSError:
                pass

    def matches(self, f, patterns):
        for pat in patterns:
            if fnmatch.fnmatch(f, pat):
                return True
        return False

    def find(self, path='.', type=None, exclude=()):
        for root, dirs, files in os.walk(path):
            if type is None or type == 'd':
                for d in dirs:
                    if self.matches(os.path.join(root, d), exclude):
                        continue
                    yield os.path.join(root, d)
            if type is None or type == 'f':
                for f in files:
                    if self.matches(os.path.join(root, f), exclude):
                        continue
                    yield os.path.join(root, f)

    def test(self, mode, f):
        if not os.path.exists(f):
            return False

        if mode == '-e':
            return True

        modes = {
            '-b': stat.S_ISBLK,
            '-c': stat.S_ISCHR,
            '-d': stat.S_ISDIR,
            '-f': stat.S_ISREG,
            '-h': stat.S_ISLNK,
            '-L': stat.S_ISLNK,
            '-p': stat.S_ISFIFO,
            '-S': stat.S_ISSOCK,
        }
        if mode in modes:
            return modes[mode](os.stat(f)[stat.ST_MODE])

        raise Exception('test: Unknown condition: %s' % mode)

sh = Shell()
