#
# $Source: /home/blais/repos/cvsroot/arubomu/old/lib/python/album-old.py,v $
# $Id: album-old.py,v 1.36 2004/02/14 18:22:31 blais Exp $
#

"""Data structures for list of CDs."""

__version__ = "$Revision: 1.36 $"
__author__ = "Martin Blais <blais@furius.ca>"


import sys, os
import string, re
import types

from utils import unique
import strxtra
from curses.ascii import isalnum

from xml.sax.saxutils import escape as esc
import xml.dom.minidom
from xml.dom.minidom import Document, Element, Text
import xml_helpers
from xml_helpers import condAddAttribute, condAddTextElement
from xml_helpers import DelegSAXHandler, BasicHandler


encoding = xml_helpers.default_enc

doctype_str = '<!DOCTYPE %%s SYSTEM "%s">'
dtd_pfx = 'http://furius.ca/jukebox/share/dtd/'

songlist_dtd = dtd_pfx + 'songlist.dtd'
album_dtd = dtd_pfx + 'album.dtd'

songlist_doctype = doctype_str % songlist_dtd
album_doctype = doctype_str % album_dtd


def outputMany(os, obj, tagname):
    values = filter(None, map(lambda x: getattr(x, tagname), obj))
    uvalues = unique(values)
    fmt = '   <%s>%%s</%s>' % (tagname, tagname)
    for v in uvalues:
        print >> os, fmt % esc(v)


def addMany(xmlobj, listobj, tagname):
    values = filter(None, map(lambda x: getattr(x, tagname), listobj))
    uvalues = unique(values)
    for v in uvalues:
        condAddTextElement(xmlobj, tagname, v)



class ParseError(StandardError):
    def __init__(self, str):
        StandardError.__init__(self, str)


def loadFile(fn, id):
    """Reads an XML file and parses as an album file."""

    album = None

    try:
        dom = xml.dom.minidom.parse(fn)
        albumxml = dom.documentElement

        album = Album(id).fromXML(albumxml)
    except IOError, e:
        print >> sys.stderr, 'Error: cannot open file "%s"' % fn
        print >> sys.stderr, e
        raise
    except xml.sax.SAXParseException, e:
        print >> sys.stderr, 'Error: malformed XML in "%s"' % fn
        raise
    except xml.parsers.expat.ExpatError, e:
        print >> sys.stderr, 'Error: XML parsing error in "%s"' % fn
        raise
    except ParseError, e:
        print >> sys.stderr, 'Error: XML file "%s" has invalid format' % fn
        raise

    return album



class Recording:
    def __init__(self):
        self.begin = self.end = None
        self.toqueref = self.cantoref = None

    def fromXML(self, xel):
        if xel.localName != 'rec':
            raise ParseError('root is not a recording "%s"' % xel.localName)
        xml_helpers.getAttributes(\
            self, xel, ['begin', 'end'], ['toqueref', 'cantoref'])
        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlrec = doc.createElement('rec')
        condAddAttribute(xmlrec, 'begin', self.begin)
        condAddAttribute(xmlrec, 'end', self.end)
        condAddAttribute(xmlrec, 'toqueref', self.toqueref)
        condAddAttribute(xmlrec, 'cantoref', self.cantoref)
        return xmlrec



class Song:
    def __init__(self, no=-1, title=''):
        self.no = no
        self.title = title
        self.title_alt = ''
        self.artist = ''
        self.duration = ''
        self.extra = ''
        self.recordings = []
        self.lyrics = None

        self.valid = 1
        
    def __cmp__(self, other):
        return cmp(self.no, other.no)

    def not__repr__(self):
        s = "%d. '%s'" % (self.no, self.title)
        if self.artist:
            s += " -- %s" % self.artist
        if self.extra:
            s += " (%s)" % self.extra
        if self.duration:
            s += ' [ %s ]' % self.duration
        return s

    def fromXML(self, xel):
        if xel.localName != 'song':
            raise ParseError('root is not a song: "%s"' % xel.localName)

        xml_helpers.getAttributes(self, xel, ['no'], [])
        if self.no: self.no = int(self.no)
        xml_helpers.getElementsText(\
            self, xel, [], ['title', 'title_alt', 'artist', 'extra',
                            'duration', 'lyrics'])

        xmlrecs = xml_helpers.getChildrenByTagName(xel, 'rec')
        self.recordings = []
        for xmlrec in xmlrecs:
            self.recordings.append( Recording().fromXML(xmlrec) )

        for att in ['title', 'artist', 'duration', 'extra']:
            if isinstance(getattr(self, att), types.ListType):
                setattr(self, att, '')
                self.valid = 0 # mark as invalid

        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlsong = doc.createElement('song')
        condAddAttribute(xmlsong, 'no', repr(self.no))
        condAddTextElement(xmlsong, 'title', self.title)
        condAddTextElement(xmlsong, 'title_alt', self.title_alt)
        condAddTextElement(xmlsong, 'artist', self.artist)
        condAddTextElement(xmlsong, 'duration', self.duration)
        condAddTextElement(xmlsong, 'extra', self.extra)
        condAddTextElement(xmlsong, 'lyrics', self.lyrics)

        for rec in self.recordings:
            xmlsong.appendChild( rec.toXML(doc) )
        return xmlsong

    def mergeXML(songs, no):
        xmlsong = Element('song')
        condAddAttribute(xmlsong, 'no', repr(no))

        addMany(xmlsong, songs, 'title')
        addMany(xmlsong, songs, 'title_alt')
        addMany(xmlsong, songs, 'artist')
        addMany(xmlsong, songs, 'duration')
        addMany(xmlsong, songs, 'extra')
        addMany(xmlsong, songs, 'lyrics')

        allrecs = reduce(lambda x,y:x+y, map(lambda x: x.recordings, songs))

        for rec in allrecs:
            xmlsong.appendChild( rec.toXML(doc) )

        return xmlsong

    mergeXML = staticmethod(mergeXML)




class Musician:
    def __init__(self, name=''):
        self.id = None
        self.name = name
        self.instrument = None

    def not__repr__(self):
        return string.join( filter( \
            lambda x: x, [self.id, self.name, self.instrument]), ', ')

    def fromXML(self, xel):
        if xel.localName != 'musician':
            raise ParseError('root is not a musician: "%s"' % xel.localName)

        xml_helpers.getAttributes(self, xel, [], ['id'])
        xml_helpers.getElementsText(\
            self, xel, [], ['name', 'instrument'])

        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlmus = doc.createElement('musician')
        condAddAttribute(xmlmus, 'id', self.id)
        condAddTextElement(xmlmus, 'name', self.name)
        condAddTextElement(xmlmus, 'instrument', self.instrument)
        return xmlmus



class Disc:
    """A disc within an album."""

    def __init__(self, no=None):
        self.disc = None
        self.no = no
        self.disctitle = ''
        self.songs = {}

        self.valid = 1

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmldisc = doc.createElement('songs')
        condAddAttribute(xmldisc, 'disc', '%d' % self.no)
        condAddAttribute(xmldisc, 'disctitle', self.disctitle)

        for song in self.songs.values():
            xmldisc.appendChild( song.toXML(doc) )

        return xmldisc

    def fromXML(self, xel):
        xml_helpers.getAttributes( self, xel, [], ['disc', 'disctitle'])
        self.no = (self.disc and int(self.disc)) or 1
        delattr(self, 'disc')

        xmlsongsl = xml_helpers.getChildrenByTagName(xel, 'song')
        for xmlsong in xmlsongsl:
            song = Song().fromXML(xmlsong)
            self.songs[song.no] = song
            if not song.valid:
                self.valid = 0

        return self



class DJSong:
    def __init__(self, disc=0, no=-1):
        self.disc = disc
        self.no = no
        self.bpm = 0
        self.desc = ''

    def __cmp__(self, other):
        return cmp(self.no, other.no)

    def fromXML(self, xel):
        if xel.localName != 'djsong':
            raise ParseError('root is not a djsong: "%s"' % xel.localName)

        xml_helpers.getAttributes(self, xel, ['no'], ['disc', 'bpm'])
        if self.no: self.no = int(self.no)
        if self.disc: self.disc = int(self.disc)
        self.desc = xml_helpers.getText(xel, encoding)

        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlsong = xml_helpers.textElement('djsong', self.desc, doc)
        condAddAttribute(xmlsong, 'no', repr(self.no))
        if self.disc:
            condAddAttribute(xmlsong, 'disc', repr(self.disc))
        condAddAttribute(xmlsong, 'bpm', self.bpm)
        return xmlsong

    def mergeXML(songs, no):
        return None # never used nor implemented.

    mergeXML = staticmethod(mergeXML)



class UserInfo:
    """User info for an album."""

    def __init__(self):
        self.category = ''
        self.type = ''
        self.src = ''
        self.date = ''
        self.conversion = ''

        self.missing_tracks = []
        self.extra_tracks = []
        self.encoding = ''
        self.djsongs = {}
        
    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmluser = doc.createElement('user')
        condAddAttribute(xmluser, 'category', self.category)
        condAddAttribute(xmluser, 'type', self.type)
        condAddAttribute(xmluser, 'src', self.src)
        condAddAttribute(xmluser, 'date', self.date)
        condAddAttribute(xmluser, 'conversion', self.conversion)

        missing = string.join(map(str, self.missing_tracks), ',')
        condAddTextElement(xmluser, 'missing_tracks', missing)
        extra = string.join(map(str, self.extra_tracks), ',')
        condAddTextElement(xmluser, 'extra_tracks', extra)
        condAddTextElement(xmluser, 'encoding', self.encoding)
            
        if self.djsongs:
            xmldj = xml.dom.minidom.Element('dj')
            xmluser.appendChild(xmldj)
            for djsong in self.djsongs.values():
                xmldj.appendChild( djsong.toXML(doc) )

        return xmluser

    def fromXML(self, xel):
        xml_helpers.getAttributes(
            self, xel, [], ['category', 'type', 'src', 'date', 'conversion'])

        xml_helpers.getElementsText( self, xel, [], ['encoding', 'missing_tracks'] )
        xml_helpers.getElementsText( self, xel, [], ['encoding', 'extra_tracks'] )
        
        if self.missing_tracks:
            self.missing_tracks = map(int, string.split(self.missing_tracks, ','))
        if self.extra_tracks:
            self.extra_tracks = map(int, string.split(self.extra_tracks, ','))

        xmldjl = xml_helpers.getChildrenByTagName(xel, 'dj')
        if xmldjl:
            assert(len(xmldjl) == 1)
            
            xmldjsongs = xml_helpers.getChildrenByTagName(xmldjl[0], 'djsong')
            for xmldjsong in xmldjsongs:
                djsong = DJSong().fromXML(xmldjsong)
                self.djsongs[djsong.no] = djsong

        return self



class Album:
    """An album."""

    nodre = re.compile('[0-9]+')

    def __init__(self, id=None):
        self.id = id

        self.descarga = None
        self.descarga_other = None
        self.amazon = None
        self.amazon_other = None

        self.artist = ''
        self.reviews = None
        self.comments = ''
        self.notes = None
        self.title = ''
        self.subtitle = ''
        self.package = ''
        self.discsstr = ''
        self.discs = []
        self.discmap = {}
        self.label = ''
        self.reldate = ''
        self.reldate_num = ''
        self.musicians = []
        self.user = None

        self.attribs = {}

        self.valid = 1

    def sortCmp(x, y):
        c = cmp(x.artist, y.artist)
        if not c:
            c = cmp(x.reldate_num, y.reldate_num)
            if not c:
                c = cmp(x.title, y.title)
        return c
    sortCmp = staticmethod(sortCmp)

    def guessId(self):
        """Tries to construct a valid id."""
        s = '%s-%s' % (strxtra.idify(self.artist), strxtra.idify(self.title))
        ss = ''
        for c in s:
            if isalnum(c) or c == '-':
                ss += c
            else:
                ss += '_'
        s = ss
        return ss

    def fullTitle(self):
        if self.subtitle:
            return '%s -- %s' % (self.title, self.subtitle)
        return self.title

    def getDJSong(self, no, disc=1):
        # Note: this is not very efficient, we could do much better with a
        # pre-computed map.
        if self.user and self.user.djsongs:
            for djs in self.user.djsongs.values():
                if djs.disc == 0:
                    djsdisc = 1
                if djs.no == no and djsdisc == disc:
                    return djs
        return None

    def not__repr__(self):
        r  = "id: %s\n" % repr(self.id)
        r += "title: %s\n" % repr(self.title)
        r += "subtitle: %s\n" % repr(self.subtitle)
        r += "artist: %s\n" % repr(self.artist)
        r += "label: %s\n" % repr(self.label)
        r += "reldate: %s\n" % repr(self.reldate)
        r += "descarga: %s\n" % repr(self.descarga)
        r += "amazon: %s\n" % repr(self.amazon)
        r += "discs:\n"
        for disc in self.discmap:
            r += "  %s\n" % (repr(disc.no))
            for no, song in disc.songs.items():
                r += "    %d - %s\n" % (no, repr(song))
        r += "musicians:\n"
        for s in self.musicians:
            r += "  '%s'\n" % repr(s)
        r += "reviews:"
        if self.reviews:
            r += "\n-----\n" + self.reviews + '\n-----\n'
        r += "comments:"
        if self.comments:
            r += "\n-----\n" + self.comments + '\n-----\n'
        return r

    def fromXML(self, xel):
        if xel.localName not in ['jbalb', 'album']:
            raise ParseError('root is not an album')

        if xel.localName == 'jbalb':
            xmluser = xml_helpers.getChildrenByTagName(xel, 'user')
            if len(xmluser) > 0:
                self.user = UserInfo().fromXML(xmluser[0])
                # override category, FIXME we should eventually remove the
                # top-level category
                self.category = self.user.category

            xmlalbums = xml_helpers.getChildrenByTagName(xel, 'album')
            if len(xmlalbums) != 1:
                raise ParseError('top node is missing album child')
            xel = xmlalbums[0]

        xml_helpers.getElementsText(\
            self, xel, ['title', 'artist'], \
            ['subtitle', 'package', 'discs', 'label', 'reldate',
             'descarga', 'amazon', 'reviews', 'comments'] )

        if self.amazon != None:
            xmlamazon = xml_helpers.getChildrenByTagName(xel, 'amazon')
            self.amazon_other = xmlamazon[0].getAttribute(\
                'other').encode(encoding).strip()

        if self.descarga != None:
            xmldescarga= xml_helpers.getChildrenByTagName(xel, 'descarga')
            self.descarga_other = xmldescarga[0].getAttribute(\
                'other').encode(encoding).strip()

        if self.reldate and not isinstance(self.reldate, types.ListType):
            self.reldate_num = string.join(\
                Album.nodre.findall(self.reldate), ', ')

        if self.discs:
            self.discsstr = self.discs
            self.discs = map(int, string.split(self.discsstr, ','))

        for att in ['title', 'artist', 'subtitle', 'label',
                    'reldate', 'descarga', 'amazon']:
            if isinstance(getattr(self, att), types.ListType):
                setattr(self, att, '')
                self.valid = 0 # mark as invalid

        xmldiscsp = xml_helpers.getChildrenByTagName(xel, 'songs')
        for xmldisc in xmldiscsp:
            disc = Disc().fromXML(xmldisc)
            if disc.no not in self.discmap:
                self.discmap[ disc.no ] = disc
            else:
                self.valid = 0
            if not disc.valid: self.valid = 0

        # create at least all the discs that we have
        if self.discs:
            for no in self.discs:
                if no not in self.discmap:
                    self.discmap[ no ] = Disc(no)

        self.musicians = []
        xmlmusiciansp = xml_helpers.getChildrenByTagName(xel, 'musicians')
        for xmlmusicians in xmlmusiciansp:
            xmlmusicianl = xml_helpers.getChildrenByTagName(\
                xmlmusicians, 'musician')
            for xmlmusician in xmlmusicianl:
                self.musicians.append( Musician().fromXML(xmlmusician) )

        notes = xml_helpers.getChildrenByTagName(xel, 'notes')
        if notes:
            assert(len(notes) == 1)
            # hunt for empty text nodes and remove them
            self.notes = xml_helpers.removeEmptyText(notes[0])
            self.notes = xml_helpers.encodeTree(self.notes, encoding)

        return self

    def toXML(self, attribs='', doc=None):

        "Returns an XML rendering of this album."

        if not doc: doc = Document()

        xmlalbum = doc.createElement('album')
        condAddAttribute(xmlalbum, 'id', self.id)
        condAddTextElement(xmlalbum, 'title', self.title)
        condAddTextElement(xmlalbum, 'subtitle', self.subtitle)
        condAddTextElement(xmlalbum, 'package', self.package)
        condAddTextElement(xmlalbum, 'discs', self.discsstr)
        condAddTextElement(xmlalbum, 'artist', self.artist)
        condAddTextElement(xmlalbum, 'label', self.label)
        condAddTextElement(xmlalbum, 'reldate', self.reldate)
        xmldescarga = condAddTextElement(xmlalbum, 'descarga', self.descarga, 1)
        if xmldescarga:
            condAddAttribute(xmldescarga, 'other', self.descarga_other)
        xmlamazon = condAddTextElement(xmlalbum, 'amazon', self.amazon, 1)
        if xmlamazon:
            condAddAttribute(xmlamazon, 'other', self.amazon_other)

        for disc in self.discmap.values():
            xmldisc = disc.toXML(doc)
            if xmldisc:
                if disc.no == 1 and len(self.discmap) == 1:
                    xmldisc.removeAttribute('disc')
                xmlalbum.appendChild(xmldisc)

        if self.musicians:
            xmlmusicians = Element('musicians')
            xmlalbum.appendChild(xmlmusicians)
            for mus in self.musicians:
                xmlmusicians.appendChild( mus.toXML(doc) )

        condAddTextElement(xmlalbum, 'reviews', self.reviews)
        condAddTextElement(xmlalbum, 'comments', self.comments)

        if self.notes:
            if type(self.notes) == types.StringType:
                condAddTextElement(xmlalbum, 'notes', self.notes)
            else:
                xmlalbum.appendChild(self.notes)

        if self.user:
            xmltop = Element('jbalb')
            xmluser = self.user.toXML(doc)
            xmltop.appendChild(xmluser)
            xmltop.appendChild(xmlalbum)
            xmlalbum = xmltop

        return xmlalbum

#     def mergeXML(albums, attribs=None):
#         """Attempt to reduce the set of information between multiple albums and
#         output in a format that can be reviewed and hand edited by a human."""
# 
#         nalbums = filter(None, albums)
# 
#         xmlalbum = Element('album')
#         condAddAttribute(xmlalbum, 'id', nalbums[0].id)
# 
#         addMany( xmlalbum, nalbums, 'title' )
#         addMany( xmlalbum, nalbums, 'subtitle' )
#         addMany( xmlalbum, nalbums, 'artist' )
#         addMany( xmlalbum, nalbums, 'label' )
#         addMany( xmlalbum, nalbums, 'reldate' )
#         addMany( xmlalbum, nalbums, 'descarga' )
#         addMany( xmlalbum, nalbums, 'amazon' )
# 
#         songsl = filter( None, map(lambda x: x.songs, nalbums) )
#         if songsl:
#             allsongs = reduce( lambda x,y:x+y, songsl )
#         else:
#             allsongs = []
#         if allsongs:
#             xmlsongs = Element('songs')
#             xmlalbum.appendChild(xmlsongs)
# 
#             numbers = unique( map( lambda x: x.no, allsongs) )
#             for n in numbers:
#                 nsongs = filter( lambda x: x.no == n, allsongs )
#                 xmlsongs.appendChild( Song.mergeXML(nsongs, n) )
# 
#         return xmlalbum
# 
#         for a in nalbums:
#             if a.musicians:
#                 xmlmusicians = Element('musicians')
#                 xmlalbum.appendChild(xmlmusicians)
#                 for mus in a.musicians:
#                     xmlmusicians.appendChild( mus.toXML(doc) )
# 
#         if len(filter(lambda x: x.reviews, nalbums)) > 0:
#             for a in nalbums:
#                 condAddTextElement(xmlalbum, 'reviews', a.reviews)
#         return xmlalbum
# 
#     mergeXML = staticmethod(mergeXML)



class SongRef:
    """A reference to a specific song."""

    def __init__(self, songxml=None):
        self.dir = None
        self.no = None
        self.begin = self.end = None
        self.stretch = self.gain = None
        self.comment = None

        if songxml:
            self.fromXML(songxml)

    def __cmp__(self, other):
        c = cmp(self.dir, other.dir)
        if c == 0:
            return cmp(int(self.no), int(other.no))
        else:
            return c

    def __repr__(self):
        s = '%s -- %d' % (self.dir, self.no)
        if self.begin: s += ', begin=%s' % self.begin
        if self.end: s += ', end=%s' % self.end
        if self.stretch: s += ', stretch=%s' % self.stretch
        return s

    def fromXML(self, xel):
        if xel.localName != 'songref':
            raise ParseError('root is not a songref: "%s"' % xel.localName)

        xml_helpers.getAttributes( \
            self, xel, \
            ['dir', 'no'], ['begin', 'end', 'stretch', 'gain', 'comment'] )
        self.no = int(self.no)
        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlsong = doc.createElement('songref')
        condAddAttribute(xmlsong, 'dir', self.dir)
        condAddAttribute(xmlsong, 'no', repr(self.no))
        condAddAttribute(xmlsong, 'begin', self.begin)
        condAddAttribute(xmlsong, 'end', self.end)
        condAddAttribute(xmlsong, 'stretch', self.stretch)
        condAddAttribute(xmlsong, 'gain', self.gain)
        condAddAttribute(xmlsong, 'comment', self.comment)
        return xmlsong


class SongList:
    """A songlist."""

    def __init__(self):
        self.name = ''
        self.songs = []

    def fromXML(self, xel):
        if xel.localName != 'songlist':
            raise ParseError('root is not a songlist: "%s"' % xel.localName)
        xml_helpers.getAttributes( self, xel, [], ['name'] )
        return self

    def toXML(self, doc=None):
        if not doc: doc = Document()

        xmlsonglist = doc.createElement('songlist')
        condAddAttribute(xmlsonglist, 'name', self.name)
        for song in self.songs:
            xmlsonglist.appendChild( song.toXML(doc) )
        return xmlsonglist


def readSongList(f):
    """Reads a file that contains one or multiple songlists and returns a list
    of songlists."""

    try:
        dom = xml.dom.minidom.parse(f)
    except IOError, e:
        raise e

    if not dom.documentElement:
        return []
    
    if dom.documentElement.localName == 'songlist':
        xmllists = [dom.documentElement]
    elif dom.documentElement.localName == 'songlists':
        xmllists = dom.documentElement.getElementsByTagName('songlist')        
    else:
        return []

    songlists = []
    for xmllist in xmllists:
        songlist = SongList()
        songlist.name = xmllist.getAttribute('name').encode(encoding)

        for songxml in xmllist.getElementsByTagName('songref'):
            songref = SongRef(songxml)
            songlist.songs.append(songref)

        songlists.append(songlist)

    dom.unlink()
    return songlists






class JbAlbHandler:
    def __init__(self):
        self.album = Album()
        self.userinfo = UserInfo()
        
    def begin_album(self, name, attrs):
        #print '++album'
        return AlbumHandler(self.album)

    def end_album(self, name, content, handler):
        pass
        #print '--album'
        #print handler

    def begin_user(self, name, attrs):
        return UserInfoHandler(self.userinfo)


class AlbumHandler:
    def __init__(self, album):
        self.album = album
        self.obj = album

    def handleCatalog(self, name, attrs):
        if 'other' in attrs.keys():
            setattr(self.album, '%s_other' % name, attrs['other'])
                  
    def end_reldate(self, name, content):
        self.album.reldate = content
        self.album.reldate_num = string.join(Album.nodre.findall(content), ', ')

    def end_discs(self, name, content):
        self.album.discsstr = content
        self.album.discs = map(int, string.split(content, ','))
        
    def begin_songs(self, name, attrs):
        return DiscHandler()

    def end_songs(self, name, attrs, handler):
        disc = handler.disc
        if disc.no not in self.album.discmap:
            self.album.discmap[ disc.no ] = disc
        else:
            # we have twice the same disc!
            self.album.valid = 0
        if not disc.valid: self.album.valid = 0

    def end_album(self, name, content):
        # for att in ['title', 'artist', 'subtitle', 'label',\
        #             'reldate', 'descarga', 'amazon']:
        #     if isinstance(getattr(self.album, att), types.ListType):
        #         setattr(self.album, att, '')
        #         self.album.valid = 0 # mark as invalid

        # create at least all the discs that we have
        if self.album.discs:
            for no in self.album.discs:
                if no not in self.album.discmap:
                    self.album.discmap[ no ] = Disc(no)

    def begin_musicians(self, name, attrs):
        return MusiciansHandler()

    def end_musicians(self, name, content, handler):
        self.album.musicians = handler.musicians

    def end_notes(self, name, content):
        if content:
            # hunt for empty text nodes and remove them
            self.album.notes = content

for i in ['title', 'artist', 'subtitle', 'package', 'label',
          'descarga', 'amazon', 'reviews', 'comments']:
    AlbumHandler.__dict__['end_%s' % i] = BasicHandler.__dict__['handleSimple']

for i in ['descarga', 'amazon']:
    AlbumHandler.__dict__['begin_%s' % i] = \
                                     AlbumHandler.__dict__['handleCatalog']


class UserInfoHandler:
    def __init__(self, uinfo):
        self.uinfo = uinfo

    def end_encoding(self, name, content):
        pass


class DiscHandler:
    def __init__(self):
        self.disc = Disc()
        
    def begin_songs(self, name, attrs):
        try:
            self.disc.no = int( attrs['disc'] )
        except KeyError:
            self.disc.no = 1
        try:
            self.disc.disctitle = attrs['disctitle']
        except KeyError:
            pass

    def begin_song(self, name, attrs):
        return SongHandler()
    
    def end_song(self, name, content, handler):
        song = handler.song
        self.disc.songs[song.no] = song
        if not song.valid:
            self.disc.valid = 0


class SongHandler:
    def __init__(self):
        self.song = Song()
        self.obj = self.song

    def begin_song(self, name, attrs):
        try:
            self.song.no = int( attrs['no'] )
        except KeyError:
            return ParseError('Error: song must have no')

    def begin_rec(self, name, attrs):
        return RecHandler()

    def end_rec(self, name, content, handler):
        self.song.recordings.append( handler.rec )
        
for i in ['title', 'title_alt', 'artist', 'extra', 'duration', 'lyrics']:
    SongHandler.__dict__['end_%s' % i] = BasicHandler.__dict__['handleSimple']



class RecHandler:
    def __init__(self):
        self.rec = Rec()
        self.obj = self.rec

    def begin_rec(self, name, attrs):
        for a in ['begin', 'end', 'toqueref', 'cantoref']:
            try:
                setattr(self.rec, a, attrs[a])
            except KeyError:
                pass



class MusiciansHandler:
    def __init__(self):
        self.musicians = []
        
    def begin_musician(self, name, attrs):
        return MusicianHandler()
    
    def end_musician(self, name, content, handler):
        self.musicians.append(handler.musician)


class MusicianHandler:
    def __init__(self):
        self.musician = Musician()
        self.obj = self.musician

    def begin_musician(self, name, attrs):
        try:
            self.musician.id = attrs.get('id', None)
        except KeyError:
            pass

for i in ['name', 'instrument']:
    MusicianHandler.__dict__['end_%s' % i] = \
                                      BasicHandler.__dict__['handleSimple']



def parseFile(fn):
    th = JbAlbHandler()
    handler = DelegSAXHandler(th)
    dom = xml.sax.parse(fn, handler)

    a = th.album
    print a.toXML().toprettyxml(indent='   ')


def testdom():
    
    import locale; locale.setlocale(locale.LC_ALL, 'en_US.iso885915')

    if len(sys.argv) < 2:
        raise StandardError("Not enough arguments for test.")

    fn = sys.argv[1]
    dom = xml.dom.minidom.parse(fn)
    albumxml = dom.documentElement

    id = os.path.splitext(os.path.basename(fn))[0]
    album = Album(id).fromXML(albumxml)
    print album.toXML().toprettyxml(indent='   ')


def testsax():

    parseFile('/users/blais/tmp/nuno.xml')


if __name__ == "__main__":
    testsax()
