"""
    Copyright 2000, 2001 Astrolabe by William McClain

    Forked in 2013 to Astronomia

    Copyright 2013 Astronomia by Tim Cera

    This file is part of Astronomia.

    Astronomia is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Astronomia is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Astronomia; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    """

"""Functions which calculate the deltaT correction to convert between
dynamical and universal time.

Reference: Jean Meeus, _Astronomical Algorithms_, second edition, 1998,
Willmann-Bell, Inc.

"""
from bisect import bisect
from astronomia.calendar import jd_to_cal
from astronomia.constants import seconds_per_day
from astronomia.util import polynomial

# _tbl is a list of tuples (jd, seconds), giving deltaT values for the
# beginnings of years in a historical range. [Meeus-1998: table 10.A]

# Updated with newer (better?) values in 2011.
# the following are not ALL from Meeus, but 1657-2010 are taken from
# http://www.iers.org/iers/earth/rotation/ut1lod/table2.html # NO LONGER THERE
# Found info here...
# http://maia.usno.navy.mil/ser7/historic_deltat.data
# With the main site here...
# http://www.usno.navy.mil/USNO/earth-orientation/eo-products/long-term

_tbl = [
    (2312752.5, 121.0),    # (cal_to_jd(1620), 121.0),
    (2313483.5, 112.0),    # (cal_to_jd(1622), 112.0),
    (2314213.5, 103.0),    # (cal_to_jd(1624), 103.0),
    (2314944.5, 95.0),     # (cal_to_jd(1626), 95.0),
    (2315674.5, 88.0),     # (cal_to_jd(1628), 88.0),
    (2316405.5, 82.0),     # (cal_to_jd(1630), 82.0),
    (2317135.5, 77.0),     # (cal_to_jd(1632), 77.0),
    (2317866.5, 72.0),     # (cal_to_jd(1634), 72.0),
    (2318596.5, 68.0),     # (cal_to_jd(1636), 68.0),
    (2319327.5, 63.0),     # (cal_to_jd(1638), 63.0),
    (2320057.5, 60.0),     # (cal_to_jd(1640), 60.0),
    (2320788.5, 56.0),     # (cal_to_jd(1642), 56.0),
    (2321518.5, 53.0),     # (cal_to_jd(1644), 53.0),
    (2322249.5, 51.0),     # (cal_to_jd(1646), 51.0),
    (2322979.5, 48.0),     # (cal_to_jd(1648), 48.0),
    (2323710.5, 46.0),     # (cal_to_jd(1650), 46.0),
    (2324440.5, 44.0),     # (cal_to_jd(1652), 44.0),
    (2325171.5, 42.0),     # (cal_to_jd(1654), 42.0),
    (2325901.5, 40.0),     # (cal_to_jd(1656), 40.0),
    (2326267.5, 44),       # (cal_to_jd(1657.0), 44),
    (2326450.0, 43),       # (cal_to_jd(1657.5), 43),
    (2326632.5, 43),       # (cal_to_jd(1658.0), 43),
    (2326815.0, 41),       # (cal_to_jd(1658.5), 41),
    (2326997.5, 40),       # (cal_to_jd(1659.0), 40),
    (2327180.0, 39),       # (cal_to_jd(1659.5), 39),
    (2327362.5, 38),       # (cal_to_jd(1660.0), 38),
    (2327545.5, 37),       # (cal_to_jd(1660.5), 37),
    (2327728.5, 37),       # (cal_to_jd(1661.0), 37),
    (2327911.0, 36),       # (cal_to_jd(1661.5), 36),
    (2328093.5, 36),       # (cal_to_jd(1662.0), 36),
    (2328276.0, 36),       # (cal_to_jd(1662.5), 36),
    (2328458.5, 37),       # (cal_to_jd(1663.0), 37),
    (2328641.0, 37),       # (cal_to_jd(1663.5), 37),
    (2328823.5, 38),       # (cal_to_jd(1664.0), 38),
    (2329006.5, 37),       # (cal_to_jd(1664.5), 37),
    (2329189.5, 36),       # (cal_to_jd(1665.0), 36),
    (2329372.0, 36),       # (cal_to_jd(1665.5), 36),
    (2329554.5, 35),       # (cal_to_jd(1666.0), 35),
    (2329737.0, 35),       # (cal_to_jd(1666.5), 35),
    (2329919.5, 34),       # (cal_to_jd(1667.0), 34),
    (2330102.0, 33),       # (cal_to_jd(1667.5), 33),
    (2330284.5, 33),       # (cal_to_jd(1668.0), 33),
    (2330467.5, 32),       # (cal_to_jd(1668.5), 32),
    (2330650.5, 32),       # (cal_to_jd(1669.0), 32),
    (2330833.0, 31),       # (cal_to_jd(1669.5), 31),
    (2331015.5, 31),       # (cal_to_jd(1670.0), 31),
    (2331198.0, 30),       # (cal_to_jd(1670.5), 30),
    (2331380.5, 30),       # (cal_to_jd(1671.0), 30),
    (2331563.0, 29),       # (cal_to_jd(1671.5), 29),
    (2331745.5, 29),       # (cal_to_jd(1672.0), 29),
    (2331928.5, 29),       # (cal_to_jd(1672.5), 29),
    (2332111.5, 29),       # (cal_to_jd(1673.0), 29),
    (2332294.0, 29),       # (cal_to_jd(1673.5), 29),
    (2332476.5, 28),       # (cal_to_jd(1674.0), 28),
    (2332659.0, 28),       # (cal_to_jd(1674.5), 28),
    (2332841.5, 27),       # (cal_to_jd(1675.0), 27),
    (2333024.0, 27),       # (cal_to_jd(1675.5), 27),
    (2333206.5, 26),       # (cal_to_jd(1676.0), 26),
    (2333389.5, 26),       # (cal_to_jd(1676.5), 26),
    (2333572.5, 25),       # (cal_to_jd(1677.0), 25),
    (2333755.0, 25),       # (cal_to_jd(1677.5), 25),
    (2333937.5, 25),       # (cal_to_jd(1678.0), 25),
    (2334120.0, 26),       # (cal_to_jd(1678.5), 26),
    (2334302.5, 26),       # (cal_to_jd(1679.0), 26),
    (2334485.0, 26),       # (cal_to_jd(1679.5), 26),
    (2334667.5, 26),       # (cal_to_jd(1680.0), 26),
    (2334850.5, 25),       # (cal_to_jd(1680.5), 25),
    (2335033.5, 25),       # (cal_to_jd(1681.0), 25),
    (2335216.0, 25),       # (cal_to_jd(1681.5), 25),
    (2335398.5, 24),       # (cal_to_jd(1682.0), 24),
    (2335581.0, 24),       # (cal_to_jd(1682.5), 24),
    (2335763.5, 24),       # (cal_to_jd(1683.0), 24),
    (2335946.0, 24),       # (cal_to_jd(1683.5), 24),
    (2336128.5, 24),       # (cal_to_jd(1684.0), 24),
    (2336311.5, 24),       # (cal_to_jd(1684.5), 24),
    (2336494.5, 24),       # (cal_to_jd(1685.0), 24),
    (2336677.0, 24),       # (cal_to_jd(1685.5), 24),
    (2336859.5, 24),       # (cal_to_jd(1686.0), 24),
    (2337042.0, 24),       # (cal_to_jd(1686.5), 24),
    (2337224.5, 23),       # (cal_to_jd(1687.0), 23),
    (2337407.0, 23),       # (cal_to_jd(1687.5), 23),
    (2337589.5, 23),       # (cal_to_jd(1688.0), 23),
    (2337772.5, 23),       # (cal_to_jd(1688.5), 23),
    (2337955.5, 22),       # (cal_to_jd(1689.0), 22),
    (2338138.0, 22),       # (cal_to_jd(1689.5), 22),
    (2338320.5, 22),       # (cal_to_jd(1690.0), 22),
    (2338503.0, 22),       # (cal_to_jd(1690.5), 22),
    (2338685.5, 22),       # (cal_to_jd(1691.0), 22),
    (2338868.0, 21),       # (cal_to_jd(1691.5), 21),
    (2339050.5, 21),       # (cal_to_jd(1692.0), 21),
    (2339233.5, 21),       # (cal_to_jd(1692.5), 21),
    (2339416.5, 21),       # (cal_to_jd(1693.0), 21),
    (2339599.0, 21),       # (cal_to_jd(1693.5), 21),
    (2339781.5, 21),       # (cal_to_jd(1694.0), 21),
    (2339964.0, 21),       # (cal_to_jd(1694.5), 21),
    (2340146.5, 21),       # (cal_to_jd(1695.0), 21),
    (2340329.0, 20),       # (cal_to_jd(1695.5), 20),
    (2340511.5, 20),       # (cal_to_jd(1696.0), 20),
    (2340694.5, 20),       # (cal_to_jd(1696.5), 20),
    (2340877.5, 20),       # (cal_to_jd(1697.0), 20),
    (2341060.0, 20),       # (cal_to_jd(1697.5), 20),
    (2341242.5, 20),       # (cal_to_jd(1698.0), 20),
    (2341425.0, 20),       # (cal_to_jd(1698.5), 20),
    (2341607.5, 20),       # (cal_to_jd(1699.0), 20),
    (2341790.0, 20),       # (cal_to_jd(1699.5), 20),
    (2341972.5, 21),       # (cal_to_jd(1700.0), 21),
    (2342155.0, 21),       # (cal_to_jd(1700.5), 21),
    (2342337.5, 21),       # (cal_to_jd(1701.0), 21),
    (2342520.0, 20),       # (cal_to_jd(1701.5), 20),
    (2342702.5, 20),       # (cal_to_jd(1702.0), 20),
    (2342885.0, 20),       # (cal_to_jd(1702.5), 20),
    (2343067.5, 20),       # (cal_to_jd(1703.0), 20),
    (2343250.0, 20),       # (cal_to_jd(1703.5), 20),
    (2343432.5, 19),       # (cal_to_jd(1704.0), 19),
    (2343615.5, 19),       # (cal_to_jd(1704.5), 19),
    (2343798.5, 19),       # (cal_to_jd(1705.0), 19),
    (2343981.0, 19),       # (cal_to_jd(1705.5), 19),
    (2344163.5, 19),       # (cal_to_jd(1706.0), 19),
    (2344346.0, 20),       # (cal_to_jd(1706.5), 20),
    (2344528.5, 20),       # (cal_to_jd(1707.0), 20),
    (2344711.0, 20),       # (cal_to_jd(1707.5), 20),
    (2344893.5, 20),       # (cal_to_jd(1708.0), 20),
    (2345076.5, 19),       # (cal_to_jd(1708.5), 19),
    (2345259.5, 20),       # (cal_to_jd(1709.0), 20),
    (2345442.0, 20),       # (cal_to_jd(1709.5), 20),
    (2345624.5, 20),       # (cal_to_jd(1710.0), 20),
    (2345807.0, 20),       # (cal_to_jd(1710.5), 20),
    (2345989.5, 20),       # (cal_to_jd(1711.0), 20),
    (2346172.0, 20),       # (cal_to_jd(1711.5), 20),
    (2346354.5, 21),       # (cal_to_jd(1712.0), 21),
    (2346537.5, 21),       # (cal_to_jd(1712.5), 21),
    (2346720.5, 21),       # (cal_to_jd(1713.0), 21),
    (2346903.0, 21),       # (cal_to_jd(1713.5), 21),
    (2347085.5, 21),       # (cal_to_jd(1714.0), 21),
    (2347268.0, 21),       # (cal_to_jd(1714.5), 21),
    (2347450.5, 21),       # (cal_to_jd(1715.0), 21),
    (2347633.0, 21),       # (cal_to_jd(1715.5), 21),
    (2347815.5, 21),       # (cal_to_jd(1716.0), 21),
    (2347998.5, 21),       # (cal_to_jd(1716.5), 21),
    (2348181.5, 21),       # (cal_to_jd(1717.0), 21),
    (2348364.0, 21),       # (cal_to_jd(1717.5), 21),
    (2348546.5, 21),       # (cal_to_jd(1718.0), 21),
    (2348729.0, 21),       # (cal_to_jd(1718.5), 21),
    (2348911.5, 21),       # (cal_to_jd(1719.0), 21),
    (2349094.0, 21),       # (cal_to_jd(1719.5), 21),
    (2349276.5, 21.1),     # (cal_to_jd(1720.0), 21.1),
    (2349459.5, 21.0),     # (cal_to_jd(1720.5), 21.0),
    (2349642.5, 21.0),     # (cal_to_jd(1721.0), 21.0),
    (2349825.0, 21.0),     # (cal_to_jd(1721.5), 21.0),
    (2350007.5, 20.9),     # (cal_to_jd(1722.0), 20.9),
    (2350190.0, 20.8),     # (cal_to_jd(1722.5), 20.8),
    (2350372.5, 20.7),     # (cal_to_jd(1723.0), 20.7),
    (2350555.0, 20.6),     # (cal_to_jd(1723.5), 20.6),
    (2350737.5, 20.4),     # (cal_to_jd(1724.0), 20.4),
    (2350920.5, 20.2),     # (cal_to_jd(1724.5), 20.2),
    (2351103.5, 20.0),     # (cal_to_jd(1725.0), 20.0),
    (2351286.0, 19.7),     # (cal_to_jd(1725.5), 19.7),
    (2351468.5, 19.4),     # (cal_to_jd(1726.0), 19.4),
    (2351651.0, 19.1),     # (cal_to_jd(1726.5), 19.1),
    (2351833.5, 18.7),     # (cal_to_jd(1727.0), 18.7),
    (2352016.0, 18.3),     # (cal_to_jd(1727.5), 18.3),
    (2352198.5, 17.8),     # (cal_to_jd(1728.0), 17.8),
    (2352381.5, 17.4),     # (cal_to_jd(1728.5), 17.4),
    (2352564.5, 17.0),     # (cal_to_jd(1729.0), 17.0),
    (2352747.0, 16.8),     # (cal_to_jd(1729.5), 16.8),
    (2352929.5, 16.6),     # (cal_to_jd(1730.0), 16.6),
    (2353112.0, 16.4),     # (cal_to_jd(1730.5), 16.4),
    (2353294.5, 16.1),     # (cal_to_jd(1731.0), 16.1),
    (2353477.0, 15.9),     # (cal_to_jd(1731.5), 15.9),
    (2353659.5, 15.7),     # (cal_to_jd(1732.0), 15.7),
    (2353842.5, 15.5),     # (cal_to_jd(1732.5), 15.5),
    (2354025.5, 15.3),     # (cal_to_jd(1733.0), 15.3),
    (2354208.0, 15.0),     # (cal_to_jd(1733.5), 15.0),
    (2354390.5, 14.7),     # (cal_to_jd(1734.0), 14.7),
    (2354573.0, 14.5),     # (cal_to_jd(1734.5), 14.5),
    (2354755.5, 14.3),     # (cal_to_jd(1735.0), 14.3),
    (2354938.0, 14.2),     # (cal_to_jd(1735.5), 14.2),
    (2355120.5, 14.1),     # (cal_to_jd(1736.0), 14.1),
    (2355303.5, 14.1),     # (cal_to_jd(1736.5), 14.1),
    (2355486.5, 14.1),     # (cal_to_jd(1737.0), 14.1),
    (2355669.0, 13.9),     # (cal_to_jd(1737.5), 13.9),
    (2355851.5, 13.7),     # (cal_to_jd(1738.0), 13.7),
    (2356034.0, 13.6),     # (cal_to_jd(1738.5), 13.6),
    (2356216.5, 13.5),     # (cal_to_jd(1739.0), 13.5),
    (2356399.0, 13.5),     # (cal_to_jd(1739.5), 13.5),
    (2356581.5, 13.5),     # (cal_to_jd(1740.0), 13.5),
    (2356764.5, 13.5),     # (cal_to_jd(1740.5), 13.5),
    (2356947.5, 13.4),     # (cal_to_jd(1741.0), 13.4),
    (2357130.0, 13.4),     # (cal_to_jd(1741.5), 13.4),
    (2357312.5, 13.4),     # (cal_to_jd(1742.0), 13.4),
    (2357495.0, 13.4),     # (cal_to_jd(1742.5), 13.4),
    (2357677.5, 13.3),     # (cal_to_jd(1743.0), 13.3),
    (2357860.0, 13.3),     # (cal_to_jd(1743.5), 13.3),
    (2358042.5, 13.2),     # (cal_to_jd(1744.0), 13.2),
    (2358225.5, 13.2),     # (cal_to_jd(1744.5), 13.2),
    (2358408.5, 13.2),     # (cal_to_jd(1745.0), 13.2),
    (2358591.0, 13.1),     # (cal_to_jd(1745.5), 13.1),
    (2358773.5, 13.1),     # (cal_to_jd(1746.0), 13.1),
    (2358956.0, 13.1),     # (cal_to_jd(1746.5), 13.1),
    (2359138.5, 13.0),     # (cal_to_jd(1747.0), 13.0),
    (2359321.0, 13.2),     # (cal_to_jd(1747.5), 13.2),
    (2359503.5, 13.3),     # (cal_to_jd(1748.0), 13.3),
    (2359686.5, 13.4),     # (cal_to_jd(1748.5), 13.4),
    (2359869.5, 13.5),     # (cal_to_jd(1749.0), 13.5),
    (2360052.0, 13.6),     # (cal_to_jd(1749.5), 13.6),
    (2360234.5, 13.7),     # (cal_to_jd(1750.0), 13.7),
    (2360417.0, 13.8),     # (cal_to_jd(1750.5), 13.8),
    (2360599.5, 13.9),     # (cal_to_jd(1751.0), 13.9),
    (2360782.0, 14.0),     # (cal_to_jd(1751.5), 14.0),
    (2360964.5, 14.0),     # (cal_to_jd(1752.0), 14.0),
    (2361147.5, 14.1),     # (cal_to_jd(1752.5), 14.1),
    (2361330.5, 14.1),     # (cal_to_jd(1753.0), 14.1),
    (2361513.0, 14.1),     # (cal_to_jd(1753.5), 14.1),
    (2361695.5, 14.1),     # (cal_to_jd(1754.0), 14.1),
    (2361878.0, 14.2),     # (cal_to_jd(1754.5), 14.2),
    (2362060.5, 14.3),     # (cal_to_jd(1755.0), 14.3),
    (2362243.0, 14.4),     # (cal_to_jd(1755.5), 14.4),
    (2362425.5, 14.4),     # (cal_to_jd(1756.0), 14.4),
    (2362608.5, 14.5),     # (cal_to_jd(1756.5), 14.5),
    (2362791.5, 14.6),     # (cal_to_jd(1757.0), 14.6),
    (2362974.0, 14.6),     # (cal_to_jd(1757.5), 14.6),
    (2363156.5, 14.7),     # (cal_to_jd(1758.0), 14.7),
    (2363339.0, 14.7),     # (cal_to_jd(1758.5), 14.7),
    (2363521.5, 14.7),     # (cal_to_jd(1759.0), 14.7),
    (2363704.0, 14.8),     # (cal_to_jd(1759.5), 14.8),
    (2363886.5, 14.8),     # (cal_to_jd(1760.0), 14.8),
    (2364069.5, 14.9),     # (cal_to_jd(1760.5), 14.9),
    (2364252.5, 14.9),     # (cal_to_jd(1761.0), 14.9),
    (2364435.0, 15.0),     # (cal_to_jd(1761.5), 15.0),
    (2364617.5, 15.0),     # (cal_to_jd(1762.0), 15.0),
    (2364800.0, 15.1),     # (cal_to_jd(1762.5), 15.1),
    (2364982.5, 15.2),     # (cal_to_jd(1763.0), 15.2),
    (2365165.0, 15.3),     # (cal_to_jd(1763.5), 15.3),
    (2365347.5, 15.4),     # (cal_to_jd(1764.0), 15.4),
    (2365530.5, 15.5),     # (cal_to_jd(1764.5), 15.5),
    (2365713.5, 15.6),     # (cal_to_jd(1765.0), 15.6),
    (2365896.0, 15.6),     # (cal_to_jd(1765.5), 15.6),
    (2366078.5, 15.6),     # (cal_to_jd(1766.0), 15.6),
    (2366261.0, 15.8),     # (cal_to_jd(1766.5), 15.8),
    (2366443.5, 15.9),     # (cal_to_jd(1767.0), 15.9),
    (2366626.0, 15.9),     # (cal_to_jd(1767.5), 15.9),
    (2366808.5, 15.9),     # (cal_to_jd(1768.0), 15.9),
    (2366991.5, 15.8),     # (cal_to_jd(1768.5), 15.8),
    (2367174.5, 15.7),     # (cal_to_jd(1769.0), 15.7),
    (2367357.0, 15.8),     # (cal_to_jd(1769.5), 15.8),
    (2367539.5, 15.7),     # (cal_to_jd(1770.0), 15.7),
    (2367722.0, 15.7),     # (cal_to_jd(1770.5), 15.7),
    (2367904.5, 15.7),     # (cal_to_jd(1771.0), 15.7),
    (2368087.0, 15.8),     # (cal_to_jd(1771.5), 15.8),
    (2368269.5, 15.9),     # (cal_to_jd(1772.0), 15.9),
    (2368452.5, 16.1),     # (cal_to_jd(1772.5), 16.1),
    (2368635.5, 16.1),     # (cal_to_jd(1773.0), 16.1),
    (2368818.0, 16.0),     # (cal_to_jd(1773.5), 16.0),
    (2369000.5, 15.9),     # (cal_to_jd(1774.0), 15.9),
    (2369183.0, 15.9),     # (cal_to_jd(1774.5), 15.9),
    (2369365.5, 15.7),     # (cal_to_jd(1775.0), 15.7),
    (2369548.0, 15.4),     # (cal_to_jd(1775.5), 15.4),
    (2369730.5, 15.3),     # (cal_to_jd(1776.0), 15.3),
    (2369913.5, 15.4),     # (cal_to_jd(1776.5), 15.4),
    (2370096.5, 15.5),     # (cal_to_jd(1777.0), 15.5),
    (2370279.0, 15.6),     # (cal_to_jd(1777.5), 15.6),
    (2370461.5, 15.6),     # (cal_to_jd(1778.0), 15.6),
    (2370644.0, 15.6),     # (cal_to_jd(1778.5), 15.6),
    (2370826.5, 15.6),     # (cal_to_jd(1779.0), 15.6),
    (2371009.0, 15.6),     # (cal_to_jd(1779.5), 15.6),
    (2371191.5, 15.6),     # (cal_to_jd(1780.0), 15.6),
    (2371374.5, 15.6),     # (cal_to_jd(1780.5), 15.6),
    (2371557.5, 15.5),     # (cal_to_jd(1781.0), 15.5),
    (2371740.0, 15.5),     # (cal_to_jd(1781.5), 15.5),
    (2371922.5, 15.4),     # (cal_to_jd(1782.0), 15.4),
    (2372105.0, 15.3),     # (cal_to_jd(1782.5), 15.3),
    (2372287.5, 15.2),     # (cal_to_jd(1783.0), 15.2),
    (2372470.0, 15.1),     # (cal_to_jd(1783.5), 15.1),
    (2372652.5, 14.9),     # (cal_to_jd(1784.0), 14.9),
    (2372835.5, 14.8),     # (cal_to_jd(1784.5), 14.8),
    (2373018.5, 14.6),     # (cal_to_jd(1785.0), 14.6),
    (2373201.0, 14.4),     # (cal_to_jd(1785.5), 14.4),
    (2373383.5, 14.3),     # (cal_to_jd(1786.0), 14.3),
    (2373566.0, 14.2),     # (cal_to_jd(1786.5), 14.2),
    (2373748.5, 14.1),     # (cal_to_jd(1787.0), 14.1),
    (2373931.0, 14.2),     # (cal_to_jd(1787.5), 14.2),
    (2374113.5, 14.2),     # (cal_to_jd(1788.0), 14.2),
    (2374296.5, 13.9),     # (cal_to_jd(1788.5), 13.9),
    (2374479.5, 13.7),     # (cal_to_jd(1789.0), 13.7),
    (2374662.0, 13.5),     # (cal_to_jd(1789.5), 13.5),
    (2374844.5, 13.3),     # (cal_to_jd(1790.0), 13.3),
    (2375027.0, 13.1),     # (cal_to_jd(1790.5), 13.1),
    (2375209.5, 13.0),     # (cal_to_jd(1791.0), 13.0),
    (2375392.0, 13.2),     # (cal_to_jd(1791.5), 13.2),
    (2375574.5, 13.2),     # (cal_to_jd(1792.0), 13.2),
    (2375757.5, 13.1),     # (cal_to_jd(1792.5), 13.1),
    (2375940.5, 13.1),     # (cal_to_jd(1793.0), 13.1),
    (2376123.0, 13.2),     # (cal_to_jd(1793.5), 13.2),
    (2376305.5, 13.3),     # (cal_to_jd(1794.0), 13.3),
    (2376488.0, 13.5),     # (cal_to_jd(1794.5), 13.5),
    (2376670.5, 13.5),     # (cal_to_jd(1795.0), 13.5),
    (2376853.0, 13.4),     # (cal_to_jd(1795.5), 13.4),
    (2377035.5, 13.2),     # (cal_to_jd(1796.0), 13.2),
    (2377218.5, 13.2),     # (cal_to_jd(1796.5), 13.2),
    (2377401.5, 13.1),     # (cal_to_jd(1797.0), 13.1),
    (2377584.0, 13.1),     # (cal_to_jd(1797.5), 13.1),
    (2377766.5, 13.0),     # (cal_to_jd(1798.0), 13.0),
    (2377949.0, 12.8),     # (cal_to_jd(1798.5), 12.8),
    (2378131.5, 12.6),     # (cal_to_jd(1799.0), 12.6),
    (2378314.0, 12.7),     # (cal_to_jd(1799.5), 12.7),
    (2378496.5, 12.6),     # (cal_to_jd(1800.0), 12.6),
    (2378679.0, 12.3),     # (cal_to_jd(1800.5), 12.3),
    (2378861.5, 12.0),     # (cal_to_jd(1801.0), 12.0),
    (2379044.0, 11.9),     # (cal_to_jd(1801.5), 11.9),
    (2379226.5, 11.8),     # (cal_to_jd(1802.0), 11.8),
    (2379409.0, 11.6),     # (cal_to_jd(1802.5), 11.6),
    (2379591.5, 11.4),     # (cal_to_jd(1803.0), 11.4),
    (2379774.0, 11.2),     # (cal_to_jd(1803.5), 11.2),
    (2379956.5, 11.1),     # (cal_to_jd(1804.0), 11.1),
    (2380139.5, 11.1),     # (cal_to_jd(1804.5), 11.1),
    (2380322.5, 11.1),     # (cal_to_jd(1805.0), 11.1),
    (2380505.0, 11.1),     # (cal_to_jd(1805.5), 11.1),
    (2380687.5, 11.1),     # (cal_to_jd(1806.0), 11.1),
    (2380870.0, 11.2),     # (cal_to_jd(1806.5), 11.2),
    (2381052.5, 11.1),     # (cal_to_jd(1807.0), 11.1),
    (2381235.0, 11.1),     # (cal_to_jd(1807.5), 11.1),
    (2381417.5, 11.2),     # (cal_to_jd(1808.0), 11.2),
    (2381600.5, 11.4),     # (cal_to_jd(1808.5), 11.4),
    (2381783.5, 11.5),     # (cal_to_jd(1809.0), 11.5),
    (2381966.0, 11.3),     # (cal_to_jd(1809.5), 11.3),
    (2382148.5, 11.2),     # (cal_to_jd(1810.0), 11.2),
    (2382331.0, 11.4),     # (cal_to_jd(1810.5), 11.4),
    (2382513.5, 11.7),     # (cal_to_jd(1811.0), 11.7),
    (2382696.0, 11.9),     # (cal_to_jd(1811.5), 11.9),
    (2382878.5, 11.9),     # (cal_to_jd(1812.0), 11.9),
    (2383061.5, 11.9),     # (cal_to_jd(1812.5), 11.9),
    (2383244.5, 11.8),     # (cal_to_jd(1813.0), 11.8),
    (2383427.0, 11.7),     # (cal_to_jd(1813.5), 11.7),
    (2383609.5, 11.8),     # (cal_to_jd(1814.0), 11.8),
    (2383792.0, 11.8),     # (cal_to_jd(1814.5), 11.8),
    (2383974.5, 11.8),     # (cal_to_jd(1815.0), 11.8),
    (2384157.0, 11.7),     # (cal_to_jd(1815.5), 11.7),
    (2384339.5, 11.6),     # (cal_to_jd(1816.0), 11.6),
    (2384522.5, 11.6),     # (cal_to_jd(1816.5), 11.6),
    (2384705.5, 11.5),     # (cal_to_jd(1817.0), 11.5),
    (2384888.0, 11.5),     # (cal_to_jd(1817.5), 11.5),
    (2385070.5, 11.4),     # (cal_to_jd(1818.0), 11.4),
    (2385253.0, 11.4),     # (cal_to_jd(1818.5), 11.4),
    (2385435.5, 11.3),     # (cal_to_jd(1819.0), 11.3),
    (2385618.0, 11.3),     # (cal_to_jd(1819.5), 11.3),
    (2385800.5, 11.13),    # (cal_to_jd(1820.0), 11.13),
    (2385983.5, 11.16),    # (cal_to_jd(1820.5), 11.16),
    (2386166.5, 10.94),    # (cal_to_jd(1821.0), 10.94),
    (2386349.0, 10.72),    # (cal_to_jd(1821.5), 10.72),
    (2386531.5, 10.29),    # (cal_to_jd(1822.0), 10.29),
    (2386714.0, 10.04),    # (cal_to_jd(1822.5), 10.04),
    (2386896.5, 9.94),     # (cal_to_jd(1823.0), 9.94),
    (2387079.0, 9.91),     # (cal_to_jd(1823.5), 9.91),
    (2387261.5, 9.88),     # (cal_to_jd(1824.0), 9.88),
    (2387444.5, 9.86),     # (cal_to_jd(1824.5), 9.86),
    (2387627.5, 9.72),     # (cal_to_jd(1825.0), 9.72),
    (2387810.0, 9.67),     # (cal_to_jd(1825.5), 9.67),
    (2387992.5, 9.66),     # (cal_to_jd(1826.0), 9.66),
    (2388175.0, 9.64),     # (cal_to_jd(1826.5), 9.64),
    (2388357.5, 9.51),     # (cal_to_jd(1827.0), 9.51),
    (2388540.0, 9.4),      # (cal_to_jd(1827.5), 9.40),
    (2388722.5, 9.21),     # (cal_to_jd(1828.0), 9.21),
    (2388905.5, 9.0),      # (cal_to_jd(1828.5), 9.),
    (2389088.5, 8.6),      # (cal_to_jd(1829.0), 8.60),
    (2389271.0, 8.29),     # (cal_to_jd(1829.5), 8.29),
    (2389453.5, 7.95),     # (cal_to_jd(1830.0), 7.95),
    (2389636.0, 7.73),     # (cal_to_jd(1830.5), 7.73),
    (2389818.5, 7.59),     # (cal_to_jd(1831.0), 7.59),
    (2390001.0, 7.49),     # (cal_to_jd(1831.5), 7.49),
    (2390183.5, 7.36),     # (cal_to_jd(1832.0), 7.36),
    (2390366.5, 7.26),     # (cal_to_jd(1832.5), 7.26),
    (2390549.5, 7.1),      # (cal_to_jd(1833.0), 7.10),
    (2390732.0, 7.0),      # (cal_to_jd(1833.5), 7.),
    (2390914.5, 6.89),     # (cal_to_jd(1834.0), 6.89),
    (2391097.0, 6.82),     # (cal_to_jd(1834.5), 6.82),
    (2391279.5, 6.73),     # (cal_to_jd(1835.0), 6.73),
    (2391462.0, 6.64),     # (cal_to_jd(1835.5), 6.64),
    (2391644.5, 6.39),     # (cal_to_jd(1836.0), 6.39),
    (2391827.5, 6.28),     # (cal_to_jd(1836.5), 6.28),
    (2392010.5, 6.25),     # (cal_to_jd(1837.0), 6.25),
    (2392193.0, 6.27),     # (cal_to_jd(1837.5), 6.27),
    (2392375.5, 6.25),     # (cal_to_jd(1838.0), 6.25),
    (2392558.0, 6.27),     # (cal_to_jd(1838.5), 6.27),
    (2392740.5, 6.22),     # (cal_to_jd(1839.0), 6.22),
    (2392923.0, 6.24),     # (cal_to_jd(1839.5), 6.24),
    (2393105.5, 6.22),     # (cal_to_jd(1840.0), 6.22),
    (2393288.5, 6.27),     # (cal_to_jd(1840.5), 6.27),
    (2393471.5, 6.3),      # (cal_to_jd(1841.0), 6.30),
    (2393654.0, 6.36),     # (cal_to_jd(1841.5), 6.36),
    (2393836.5, 6.35),     # (cal_to_jd(1842.0), 6.35),
    (2394019.0, 6.37),     # (cal_to_jd(1842.5), 6.37),
    (2394201.5, 6.32),     # (cal_to_jd(1843.0), 6.32),
    (2394384.0, 6.33),     # (cal_to_jd(1843.5), 6.33),
    (2394566.5, 6.33),     # (cal_to_jd(1844.0), 6.33),
    (2394749.5, 6.37),     # (cal_to_jd(1844.5), 6.37),
    (2394932.5, 6.37),     # (cal_to_jd(1845.0), 6.37),
    (2395115.0, 6.41),     # (cal_to_jd(1845.5), 6.41),
    (2395297.5, 6.4),      # (cal_to_jd(1846.0), 6.40),
    (2395480.0, 6.44),     # (cal_to_jd(1846.5), 6.44),
    (2395662.5, 6.46),     # (cal_to_jd(1847.0), 6.46),
    (2395845.0, 6.51),     # (cal_to_jd(1847.5), 6.51),
    (2396027.5, 6.48),     # (cal_to_jd(1848.0), 6.48),
    (2396210.5, 6.51),     # (cal_to_jd(1848.5), 6.51),
    (2396393.5, 6.53),     # (cal_to_jd(1849.0), 6.53),
    (2396576.0, 6.58),     # (cal_to_jd(1849.5), 6.58),
    (2396758.5, 6.55),     # (cal_to_jd(1850.0), 6.55),
    (2396941.0, 6.61),     # (cal_to_jd(1850.5), 6.61),
    (2397123.5, 6.69),     # (cal_to_jd(1851.0), 6.69),
    (2397306.0, 6.8),      # (cal_to_jd(1851.5), 6.80),
    (2397488.5, 6.84),     # (cal_to_jd(1852.0), 6.84),
    (2397671.5, 6.94),     # (cal_to_jd(1852.5), 6.94),
    (2397854.5, 7.03),     # (cal_to_jd(1853.0), 7.03),
    (2398037.0, 7.13),     # (cal_to_jd(1853.5), 7.13),
    (2398219.5, 7.15),     # (cal_to_jd(1854.0), 7.15),
    (2398402.0, 7.22),     # (cal_to_jd(1854.5), 7.22),
    (2398584.5, 7.26),     # (cal_to_jd(1855.0), 7.26),
    (2398767.0, 7.3),      # (cal_to_jd(1855.5), 7.30),
    (2398949.5, 7.23),     # (cal_to_jd(1856.0), 7.23),
    (2399132.5, 7.22),     # (cal_to_jd(1856.5), 7.22),
    (2399315.5, 7.21),     # (cal_to_jd(1857.0), 7.21),
    (2399498.0, 7.2),      # (cal_to_jd(1857.5), 7.20),
    (2399680.5, 6.99),     # (cal_to_jd(1858.0), 6.99),
    (2399863.0, 6.98),     # (cal_to_jd(1858.5), 6.98),
    (2400045.5, 7.19),     # (cal_to_jd(1859.0), 7.19),
    (2400228.0, 7.36),     # (cal_to_jd(1859.5), 7.36),
    (2400410.5, 7.35),     # (cal_to_jd(1860.0), 7.35),
    (2400593.5, 7.39),     # (cal_to_jd(1860.5), 7.39),
    (2400776.5, 7.41),     # (cal_to_jd(1861.0), 7.41),
    (2400959.0, 7.45),     # (cal_to_jd(1861.5), 7.45),
    (2401141.5, 7.36),     # (cal_to_jd(1862.0), 7.36),
    (2401324.0, 7.18),     # (cal_to_jd(1862.5), 7.18),
    (2401506.5, 6.95),     # (cal_to_jd(1863.0), 6.95),
    (2401689.0, 6.72),     # (cal_to_jd(1863.5), 6.72),
    (2401871.5, 6.45),     # (cal_to_jd(1864.0), 6.45),
    (2402054.5, 6.24),     # (cal_to_jd(1864.5), 6.24),
    (2402237.5, 5.92),     # (cal_to_jd(1865.0), 5.92),
    (2402420.0, 5.59),     # (cal_to_jd(1865.5), 5.59),
    (2402602.5, 5.15),     # (cal_to_jd(1866.0), 5.15),
    (2402785.0, 4.67),     # (cal_to_jd(1866.5), 4.67),
    (2402967.5, 4.11),     # (cal_to_jd(1867.0), 4.11),
    (2403150.0, 3.52),     # (cal_to_jd(1867.5), 3.52),
    (2403332.5, 2.94),     # (cal_to_jd(1868.0), 2.94),
    (2403515.5, 2.47),     # (cal_to_jd(1868.5), 2.47),
    (2403698.5, 1.97),     # (cal_to_jd(1869.0), 1.97),
    (2403881.0, 1.52),     # (cal_to_jd(1869.5), 1.52),
    (2404063.5, 1.04),     # (cal_to_jd(1870.0), 1.04),
    (2404246.0, 0.6),      # (cal_to_jd(1870.5), 0.60),
    (2404428.5, 0.11),     # (cal_to_jd(1871.0), 0.11),
    (2404611.0, -0.34),    # (cal_to_jd(1871.5), -0.34),
    (2404793.5, -0.82),    # (cal_to_jd(1872.0), -0.82),
    (2404976.5, -1.25),    # (cal_to_jd(1872.5), -1.25),
    (2405159.5, -1.7),     # (cal_to_jd(1873.0), -1.70),
    (2405342.0, -2.08),    # (cal_to_jd(1873.5), -2.08),
    (2405524.5, -2.48),    # (cal_to_jd(1874.0), -2.48),
    (2405707.0, -2.82),    # (cal_to_jd(1874.5), -2.82),
    (2405889.5, -3.19),    # (cal_to_jd(1875.0), -3.19),
    (2406072.0, -3.5),     # (cal_to_jd(1875.5), -3.50),
    (2406254.5, -3.84),    # (cal_to_jd(1876.0), -3.84),
    (2406437.5, -4.14),    # (cal_to_jd(1876.5), -4.14),
    (2406620.5, -4.43),    # (cal_to_jd(1877.0), -4.43),
    (2406803.0, -4.59),    # (cal_to_jd(1877.5), -4.59),
    (2406985.5, -4.79),    # (cal_to_jd(1878.0), -4.79),
    (2407168.0, -4.92),    # (cal_to_jd(1878.5), -4.92),
    (2407350.5, -5.09),    # (cal_to_jd(1879.0), -5.09),
    (2407533.0, -5.24),    # (cal_to_jd(1879.5), -5.24),
    (2407715.5, -5.36),    # (cal_to_jd(1880.0), -5.36),
    (2407898.5, -5.34),    # (cal_to_jd(1880.5), -5.34),
    (2408081.5, -5.37),    # (cal_to_jd(1881.0), -5.37),
    (2408264.0, -5.32),    # (cal_to_jd(1881.5), -5.32),
    (2408446.5, -5.34),    # (cal_to_jd(1882.0), -5.34),
    (2408629.0, -5.33),    # (cal_to_jd(1882.5), -5.33),
    (2408811.5, -5.4),     # (cal_to_jd(1883.0), -5.40),
    (2408994.0, -5.47),    # (cal_to_jd(1883.5), -5.47),
    (2409176.5, -5.58),    # (cal_to_jd(1884.0), -5.58),
    (2409359.5, -5.66),    # (cal_to_jd(1884.5), -5.66),
    (2409542.5, -5.74),    # (cal_to_jd(1885.0), -5.74),
    (2409725.0, -5.68),    # (cal_to_jd(1885.5), -5.68),
    (2409907.5, -5.69),    # (cal_to_jd(1886.0), -5.69),
    (2410090.0, -5.65),    # (cal_to_jd(1886.5), -5.65),
    (2410272.5, -5.67),    # (cal_to_jd(1887.0), -5.67),
    (2410455.0, -5.68),    # (cal_to_jd(1887.5), -5.68),
    (2410637.5, -5.73),    # (cal_to_jd(1888.0), -5.73),
    (2410820.5, -5.72),    # (cal_to_jd(1888.5), -5.72),
    (2411003.5, -5.78),    # (cal_to_jd(1889.0), -5.78),
    (2411186.0, -5.79),    # (cal_to_jd(1889.5), -5.79),
    (2411368.5, -5.86),    # (cal_to_jd(1890.0), -5.86),
    (2411551.0, -5.89),    # (cal_to_jd(1890.5), -5.89),
    (2411733.5, -6.01),    # (cal_to_jd(1891.0), -6.01),
    (2411916.0, -6.13),    # (cal_to_jd(1891.5), -6.13),
    (2412098.5, -6.28),    # (cal_to_jd(1892.0), -6.28),
    (2412281.5, -6.41),    # (cal_to_jd(1892.5), -6.41),
    (2412464.5, -6.53),    # (cal_to_jd(1893.0), -6.53),
    (2412647.0, -6.49),    # (cal_to_jd(1893.5), -6.49),
    (2412829.5, -6.5),     # (cal_to_jd(1894.0), -6.50),
    (2413012.0, -6.45),    # (cal_to_jd(1894.5), -6.45),
    (2413194.5, -6.41),    # (cal_to_jd(1895.0), -6.41),
    (2413377.0, -6.26),    # (cal_to_jd(1895.5), -6.26),
    (2413559.5, -6.11),    # (cal_to_jd(1896.0), -6.11),
    (2413742.5, -5.9),     # (cal_to_jd(1896.5), -5.90),
    (2413925.5, -5.63),    # (cal_to_jd(1897.0), -5.63),
    (2414108.0, -5.13),    # (cal_to_jd(1897.5), -5.13),
    (2414290.5, -4.68),    # (cal_to_jd(1898.0), -4.68),
    (2414473.0, -4.19),    # (cal_to_jd(1898.5), -4.19),
    (2414655.5, -3.72),    # (cal_to_jd(1899.0), -3.72),
    (2414838.0, -3.21),    # (cal_to_jd(1899.5), -3.21),
    (2415020.5, -2.7),     # (cal_to_jd(1900.0), -2.70),
    (2415203.0, -2.09),    # (cal_to_jd(1900.5), -2.09),
    (2415385.5, -1.48),    # (cal_to_jd(1901.0), -1.48),
    (2415568.0, -0.75),    # (cal_to_jd(1901.5), -0.75),
    (2415750.5, -0.08),    # (cal_to_jd(1902.0), -0.08),
    (2415933.0, 0.62),     # (cal_to_jd(1902.5), 0.62),
    (2416115.5, 1.26),     # (cal_to_jd(1903.0), 1.26),
    (2416298.0, 1.95),     # (cal_to_jd(1903.5), 1.95),
    (2416480.5, 2.59),     # (cal_to_jd(1904.0), 2.59),
    (2416663.5, 3.28),     # (cal_to_jd(1904.5), 3.28),
    (2416846.5, 3.92),     # (cal_to_jd(1905.0), 3.92),
    (2417029.0, 4.61),     # (cal_to_jd(1905.5), 4.61),
    (2417211.5, 5.2),      # (cal_to_jd(1906.0), 5.20),
    (2417394.0, 5.73),     # (cal_to_jd(1906.5), 5.73),
    (2417576.5, 6.29),     # (cal_to_jd(1907.0), 6.29),
    (2417759.0, 7.0),      # (cal_to_jd(1907.5), 7.),
    (2417941.5, 7.68),     # (cal_to_jd(1908.0), 7.68),
    (2418124.5, 8.45),     # (cal_to_jd(1908.5), 8.45),
    (2418307.5, 9.13),     # (cal_to_jd(1909.0), 9.13),
    (2418490.0, 9.78),     # (cal_to_jd(1909.5), 9.78),
    (2418672.5, 10.38),    # (cal_to_jd(1910.0), 10.38),
    (2418855.0, 10.99),    # (cal_to_jd(1910.5), 10.99),
    (2419037.5, 11.64),    # (cal_to_jd(1911.0), 11.64),
    (2419220.0, 12.47),    # (cal_to_jd(1911.5), 12.47),
    (2419402.5, 13.23),    # (cal_to_jd(1912.0), 13.23),
    (2419585.5, 14.0),     # (cal_to_jd(1912.5), 14.),
    (2419768.5, 14.69),    # (cal_to_jd(1913.0), 14.69),
    (2419951.0, 15.38),    # (cal_to_jd(1913.5), 15.38),
    (2420133.5, 16.0),     # (cal_to_jd(1914.0), 16.),
    (2420316.0, 16.64),    # (cal_to_jd(1914.5), 16.64),
    (2420498.5, 17.19),    # (cal_to_jd(1915.0), 17.19),
    (2420681.0, 17.72),    # (cal_to_jd(1915.5), 17.72),
    (2420863.5, 18.19),    # (cal_to_jd(1916.0), 18.19),
    (2421046.5, 18.67),    # (cal_to_jd(1916.5), 18.67),
    (2421229.5, 19.13),    # (cal_to_jd(1917.0), 19.13),
    (2421412.0, 19.69),    # (cal_to_jd(1917.5), 19.69),
    (2421594.5, 20.14),    # (cal_to_jd(1918.0), 20.14),
    (2421777.0, 20.54),    # (cal_to_jd(1918.5), 20.54),
    (2421959.5, 20.86),    # (cal_to_jd(1919.0), 20.86),
    (2422142.0, 21.14),    # (cal_to_jd(1919.5), 21.14),
    (2422324.5, 21.41),    # (cal_to_jd(1920.0), 21.41),
    (2422507.5, 21.78),    # (cal_to_jd(1920.5), 21.78),
    (2422690.5, 22.06),    # (cal_to_jd(1921.0), 22.06),
    (2422873.0, 22.3),     # (cal_to_jd(1921.5), 22.30),
    (2423055.5, 22.51),    # (cal_to_jd(1922.0), 22.51),
    (2423238.0, 22.79),    # (cal_to_jd(1922.5), 22.79),
    (2423420.5, 23.01),    # (cal_to_jd(1923.0), 23.01),
    (2423603.0, 23.29),    # (cal_to_jd(1923.5), 23.29),
    (2423785.5, 23.46),    # (cal_to_jd(1924.0), 23.46),
    (2423968.5, 23.55),    # (cal_to_jd(1924.5), 23.55),
    (2424151.5, 23.63),    # (cal_to_jd(1925.0), 23.63),
    (2424334.0, 23.8),     # (cal_to_jd(1925.5), 23.80),
    (2424516.5, 23.95),    # (cal_to_jd(1926.0), 23.95),
    (2424699.0, 24.25),    # (cal_to_jd(1926.5), 24.25),
    (2424881.5, 24.39),    # (cal_to_jd(1927.0), 24.39),
    (2425064.0, 24.42),    # (cal_to_jd(1927.5), 24.42),
    (2425246.5, 24.34),    # (cal_to_jd(1928.0), 24.34),
    (2425429.5, 24.22),    # (cal_to_jd(1928.5), 24.22),
    (2425612.5, 24.1),     # (cal_to_jd(1929.0), 24.10),
    (2425795.0, 24.08),    # (cal_to_jd(1929.5), 24.08),
    (2425977.5, 24.02),    # (cal_to_jd(1930.0), 24.02),
    (2426160.0, 24.04),    # (cal_to_jd(1930.5), 24.04),
    (2426342.5, 23.98),    # (cal_to_jd(1931.0), 23.98),
    (2426525.0, 23.91),    # (cal_to_jd(1931.5), 23.91),
    (2426707.5, 23.89),    # (cal_to_jd(1932.0), 23.89),
    (2426890.5, 23.95),    # (cal_to_jd(1932.5), 23.95),
    (2427073.5, 23.93),    # (cal_to_jd(1933.0), 23.93),
    (2427256.0, 23.92),    # (cal_to_jd(1933.5), 23.92),
    (2427438.5, 23.88),    # (cal_to_jd(1934.0), 23.88),
    (2427621.0, 23.94),    # (cal_to_jd(1934.5), 23.94),
    (2427803.5, 23.91),    # (cal_to_jd(1935.0), 23.91),
    (2427986.0, 23.82),    # (cal_to_jd(1935.5), 23.82),
    (2428168.5, 23.76),    # (cal_to_jd(1936.0), 23.76),
    (2428351.5, 23.87),    # (cal_to_jd(1936.5), 23.87),
    (2428534.5, 23.91),    # (cal_to_jd(1937.0), 23.91),
    (2428717.0, 23.95),    # (cal_to_jd(1937.5), 23.95),
    (2428899.5, 23.96),    # (cal_to_jd(1938.0), 23.96),
    (2429082.0, 24.0),     # (cal_to_jd(1938.5), 24.),
    (2429264.5, 24.04),    # (cal_to_jd(1939.0), 24.04),
    (2429447.0, 24.2),     # (cal_to_jd(1939.5), 24.20),
    (2429629.5, 24.35),    # (cal_to_jd(1940.0), 24.35),
    (2429812.5, 24.61),    # (cal_to_jd(1940.5), 24.61),
    (2429995.5, 24.82),    # (cal_to_jd(1941.0), 24.82),
    (2430178.0, 25.09),    # (cal_to_jd(1941.5), 25.09),
    (2430360.5, 25.3),     # (cal_to_jd(1942.0), 25.30),
    (2430543.0, 25.56),    # (cal_to_jd(1942.5), 25.56),
    (2430725.5, 25.77),    # (cal_to_jd(1943.0), 25.77),
    (2430908.0, 26.05),    # (cal_to_jd(1943.5), 26.05),
    (2431090.5, 26.27),    # (cal_to_jd(1944.0), 26.27),
    (2431273.5, 26.54),    # (cal_to_jd(1944.5), 26.54),
    (2431456.5, 26.76),    # (cal_to_jd(1945.0), 26.76),
    (2431639.0, 27.04),    # (cal_to_jd(1945.5), 27.04),
    (2431821.5, 27.27),    # (cal_to_jd(1946.0), 27.27),
    (2432004.0, 27.55),    # (cal_to_jd(1946.5), 27.55),
    (2432186.5, 27.77),    # (cal_to_jd(1947.0), 27.77),
    (2432369.0, 28.03),    # (cal_to_jd(1947.5), 28.03),
    (2432551.5, 28.25),    # (cal_to_jd(1948.0), 28.25),
    (2432734.5, 28.5),     # (cal_to_jd(1948.5), 28.50),
    (2432917.5, 28.7),     # (cal_to_jd(1949.0), 28.70),
    (2433100.0, 28.95),    # (cal_to_jd(1949.5), 28.95),
    (2433282.5, 29.15),    # (cal_to_jd(1950.0), 29.15),
    (2433465.0, 29.38),    # (cal_to_jd(1950.5), 29.38),
    (2433647.5, 29.57),    # (cal_to_jd(1951.0), 29.57),
    (2433830.0, 29.8),     # (cal_to_jd(1951.5), 29.80),
    (2434012.5, 29.97),    # (cal_to_jd(1952.0), 29.97),
    (2434195.5, 30.19),    # (cal_to_jd(1952.5), 30.19),
    (2434378.5, 30.36),    # (cal_to_jd(1953.0), 30.36),
    (2434561.0, 30.57),    # (cal_to_jd(1953.5), 30.57),
    (2434743.5, 30.72),    # (cal_to_jd(1954.0), 30.72),
    (2434926.0, 30.93),    # (cal_to_jd(1954.5), 30.93),
    (2435108.5, 31.07),    # (cal_to_jd(1955.0), 31.07),
    (2435291.0, 31.24),    # (cal_to_jd(1955.5), 31.24),
    (2435473.5, 31.349),   # (cal_to_jd(1956.0), 31.349),
    (2435656.5, 31.516),   # (cal_to_jd(1956.5), 31.516),
    (2435839.5, 31.677),   # (cal_to_jd(1957.0), 31.677),
    (2436022.0, 31.923),   # (cal_to_jd(1957.5), 31.923),
    (2436204.5, 32.166),   # (cal_to_jd(1958.0), 32.166),
    (2436387.0, 32.449),   # (cal_to_jd(1958.5), 32.449),
    (2436569.5, 32.671),   # (cal_to_jd(1959.0), 32.671),
    (2436752.0, 32.919),   # (cal_to_jd(1959.5), 32.919),
    (2436934.5, 33.15),    # (cal_to_jd(1960.0), 33.150),
    (2437117.5, 33.397),   # (cal_to_jd(1960.5), 33.397),
    (2437300.5, 33.584),   # (cal_to_jd(1961.0), 33.584),
    (2437483.0, 33.804),   # (cal_to_jd(1961.5), 33.804),
    (2437665.5, 33.992),   # (cal_to_jd(1962.0), 33.992),
    (2437848.0, 34.24),    # (cal_to_jd(1962.5), 34.240),
    (2438030.5, 34.466),   # (cal_to_jd(1963.0), 34.466),
    (2438213.0, 34.731),   # (cal_to_jd(1963.5), 34.731),
    (2438395.5, 35.03),    # (cal_to_jd(1964.0), 35.030),
    (2438578.5, 35.4),     # (cal_to_jd(1964.5), 35.4),
    (2438761.5, 35.738),   # (cal_to_jd(1965.0), 35.738),
    (2438944.0, 36.147),   # (cal_to_jd(1965.5), 36.147),
    (2439126.5, 36.546),   # (cal_to_jd(1966.0), 36.546),
    (2439309.0, 36.995),   # (cal_to_jd(1966.5), 36.995),
    (2439491.5, 37.429),   # (cal_to_jd(1967.0), 37.429),
    (2439674.0, 37.879),   # (cal_to_jd(1967.5), 37.879),
    (2439856.5, 38.291),   # (cal_to_jd(1968.0), 38.291),
    (2440039.5, 38.753),   # (cal_to_jd(1968.5), 38.753),
    (2440222.5, 39.204),   # (cal_to_jd(1969.0), 39.204),
    (2440405.0, 39.707),   # (cal_to_jd(1969.5), 39.707),
    (2440587.5, 40.182),   # (cal_to_jd(1970.0), 40.182),
    (2440770.0, 40.706),   # (cal_to_jd(1970.5), 40.706),
    (2440952.5, 41.17),    # (cal_to_jd(1971.0), 41.170),
    (2441135.0, 41.686),   # (cal_to_jd(1971.5), 41.686),
    (2441317.5, 42.227),   # (cal_to_jd(1972.0), 42.227),
    (2441500.5, 42.825),   # (cal_to_jd(1972.5), 42.825),
    (2441683.5, 43.373),   # (cal_to_jd(1973.0), 43.373),
    (2441714.5, 43.4724),  # (cal_to_jd(1973, 2), 43.4724),
    (2441742.5, 43.5648),  # (cal_to_jd(1973, 3), 43.5648),
    (2441773.5, 43.6737),  # (cal_to_jd(1973, 4), 43.6737),
    (2441803.5, 43.7782),  # (cal_to_jd(1973, 5), 43.7782),
    (2441834.5, 43.8763),  # (cal_to_jd(1973, 6), 43.8763),
    (2441864.5, 43.9562),  # (cal_to_jd(1973, 7), 43.9562),
    (2441895.5, 44.0315),  # (cal_to_jd(1973, 8), 44.0315),
    (2441926.5, 44.1132),  # (cal_to_jd(1973, 9), 44.1132),
    (2441956.5, 44.1982),  # (cal_to_jd(1973, 10), 44.1982),
    (2441987.5, 44.2952),  # (cal_to_jd(1973, 11), 44.2952),
    (2442017.5, 44.3936),  # (cal_to_jd(1973, 12), 44.3936),
    (2442048.5, 44.4841),  # (cal_to_jd(1974, 1), 44.4841),
    (2442079.5, 44.5646),  # (cal_to_jd(1974, 2), 44.5646),
    (2442107.5, 44.6425),  # (cal_to_jd(1974, 3), 44.6425),
    (2442138.5, 44.7386),  # (cal_to_jd(1974, 4), 44.7386),
    (2442168.5, 44.837),   # (cal_to_jd(1974, 5), 44.8370),
    (2442199.5, 44.9302),  # (cal_to_jd(1974, 6), 44.9302),
    (2442229.5, 44.9986),  # (cal_to_jd(1974, 7), 44.9986),
    (2442260.5, 45.0584),  # (cal_to_jd(1974, 8), 45.0584),
    (2442291.5, 45.1284),  # (cal_to_jd(1974, 9), 45.1284),
    (2442321.5, 45.2064),  # (cal_to_jd(1974, 10), 45.2064),
    (2442352.5, 45.298),   # (cal_to_jd(1974, 11), 45.2980),
    (2442382.5, 45.3897),  # (cal_to_jd(1974, 12), 45.3897),
    (2442413.5, 45.4761),  # (cal_to_jd(1975, 1), 45.4761),
    (2442444.5, 45.5633),  # (cal_to_jd(1975, 2), 45.5633),
    (2442472.5, 45.645),   # (cal_to_jd(1975, 3), 45.6450),
    (2442503.5, 45.7375),  # (cal_to_jd(1975, 4), 45.7375),
    (2442533.5, 45.8284),  # (cal_to_jd(1975, 5), 45.8284),
    (2442564.5, 45.9133),  # (cal_to_jd(1975, 6), 45.9133),
    (2442594.5, 45.982),   # (cal_to_jd(1975, 7), 45.9820),
    (2442625.5, 46.0408),  # (cal_to_jd(1975, 8), 46.0408),
    (2442656.5, 46.1067),  # (cal_to_jd(1975, 9), 46.1067),
    (2442686.5, 46.1825),  # (cal_to_jd(1975, 10), 46.1825),
    (2442717.5, 46.2789),  # (cal_to_jd(1975, 11), 46.2789),
    (2442747.5, 46.3713),  # (cal_to_jd(1975, 12), 46.3713),
    (2442778.5, 46.4567),  # (cal_to_jd(1976, 1), 46.4567),
    (2442809.5, 46.5445),  # (cal_to_jd(1976, 2), 46.5445),
    (2442838.5, 46.6311),  # (cal_to_jd(1976, 3), 46.6311),
    (2442869.5, 46.7302),  # (cal_to_jd(1976, 4), 46.7302),
    (2442899.5, 46.8284),  # (cal_to_jd(1976, 5), 46.8284),
    (2442930.5, 46.9247),  # (cal_to_jd(1976, 6), 46.9247),
    (2442960.5, 46.997),   # (cal_to_jd(1976, 7), 46.9970),
    (2442991.5, 47.0709),  # (cal_to_jd(1976, 8), 47.0709),
    (2443022.5, 47.1451),  # (cal_to_jd(1976, 9), 47.1451),
    (2443052.5, 47.2362),  # (cal_to_jd(1976, 10), 47.2362),
    (2443083.5, 47.3413),  # (cal_to_jd(1976, 11), 47.3413),
    (2443113.5, 47.4319),  # (cal_to_jd(1976, 12), 47.4319),
    (2443144.5, 47.5214),  # (cal_to_jd(1977, 1), 47.5214),
    (2443175.5, 47.6049),  # (cal_to_jd(1977, 2), 47.6049),
    (2443203.5, 47.6837),  # (cal_to_jd(1977, 3), 47.6837),
    (2443234.5, 47.7781),  # (cal_to_jd(1977, 4), 47.7781),
    (2443264.5, 47.8771),  # (cal_to_jd(1977, 5), 47.8771),
    (2443295.5, 47.9687),  # (cal_to_jd(1977, 6), 47.9687),
    (2443325.5, 48.0348),  # (cal_to_jd(1977, 7), 48.0348),
    (2443356.5, 48.0942),  # (cal_to_jd(1977, 8), 48.0942),
    (2443387.5, 48.1608),  # (cal_to_jd(1977, 9), 48.1608),
    (2443417.5, 48.246),   # (cal_to_jd(1977, 10), 48.2460),
    (2443448.5, 48.3439),  # (cal_to_jd(1977, 11), 48.3439),
    (2443478.5, 48.4355),  # (cal_to_jd(1977, 12), 48.4355),
    (2443509.5, 48.5344),  # (cal_to_jd(1978, 1), 48.5344),
    (2443540.5, 48.6325),  # (cal_to_jd(1978, 2), 48.6325),
    (2443568.5, 48.7294),  # (cal_to_jd(1978, 3), 48.7294),
    (2443599.5, 48.8365),  # (cal_to_jd(1978, 4), 48.8365),
    (2443629.5, 48.9353),  # (cal_to_jd(1978, 5), 48.9353),
    (2443660.5, 49.0319),  # (cal_to_jd(1978, 6), 49.0319),
    (2443690.5, 49.1013),  # (cal_to_jd(1978, 7), 49.1013),
    (2443721.5, 49.1591),  # (cal_to_jd(1978, 8), 49.1591),
    (2443752.5, 49.2286),  # (cal_to_jd(1978, 9), 49.2286),
    (2443782.5, 49.307),   # (cal_to_jd(1978, 10), 49.3070),
    (2443813.5, 49.4018),  # (cal_to_jd(1978, 11), 49.4018),
    (2443843.5, 49.4945),  # (cal_to_jd(1978, 12), 49.4945),
    (2443874.5, 49.5862),  # (cal_to_jd(1979, 1), 49.5862),
    (2443905.5, 49.6805),  # (cal_to_jd(1979, 2), 49.6805),
    (2443933.5, 49.7602),  # (cal_to_jd(1979, 3), 49.7602),
    (2443964.5, 49.8556),  # (cal_to_jd(1979, 4), 49.8556),
    (2443994.5, 49.9489),  # (cal_to_jd(1979, 5), 49.9489),
    (2444025.5, 50.0347),  # (cal_to_jd(1979, 6), 50.0347),
    (2444055.5, 50.1019),  # (cal_to_jd(1979, 7), 50.1019),
    (2444086.5, 50.1622),  # (cal_to_jd(1979, 8), 50.1622),
    (2444117.5, 50.226),   # (cal_to_jd(1979, 9), 50.2260),
    (2444147.5, 50.2968),  # (cal_to_jd(1979, 10), 50.2968),
    (2444178.5, 50.3831),  # (cal_to_jd(1979, 11), 50.3831),
    (2444208.5, 50.4599),  # (cal_to_jd(1979, 12), 50.4599),
    (2444239.5, 50.5387),  # (cal_to_jd(1980, 1), 50.5387),
    (2444270.5, 50.6161),  # (cal_to_jd(1980, 2), 50.6161),
    (2444299.5, 50.6866),  # (cal_to_jd(1980, 3), 50.6866),
    (2444330.5, 50.7658),  # (cal_to_jd(1980, 4), 50.7658),
    (2444360.5, 50.8454),  # (cal_to_jd(1980, 5), 50.8454),
    (2444391.5, 50.9187),  # (cal_to_jd(1980, 6), 50.9187),
    (2444421.5, 50.9761),  # (cal_to_jd(1980, 7), 50.9761),
    (2444452.5, 51.0278),  # (cal_to_jd(1980, 8), 51.0278),
    (2444483.5, 51.0843),  # (cal_to_jd(1980, 9), 51.0843),
    (2444513.5, 51.1538),  # (cal_to_jd(1980, 10), 51.1538),
    (2444544.5, 51.2319),  # (cal_to_jd(1980, 11), 51.2319),
    (2444574.5, 51.3063),  # (cal_to_jd(1980, 12), 51.3063),
    (2444605.5, 51.3808),  # (cal_to_jd(1981, 1), 51.3808),
    (2444636.5, 51.4526),  # (cal_to_jd(1981, 2), 51.4526),
    (2444664.5, 51.516),   # (cal_to_jd(1981, 3), 51.5160),
    (2444695.5, 51.5985),  # (cal_to_jd(1981, 4), 51.5985),
    (2444725.5, 51.6809),  # (cal_to_jd(1981, 5), 51.6809),
    (2444756.5, 51.7573),  # (cal_to_jd(1981, 6), 51.7573),
    (2444786.5, 51.8133),  # (cal_to_jd(1981, 7), 51.8133),
    (2444817.5, 51.8532),  # (cal_to_jd(1981, 8), 51.8532),
    (2444848.5, 51.9014),  # (cal_to_jd(1981, 9), 51.9014),
    (2444878.5, 51.9603),  # (cal_to_jd(1981, 10), 51.9603),
    (2444909.5, 52.0328),  # (cal_to_jd(1981, 11), 52.0328),
    (2444939.5, 52.0985),  # (cal_to_jd(1981, 12), 52.0985),
    (2444970.5, 52.1668),  # (cal_to_jd(1982, 1), 52.1668),
    (2445001.5, 52.2316),  # (cal_to_jd(1982, 2), 52.2316),
    (2445029.5, 52.2938),  # (cal_to_jd(1982, 3), 52.2938),
    (2445060.5, 52.368),   # (cal_to_jd(1982, 4), 52.3680),
    (2445090.5, 52.4465),  # (cal_to_jd(1982, 5), 52.4465),
    (2445121.5, 52.518),   # (cal_to_jd(1982, 6), 52.5180),
    (2445151.5, 52.5752),  # (cal_to_jd(1982, 7), 52.5752),
    (2445182.5, 52.6178),  # (cal_to_jd(1982, 8), 52.6178),
    (2445213.5, 52.6668),  # (cal_to_jd(1982, 9), 52.6668),
    (2445243.5, 52.734),   # (cal_to_jd(1982, 10), 52.7340),
    (2445274.5, 52.8056),  # (cal_to_jd(1982, 11), 52.8056),
    (2445304.5, 52.8792),  # (cal_to_jd(1982, 12), 52.8792),
    (2445335.5, 52.9565),  # (cal_to_jd(1983, 1), 52.9565),
    (2445366.5, 53.0445),  # (cal_to_jd(1983, 2), 53.0445),
    (2445394.5, 53.1268),  # (cal_to_jd(1983, 3), 53.1268),
    (2445425.5, 53.2197),  # (cal_to_jd(1983, 4), 53.2197),
    (2445455.5, 53.3024),  # (cal_to_jd(1983, 5), 53.3024),
    (2445486.5, 53.3747),  # (cal_to_jd(1983, 6), 53.3747),
    (2445516.5, 53.4335),  # (cal_to_jd(1983, 7), 53.4335),
    (2445547.5, 53.4778),  # (cal_to_jd(1983, 8), 53.4778),
    (2445578.5, 53.53),    # (cal_to_jd(1983, 9), 53.5300),
    (2445608.5, 53.5845),  # (cal_to_jd(1983, 10), 53.5845),
    (2445639.5, 53.6523),  # (cal_to_jd(1983, 11), 53.6523),
    (2445669.5, 53.7256),  # (cal_to_jd(1983, 12), 53.7256),
    (2445700.5, 53.7882),  # (cal_to_jd(1984, 1), 53.7882),
    (2445731.5, 53.8367),  # (cal_to_jd(1984, 2), 53.8367),
    (2445760.5, 53.883),   # (cal_to_jd(1984, 3), 53.8830),
    (2445791.5, 53.9443),  # (cal_to_jd(1984, 4), 53.9443),
    (2445821.5, 54.0042),  # (cal_to_jd(1984, 5), 54.0042),
    (2445852.5, 54.0536),  # (cal_to_jd(1984, 6), 54.0536),
    (2445882.5, 54.0856),  # (cal_to_jd(1984, 7), 54.0856),
    (2445913.5, 54.1084),  # (cal_to_jd(1984, 8), 54.1084),
    (2445944.5, 54.1463),  # (cal_to_jd(1984, 9), 54.1463),
    (2445974.5, 54.1914),  # (cal_to_jd(1984, 10), 54.1914),
    (2446005.5, 54.2452),  # (cal_to_jd(1984, 11), 54.2452),
    (2446035.5, 54.2958),  # (cal_to_jd(1984, 12), 54.2958),
    (2446066.5, 54.3427),  # (cal_to_jd(1985, 1), 54.3427),
    (2446097.5, 54.3911),  # (cal_to_jd(1985, 2), 54.3911),
    (2446125.5, 54.432),   # (cal_to_jd(1985, 3), 54.4320),
    (2446156.5, 54.4898),  # (cal_to_jd(1985, 4), 54.4898),
    (2446186.5, 54.5456),  # (cal_to_jd(1985, 5), 54.5456),
    (2446217.5, 54.5977),  # (cal_to_jd(1985, 6), 54.5977),
    (2446247.5, 54.6355),  # (cal_to_jd(1985, 7), 54.6355),
    (2446278.5, 54.6532),  # (cal_to_jd(1985, 8), 54.6532),
    (2446309.5, 54.6776),  # (cal_to_jd(1985, 9), 54.6776),
    (2446339.5, 54.7174),  # (cal_to_jd(1985, 10), 54.7174),
    (2446370.5, 54.7741),  # (cal_to_jd(1985, 11), 54.7741),
    (2446400.5, 54.8253),  # (cal_to_jd(1985, 12), 54.8253),
    (2446431.5, 54.8713),  # (cal_to_jd(1986, 1), 54.8713),
    (2446462.5, 54.9161),  # (cal_to_jd(1986, 2), 54.9161),
    (2446490.5, 54.9581),  # (cal_to_jd(1986, 3), 54.9581),
    (2446521.5, 54.9997),  # (cal_to_jd(1986, 4), 54.9997),
    (2446551.5, 55.0476),  # (cal_to_jd(1986, 5), 55.0476),
    (2446582.5, 55.0912),  # (cal_to_jd(1986, 6), 55.0912),
    (2446612.5, 55.1132),  # (cal_to_jd(1986, 7), 55.1132),
    (2446643.5, 55.1328),  # (cal_to_jd(1986, 8), 55.1328),
    (2446674.5, 55.1532),  # (cal_to_jd(1986, 9), 55.1532),
    (2446704.5, 55.1898),  # (cal_to_jd(1986, 10), 55.1898),
    (2446735.5, 55.2416),  # (cal_to_jd(1986, 11), 55.2416),
    (2446765.5, 55.2838),  # (cal_to_jd(1986, 12), 55.2838),
    (2446796.5, 55.3222),  # (cal_to_jd(1987, 1), 55.3222),
    (2446827.5, 55.3613),  # (cal_to_jd(1987, 2), 55.3613),
    (2446855.5, 55.4063),  # (cal_to_jd(1987, 3), 55.4063),
    (2446886.5, 55.4629),  # (cal_to_jd(1987, 4), 55.4629),
    (2446916.5, 55.5111),  # (cal_to_jd(1987, 5), 55.5111),
    (2446947.5, 55.5524),  # (cal_to_jd(1987, 6), 55.5524),
    (2446977.5, 55.5812),  # (cal_to_jd(1987, 7), 55.5812),
    (2447008.5, 55.6004),  # (cal_to_jd(1987, 8), 55.6004),
    (2447039.5, 55.6262),  # (cal_to_jd(1987, 9), 55.6262),
    (2447069.5, 55.6656),  # (cal_to_jd(1987, 10), 55.6656),
    (2447100.5, 55.7168),  # (cal_to_jd(1987, 11), 55.7168),
    (2447130.5, 55.7698),  # (cal_to_jd(1987, 12), 55.7698),
    (2447161.5, 55.8197),  # (cal_to_jd(1988, 1), 55.8197),
    (2447192.5, 55.8615),  # (cal_to_jd(1988, 2), 55.8615),
    (2447221.5, 55.913),   # (cal_to_jd(1988, 3), 55.9130),
    (2447252.5, 55.9663),  # (cal_to_jd(1988, 4), 55.9663),
    (2447282.5, 56.022),   # (cal_to_jd(1988, 5), 56.0220),
    (2447313.5, 56.07),    # (cal_to_jd(1988, 6), 56.0700),
    (2447343.5, 56.0939),  # (cal_to_jd(1988, 7), 56.0939),
    (2447374.5, 56.1105),  # (cal_to_jd(1988, 8), 56.1105),
    (2447405.5, 56.1314),  # (cal_to_jd(1988, 9), 56.1314),
    (2447435.5, 56.1611),  # (cal_to_jd(1988, 10), 56.1611),
    (2447466.5, 56.2068),  # (cal_to_jd(1988, 11), 56.2068),
    (2447496.5, 56.2583),  # (cal_to_jd(1988, 12), 56.2583),
    (2447527.5, 56.3),     # (cal_to_jd(1989, 1), 56.3000),
    (2447558.5, 56.3399),  # (cal_to_jd(1989, 2), 56.3399),
    (2447586.5, 56.379),   # (cal_to_jd(1989, 3), 56.3790),
    (2447617.5, 56.4283),  # (cal_to_jd(1989, 4), 56.4283),
    (2447647.5, 56.4804),  # (cal_to_jd(1989, 5), 56.4804),
    (2447678.5, 56.5352),  # (cal_to_jd(1989, 6), 56.5352),
    (2447708.5, 56.5697),  # (cal_to_jd(1989, 7), 56.5697),
    (2447739.5, 56.5983),  # (cal_to_jd(1989, 8), 56.5983),
    (2447770.5, 56.6328),  # (cal_to_jd(1989, 9), 56.6328),
    (2447800.5, 56.6739),  # (cal_to_jd(1989, 10), 56.6739),
    (2447831.5, 56.7332),  # (cal_to_jd(1989, 11), 56.7332),
    (2447861.5, 56.7972),  # (cal_to_jd(1989, 12), 56.7972),
    (2447892.5, 56.8553),  # (cal_to_jd(1990, 1), 56.8553),
    (2447923.5, 56.9111),  # (cal_to_jd(1990, 2), 56.9111),
    (2447951.5, 56.9755),  # (cal_to_jd(1990, 3), 56.9755),
    (2447982.5, 57.0471),  # (cal_to_jd(1990, 4), 57.0471),
    (2448012.5, 57.1136),  # (cal_to_jd(1990, 5), 57.1136),
    (2448043.5, 57.1738),  # (cal_to_jd(1990, 6), 57.1738),
    (2448073.5, 57.2226),  # (cal_to_jd(1990, 7), 57.2226),
    (2448104.5, 57.2597),  # (cal_to_jd(1990, 8), 57.2597),
    (2448135.5, 57.3073),  # (cal_to_jd(1990, 9), 57.3073),
    (2448165.5, 57.3643),  # (cal_to_jd(1990, 10), 57.3643),
    (2448196.5, 57.4334),  # (cal_to_jd(1990, 11), 57.4334),
    (2448226.5, 57.5016),  # (cal_to_jd(1990, 12), 57.5016),
    (2448257.5, 57.5653),  # (cal_to_jd(1991, 1), 57.5653),
    (2448288.5, 57.6333),  # (cal_to_jd(1991, 2), 57.6333),
    (2448316.5, 57.6973),  # (cal_to_jd(1991, 3), 57.6973),
    (2448347.5, 57.7711),  # (cal_to_jd(1991, 4), 57.7711),
    (2448377.5, 57.8407),  # (cal_to_jd(1991, 5), 57.8407),
    (2448408.5, 57.9058),  # (cal_to_jd(1991, 6), 57.9058),
    (2448438.5, 57.9576),  # (cal_to_jd(1991, 7), 57.9576),
    (2448469.5, 57.9975),  # (cal_to_jd(1991, 8), 57.9975),
    (2448500.5, 58.0426),  # (cal_to_jd(1991, 9), 58.0426),
    (2448530.5, 58.1043),  # (cal_to_jd(1991, 10), 58.1043),
    (2448561.5, 58.1679),  # (cal_to_jd(1991, 11), 58.1679),
    (2448591.5, 58.2389),  # (cal_to_jd(1991, 12), 58.2389),
    (2448622.5, 58.3092),  # (cal_to_jd(1992, 1), 58.3092),
    (2448653.5, 58.3833),  # (cal_to_jd(1992, 2), 58.3833),
    (2448682.5, 58.4537),  # (cal_to_jd(1992, 3), 58.4537),
    (2448713.5, 58.5401),  # (cal_to_jd(1992, 4), 58.5401),
    (2448743.5, 58.6228),  # (cal_to_jd(1992, 5), 58.6228),
    (2448774.5, 58.6917),  # (cal_to_jd(1992, 6), 58.6917),
    (2448804.5, 58.741),   # (cal_to_jd(1992, 7), 58.7410),
    (2448835.5, 58.7836),  # (cal_to_jd(1992, 8), 58.7836),
    (2448866.5, 58.8406),  # (cal_to_jd(1992, 9), 58.8406),
    (2448896.5, 58.8986),  # (cal_to_jd(1992, 10), 58.8986),
    (2448927.5, 58.9714),  # (cal_to_jd(1992, 11), 58.9714),
    (2448957.5, 59.0438),  # (cal_to_jd(1992, 12), 59.0438),
    (2448988.5, 59.1218),  # (cal_to_jd(1993, 1), 59.1218),
    (2449019.5, 59.2003),  # (cal_to_jd(1993, 2), 59.2003),
    (2449047.5, 59.2747),  # (cal_to_jd(1993, 3), 59.2747),
    (2449078.5, 59.3574),  # (cal_to_jd(1993, 4), 59.3574),
    (2449108.5, 59.4434),  # (cal_to_jd(1993, 5), 59.4434),
    (2449139.5, 59.5242),  # (cal_to_jd(1993, 6), 59.5242),
    (2449169.5, 59.585),   # (cal_to_jd(1993, 7), 59.5850),
    (2449200.5, 59.6344),  # (cal_to_jd(1993, 8), 59.6344),
    (2449231.5, 59.6928),  # (cal_to_jd(1993, 9), 59.6928),
    (2449261.5, 59.7588),  # (cal_to_jd(1993, 10), 59.7588),
    (2449292.5, 59.8386),  # (cal_to_jd(1993, 11), 59.8386),
    (2449322.5, 59.9111),  # (cal_to_jd(1993, 12), 59.9111),
    (2449353.5, 59.9845),  # (cal_to_jd(1994, 1), 59.9845),
    (2449384.5, 60.0564),  # (cal_to_jd(1994, 2), 60.0564),
    (2449412.5, 60.1231),  # (cal_to_jd(1994, 3), 60.1231),
    (2449443.5, 60.2042),  # (cal_to_jd(1994, 4), 60.2042),
    (2449473.5, 60.2804),  # (cal_to_jd(1994, 5), 60.2804),
    (2449504.5, 60.353),   # (cal_to_jd(1994, 6), 60.3530),
    (2449534.5, 60.4012),  # (cal_to_jd(1994, 7), 60.4012),
    (2449565.5, 60.444),   # (cal_to_jd(1994, 8), 60.4440),
    (2449596.5, 60.49),    # (cal_to_jd(1994, 9), 60.4900),
    (2449626.5, 60.5578),  # (cal_to_jd(1994, 10), 60.5578),
    (2449657.5, 60.6324),  # (cal_to_jd(1994, 11), 60.6324),
    (2449687.5, 60.7059),  # (cal_to_jd(1994, 12), 60.7059),
    (2449718.5, 60.7853),  # (cal_to_jd(1995, 1), 60.7853),
    (2449749.5, 60.8664),  # (cal_to_jd(1995, 2), 60.8664),
    (2449777.5, 60.9387),  # (cal_to_jd(1995, 3), 60.9387),
    (2449808.5, 61.0277),  # (cal_to_jd(1995, 4), 61.0277),
    (2449838.5, 61.1103),  # (cal_to_jd(1995, 5), 61.1103),
    (2449869.5, 61.187),   # (cal_to_jd(1995, 6), 61.1870),
    (2449899.5, 61.2454),  # (cal_to_jd(1995, 7), 61.2454),
    (2449930.5, 61.2881),  # (cal_to_jd(1995, 8), 61.2881),
    (2449961.5, 61.3378),  # (cal_to_jd(1995, 9), 61.3378),
    (2449991.5, 61.4036),  # (cal_to_jd(1995, 10), 61.4036),
    (2450022.5, 61.476),   # (cal_to_jd(1995, 11), 61.4760),
    (2450052.5, 61.5525),  # (cal_to_jd(1995, 12), 61.5525),
    (2450083.5, 61.6287),  # (cal_to_jd(1996, 1), 61.6287),
    (2450114.5, 61.6846),  # (cal_to_jd(1996, 2), 61.6846),
    (2450143.5, 61.7433),  # (cal_to_jd(1996, 3), 61.7433),
    (2450174.5, 61.8132),  # (cal_to_jd(1996, 4), 61.8132),
    (2450204.5, 61.8823),  # (cal_to_jd(1996, 5), 61.8823),
    (2450235.5, 61.9497),  # (cal_to_jd(1996, 6), 61.9497),
    (2450265.5, 61.9969),  # (cal_to_jd(1996, 7), 61.9969),
    (2450296.5, 62.0343),  # (cal_to_jd(1996, 8), 62.0343),
    (2450327.5, 62.0714),  # (cal_to_jd(1996, 9), 62.0714),
    (2450357.5, 62.1202),  # (cal_to_jd(1996, 10), 62.1202),
    (2450388.5, 62.181),   # (cal_to_jd(1996, 11), 62.1810),
    (2450418.5, 62.2382),  # (cal_to_jd(1996, 12), 62.2382),
    (2450449.5, 62.295),   # (cal_to_jd(1997, 1), 62.2950),
    (2450480.5, 62.3506),  # (cal_to_jd(1997, 2), 62.3506),
    (2450508.5, 62.3995),  # (cal_to_jd(1997, 3), 62.3995),
    (2450539.5, 62.4754),  # (cal_to_jd(1997, 4), 62.4754),
    (2450569.5, 62.5463),  # (cal_to_jd(1997, 5), 62.5463),
    (2450600.5, 62.6136),  # (cal_to_jd(1997, 6), 62.6136),
    (2450630.5, 62.6571),  # (cal_to_jd(1997, 7), 62.6571),
    (2450661.5, 62.6942),  # (cal_to_jd(1997, 8), 62.6942),
    (2450692.5, 62.7383),  # (cal_to_jd(1997, 9), 62.7383),
    (2450722.5, 62.7926),  # (cal_to_jd(1997, 10), 62.7926),
    (2450753.5, 62.8567),  # (cal_to_jd(1997, 11), 62.8567),
    (2450783.5, 62.9146),  # (cal_to_jd(1997, 12), 62.9146),
    (2450814.5, 62.9659),  # (cal_to_jd(1998, 1), 62.9659),
    (2450845.5, 63.0217),  # (cal_to_jd(1998, 2), 63.0217),
    (2450873.5, 63.0807),  # (cal_to_jd(1998, 3), 63.0807),
    (2450904.5, 63.1462),  # (cal_to_jd(1998, 4), 63.1462),
    (2450934.5, 63.2053),  # (cal_to_jd(1998, 5), 63.2053),
    (2450965.5, 63.2599),  # (cal_to_jd(1998, 6), 63.2599),
    (2450995.5, 63.2844),  # (cal_to_jd(1998, 7), 63.2844),
    (2451026.5, 63.2961),  # (cal_to_jd(1998, 8), 63.2961),
    (2451057.5, 63.3126),  # (cal_to_jd(1998, 9), 63.3126),
    (2451087.5, 63.3422),  # (cal_to_jd(1998, 10), 63.3422),
    (2451118.5, 63.3871),  # (cal_to_jd(1998, 11), 63.3871),
    (2451148.5, 63.4339),  # (cal_to_jd(1998, 12), 63.4339),
    (2451179.5, 63.4673),  # (cal_to_jd(1999, 1), 63.4673),
    (2451210.5, 63.4979),  # (cal_to_jd(1999, 2), 63.4979),
    (2451238.5, 63.5319),  # (cal_to_jd(1999, 3), 63.5319),
    (2451269.5, 63.5679),  # (cal_to_jd(1999, 4), 63.5679),
    (2451299.5, 63.6104),  # (cal_to_jd(1999, 5), 63.6104),
    (2451330.5, 63.6444),  # (cal_to_jd(1999, 6), 63.6444),
    (2451360.5, 63.6642),  # (cal_to_jd(1999, 7), 63.6642),
    (2451391.5, 63.6739),  # (cal_to_jd(1999, 8), 63.6739),
    (2451422.5, 63.6926),  # (cal_to_jd(1999, 9), 63.6926),
    (2451452.5, 63.7147),  # (cal_to_jd(1999, 10), 63.7147),
    (2451483.5, 63.7518),  # (cal_to_jd(1999, 11), 63.7518),
    (2451513.5, 63.7927),  # (cal_to_jd(1999, 12), 63.7927),
    (2451544.5, 63.8285),  # (cal_to_jd(2000, 1), 63.8285),
    (2451575.5, 63.8557),  # (cal_to_jd(2000, 2), 63.8557),
    (2451604.5, 63.8804),  # (cal_to_jd(2000, 3), 63.8804),
    (2451635.5, 63.9075),  # (cal_to_jd(2000, 4), 63.9075),
    (2451665.5, 63.9393),  # (cal_to_jd(2000, 5), 63.9393),
    (2451696.5, 63.9691),  # (cal_to_jd(2000, 6), 63.9691),
    (2451726.5, 63.9799),  # (cal_to_jd(2000, 7), 63.9799),
    (2451757.5, 63.9833),  # (cal_to_jd(2000, 8), 63.9833),
    (2451788.5, 63.9938),  # (cal_to_jd(2000, 9), 63.9938),
    (2451818.5, 64.0093),  # (cal_to_jd(2000, 10), 64.0093),
    (2451849.5, 64.04),    # (cal_to_jd(2000, 11), 64.0400),
    (2451879.5, 64.067),   # (cal_to_jd(2000, 12), 64.0670),
    (2451910.5, 64.0908),  # (cal_to_jd(2001, 1), 64.0908),
    (2451941.5, 64.1068),  # (cal_to_jd(2001, 2), 64.1068),
    (2451969.5, 64.1282),  # (cal_to_jd(2001, 3), 64.1282),
    (2452000.5, 64.1584),  # (cal_to_jd(2001, 4), 64.1584),
    (2452030.5, 64.1833),  # (cal_to_jd(2001, 5), 64.1833),
    (2452061.5, 64.2094),  # (cal_to_jd(2001, 6), 64.2094),
    (2452091.5, 64.2117),  # (cal_to_jd(2001, 7), 64.2117),
    (2452122.5, 64.2073),  # (cal_to_jd(2001, 8), 64.2073),
    (2452153.5, 64.2116),  # (cal_to_jd(2001, 9), 64.2116),
    (2452183.5, 64.2223),  # (cal_to_jd(2001, 10), 64.2223),
    (2452214.5, 64.25),    # (cal_to_jd(2001, 11), 64.2500),
    (2452244.5, 64.2761),  # (cal_to_jd(2001, 12), 64.2761),
    (2452275.5, 64.2998),  # (cal_to_jd(2002, 1), 64.2998),
    (2452306.5, 64.3192),  # (cal_to_jd(2002, 2), 64.3192),
    (2452334.5, 64.345),   # (cal_to_jd(2002, 3), 64.3450),
    (2452365.5, 64.3735),  # (cal_to_jd(2002, 4), 64.3735),
    (2452395.5, 64.3943),  # (cal_to_jd(2002, 5), 64.3943),
    (2452426.5, 64.4151),  # (cal_to_jd(2002, 6), 64.4151),
    (2452456.5, 64.4132),  # (cal_to_jd(2002, 7), 64.4132),
    (2452487.5, 64.4118),  # (cal_to_jd(2002, 8), 64.4118),
    (2452518.5, 64.4097),  # (cal_to_jd(2002, 9), 64.4097),
    (2452548.5, 64.4168),  # (cal_to_jd(2002, 10), 64.4168),
    (2452579.5, 64.4329),  # (cal_to_jd(2002, 11), 64.4329),
    (2452609.5, 64.4511),  # (cal_to_jd(2002, 12), 64.4511),
    (2452640.5, 64.4734),  # (cal_to_jd(2003, 1), 64.4734),
    (2452671.5, 64.4893),  # (cal_to_jd(2003, 2), 64.4893),
    (2452699.5, 64.5053),  # (cal_to_jd(2003, 3), 64.5053),
    (2452730.5, 64.5269),  # (cal_to_jd(2003, 4), 64.5269),
    (2452760.5, 64.5471),  # (cal_to_jd(2003, 5), 64.5471),
    (2452791.5, 64.5597),  # (cal_to_jd(2003, 6), 64.5597),
    (2452821.5, 64.5512),  # (cal_to_jd(2003, 7), 64.5512),
    (2452852.5, 64.5371),  # (cal_to_jd(2003, 8), 64.5371),
    (2452883.5, 64.5359),  # (cal_to_jd(2003, 9), 64.5359),
    (2452913.5, 64.5415),  # (cal_to_jd(2003, 10), 64.5415),
    (2452944.5, 64.5544),  # (cal_to_jd(2003, 11), 64.5544),
    (2452974.5, 64.5654),  # (cal_to_jd(2003, 12), 64.5654),
    (2453005.5, 64.5736),  # (cal_to_jd(2004, 1), 64.5736),
    (2453036.5, 64.5891),  # (cal_to_jd(2004, 2), 64.5891),
    (2453065.5, 64.6015),  # (cal_to_jd(2004, 3), 64.6015),
    (2453096.5, 64.6176),  # (cal_to_jd(2004, 4), 64.6176),
    (2453126.5, 64.6374),  # (cal_to_jd(2004, 5), 64.6374),
    (2453157.5, 64.6549),  # (cal_to_jd(2004, 6), 64.6549),
    (2453187.5, 64.653),   # (cal_to_jd(2004, 7), 64.6530),
    (2453218.5, 64.6379),  # (cal_to_jd(2004, 8), 64.6379),
    (2453249.5, 64.6372),  # (cal_to_jd(2004, 9), 64.6372),
    (2453279.5, 64.64),    # (cal_to_jd(2004, 10), 64.6400),
    (2453310.5, 64.6543),  # (cal_to_jd(2004, 11), 64.6543),
    (2453340.5, 64.6723),  # (cal_to_jd(2004, 12), 64.6723),
    (2453371.5, 64.6876),  # (cal_to_jd(2005, 1), 64.6876),
    (2453402.5, 64.7052),  # (cal_to_jd(2005, 2), 64.7052),
    (2453430.5, 64.7313),  # (cal_to_jd(2005, 3), 64.7313),
    (2453461.5, 64.7575),  # (cal_to_jd(2005, 4), 64.7575),
    (2453491.5, 64.7811),  # (cal_to_jd(2005, 5), 64.7811),
    (2453522.5, 64.8001),  # (cal_to_jd(2005, 6), 64.8001),
    (2453552.5, 64.7995),  # (cal_to_jd(2005, 7), 64.7995),
    (2453583.5, 64.7876),  # (cal_to_jd(2005, 8), 64.7876),
    (2453614.5, 64.7831),  # (cal_to_jd(2005, 9), 64.7831),
    (2453644.5, 64.7921),  # (cal_to_jd(2005, 10), 64.7921),
    (2453675.5, 64.8096),  # (cal_to_jd(2005, 11), 64.8096),
    (2453705.5, 64.8311),  # (cal_to_jd(2005, 12), 64.8311),
    (2453736.5, 64.8452),  # (cal_to_jd(2006, 1), 64.8452),
    (2453767.5, 64.8597),  # (cal_to_jd(2006, 2), 64.8597),
    (2453795.5, 64.885),   # (cal_to_jd(2006, 3), 64.8850),
    (2453826.5, 64.9175),  # (cal_to_jd(2006, 4), 64.9175),
    (2453856.5, 64.948),   # (cal_to_jd(2006, 5), 64.9480),
    (2453887.5, 64.9794),  # (cal_to_jd(2006, 6), 64.9794),
    (2453917.5, 64.9895),  # (cal_to_jd(2006, 7), 64.9895),
    (2453948.5, 65.0028),  # (cal_to_jd(2006, 8), 65.0028),
    (2453979.5, 65.0138),  # (cal_to_jd(2006, 9), 65.0138),
    (2454009.5, 65.0371),  # (cal_to_jd(2006, 10), 65.0371),
    (2454040.5, 65.0773),  # (cal_to_jd(2006, 11), 65.0773),
    (2454070.5, 65.1122),  # (cal_to_jd(2006, 12), 65.1122),
    (2454101.5, 65.1464),  # (cal_to_jd(2007, 1), 65.1464),
    (2454132.5, 65.1833),  # (cal_to_jd(2007, 2), 65.1833),
    (2454160.5, 65.2145),  # (cal_to_jd(2007, 3), 65.2145),
    (2454191.5, 65.2494),  # (cal_to_jd(2007, 4), 65.2494),
    (2454221.5, 65.2921),  # (cal_to_jd(2007, 5), 65.2921),
    (2454252.5, 65.3279),  # (cal_to_jd(2007, 6), 65.3279),
    (2454282.5, 65.3413),  # (cal_to_jd(2007, 7), 65.3413),
    (2454313.5, 65.3452),  # (cal_to_jd(2007, 8), 65.3452),
    (2454344.5, 65.3496),  # (cal_to_jd(2007, 9), 65.3496),
    (2454374.5, 65.3711),  # (cal_to_jd(2007, 10), 65.3711),
    (2454405.5, 65.3972),  # (cal_to_jd(2007, 11), 65.3972),
    (2454435.5, 65.4296),  # (cal_to_jd(2007, 12), 65.4296),
    (2454466.5, 65.4573),  # (cal_to_jd(2008, 1), 65.4573),
    (2454497.5, 65.4868),  # (cal_to_jd(2008, 2), 65.4868),
    (2454526.5, 65.5152),  # (cal_to_jd(2008, 3), 65.5152),
    (2454557.5, 65.545),   # (cal_to_jd(2008, 4), 65.5450),
    (2454587.5, 65.5781),  # (cal_to_jd(2008, 5), 65.5781),
    (2454618.5, 65.6127),  # (cal_to_jd(2008, 6), 65.6127),
    (2454648.5, 65.6288),  # (cal_to_jd(2008, 7), 65.6288),
    (2454679.5, 65.637),   # (cal_to_jd(2008, 8), 65.6370),
    (2454710.5, 65.6493),  # (cal_to_jd(2008, 9), 65.6493),
    (2454740.5, 65.676),   # (cal_to_jd(2008, 10), 65.6760),
    (2454771.5, 65.7097),  # (cal_to_jd(2008, 11), 65.7097),
    (2454801.5, 65.7461),  # (cal_to_jd(2008, 12), 65.7461),
    (2454832.5, 65.7768),  # (cal_to_jd(2009, 1), 65.7768),
    (2454863.5, 65.8025),  # (cal_to_jd(2009, 2), 65.8025),
    (2454891.5, 65.8237),  # (cal_to_jd(2009, 3), 65.8237),
    (2454922.5, 65.8595),  # (cal_to_jd(2009, 4), 65.8595),
    (2454952.5, 65.8973),  # (cal_to_jd(2009, 5), 65.8973),
    (2454983.5, 65.9323),  # (cal_to_jd(2009, 6), 65.9323),
    (2455013.5, 65.9509),  # (cal_to_jd(2009, 7), 65.9509),
    (2455044.5, 65.9534),  # (cal_to_jd(2009, 8), 65.9534),
    (2455075.5, 65.9628),  # (cal_to_jd(2009, 9), 65.9628),
    (2455105.5, 65.9839),  # (cal_to_jd(2009, 10), 65.9839),
    (2455136.5, 66.0147),  # (cal_to_jd(2009, 11), 66.0147),
    (2455166.5, 66.042),   # (cal_to_jd(2009, 12), 66.0420),
    (2455197.5, 66.0699),  # (cal_to_jd(2010, 1), 66.0699),
    (2455228.5, 66.0961),  # (cal_to_jd(2010, 2), 66.0961),
    (2455256.5, 66.131),   # (cal_to_jd(2010, 3), 66.1310),
    (2455287.5, 66.1683),  # (cal_to_jd(2010, 4), 66.1683),
    (2455317.5, 66.2072),  # (cal_to_jd(2010, 5), 66.2072),
    (2455348.5, 66.2356),  # (cal_to_jd(2010, 6), 66.2356),
    (2455378.5, 66.2409),  # (cal_to_jd(2010, 7), 66.2409),
    (2455409.5, 66.2335),  # (cal_to_jd(2010, 8), 66.2335),
    (2455440.5, 66.2349),  # (cal_to_jd(2010, 9), 66.2349),
    (2455470.5, 66.2441),  # (cal_to_jd(2010, 10), 66.2441),
    (2455501.5, 66.2751),  # (cal_to_jd(2010, 11), 66.2751),
    (2455531.5, 66.3054),  # (cal_to_jd(2010, 12), 66.3054),
    (2455562.5, 66.3246),  # (cal_to_jd(2011, 1), 66.3246),
    (2455593.5, 66.3406),  # (cal_to_jd(2011, 2), 66.3406),
    (2455621.5, 66.3624),  # (cal_to_jd(2011, 3), 66.3624),
    (2455652.5, 66.3957),  # (cal_to_jd(2011, 4), 66.3957),
    (2455682.5, 66.4289),  # (cal_to_jd(2011, 5), 66.4289),
    (2455713.5, 66.4619),  # (cal_to_jd(2011, 6), 66.4619),
    (2455743.5, 66.4749),  # (cal_to_jd(2011, 7), 66.4749),
    (2455774.5, 66.4751),  # (cal_to_jd(2011, 8), 66.4751),
    (2455805.5, 66.4829),  # (cal_to_jd(2011, 9), 66.4829),
    (2455835.5, 66.5056),  # (cal_to_jd(2011, 10), 66.5056),
    (2455866.5, 66.5383),  # (cal_to_jd(2011, 11), 66.5383),
    (2455896.5, 66.5706),  # (cal_to_jd(2011, 12), 66.5706),
    (2455927.5, 66.603),   # (cal_to_jd(2012, 1), 66.6030),
    (2455958.5, 66.634),   # (cal_to_jd(2012, 2), 66.6340),
    (2455987.5, 66.6569),  # (cal_to_jd(2012, 3), 66.6569),
    (2456018.5, 66.6925),  # (cal_to_jd(2012, 4), 66.6925),
    (2456048.5, 66.7289),  # (cal_to_jd(2012, 5), 66.7289),
    (2456079.5, 66.7579),  # (cal_to_jd(2012, 6), 66.7579),
    (2456109.5, 66.7708),  # (cal_to_jd(2012, 7), 66.7708),
    (2456140.5, 66.774),   # (cal_to_jd(2012, 8), 66.7740),
    (2456171.5, 66.7846),  # (cal_to_jd(2012, 9), 66.7846),
    (2456201.5, 66.8103),  # (cal_to_jd(2012, 10), 66.8103),
    (2456232.5, 66.84),    # (cal_to_jd(2012, 11), 66.8400),
    (2456262.5, 66.8779),  # (cal_to_jd(2012, 12), 66.8779),
    # (cal_to_jd(2013, 1), 66.9069),
    # (cal_to_jd(2013, 2), 66.9443),
    # (cal_to_jd(2013, 3), 66.9763),
    # (cal_to_jd(2013, 4), 67.0258),

    # All these final values are predicted values
    # as taken from http://maia.usno.navy.mil/ser7/deltat.preds

    (2456293.5, 67.1),
    (2456384.75, 67.2),
    (2456476.0, 67.3),
    (2456567.25, 67.5),
    (2456658.5, 67.6),
    (2456749.75, 67.7),
    (2456841.0, 67.8),
    (2456932.25, 67.9),

    # At this point, per AAPLUS advice, move to equations rather than
    # interpolate from table.  Naughter (AAPLUS developer) starts equation
    # in April, but I have to end the table on a January 1st.
]

_tbl_start = 1620
_tbl_end = 2015


def deltaT_seconds(jd):
    """Return deltaT as seconds of time.

    For a historical range from 1620 to a recent year, we interpolate from a
    table of observed values. Outside that range we use formulae.

    Arguments:
      - `jd` : (int) Julian Day number

    Returns:
      - deltaT in seconds

    """
    yr, mo, day = jd_to_cal(jd)
    #
    # 1620 - 20xx
    #
    if _tbl_start < yr < _tbl_end:
        idx = bisect(_tbl, (jd, 0))
        try:
            jd1, secs1 = _tbl[idx]
            jd0, secs0 = _tbl[idx-1]
            # simple linear interpolation between two values
            return ((jd - jd0) * (secs1 - secs0) / (jd1 - jd0)) + secs0
        except IndexError:
            pass

    # I decided to replicate Naughter for times outside of table.
    # Naughter claims that this is better since there are
    # no discontinuties.
    # Found Naughter source NASA Eclipse Web site,
    # 'Polynomial Expressions for Delta T'.  Adapted from
    # 'Five Millennium Canon of Solar Eclipses'

    # Some of these if blocks will not be used because
    # first will interpolate from values in _tbl.
    # Decided to keep for completeness.

    # Wants middle of month... accurate enough?
    y = yr + (mo - 0.5)/12
    if (yr < -500):
        t = (y - 1820)/100.0
        result = polynomial([-20, 0, 32], t)
    elif (yr < 500):
        t = y/100.0
        result = polynomial([10583.6,
                             -1014.41,
                             33.78311,
                             -5.952053,
                             -0.1798452,
                             0.022174192,
                             0.0090316521], t)
    elif (yr < 1600):
        t = (y - 1000)/100.0
        result = polynomial([1574.2,
                             -556.01,
                             71.23472,
                             0.319781,
                             -0.8503463,
                             -0.005050998,
                             0.0083572073], t)
    elif (yr < 1700):
        t = (y - 1600)/100.0
        result = polynomial([120, -98.08, -153.2, 1/0.007129], t)
    elif (yr < 1800):
        t = (y - 1700)/100.0
        result = polynomial([8.83, 16.03, -59.285, 133.36, -1/0.01174], t)
    elif (yr < 1860):
        t = (y - 1800)/100.0
        result = polynomial([13.72,
                             -33.2447,
                             68.612,
                             4111.6,
                             -37436,
                             121272,
                             -169900,
                             87500], t)
    elif (yr < 1900):
        t = (y - 1860)/100.0
        result = polynomial([7.62,
                             57.37,
                             -2517.54,
                             16806.68,
                             -44736.24,
                             1/0.0000233174], t)
    elif (yr < 1920):
        t = (y - 1900)/100.0
        result = polynomial([-2.79, 149.4119, -598.939, 6196.6, -19700], t)
    elif (yr < 1941):
        t = (y - 1920)/100.0
        result = polynomial([21.20, 84.493, -761.00, 2093.6], t)
    elif (yr < 1961):
        t = (y - 1950)/100.0
        result = polynomial([29.07, 40.7, -1/0.0233, 1/0.002547], t)
    elif (yr < 1986):
        t = (y - 1975)/100.0
        result = polynomial([45.45, 106.7, -1/0.026, -1/0.000718], t)
    elif (yr < 2005):
        t = (y - 2000)/100.0
        result = polynomial([63.86,
                             33.45,
                             -603.74,
                             1727.5,
                             65181.4,
                             237359.9], t)
    elif (yr < 2050):
        t = (y - 2000)/100.0
        result = polynomial([62.92, 32.217, 55.89], t)
    elif (yr < 2150):
        t = (y - 1820)/100.0
        result = -20.0 + 32.0*(t)**2 - 0.5628*(2150 - y)
    else:
        t = (y - 1820)/100.0
        result = polynomial([-20.0, 0.0, 32.0], t)

    result = result + (-0.000012932*(y - 1955)**2)
    return result


def dt_to_ut(jd):
    """Convert Julian Day from dynamical to universal time.

    Arguments:
      - `jd` : (int) Julian Day number (dynamical time)

    Returns:
      - Julian Day number : (int) (universal time)

    """
    return jd - deltaT_seconds(jd) / seconds_per_day
