from optparse import make_option

from django.utils.translation import ugettext as _
from django.core.management.color import no_style
from django.utils import translation
from django.core.management.base import BaseCommand
from django.db import models

from autodocs.base import AutodocsBaseClass
from autodocs.utils import element_encode


class AutodocsDataTypes(AutodocsBaseClass):

    ATTRS_INSPECTED = (
        'verbose_name',
        'null',
        'unique',
        'flatchoices',
        'get_default',
        'max_length',
        'validators',
    )

    FIELDS_DESCRIPTION = (
        'Application',
        'Model',
        'Column',
        'Field',
        'Is null',
        'Is unique',
        'Choices',
        'Default',
        'Max. length',
        'Validators',
    )
    
    VALID_ATTRS = (
        'code',
        'limit_value',
        'regex',
    )
    
    def __init__(self, **kwargs):
        self.models = kwargs.pop('models')
        self.output_rows = []
        self.run()

    def cleanup(self, input_list):
        output_list = []
        for element in input_list:
            if element is None:
                value = _('None')
            elif isinstance(element, bool):
                value = element and _('Yes') or _('No')
            else:
                value = element
            output_list.append(element_encode(value))
        return output_list
        
    def process_rules(self, rule, attr_val):
        if attr_val == 'regex':
            return str(rule.regex.pattern)
        return str(getattr(rule, attr_val))
    
    def run(self):
        self.output_rows.append(self.FIELDS_DESCRIPTION)
        for model in self.models:
            meta = model._meta
            for field in meta.fields:
                doc_row = [meta.app_label, meta.verbose_name, field.column]
                for attr in self.ATTRS_INSPECTED:
                    field_attr = getattr(field, attr)
                    if attr == 'validators':
                        validation_rules = [self.process_rules(rule, attr_val)
                                                for rule in field_attr
                                                for attr_val in rule.__dict__
                                                if attr_val in self.VALID_ATTRS]
                        attr_value = ', '.join(validation_rules)
                    elif isinstance(field_attr, list):
                        try:
                            attr_value = ', '.join(field_attr)
                        except TypeError:
                            cleaned_list = self.cleanup(dict(field_attr).values())
                            attr_value = ', '.join(cleaned_list)
                    elif callable(field_attr):
                        attr_value = field_attr()
                    else:
                        attr_value = field_attr
                    doc_row.append(attr_value)                            
                self.output_rows.append(self.cleanup(doc_row))


class Command(BaseCommand):
    
    option_list = BaseCommand.option_list + (
        make_option('--output-file', '-o', dest='output_file',
            help='CSV output file name. Defaults to "types.csv".',
            default='types.csv'),
        make_option('-e', '--exclude', dest='excluded_apps',
            default='',
            help='Application names to exclude, comma separated. Defaults to '
                 'all applications. E.g.: ./manage.py data_types --exclude admin,auth'),
        make_option('--language', '-l', dest='language',
            default='en-us',
            help='Language. Defaults to "en-us".'),
    )

    help = 'Inspects applications and lists data types.'

    def handle(self, *args, **options):
        self.style = no_style()
        output_file = options.get('output_file')
        excluded_apps = options.get('excluded_apps').split(',')
        language = options.get('language')
            
        translation.activate(language)
        
        app_name = lambda app: app.__name__.split('.')[-2]
        apps = [app for app in models.get_apps()
                    if app_name(app) not in excluded_apps]
        all_models = []
        for app in apps:
            all_models.extend(models.get_models(app, include_auto_created=True))

        data_types = AutodocsDataTypes(models=all_models)
        data_types.export_csv(output_file)