import hashlib
from optparse import make_option

from django.core.management.color import no_style
from django.core.management.base import BaseCommand
from django.core.management.base import CommandError
from django.core.exceptions import ImproperlyConfigured
from django.db import models
from django.db.models.loading import cache

from autodocs.base import AutodocsBaseClass
from autodocs.utils import element_encode


class AutodocsDataIntegrity(AutodocsBaseClass):

    FIELDS_DESCRIPTION = (
        'Application',
        'Model',
        'Class',
        'Field',
        'Checksum',
    )

    def __init__(self, **kwargs):
        self.models = kwargs.pop('models')
        self.output_rows = []
        self.run()

    def run(self):
        def field_eligible(field):
            if field.primary_key:
                return False
            if field.rel:
                return False
            if field.attname.endswith('_id'):
                return False
            return True

        self.output_rows.append(self.FIELDS_DESCRIPTION)
        for model in self.models:
            meta = model._meta
            fields = [f.attname for f in meta.local_fields if field_eligible(f)]
            values = {}
            for obj in model._default_manager.all().order_by('id'):
                for field in fields:
                    value = getattr(obj, field)
                    if field not in values:
                        values[field] = []
                    values[field].append(u'%s-%s' % (obj.id, value))

            for attname, attval in values.items():
                consolidated = '+'.join([element_encode(chk) for chk in attval])
                md5 = hashlib.md5()
                md5.update(consolidated)
                checksum = md5.hexdigest()
                doc_row = [
                    meta.app_label,
                    meta.verbose_name,
                    model.__name__,
                    attname,
                    checksum,
                ]
                self.output_rows.append([element_encode(ele) for ele in doc_row])


class Command(BaseCommand):

    option_list = BaseCommand.option_list + (
        make_option('-o', '--output-file', dest='output_file',
            help='CSV output file name. Defaults to "checksums.csv".',
            default='checksums.csv'),
        make_option('-i', '--include', dest='included_apps',
            help='Application names to include, comma separated. Defaults to '
            'empty list. E.g.: ./manage.py data_integrity --include admin,auth'),
    )

    help = 'Inspects applications and calculates checksums for each field.'

    def handle(self, *args, **options):
        self.style = no_style()
        output_file = options.get('output_file')
        included_apps = options.get('included_apps')

        if not included_apps:
            raise CommandError('You must supply application names to include.')

        try:
            apps = [cache.get_app(app) for app in included_apps.split(',')]
        except ImproperlyConfigured as exc:
            raise CommandError(exc)

        all_models = []
        for app in apps:
            all_models.extend(models.get_models(app, include_auto_created=True))

        data_types = AutodocsDataIntegrity(models=all_models)
        data_types.export_csv(output_file)
