#!/usr/bin/env python3
# Copyright © 2014 Amirouche Boubekki <amirouche@hypermove.net>
# This work is free. You can redistribute it and/or modify it under the
# terms of the Frak It To Public License, Version 14.08 or later.
#
#
#                    FRAK IT TO PUBLIC LICENSE
#                          Version 14.08

# Copyright (C) 2014 Amirouche Boubekki <amirouche@hypermove.net>

# Everyone is permitted to copy and distribute verbatim or modified
# copies of this license document, and changing it is allowed as long
# as the name is changed.

#                    FRAK IT TO PUBLIC LICENSE
#   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

#  0. Do what the FRAK you can do.
#  1. Do FRAK what the FRAK you can do
#  3. Do FRAK what the FRAK you can FRAK do
import os
from os import write
from os import close
from json import loads
from json import dumps
from shutil import move
from shutil import copy
from tempfile import mkstemp
from datetime import datetime
from subprocess import check_call

from subprocess import check_output
from xml.sax.saxutils import escape

from docopt import docopt
from slugify import slugify
from jinja2 import Environment
from jinja2 import FileSystemLoader
from pygments import highlight
from pygments.lexers import get_lexer_by_name
from pygments.formatters import get_formatter_by_name


VERSION = '14.09.07-dev'


class Parser:
    """Generic parser"""

    def __init__(self, string):
        self.source = string
        self.position = 0
        self.context = dict()

    @classmethod
    def from_file(cls, file):
        with open(file) as f:
            string = f.read()
        return cls(string).parse()

    @classmethod
    def from_string(cls, string):
        return cls(string).parse()

    def next(self):
        self.position += 1
        return self.source[self.position]

    def peek(self, relative=0):
        return self.source[self.position + relative]

    def parse(self):
        raise NotImplementedError


class Azoufzouf(Parser):

    COMMAND_CHARACTER = 'ⵣ'

    def parse(self):
        output = list()
        next = self.peek()
        while True:
            if next != self.COMMAND_CHARACTER:
                text = self._consume_text()
                output.append(text)
            else:
                command = self._consume_command()
                output.append(command)
            try:
                next = self.peek()
            except IndexError:
                return output

    def _consume_text(self):
        output = self.peek()
        try:
            next = self.next()
        except IndexError:
            return output
        else:
            while True:
                try:
                    ok = self.peek(1) == self.COMMAND_CHARACTER
                except IndexError:
                    return output + next
                ok = ok and (next == self.COMMAND_CHARACTER)
                if ok:
                    output += next
                    self.next()  # consume escape char
                    next = self.next()
                elif next == self.COMMAND_CHARACTER:
                    # and self.peek(1) != self.COMMAND_CHARACTER
                    return output
                else:  # continue to consume text
                    output += next
                    try:
                        next = self.next()
                    except IndexError:
                        return output

    # methods for parsing commands like øcommand_name[fuu]{bar}[123]

    def _consume_command(self):
        command = list()
        name = self.next()
        next = self.next()
        while True:
            if next == '[':
                break
            elif next == '{':
                break
            elif next.isspace():
                break
            else:
                name += next
                try:
                    next = self.next()
                except IndexError:
                    command.append(name)
                    return command
        command.append(name)
        while True:
            if next == '[':
                argument = self._consume_argument()
                command.append(argument)
            elif next == '{':
                argument = self._consume_text_argument()
                command.append(argument)
            elif next not in '{[':
                return command
            try:
                next = self.peek()
            except IndexError:
                return command

    def _consume_argument(self):
        argument = self.next()
        next = self.next()
        while True:
            if next == ']':
                try:
                    # try to consume closing bracket
                    self.next()
                finally:
                    # anyway return argument
                    break
            else:
                argument += next
                try:
                    next = self.next()
                except IndexError:
                    return self._to_python(argument)
        # convert to integer or float if possible
        try:
            argument = int(argument)
        except ValueError:
            try:
                argument = float(argument)
            except ValueError:
                pass
            else:
                pass
        else:
            pass
        return argument

    def _consume_text_argument(self):
        argument = self.next()
        next = self.next()
        while True:
            if next == '}':
                try:
                    continuation = self.peek(1)
                except IndexError:
                    break
                else:
                    if continuation != '}':
                        # end of text argument
                        break
            elif next == '}':
                argument += next
                self.next()  # consume escape bracket
                next = self.next()
            else:
                argument += next
                next = self.next()
        # try to consume closing bracket
        try:
            self.next()
        except IndexError:
            # end of string
            pass
        # return parsed string
        return argument


parse_string = Azoufzouf.from_string
parse_file = Azoufzouf.from_file


def dung(path, output=None, indent=None):
    value = parse_file(path)
    if indent:
        json = dumps(value, indent=indent)
    else:
        json = dumps(value)
    # dispatch options
    if output:
        with open(output, 'w') as f:
            f.write(json)
        return value
    else:
        return value


class HTMLNoteBuilder:
    """Command calls are translated to method calls."""

    def __init__(self):
        self.last_is_inline = False
        self.context = dict()
        self._keywords = list()
        self._title = None
        self._introduction = None
        self._first_section_slug = None

    def render(self, dungs):
        output = ''
        for dung in dungs:
            if isinstance(dung, list):
                name = dung[0]
                method = getattr(self, name)
                arguments = dung[1:]
                text = method(*arguments)
                output += text
            else:
                if dung.isspace():
                    continue
                lines = dung.split('\n')
                empty_line_sequence_size = 0
                if len(lines) == 2:
                    head, tail = lines
                    if head == '':
                        output += ' %s' % tail
                    else:
                        output += '%s' % head
                else:
                    for line in lines:
                        if line == '':
                            if self.last_is_inline:
                                empty_line_sequence_size += 1
                                if empty_line_sequence_size == 2:
                                    output += '</p>'
                                    self.last_is_inline = False
                                    empty_line_sequence_size += 1
                                else:
                                    empty_line_sequence_size += 1
                            # else: nothing to do
                        else:
                            line = line.replace('---', '—')
                            line = line.replace('--', '–')
                            if self.last_is_inline:
                                if empty_line_sequence_size:
                                    output += ' %s' % line
                                else:
                                    output += '%s' % line
                            else:
                                output += '<p>%s' % line
                                self.last_is_inline = True
                            empty_line_sequence_size = 1

        context = dict(
            title=self._title,
            introduction=self._introduction,
            first_section_slug=self._first_section_slug,
            keywords=self._keywords,
            body=output,
        )
        return context

    def keyword(self, name):
        self._keywords.append(name)
        self.last_is_inline = False
        return ''

    def introduction(self, text):
        self._introduction = text
        self.last_is_inline = False
        return '<p>%s</p>' % text

    def title(self, title):
        self._title = title
        self.last_is_inline = False
        return '<h1>%s</h1>' % title

    def set(self, ref, value):
        self.last_is_inline = False
        self.context[ref] = value
        return ''

    def section(self, text):
        self.last_is_inline = False
        slug = 'section-%s' % slugify(text)
        if not self._first_section_slug:
            self._first_section_slug = slug
        return '<h1 id="%s">%s</h1>' % (slug, text)

    def subsection(self, text):
        self.last_is_inline = False
        slug = 'subsection-%s' % slugify(text)
        return '<h2 id="%s">%s</h1>' % (slug, text)

    def subsubsection(self, text):
        self.last_is_inline = False
        slug = 'subsubsection-%s' % slugify(text)
        return '<h3 id="%s">%s</h1>' % (slug, text)

    def href(self, text, ref=None):
        self.last_is_inline = True
        ref = ref if ref else text
        try:
            url = self.context[ref]
        except KeyError:
            url = ref
        return '<a href="%s">%s</a>' % (url, text)

    def citation(self, text, who=None):
        self.last_is_inline = False
        if who:
            output = '<div class="citation"><div class="body">'
            output += '%s</div><p>%s</p></div>'
            output = output % (text, who)
        else:
            output = '<div class="citation"><div class="body">%s</div></div>'
            output = output % text
        return output

    def emphase(self, text):
        self.last_is_inline = True
        return ' <i>%s</i>' % text

    def code(self, text):
        self.last_is_inline = True
        return '<code>%s</code>' % text

    def include(self, filepath, lang=None):
        self.last_is_inline = False
        file = filepath
        with open(file) as f:
            output = f.read()
        if lang:
            html = get_formatter_by_name('html')
            lexer = get_lexer_by_name(lang)
            output = highlight(output, lexer, html)
            # remove wrapping div to insert legend
            start = len('<div class="highlight">')
            end = -len('</div>\n')
            output = output[start:end]
        else:
            output = escape(output)
            output = '<pre>%s</pre>' % output
        icon = '<span class="glyphicon glyphicon-cloud-download"></span>'
        legend = '<a class="legend" href="%s">%s <code>%s</code></a>' % (
            filepath,
            icon,
            filepath,
        )
        output = '<div class="highlight">%s%s</div>' % (output, legend)
        return output

    def blockdiag(self, name, command):
        self.last_is_inline = False
        command = 'blockdiag {' + command + '}'
        buffer = command.encode('utf-8')
        f, path = mkstemp()
        write(f, buffer)
        close(f)
        # generate svg file
        command = 'blockdiag -Tsvg %s' % path
        command = command.split()
        check_call(command)
        target = name + '.svg'
        initial = path + '.svg'
        copy(initial, target)
        return '<div><img src="%s" /></div>' % target

    def python(self, filepath):
        self.last_is_inline = False
        filepath = os.path.abspath(filepath)
        command = 'pygmentize -f html %s' % filepath
        output = check_output(command.split())
        output = output.decode('utf')
        return output


def new_draft(title):
    title = title.capitalize()
    slug = slugify(title)
    directory = os.path.join('_drafts', slug)
    if os.path.exists(directory):
        print('Ark! There is already a draft with this name.')
        exit(1)
    else:
        os.makedirs(directory)
        index = os.path.join(directory, 'index.azf')
        with open(index, 'w') as f:
            body = 'ⵣtitle{%s}\n\n' % title
            f.write(body)
    print('Done. You can edit %s' % index)


def render_html(path):
    # move curdir to the note directory
    # because the note file is written in the
    # note file perspective
    origin_directory = os.curdir
    origin_directory = os.path.abspath(origin_directory)
    directory = os.path.abspath(path)
    os.chdir(directory)
    index = os.path.abspath('index.azf')
    json = dung(index)
    # * Render body and retrieve a few variables
    # translate
    render = HTMLNoteBuilder().render
    context = render(json)
    os.chdir(origin_directory)
    # render note with template
    now = datetime.now()
    created_at = now.year, now.month, now.day
    context['created_at'] = created_at

    templates = os.path.abspath(os.curdir)
    templates = os.path.join(templates, '_templates')
    environment = Environment(
        loader=FileSystemLoader(templates),
    )
    environment.filters['slugify'] = slugify
    template = environment.get_template('note.jinja2')
    main = template.render(**context)

    # store rendered note in the same directory
    html = os.path.join(path, 'index.html')
    with open(html, 'w') as f:
        f.write(main)

    # cache rendering context
    json = os.path.join(path, 'index.json')
    with open(json, 'w') as f:
        f.write(dumps(context, indent=4))
    return context


def render_jinja2(template, context, output):
    templates = os.path.abspath(os.curdir)
    templates = os.path.join(templates, '_templates')
    environment = Environment(
        loader=FileSystemLoader(templates),
    )
    environment.filters['slugify'] = slugify
    template = environment.get_template(template)
    out = template.render(**context)
    with open(output, 'w') as f:
        f.write(out)


def new_book(title):
    text = """ⵣtitle{%s}

There is the new book!

ⵣbegin_toc
* path/to/part-one
* path/to/part-two
ⵣend_toc
"""
    text = text % title
    with open('index.azf', 'w') as f:
        f.write(text)

def new_book_part(title):
    text = """ⵣtitle{%s}

There is the new part of the book!

ⵣbegin_toc
* path/to/chapter-one
* path/to/chapter-two
ⵣend_toc
"""
    text = text % title
    slug = slugify(title)
    os.makedirs(slug)
    index = os.path.join(slug, 'index.azf')
    with open(index, 'w') as f:
        f.write(text)


def new_book_chapter(part, title):
    text = """ⵣtitle{%s}

A full chapter!
"""
    text = text % title
    slug = slugify(title)
    path = os.path.join(part, slug)
    os.makedirs(path)
    index = os.path.join(path, 'index.azf')
    with open(index, 'w') as f:
        f.write(text)


def main():
    doc = """main.py.

Usage:
  main.py new book <title>
  main.py new book part <title>
  main.py new book chapter <part> <title>
  main.py new draft <title>
  main.py publish note <path>
  main.py publish page <path>
  main.py dung <path> [<output>] [--pretty-print=<indent>] [--ugly]
  main.py render html <path>
  main.py render jinja2 <template> <context> <output>
  main.py build
  main.py -h | --help
  main.py --version

Options:
  --pretty-print=<indent> Pretty print indentation [default: 4].
  --ugly                  Ugly JSON.
  -h --help               Show this screen.
  --version               Show version.
"""
    arguments = docopt(doc, version='14.09.07-dev Pyramid of Cheops')
    if os.environ.get('DEBUG', False):
        print(arguments)
    # dispatch
    if arguments['dung']:
        path = arguments['<path>']
        output = arguments['<output>']
        if arguments['--ugly']:
            indent = None
        else:
            if arguments['--pretty-print']:
                indent = arguments['<indent>']
            else:
                indent = 4
        dung(path, output, indent)
        print('Done!')
    elif arguments['render'] and arguments['html']:
        path = arguments['<path>']
        render_html(path)
        print('Done!')
    elif arguments['render'] and arguments['jinja2']:
        template = arguments['<template>']
        context = arguments['<context>']
        output = arguments['<output>']
        with open(context) as f:
            context = loads(f.read())
        render_jinja2(template, context, output)
        print('Done!')
    elif arguments['new'] and arguments['draft']:
        title = arguments['<title>']
        new_draft(title)
        print('Done!')
    elif arguments['new'] and arguments['book']:
        title = arguments['<title>']
        if arguments['part']:
            new_book_part(title)
        elif arguments['chapter']:
            part = arguments['<part>']
            new_book_chapter(part, title)
        else:
            new_book(title)
        print('Done!')
    elif arguments['publish'] and arguments['note']:
        path = arguments['<path>']
        context = render_html(path)
        # move to published notes
        slug = slugify(context['title'])
        created_at = map(str, context['created_at'])
        created_at = os.path.join(*created_at)
        target = os.path.join('notes', created_at, slug)
        move(path, target)
        print('Done!')
    elif arguments['publish'] and arguments['page']:
        path = arguments['<path>']
        # prepare directory for publication
        # add/rewrite created_at file with publication date
        now = datetime.now()
        now = now.year, now.month, now.day
        now = list(map(str, now))
        created_at = os.path.join(path, 'created_at')
        with open(created_at, 'w') as f:
            f.write(dumps(now))
        # render page
        index = os.path.join(path, 'index.azf')
        render_html(index)
        # move every to pages directory
        if path.endswith('/'):
            slug = os.path.split(path[:-1])[1]
        else:
            slug = os.path.split(path)[1]
        target = os.path.join('pages', slug)
        move(path, target)
        print('Done!')
    elif arguments['build']:
        # TODO: render atom feed, pdf and epub
        # build index page
        # group notes by year
        keywords = dict()
        notes = list()
        years = os.listdir('notes')
        years.sort(reverse=True)
        for year in years:
            year_path = os.path.join('notes', year)
            months = os.listdir(year_path)
            months.sort()
            for month in months:
                month_path = os.path.join(year_path, month)
                days = os.listdir(month_path)
                days.sort()
                for day in days:
                    day_path = os.path.join(month_path, day)
                    slugs = os.listdir(day_path)
                    slugs.sort()
                    for slug in slugs:
                        path = os.path.join(day_path, slug)
                        json = os.path.join(path, 'index.json')
                        with open(json) as f:
                            note = loads(f.read())
                        # make easier to group by year in template
                        note['year'] = note['created_at'][0]
                        # build the shortcut pass before end
                        # XXX: should done be in template
                        slug = note['first_section_slug']
                        note['path'] = '%s#%s' % (path, slug)
                        # populate inverted index: keyword -> list notes
                        for keyword in note['keywords']:
                            try:
                                keywords[keyword].append(note)
                            except:
                                keywords[keyword] = list()
                                keywords[keyword].append(note)
                        notes.append(note)
        # render index.html
        templates = os.path.abspath(os.curdir)
        templates = os.path.join(templates, '_templates')
        environment = Environment(
            loader=FileSystemLoader(templates),
        )
        environment.filters['slugify'] = slugify
        template = environment.get_template('index.jinja2')
        html = template.render(
            notes=notes,
            title='Wellcome',
            bodyclass='index',
            keywords=keywords.keys(),
        )
        # save
        with open('index.html', 'w') as f:
            f.write(html)
        # save notes data
        with open('index.json', 'w') as f:
            f.write(dumps(notes))
        # build keywords pages
        with open('keywords.json', 'w') as f:
            f.write(dumps(keywords))
        for keyword, notes in keywords.items():
            slug = slugify(keyword)
            path = os.path.join('keyword', slug)
            try:
                # new keyword
                os.makedirs(path)
            except FileExistsError:
                pass  # this is an update of the keyword page
            template = environment.get_template('keyword.jinja2')
            title = 'Notes about %s' % keyword
            html = template.render(
                notes=notes,
                title=title,
                bodyclass='index',
                keyword=keyword,
            )
            # save
            index = os.path.join(path, 'index.html')
            with open(index, 'w') as f:
                f.write(html)
    else:
        message = 'Oops! What did happen here?! Please fill '
        message += 'a bug report with a lot of details using DEBUG=FIXME '
        message += 'and send output at amirouche@hypermove.net'
        message += ', thanks!'
        raise Exception(message)


if __name__ == '__main__':
    main()
