# bard/log.py
#
#

""" provide logging infrastructure. """

__copyright__ = "Copyright 2014 B.H.J Thate"

## IMPORTS

from bard.defines import BOLD, BLA, BLUE, RED, GREEN, YELLOW, ENDC
from bard.utils import error, j, cdir, short_date

import logging.handlers
import logging
import time
import sys
import os
import re

## DEFINES

homedir = os.path.expanduser("~")
logdir = j(homedir, "bard.logs", "")
enabled = False
datefmt = '%H:%M:%S'
format = "%(asctime)-8s %(message)s %(module)s.%(lineno)s"
format_small = "%(asctime)-8s %(message)s"

LEVELS = {'debug': logging.DEBUG,
          'info': logging.INFO,
          'warning': logging.WARNING,
          'warn': logging.WARNING,
          'error': logging.ERROR,
          'critical': logging.CRITICAL
        }

## CLASSES

class Formatter(logging.Formatter):

    """ hooks into the logging system. """

    def format(zelf, record):
        target = str(record.msg)
        if not target: target = " "
        if target[0] in [">", ]: target = "%s%s%s%s" % (BLUE, target[0], ENDC, target[1:])
        elif target[0] in ["<", ]: target = "%s%s%s%s" % (RED, target[0], ENDC, target[1:])
        elif target[0] in ["!", ]: target = "%s%s%s%s" % (YELLOW, target[0], ENDC, target[1:])
        elif target[0] in ["#", ]: target = "%s%s%s%s" % (BLA, target[0], ENDC, target[1:])
        record.msg = target
        return logging.Formatter.format(zelf, record)

## INIT

def init(loglevel):
    """ return a properly configured logger. """
    logger = logging.getLogger("")
    formatter = Formatter(format, datefmt=datefmt)
    formatter_short = Formatter(format_small, datefmt=datefmt)
    level = LEVELS.get(str(loglevel).lower(), logging.NOTSET)
    filehandler = None
    logger.setLevel(level)
    if logger.handlers:
        for handler in logger.handlers: logger.removeHandler(handler)
    if not os.path.exists(logdir): cdir(logdir)
    try: filehandler = logging.handlers.TimedRotatingFileHandler(j(logdir, "log"), 'midnight')
    except: error()
    ch = logging.StreamHandler()
    ch.setLevel(level)
    ch.setFormatter(formatter_short)
    logger.addHandler(ch)
    if filehandler:
        filehandler.setFormatter(formatter_short)
        filehandler.setLevel(level)
        logger.addHandler(filehandler)
    global enabled
    enabled = True
    return logger
