from __future__ import absolute_import

"""
Wrapper around PyCrypto providing basic symmetric key
encryption with AES in CFB mode.

Plaintext is raw bytes(str in python 2.X or bytes in
python 3.X). If it's actual text you're transmitting,
ensure you encode/decode as necessary.

Includes four functions:

generate_salt()
    Generates random data for use as salt.

derive_key(password, salt)
    Returns the encryption/decryption key for the given
    password and salt, using the standard Password-Based Key
    Derivation Function 2.  The password should be secret and
    preshared, the salt should be preshared too but need not
    be secret, and can be transmitted for example insecurely
    at the beginning of communication.

encrypt(plaintext, key)
    Encrypts plaintext using the Advanced Encryption
    Standard in cipher feedback mode (AES CFB). Uses a random
    initialisation vector each time it is called, which is
    prepended to the ciphertext.

decrypt(ciphertext, key)
    Decrypts the ciphertext using the Advanced Encryption
    Standard in cipher feedback mode (AES CFB), using a key
    as returned by derive_key. Uses the initialisation vector
    prepended to the ciphertext by encrypt().
"""


from Crypto.Cipher import AES
from Crypto import Random
from Crypto.Protocol import KDF

try:
    from .__version__ import __version__
except ImportError:
    raise
    # Version file has not been autogenerated from build process:
    __version__ = None


    
def generate_salt():
    """Generates random data for use as salt."""
    return Random.new().read(8)
    
def derive_key(password, salt):
    """Returns the encryption/decryption key for the given
    password and salt, using the standard Password-Based
    Key Derivation Function 2.  The password should
    be secret and preshared, the salt should be
    preshared too but need not be secret, and can be
    transmitted for example insecurely at the beginning
    of communication."""
    return KDF.PBKDF2(password, salt)
     
def encrypt(plaintext, key):
    """Encrypts plaintext using the Advanced Encryption
    Standard in cipher feedback mode (AES CFB). Uses a
    random initialisation vector each time it is called,
    which is prepended to the ciphertext."""
    iv = Random.new().read(AES.block_size)
    cipher = AES.new(key, AES.MODE_CFB, iv)
    ciphertext = iv + cipher.encrypt(plaintext)
    return ciphertext
    
def decrypt(ciphertext, key):
    """Decrypts the ciphertext using the Advanced
    Encryption Standard in cipher feedback mode (AES
    CFB), using a key as returned by derive_key. Uses the
    initialisation vector prepended to the ciphertext by
    encrypt(). """
    iv = ciphertext[:AES.block_size]
    cipher = AES.new(key, AES.MODE_CFB, iv)
    plaintext = cipher.decrypt(ciphertext[AES.block_size:])
    return plaintext
    

