"""Bayesian variant calling with FreeBayes.

http://bioinformatics.bc.edu/marthlab/FreeBayes
"""
import os
import shutil
import subprocess

from bcbio import broad
from bcbio.utils import file_exists
from bcbio.distributed.transaction import file_transaction
from bcbio.variation import annotation, genotype
from bcbio.log import logger
from bcbio.pipeline.shared import subset_variant_regions

def _freebayes_options_from_config(aconfig, out_file, region=None):
    opts = []
    opts += ["--ploidy", str(aconfig.get("ploidy", 2))]
    variant_regions = aconfig.get("variant_regions", None)
    target = subset_variant_regions(variant_regions, region, out_file)
    if target:
        opts += ["--region" if target == region else "--targets", target]
    background = aconfig.get("call_background", None)
    if background and os.path.exists(background):
        opts += ["--variant-input", background]
    return opts

def run_freebayes(align_bam, ref_file, config, dbsnp=None, region=None,
                  out_file=None):
    """Detect small polymorphisms with FreeBayes.
    """
    broad_runner = broad.runner_from_config(config)
    broad_runner.run_fn("picard_index", align_bam)
    if out_file is None:
        out_file = "%s-variants.vcf" % os.path.splitext(align_bam)[0]
    if not file_exists(out_file):
        logger.info("Genotyping with FreeBayes: {region} {fname}".format(
            region=region, fname=os.path.basename(align_bam)))
        with file_transaction(out_file) as tx_out_file:
            cl = [config["program"].get("freebayes", "freebayes"),
                  "-b", align_bam, "-v", tx_out_file, "-f", ref_file,
                  "--left-align-indels", "--use-mapping-quality"]
            cl += _freebayes_options_from_config(config["algorithm"], out_file, region)
            subprocess.check_call(cl)
    _remove_freebayes_refalt_dups(out_file)
    return out_file

def _remove_freebayes_refalt_dups(in_file):
    """Remove lines from FreeBayes outputs where REF/ALT are identical.
    2       22816178        .       G       G       0.0339196
    """
    out_file = apply("{0}-nodups{1}".format, os.path.splitext(in_file))
    if not file_exists(out_file):
        with open(in_file) as in_handle:
            with open(out_file, "w") as out_handle:
                for line in in_handle:
                    if line.startswith("#"):
                        out_handle.write(line)
                    else:
                        parts = line.split("\t")
                        if parts[3] != parts[4]:
                            out_handle.write(line)
        shutil.move(in_file, "{0}.orig".format(in_file))
        shutil.move(out_file, in_file)
        with open(out_file, "w") as out_handle:
            out_handle.write("Moved to {0}".format(in_file))

def postcall_annotate(in_file, ref_file, vrn_files, config):
    """Perform post-call annotation of FreeBayes calls in preparation for filtering.
    """
    #out_file = _check_file_gatk_merge(in_file)
    out_file = annotation.annotate_nongatk_vcf(in_file, vrn_files.dbsnp,
                                               ref_file, config)
    #filters = ["QUAL < 20.0", "DP < 5"]
    #out_file = genotype.variant_filtration_with_exp(broad.runner_from_config(config),
    #                                                out_file, ref_file, "", filters)
    return out_file

def _check_file_gatk_merge(vcf_file):
    """Remove problem lines generated by GATK merging from FreeBayes calls.

    Works around this issue until next GATK release:
    http://getsatisfaction.com/gsa/topics/
    variantcontext_creates_empty_allele_from_vcf_input_with_multiple_alleles
    """
    def _not_empty_allele(line):
        parts = line.split("\t")
        alt = parts[4]
        return not alt[0] == ","
    orig_file = "{0}.orig".format(vcf_file)
    if not file_exists(orig_file):
        shutil.move(vcf_file, orig_file)
        with open(orig_file) as in_handle:
            with open(vcf_file, "w") as out_handle:
                for line in in_handle:
                    if line.startswith("#") or _not_empty_allele(line):
                        out_handle.write(line)
    return vcf_file
