"""
Unit tests for the Position class.
"""
import datetime
import unittest
import copy
from datetime import date

from .amount import ZERO, D, Amount
from .position import Lot, Position, from_string, from_amounts


class TestPosition(unittest.TestCase):

    def test_from_string(self):
        with self.assertRaises(ValueError):
            pos = from_string('')

        pos = from_string('10 USD')
        self.assertEqual(Position(Lot("USD", None, None), D('10')), pos)

        pos = from_string(' 111.2934  CAD ')
        self.assertEqual(Position(Lot("CAD", None, None), D('111.2934')), pos)

        pos = from_string('2.2 GOOG {532.43 USD}')
        cost = Amount(D('532.43'), 'USD')
        self.assertEqual(Position(Lot("GOOG", cost, None), D('2.2')), pos)

        pos = from_string('2.2 GOOG {532.43 USD / 2014-06-15}')
        cost = Amount(D('532.43'), 'USD')
        lot_date = datetime.date(2014, 6, 15)
        self.assertEqual(Position(Lot("GOOG", cost, lot_date), D('2.2')), pos)

    def test_from_amounts(self):
        pos = from_amounts(Amount(D('10.00'), 'USD'))
        self.assertEqual(Position(Lot("USD", None, None), D('10')), pos)

        pos = from_amounts(Amount(D('10'), 'GOOG'),
                           Amount(D('510.00'), 'USD'))
        self.assertEqual(
            Position(Lot("GOOG", Amount(D('510'), 'USD'), None), D('10')), pos)

    def test_constructors(self):
        position = Position(Lot('USD', None, None),
                            D('123.45'))
        position = Position(Lot('USD', Amount('74.00', 'CAD'), None),
                            D('123.45'))
        position = Position(Lot('USD', Amount('74.00', 'CAD'), date(2013, 2, 3)),
                            D('123.45'))
        with self.assertRaises(Exception):
            Position(None, D('123.45'))
        with self.assertRaises(Exception):
            Position(Lot('USD', None, None), None)

    def test_compare_zero_to_none(self):
        pos1 = Position(Lot("CAD", None, None), ZERO)
        pos_none = None
        self.assertEqual(pos1, pos_none)
        self.assertEqual(pos_none, pos1)

        pos2 = Position(Lot("USD", None, None), ZERO)
        self.assertNotEqual(pos1, pos2)

    def test_eq_and_sortkey(self):
        pos1 = Position(Lot("USD", None, None), D('200'))
        pos2 = Position(Lot("USD", None, None), D('201'))
        pos3 = Position(Lot("CAD", None, None), D('100'))
        pos4 = Position(Lot("CAD", None, None), D('101'))
        pos5 = Position(Lot("ZZZ", None, None), D('50'))
        positions = [pos5, pos4, pos3, pos2, pos1]
        positions.sort()

        self.assertTrue(pos1 < pos2)
        self.assertTrue(pos2 < pos3)
        self.assertTrue(pos3 < pos4)
        self.assertTrue(pos4 < pos5)

        self.assertTrue(positions[0] is pos1)
        self.assertTrue(positions[1] is pos2)
        self.assertTrue(positions[2] is pos3)
        self.assertTrue(positions[3] is pos4)
        self.assertTrue(positions[4] is pos5)

    def test_copy(self):
        # Ensure that the lot instances are shared.
        pos1 = Position(Lot("USD", None, None), D('200'))
        pos2 = copy.copy(pos1)
        self.assertTrue(pos1.lot is pos2.lot)

    def test_getamount(self):
        pos = Position(Lot("USD", Amount('10', 'AUD'), None), D('28372'))
        self.assertEqual(Amount('28372', 'USD'), pos.get_amount())
        self.assertEqual(Amount('283720', 'AUD'), pos.get_cost())

        cpos = pos.at_cost()
        self.assertTrue(isinstance(cpos, Position))
        self.assertEqual(Amount('283720', 'AUD'), cpos.get_amount())
        self.assertEqual(Amount('283720', 'AUD'), cpos.get_cost())

    def test_add(self):
        pos = Position(Lot("USD", Amount('10', 'AUD'), None), D('28372'))
        pos.add(D('337'))
        self.assertEqual(Amount('28709', 'USD'), pos.get_amount())
        self.assertEqual(Amount('287090', 'AUD'), pos.get_cost())

    def test_negative(self):
        pos = Position(Lot("USD", Amount('10', 'AUD'), None), D('28372'))
        negpos = pos.get_negative()
        self.assertEqual(Amount('-28372', 'USD'), negpos.get_amount())
        self.assertEqual(Amount('-283720', 'AUD'), negpos.get_cost())

    def test_is_negative_at_cost(self):
        pos1 = Position(Lot("USD", Amount('10', 'AUD'), None), D('1'))
        pos2 = Position(Lot("USD", Amount('10', 'AUD'), None), D('-1'))
        self.assertFalse(pos1.is_negative_at_cost())
        self.assertTrue(pos2.is_negative_at_cost())
