# Copyright (c) 2012, Sven Thiele <sthiele78@gmail.com>
#
# This file is part of BioASP.
#
# BioASP is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# BioASP is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with BioASP.  If not, see <http://www.gnu.org/licenses/>.import random

# -*- coding: utf-8 -*-
from bioasp import asp
from bioasp.misc import *
import tempfile
import os

def vertices(termset):
    v = filter(lambda t: t.p('vertex'), termset)
    return map(lambda t: t.arg(0),v)

def edges(termset):
    e = filter(lambda t: t.p('edge'), termset)
    return map(lambda t: (t.arg(0),t.arg(1)),e)

def extract(termset,pred,arg):
    '''
    Computes the set of strings which argument number [arg] of
    terms with predicate [pred] in TermSet [termset].
    '''
    accu = set()
    for t in termset:
        if t.p(pred):
            accu.add(t.arg(arg))
    return accu

def vertex_set(termset): return extract(termset,'vertex',0)
def input_set(termset): return extract(termset,'input',0)
def exp_set(termset): return extract(termset,'exp',0)

def constraint_set(termset):
    '''
    if [termset] is a TermSet object representing a system, then
    [constraint_set(termset)] is a set of pairs of strings,
    representing the set of constraints derived from [termset]
    for the sign consistency constraint. Each pair in the set
    represents one vertex and one experiment.
    '''
    constr_vtx = vertex_set(termset).difference(input_set(termset))
    accu = set()
    for t in constr_vtx:
        for u in exp_set(termset):
            accu.add((t,u))
    return accu

def idconcat(e,v):
    return quote(unquote(e) + unquote(v))
    


class CroppedTermSet(dict):
    def __init__(self,termset,constr_set):
        '''
        Given a system [termset], computes a sub-system where all
        parts (vertices and edges) that do not appear in a constraint
        in [constr_set] are removed. This is useful to
        compute a MIC from the entire system. [constr_set] is a set
        of pairs of strings (vertex x experiment).
        The result is a dictionary whose keys are the experiments
        appearing in [constr_set]. For each such experiment [e],
        the dictionary associates a TermSet describing the subsystem
        of [termset] generated by [constr_set] for experiment [e].
        '''
        dict.__init__(self)
        exp = set(map(lambda (v,e): e,constr_set))
        vertices = {}
        incoming = {}
        for e in exp:
            self[e] = asp.TermSet()
            vertices[e] = set()
            incoming[e] = set()
        for v,e in constr_set: vertices[e].add(v)
        for t in termset:
            if t.p('edge'):
                for e in exp:
                    if t.arg(1) in vertices[e] and not t.arg(0) in vertices[e]:
                        incoming[e].add(t.arg(0))
                        self[e].add(asp.Term('input',[t.arg(0)]))
        for e in exp: vertices[e] = vertices[e].union(incoming[e])
        for t in termset:
            if t.p('vertex'):
                for e in exp:
                    if t.arg(0) in vertices[e]:
                        self[e].add(t)
            elif t.p('edge'):
                for e in exp:
                    if t.arg(0) in vertices[e] and t.arg(1) in vertices[e] and t.arg(1) not in incoming[e]:
                        self[e].add(t)
            elif t.p('input'):
                for e in exp:
                    if t.arg(0) in vertices[e]:
                        self[e].add(t)
            elif t.p('exp'):
                if t.arg(0) in exp: self[t.arg(0)].add(t)
            elif t.p('obs_vlabel'):
                if t.arg(0) in exp and t.arg(1) in vertices[t.arg(0)]:
                    self[t.arg(0)].add(t)
            elif t.p('obs_elabel'):
                for e in exp:
                    if t.arg(0) in vertices[e] and t.arg(1) in vertices[e]:
                        self[e].add(t)

    def collapse(self):
        '''
        Given a family of subsystems (one by experiment, as given by the
        [crop] function), returns the union of all subsystems.
        '''
        accu = asp.TermSet()
        for _, ts in self.iteritems():
            accu = accu.union(ts)
        return accu

    def dot_output(self,fn=None):
        if fn:
            file = open(fn,'w')
        else:
            fd, fn = tempfile.mkstemp('.lp')
            file = os.fdopen(fd,'w')
        f = open(fn,'w')
        for e,sub_e in self.iteritems():
            f.write('digraph ' + e + '{\n')
            for v in vertex_set(sub_e):
                if asp.Term('obs_vlabel', [ e, v, '-1']) in sub_e:
                    style = ' [style=filled,color="#BB1111",fillcolor="#FFAAAA"'
                elif asp.Term('obs_vlabel', [ e, v, '1']) in sub_e:
                    style = ' [style=filled,color="#11BB11",fillcolor="#AAFFAA"'
                else:
                    style = ' [style=filled,fillcolor=white'
                if asp.Term('input',[v]) in sub_e:
                    style += ',shape=box'
                style += ',label=' + v + ']'
                f.write(idconcat(e,v) + style + ';\n')
            for t in sub_e.filter(lambda t: t.p('edge')):
                if asp.Term('obs_elabel', [ t.arg(0), t.arg(1), '-1']) in sub_e:
                    style = ' [color="#BB1111",arrowhead=tee] '
                elif asp.Term('obs_elabel', [ t.arg(0), t.arg(1), '1']) in sub_e:
                    style = ' [color="#11BB11"] '
                else:
                    style = ' [color=black,arrowhead=odot] '
                f.write(idconcat(e,t.arg(0)) + ' -> ' + idconcat(e,t.arg(1)) + style + ';')
            f.write('}\n')
        f.close()
        return fn

    def dot_show(self):
        fn = self.dot_output()
        os.system('dot -Tps ' + fn + ' > ' + fn + '.ps && gv ' + fn + '.ps')
#        os.unlink(fn)
#        os.unlink(fn + '.ps')

class Dataset(CroppedTermSet):
    def __init__(self,termset):
        CroppedTermSet.__init__(self, termset,constraint_set(termset))
