'''Testing code for Bio.Entrez parsers.
'''

import unittest

import os
if os.name == 'java':
    from Bio import MissingExternalDependencyError
    #This is a slight miss-use of MissingExternalDependencyError,
    #but it will do in the short term to skip this unit test on Jython
    raise MissingExternalDependencyError("The Bio.Entrez XML parser fails "
        "on Jython, see http://bugs.jython.org/issue1447")

from Bio import Entrez


class EInfoTest(unittest.TestCase):
    '''Tests for parsing XML output returned by EInfo
    '''
    def test_list(self):
        '''Test parsing database list returned by EInfo
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.einfo()
        handle = open('Entrez/einfo1.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record["DbList"], ['pubmed',
                                            'protein',
                                            'nucleotide',
                                            'nuccore',
                                            'nucgss',
                                            'nucest',
                                            'structure',
                                            'genome',
                                            'books',
                                            'cancerchromosomes',
                                            'cdd',
                                            'gap',
                                            'domains',
                                            'gene',
                                            'genomeprj',
                                            'gensat',
                                            'geo',
                                            'gds',
                                            'homologene',
                                            'journals',
                                            'mesh',
                                            'ncbisearch',
                                            'nlmcatalog',
                                            'omia',
                                            'omim',
                                            'pmc',
                                            'popset',
                                            'probe',
                                            'proteinclusters',
                                            'pcassay',
                                            'pccompound',
                                            'pcsubstance',
                                            'snp',
                                            'taxonomy',
                                            'toolkit',
                                            'unigene',
                                            'unists'
                                           ])

    def test_pubmed(self):
        '''Test parsing database info returned by EInfo
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.einfo(db="pubmed")
        handle = open('Entrez/einfo2.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record["DbInfo"]['DbName'], 'pubmed')
        self.assertEqual(record["DbInfo"]['MenuName'], 'PubMed')
        self.assertEqual(record["DbInfo"]['Description'], 'PubMed bibliographic record')
        self.assertEqual(record["DbInfo"]['Count'], "17905967")
        self.assertEqual(record["DbInfo"]['LastUpdate'], '2008/04/15 06:42')

        self.assertEqual(len(record["DbInfo"]['FieldList']), 40)

        self.assertEqual(record["DbInfo"]['FieldList'][0]['Name'], 'ALL')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['FullName'], 'All Fields')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['Description'], 'All terms from all searchable fields')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['TermCount'], "70792830")
        self.assertEqual(record["DbInfo"]['FieldList'][0]['IsDate'], 'N')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['IsNumerical'], 'N')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['SingleToken'], 'N')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['Hierarchy'], 'N')
        self.assertEqual(record["DbInfo"]['FieldList'][0]['IsHidden'], 'N')

        self.assertEqual(len(record["DbInfo"]['LinkList']), 46)

        self.assertEqual(record["DbInfo"]['LinkList'][0]['Name'], 'pubmed_books_refs')
        self.assertEqual(record["DbInfo"]['LinkList'][0]['Menu'], 'Cited in Books')
        self.assertEqual(record["DbInfo"]['LinkList'][0]['Description'], 'PubMed links associated with Books')
        self.assertEqual(record["DbInfo"]['LinkList'][0]['DbTo'], 'books')

class ESearchTest(unittest.TestCase):
    '''Tests for parsing XML output returned by ESearch
    '''
    def test_pubmed1(self):
        '''Test parsing XML returned by ESearch from PubMed (first test)
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="pubmed", term="biopython")
        handle = open('Entrez/esearch1.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '5')
        self.assertEqual(record['RetMax'], '5')
        self.assertEqual(record['RetStart'], '0')
        self.assertEqual(len(record['IdList']), 5)
        self.assertEqual(record['IdList'][0], '16403221')
        self.assertEqual(record['IdList'][1], '16377612')
        self.assertEqual(record['IdList'][2], '14871861')
        self.assertEqual(record['IdList'][3], '14630660')
        self.assertEqual(record['IdList'][4], '12230038')
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(len(record['TranslationStack']), 2)
        self.assertEqual(record['TranslationStack'][0]['Term'], 'biopython[All Fields]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'All Fields')
        self.assertEqual(record['TranslationStack'][0]['Count'], '5')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1], 'GROUP')
        self.assertEqual(record['QueryTranslation'], 'biopython[All Fields]')

    def test_pubmed2(self):
        '''Test parsing XML returned by ESearch from PubMed (second test)
        '''
        # Search in PubMed for the term cancer for the entrez date from
        # the last 60 days and retrieve the first 100 IDs and translations
        # using the history parameter. 
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="pubmed", term="cancer", reldate=60,
        #                        datetype="edat", retmax=100, usehistory="y")
        handle = open('Entrez/esearch2.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], "10238")
        self.assertEqual(record['RetMax'], "100")
        self.assertEqual(record['RetStart'], "0")
        self.assertEqual(record['QueryKey'], '12')
        self.assertEqual(record['WebEnv'], '0rYFb69LfbTFXfG7-0HPo2BU-ZFWF1s_51WtYR5e0fAzThQCR0WIW12inPQRRIj1xUzSfGgG9ovT9-@263F6CC86FF8F760_0173SID')
        self.assertEqual(len(record['IdList']), 100)
        self.assertEqual(record['IdList'][0], '18411453')
        self.assertEqual(record['IdList'][1], '18411431')
        self.assertEqual(record['IdList'][2], '18411430')
        self.assertEqual(record['IdList'][3], '18411429')
        self.assertEqual(record['IdList'][4], '18411428')
        self.assertEqual(record['IdList'][5], '18411402')
        self.assertEqual(record['IdList'][6], '18411381')
        self.assertEqual(record['IdList'][7], '18411373')
        self.assertEqual(record['IdList'][8], '18411372')
        self.assertEqual(record['IdList'][9], '18411371')
        self.assertEqual(record['IdList'][10], '18411370')
        self.assertEqual(record['IdList'][11], '18411367')
        self.assertEqual(record['IdList'][12], '18411306')
        self.assertEqual(record['IdList'][13], '18411292')
        self.assertEqual(record['IdList'][14], '18411277')
        self.assertEqual(record['IdList'][15], '18411260')
        self.assertEqual(record['IdList'][16], '18411234')
        self.assertEqual(record['IdList'][17], '18411200')
        self.assertEqual(record['IdList'][18], '18411199')
        self.assertEqual(record['IdList'][19], '18411198')
        self.assertEqual(record['IdList'][20], '18411197')
        self.assertEqual(record['IdList'][21], '18411195')
        self.assertEqual(record['IdList'][22], '18411194')
        self.assertEqual(record['IdList'][23], '18411193')
        self.assertEqual(record['IdList'][24], '18411192')
        self.assertEqual(record['IdList'][25], '18411191')
        self.assertEqual(record['IdList'][26], '18411052')
        self.assertEqual(record['IdList'][27], '18411048')
        self.assertEqual(record['IdList'][28], '18411046')
        self.assertEqual(record['IdList'][29], '18411019')
        self.assertEqual(record['IdList'][30], '18411018')
        self.assertEqual(record['IdList'][31], '18411017')
        self.assertEqual(record['IdList'][32], '18411015')
        self.assertEqual(record['IdList'][33], '18411014')
        self.assertEqual(record['IdList'][34], '18411011')
        self.assertEqual(record['IdList'][35], '18411010')
        self.assertEqual(record['IdList'][36], '18411005')
        self.assertEqual(record['IdList'][37], '18411003')
        self.assertEqual(record['IdList'][38], '18411001')
        self.assertEqual(record['IdList'][39], '18411000')
        self.assertEqual(record['IdList'][40], '18410999')
        self.assertEqual(record['IdList'][41], '18410998')
        self.assertEqual(record['IdList'][42], '18410997')
        self.assertEqual(record['IdList'][43], '18410995')
        self.assertEqual(record['IdList'][44], '18410977')
        self.assertEqual(record['IdList'][45], '18410975')
        self.assertEqual(record['IdList'][46], '18410966')
        self.assertEqual(record['IdList'][47], '18410954')
        self.assertEqual(record['IdList'][48], '18410953')
        self.assertEqual(record['IdList'][49], '18410934')
        self.assertEqual(record['IdList'][50], '18410925')
        self.assertEqual(record['IdList'][51], '18410903')
        self.assertEqual(record['IdList'][52], '18410826')
        self.assertEqual(record['IdList'][53], '18410739')
        self.assertEqual(record['IdList'][54], '18410720')
        self.assertEqual(record['IdList'][55], '18410716')
        self.assertEqual(record['IdList'][56], '18410709')
        self.assertEqual(record['IdList'][57], '18410705')
        self.assertEqual(record['IdList'][58], '18410692')
        self.assertEqual(record['IdList'][59], '18410690')
        self.assertEqual(record['IdList'][60], '18410634')
        self.assertEqual(record['IdList'][61], '18410618')
        self.assertEqual(record['IdList'][62], '18410610')
        self.assertEqual(record['IdList'][63], '18410593')
        self.assertEqual(record['IdList'][64], '18410587')
        self.assertEqual(record['IdList'][65], '18410567')
        self.assertEqual(record['IdList'][66], '18410539')
        self.assertEqual(record['IdList'][67], '18410530')
        self.assertEqual(record['IdList'][68], '18410528')
        self.assertEqual(record['IdList'][69], '18410461')
        self.assertEqual(record['IdList'][70], '18410455')
        self.assertEqual(record['IdList'][71], '18410444')
        self.assertEqual(record['IdList'][72], '18410443')
        self.assertEqual(record['IdList'][73], '18410442')
        self.assertEqual(record['IdList'][74], '18410441')
        self.assertEqual(record['IdList'][75], '18410440')
        self.assertEqual(record['IdList'][76], '18410439')
        self.assertEqual(record['IdList'][77], '18410437')
        self.assertEqual(record['IdList'][78], '18410436')
        self.assertEqual(record['IdList'][79], '18410435')
        self.assertEqual(record['IdList'][80], '18410431')
        self.assertEqual(record['IdList'][81], '18410430')
        self.assertEqual(record['IdList'][82], '18410428')
        self.assertEqual(record['IdList'][83], '18410427')
        self.assertEqual(record['IdList'][84], '18410405')
        self.assertEqual(record['IdList'][85], '18410404')
        self.assertEqual(record['IdList'][86], '18410355')
        self.assertEqual(record['IdList'][87], '18410327')
        self.assertEqual(record['IdList'][88], '18410312')
        self.assertEqual(record['IdList'][89], '18410311')
        self.assertEqual(record['IdList'][90], '18410307')
        self.assertEqual(record['IdList'][91], '18410259')
        self.assertEqual(record['IdList'][92], '18410249')
        self.assertEqual(record['IdList'][93], '18410245')
        self.assertEqual(record['IdList'][94], '18410243')
        self.assertEqual(record['IdList'][95], '18410242')
        self.assertEqual(record['IdList'][96], '18410060')
        self.assertEqual(record['IdList'][97], '18410013')
        self.assertEqual(record['IdList'][98], '18409992')
        self.assertEqual(record['IdList'][99], '18409991')
        self.assertEqual(len(record['TranslationSet']), 1)
        self.assertEqual(record['TranslationSet'][0]['From'], 'cancer')
        self.assertEqual(record['TranslationSet'][0]['To'], '(("neoplasms"[TIAB] NOT Medline[SB]) OR "neoplasms"[MeSH Terms] OR cancer[Text Word])')
        self.assertEqual(len(record['TranslationStack']), 13)
        self.assertEqual(record['TranslationStack'][0]['Term'], '"neoplasms"[TIAB]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'TIAB')
        self.assertEqual(record['TranslationStack'][0]['Count'], "52104")
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1]['Term'], 'Medline[SB]')
        self.assertEqual(record['TranslationStack'][1]['Field'], 'SB')
        self.assertEqual(record['TranslationStack'][1]['Count'], "16509514")
        self.assertEqual(record['TranslationStack'][1]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][2], 'NOT')
        self.assertEqual(record['TranslationStack'][3], 'GROUP')
        self.assertEqual(record['TranslationStack'][4]['Term'], '"neoplasms"[MeSH Terms]')
        self.assertEqual(record['TranslationStack'][4]['Field'], 'MeSH Terms')
        self.assertEqual(record['TranslationStack'][4]['Count'], "1918010")
        self.assertEqual(record['TranslationStack'][4]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][5], 'OR')
        self.assertEqual(record['TranslationStack'][6]['Term'], 'cancer[Text Word]')
        self.assertEqual(record['TranslationStack'][6]['Field'], 'Text Word')
        self.assertEqual(record['TranslationStack'][6]['Count'], "638849")
        self.assertEqual(record['TranslationStack'][6]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][7], 'OR')
        self.assertEqual(record['TranslationStack'][8], 'GROUP')
        self.assertEqual(record['TranslationStack'][9]['Term'], '2008/02/16[EDAT]')
        self.assertEqual(record['TranslationStack'][9]['Field'], 'EDAT')
        self.assertEqual(record['TranslationStack'][9]['Count'], "-1")
        self.assertEqual(record['TranslationStack'][9]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][10]['Term'], '2008/04/16[EDAT]')
        self.assertEqual(record['TranslationStack'][10]['Field'], 'EDAT')
        self.assertEqual(record['TranslationStack'][10]['Count'], "-1")
        self.assertEqual(record['TranslationStack'][10]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][11], 'RANGE')
        self.assertEqual(record['TranslationStack'][12], 'AND')
        self.assertEqual(record['QueryTranslation'], '(("neoplasms"[TIAB] NOT Medline[SB]) OR "neoplasms"[MeSH Terms] OR cancer[Text Word]) AND 2008/02/16[EDAT] : 2008/04/16[EDAT]')

    def test_pubmed3(self):
        '''Test parsing XML returned by ESearch from PubMed (third test)
        '''
        # Search in PubMed for the journal PNAS Volume 97, and retrieve
        # 6 IDs starting at ID 7.
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="pubmed", term="PNAS[ta] AND 97[vi]",
        #                        retstart=6, retmax=6)
        handle = open('Entrez/esearch3.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '2652')
        self.assertEqual(record['RetMax'], '6')
        self.assertEqual(record['RetStart'], '6')
        self.assertEqual(len(record['IdList']), 6)
        self.assertEqual(record['IdList'][0], '11121077')
        self.assertEqual(record['IdList'][1], '11121076')
        self.assertEqual(record['IdList'][2], '11121075')
        self.assertEqual(record['IdList'][3], '11121074')
        self.assertEqual(record['IdList'][4], '11121073')
        self.assertEqual(record['IdList'][5], '11121072')
        self.assertEqual(len(record['TranslationSet']), 1)
        self.assertEqual(record['TranslationSet'][0]['From'], 'PNAS[ta]')
        self.assertEqual(record['TranslationSet'][0]['To'], '"Proc Natl Acad Sci U S A"[Journal:__jrid6653]')
        self.assertEqual(len(record['TranslationStack']), 3)
        self.assertEqual(record['TranslationStack'][0]['Term'], '"Proc Natl Acad Sci U S A"[Journal]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'Journal')
        self.assertEqual(record['TranslationStack'][0]['Count'], '91806')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1]['Term'], '97[vi]')
        self.assertEqual(record['TranslationStack'][1]['Field'], 'vi')
        self.assertEqual(record['TranslationStack'][1]['Count'], '58681')
        self.assertEqual(record['TranslationStack'][1]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][2], 'AND')
        self.assertEqual(record['QueryTranslation'], '"Proc Natl Acad Sci U S A"[Journal] AND 97[vi]')

    def test_journals(self):
        '''Test parsing XML returned by ESearch from the Journals database
        '''
        # Search in Journals for the term obstetrics.
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="journals", term="obstetrics")
        handle = open('Entrez/esearch4.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '177')
        self.assertEqual(record['RetMax'], '20')
        self.assertEqual(record['RetStart'], '0')
        self.assertEqual(len(record['IdList']), 20)
        self.assertEqual(record['IdList'][0], '75')
        self.assertEqual(record['IdList'][1], '138')
        self.assertEqual(record['IdList'][2], '136')
        self.assertEqual(record['IdList'][3], '137')
        self.assertEqual(record['IdList'][4], '139')
        self.assertEqual(record['IdList'][5], '140')
        self.assertEqual(record['IdList'][6], '355')
        self.assertEqual(record['IdList'][7], '354')
        self.assertEqual(record['IdList'][8], '27731')
        self.assertEqual(record['IdList'][9], '439')
        self.assertEqual(record['IdList'][10], '564')
        self.assertEqual(record['IdList'][11], '617')
        self.assertEqual(record['IdList'][12], '749')
        self.assertEqual(record['IdList'][13], '735')
        self.assertEqual(record['IdList'][14], '815')
        self.assertEqual(record['IdList'][15], '905')
        self.assertEqual(record['IdList'][16], '903')
        self.assertEqual(record['IdList'][17], '932')
        self.assertEqual(record['IdList'][18], '933')
        self.assertEqual(record['IdList'][19], '875')
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(len(record['TranslationStack']), 2)
        self.assertEqual(record['TranslationStack'][0]['Term'], 'obstetrics[All Fields]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'All Fields')
        self.assertEqual(record['TranslationStack'][0]['Count'], '177')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][0].tag, "TermSet")
        self.assertEqual(record['TranslationStack'][1], 'GROUP')
        self.assertEqual(record['TranslationStack'][1].tag, "OP")
        self.assertEqual(record['QueryTranslation'], 'obstetrics[All Fields]')

    def test_pmc(self):
        '''Test parsing XML returned by ESearch from PubMed Central
        '''
        # Search in PubMed Central for stem cells in free fulltext articles.
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="pmc",
        #                        term="stem cells AND free fulltext[filter]")
        handle = open('Entrez/esearch5.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '23492')
        self.assertEqual(record['RetMax'], '20')
        self.assertEqual(record['RetStart'], '0')
        self.assertEqual(len(record['IdList']), 20)
        self.assertEqual(record['IdList'][0], '1894783')
        self.assertEqual(record['IdList'][1], '2064507')
        self.assertEqual(record['IdList'][2], '520747')
        self.assertEqual(record['IdList'][3], '2043120')
        self.assertEqual(record['IdList'][4], '2118723')
        self.assertEqual(record['IdList'][5], '1815228')
        self.assertEqual(record['IdList'][6], '1253596')
        self.assertEqual(record['IdList'][7], '2077853')
        self.assertEqual(record['IdList'][8], '1308908')
        self.assertEqual(record['IdList'][9], '2233634')
        self.assertEqual(record['IdList'][10], '556262')
        self.assertEqual(record['IdList'][11], '1925137')
        self.assertEqual(record['IdList'][12], '1860068')
        self.assertEqual(record['IdList'][13], '1626529')
        self.assertEqual(record['IdList'][14], '2217616')
        self.assertEqual(record['IdList'][15], '1584276')
        self.assertEqual(record['IdList'][16], '2000702')
        self.assertEqual(record['IdList'][17], '186324')
        self.assertEqual(record['IdList'][18], '1959362')
        self.assertEqual(record['IdList'][19], '1413911')
        self.assertEqual(len(record['TranslationSet']), 1)
        self.assertEqual(record['TranslationSet'][0]['From'], 'stem cells')
        self.assertEqual(record['TranslationSet'][0]['To'], '("stem cells"[MeSH Terms] OR stem cells[Acknowledgments] OR stem cells[Figure/Table Caption] OR stem cells[Section Title] OR stem cells[Body - All Words] OR stem cells[Title] OR stem cells[Abstract])')
        self.assertEqual(len(record['TranslationStack']), 16)
        self.assertEqual(record['TranslationStack'][0]['Term'], '"stem cells"[MeSH Terms]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'MeSH Terms')
        self.assertEqual(record['TranslationStack'][0]['Count'], '12224')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1]['Term'], 'stem cells[Acknowledgments]')
        self.assertEqual(record['TranslationStack'][1]['Field'], 'Acknowledgments')
        self.assertEqual(record['TranslationStack'][1]['Count'], '79')
        self.assertEqual(record['TranslationStack'][1]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][2], 'OR')
        self.assertEqual(record['TranslationStack'][3]['Term'], 'stem cells[Figure/Table Caption]')
        self.assertEqual(record['TranslationStack'][3]['Field'], 'Figure/Table Caption')
        self.assertEqual(record['TranslationStack'][3]['Count'], '806')
        self.assertEqual(record['TranslationStack'][3]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][4], 'OR')
        self.assertEqual(record['TranslationStack'][5]['Term'], 'stem cells[Section Title]')
        self.assertEqual(record['TranslationStack'][5]['Field'], 'Section Title')
        self.assertEqual(record['TranslationStack'][5]['Count'], '522')
        self.assertEqual(record['TranslationStack'][5]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][6], 'OR')
        self.assertEqual(record['TranslationStack'][7]['Term'], 'stem cells[Body - All Words]')
        self.assertEqual(record['TranslationStack'][7]['Field'], 'Body - All Words')
        self.assertEqual(record['TranslationStack'][7]['Count'], '13936')
        self.assertEqual(record['TranslationStack'][7]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][8], 'OR')
        self.assertEqual(record['TranslationStack'][9]['Term'], 'stem cells[Title]')
        self.assertEqual(record['TranslationStack'][9]['Field'], 'Title')
        self.assertEqual(record['TranslationStack'][9]['Count'], '1005')
        self.assertEqual(record['TranslationStack'][9]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][10], 'OR')
        self.assertEqual(record['TranslationStack'][11]['Term'], 'stem cells[Abstract]')
        self.assertEqual(record['TranslationStack'][11]['Field'], 'Abstract')
        self.assertEqual(record['TranslationStack'][11]['Count'], '2503')
        self.assertEqual(record['TranslationStack'][11]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][12], 'OR')
        self.assertEqual(record['TranslationStack'][13], 'GROUP')
        self.assertEqual(record['TranslationStack'][14]['Term'], 'free fulltext[filter]')
        self.assertEqual(record['TranslationStack'][14]['Field'], 'filter')
        self.assertEqual(record['TranslationStack'][14]['Count'], '1412839')
        self.assertEqual(record['TranslationStack'][14]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][15], 'AND')
        self.assertEqual(record['QueryTranslation'], '("stem cells"[MeSH Terms] OR stem cells[Acknowledgments] OR stem cells[Figure/Table Caption] OR stem cells[Section Title] OR stem cells[Body - All Words] OR stem cells[Title] OR stem cells[Abstract]) AND free fulltext[filter]')

    def test_nucleotide(self):
        '''Test parsing XML returned by ESearch from the Nucleotide database
        '''
        # Search in Nucleotide for a property of the sequence,
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="nucleotide", term="biomol trna[prop]")
        handle = open('Entrez/esearch6.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], "699")
        self.assertEqual(record['RetMax'], "20")
        self.assertEqual(record['RetStart'], "0")
        self.assertEqual(len(record['IdList']), 20)
        self.assertEqual(record['IdList'][0], '220161')
        self.assertEqual(record['IdList'][1], '220160')
        self.assertEqual(record['IdList'][2], '220159')
        self.assertEqual(record['IdList'][3], '220263')
        self.assertEqual(record['IdList'][4], '220162')
        self.assertEqual(record['IdList'][5], '159885659')
        self.assertEqual(record['IdList'][6], '156572228')
        self.assertEqual(record['IdList'][7], '2648075')
        self.assertEqual(record['IdList'][8], '287595')
        self.assertEqual(record['IdList'][9], '402544')
        self.assertEqual(record['IdList'][10], '402506')
        self.assertEqual(record['IdList'][11], '402505')
        self.assertEqual(record['IdList'][12], '287594')
        self.assertEqual(record['IdList'][13], '287593')
        self.assertEqual(record['IdList'][14], '287592')
        self.assertEqual(record['IdList'][15], '287591')
        self.assertEqual(record['IdList'][16], '287590')
        self.assertEqual(record['IdList'][17], '287589')
        self.assertEqual(record['IdList'][18], '287588')
        self.assertEqual(record['IdList'][19], '287587')
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(record['QueryTranslation'], '')

    def test_protein(self):
        '''Test parsing XML returned by ESearch from the Protein database
        '''
        # Search in Protein for a molecular weight
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="protein", term="200020[molecular weight]")
        handle = open('Entrez/esearch7.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '3')
        self.assertEqual(record['RetMax'], '3')
        self.assertEqual(record['RetStart'], '0')
        self.assertEqual(len(record['IdList']), 3)
        self.assertEqual(record['IdList'][0], '16766766')
        self.assertEqual(record['IdList'][1], '16422035')
        self.assertEqual(record['IdList'][2], '4104812')
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(len(record['TranslationStack']), 2)
        self.assertEqual(record['TranslationStack'][0]['Term'], '000200020[molecular weight]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'molecular weight')
        self.assertEqual(record['TranslationStack'][0]['Count'], '3')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1], 'GROUP')
        self.assertEqual(record['QueryTranslation'], '000200020[molecular weight]')

    def test_notfound(self):
        '''Test parsing XML returned by ESearch when no items were found
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="protein", term="abcXYZ")
        handle = open('Entrez/esearch8.xml')
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], '3')
        self.assertEqual(record['RetMax'], '3')
        self.assertEqual(record['RetStart'], '0')
        self.assertEqual(len(record['IdList']), 3)
        self.assertEqual(record['IdList'][0], '16766766')
        self.assertEqual(record['IdList'][1], '16422035')
        self.assertEqual(record['IdList'][2], '4104812')
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(len(record['TranslationStack']), 2)
        self.assertEqual(record['TranslationStack'][0]['Term'], '000200020[molecular weight]')
        self.assertEqual(record['TranslationStack'][0]['Field'], 'molecular weight')
        self.assertEqual(record['TranslationStack'][0]['Count'], '3')
        self.assertEqual(record['TranslationStack'][0]['Explode'], 'Y')
        self.assertEqual(record['TranslationStack'][1], 'GROUP')
        self.assertEqual(record['QueryTranslation'], '000200020[molecular weight]')

    def test_notfound(self):
        '''Test parsing XML returned by ESearch when no items were found
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.esearch(db="protein", term="abcXYZ")
        handle = open('Entrez/esearch8.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record['Count'], "0")
        self.assertEqual(record['RetMax'], "0")
        self.assertEqual(record['RetStart'], "0")
        self.assertEqual(len(record['IdList']), 0)
        self.assertEqual(len(record['TranslationSet']), 0)
        self.assertEqual(record['QueryTranslation'], '')
        self.assertEqual(len(record['ErrorList']), 2)
        self.assertTrue("PhraseNotFound" in record['ErrorList'])
        self.assertTrue("FieldNotFound" in record['ErrorList'])
        self.assertEqual(len(record['ErrorList']["PhraseNotFound"]), 1)
        self.assertEqual(len(record['ErrorList']["FieldNotFound"]), 0)
        self.assertEqual(record['ErrorList']["PhraseNotFound"][0], "abcXYZ")
        self.assertEqual(len(record['WarningList']), 3)
        self.assertTrue("PhraseIgnored" in record['WarningList'])
        self.assertTrue("QuotedPhraseNotFound" in record['WarningList'])
        self.assertTrue("OutputMessage" in record['WarningList'])
        self.assertEqual(len(record['WarningList']["PhraseIgnored"]), 0)
        self.assertEqual(len(record['WarningList']["QuotedPhraseNotFound"]), 0)
        self.assertEqual(len(record['WarningList']["OutputMessage"]), 1)
        self.assertEqual(record['WarningList']["OutputMessage"][0], "No items found.")

class EPostTest(unittest.TestCase):
    '''Tests for parsing XML output returned by EPost
    '''
    # Don't know how to get an InvalidIdList in the XML returned by EPost;
    # unable to test if we are parsing it correctly.
    def test_epost(self):
        '''Test parsing XML returned by EPost
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.epost(db="pubmed", id="11237011")
        handle = open('Entrez/epost1.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record["QueryKey"],  '1')
        self.assertEqual(record["WebEnv"], '0zYsuLk3zG_lRMkblPBEqnT8nIENUGw4HAy8xXChTnoVm7GEnWY71jv3nz@1FC077F3806DE010_0042SID')

    def test_wrong(self):
        '''Test parsing XML returned by EPost with incorrect arguments
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.epost(db="nothing")
        handle = open('Entrez/epost2.xml', "rb")
        self.assertRaises(RuntimeError, Entrez.read, handle)

    def test_invalid(self):
        '''Test parsing XML returned by EPost with an invalid id (overflow tag)
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.epost(db="pubmed", id=99999999999999999999999999999999)
        handle = open('Entrez/epost3.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record["InvalidIdList"], ["-1"])
        self.assertEqual(record["QueryKey"], "1")
        self.assertEqual(record["WebEnv"], "08AIUeBsfIk6BfdzKnd3GM2RtCudczC9jm5aeb4US0o7azCTQCeCsr-xg0@1EDE54E680D03C40_0011SID")


class ESummaryTest(unittest.TestCase):
    '''Tests for parsing XML output returned by ESummary
    '''
    # Items have a type, which can be
    # (Integer|Date|String|Structure|List|Flags|Qualifier|Enumerator|Unknown)
    # I don't have an XML file where the type "Flags", "Qualifier",
    # "Enumerator", or "Unknown" is used, so they are not tested here.
    def test_pubmed(self):
        '''Test parsing XML returned by ESummary from PubMed
        '''
        # In PubMed display records for PMIDs 11850928 and 11482001 in
        # xml retrieval mode
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="pubmed", id=["11850928","11482001"],
        #                         retmode="xml")
        handle = open('Entrez/esummary1.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record[0]["Id"], "11850928")
        self.assertEqual(record[0]["PubDate"], "1965 Aug")
        self.assertEqual(record[0]["EPubDate"], "")
        self.assertEqual(record[0]["Source"], "Arch Dermatol")
        self.assertEqual(len(record[0]["AuthorList"]), 2)
        self.assertEqual(record[0]["AuthorList"][0], "LoPresti PJ")
        self.assertEqual(record[0]["AuthorList"][1], "Hambrick GW Jr")
        self.assertEqual(record[0]["LastAuthor"], "Hambrick GW Jr")
        self.assertEqual(record[0]["Title"], "Zirconium granuloma following treatment of rhus dermatitis.")
        self.assertEqual(record[0]["Volume"], "92")
        self.assertEqual(record[0]["Issue"], "2")
        self.assertEqual(record[0]["Pages"], "188-91")
        self.assertEqual(record[0]["LangList"], ["English"])
        self.assertEqual(record[0]["NlmUniqueID"], "0372433")
        self.assertEqual(record[0]["ISSN"], "0003-987X")
        self.assertEqual(record[0]["ESSN"], "1538-3652")
        self.assertEqual(len(record[0]["PubTypeList"]), 1)
        self.assertEqual(record[0]["PubTypeList"][0], "Journal Article")
        self.assertEqual(record[0]["RecordStatus"], "PubMed - indexed for MEDLINE")
        self.assertEqual(record[0]["PubStatus"], "ppublish")
        self.assertEqual(len(record[0]["ArticleIds"]), 2)
        self.assertEqual(record[0]["ArticleIds"]["pubmed"], ["11850928"])
        self.assertEqual(record[0]["ArticleIds"]["medline"], [])
        self.assertEqual(len(record[0]["History"]), 2)
        self.assertEqual(record[0]["History"]["pubmed"], ["1965/08/01 00:00"])
        self.assertEqual(record[0]["History"]["medline"], ["2002/03/09 10:01"])
        self.assertEqual(len(record[0]["References"]), 0)
        self.assertEqual(record[0]["HasAbstract"], 1)
        self.assertEqual(record[0]["PmcRefCount"], 0)
        self.assertEqual(record[0]["FullJournalName"], "Archives of dermatology")
        self.assertEqual(record[0]["ELocationID"], "")
        self.assertEqual(record[0]["SO"], "1965 Aug;92(2):188-91")

        self.assertEqual(record[1]["Id"], "11482001")
        self.assertEqual(record[1]["PubDate"], "2001 Jun")
        self.assertEqual(record[1]["EPubDate"], "")
        self.assertEqual(record[1]["Source"], "Adverse Drug React Toxicol Rev")
        self.assertEqual(len(record[1]["AuthorList"]), 3)
        self.assertEqual(record[1]["AuthorList"][0], "Mantle D")
        self.assertEqual(record[1]["AuthorList"][1], "Gok MA")
        self.assertEqual(record[1]["AuthorList"][2], "Lennard TW")
        self.assertEqual(record[1]["LastAuthor"], "Lennard TW")
        self.assertEqual(record[1]["Title"], "Adverse and beneficial effects of plant extracts on skin and skin disorders.")
        self.assertEqual(record[1]["Volume"], "20")
        self.assertEqual(record[1]["Issue"], "2")
        self.assertEqual(record[1]["Pages"], "89-103")
        self.assertEqual(len(record[1]["LangList"]), 1)
        self.assertEqual(record[1]["LangList"][0], "English")
        self.assertEqual(record[1]["NlmUniqueID"], "9109474")
        self.assertEqual(record[1]["ISSN"], "0964-198X")
        self.assertEqual(record[1]["ESSN"], "")
        self.assertEqual(len(record[1]["PubTypeList"]), 2)
        self.assertEqual(record[1]["PubTypeList"][0], "Journal Article")
        self.assertEqual(record[1]["PubTypeList"][1], "Review")
        self.assertEqual(record[1]["RecordStatus"], "PubMed - indexed for MEDLINE")
        self.assertEqual(record[1]["PubStatus"], "ppublish")
        self.assertEqual(len(record[1]["ArticleIds"]), 2)
        self.assertEqual(record[1]["ArticleIds"]["pubmed"], ["11482001"])
        self.assertEqual(record[1]["ArticleIds"]["medline"], [])
        self.assertEqual(len(record[1]["History"]), 2)
        self.assertEqual(record[1]["History"]["pubmed"], ["2001/08/03 10:00"])
        self.assertEqual(record[1]["History"]["medline"], ["2002/01/23 10:01"])
        self.assertEqual(len(record[1]["References"]), 0)
        self.assertEqual(record[1]["HasAbstract"], 1)
        self.assertEqual(record[1]["PmcRefCount"], 0)
        self.assertEqual(record[1]["FullJournalName"], "Adverse drug reactions and toxicological reviews")
        self.assertEqual(record[1]["ELocationID"], "")
        self.assertEqual(record[1]["SO"], "2001 Jun;20(2):89-103")

    def test_journals(self):
        '''Test parsing XML returned by ESummary from the Journals database
        '''
        # In Journals display records for journal IDs 27731,439,735,905 
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="journals", id="27731,439,735,905")
        handle = open('Entrez/esummary2.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record[0]["Id"], "27731")
        self.assertEqual(record[0]["Title"], "The American journal of obstetrics and diseases of women and children")
        self.assertEqual(record[0]["MedAbbr"], "Am J Obstet Dis Women Child")
        self.assertEqual(record[0]["IsoAbbr"], "")
        self.assertEqual(record[0]["NlmId"], "14820330R")
        self.assertEqual(record[0]["pISSN"], "0894-5543")
        self.assertEqual(record[0]["eISSN"], "")
        self.assertEqual(record[0]["PublicationStartYear"], "1868")
        self.assertEqual(record[0]["PublicationEndYear"], "1919")
        self.assertEqual(record[0]["Publisher"], "W.A. Townsend & Adams, $c [1868-1919]")
        self.assertEqual(record[0]["Language"], "eng")
        self.assertEqual(record[0]["Country"], "United States")
        self.assertEqual(len(record[0]["BroadHeading"]), 0)
        self.assertEqual(record[0]["ContinuationNotes"], "")

        self.assertEqual(record[1]["Id"], "439")
        self.assertEqual(record[1]["Title"], "American journal of obstetrics and gynecology")
        self.assertEqual(record[1]["MedAbbr"], "Am J Obstet Gynecol")
        self.assertEqual(record[1]["IsoAbbr"], "Am. J. Obstet. Gynecol.")
        self.assertEqual(record[1]["NlmId"], "0370476")
        self.assertEqual(record[1]["pISSN"], "0002-9378")
        self.assertEqual(record[1]["eISSN"], "1097-6868")
        self.assertEqual(record[1]["PublicationStartYear"], "1920")
        self.assertEqual(record[1]["PublicationEndYear"], "")
        self.assertEqual(record[1]["Publisher"], "Elsevier,")
        self.assertEqual(record[1]["Language"], "eng")
        self.assertEqual(record[1]["Country"], "United States")
        self.assertEqual(len(record[1]["BroadHeading"]), 2)
        self.assertEqual(record[1]["BroadHeading"][0], "Gynecology")
        self.assertEqual(record[1]["BroadHeading"][1], "Obstetrics")
        self.assertEqual(record[1]["ContinuationNotes"], "Continues: American journal of obstetrics and diseases of women and children. ")

        self.assertEqual(record[2]["Id"], "735")
        self.assertEqual(record[2]["Title"], "Archives of gynecology and obstetrics")
        self.assertEqual(record[2]["MedAbbr"], "Arch Gynecol Obstet")
        self.assertEqual(record[2]["IsoAbbr"], "Arch. Gynecol. Obstet.")
        self.assertEqual(record[2]["NlmId"], "8710213")
        self.assertEqual(record[2]["pISSN"], "0932-0067")
        self.assertEqual(record[2]["eISSN"], "1432-0711")
        self.assertEqual(record[2]["PublicationStartYear"], "1987")
        self.assertEqual(record[2]["PublicationEndYear"], "")
        self.assertEqual(record[2]["Publisher"], "Springer Verlag")
        self.assertEqual(record[2]["Language"], "eng")
        self.assertEqual(record[2]["Country"], "Germany")
        self.assertEqual(len(record[2]["BroadHeading"]), 2)
        self.assertEqual(record[2]["BroadHeading"][0], "Gynecology")
        self.assertEqual(record[2]["BroadHeading"][1], "Obstetrics")
        self.assertEqual(record[2]["ContinuationNotes"], "Continues: Archives of gynecology. ")

        self.assertEqual(record[3]["Id"], "905")
        self.assertEqual(record[3]["Title"], "Asia-Oceania journal of obstetrics and gynaecology / AOFOG")
        self.assertEqual(record[3]["MedAbbr"], "Asia Oceania J Obstet Gynaecol")
        self.assertEqual(record[3]["IsoAbbr"], "")
        self.assertEqual(record[3]["NlmId"], "8102781")
        self.assertEqual(record[3]["pISSN"], "0389-2328")
        self.assertEqual(record[3]["eISSN"], "")
        self.assertEqual(record[3]["PublicationStartYear"], "1980")
        self.assertEqual(record[3]["PublicationEndYear"], "1994")
        self.assertEqual(record[3]["Publisher"], "University Of Tokyo Press")
        self.assertEqual(record[3]["Language"], "eng")
        self.assertEqual(record[3]["Country"], "Japan")
        self.assertEqual(len(record[3]["BroadHeading"]), 2)
        self.assertEqual(record[3]["BroadHeading"][0], "Gynecology")
        self.assertEqual(record[3]["BroadHeading"][1], "Obstetrics")
        self.assertEqual(record[3]["ContinuationNotes"], "Continues: Journal of the Asian Federation of Obstetrics and Gynaecology. Continued by: Journal of obstetrics and gynaecology (Tokyo, Japan). ")

    def test_protein(self):
        '''Test parsing XML returned by ESummary from the Protein database
        '''
        # In Protein display records for GIs 28800982 and 28628843 in xml retrieval mode
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="protein", id="28800982,28628843", retmode="xml")
        handle = open('Entrez/esummary3.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["Id"], "28800982")
        self.assertEqual(record[0]["Caption"], "AAO47091")
        self.assertEqual(record[0]["Title"], "hemochromatosis [Homo sapiens]")
        self.assertEqual(record[0]["Extra"], "gi|28800982|gb|AAO47091.1|[28800982]")
        self.assertEqual(record[0]["Gi"], 28800982)
        self.assertEqual(record[0]["CreateDate"], "2003/03/03")
        self.assertEqual(record[0]["UpdateDate"], "2003/03/03")
        self.assertEqual(record[0]["Flags"], 0)
        self.assertEqual(record[0]["TaxId"], 9606)
        self.assertEqual(record[0]["Length"], 268)
        self.assertEqual(record[0]["Status"], "live")
        self.assertEqual(record[0]["ReplacedBy"], "")
        self.assertEqual(record[0]["Comment"], "  ")

        self.assertEqual(record[1]["Id"], "28628843")
        self.assertEqual(record[1]["Caption"], "AAO49381")
        self.assertEqual(record[1]["Title"], "erythroid associated factor [Homo sapiens]")
        self.assertEqual(record[1]["Extra"], "gi|28628843|gb|AAO49381.1|AF485325_1[28628843]")
        self.assertEqual(record[1]["Gi"],  28628843)
        self.assertEqual(record[1]["CreateDate"], "2003/03/02")
        self.assertEqual(record[1]["UpdateDate"], "2003/03/02")
        self.assertEqual(record[1]["Flags"], 0)
        self.assertEqual(record[1]["TaxId"], 9606)
        self.assertEqual(record[1]["Length"], 102)
        self.assertEqual(record[1]["Status"], "live")
        self.assertEqual(record[1]["ReplacedBy"], "")
        self.assertEqual(record[1]["Comment"], "  ")

    def test_nucleotide(self):
        '''Test parsing XML returned by ESummary from the Nucleotide database
        '''
        # In Nucleotide display records for GIs 28864546 and 28800981
        # in xml retrieval mode
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="nucleotide", id="28864546,28800981",
        #                         retmode="xml")
        handle = open('Entrez/esummary4.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["Id"], "28864546")
        self.assertEqual(record[0]["Caption"], "AY207443")
        self.assertEqual(record[0]["Title"], "Homo sapiens alpha hemoglobin (HBZP) pseudogene 3' UTR/AluJo repeat breakpoint junction")
        self.assertEqual(record[0]["Extra"], "gi|28864546|gb|AY207443.1|[28864546]")
        self.assertEqual(record[0]["Gi"], 28864546)
        self.assertEqual(record[0]["CreateDate"], "2003/03/05")
        self.assertEqual(record[0]["UpdateDate"], "2003/03/05")
        self.assertEqual(record[0]["Flags"], 0)
        self.assertEqual(record[0]["TaxId"], 9606)
        self.assertEqual(record[0]["Length"], 491)
        self.assertEqual(record[0]["Status"], "live")
        self.assertEqual(record[0]["ReplacedBy"], "")
        self.assertEqual(record[0]["Comment"], "  ")

        self.assertEqual(record[1]["Id"], "28800981")
        self.assertEqual(record[1]["Caption"], "AY205604")
        self.assertEqual(record[1]["Title"], "Homo sapiens hemochromatosis (HFE) mRNA, partial cds")
        self.assertEqual(record[1]["Extra"], "gi|28800981|gb|AY205604.1|[28800981]")
        self.assertEqual(record[1]["Gi"], 28800981)
        self.assertEqual(record[1]["CreateDate"], "2003/03/03")
        self.assertEqual(record[1]["UpdateDate"], "2003/03/03")
        self.assertEqual(record[1]["Flags"], 0)
        self.assertEqual(record[1]["TaxId"], 9606)
        self.assertEqual(record[1]["Length"], 860)
        self.assertEqual(record[1]["Status"], "live")
        self.assertEqual(record[1]["ReplacedBy"], "")
        self.assertEqual(record[1]["Comment"], "  ")

    def test_structure(self):
        '''Test parsing XML returned by ESummary from the Structure database
        '''
        # In Nucleotide display records for GIs 28864546 and 28800981
        # in xml retrieval mode
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="structure", id=["19923","12120"],
        #                         retmode="xml")
        handle = open('Entrez/esummary5.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record[0]["Id"], "19923")
        self.assertEqual(record[0]["PdbAcc"], "1L5J")
        self.assertEqual(record[0]["PdbDescr"], "Crystal Structure Of E. Coli Aconitase B")
        self.assertEqual(record[0]["EC"], "4.2.1.3")
        self.assertEqual(record[0]["Resolution"], "2.4")
        self.assertEqual(record[0]["ExpMethod"], "X-Ray Diffraction")
        self.assertEqual(record[0]["PdbClass"], "Lyase")
        self.assertEqual(record[0]["PdbReleaseDate"], "2007/8/27")
        self.assertEqual(record[0]["PdbDepositDate"], "2002/3/7")
        self.assertEqual(record[0]["DepositDate"], "2007/10/25")
        self.assertEqual(record[0]["ModifyDate"], "2007/10/25")
        self.assertEqual(record[0]["LigCode"], "F3S|TRA")
        self.assertEqual(record[0]["LigCount"], "2")
        self.assertEqual(record[0]["ModProteinResCount"], "0")
        self.assertEqual(record[0]["ModDNAResCount"], "0")
        self.assertEqual(record[0]["ModRNAResCount"], "0")
        self.assertEqual(record[0]["ProteinChainCount"], "2")
        self.assertEqual(record[0]["DNAChainCount"], "0")
        self.assertEqual(record[0]["RNAChainCount"], "0")

        self.assertEqual(record[1]["Id"], "12120")
        self.assertEqual(record[1]["PdbAcc"], "1B0K")
        self.assertEqual(record[1]["PdbDescr"], "S642a:fluorocitrate Complex Of Aconitase")
        self.assertEqual(record[1]["EC"], "4.2.1.3")
        self.assertEqual(record[1]["Resolution"], "2.5")
        self.assertEqual(record[1]["ExpMethod"], "X-Ray Diffraction")
        self.assertEqual(record[1]["PdbClass"], "Lyase")
        self.assertEqual(record[1]["PdbReleaseDate"], "2007/8/27")
        self.assertEqual(record[1]["PdbDepositDate"], "1998/11/11")
        self.assertEqual(record[1]["DepositDate"], "2007/10/07")
        self.assertEqual(record[1]["ModifyDate"], "2007/10/07")
        self.assertEqual(record[1]["LigCode"], "FLC|O|SF4")
        self.assertEqual(record[1]["LigCount"], "3")
        self.assertEqual(record[1]["ModProteinResCount"], "0")
        self.assertEqual(record[1]["ModDNAResCount"], "0")
        self.assertEqual(record[1]["ModRNAResCount"], "0")
        self.assertEqual(record[1]["ProteinChainCount"], "1")
        self.assertEqual(record[1]["DNAChainCount"], "0")
        self.assertEqual(record[1]["RNAChainCount"], "0")

    def test_taxonomy(self):
        '''Test parsing XML returned by ESummary from the Taxonomy database
        '''
        # In Taxonomy display records for TAXIDs 9913 and 30521 in
        # xml retrieval mode 
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="taxonomy", id=["9913","30521"],
        #                         retmode="xml")
        handle = open('Entrez/esummary6.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record[0]["Id"], "9913")
        self.assertEqual(record[0]["Rank"], "species")
        self.assertEqual(record[0]["Division"], "even-toed ungulates")
        self.assertEqual(record[0]["ScientificName"], "Bos taurus")
        self.assertEqual(record[0]["CommonName"], "cattle")
        self.assertEqual(record[0]["TaxId"], 9913)
        self.assertEqual(record[0]["NucNumber"], 2264214)
        self.assertEqual(record[0]["ProtNumber"], 55850)
        self.assertEqual(record[0]["StructNumber"], 1517)
        self.assertEqual(record[0]["GenNumber"], 31)
        self.assertEqual(record[0]["GeneNumber"], 29651)
        self.assertEqual(record[0]["Genus"], "")
        self.assertEqual(record[0]["Species"], "")
        self.assertEqual(record[0]["Subsp"], "")

        self.assertEqual(record[1]["Id"], "30521")
        self.assertEqual(record[1]["Rank"], "species")
        self.assertEqual(record[1]["Division"], "even-toed ungulates")
        self.assertEqual(record[1]["ScientificName"], "Bos grunniens")
        self.assertEqual(record[1]["CommonName"], "domestic yak")
        self.assertEqual(record[1]["TaxId"], 30521)
        self.assertEqual(record[1]["NucNumber"], 560)
        self.assertEqual(record[1]["ProtNumber"], 254)
        self.assertEqual(record[1]["StructNumber"], 0)
        self.assertEqual(record[1]["GenNumber"], 1)
        self.assertEqual(record[1]["GeneNumber"], 13)
        self.assertEqual(record[1]["Genus"], "")
        self.assertEqual(record[1]["Species"], "")
        self.assertEqual(record[1]["Subsp"], "")

    def test_unists(self):
        '''Test parsing XML returned by ESummary from the UniSTS database
        '''
        # In UniSTS display records for IDs 254085 and 254086 in xml
        # retrieval mode 
        # To create the XML file, use
        # >>> Bio.Entrez.esummary(db="unists", id=["254085","254086"],
        #                         retmode="xml")
        handle = open('Entrez/esummary7.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["Id"], "254085")
        self.assertEqual(record[0]["Marker_Name"], "SE234324")
        self.assertEqual(len(record[0]["Map_Gene_Summary_List"]), 1)
        self.assertEqual(record[0]["Map_Gene_Summary_List"][0]["Org"], "Sus scrofa")
        self.assertEqual(record[0]["Map_Gene_Summary_List"][0]["Chr"], " chromosome 7")
        self.assertEqual(record[0]["Map_Gene_Summary_List"][0]["Locus"], "")
        self.assertEqual(record[0]["EPCR_Summary"], "Found by e-PCR in sequences from Sus scrofa.")
        self.assertEqual(record[0]["LocusId"], "")

        self.assertEqual(record[1]["Id"], "254086")
        self.assertEqual(record[1]["Marker_Name"], "SE259162")
        self.assertEqual(len(record[1]["Map_Gene_Summary_List"]), 1)
        self.assertEqual(record[1]["Map_Gene_Summary_List"][0]["Org"], "Sus scrofa")
        self.assertEqual(record[1]["Map_Gene_Summary_List"][0]["Chr"], " chromosome 12")
        self.assertEqual(record[1]["Map_Gene_Summary_List"][0]["Locus"], "")
        self.assertEqual(record[1]["EPCR_Summary"], "Found by e-PCR in sequences from Sus scrofa.")
        self.assertEqual(record[1]["LocusId"], "")

    def test_wrong(self):
        '''Test parsing XML returned by ESummary with incorrect arguments
        '''
        # To create the XML file, use
        # >>> Bio.Entrez.esummary()
        handle = open('Entrez/esummary8.xml', "rb")
        self.assertRaises(RuntimeError, Entrez.read, handle)


class ELinkTest(unittest.TestCase):
    '''Tests for parsing XML output returned by ELink
    '''
    def test_pubmed1(self):
        '''Test parsing pubmed links returned by ELink (first test)
        '''
        # Retrieve IDs from PubMed for PMID 9298984 to the PubMed database
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="9298984", cmd="neighbor")
        handle = open('Entrez/elink1.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(record[0]["IdList"], ["9298984"])
        self.assertEqual(record[0]["LinkSetDb"][0]["DbTo"], "pubmed")
        self.assertEqual(record[0]["LinkSetDb"][0]["LinkName"], "pubmed_pubmed")
        self.assertEqual(len(record[0]["LinkSetDb"][0]["Link"]), 144)
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Id"], "9298984")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Score"], "2147483647")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Id"], "8794856")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Score"], "65259341")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Id"], "9700164")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Score"], "60347327")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Id"], "7914521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Score"], "54343405")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Id"], "1339459")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Score"], "53014422")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Id"], "9914369")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Score"], "52741538")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Id"], "11590237")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Score"], "52493903")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Id"], "12686595")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Score"], "48734007")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Id"], "9074495")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Score"], "48220447")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Id"], "11146659")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Score"], "46604626")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Id"], "10893249")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Score"], "46254167")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Id"], "8978614")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Score"], "46166362")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Id"], "15371539")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Score"], "45060488")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Id"], "10806105")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Score"], "44825774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Id"], "10402457")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Score"], "44338522")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Id"], "10545493")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Score"], "43860609")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Id"], "10523511")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Score"], "43268800")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Id"], "12515822")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Score"], "43215343")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Id"], "15915585")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Score"], "43029760")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Id"], "11483958")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Score"], "42348877")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Id"], "11685532")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Score"], "42262104")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Id"], "9869638")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Score"], "41143593")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Id"], "12080088")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Score"], "40849490")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Id"], "12034769")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Score"], "40841328")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Id"], "9852156")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Score"], "40793501")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Id"], "9735366")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Score"], "40661605")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Id"], "10749938")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Score"], "40486739")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Id"], "9490715")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Score"], "40311339")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Id"], "9425896")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Score"], "40056298")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Id"], "11266459")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Score"], "39883140")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Id"], "14522947")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Score"], "39683976")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Id"], "15616189")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Score"], "39518630")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Id"], "16732327")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Score"], "39425668")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Id"], "11179694")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Score"], "39183565")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Id"], "10898791")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Score"], "39159761")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Id"], "11146661")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Score"], "39116609")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Id"], "11914278")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Score"], "39028004")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Id"], "10985388")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Score"], "39002572")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Id"], "16839185")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Score"], "38916726")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Id"], "7585942")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Score"], "38747288")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Id"], "2022189")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Score"], "38717145")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Id"], "7690762")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Score"], "38647275")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Id"], "7904902")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Score"], "38557343")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Id"], "9378750")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Score"], "38484849")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Id"], "12388768")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Score"], "38454422")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Id"], "11352945")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Score"], "38449836")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Id"], "11267866")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Score"], "38419058")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Id"], "17222555")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Score"], "38368546")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Id"], "11252055")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Score"], "38257516")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Id"], "16585270")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Score"], "37800856")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Id"], "9606208")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Score"], "37669054")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Id"], "17182852")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Score"], "37621285")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Id"], "9933569")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Score"], "37398470")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Id"], "15268859")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Score"], "37340582")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Id"], "12235289")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Score"], "37247450")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Id"], "16741559")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Score"], "37198716")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Id"], "11266451")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Score"], "37142542")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Id"], "15075237")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Score"], "36897578")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Id"], "15485811")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Score"], "36804297")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Id"], "14699129")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Score"], "36782062")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Id"], "16510521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Score"], "36724370")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Id"], "15824131")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Score"], "36695341")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Id"], "15371340")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Score"], "36685694")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Id"], "9878245")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Score"], "36684230")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Id"], "10398680")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Score"], "36573411")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Id"], "16516834")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Score"], "36525654")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Id"], "11715021")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Score"], "36518129")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Id"], "14622138")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Score"], "36496009")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Id"], "11092768")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Score"], "36457186")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Id"], "12514103")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Score"], "36385909")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Id"], "17525528")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Score"], "36316439")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Id"], "11402064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Score"], "36172957")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Id"], "9258677")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Score"], "35989143")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Id"], "14499625")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Score"], "35978627")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Id"], "10428958")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Score"], "35924800")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Id"], "14972679")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Score"], "35915578")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Id"], "9396743")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Score"], "35883749")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Id"], "16219694")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Score"], "35870689")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Id"], "11369198")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Score"], "35838048")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Id"], "17333235")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Score"], "35815282")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Id"], "11102811")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Score"], "35783566")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Id"], "10207147")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Score"], "35594009")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Id"], "10477755")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Score"], "35589601")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Id"], "10747094")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Score"], "35548072")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Id"], "15215209")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Score"], "35526869")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Id"], "11157774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Score"], "35510607")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Id"], "10669599")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Score"], "35462246")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Id"], "17448445")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Score"], "35398470")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Id"], "17878237")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Score"], "35231311")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Id"], "10411903")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Score"], "35202708")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Id"], "12773390")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Score"], "35171743")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Id"], "12498686")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Score"], "35131906")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Id"], "9009204")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Score"], "34993776")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Id"], "17576797")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Score"], "34988639")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Id"], "10225945")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Score"], "34950419")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Id"], "11161560")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Score"], "34912466")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Id"], "11967147")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Score"], "34900540")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Id"], "14711415")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Score"], "34883714")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Id"], "2211824")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Score"], "34843507")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Id"], "15737064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Score"], "34828187")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Id"], "7720068")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Score"], "34811182")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Id"], "9472001")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Score"], "34799321")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Id"], "11792803")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Score"], "34697393")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Id"], "11386760")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Score"], "34684610")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Id"], "15094189")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Score"], "34684021")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Id"], "9763420")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Score"], "34666950")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Id"], "10629219")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Score"], "34422925")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Id"], "11238410")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Score"], "34318521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Id"], "17199038")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Score"], "34255594")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Id"], "12944469")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Score"], "34249507")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Id"], "15616192")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Score"], "34110517")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Id"], "11146660")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Score"], "34063257")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Id"], "11402066")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Score"], "34012520")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Id"], "6791901")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Score"], "33311119")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Id"], "7172865")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Score"], "32934223")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Id"], "8270646")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Score"], "32898701")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Id"], "1916263")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Score"], "32707765")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Id"], "7588080")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Score"], "32503526")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Id"], "7391142")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Score"], "31632645")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Id"], "6793236")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Score"], "31522175")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Id"], "2512302")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Score"], "30339372")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Id"], "7720069")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Score"], "30024525")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Id"], "8257792")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Score"], "29834355")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Id"], "3417141")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Score"], "27920818")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Id"], "3315496")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Score"], "27422009")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Id"], "1993311")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Score"], "26763828")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Id"], "6185450")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Score"], "26100420")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Id"], "1819515")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Score"], "26036804")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Id"], "7250964")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Score"], "25738652")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Id"], "8489280")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Score"], "25587858")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Id"], "7096444")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Score"], "24642544")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Id"], "348629")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Score"], "24432498")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Id"], "2275018")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Score"], "23077593")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Id"], "1747872")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Score"], "22933494")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Id"], "3547036")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Score"], "22925639")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Id"], "18291669")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Score"], "22762310")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Id"], "1576878")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Score"], "20846041")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Id"], "6230555")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Score"], "19354488")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Id"], "7627547")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Score"], "18940607")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Id"], "17678444")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Score"], "18834135")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Id"], "3366468")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Score"], "14831756")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Id"], "1959920")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Score"], "14156414")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Id"], "13242628")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Score"], "12584732")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Id"], "17248312")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Score"], "7610436")

    def test_nucleotide(self):
        '''Test parsing Nucleotide to Protein links returned by ELink
        '''
        # Retrieve IDs from Nucleotide for GI  48819, 7140345 to Protein
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="nucleotide", db="protein",
        #                      id="48819,7140345")
        handle = open('Entrez/elink2.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["DbFrom"], "nucleotide")
        self.assertEqual(record[0]["IdList"], ["48819", "7140345"])

    def test_pubmed2(self):
        '''Test parsing pubmed links returned by ELink (second test)
        '''
        # Retrieve PubMed related articles for PMIDs 11812492 11774222
        # with a publication date from 1995 to the present
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="11812492,11774222",
        #                      db="pubmed", mindate="1995", datetype="pdat")
        handle = open('Entrez/elink3.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(len(record[0]['IdList']), 2)
        self.assertEqual(record[0]['IdList'][0], "11812492")
        self.assertEqual(record[0]['IdList'][1], "11774222")
        self.assertEqual(record[0]["LinkSetDb"][0]["DbTo"], "pubmed")
        self.assertEqual(record[0]["LinkSetDb"][0]["LinkName"], "pubmed_pubmed")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Id"], "11812492")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Score"], "2147483647")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Id"], "11774222")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Score"], "2147483647")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Id"], "11668631")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Score"], "86345306")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Id"], "15111095")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Score"], "81604359")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Id"], "10731564")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Score"], "65665112")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Id"], "15780005")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Score"], "62251079")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Id"], "17885136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Score"], "50322134")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Id"], "17470297")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Score"], "49148434")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Id"], "16005284")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Score"], "49035508")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Id"], "10856373")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Score"], "48363137")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Id"], "15383292")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Score"], "48347159")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Id"], "17040125")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Score"], "48301243")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Id"], "10770808")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Score"], "47696325")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Id"], "11125122")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Score"], "45889695")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Id"], "15287587")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Score"], "45599733")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Id"], "15839745")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Score"], "44650620")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Id"], "10612825")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Score"], "44445812")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Id"], "15024419")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Score"], "44075047")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Id"], "12743802")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Score"], "43873158")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Id"], "15238684")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Score"], "43856864")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Id"], "12386340")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Score"], "43770229")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Id"], "16269725")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Score"], "43712594")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Id"], "10592273")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Score"], "43640108")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Id"], "15383308")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Score"], "42835474")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Id"], "15676075")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Score"], "42272663")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Id"], "11774221")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Score"], "42058380")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Id"], "10592272")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Score"], "41719917")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Id"], "15997407")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Score"], "41535461")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Id"], "15774024")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Score"], "41351079")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Id"], "11233160")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Score"], "41268965")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Id"], "14702162")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Score"], "41147661")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Id"], "16616613")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Score"], "41073100")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Id"], "17202370")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Score"], "40819600")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Id"], "15478601")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Score"], "40578911")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Id"], "15322925")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Score"], "40548101")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Id"], "11472559")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Score"], "40508356")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Id"], "11925998")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Score"], "39844751")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Id"], "12372145")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Score"], "39809277")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Id"], "17562224")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Score"], "38850094")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Id"], "15037105")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Score"], "38758229")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Id"], "14998511")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Score"], "38608049")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Id"], "10092480")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Score"], "38410463")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Id"], "7729881")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Score"], "38329800")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Id"], "12933853")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Score"], "37881850")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Id"], "16818783")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Score"], "37835096")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Id"], "16406333")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Score"], "37775136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Id"], "11472553")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Score"], "37750745")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Id"], "11403387")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Score"], "37707525")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Id"], "17306254")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Score"], "37685833")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Id"], "11516587")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Score"], "37620966")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Id"], "9274032")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Score"], "37528832")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Id"], "12856318")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Score"], "37484650")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Id"], "14695526")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Score"], "37429895")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Id"], "12481045")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Score"], "37051674")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Id"], "11752345")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Score"], "36875760")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Id"], "12467974")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Score"], "36787103")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Id"], "11214099")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Score"], "36710749")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Id"], "14638788")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Score"], "36667774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Id"], "16278157")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Score"], "36598908")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Id"], "11752242")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Score"], "36555638")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Id"], "14681474")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Score"], "36317853")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Id"], "15944077")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Score"], "36264027")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Id"], "12625936")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Score"], "36088314")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Id"], "16672453")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Score"], "35985060")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Id"], "14695451")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Score"], "35971708")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Id"], "12402526")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Score"], "35942170")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Id"], "10592200")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Score"], "35932875")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Id"], "17584494")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Score"], "35869907")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Id"], "17761848")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Score"], "35868206")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Id"], "16697384")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Score"], "35792791")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Id"], "8784774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Score"], "35787497")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Id"], "18000556")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Score"], "35701408")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Id"], "15828434")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Score"], "35604052")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Id"], "10511685")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Score"], "35598319")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Id"], "15608284")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Score"], "35439627")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Id"], "11125071")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Score"], "35414962")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Id"], "11791238")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Score"], "35411948")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Id"], "15710433")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Score"], "35197152")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Id"], "16164550")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Score"], "35172458")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Id"], "17697334")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Score"], "35121478")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Id"], "12537121")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Score"], "35054632")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Id"], "12860672")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Score"], "35046651")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Id"], "15630619")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Score"], "35034076")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Id"], "15125639")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Score"], "35007338")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Id"], "11443570")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Score"], "34935553")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Id"], "12208043")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Score"], "34923107")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Id"], "11731507")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Score"], "34875290")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Id"], "11988510")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Score"], "34773036")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Id"], "11125038")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Score"], "34754724")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Id"], "16381944")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Score"], "34747225")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Id"], "17135206")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Score"], "34735015")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Id"], "17099226")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Score"], "34698054")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Id"], "15608233")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Score"], "34588400")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Id"], "16672057")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Score"], "34583177")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Id"], "15687015")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Score"], "34357840")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Id"], "10782070")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Score"], "34326746")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Id"], "14970722")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Score"], "34217911")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Id"], "18027007")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Score"], "34185436")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Id"], "12387845")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Score"], "34083368")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Id"], "16237012")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Score"], "34070163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Id"], "16351742")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Score"], "33775198")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Id"], "12203989")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Score"], "33759170")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Id"], "15474306")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Score"], "33737675")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Id"], "15270538")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Score"], "33697306")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Id"], "17518759")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Score"], "33695140")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Id"], "16085497")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Score"], "33652537")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Id"], "16423288")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Score"], "33564554")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Id"], "16251775")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Score"], "33547325")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Id"], "12632152")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Score"], "33497998")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Id"], "11269648")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Score"], "33493800")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Id"], "16103603")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Score"], "33378796")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Id"], "12816546")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Score"], "33316167")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Id"], "10221636")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Score"], "33310814")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Id"], "16381973")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Score"], "33236048")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Id"], "15977173")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Score"], "33222497")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Id"], "16351753")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Score"], "33205084")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Id"], "15317790")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Score"], "33195439")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Id"], "17135198")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Score"], "33189951")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Id"], "12701381")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Score"], "33172200")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Id"], "12203988")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Score"], "33172077")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Id"], "11456466")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Score"], "33124900")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Id"], "16936055")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Score"], "33081742")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Id"], "17183477")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Score"], "33005068")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Id"], "9455480")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Score"], "32997067")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Id"], "12490454")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Score"], "32995041")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Id"], "12435493")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Score"], "32990122")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Id"], "11038309")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Score"], "32977663")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Id"], "10366827")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Score"], "32903347")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Id"], "10466136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Score"], "32869387")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Id"], "16381840")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Score"], "32816923")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Id"], "11825250")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Score"], "32776183")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Id"], "12234534")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Score"], "32708547")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Id"], "14624247")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Score"], "32708542")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Id"], "12886019")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Score"], "32653276")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Id"], "12041732")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Score"], "32607185")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Id"], "15336912")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Score"], "32596453")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Id"], "12652910")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Score"], "32567397")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Id"], "14681353")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Score"], "32549157")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Id"], "12586873")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Score"], "32504063")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Id"], "11481430")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Score"], "32462602")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Id"], "15254259")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Score"], "32441737")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Id"], "16873516")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Score"], "32433603")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Id"], "17170002")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Score"], "32425626")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Id"], "12519941")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Score"], "32367760")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][144]["Id"], "11197770")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][144]["Score"], "32362623")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][145]["Id"], "11240843")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][145]["Score"], "32347064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][146]["Id"], "11328780")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][146]["Score"], "32333807")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][147]["Id"], "11875041")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][147]["Score"], "32312036")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][148]["Id"], "11752243")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][148]["Score"], "32268199")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][149]["Id"], "16907992")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][149]["Score"], "32247019")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][150]["Id"], "15046636")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][150]["Score"], "32214942")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][151]["Id"], "10592169")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][151]["Score"], "32137798")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][152]["Id"], "17919582")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][152]["Score"], "32137767")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][153]["Id"], "18025705")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][153]["Score"], "32131322")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][154]["Id"], "11029673")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][154]["Score"], "32126363")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][155]["Id"], "9047337")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][155]["Score"], "32090163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][156]["Id"], "11080372")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][156]["Score"], "31924475")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][157]["Id"], "18045790")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][157]["Score"], "31834367")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][158]["Id"], "10215019")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][158]["Score"], "31823989")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][159]["Id"], "14706096")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][159]["Score"], "31781977")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][160]["Id"], "17537593")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][160]["Score"], "31771566")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][161]["Id"], "12819149")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][161]["Score"], "31683943")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][162]["Id"], "17880721")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][162]["Score"], "31630816")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][163]["Id"], "14681478")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][163]["Score"], "31620457")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][164]["Id"], "11985867")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][164]["Score"], "31544318")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][165]["Id"], "15608248")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][165]["Score"], "31542256")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][166]["Id"], "17401150")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][166]["Score"], "31497289")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][167]["Id"], "10359795")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][167]["Score"], "31460779")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][168]["Id"], "15608286")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][168]["Score"], "31435112")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][169]["Id"], "15774022")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][169]["Score"], "31425851")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][170]["Id"], "9921679")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][170]["Score"], "31396086")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][171]["Id"], "17038195")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][171]["Score"], "31380822")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][172]["Id"], "15491544")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][172]["Score"], "31294370")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][173]["Id"], "10469257")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][173]["Score"], "31291548")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][174]["Id"], "15487498")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][174]["Score"], "31268351")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][175]["Id"], "15383303")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][175]["Score"], "31264596")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][176]["Id"], "15643605")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][176]["Score"], "31259953")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][177]["Id"], "16418238")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][177]["Score"], "31259003")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][178]["Id"], "15500248")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][178]["Score"], "31252080")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][179]["Id"], "15479945")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][179]["Score"], "31249988")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][180]["Id"], "16962738")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][180]["Score"], "31249405")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][181]["Id"], "15094394")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][181]["Score"], "31200337")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][182]["Id"], "11758285")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][182]["Score"], "31180435")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][183]["Id"], "15723693")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][183]["Score"], "31083464")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][184]["Id"], "16710453")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][184]["Score"], "31083136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][185]["Id"], "15311460")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][185]["Score"], "31068402")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][186]["Id"], "16549670")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][186]["Score"], "30995148")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][187]["Id"], "18180957")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][187]["Score"], "30973190")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][188]["Id"], "14681351")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][188]["Score"], "30968930")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][189]["Id"], "10902212")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][189]["Score"], "30960861")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][190]["Id"], "15357877")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][190]["Score"], "30947680")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][191]["Id"], "12356773")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][191]["Score"], "30910321")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][192]["Id"], "17537669")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][192]["Score"], "30893205")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][193]["Id"], "16551372")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][193]["Score"], "30889080")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][194]["Id"], "15231810")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][194]["Score"], "30863616")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][195]["Id"], "12819150")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][195]["Score"], "30847027")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][196]["Id"], "15608257")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][196]["Score"], "30840234")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][197]["Id"], "17384426")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][197]["Score"], "30827754")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][198]["Id"], "15811532")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][198]["Score"], "30823185")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][199]["Id"], "10612821")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][199]["Score"], "30822187")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][200]["Id"], "17062145")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][200]["Score"], "30813605")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][201]["Id"], "11355885")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][201]["Score"], "30810648")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][202]["Id"], "15746365")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][202]["Score"], "30784209")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][203]["Id"], "16282300")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][203]["Score"], "30782807")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][204]["Id"], "15546336")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][204]["Score"], "30773578")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][205]["Id"], "11741630")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][205]["Score"], "30764995")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][206]["Id"], "15980532")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][206]["Score"], "30735790")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][207]["Id"], "12519977")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][207]["Score"], "30707395")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][208]["Id"], "12436197")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][208]["Score"], "30705501")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][209]["Id"], "11125059")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][209]["Score"], "30614888")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][210]["Id"], "11163442")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][210]["Score"], "30550965")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][211]["Id"], "12519964")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][211]["Score"], "30518025")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][212]["Id"], "12083398")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][212]["Score"], "30466595")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][213]["Id"], "11908756")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][213]["Score"], "30462080")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][214]["Id"], "15608226")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][214]["Score"], "30335152")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][215]["Id"], "16845091")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][215]["Score"], "30277120")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][216]["Id"], "17338820")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][216]["Score"], "30208452")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][217]["Id"], "10407783")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][217]["Score"], "30171504")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][218]["Id"], "17130148")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][218]["Score"], "30160136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][219]["Id"], "14681471")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][219]["Score"], "30155757")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][220]["Id"], "17445272")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][220]["Score"], "30015229")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][221]["Id"], "11279516")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][221]["Score"], "29947199")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][222]["Id"], "17221864")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][222]["Score"], "29893674")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][223]["Id"], "15827081")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][223]["Score"], "29891924")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][224]["Id"], "11222582")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][224]["Score"], "29878915")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][225]["Id"], "11384164")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][225]["Score"], "29871698")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][226]["Id"], "17877839")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][226]["Score"], "29843765")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][227]["Id"], "17151077")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][227]["Score"], "29841695")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][228]["Id"], "16381974")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][228]["Score"], "29740312")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][229]["Id"], "10592263")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][229]["Score"], "29633946")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][230]["Id"], "15608212")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][230]["Score"], "29621479")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][231]["Id"], "9847217")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][231]["Score"], "29618439")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][232]["Id"], "17142236")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][232]["Score"], "29577611")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][233]["Id"], "17059604")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][233]["Score"], "29569767")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][234]["Id"], "16845079")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][234]["Score"], "29506663")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][235]["Id"], "14727153")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][235]["Score"], "29368276")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][236]["Id"], "18045498")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][236]["Score"], "29364312")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][237]["Id"], "17185755")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][237]["Score"], "29331905")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][238]["Id"], "18025704")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][238]["Score"], "29323161")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][239]["Id"], "15215374")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][239]["Score"], "29306559")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][240]["Id"], "17135185")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][240]["Score"], "29236297")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][241]["Id"], "10466135")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][241]["Score"], "29231855")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][242]["Id"], "17148475")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][242]["Score"], "29229044")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][243]["Id"], "15657101")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][243]["Score"], "29209567")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][244]["Id"], "14681490")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][244]["Score"], "29189708")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][245]["Id"], "15714328")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][245]["Score"], "29183488")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][246]["Id"], "14960477")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][246]["Score"], "29040531")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][247]["Id"], "11015564")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][247]["Score"], "29011368")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][248]["Id"], "18064491")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][248]["Score"], "28956740")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][249]["Id"], "12734009")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][249]["Score"], "28950064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][250]["Id"], "17094804")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][250]["Score"], "28906953")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][251]["Id"], "17908294")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][251]["Score"], "28897717")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][252]["Id"], "16176584")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][252]["Score"], "28874470")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][253]["Id"], "14715089")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][253]["Score"], "28763886")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][254]["Id"], "14681408")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][254]["Score"], "28697827")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][255]["Id"], "14594716")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][255]["Score"], "28686075")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][256]["Id"], "16528802")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][256]["Score"], "28644452")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][257]["Id"], "16010002")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][257]["Score"], "28637570")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][258]["Id"], "17430565")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][258]["Score"], "28635513")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][259]["Id"], "16452787")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][259]["Score"], "28631832")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][260]["Id"], "11197127")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][260]["Score"], "28619225")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][261]["Id"], "8682188")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][261]["Score"], "28592521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][262]["Id"], "12519940")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][262]["Score"], "28573991")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][263]["Id"], "17121775")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][263]["Score"], "28448726")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][264]["Id"], "16371163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][264]["Score"], "28373394")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][265]["Id"], "15300845")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][265]["Score"], "28338477")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][266]["Id"], "15248903")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][266]["Score"], "28323328")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][267]["Id"], "11319266")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][267]["Score"], "28293166")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][268]["Id"], "16336665")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][268]["Score"], "28231249")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][269]["Id"], "14681350")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][269]["Score"], "28227327")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][270]["Id"], "16216831")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][270]["Score"], "28224610")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][271]["Id"], "15494741")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][271]["Score"], "28190925")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][272]["Id"], "17088289")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][272]["Score"], "28168901")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][273]["Id"], "17099235")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][273]["Score"], "28159766")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][274]["Id"], "15215372")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][274]["Score"], "28129693")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][275]["Id"], "9169870")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][275]["Score"], "28117392")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][276]["Id"], "10077537")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][276]["Score"], "27911205")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][277]["Id"], "18172929")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][277]["Score"], "27885172")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][278]["Id"], "9571806")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][278]["Score"], "27841468")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][279]["Id"], "11752280")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][279]["Score"], "27795833")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][280]["Id"], "11414208")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][280]["Score"], "27725996")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][281]["Id"], "9298642")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][281]["Score"], "27716027")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][282]["Id"], "18073380")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][282]["Score"], "27437383")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][283]["Id"], "14527308")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][283]["Score"], "27332641")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][284]["Id"], "9847220")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][284]["Score"], "27083894")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][285]["Id"], "10413661")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][285]["Score"], "27073030")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][286]["Id"], "10407677")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][286]["Score"], "26907635")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][287]["Id"], "11244060")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][287]["Score"], "26897688")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][288]["Id"], "10227170")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][288]["Score"], "26766431")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][289]["Id"], "8719164")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][289]["Score"], "26515360")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][290]["Id"], "18359019")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][290]["Score"], "26225983")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][291]["Id"], "10511680")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][291]["Score"], "26031196")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][292]["Id"], "9884329")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][292]["Score"], "25992564")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][293]["Id"], "17827295")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][293]["Score"], "25989152")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][294]["Id"], "10899154")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][294]["Score"], "25843128")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][295]["Id"], "11668619")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][295]["Score"], "25822950")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][296]["Id"], "18386064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][296]["Score"], "25702942")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][297]["Id"], "11092731")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][297]["Score"], "25618899")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][298]["Id"], "9520376")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][298]["Score"], "25549761")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][299]["Id"], "11756688")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][299]["Score"], "25440634")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][300]["Id"], "10737802")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][300]["Score"], "25362744")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][301]["Id"], "9879937")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][301]["Score"], "25277089")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][302]["Id"], "17822801")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][302]["Score"], "25252984")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][303]["Id"], "10965872")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][303]["Score"], "25208185")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][304]["Id"], "10511682")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][304]["Score"], "25183443")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][305]["Id"], "10851186")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][305]["Score"], "25092764")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][306]["Id"], "9775388")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][306]["Score"], "25026910")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][307]["Id"], "10810023")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][307]["Score"], "24904718")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][308]["Id"], "18032438")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][308]["Score"], "24509777")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][309]["Id"], "18377816")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][309]["Score"], "24373788")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][310]["Id"], "11774190")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][310]["Score"], "24185658")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][311]["Id"], "10484179")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][311]["Score"], "24122767")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][312]["Id"], "9625791")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][312]["Score"], "24049917")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][313]["Id"], "11446511")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][313]["Score"], "24048253")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][314]["Id"], "10066467")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][314]["Score"], "23968405")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][315]["Id"], "11783003")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][315]["Score"], "23393870")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][316]["Id"], "10611059")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][316]["Score"], "23255298")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][317]["Id"], "10587943")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][317]["Score"], "23014503")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][318]["Id"], "10612820")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][318]["Score"], "22990878")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][319]["Id"], "9685316")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][319]["Score"], "22771348")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][320]["Id"], "11125121")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][320]["Score"], "22732820")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][321]["Id"], "10075567")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][321]["Score"], "22670427")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][322]["Id"], "11084929")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][322]["Score"], "22397665")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][323]["Id"], "11357826")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][323]["Score"], "22362882")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][324]["Id"], "17983575")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][324]["Score"], "22305320")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][325]["Id"], "11038308")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][325]["Score"], "22115670")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][326]["Id"], "18257289")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][326]["Score"], "22053176")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][327]["Id"], "10419978")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][327]["Score"], "22016184")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][328]["Id"], "9421619")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][328]["Score"], "21957407")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][329]["Id"], "10592198")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][329]["Score"], "21803908")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][330]["Id"], "11483982")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][330]["Score"], "20783817")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][331]["Id"], "11329386")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][331]["Score"], "20223493")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][332]["Id"], "10587942")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][332]["Score"], "20208799")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][333]["Id"], "10810024")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][333]["Score"], "19989188")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][334]["Id"], "11480780")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][334]["Score"], "19974101")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][335]["Id"], "11802378")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][335]["Score"], "19738532")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][336]["Id"], "10610803")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][336]["Score"], "19359100")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][337]["Id"], "10407668")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][337]["Score"], "19070525")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][338]["Id"], "18287701")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][338]["Score"], "19065945")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][339]["Id"], "10963611")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][339]["Score"], "18962273")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][340]["Id"], "10447503")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][340]["Score"], "17406980")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][341]["Id"], "9830540")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][341]["Score"], "17143709")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][342]["Id"], "11462837")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][342]["Score"], "16819799")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][343]["Id"], "10637631")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][343]["Score"], "16390796")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][344]["Id"], "11387032")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][344]["Score"], "15698695")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][345]["Id"], "18365535")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][345]["Score"], "15494816")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][346]["Id"], "15181901")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][346]["Score"], "14385628")

    def test_medline(self):
        '''Test parsing medline indexed articles returned by ELink
        '''
        # Retrieve MEDLINE indexed only related articles for PMID 12242737
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="12242737", db="pubmed",
        #                      term="medline[sb]")
        handle = open('Entrez/elink4.xml', "rb")
        record = Entrez.read(handle)
        
        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(record[0]["IdList"], ["12242737"])
        self.assertEqual(record[0]["LinkSetDb"][0]["DbTo"], "pubmed")
        self.assertEqual(record[0]["LinkSetDb"][0]["LinkName"], "pubmed_pubmed")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Id"], "12242737")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][0]["Score"], "2147483647")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Id"], "11218011")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][1]["Score"], "50825961")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Id"], "11329656")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][2]["Score"], "49822043")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Id"], "9757294")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][3]["Score"], "42645380")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Id"], "9456947")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][4]["Score"], "39871666")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Id"], "17193860")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][5]["Score"], "39717388")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Id"], "11274884")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][6]["Score"], "39233276")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Id"], "12878072")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][7]["Score"], "37748327")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Id"], "11125632")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][8]["Score"], "36227857")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Id"], "12822521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][9]["Score"], "36170366")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Id"], "16999328")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][10]["Score"], "36107139")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Id"], "17875142")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][11]["Score"], "35736802")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Id"], "9510579")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][12]["Score"], "35206779")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Id"], "17354190")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][13]["Score"], "34792954")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Id"], "11702119")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][14]["Score"], "34618984")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Id"], "10024396")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][15]["Score"], "33877753")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Id"], "14650118")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][16]["Score"], "33746160")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Id"], "17243036")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][17]["Score"], "33198930")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Id"], "16580806")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][18]["Score"], "33117197")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Id"], "15278705")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][19]["Score"], "33002826")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Id"], "15236131")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][20]["Score"], "32808406")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Id"], "11368937")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][21]["Score"], "32277701")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Id"], "10688065")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][22]["Score"], "32052850")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Id"], "15635471")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][23]["Score"], "31938251")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Id"], "16357381")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][24]["Score"], "31780147")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Id"], "8153333")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][25]["Score"], "31542202")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Id"], "16284132")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][26]["Score"], "31290577")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Id"], "11329162")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][27]["Score"], "31163088")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Id"], "11973040")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][28]["Score"], "31156707")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Id"], "15143223")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][29]["Score"], "31025329")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Id"], "17040637")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][30]["Score"], "30990506")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Id"], "11016058")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][31]["Score"], "30966482")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Id"], "9317094")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][32]["Score"], "30935529")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Id"], "16133609")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][33]["Score"], "30580027")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Id"], "17325998")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][34]["Score"], "30130533")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Id"], "15505294")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][35]["Score"], "29430378")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Id"], "17268692")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][36]["Score"], "29166153")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Id"], "11329655")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][37]["Score"], "29112282")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Id"], "11775722")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][38]["Score"], "28940754")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Id"], "11907356")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][39]["Score"], "28860163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Id"], "10222515")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][40]["Score"], "28807143")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Id"], "17174054")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][41]["Score"], "28790302")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Id"], "9314960")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][42]["Score"], "28750160")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Id"], "14661661")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][43]["Score"], "28361423")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Id"], "17879696")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][44]["Score"], "28120568")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Id"], "4818442")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][45]["Score"], "28058957")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Id"], "15141648")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][46]["Score"], "28011681")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Id"], "8855688")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][47]["Score"], "27711822")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Id"], "17875143")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][48]["Score"], "27711025")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Id"], "1481295")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][49]["Score"], "27707751")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Id"], "8599783")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][50]["Score"], "27683273")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Id"], "10499696")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][51]["Score"], "27623848")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Id"], "12733684")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][52]["Score"], "27527242")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Id"], "18021675")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][53]["Score"], "27495074")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Id"], "12226761")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][54]["Score"], "27366064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Id"], "4808999")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][55]["Score"], "27304472")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Id"], "16988291")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][56]["Score"], "27295295")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Id"], "10575758")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][57]["Score"], "27243181")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Id"], "8903064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][58]["Score"], "27206664")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Id"], "10811354")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][59]["Score"], "27088219")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Id"], "16096604")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][60]["Score"], "26862979")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Id"], "15788584")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][61]["Score"], "26759584")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Id"], "17376366")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][62]["Score"], "26743241")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Id"], "16566645")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][63]["Score"], "26725076")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Id"], "17259035")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][64]["Score"], "26595433")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Id"], "9314959")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][65]["Score"], "26445900")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Id"], "11895298")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][66]["Score"], "26256774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Id"], "11740602")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][67]["Score"], "26158189")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Id"], "15022983")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][68]["Score"], "25889186")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Id"], "15300544")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][69]["Score"], "25837458")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Id"], "12719915")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][70]["Score"], "25831232")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Id"], "14661306")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][71]["Score"], "25788023")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Id"], "16362812")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][72]["Score"], "25565076")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Id"], "17320773")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][73]["Score"], "25504305")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Id"], "11762248")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][74]["Score"], "25504002")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Id"], "10665303")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][75]["Score"], "25384388")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Id"], "17453494")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][76]["Score"], "25226372")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Id"], "9575723")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][77]["Score"], "25174136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Id"], "12744498")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][78]["Score"], "24971179")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Id"], "12352163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][79]["Score"], "24915990")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Id"], "8290724")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][80]["Score"], "24909462")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Id"], "11973504")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][81]["Score"], "24878058")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Id"], "14661668")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][82]["Score"], "24779779")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Id"], "16552382")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][83]["Score"], "24760919")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Id"], "17709829")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][84]["Score"], "24743292")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Id"], "14528718")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][85]["Score"], "24686212")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Id"], "15008163")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][86]["Score"], "24612994")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Id"], "10051883")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][87]["Score"], "24492331")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Id"], "11027076")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][88]["Score"], "24410525")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Id"], "17543650")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][89]["Score"], "24371825")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Id"], "17658095")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][90]["Score"], "24331965")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Id"], "9193407")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][91]["Score"], "24240252")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Id"], "10578418")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][92]["Score"], "24091226")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Id"], "12592155")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][93]["Score"], "24001341")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Id"], "17157468")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][94]["Score"], "23984321")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Id"], "15094630")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][95]["Score"], "23912874")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Id"], "8794574")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][96]["Score"], "23900764")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Id"], "9125660")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][97]["Score"], "23884352")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Id"], "8819381")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][98]["Score"], "23839719")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Id"], "14661666")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][99]["Score"], "23748510")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Id"], "9658901")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][100]["Score"], "23667126")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Id"], "12744499")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][101]["Score"], "23647189")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Id"], "12164574")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][102]["Score"], "23623853")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Id"], "15136027")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][103]["Score"], "23572558")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Id"], "14872380")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][104]["Score"], "23460906")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Id"], "3905087")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][105]["Score"], "23305022")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Id"], "15642291")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][106]["Score"], "23234831")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Id"], "16928974")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][107]["Score"], "23223298")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Id"], "6072516")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][108]["Score"], "23042548")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Id"], "12949462")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][109]["Score"], "23001441")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Id"], "10761553")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][110]["Score"], "22995991")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Id"], "14661663")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][111]["Score"], "22986720")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Id"], "16338316")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][112]["Score"], "22933288")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Id"], "17464254")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][113]["Score"], "22912253")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Id"], "15529836")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][114]["Score"], "22892154")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Id"], "12361530")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][115]["Score"], "22871698")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Id"], "12876813")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][116]["Score"], "22822515")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Id"], "10749221")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][117]["Score"], "22794373")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Id"], "6482054")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][118]["Score"], "22791927")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Id"], "9016217")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][119]["Score"], "22738432")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Id"], "14702442")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][120]["Score"], "22722123")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Id"], "15279747")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][121]["Score"], "22698787")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Id"], "7892443")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][122]["Score"], "22642038")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Id"], "616459")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][123]["Score"], "22591277")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Id"], "8886718")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][124]["Score"], "22542938")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Id"], "17245521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][125]["Score"], "22538649")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Id"], "1535863")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][126]["Score"], "22468774")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Id"], "15537403")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][127]["Score"], "22458002")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Id"], "16040910")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][128]["Score"], "22452119")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Id"], "16929028")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][129]["Score"], "22433988")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Id"], "16697589")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][130]["Score"], "22366606")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Id"], "531835")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][131]["Score"], "22366454")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Id"], "2308313")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][132]["Score"], "22330898")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Id"], "12522920")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][133]["Score"], "22178764")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Id"], "10222521")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][134]["Score"], "22135023")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Id"], "10499697")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][135]["Score"], "22130302")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Id"], "8903058")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][136]["Score"], "22113132")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Id"], "17441569")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][137]["Score"], "22085858")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Id"], "15284932")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][138]["Score"], "22075791")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Id"], "15466771")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][139]["Score"], "22075418")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Id"], "17145267")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][140]["Score"], "22033864")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Id"], "11329662")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][141]["Score"], "22012948")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Id"], "10222514")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][142]["Score"], "22009777")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Id"], "17383530")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][143]["Score"], "22003600")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][144]["Id"], "12455800")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][144]["Score"], "21992674")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][145]["Id"], "15845051")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][145]["Score"], "21946257")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][146]["Id"], "11443295")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][146]["Score"], "21908841")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][147]["Id"], "15162233")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][147]["Score"], "21903624")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][148]["Id"], "16133610")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][148]["Score"], "21872203")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][149]["Id"], "12845461")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][149]["Score"], "21864314")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][150]["Id"], "16947073")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][150]["Score"], "21832153")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][151]["Id"], "7415301")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][151]["Score"], "21822396")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][152]["Id"], "16416239")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][152]["Score"], "21820165")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][153]["Id"], "4848922")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][153]["Score"], "21786194")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][154]["Id"], "12720164")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][154]["Score"], "21785319")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][155]["Id"], "17093987")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][155]["Score"], "21750370")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][156]["Id"], "16769006")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][156]["Score"], "21735873")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][157]["Id"], "17954835")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][157]["Score"], "21733933")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][158]["Id"], "15236134")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][158]["Score"], "21640099")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][159]["Id"], "12524603")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][159]["Score"], "21636724")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][160]["Id"], "16749985")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][160]["Score"], "21628926")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][161]["Id"], "3213296")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][161]["Score"], "21490232")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][162]["Id"], "11409026")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][162]["Score"], "21061296")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][163]["Id"], "9725288")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][163]["Score"], "21053585")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][164]["Id"], "6217136")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][164]["Score"], "21042914")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][165]["Id"], "663071")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][165]["Score"], "20926141")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][166]["Id"], "10341802")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][166]["Score"], "20797282")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][167]["Id"], "6473764")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][167]["Score"], "20757680")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][168]["Id"], "2584497")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][168]["Score"], "20521350")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][169]["Id"], "8338105")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][169]["Score"], "20501334")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][170]["Id"], "18053822")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][170]["Score"], "20275078")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][171]["Id"], "4058411")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][171]["Score"], "20161667")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][172]["Id"], "11669077")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][172]["Score"], "19993282")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][173]["Id"], "11781922")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][173]["Score"], "19969425")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][174]["Id"], "9793138")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][174]["Score"], "19952972")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][175]["Id"], "9391495")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][175]["Score"], "19815538")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][176]["Id"], "10803203")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][176]["Score"], "19495693")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][177]["Id"], "7326186")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][177]["Score"], "19273989")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][178]["Id"], "11868066")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][178]["Score"], "19220137")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][179]["Id"], "10904988")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][179]["Score"], "19203510")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][180]["Id"], "3288780")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][180]["Score"], "18958114")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][181]["Id"], "2047316")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][181]["Score"], "18907473")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][182]["Id"], "12237004")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][182]["Score"], "18751474")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][183]["Id"], "5627987")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][183]["Score"], "18741903")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][184]["Id"], "9269670")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][184]["Score"], "18666426")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][185]["Id"], "8903059")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][185]["Score"], "18653874")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][186]["Id"], "5594242")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][186]["Score"], "18548780")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][187]["Id"], "7068417")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][187]["Score"], "18390022")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][188]["Id"], "7330196")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][188]["Score"], "18371587")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][189]["Id"], "7408592")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][189]["Score"], "18275541")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][190]["Id"], "8835983")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][190]["Score"], "18176923")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][191]["Id"], "6940010")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][191]["Score"], "18011066")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][192]["Id"], "10499712")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][192]["Score"], "17943586")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][193]["Id"], "4539876")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][193]["Score"], "17915154")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][194]["Id"], "1943587")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][194]["Score"], "17752606")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][195]["Id"], "9847909")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][195]["Score"], "17568386")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][196]["Id"], "11578071")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][196]["Score"], "17561413")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][197]["Id"], "11789473")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][197]["Score"], "17435433")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][198]["Id"], "9885599")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][198]["Score"], "17383598")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][199]["Id"], "7423836")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][199]["Score"], "17196872")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][200]["Id"], "10688063")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][200]["Score"], "16453112")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][201]["Id"], "11695100")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][201]["Score"], "16352760")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][202]["Id"], "11329658")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][202]["Score"], "16089885")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][203]["Id"], "11939665")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][203]["Score"], "15947974")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][204]["Id"], "5512349")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][204]["Score"], "15647685")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][205]["Id"], "2222794")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][205]["Score"], "14981157")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][206]["Id"], "5998281")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][206]["Score"], "14226588")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][207]["Id"], "10475937")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][207]["Score"], "13934390")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][208]["Id"], "5046513")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][208]["Score"], "12769605")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][209]["Id"], "1539132")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][209]["Score"], "12395064")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][210]["Id"], "4414214")
        self.assertEqual(record[0]["LinkSetDb"][0]["Link"][210]["Score"], "10113539")

    def test_pubmed3(self):
        '''Test parsing pubmed link returned by ELink (third test)
        '''
        # Create a hyperlink to the first link available for PMID 10611131
        # in PubMed
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="10611131", cmd="prlinks")

        handle = open('Entrez/elink5.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(len(record[0]["IdUrlList"]), 1)
        self.assertEqual(len(record[0]["IdUrlList"][0]), 2)
        self.assertEqual(record[0]["IdUrlList"][0]["Id"], "10611131")
        self.assertEqual(len(record[0]["IdUrlList"][0]["ObjUrl"]), 1)
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Url"], "http://brain.oxfordjournals.org/cgi/pmidlookup?view=long&pmid=10611131")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--highwire.stanford.edu-icons-externalservices-pubmed-custom-oxfordjournals_final_free.gif")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["SubjectType"], ["publishers/providers"])
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Attribute"][0], "publisher of information in URL")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Attribute"][1], "full-text online")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Name"], "HighWire Press")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["NameAbbr"], "HighWire")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Id"], "3051")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Url"], "http://highwire.stanford.edu")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["IconUrl"], "http://highwire.stanford.edu/icons/externalservices/pubmed/highwirepress.jpg")


    def test_pubmed4(self):
        '''Test parsing pubmed links returned by ELink (fourth test)
        '''
        # List all available links in PubMed, except for libraries, for
        # PMIDs 12085856 and 12085853
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="12085856,12085853", cmd="llinks")
        handle = open('Entrez/elink6.xml', "rb")
        record = Entrez.read(handle)
        
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(len(record[0]["IdUrlList"]), 2)
        self.assertEqual(record[0]["IdUrlList"][0]["Id"], "12085856")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Url"], "http://symptomresearch.nih.gov/chapter_1/index.htm")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["SubjectType"], ["online tutorials/courses"])
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Name"], "New England Research Institutes Inc.")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["NameAbbr"], "NERI")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Id"], "3291")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][0]["Provider"]["Url"], "http://www.symptomresearch.com")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Url"], "http://www.nlm.nih.gov/medlineplus/coronaryarterybypasssurgery.html")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--www.nlm.nih.gov-medlineplus-images-linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["LinkName"], "Coronary Artery Bypass Surgery")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["SubjectType"], ["consumer health"])
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Provider"]["Name"], "MedlinePlus Health Information")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Provider"]["NameAbbr"], "MEDPLUS")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Provider"]["Id"], "3162")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Provider"]["Url"], "http://medlineplus.gov/")
        self.assertEqual(record[0]["IdUrlList"][0]["ObjUrl"][1]["Provider"]["IconUrl"], "http://www.nlm.nih.gov/medlineplus/images/linkout_sm.gif")

        self.assertEqual(record[0]["IdUrlList"][1]["Id"], "12085853")
        self.assertEqual(len(record[0]["IdUrlList"][1]["ObjUrl"]), 4)
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Url"], "http://www.nlm.nih.gov/medlineplus/arrhythmia.html")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--www.nlm.nih.gov-medlineplus-images-linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["LinkName"], "Arrhythmia")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["SubjectType"], ["consumer health"])
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Provider"]["Name"], "MedlinePlus Health Information")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Provider"]["NameAbbr"], "MEDPLUS")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Provider"]["Id"], "3162")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Provider"]["Url"], "http://medlineplus.gov/")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][0]["Provider"]["IconUrl"], "http://www.nlm.nih.gov/medlineplus/images/linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Url"], "http://www.nlm.nih.gov/medlineplus/exerciseandphysicalfitness.html")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--www.nlm.nih.gov-medlineplus-images-linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["LinkName"], "Exercise and Physical Fitness")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["SubjectType"], ["consumer health"])
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Provider"]["Name"], "MedlinePlus Health Information")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Provider"]["NameAbbr"], "MEDPLUS")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Provider"]["Id"], "3162")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Provider"]["Url"], "http://medlineplus.gov/")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][1]["Provider"]["IconUrl"], "http://www.nlm.nih.gov/medlineplus/images/linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Url"], "http://www.nlm.nih.gov/medlineplus/exerciseforchildren.html")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--www.nlm.nih.gov-medlineplus-images-linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["LinkName"], "Exercise for Children")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["SubjectType"], ["consumer health"])
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Provider"]["Name"], "MedlinePlus Health Information")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Provider"]["NameAbbr"], "MEDPLUS")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Provider"]["Id"], "3162")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Provider"]["Url"], "http://medlineplus.gov/")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][2]["Provider"]["IconUrl"], "http://www.nlm.nih.gov/medlineplus/images/linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Url"], "http://www.nlm.nih.gov/medlineplus/pacemakersandimplantabledefibrillators.html")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["IconUrl"], "http://www.ncbi.nlm.nih.gov/entrez/query/egifs/http:--www.nlm.nih.gov-medlineplus-images-linkout_sm.gif")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["LinkName"], "Pacemakers and Implantable Defibrillators")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["SubjectType"], ["consumer health"])
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Provider"]["Name"], "MedlinePlus Health Information")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Provider"]["NameAbbr"], "MEDPLUS")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Provider"]["Id"], "3162")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Provider"]["Url"], "http://medlineplus.gov/")
        self.assertEqual(record[0]["IdUrlList"][1]["ObjUrl"][3]["Provider"]["IconUrl"], "http://www.nlm.nih.gov/medlineplus/images/linkout_sm.gif")

    def test_pubmed5(self):
        '''Test parsing pubmed links returned by ELink (fifth test)
        '''
        # List Entrez database links for PubMed PMIDs 12169658 and 11748140
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="12169658,11748140",
        #                      cmd="acheck")
        handle = open('Entrez/elink7.xml', "rb")
        record = Entrez.read(handle)
        
        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(len(record[0]["IdCheckList"]), 2)
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["Id"], "12169658")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][0]["DbTo"], "books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][0]["LinkName"], "pubmed_books_refs")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][0]["MenuTag"], "Cited in Books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][0]["HtmlTag"], "Cited in Books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][0]["Priority"], "185")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][1]["DbTo"], "gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][1]["LinkName"], "pubmed_gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][1]["MenuTag"], "Gene Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][1]["HtmlTag"], "Gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][1]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][2]["DbTo"], "geo")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][2]["LinkName"], "pubmed_geo")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][2]["MenuTag"], "GEO Profile Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][2]["HtmlTag"], "GEO Profiles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][2]["Priority"], "170")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][3]["DbTo"], "homologene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][3]["LinkName"], "pubmed_homologene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][3]["MenuTag"], "HomoloGene Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][3]["HtmlTag"], "HomoloGene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][3]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][4]["DbTo"], "nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][4]["LinkName"], "pubmed_nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][4]["MenuTag"], "CoreNucleotide Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][4]["HtmlTag"], "CoreNucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][4]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][5]["DbTo"], "nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][5]["LinkName"], "pubmed_nuccore_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][5]["MenuTag"], "CoreNucleotide (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][5]["HtmlTag"], "CoreNucleotide (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][5]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][6]["DbTo"], "nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][6]["LinkName"], "pubmed_nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][6]["MenuTag"], "Nucleotide Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][6]["HtmlTag"], "Nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][6]["Priority"], "135")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][7]["DbTo"], "nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][7]["LinkName"], "pubmed_nucleotide_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][7]["MenuTag"], "Nucleotide (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][7]["HtmlTag"], "Nucleotide (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][7]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][8]["DbTo"], "pcsubstance")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][8]["LinkName"], "pubmed_pcsubstance_mesh")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][8]["MenuTag"], "Substance (MeSH Keyword)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][8]["HtmlTag"], "Substance (MeSH Keyword)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][8]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["DbTo"], "pmc")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["LinkName"], "pubmed_pmc_refs")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["MenuTag"], "Cited in PMC")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["HtmlTag"], "Cited in PMC")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["Url"], "http://www.pubmedcentral.gov/tocrender.fcgi?action=cited&tool=pubmed&pubmedid=<@UID@>")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][9]["Priority"], "180")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][10]["DbTo"], "protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][10]["LinkName"], "pubmed_protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][10]["MenuTag"], "Protein Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][10]["HtmlTag"], "Protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][10]["Priority"], "140")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][11]["DbTo"], "protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][11]["LinkName"], "pubmed_protein_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][11]["MenuTag"], "Protein (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][11]["HtmlTag"], "Protein (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][11]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][12]["DbTo"], "pubmed")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][12]["LinkName"], "pubmed_pubmed")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][12]["MenuTag"], "Related Articles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][12]["HtmlTag"], "Related Articles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][12]["Priority"], "1")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][13]["DbTo"], "taxonomy")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][13]["LinkName"], "pubmed_taxonomy_entrez")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][13]["MenuTag"], "Taxonomy via GenBank")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][13]["HtmlTag"], "Taxonomy via GenBank")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][13]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][14]["DbTo"], "unigene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][14]["LinkName"], "pubmed_unigene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][14]["MenuTag"], "UniGene Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][14]["HtmlTag"], "UniGene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][14]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][15]["DbTo"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][15]["LinkName"], "ExternalLink")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][15]["MenuTag"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][15]["HtmlTag"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][0]["LinkInfo"][15]["Priority"], "255")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["Id"], "11748140")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][0]["DbTo"], "books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][0]["LinkName"], "pubmed_books_refs")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][0]["MenuTag"], "Cited in Books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][0]["HtmlTag"], "Cited in Books")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][0]["Priority"], "185")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][1]["DbTo"], "gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][1]["LinkName"], "pubmed_gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][1]["MenuTag"], "Gene Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][1]["HtmlTag"], "Gene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][1]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][2]["DbTo"], "geo")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][2]["LinkName"], "pubmed_geo")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][2]["MenuTag"], "GEO Profile Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][2]["HtmlTag"], "GEO Profiles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][2]["Priority"], "170")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][3]["DbTo"], "nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][3]["LinkName"], "pubmed_nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][3]["MenuTag"], "CoreNucleotide Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][3]["HtmlTag"], "CoreNucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][3]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][4]["DbTo"], "nuccore")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][4]["LinkName"], "pubmed_nuccore_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][4]["MenuTag"], "CoreNucleotide (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][4]["HtmlTag"], "CoreNucleotide (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][4]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][5]["DbTo"], "nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][5]["LinkName"], "pubmed_nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][5]["MenuTag"], "Nucleotide Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][5]["HtmlTag"], "Nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][5]["Priority"], "135")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][6]["DbTo"], "nucleotide")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][6]["LinkName"], "pubmed_nucleotide_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][6]["MenuTag"], "Nucleotide (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][6]["HtmlTag"], "Nucleotide (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][6]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["DbTo"], "pmc")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["LinkName"], "pubmed_pmc_refs")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["MenuTag"], "Cited in PMC")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["HtmlTag"], "Cited in PMC")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["Url"], "http://www.pubmedcentral.gov/tocrender.fcgi?action=cited&tool=pubmed&pubmedid=<@UID@>")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][7]["Priority"], "180")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][8]["DbTo"], "protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][8]["LinkName"], "pubmed_protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][8]["MenuTag"], "Protein Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][8]["HtmlTag"], "Protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][8]["Priority"], "140")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][9]["DbTo"], "protein")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][9]["LinkName"], "pubmed_protein_refseq")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][9]["MenuTag"], "Protein (RefSeq) Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][9]["HtmlTag"], "Protein (RefSeq)")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][9]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][10]["DbTo"], "pubmed")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][10]["LinkName"], "pubmed_pubmed")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][10]["MenuTag"], "Related Articles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][10]["HtmlTag"], "Related Articles")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][10]["Priority"], "1")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][11]["DbTo"], "taxonomy")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][11]["LinkName"], "pubmed_taxonomy_entrez")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][11]["MenuTag"], "Taxonomy via GenBank")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][11]["HtmlTag"], "Taxonomy via GenBank")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][11]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][12]["DbTo"], "unigene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][12]["LinkName"], "pubmed_unigene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][12]["MenuTag"], "UniGene Links")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][12]["HtmlTag"], "UniGene")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][12]["Priority"], "128")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][13]["DbTo"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][13]["LinkName"], "ExternalLink")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][13]["MenuTag"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][13]["HtmlTag"], "LinkOut")
        self.assertEqual(record[0]["IdCheckList"]["IdLinkSet"][1]["LinkInfo"][13]["Priority"], "255")


    def test_pubmed6(self):
        '''Test parsing pubmed links returned by ELink (sixth test)
        '''
        # Check for the existence of a Related Articles link for PMIDs
        # 0611131, 111645 and 12068369
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="pubmed", id="10611131,111645",
        #                      id="12068369", cmd="ncheck")
        handle = open('Entrez/elink8.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["DbFrom"], "pubmed")
        self.assertEqual(len(record[0]["IdCheckList"]), 2)
        self.assertEqual(len(record[0]["IdCheckList"]["Id"]), 1)
        self.assertEqual(record[0]["IdCheckList"]["Id"][0], "12068369")
        self.assertEqual(len(record[0]["IdCheckList"]["Id"][0].attributes), 1)
        self.assertEqual(record[0]["IdCheckList"]["Id"][0].attributes["HasNeighbor"], "Y")

    def test_cancerchromosomes(self):
        '''Test parsing cancerchromosomes links returned by ELink
        '''
        # Retrieve neighbors for Cancer Chromosomes ID 2662 to the link
        # subset related by cytogenetics
        # To create the XML file, use
        # >>> Bio.Entrez.elink(dbfrom="cancerchromosomes",
        #                       db="cancerchromosomes", id="2662",
        #                       cmd="neighbor",
        #                       linkname="cancerchromosomes_cancerchromosomes_cyto")
        handle = open('Entrez/elink9.xml', "rb")
        record = Entrez.read(handle)
        
        self.assertEqual(record[0]["DbFrom"], "cancerchromosomes")
        self.assertEqual(record[0]["IdList"], ["2662"])


class EGQueryTest(unittest.TestCase):
    '''Tests for parsing XML output returned by EGQuery
    '''
    def test_egquery1(self):
        '''Test parsing XML output returned by EGQuery (first test)
        '''
        # Display counts in XML for stem cells in each Entrez database
        # To create the XML file, use
        # >>> Bio.Entrez.egquery(term="stem cells")
        handle = open('Entrez/egquery1.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record["Term"], "stem cells")

        self.assertEqual(record["eGQueryResult"][0]["DbName"], "pubmed")
        self.assertEqual(record["eGQueryResult"][0]["MenuName"], "PubMed")
        self.assertEqual(record["eGQueryResult"][0]["Count"], "392")
        self.assertEqual(record["eGQueryResult"][0]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][1]["DbName"], "pmc")
        self.assertEqual(record["eGQueryResult"][1]["MenuName"], "PMC")
        self.assertEqual(record["eGQueryResult"][1]["Count"], "173")
        self.assertEqual(record["eGQueryResult"][1]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][2]["DbName"], "journals")
        self.assertEqual(record["eGQueryResult"][2]["MenuName"], "Journals")
        self.assertEqual(record["eGQueryResult"][2]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][2]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][3]["DbName"], "mesh")
        self.assertEqual(record["eGQueryResult"][3]["MenuName"], "MeSH")
        self.assertEqual(record["eGQueryResult"][3]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][3]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][4]["DbName"], "books")
        self.assertEqual(record["eGQueryResult"][4]["MenuName"], "Books")
        self.assertEqual(record["eGQueryResult"][4]["Count"], "10")
        self.assertEqual(record["eGQueryResult"][4]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][5]["DbName"], "omim")
        self.assertEqual(record["eGQueryResult"][5]["MenuName"], "OMIM")
        self.assertEqual(record["eGQueryResult"][5]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][5]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][6]["DbName"], "omia")
        self.assertEqual(record["eGQueryResult"][6]["MenuName"], "OMIA")
        self.assertEqual(record["eGQueryResult"][6]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][6]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][7]["DbName"], "ncbisearch")
        self.assertEqual(record["eGQueryResult"][7]["MenuName"], "NCBI Web Site")
        self.assertEqual(record["eGQueryResult"][7]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][7]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][8]["DbName"], "nuccore")
        self.assertEqual(record["eGQueryResult"][8]["MenuName"], "CoreNucleotide")
        self.assertEqual(record["eGQueryResult"][8]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][8]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][9]["DbName"], "nucgss")
        self.assertEqual(record["eGQueryResult"][9]["MenuName"], "GSS")
        self.assertEqual(record["eGQueryResult"][9]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][9]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][10]["DbName"], "nucest")
        self.assertEqual(record["eGQueryResult"][10]["MenuName"], "EST")
        self.assertEqual(record["eGQueryResult"][10]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][10]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][11]["DbName"], "protein")
        self.assertEqual(record["eGQueryResult"][11]["MenuName"], "Protein")
        self.assertEqual(record["eGQueryResult"][11]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][11]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][12]["DbName"], "genome")
        self.assertEqual(record["eGQueryResult"][12]["MenuName"], "Genome")
        self.assertEqual(record["eGQueryResult"][12]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][12]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][13]["DbName"], "structure")
        self.assertEqual(record["eGQueryResult"][13]["MenuName"], "Structure")
        self.assertEqual(record["eGQueryResult"][13]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][13]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][14]["DbName"], "taxonomy")
        self.assertEqual(record["eGQueryResult"][14]["MenuName"], "Taxonomy")
        self.assertEqual(record["eGQueryResult"][14]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][14]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][15]["DbName"], "snp")
        self.assertEqual(record["eGQueryResult"][15]["MenuName"], "SNP")
        self.assertEqual(record["eGQueryResult"][15]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][15]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][16]["DbName"], "gene")
        self.assertEqual(record["eGQueryResult"][16]["MenuName"], "Gene")
        self.assertEqual(record["eGQueryResult"][16]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][16]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][17]["DbName"], "unigene")
        self.assertEqual(record["eGQueryResult"][17]["MenuName"], "UniGene")
        self.assertEqual(record["eGQueryResult"][17]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][17]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][18]["DbName"], "cdd")
        self.assertEqual(record["eGQueryResult"][18]["MenuName"], "Conserved Domains")
        self.assertEqual(record["eGQueryResult"][18]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][18]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][19]["DbName"], "domains")
        self.assertEqual(record["eGQueryResult"][19]["MenuName"], "3D Domains")
        self.assertEqual(record["eGQueryResult"][19]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][19]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][20]["DbName"], "unists")
        self.assertEqual(record["eGQueryResult"][20]["MenuName"], "UniSTS")
        self.assertEqual(record["eGQueryResult"][20]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][20]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][21]["DbName"], "popset")
        self.assertEqual(record["eGQueryResult"][21]["MenuName"], "PopSet")
        self.assertEqual(record["eGQueryResult"][21]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][21]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][22]["DbName"], "geo")
        self.assertEqual(record["eGQueryResult"][22]["MenuName"], "GEO Profiles")
        self.assertEqual(record["eGQueryResult"][22]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][22]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][23]["DbName"], "gds")
        self.assertEqual(record["eGQueryResult"][23]["MenuName"], "GEO DataSets")
        self.assertEqual(record["eGQueryResult"][23]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][23]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][24]["DbName"], "homologene")
        self.assertEqual(record["eGQueryResult"][24]["MenuName"], "HomoloGene")
        self.assertEqual(record["eGQueryResult"][24]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][24]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][25]["DbName"], "cancerchromosomes")
        self.assertEqual(record["eGQueryResult"][25]["MenuName"], "CancerChromosomes")
        self.assertEqual(record["eGQueryResult"][25]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][25]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][26]["DbName"], "pccompound")
        self.assertEqual(record["eGQueryResult"][26]["MenuName"], "PubChem Compound")
        self.assertEqual(record["eGQueryResult"][26]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][26]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][27]["DbName"], "pcsubstance")
        self.assertEqual(record["eGQueryResult"][27]["MenuName"], "PubChem Substance")
        self.assertEqual(record["eGQueryResult"][27]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][27]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][28]["DbName"], "pcassay")
        self.assertEqual(record["eGQueryResult"][28]["MenuName"], "PubChem BioAssay")
        self.assertEqual(record["eGQueryResult"][28]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][28]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][29]["DbName"], "nlmcatalog")
        self.assertEqual(record["eGQueryResult"][29]["MenuName"], "NLM Catalog")
        self.assertEqual(record["eGQueryResult"][29]["Count"], "2")
        self.assertEqual(record["eGQueryResult"][29]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][30]["DbName"], "gensat")
        self.assertEqual(record["eGQueryResult"][30]["MenuName"], "GENSAT")
        self.assertEqual(record["eGQueryResult"][30]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][30]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][31]["DbName"], "probe")
        self.assertEqual(record["eGQueryResult"][31]["MenuName"], "Probe")
        self.assertEqual(record["eGQueryResult"][31]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][31]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][32]["DbName"], "genomeprj")
        self.assertEqual(record["eGQueryResult"][32]["MenuName"], "Genome Project")
        self.assertEqual(record["eGQueryResult"][32]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][32]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][33]["DbName"], "gap")
        self.assertEqual(record["eGQueryResult"][33]["MenuName"], "dbGaP")
        self.assertEqual(record["eGQueryResult"][33]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][33]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][34]["DbName"], "proteinclusters")
        self.assertEqual(record["eGQueryResult"][34]["MenuName"], "Protein Clusters")
        self.assertEqual(record["eGQueryResult"][34]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][34]["Status"], "Term or Database is not found")

    def test_egquery2(self):
        '''Test parsing XML output returned by EGQuery (second test)
        '''
        # Display counts in XML for brca1 or brca2 for each Entrez database
        # To create the XML file, use
        # >>> Bio.Entrez.egquery(term="brca1 OR brca2")
        handle = open('Entrez/egquery2.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record["Term"], "brca1 OR brca2")

        self.assertEqual(record["eGQueryResult"][0]["DbName"], "pubmed")
        self.assertEqual(record["eGQueryResult"][0]["MenuName"], "PubMed")
        self.assertEqual(record["eGQueryResult"][0]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][0]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][1]["DbName"], "pmc")
        self.assertEqual(record["eGQueryResult"][1]["MenuName"], "PMC")
        self.assertEqual(record["eGQueryResult"][1]["Count"], "2739")
        self.assertEqual(record["eGQueryResult"][1]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][2]["DbName"], "journals")
        self.assertEqual(record["eGQueryResult"][2]["MenuName"], "Journals")
        self.assertEqual(record["eGQueryResult"][2]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][2]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][3]["DbName"], "mesh")
        self.assertEqual(record["eGQueryResult"][3]["MenuName"], "MeSH")
        self.assertEqual(record["eGQueryResult"][3]["Count"], "29")
        self.assertEqual(record["eGQueryResult"][3]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][4]["DbName"], "books")
        self.assertEqual(record["eGQueryResult"][4]["MenuName"], "Books")
        self.assertEqual(record["eGQueryResult"][4]["Count"], "392")
        self.assertEqual(record["eGQueryResult"][4]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][5]["DbName"], "omim")
        self.assertEqual(record["eGQueryResult"][5]["MenuName"], "OMIM")
        self.assertEqual(record["eGQueryResult"][5]["Count"], "149")
        self.assertEqual(record["eGQueryResult"][5]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][6]["DbName"], "omia")
        self.assertEqual(record["eGQueryResult"][6]["MenuName"], "OMIA")
        self.assertEqual(record["eGQueryResult"][6]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][6]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][7]["DbName"], "ncbisearch")
        self.assertEqual(record["eGQueryResult"][7]["MenuName"], "NCBI Web Site")
        self.assertEqual(record["eGQueryResult"][7]["Count"], "13")
        self.assertEqual(record["eGQueryResult"][7]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][8]["DbName"], "nuccore")
        self.assertEqual(record["eGQueryResult"][8]["MenuName"], "CoreNucleotide")
        self.assertEqual(record["eGQueryResult"][8]["Count"], "4917")
        self.assertEqual(record["eGQueryResult"][8]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][9]["DbName"], "nucgss")
        self.assertEqual(record["eGQueryResult"][9]["MenuName"], "GSS")
        self.assertEqual(record["eGQueryResult"][9]["Count"], "184")
        self.assertEqual(record["eGQueryResult"][9]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][10]["DbName"], "nucest")
        self.assertEqual(record["eGQueryResult"][10]["MenuName"], "EST")
        self.assertEqual(record["eGQueryResult"][10]["Count"], "600")
        self.assertEqual(record["eGQueryResult"][10]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][11]["DbName"], "protein")
        self.assertEqual(record["eGQueryResult"][11]["MenuName"], "Protein")
        self.assertEqual(record["eGQueryResult"][11]["Count"], "6779")
        self.assertEqual(record["eGQueryResult"][11]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][12]["DbName"], "genome")
        self.assertEqual(record["eGQueryResult"][12]["MenuName"], "Genome")
        self.assertEqual(record["eGQueryResult"][12]["Count"], "44")
        self.assertEqual(record["eGQueryResult"][12]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][13]["DbName"], "structure")
        self.assertEqual(record["eGQueryResult"][13]["MenuName"], "Structure")
        self.assertEqual(record["eGQueryResult"][13]["Count"], "29")
        self.assertEqual(record["eGQueryResult"][13]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][14]["DbName"], "taxonomy")
        self.assertEqual(record["eGQueryResult"][14]["MenuName"], "Taxonomy")
        self.assertEqual(record["eGQueryResult"][14]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][14]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][15]["DbName"], "snp")
        self.assertEqual(record["eGQueryResult"][15]["MenuName"], "SNP")
        self.assertEqual(record["eGQueryResult"][15]["Count"], "2013")
        self.assertEqual(record["eGQueryResult"][15]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][16]["DbName"], "gene")
        self.assertEqual(record["eGQueryResult"][16]["MenuName"], "Gene")
        self.assertEqual(record["eGQueryResult"][16]["Count"], "1775")
        self.assertEqual(record["eGQueryResult"][16]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][17]["DbName"], "unigene")
        self.assertEqual(record["eGQueryResult"][17]["MenuName"], "UniGene")
        self.assertEqual(record["eGQueryResult"][17]["Count"], "207")
        self.assertEqual(record["eGQueryResult"][17]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][18]["DbName"], "cdd")
        self.assertEqual(record["eGQueryResult"][18]["MenuName"], "Conserved Domains")
        self.assertEqual(record["eGQueryResult"][18]["Count"], "17")
        self.assertEqual(record["eGQueryResult"][18]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][19]["DbName"], "domains")
        self.assertEqual(record["eGQueryResult"][19]["MenuName"], "3D Domains")
        self.assertEqual(record["eGQueryResult"][19]["Count"], "131")
        self.assertEqual(record["eGQueryResult"][19]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][20]["DbName"], "unists")
        self.assertEqual(record["eGQueryResult"][20]["MenuName"], "UniSTS")
        self.assertEqual(record["eGQueryResult"][20]["Count"], "198")
        self.assertEqual(record["eGQueryResult"][20]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][21]["DbName"], "popset")
        self.assertEqual(record["eGQueryResult"][21]["MenuName"], "PopSet")
        self.assertEqual(record["eGQueryResult"][21]["Count"], "43")
        self.assertEqual(record["eGQueryResult"][21]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][22]["DbName"], "geo")
        self.assertEqual(record["eGQueryResult"][22]["MenuName"], "GEO Profiles")
        self.assertEqual(record["eGQueryResult"][22]["Count"], "128692")
        self.assertEqual(record["eGQueryResult"][22]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][23]["DbName"], "gds")
        self.assertEqual(record["eGQueryResult"][23]["MenuName"], "GEO DataSets")
        self.assertEqual(record["eGQueryResult"][23]["Count"], "21")
        self.assertEqual(record["eGQueryResult"][23]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][24]["DbName"], "homologene")
        self.assertEqual(record["eGQueryResult"][24]["MenuName"], "HomoloGene")
        self.assertEqual(record["eGQueryResult"][24]["Count"], "50")
        self.assertEqual(record["eGQueryResult"][24]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][25]["DbName"], "cancerchromosomes")
        self.assertEqual(record["eGQueryResult"][25]["MenuName"], "CancerChromosomes")
        self.assertEqual(record["eGQueryResult"][25]["Count"], "18")
        self.assertEqual(record["eGQueryResult"][25]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][26]["DbName"], "pccompound")
        self.assertEqual(record["eGQueryResult"][26]["MenuName"], "PubChem Compound")
        self.assertEqual(record["eGQueryResult"][26]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][26]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][27]["DbName"], "pcsubstance")
        self.assertEqual(record["eGQueryResult"][27]["MenuName"], "PubChem Substance")
        self.assertEqual(record["eGQueryResult"][27]["Count"], "26")
        self.assertEqual(record["eGQueryResult"][27]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][28]["DbName"], "pcassay")
        self.assertEqual(record["eGQueryResult"][28]["MenuName"], "PubChem BioAssay")
        self.assertEqual(record["eGQueryResult"][28]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][28]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][29]["DbName"], "nlmcatalog")
        self.assertEqual(record["eGQueryResult"][29]["MenuName"], "NLM Catalog")
        self.assertEqual(record["eGQueryResult"][29]["Count"], "31")
        self.assertEqual(record["eGQueryResult"][29]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][30]["DbName"], "gensat")
        self.assertEqual(record["eGQueryResult"][30]["MenuName"], "GENSAT")
        self.assertEqual(record["eGQueryResult"][30]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][30]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][31]["DbName"], "probe")
        self.assertEqual(record["eGQueryResult"][31]["MenuName"], "Probe")
        self.assertEqual(record["eGQueryResult"][31]["Count"], "1410")
        self.assertEqual(record["eGQueryResult"][31]["Status"], "Ok")
        self.assertEqual(record["eGQueryResult"][32]["DbName"], "genomeprj")
        self.assertEqual(record["eGQueryResult"][32]["MenuName"], "Genome Project")
        self.assertEqual(record["eGQueryResult"][32]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][32]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][33]["DbName"], "gap")
        self.assertEqual(record["eGQueryResult"][33]["MenuName"], "dbGaP")
        self.assertEqual(record["eGQueryResult"][33]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][33]["Status"], "Term or Database is not found")
        self.assertEqual(record["eGQueryResult"][34]["DbName"], "proteinclusters")
        self.assertEqual(record["eGQueryResult"][34]["MenuName"], "Protein Clusters")
        self.assertEqual(record["eGQueryResult"][34]["Count"], "0")
        self.assertEqual(record["eGQueryResult"][34]["Status"], "Term or Database is not found")

class ESpellTest(unittest.TestCase):
    '''Tests for parsing XML output returned by ESpell
    '''
    def test_espell(self):
        '''Test parsing XML output returned by ESpell
        '''
        # Request suggestions for the PubMed search biopythooon
        # To create the XML file, use
        # >>> Bio.Entrez.espell(db="pubmed", term="biopythooon")
        handle = open('Entrez/espell.xml', "rb")
        record = Entrez.read(handle)
        self.assertEqual(record["Database"], "pubmed")
        self.assertEqual(record["Query"], "biopythooon")
        self.assertEqual(record["CorrectedQuery"], "biopython")
        self.assertEqual(len(record["SpelledQuery"]), 1)
        self.assertEqual(record["SpelledQuery"][0], "biopython")
        self.assertEqual(record["SpelledQuery"][0].tag, "Replaced")


class EFetchTest(unittest.TestCase):
    '''Tests for parsing XML output returned by EFetch
    '''
    def test_pubmed1(self):
        '''Test parsing XML returned by EFetch, PubMed database (first test)
        '''
        # In PubMed display PMIDs 12091962 and 9997 in xml retrieval mode
        # and abstract retrieval type.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db='pubmed', id='12091962,9997',
        #                       retmode='xml', rettype='abstract')
        handle = open('Entrez/pubmed1.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["MedlineCitation"].attributes["Owner"], "KIE")
        self.assertEqual(record[0]["MedlineCitation"].attributes["Status"], "MEDLINE")
        self.assertEqual(record[0]["MedlineCitation"]["PMID"], "12091962")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Year"], "1991")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Month"], "01")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Day"], "22")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Year"], "1991")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Month"], "01")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Day"], "22")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Year"], "2007")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Month"], "11")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Day"], "15")
        self.assertEqual(record[0]["MedlineCitation"]["Article"].attributes["PubModel"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["ISSN"], "1043-1578")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["ISSN"].attributes["IssnType"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"].attributes["CitedMedium"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Volume"], "17")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Issue"], "1")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Year"], "1990")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Season"], "Spring")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["Title"], "Social justice (San Francisco, Calif.)")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["ArticleTitle"], "The treatment of AIDS behind the walls of correctional facilities.")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Pagination"]["MedlinePgn"], "113-25")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"].attributes["CompleteYN"], 'Y')
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["LastName"], "Olivero")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["ForeName"], "J Michael")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["Initials"], "JM")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Language"], ["eng"])
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["PublicationTypeList"], ["Journal Article", "Review"])
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["Country"], "United States")
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["MedlineTA"], "Soc Justice")
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["NlmUniqueID"], "9891830")
        self.assertEqual(record[0]["MedlineCitation"]["CitationSubset"], ["E"])
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"], "AIDS Serodiagnosis")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"], "Acquired Immunodeficiency Syndrome")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"], "Civil Rights")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"], "HIV Seropositivity")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"], "Humans")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"], "Jurisprudence")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"], "Law Enforcement")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"], "Mass Screening")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"], "Minority Groups")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"], "Organizational Policy")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"], "Patient Care")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][11]["DescriptorName"], "Prejudice")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][11]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][12]["DescriptorName"], "Prisoners")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][12]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][13]["DescriptorName"], "Public Policy")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][13]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][14]["DescriptorName"], "Quarantine")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][14]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][15]["DescriptorName"], "Social Control, Formal")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][15]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][16]["DescriptorName"], "Statistics as Topic")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][16]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][17]["DescriptorName"], "Stereotyping")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][17]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][18]["DescriptorName"], "United States")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][18]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["NumberOfReferences"], "63")
        self.assertEqual(record[0]["MedlineCitation"]["OtherID"][0], "31840")
        self.assertEqual(record[0]["MedlineCitation"]["OtherID"][0].attributes["Source"], "KIE")
        self.assertEqual(record[0]["MedlineCitation"]["KeywordList"][0].attributes["Owner"], "KIE")
        self.assertEqual(record[0]["MedlineCitation"]["KeywordList"][0][0], "Health Care and Public Health")
        self.assertEqual(record[0]["MedlineCitation"]["KeywordList"][0][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["KeywordList"][0][1], "Legal Approach")
        self.assertEqual(record[0]["MedlineCitation"]["KeywordList"][0][1].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][0], "14 fn.")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][0].attributes["Owner"], "KIE")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][1], "KIE BoB Subject Heading: AIDS")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][1].attributes["Owner"], "KIE")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][2], "63 refs.")
        self.assertEqual(record[0]["MedlineCitation"]["GeneralNote"][2].attributes["Owner"], "KIE")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0].attributes["PubStatus"], "pubmed")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Year"], "1990")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Month"], "4")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Day"], "1")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Hour"], "0")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Minute"], "0")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1].attributes["PubStatus"], "medline")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Year"], "2002")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Month"], "7")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Day"], "16")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Hour"], "10")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Minute"], "1")
        self.assertEqual(record[0]["PubmedData"]["PublicationStatus"], "ppublish")
        self.assertEqual(len(record[0]["PubmedData"]["ArticleIdList"]), 1)
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][0], "12091962")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][0].attributes["IdType"], "pubmed")
        self.assertEqual(record[1]["MedlineCitation"].attributes["Owner"], "NLM")
        self.assertEqual(record[1]["MedlineCitation"].attributes["Status"], "MEDLINE")
        self.assertEqual(record[1]["MedlineCitation"]["PMID"], "9997")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Year"], "1976")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Month"], "12")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Day"], "30")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Year"], "1976")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Month"], "12")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Day"], "30")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Year"], "2003")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Month"], "11")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Day"], "14")
        self.assertEqual(record[1]["MedlineCitation"]["Article"].attributes["PubModel"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISSN"], "0006-3002")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISSN"].attributes["IssnType"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"].attributes["CitedMedium"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Volume"], "446")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Issue"], "1")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Year"], "1976")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Month"], "Sep")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Day"], "28")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["Title"], "Biochimica et biophysica acta")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISOAbbreviation"], "Biochim. Biophys. Acta")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["ArticleTitle"], "Magnetic studies of Chromatium flavocytochrome C552. A mechanism for heme-flavin interaction.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Pagination"]["MedlinePgn"], "179-91")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Abstract"]["AbstractText"], "Electron paramagnetic resonance and magnetic susceptibility studies of Chromatium flavocytochrome C552 and its diheme flavin-free subunit at temperatures below 45 degrees K are reported. The results show that in the intact protein and the subunit the two low-spin (S = 1/2) heme irons are distinguishable, giving rise to separate EPR signals. In the intact protein only, one of the heme irons exists in two different low spin environments in the pH range 5.5 to 10.5, while the other remains in a constant environment. Factors influencing the variable heme iron environment also influence flavin reactivity, indicating the existence of a mechanism for heme-flavin interaction.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"].attributes["CompleteYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["LastName"], "Strekas")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["ForeName"], "T C")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["Initials"], "TC")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Language"], ["eng"])
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["PublicationTypeList"], ["Journal Article"])
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["Country"], "NETHERLANDS")
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["MedlineTA"], "Biochim Biophys Acta")
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["NlmUniqueID"], "0217513")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][0]["RegistryNumber"], "0")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][0]["NameOfSubstance"], "Cytochrome c Group")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][1]["RegistryNumber"], "0")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][1]["NameOfSubstance"], "Flavins")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][2]["RegistryNumber"], "14875-96-8")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][2]["NameOfSubstance"], "Heme")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][3]["RegistryNumber"], "7439-89-6")
        self.assertEqual(record[1]["MedlineCitation"]["ChemicalList"][3]["NameOfSubstance"], "Iron")
        self.assertEqual(record[1]["MedlineCitation"]["CitationSubset"], ["IM"])
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"], "Binding Sites")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"], "Chromatium")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][1]["QualifierName"][0], "enzymology")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][1]["QualifierName"][0].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"], "Cytochrome c Group")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"], "Electron Spin Resonance Spectroscopy")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"], "Flavins")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"], "Heme")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"], "Hydrogen-Ion Concentration")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"], "Iron")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][0], "analysis")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"], "Magnetics")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"], "Oxidation-Reduction")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"], "Protein Binding")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][11]["DescriptorName"], "Protein Conformation")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][11]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][12]["DescriptorName"], "Temperature")
        self.assertEqual(record[1]["MedlineCitation"]["MeshHeadingList"][12]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0].attributes["PubStatus"], "pubmed")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Year"], "1976")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Month"], "9")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Day"], "28")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1].attributes["PubStatus"], "medline")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Year"], "1976")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Month"], "9")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Day"], "28")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Hour"], "0")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Minute"], "1")
        self.assertEqual(record[1]["PubmedData"]["PublicationStatus"], "ppublish")
        self.assertEqual(len(record[1]["PubmedData"]["ArticleIdList"]), 1)
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][0], "9997")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][0].attributes["IdType"], "pubmed")


    def test_pubmed2(self):
        '''Test parsing XML returned by EFetch, PubMed database (second test)
        '''
        # In PubMed display PMIDs in xml retrieval mode.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db='pubmed', id="11748933,11700088",
        #                       retmode="xml")
        handle = open('Entrez/pubmed2.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["MedlineCitation"].attributes["Owner"], "NLM")
        self.assertEqual(record[0]["MedlineCitation"].attributes["Status"], "MEDLINE")
        self.assertEqual(record[0]["MedlineCitation"]["PMID"], "11748933")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Year"], "2001")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Month"], "12")
        self.assertEqual(record[0]["MedlineCitation"]["DateCreated"]["Day"], "25")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Year"], "2002")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Month"], "03")
        self.assertEqual(record[0]["MedlineCitation"]["DateCompleted"]["Day"], "04")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Year"], "2006")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Month"], "11")
        self.assertEqual(record[0]["MedlineCitation"]["DateRevised"]["Day"], "15")
        self.assertEqual(record[0]["MedlineCitation"]["Article"].attributes["PubModel"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["ISSN"], "0011-2240")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["ISSN"].attributes["IssnType"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"].attributes["CitedMedium"], "Print")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Volume"], "42")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Issue"], "4")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Year"], "2001")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Month"], "Jun")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["Title"], "Cryobiology")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Journal"]["ISOAbbreviation"], "Cryobiology")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["ArticleTitle"], "Is cryopreservation a homogeneous process? Ultrastructure and motility of untreated, prefreezing, and postthawed spermatozoa of Diplodus puntazzo (Cetti).")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Pagination"]["MedlinePgn"], "244-55")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Abstract"]["AbstractText"], "This study subdivides the cryopreservation procedure for Diplodus puntazzo spermatozoa into three key phases, fresh, prefreezing (samples equilibrated in cryosolutions), and postthawed stages, and examines the ultrastructural anomalies and motility profiles of spermatozoa in each stage, with different cryodiluents. Two simple cryosolutions were evaluated: 0.17 M sodium chloride containing a final concentration of 15% dimethyl sulfoxide (Me(2)SO) (cryosolution A) and 0.1 M sodium citrate containing a final concentration of 10% Me(2)SO (cryosolution B). Ultrastructural anomalies of the plasmatic and nuclear membranes of the sperm head were common and the severity of the cryoinjury differed significantly between the pre- and the postfreezing phases and between the two cryosolutions. In spermatozoa diluted with cryosolution A, during the prefreezing phase, the plasmalemma of 61% of the cells was absent or damaged compared with 24% in the fresh sample (P < 0.001). In spermatozoa diluted with cryosolution B, there was a pronounced increase in the number of cells lacking the head plasmatic membrane from the prefreezing to the postthawed stages (from 32 to 52%, P < 0.01). In both cryosolutions, damages to nuclear membrane were significantly higher after freezing (cryosolution A: 8 to 23%, P < 0.01; cryosolution B: 5 to 38%, P < 0.001). With cryosolution A, the after-activation motility profile confirmed a consistent drop from fresh at the prefreezing stage, whereas freezing and thawing did not affect the motility much further and 50% of the cells were immotile by 60-90 s after activation. With cryosolution B, only the postthawing stage showed a sharp drop of motility profile. This study suggests that the different phases of the cryoprocess should be investigated to better understand the process of sperm damage.")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Abstract"]["CopyrightInformation"], "Copyright 2001 Elsevier Science.")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Affiliation"], u'Dipartimento di Scienze Ambientali, Universit\xe0 degli Studi della Tuscia, 01100 Viterbo, Italy.')
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"].attributes["CompleteYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["LastName"], "Taddei")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["ForeName"], "A R")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][0]["Initials"], "AR")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][1].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][1]["LastName"], "Barbato")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][1]["ForeName"], "F")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][1]["Initials"], "F")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][2].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][2]["LastName"], "Abelli")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][2]["ForeName"], "L")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][2]["Initials"], "L")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][3].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][3]["LastName"], "Canese")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][3]["ForeName"], "S")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][3]["Initials"], "S")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][4].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][4]["LastName"], "Moretti")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][4]["ForeName"], "F")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][4]["Initials"], "F")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][5].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][5]["LastName"], "Rana")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][5]["ForeName"], "K J")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][5]["Initials"], "KJ")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][6].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][6]["LastName"], "Fausto")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][6]["ForeName"], "A M")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][6]["Initials"], "AM")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][7].attributes["ValidYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][7]["LastName"], "Mazzini")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][7]["ForeName"], "M")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["AuthorList"][7]["Initials"], "M")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["Language"], ["eng"])
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["PublicationTypeList"][0], "Journal Article")
        self.assertEqual(record[0]["MedlineCitation"]["Article"]["PublicationTypeList"][1], "Research Support, Non-U.S. Gov't")
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["Country"], "United States")
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["MedlineTA"], "Cryobiology")
        self.assertEqual(record[0]["MedlineCitation"]["MedlineJournalInfo"]["NlmUniqueID"], "0006252")
        self.assertEqual(record[0]["MedlineCitation"]["CitationSubset"], ["IM"])
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"], "Animals")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][0]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"], "Cell Membrane")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["QualifierName"][0], "ultrastructure")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][1]["QualifierName"][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"], "Cryopreservation")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["QualifierName"][0], "methods")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][2]["QualifierName"][0].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"], "Male")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][3]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"], "Microscopy, Electron")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][4]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"], "Microscopy, Electron, Scanning")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][5]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"], "Nuclear Envelope")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["QualifierName"][0], "ultrastructure")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][6]["QualifierName"][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"], "Sea Bream")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][0], "anatomy & histology")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][0].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][1], "physiology")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][7]["QualifierName"][1].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"], "Semen Preservation")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["QualifierName"][0], "adverse effects")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["QualifierName"][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["QualifierName"][1], "methods")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][8]["QualifierName"][1].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"], "Sperm Motility")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][9]["DescriptorName"].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"], "Spermatozoa")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["DescriptorName"].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["QualifierName"][0], "physiology")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["QualifierName"][0].attributes["MajorTopicYN"], "N")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["QualifierName"][1], "ultrastructure")
        self.assertEqual(record[0]["MedlineCitation"]["MeshHeadingList"][10]["QualifierName"][1].attributes["MajorTopicYN"], "Y")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0].attributes["PubStatus"], "pubmed")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Year"], "2001")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Month"], "12")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Day"], "26")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Hour"], "10")
        self.assertEqual(record[0]["PubmedData"]["History"][0][0]["Minute"], "0")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1].attributes["PubStatus"], "medline")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Year"], "2002")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Month"], "3")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Day"], "5")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Hour"], "10")
        self.assertEqual(record[0]["PubmedData"]["History"][0][1]["Minute"], "1")
        self.assertEqual(record[0]["PubmedData"]["PublicationStatus"], "ppublish")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][0], "11748933")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][0].attributes["IdType"], "pubmed")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][1], "10.1006/cryo.2001.2328")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][1].attributes["IdType"], "doi")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][2], "S0011-2240(01)92328-4")
        self.assertEqual(record[0]["PubmedData"]["ArticleIdList"][2].attributes["IdType"], "pii")

        self.assertEqual(record[1]["MedlineCitation"].attributes["Owner"], "NLM")
        self.assertEqual(record[1]["MedlineCitation"].attributes["Status"], "PubMed-not-MEDLINE")
        self.assertEqual(record[1]["MedlineCitation"]["PMID"], "11700088")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Year"], "2001")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Month"], "11")
        self.assertEqual(record[1]["MedlineCitation"]["DateCreated"]["Day"], "08")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Year"], "2001")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Month"], "12")
        self.assertEqual(record[1]["MedlineCitation"]["DateCompleted"]["Day"], "20")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Year"], "2003")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Month"], "10")
        self.assertEqual(record[1]["MedlineCitation"]["DateRevised"]["Day"], "31")
        self.assertEqual(record[1]["MedlineCitation"]["Article"].attributes["PubModel"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISSN"], "1090-7807")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISSN"].attributes["IssnType"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"].attributes["CitedMedium"], "Print")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Volume"], "153")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["Issue"], "1")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Year"], "2001")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["JournalIssue"]["PubDate"]["Month"], "Nov")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["Title"], "Journal of magnetic resonance (San Diego, Calif. : 1997)")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Journal"]["ISOAbbreviation"], "J. Magn. Reson.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["ArticleTitle"], "Proton MRI of (13)C distribution by J and chemical shift editing.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Pagination"]["MedlinePgn"], "117-23")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Abstract"]["AbstractText"], "The sensitivity of (13)C NMR imaging can be considerably favored by detecting the (1)H nuclei bound to (13)C nuclei via scalar J-interaction (X-filter). However, the J-editing approaches have difficulty in discriminating between compounds with similar J-constant as, for example, different glucose metabolites. In such cases, it is almost impossible to get J-edited images of a single-compound distribution, since the various molecules are distinguishable only via their chemical shift. In a recent application of J-editing to high-resolution spectroscopy, it has been shown that a more efficient chemical selectivity could be obtained by utilizing the larger chemical shift range of (13)C. This has been made by introducing frequency-selective (13)C pulses that allow a great capability of indirect chemical separation. Here a double-resonance imaging approach is proposed, based on both J-editing and (13)C chemical shift editing, which achieves a powerful chemical selectivity and is able to produce full maps of specific chemical compounds. Results are presented on a multicompartments sample containing solutions of glucose and lactic and glutamic acid in water.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Abstract"]["CopyrightInformation"], "Copyright 2001 Academic Press.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Affiliation"], "INFM and Department of Physics, University of L'Aquila, I-67100 L'Aquila, Italy.")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"].attributes["CompleteYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["LastName"], "Casieri")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["ForeName"], "C")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][0]["Initials"], "C")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][1].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][1]["LastName"], "Testa")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][1]["ForeName"], "C")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][1]["Initials"], "C")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][2].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][2]["LastName"], "Carpinelli")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][2]["ForeName"], "G")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][2]["Initials"], "G")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][3].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][3]["LastName"], "Canese")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][3]["ForeName"], "R")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][3]["Initials"], "R")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][4].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][4]["LastName"], "Podo")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][4]["ForeName"], "F")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][4]["Initials"], "F")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][5].attributes["ValidYN"], "Y")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][5]["LastName"], "De Luca")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][5]["ForeName"], "F")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["AuthorList"][5]["Initials"], "F")
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["Language"], ["eng"])
        self.assertEqual(record[1]["MedlineCitation"]["Article"]["PublicationTypeList"][0], "Journal Article")
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["Country"], "United States")
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["MedlineTA"], "J Magn Reson")
        self.assertEqual(record[1]["MedlineCitation"]["MedlineJournalInfo"]["NlmUniqueID"], "9707935")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0].attributes["PubStatus"], "pubmed")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Year"], "2001")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Month"], "11")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Day"], "9")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Hour"], "10")
        self.assertEqual(record[1]["PubmedData"]["History"][0][0]["Minute"], "0")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1].attributes["PubStatus"], "medline")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Year"], "2001")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Month"], "11")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Day"], "9")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Hour"], "10")
        self.assertEqual(record[1]["PubmedData"]["History"][0][1]["Minute"], "1")
        self.assertEqual(record[1]["PubmedData"]["PublicationStatus"], "ppublish")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][0], "11700088")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][0].attributes["IdType"], "pubmed")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][1], "10.1006/jmre.2001.2429")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][1].attributes["IdType"], "doi")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][2], "S1090-7807(01)92429-2")
        self.assertEqual(record[1]["PubmedData"]["ArticleIdList"][2].attributes["IdType"], "pii")

    def test_journals(self):
        '''Test parsing XML returned by EFetch, Journals database
        '''
        # In Journals display records for journal IDs 22682,21698,1490
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db="journals", id=["22682","21698","1490"],
        #                       rettype="full", retmode='xml')
        handle = open('Entrez/serialset.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["NlmUniqueID"], "100971611")
        self.assertEqual(record[0]["Title"], "21st century science & technology")
        self.assertEqual(record[0]["MedlineTA"], "21st Century Sci Technol")
        self.assertEqual(len(record[0]["PublicationInfo"]), 8)
        self.assertEqual(record[0]["PublicationInfo"]["Country"], "United States")
        self.assertEqual(record[0]["PublicationInfo"]["Place"], ["[Washington, D.C. :"])
        self.assertEqual(record[0]["PublicationInfo"]["Publisher"], ["21st Century Science Associates,"])
        self.assertEqual(record[0]["PublicationInfo"]["PublicationFirstYear"], "1988")
        self.assertEqual(record[0]["PublicationInfo"]["Frequency"], ["Quarterly,"])
        self.assertEqual(record[0]["PublicationInfo"]["Imprint"], [])
        self.assertEqual(record[0]["PublicationInfo"]["DateIssued"], [])
        self.assertEqual(record[0]["PublicationInfo"]["DatesOfSerialPublication"], [])
        self.assertEqual(record[0]["ISSN"], ["0895-6820"])
        self.assertEqual(record[0]["ISSN"][0].attributes["IssnType"], "Print")
        self.assertEqual(record[0]["Language"], ["eng"])
        self.assertEqual(record[0]["AcidFreeYN"], "N")
        self.assertEqual(record[0]["MinorTitleChangeYN"], "N")
        self.assertEqual(record[0]["CurrentlyIndexedYN"], "N")
        self.assertEqual(record[0]["IndexOnlineYN"], "N")
        self.assertEqual(record[0]["IndexingSubset"], "S")
        self.assertEqual(len(record[0]["CrossReferenceList"]), 5)
        self.assertEqual(record[0]["CrossReferenceList"][0]["XrTitle"], "21 century")
        self.assertEqual(record[0]["CrossReferenceList"][0].attributes["XrType"], "X")
        self.assertEqual(record[0]["CrossReferenceList"][1]["XrTitle"], "21st century science & technology.")
        self.assertEqual(record[0]["CrossReferenceList"][1].attributes["XrType"], "A")
        self.assertEqual(record[0]["CrossReferenceList"][2]["XrTitle"], "21st century science and technology")
        self.assertEqual(record[0]["CrossReferenceList"][2].attributes["XrType"], "X")
        self.assertEqual(record[0]["CrossReferenceList"][3]["XrTitle"], "Twenty-first century science & technology")
        self.assertEqual(record[0]["CrossReferenceList"][3].attributes["XrType"], "X")
        self.assertEqual(record[0]["CrossReferenceList"][4]["XrTitle"], "Twenty-first century science and technology")
        self.assertEqual(record[0]["CrossReferenceList"][4].attributes["XrType"], "X")
        self.assertEqual(record[0]["SortSerialName"], "21ST CENTURY SCIENCE & TECHNOLOGY")
        self.assertEqual(record[0]["IlsCreatedTimestamp"]["Year"], "2000")
        self.assertEqual(record[0]["IlsCreatedTimestamp"]["Month"], "11")
        self.assertEqual(record[0]["IlsCreatedTimestamp"]["Day"], "22")
        self.assertEqual(record[0]["IlsUpdatedTimestamp"]["Year"], "2006")
        self.assertEqual(record[0]["IlsUpdatedTimestamp"]["Month"], "10")
        self.assertEqual(record[0]["IlsUpdatedTimestamp"]["Day"], "21")

        self.assertEqual(record[1].attributes["DataCreationMethod"], "P")
        self.assertEqual(record[1]["NlmUniqueID"], "100939625")
        self.assertEqual(record[1]["Title"], "AIHAJ : a journal for the science of occupational and environmental\nhealth and safety")
        self.assertEqual(record[1]["MedlineTA"], "AIHAJ")
        self.assertEqual(len(record[1]["PublicationInfo"]), 9)
        self.assertEqual(record[1]["PublicationInfo"]["Country"], "United States")
        self.assertEqual(record[1]["PublicationInfo"]["Place"], ["Fairfax, VA :"])
        self.assertEqual(record[1]["PublicationInfo"]["Publisher"], ["American Industrial Hygiene Association,"])
        self.assertEqual(record[1]["PublicationInfo"]["PublicationFirstYear"], "2000")
        self.assertEqual(record[1]["PublicationInfo"]["PublicationEndYear"], "2001")
        self.assertEqual(record[1]["PublicationInfo"]["Frequency"], ["Bimonthly"])
        self.assertEqual(record[1]["PublicationInfo"]["Imprint"], [])
        self.assertEqual(record[1]["PublicationInfo"]["DateIssued"], [])
        self.assertEqual(record[1]["PublicationInfo"]["DatesOfSerialPublication"], [])
        self.assertEqual(record[1]["ISSN"], ["1529-8663"])
        self.assertEqual(record[1]["ISSN"][0].attributes["IssnType"], "Print")
        self.assertEqual(record[1]["Language"], ["eng"])
        self.assertEqual(record[1]["AcidFreeYN"], "N")
        self.assertEqual(record[1]["ContinuationNotes"], "Continues: American Industrial Hygiene Association\njournal. Continued by: AIHA journal. ")
        self.assertEqual(record[1]["MinorTitleChangeYN"], "N")
        self.assertEqual(len(record[1]["IndexingHistoryList"]), 2)
        self.assertEqual(record[1]["IndexingHistoryList"][0].attributes["CitationSubset"], "IM")
        self.assertEqual(record[1]["IndexingHistoryList"][0].attributes["IndexingTreatment"], "Full")
        self.assertEqual(record[1]["IndexingHistoryList"][0].attributes["IndexingStatus"], "Continued-by-another-indexed-title")
        self.assertEqual(record[1]["IndexingHistoryList"][0]["DateOfAction"]["Year"], "2002")
        self.assertEqual(record[1]["IndexingHistoryList"][0]["DateOfAction"]["Month"], "06")
        self.assertEqual(record[1]["IndexingHistoryList"][0]["DateOfAction"]["Day"], "03")
        self.assertEqual(record[1]["IndexingHistoryList"][1].attributes["CitationSubset"], "IM")
        self.assertEqual(record[1]["IndexingHistoryList"][1].attributes["IndexingTreatment"], "Full")
        self.assertEqual(record[1]["IndexingHistoryList"][1].attributes["IndexingStatus"], "Currently-indexed-Title-changed")
        self.assertEqual(record[1]["IndexingHistoryList"][1]["DateOfAction"]["Year"], "2000")
        self.assertEqual(record[1]["IndexingHistoryList"][1]["DateOfAction"]["Month"], "03")
        self.assertEqual(record[1]["IndexingHistoryList"][1]["DateOfAction"]["Day"], "24")
        self.assertEqual(record[1]["IndexingHistoryList"][1]["Coverage"], "v61n1,Jan./Feb. 2000-v62n6,Nov./Dec. 2001")
        self.assertEqual(record[1]["CurrentlyIndexedYN"], "N")
        self.assertEqual(record[1]["IndexOnlineYN"], "N")
        self.assertEqual(record[1]["IndexingSubset"], "IM")
        self.assertEqual(record[1]["BroadJournalHeadingList"][0], "Occupational Medicine")
        self.assertEqual(len(record[1]["CrossReferenceList"]), 2)
        self.assertEqual(record[1]["CrossReferenceList"][0]["XrTitle"], "AIHAJ :")
        self.assertEqual(record[1]["CrossReferenceList"][0].attributes["XrType"], "A")
        self.assertEqual(record[1]["CrossReferenceList"][1]["XrTitle"], "American Industrial Hygiene Association journal")
        self.assertEqual(record[1]["CrossReferenceList"][1].attributes["XrType"], "X")
        self.assertEqual(record[1]["SortSerialName"], "AIHAJ : A JOURNAL FOR THE SCIENCE OF OCCUPATIONAL AND\nENVIRONMENTAL HEALTH AND SAFETY")
        self.assertEqual(record[1]["IlsCreatedTimestamp"]["Year"], "2000")
        self.assertEqual(record[1]["IlsCreatedTimestamp"]["Month"], "03")
        self.assertEqual(record[1]["IlsCreatedTimestamp"]["Day"], "22")
        self.assertEqual(record[1]["IlsUpdatedTimestamp"]["Year"], "2005")
        self.assertEqual(record[1]["IlsUpdatedTimestamp"]["Month"], "11")
        self.assertEqual(record[1]["IlsUpdatedTimestamp"]["Day"], "20")

        self.assertEqual(record[2].attributes["DataCreationMethod"], "P")
        self.assertEqual(record[2]["NlmUniqueID"], "8403252")
        self.assertEqual(record[2]["Title"], "Acta crystallographica. Section B, Structural science")
        self.assertEqual(record[2]["MedlineTA"], "Acta Crystallogr B")
        self.assertEqual(len(record[2]["PublicationInfo"]), 8)
        self.assertEqual(record[2]["PublicationInfo"]["Country"], "Denmark")
        self.assertEqual(record[2]["PublicationInfo"]["Place"], ["Copenhagen"])
        self.assertEqual(record[2]["PublicationInfo"]["Publisher"], ["Munksgaard International Publishers For The International\nUnion Of Crystallography"])
        self.assertEqual(record[2]["PublicationInfo"]["PublicationFirstYear"], "1983")
        self.assertEqual(record[2]["PublicationInfo"]["Frequency"], ["Bimonthly"])
        self.assertEqual(record[2]["PublicationInfo"]["Imprint"], [])
        self.assertEqual(record[2]["PublicationInfo"]["DateIssued"], [])
        self.assertEqual(record[2]["PublicationInfo"]["DatesOfSerialPublication"], [])
        self.assertEqual(len(record[2]["ISSN"]), 2)
        self.assertEqual(record[2]["ISSN"][0], "0108-7681")
        self.assertEqual(record[2]["ISSN"][0].attributes["IssnType"], "Print")
        self.assertEqual(record[2]["ISSN"][1], "1600-5740")
        self.assertEqual(record[2]["ISSN"][1].attributes["IssnType"], "Electronic")
        self.assertEqual(record[2]["ISOAbbreviation"], "Acta Crystallogr., B")
        self.assertEqual(record[2]["Language"], ["eng", "fre", "ger"])
        self.assertEqual(record[2]["AcidFreeYN"], "N")
        self.assertEqual(record[2]["Coden"], "ASBSDK")
        self.assertEqual(record[2]["ContinuationNotes"], "Continues: Acta crystallographica. Section B, Structural\ncrystallography and crystal chemistry. ")
        self.assertEqual(record[2]["MinorTitleChangeYN"], "N")
        self.assertEqual(len(record[2]["IndexingHistoryList"]), 1)
        self.assertEqual(record[2]["IndexingHistoryList"][0].attributes["CitationSubset"], "IM")
        self.assertEqual(record[2]["IndexingHistoryList"][0].attributes["IndexingTreatment"], "Selective")
        self.assertEqual(record[2]["IndexingHistoryList"][0].attributes["IndexingStatus"], "Currently-indexed")
        self.assertEqual(record[2]["IndexingHistoryList"][0]["DateOfAction"]["Year"], "1989")
        self.assertEqual(record[2]["IndexingHistoryList"][0]["DateOfAction"]["Month"], "11")
        self.assertEqual(record[2]["IndexingHistoryList"][0]["DateOfAction"]["Day"], "06")
        self.assertEqual(record[2]["IndexingHistoryList"][0]["Coverage"], "v44n1, 1988-")
        self.assertEqual(record[2]["CurrentlyIndexedYN"], "Y")
        self.assertEqual(record[2]["CurrentlyIndexedForSubset"], [""])
        self.assertEqual(record[2]["CurrentlyIndexedForSubset"][0].attributes["CurrentSubset"], "IM")
        self.assertEqual(record[2]["CurrentlyIndexedForSubset"][0].attributes["CurrentIndexingTreatment"], "Selective")
        self.assertEqual(record[2]["IndexOnlineYN"], "Y")
        self.assertEqual(record[2]["IndexingSubset"], "IM")
        self.assertEqual(record[2]["BroadJournalHeadingList"][0], "Chemistry, Analytical")
        self.assertEqual(len(record[2]["CrossReferenceList"]), 4)
        self.assertEqual(record[2]["CrossReferenceList"][0]["XrTitle"], "ACTA CRYSTALLOGR B")
        self.assertEqual(record[2]["CrossReferenceList"][0].attributes["XrType"], "A")
        self.assertEqual(record[2]["CrossReferenceList"][1]["XrTitle"], "Acta Crystallogr.,Sect.B")
        self.assertEqual(record[2]["CrossReferenceList"][1].attributes["XrType"], "A")
        self.assertEqual(record[2]["CrossReferenceList"][2]["XrTitle"], "Acta crystallographica. Section B, Structural\nscience.")
        self.assertEqual(record[2]["CrossReferenceList"][2].attributes["XrType"], "A")
        self.assertEqual(record[2]["CrossReferenceList"][3]["XrTitle"], "Structural science")
        self.assertEqual(record[2]["CrossReferenceList"][3].attributes["XrType"], "X")
        self.assertEqual(record[2]["SortSerialName"], "ACTA CRYSTALLOGRAPHICA. SECTION B, STRUCTURAL\nSCIENCE")
        self.assertEqual(record[2]["IlsCreatedTimestamp"]["Year"], "1998")
        self.assertEqual(record[2]["IlsCreatedTimestamp"]["Month"], "11")
        self.assertEqual(record[2]["IlsCreatedTimestamp"]["Day"], "05")
        self.assertEqual(record[2]["IlsUpdatedTimestamp"]["Year"], "2008")
        self.assertEqual(record[2]["IlsUpdatedTimestamp"]["Month"], "10")
        self.assertEqual(record[2]["IlsUpdatedTimestamp"]["Day"], "15")

    def test_omim(self):
        '''Test parsing XML returned by EFetch, OMIM database
        '''
        # In OMIM show the full record for MIM number 601100 as XML
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db="omim", id="601100", retmode='xml',
        #                       rettype='full')
        handle = open('Entrez/ncbi_mim.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["Mim-entry_mimNumber"], "601100")
        self.assertEqual(record[0]["Mim-entry_mimType"], "1")
        self.assertEqual(record[0]["Mim-entry_mimType"].attributes["value"], "star")
        self.assertEqual(record[0]["Mim-entry_title"], "STRESS 70 PROTEIN CHAPERONE, MICROSOME-ASSOCIATED, 60-KD; STCH")
        self.assertEqual(record[0]["Mim-entry_copyright"], "Copyright (c) 1966-2008 Johns Hopkins University")
        self.assertEqual(record[0]["Mim-entry_symbol"], "STCH")
        self.assertEqual(record[0]["Mim-entry_locus"], "21q11.1")
        self.assertEqual(len(record[0]["Mim-entry_text"]), 2)
        self.assertEqual(record[0]["Mim-entry_text"][0]["Mim-text_label"], "TEXT")
        self.assertEqual(record[0]["Mim-entry_text"][0]["Mim-text_text"], "The stress-70 chaperone family consists of proteins that bind to denatured or incorrectly folded polypeptides and play a major role in the processing of cytosolic and secretory proteins. {2:Otterson et al. (1994)} cloned a human cDNA encoding a predicted 471-amino acid protein (60 kD) which they designated STCH. {1:Brodsky et al. (1995)} stated that the protein sequence is very similar to that of HSP70 ({140550}) and BiP ({138120}). As with other members of the family, the STCH protein contains an ATPase domain at the amino terminus whose activity was shown to be independent of peptide stimulation. The protein was found to be microsome-associated and constitutively expressed in all cell types examined.")
        self.assertEqual(len(record[0]["Mim-entry_text"][0]["Mim-text_neighbors"]), 1)
        self.assertEqual(record[0]["Mim-entry_text"][0]["Mim-text_neighbors"]["Mim-link"]["Mim-link_num"], "30")
        self.assertEqual(record[0]["Mim-entry_text"][0]["Mim-text_neighbors"]["Mim-link"]["Mim-link_uids"], "8131751,9358068,10675567,9488737,8757872,11048651,2559088,10982831,2105497,16572726,9083109,17181539,14508011,15028727,10651811,9108392,11599566,2661019,11836248,7594475,12406544,8536694,12389629,10430932,9177027,9837933,8522346,2928112,12834280,8702658")
        self.assertEqual(record[0]["Mim-entry_text"][0]["Mim-text_neighbors"]["Mim-link"]["Mim-link_numRelevant"], "0")
        self.assertEqual(record[0]["Mim-entry_text"][1]["Mim-text_label"], "TEXT")
        self.assertEqual(record[0]["Mim-entry_text"][1]["Mim-text_text"], "{1:Brodsky et al. (1995)} mapped the STCH gene to chromosome 21q11.1 with a high-resolution somatic cell hybrid panel for chromosome 21 and by fluorescence in situ hybridization with a YAC containing the gene. By interspecific backcross analysis, {3:Reeves et al. (1998)} mapped the mouse Stch gene to chromosome 16.")
        self.assertEqual(len(record[0]["Mim-entry_text"][1]["Mim-text_neighbors"]), 1)
        self.assertEqual(record[0]["Mim-entry_text"][1]["Mim-text_neighbors"]["Mim-link"]["Mim-link_num"], "30")
        self.assertEqual(record[0]["Mim-entry_text"][1]["Mim-text_neighbors"]["Mim-link"]["Mim-link_uids"], "1354597,8244375,8597637,8838809,9143508,1427875,7806216,9852683,7835904,11060461,10083745,7789175,7806232,7513297,8020937,12014109,1769649,2045096,9747039,8034329,8088815,1783375,8275716,8020959,7956352,8020952,10198174,7655454,8750197,11272792")
        self.assertEqual(record[0]["Mim-entry_text"][1]["Mim-text_neighbors"]["Mim-link"]["Mim-link_numRelevant"], "0")
        self.assertEqual(record[0]["Mim-entry_hasSummary"], "")
        self.assertEqual(record[0]["Mim-entry_hasSummary"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_hasSynopsis"], "")
        self.assertEqual(record[0]["Mim-entry_hasSynopsis"].attributes["value"], "false")
        self.assertEqual(len(record[0]["Mim-entry_editHistory"]), 6)
        self.assertEqual(record[0]["Mim-entry_editHistory"][0]["Mim-edit-item_author"], "terry")
        self.assertEqual(record[0]["Mim-entry_editHistory"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1999")
        self.assertEqual(record[0]["Mim-entry_editHistory"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_editHistory"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "9")
        self.assertEqual(record[0]["Mim-entry_editHistory"][1]["Mim-edit-item_author"], "carol")
        self.assertEqual(record[0]["Mim-entry_editHistory"][1]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1999")
        self.assertEqual(record[0]["Mim-entry_editHistory"][1]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_editHistory"][1]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "7")
        self.assertEqual(record[0]["Mim-entry_editHistory"][2]["Mim-edit-item_author"], "carol")
        self.assertEqual(record[0]["Mim-entry_editHistory"][2]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1998")
        self.assertEqual(record[0]["Mim-entry_editHistory"][2]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "7")
        self.assertEqual(record[0]["Mim-entry_editHistory"][2]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "8")
        self.assertEqual(record[0]["Mim-entry_editHistory"][3]["Mim-edit-item_author"], "terry")
        self.assertEqual(record[0]["Mim-entry_editHistory"][3]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1996")
        self.assertEqual(record[0]["Mim-entry_editHistory"][3]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "5")
        self.assertEqual(record[0]["Mim-entry_editHistory"][3]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "24")
        self.assertEqual(record[0]["Mim-entry_editHistory"][4]["Mim-edit-item_author"], "mark")
        self.assertEqual(record[0]["Mim-entry_editHistory"][4]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1996")
        self.assertEqual(record[0]["Mim-entry_editHistory"][4]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_editHistory"][4]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "1")
        self.assertEqual(record[0]["Mim-entry_editHistory"][5]["Mim-edit-item_author"], "mark")
        self.assertEqual(record[0]["Mim-entry_editHistory"][5]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1996")
        self.assertEqual(record[0]["Mim-entry_editHistory"][5]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_editHistory"][5]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "1")
        self.assertEqual(record[0]["Mim-entry_creationDate"]["Mim-edit-item"]["Mim-edit-item_author"], "Alan F. Scott")
        self.assertEqual(record[0]["Mim-entry_creationDate"]["Mim-edit-item"]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1996")
        self.assertEqual(record[0]["Mim-entry_creationDate"]["Mim-edit-item"]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_creationDate"]["Mim-edit-item"]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "1")
        self.assertEqual(len(record[0]["Mim-entry_references"]), 3)
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_number"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_origNumber"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_type"], "")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_type"].attributes["value"], "citation")
        self.assertEqual(len(record[0]["Mim-entry_references"][0]["Mim-reference_authors"]), 6)
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][0]["Mim-author_name"], "Brodsky, G.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][0]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][1]["Mim-author_name"], "Otterson, G. A.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][1]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][2]["Mim-author_name"], "Parry, B. B.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][2]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][3]["Mim-author_name"], "Hart, I.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][3]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][4]["Mim-author_name"], "Patterson, D.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][4]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][5]["Mim-author_name"], "Kaye, F. J.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_authors"][5]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_primaryAuthor"], "Brodsky")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_otherAuthors"], "et al.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_citationTitle"], "Localization of STCH to human chromosome 21q11.1.")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_citationType"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_volume"], "30")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_journal"], "Genomics")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_year"], "1995")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_month"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_day"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pages"][0]["Mim-page_from"], "627")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pages"][0]["Mim-page_to"], "628")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_pubmedUID"], "8825657")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_ambiguous"], "")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_ambiguous"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_noLink"], "")
        self.assertEqual(record[0]["Mim-entry_references"][0]["Mim-reference_noLink"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_number"], "2")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_origNumber"], "2")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_type"], "")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_type"].attributes["value"], "citation")
        self.assertEqual(len(record[0]["Mim-entry_references"][1]["Mim-reference_authors"]), 6)
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][0]["Mim-author_name"], "Otterson, G. A.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][0]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][1]["Mim-author_name"], "Flynn, G. C.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][1]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][2]["Mim-author_name"], "Kratzke, R. A.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][2]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][3]["Mim-author_name"], "Coxon, A.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][3]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][4]["Mim-author_name"], "Johnston, P. G.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][4]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][5]["Mim-author_name"], "Kaye, F. J.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_authors"][5]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_primaryAuthor"], "Otterson")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_otherAuthors"], "et al.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_citationTitle"], "Stch encodes the 'ATPase core' of a microsomal stress70 protein.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_citationType"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_volume"], "13")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_journal"], "EMBO J.")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_year"], "1994")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_month"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_day"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pages"][0]["Mim-page_from"], "1216")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pages"][0]["Mim-page_to"], "1225")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_pubmedUID"], "8131751")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_ambiguous"], "")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_ambiguous"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_noLink"], "")
        self.assertEqual(record[0]["Mim-entry_references"][1]["Mim-reference_noLink"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_number"], "3")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_origNumber"], "3")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_type"], "")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_type"].attributes["value"], "citation")
        self.assertEqual(len(record[0]["Mim-entry_references"][2]["Mim-reference_authors"]), 4)
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][0]["Mim-author_name"], "Reeves, R. H.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][0]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][1]["Mim-author_name"], "Rue, E.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][1]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][2]["Mim-author_name"], "Yu, J.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][2]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][3]["Mim-author_name"], "Kao, F.-T.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_authors"][3]["Mim-author_index"], "1")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_primaryAuthor"], "Reeves")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_otherAuthors"], "et al.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_citationTitle"], "Stch maps to mouse chromosome 16, extending the conserved synteny with human chromosome 21.")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_citationType"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_volume"], "49")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_journal"], "Genomics")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_year"], "1998")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_month"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pubDate"]["Mim-date"]["Mim-date_day"], "0")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pages"][0]["Mim-page_from"], "156")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pages"][0]["Mim-page_to"], "157")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_pubmedUID"], "9570963")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_ambiguous"], "")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_ambiguous"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_noLink"], "")
        self.assertEqual(record[0]["Mim-entry_references"][2]["Mim-reference_noLink"].attributes["value"], "false")
        self.assertEqual(record[0]["Mim-entry_attribution"][0]["Mim-edit-item_author"], "Carol A. Bocchini - updated")
        self.assertEqual(record[0]["Mim-entry_attribution"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_year"], "1999")
        self.assertEqual(record[0]["Mim-entry_attribution"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_month"], "3")
        self.assertEqual(record[0]["Mim-entry_attribution"][0]["Mim-edit-item_modDate"]["Mim-date"]["Mim-date_day"], "7")
        self.assertEqual(record[0]["Mim-entry_numGeneMaps"], "1")
        self.assertEqual(len(record[0]["Mim-entry_medlineLinks"]), 1)
        self.assertEqual(record[0]["Mim-entry_medlineLinks"]["Mim-link"]["Mim-link_num"], "3")
        self.assertEqual(record[0]["Mim-entry_medlineLinks"]["Mim-link"]["Mim-link_uids"], "8825657,8131751,9570963")
        self.assertEqual(record[0]["Mim-entry_medlineLinks"]["Mim-link"]["Mim-link_numRelevant"], "0")
        self.assertEqual(len(record[0]["Mim-entry_proteinLinks"]), 1)
        self.assertEqual(record[0]["Mim-entry_proteinLinks"]["Mim-link"]["Mim-link_num"], "7")
        self.assertEqual(record[0]["Mim-entry_proteinLinks"]["Mim-link"]["Mim-link_uids"], "148747550,67461586,48928056,30089677,2352621,1351125,460148")
        self.assertEqual(record[0]["Mim-entry_proteinLinks"]["Mim-link"]["Mim-link_numRelevant"], "0")
        self.assertEqual(len(record[0]["Mim-entry_nucleotideLinks"]), 1)
        self.assertEqual(record[0]["Mim-entry_nucleotideLinks"]["Mim-link"]["Mim-link_num"], "5")
        self.assertEqual(record[0]["Mim-entry_nucleotideLinks"]["Mim-link"]["Mim-link_uids"], "148747549,55741785,48928055,2352620,460147")
        self.assertEqual(record[0]["Mim-entry_nucleotideLinks"]["Mim-link"]["Mim-link_numRelevant"], "0")

    def test_taxonomy(self):
        '''Test parsing XML returned by EFetch, Taxonomy database
        '''
        # Access the Taxonomy database using efetch.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db="taxonomy", id="9685", retmode="xml")
        handle = open('Entrez/taxonomy.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(len(record), 1)
        self.assertEqual(record[0]["TaxId"], "9685")
        self.assertEqual(record[0]["ScientificName"], "Felis catus")
        self.assertEqual(record[0]["OtherNames"]["GenbankCommonName"], "domestic cat")
        self.assertEqual(record[0]["OtherNames"]["Synonym"][0], "Felis silvestris catus")
        self.assertEqual(record[0]["OtherNames"]["Synonym"][1], "Felis domesticus")
        self.assertEqual(record[0]["OtherNames"]["CommonName"][0], "cat")
        self.assertEqual(record[0]["OtherNames"]["CommonName"][1], "cats")
        self.assertEqual(record[0]["OtherNames"]["Includes"][0], "Korat cats")
        self.assertEqual(record[0]["ParentTaxId"], "9682")
        self.assertEqual(record[0]["Rank"], "species")
        self.assertEqual(record[0]["Division"], "Mammals")
        self.assertEqual(record[0]["GeneticCode"]["GCId"], "1")
        self.assertEqual(record[0]["GeneticCode"]["GCName"], "Standard")
        self.assertEqual(record[0]["MitoGeneticCode"]["MGCId"], "2")
        self.assertEqual(record[0]["MitoGeneticCode"]["MGCName"], "Vertebrate Mitochondrial")
        self.assertEqual(record[0]["Lineage"], "cellular organisms; Eukaryota; Fungi/Metazoa group; Metazoa; Eumetazoa; Bilateria; Coelomata; Deuterostomia; Chordata; Craniata; Vertebrata; Gnathostomata; Teleostomi; Euteleostomi; Sarcopterygii; Tetrapoda; Amniota; Mammalia; Theria; Eutheria; Laurasiatheria; Carnivora; Feliformia; Felidae; Felinae; Felis")

        self.assertEqual(record[0]["LineageEx"][0]["TaxId"], "131567")
        self.assertEqual(record[0]["LineageEx"][0]["ScientificName"], "cellular organisms")
        self.assertEqual(record[0]["LineageEx"][0]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][1]["TaxId"], "2759")
        self.assertEqual(record[0]["LineageEx"][1]["ScientificName"], "Eukaryota")
        self.assertEqual(record[0]["LineageEx"][1]["Rank"], "superkingdom")
        self.assertEqual(record[0]["LineageEx"][2]["TaxId"], "33154")
        self.assertEqual(record[0]["LineageEx"][2]["ScientificName"], "Fungi/Metazoa group")
        self.assertEqual(record[0]["LineageEx"][2]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][3]["TaxId"], "33208")
        self.assertEqual(record[0]["LineageEx"][3]["ScientificName"], "Metazoa")
        self.assertEqual(record[0]["LineageEx"][3]["Rank"], "kingdom")
        self.assertEqual(record[0]["LineageEx"][4]["TaxId"], "6072")
        self.assertEqual(record[0]["LineageEx"][4]["ScientificName"], "Eumetazoa")
        self.assertEqual(record[0]["LineageEx"][4]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][5]["TaxId"], "33213")
        self.assertEqual(record[0]["LineageEx"][5]["ScientificName"], "Bilateria")
        self.assertEqual(record[0]["LineageEx"][5]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][6]["TaxId"], "33316")
        self.assertEqual(record[0]["LineageEx"][6]["ScientificName"], "Coelomata")
        self.assertEqual(record[0]["LineageEx"][6]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][7]["TaxId"], "33511")
        self.assertEqual(record[0]["LineageEx"][7]["ScientificName"], "Deuterostomia")
        self.assertEqual(record[0]["LineageEx"][7]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][8]["TaxId"], "7711")
        self.assertEqual(record[0]["LineageEx"][8]["ScientificName"], "Chordata")
        self.assertEqual(record[0]["LineageEx"][8]["Rank"], "phylum")
        self.assertEqual(record[0]["LineageEx"][9]["TaxId"], "89593")
        self.assertEqual(record[0]["LineageEx"][9]["ScientificName"], "Craniata")
        self.assertEqual(record[0]["LineageEx"][9]["Rank"], "subphylum")
        self.assertEqual(record[0]["LineageEx"][10]["TaxId"], "7742")
        self.assertEqual(record[0]["LineageEx"][10]["ScientificName"], "Vertebrata")
        self.assertEqual(record[0]["LineageEx"][10]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][11]["TaxId"], "7776")
        self.assertEqual(record[0]["LineageEx"][11]["ScientificName"], "Gnathostomata")
        self.assertEqual(record[0]["LineageEx"][11]["Rank"], "superclass")
        self.assertEqual(record[0]["LineageEx"][12]["TaxId"], "117570")
        self.assertEqual(record[0]["LineageEx"][12]["ScientificName"], "Teleostomi")
        self.assertEqual(record[0]["LineageEx"][12]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][13]["TaxId"], "117571")
        self.assertEqual(record[0]["LineageEx"][13]["ScientificName"], "Euteleostomi")
        self.assertEqual(record[0]["LineageEx"][13]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][14]["TaxId"], "8287")
        self.assertEqual(record[0]["LineageEx"][14]["ScientificName"], "Sarcopterygii")
        self.assertEqual(record[0]["LineageEx"][14]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][15]["TaxId"], "32523")
        self.assertEqual(record[0]["LineageEx"][15]["ScientificName"], "Tetrapoda")
        self.assertEqual(record[0]["LineageEx"][15]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][16]["TaxId"], "32524")
        self.assertEqual(record[0]["LineageEx"][16]["ScientificName"], "Amniota")
        self.assertEqual(record[0]["LineageEx"][16]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][17]["TaxId"], "40674")
        self.assertEqual(record[0]["LineageEx"][17]["ScientificName"], "Mammalia")
        self.assertEqual(record[0]["LineageEx"][17]["Rank"], "class")
        self.assertEqual(record[0]["LineageEx"][18]["TaxId"], "32525")
        self.assertEqual(record[0]["LineageEx"][18]["ScientificName"], "Theria")
        self.assertEqual(record[0]["LineageEx"][18]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][19]["TaxId"], "9347")
        self.assertEqual(record[0]["LineageEx"][19]["ScientificName"], "Eutheria")
        self.assertEqual(record[0]["LineageEx"][19]["Rank"], "no rank")
        self.assertEqual(record[0]["LineageEx"][20]["TaxId"], "314145")
        self.assertEqual(record[0]["LineageEx"][20]["ScientificName"], "Laurasiatheria")
        self.assertEqual(record[0]["LineageEx"][20]["Rank"], "superorder")
        self.assertEqual(record[0]["LineageEx"][21]["TaxId"], "33554")
        self.assertEqual(record[0]["LineageEx"][21]["ScientificName"], "Carnivora")
        self.assertEqual(record[0]["LineageEx"][21]["Rank"], "order")
        self.assertEqual(record[0]["LineageEx"][22]["TaxId"], "379583")
        self.assertEqual(record[0]["LineageEx"][22]["ScientificName"], "Feliformia")
        self.assertEqual(record[0]["LineageEx"][22]["Rank"], "suborder")
        self.assertEqual(record[0]["LineageEx"][23]["TaxId"], "9681")
        self.assertEqual(record[0]["LineageEx"][23]["ScientificName"], "Felidae")
        self.assertEqual(record[0]["LineageEx"][23]["Rank"], "family")
        self.assertEqual(record[0]["LineageEx"][24]["TaxId"], "338152")
        self.assertEqual(record[0]["LineageEx"][24]["ScientificName"], "Felinae")
        self.assertEqual(record[0]["LineageEx"][24]["Rank"], "subfamily")
        self.assertEqual(record[0]["LineageEx"][25]["TaxId"], "9682")
        self.assertEqual(record[0]["LineageEx"][25]["ScientificName"], "Felis")
        self.assertEqual(record[0]["LineageEx"][25]["Rank"], "genus")
        self.assertEqual(record[0]["CreateDate"], "1995/02/27")
        self.assertEqual(record[0]["UpdateDate"], "2007/09/04")
        self.assertEqual(record[0]["PubDate"], "1993/07/26")

    def test_nucleotide1(self):
        '''Test parsing XML returned by EFetch, Nucleotide database (first test)
        '''
        # Access the nucleotide database using efetch.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db='nucleotide', id=5, retmode='xml')
        handle = open('Entrez/nucleotide1.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["GBSeq_locus"], "X60065")
        self.assertEqual(record[0]["GBSeq_length"], "1136")
        self.assertEqual(record[0]["GBSeq_strandedness"], "single")
        self.assertEqual(record[0]["GBSeq_moltype"], "mRNA")
        self.assertEqual(record[0]["GBSeq_topology"], "linear")
        self.assertEqual(record[0]["GBSeq_division"], "MAM")
        self.assertEqual(record[0]["GBSeq_update-date"], "14-NOV-2006")
        self.assertEqual(record[0]["GBSeq_create-date"], "05-MAY-1992")
        self.assertEqual(record[0]["GBSeq_definition"], "B.bovis beta-2-gpI mRNA for beta-2-glycoprotein I")
        self.assertEqual(record[0]["GBSeq_primary-accession"], "X60065")
        self.assertEqual(record[0]["GBSeq_accession-version"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_other-seqids"][0], "emb|X60065.1|")
        self.assertEqual(record[0]["GBSeq_other-seqids"][1], "gi|5")
        self.assertEqual(record[0]["GBSeq_keywords"][0], "beta-2 glycoprotein I")
        self.assertEqual(record[0]["GBSeq_source"], "Bos taurus (cattle)")
        self.assertEqual(record[0]["GBSeq_organism"], "Bos taurus")
        self.assertEqual(record[0]["GBSeq_taxonomy"], "Eukaryota; Metazoa; Chordata; Craniata; Vertebrata; Euteleostomi; Mammalia; Eutheria; Laurasiatheria; Cetartiodactyla; Ruminantia; Pecora; Bovidae; Bovinae; Bos")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_reference"], "1")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][0], "Bendixen,E.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][1], "Halkier,T.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][2], "Magnusson,S.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][3], "Sottrup-Jensen,L.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][4], "Kristensen,T.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_title"], "Complete primary structure of bovine beta 2-glycoprotein I: localization of the disulfide bridges")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_journal"], "Biochemistry 31 (14), 3611-3617 (1992)")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_pubmed"], "1567819")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_reference"], "2")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_position"], "1..1136")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_authors"][0], "Kristensen,T.")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_title"], "Direct Submission")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_journal"], "Submitted (11-JUN-1991) T. Kristensen, Dept of Mol Biology, University of Aarhus, C F Mollers Alle 130, DK-8000 Aarhus C, DENMARK")
        self.assertEqual(len(record[0]["GBSeq_feature-table"]), 7)
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_key"], "source")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_location"], "1..1136")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_to"], "1136")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][0]["GBQualifier_name"], "organism")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][0]["GBQualifier_value"], "Bos taurus")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][1]["GBQualifier_name"], "mol_type")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][1]["GBQualifier_value"], "mRNA")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][2]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][2]["GBQualifier_value"], "taxon:9913")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][3]["GBQualifier_name"], "clone")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][3]["GBQualifier_value"], "pBB2I")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][4]["GBQualifier_name"], "tissue_type")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][4]["GBQualifier_value"], "liver")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_key"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_location"], "<1..1136")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_to"], "1136")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_partial5"], "")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_partial5"].attributes["value"], "true")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_key"], "CDS")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_location"], "<1..1029")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_to"], "1029")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_partial5"], "")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_partial5"].attributes["value"], "true")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][1]["GBQualifier_name"], "codon_start")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][1]["GBQualifier_value"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][2]["GBQualifier_name"], "transl_table")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][2]["GBQualifier_value"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][3]["GBQualifier_name"], "product")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][3]["GBQualifier_value"], "beta-2-glycoprotein I")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][4]["GBQualifier_name"], "protein_id")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][4]["GBQualifier_value"], "CAA42669.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][5]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][5]["GBQualifier_value"], "GI:6")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][6]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][6]["GBQualifier_value"], "GOA:P17690")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][7]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][7]["GBQualifier_value"], "UniProtKB/Swiss-Prot:P17690")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][8]["GBQualifier_name"], "translation")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][8]["GBQualifier_value"], "PALVLLLGFLCHVAIAGRTCPKPDELPFSTVVPLKRTYEPGEQIVFSCQPGYVSRGGIRRFTCPLTGLWPINTLKCMPRVCPFAGILENGTVRYTTFEYPNTISFSCHTGFYLKGASSAKCTEEGKWSPDLPVCAPITCPPPPIPKFASLSVYKPLAGNNSFYGSKAVFKCLPHHAMFGNDTVTCTEHGNWTQLPECREVRCPFPSRPDNGFVNHPANPVLYYKDTATFGCHETYSLDGPEEVECSKFGNWSAQPSCKASCKLSIKRATVIYEGERVAIQNKFKNGMLHGQKVSFFCKHKEKKCSYTEDAQCIDGTIEIPKCFKEHSSLAFWKTDASDVKPC")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_key"], "sig_peptide")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_location"], "<1..48")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_to"], "48")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_partial5"], "")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_partial5"].attributes["value"], "true")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_key"], "mat_peptide")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_location"], "49..1026")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_intervals"][0]["GBInterval_from"], "49")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_intervals"][0]["GBInterval_to"], "1026")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_quals"][1]["GBQualifier_name"], "product")
        self.assertEqual(record[0]["GBSeq_feature-table"][4]["GBFeature_quals"][1]["GBQualifier_value"], "beta-2-glycoprotein I")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_key"], "polyA_signal")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_location"], "1101..1106")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_intervals"][0]["GBInterval_from"], "1101")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_intervals"][0]["GBInterval_to"], "1106")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][5]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_key"], "polyA_site")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_location"], "1130")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_intervals"][0]["GBInterval_point"], "1130")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_intervals"][0]["GBInterval_accession"], "X60065.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_quals"][0]["GBQualifier_name"], "gene")
        self.assertEqual(record[0]["GBSeq_feature-table"][6]["GBFeature_quals"][0]["GBQualifier_value"], "beta-2-gpI")
        self.assertEqual(record[0]["GBSeq_sequence"], "ccagcgctcgtcttgctgttggggtttctctgccacgttgctatcgcaggacgaacctgccccaagccagatgagctaccgttttccacggtggttccactgaaacggacctatgagcccggggagcagatagtcttctcctgccagccgggctacgtgtcccggggagggatccggcggtttacatgcccgctcacaggactctggcccatcaacacgctgaaatgcatgcccagagtatgtccttttgctgggatcttagaaaacggaacggtacgctatacaacgtttgagtatcccaacaccatcagcttttcttgccacacggggttttatctgaaaggagctagttctgcaaaatgcactgaggaagggaagtggagcccagaccttcctgtctgtgcccctataacctgccctccaccacccatacccaagtttgcaagtctcagcgtttacaagccgttggctgggaacaactccttctatggcagcaaggcagtctttaagtgcttgccacaccacgcgatgtttggaaatgacaccgttacctgcacggaacatgggaactggacgcagttgccagaatgcagggaagtaagatgcccattcccatcaagaccagacaatgggtttgtgaaccatcctgcaaatccagtgctctactataaggacaccgccacctttggctgccatgaaacgtattccttggatggaccggaagaagtagaatgcagcaaattcggaaactggtctgcacagccaagctgtaaagcatcttgtaagttatctattaaaagagctactgtgatatatgaaggagagagagtagctatccagaacaaatttaagaatggaatgctgcatggccaaaaggtttctttcttctgcaagcataaggaaaagaagtgcagctacacagaagatgctcagtgcatagacggcaccatcgagattcccaaatgcttcaaggagcacagttctttagctttctggaaaacggatgcatctgacgtaaaaccatgctaagctggttttcacactgaaaattaaatgtcatgcttatatgtgtctgtctgagaatctgatggaaacggaaaaataaagagactgaatttaccgtgtcaagaaaaaaa")

    def test_nucleotide2(self):
        '''Test parsing XML returned by EFetch, Nucleotide database (second test)
        '''
        # Access the nucleotide database using efetch.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db='nucleotide', id=5,
        #                       rettype='fasta', complexity=0, retmode='xml')
        handle = open('Entrez/nucleotide2.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["TSeq_seqtype"], "")
        self.assertEqual(record[0]["TSeq_seqtype"].attributes["value"], "nucleotide")
        self.assertEqual(record[0]["TSeq_gi"], "5")
        self.assertEqual(record[0]["TSeq_accver"], "X60065.1")
        self.assertEqual(record[0]["TSeq_taxid"], "9913")
        self.assertEqual(record[0]["TSeq_orgname"], "Bos taurus")
        self.assertEqual(record[0]["TSeq_defline"], "B.bovis beta-2-gpI mRNA for beta-2-glycoprotein I")
        self.assertEqual(record[0]["TSeq_length"], "1136")
        self.assertEqual(record[0]["TSeq_sequence"], "CCAGCGCTCGTCTTGCTGTTGGGGTTTCTCTGCCACGTTGCTATCGCAGGACGAACCTGCCCCAAGCCAGATGAGCTACCGTTTTCCACGGTGGTTCCACTGAAACGGACCTATGAGCCCGGGGAGCAGATAGTCTTCTCCTGCCAGCCGGGCTACGTGTCCCGGGGAGGGATCCGGCGGTTTACATGCCCGCTCACAGGACTCTGGCCCATCAACACGCTGAAATGCATGCCCAGAGTATGTCCTTTTGCTGGGATCTTAGAAAACGGAACGGTACGCTATACAACGTTTGAGTATCCCAACACCATCAGCTTTTCTTGCCACACGGGGTTTTATCTGAAAGGAGCTAGTTCTGCAAAATGCACTGAGGAAGGGAAGTGGAGCCCAGACCTTCCTGTCTGTGCCCCTATAACCTGCCCTCCACCACCCATACCCAAGTTTGCAAGTCTCAGCGTTTACAAGCCGTTGGCTGGGAACAACTCCTTCTATGGCAGCAAGGCAGTCTTTAAGTGCTTGCCACACCACGCGATGTTTGGAAATGACACCGTTACCTGCACGGAACATGGGAACTGGACGCAGTTGCCAGAATGCAGGGAAGTAAGATGCCCATTCCCATCAAGACCAGACAATGGGTTTGTGAACCATCCTGCAAATCCAGTGCTCTACTATAAGGACACCGCCACCTTTGGCTGCCATGAAACGTATTCCTTGGATGGACCGGAAGAAGTAGAATGCAGCAAATTCGGAAACTGGTCTGCACAGCCAAGCTGTAAAGCATCTTGTAAGTTATCTATTAAAAGAGCTACTGTGATATATGAAGGAGAGAGAGTAGCTATCCAGAACAAATTTAAGAATGGAATGCTGCATGGCCAAAAGGTTTCTTTCTTCTGCAAGCATAAGGAAAAGAAGTGCAGCTACACAGAAGATGCTCAGTGCATAGACGGCACCATCGAGATTCCCAAATGCTTCAAGGAGCACAGTTCTTTAGCTTTCTGGAAAACGGATGCATCTGACGTAAAACCATGCTAAGCTGGTTTTCACACTGAAAATTAAATGTCATGCTTATATGTGTCTGTCTGAGAATCTGATGGAAACGGAAAAATAAAGAGACTGAATTTACCGTGTCAAGAAAAAAA")
        self.assertEqual(record[1]["TSeq_seqtype"], "")
        self.assertEqual(record[1]["TSeq_seqtype"].attributes["value"], "protein")
        self.assertEqual(record[1]["TSeq_gi"], "6")
        self.assertEqual(record[1]["TSeq_accver"], "CAA42669.1")
        self.assertEqual(record[1]["TSeq_taxid"], "9913")
        self.assertEqual(record[1]["TSeq_orgname"], "Bos taurus")
        self.assertEqual(record[1]["TSeq_defline"], "beta-2-glycoprotein I [Bos taurus]")
        self.assertEqual(record[1]["TSeq_length"], "342")
        self.assertEqual(record[1]["TSeq_sequence"], "PALVLLLGFLCHVAIAGRTCPKPDELPFSTVVPLKRTYEPGEQIVFSCQPGYVSRGGIRRFTCPLTGLWPINTLKCMPRVCPFAGILENGTVRYTTFEYPNTISFSCHTGFYLKGASSAKCTEEGKWSPDLPVCAPITCPPPPIPKFASLSVYKPLAGNNSFYGSKAVFKCLPHHAMFGNDTVTCTEHGNWTQLPECREVRCPFPSRPDNGFVNHPANPVLYYKDTATFGCHETYSLDGPEEVECSKFGNWSAQPSCKASCKLSIKRATVIYEGERVAIQNKFKNGMLHGQKVSFFCKHKEKKCSYTEDAQCIDGTIEIPKCFKEHSSLAFWKTDASDVKPC")

    def test_nucleotide2(self):
        '''Test parsing XML returned by EFetch, Protein database
        '''
        # Access the protein database using efetch.
        # To create the XML file, use
        # >>> Bio.Entrez.efetch(db='protein', id=8, rettype='gp', retmode='xml')
        handle = open('Entrez/protein.xml', "rb")
        record = Entrez.read(handle)

        self.assertEqual(record[0]["GBSeq_locus"], "CAA35997")
        self.assertEqual(record[0]["GBSeq_length"], "100")
        self.assertEqual(record[0]["GBSeq_moltype"], "AA")
        self.assertEqual(record[0]["GBSeq_topology"], "linear")
        self.assertEqual(record[0]["GBSeq_division"], "MAM")
        self.assertEqual(record[0]["GBSeq_update-date"], "12-SEP-1993")
        self.assertEqual(record[0]["GBSeq_create-date"], "03-APR-1990")
        self.assertEqual(record[0]["GBSeq_definition"], "unnamed protein product [Bos taurus]")
        self.assertEqual(record[0]["GBSeq_primary-accession"], "CAA35997")
        self.assertEqual(record[0]["GBSeq_accession-version"], "CAA35997.1")
        self.assertEqual(record[0]["GBSeq_other-seqids"][0], "emb|CAA35997.1|")
        self.assertEqual(record[0]["GBSeq_other-seqids"][1], "gi|8")
        self.assertEqual(record[0]["GBSeq_source"], "Bos taurus (cattle)")
        self.assertEqual(record[0]["GBSeq_organism"], "Bos taurus")
        self.assertEqual(record[0]["GBSeq_taxonomy"], "Eukaryota; Metazoa; Chordata; Craniata; Vertebrata; Euteleostomi; Mammalia; Eutheria; Laurasiatheria; Cetartiodactyla; Ruminantia; Pecora; Bovidae; Bovinae; Bos")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_reference"], "1")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_position"], "1..100")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][0], "Kiefer,M.C.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][1], "Saphire,A.C.S.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][2], "Bauer,D.M.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_authors"][3], "Barr,P.J.")
        self.assertEqual(record[0]["GBSeq_references"][0]["GBReference_journal"], "Unpublished")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_reference"], "2")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_position"], "1..100")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_authors"][0], "Kiefer,M.C.")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_title"], "Direct Submission")
        self.assertEqual(record[0]["GBSeq_references"][1]["GBReference_journal"], "Submitted (30-JAN-1990) Kiefer M.C., Chiron Corporation, 4560 Hortom St, Emeryville CA 94608-2916, U S A")
        self.assertEqual(record[0]["GBSeq_comment"], "See <X15699> for Human sequence.~Data kindly reviewed (08-MAY-1990) by Kiefer M.C.")
        self.assertEqual(record[0]["GBSeq_source-db"], "embl accession X51700.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_key"], "source")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_location"], "1..100")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_to"], "100")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_intervals"][0]["GBInterval_accession"], "CAA35997.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][0]["GBQualifier_name"], "organism")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][0]["GBQualifier_value"], "Bos taurus")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][1]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][1]["GBQualifier_value"], "taxon:9913")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][2]["GBQualifier_name"], "clone")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][2]["GBQualifier_value"], "bBGP-3")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][3]["GBQualifier_name"], "tissue_type")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][3]["GBQualifier_value"], "bone matrix")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][4]["GBQualifier_name"], "clone_lib")
        self.assertEqual(record[0]["GBSeq_feature-table"][0]["GBFeature_quals"][4]["GBQualifier_value"], "Zap-bb")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_key"], "Protein")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_location"], "1..100")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_to"], "100")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_intervals"][0]["GBInterval_accession"], "CAA35997.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_quals"][0]["GBQualifier_name"], "name")
        self.assertEqual(record[0]["GBSeq_feature-table"][1]["GBFeature_quals"][0]["GBQualifier_value"], "unnamed protein product")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_key"], "Region")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_location"], "33..97")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_from"], "33")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_to"], "97")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_intervals"][0]["GBInterval_accession"], "CAA35997.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][0]["GBQualifier_name"], "region_name")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][0]["GBQualifier_value"], "Gla")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][1]["GBQualifier_name"], "note")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][1]["GBQualifier_value"], "Vitamin K-dependent carboxylation/gamma-carboxyglutamic (GLA) domain. This domain is responsible for the high-affinity binding of calcium ions. This domain contains post-translational modifications of many glutamate residues by Vitamin K-dependent...; cl02449")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][2]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][2]["GBFeature_quals"][2]["GBQualifier_value"], "CDD:92835")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_key"], "CDS")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_location"], "1..100")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_from"], "1")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_to"], "100")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_intervals"][0]["GBInterval_accession"], "CAA35997.1")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][0]["GBQualifier_name"], "coded_by")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][0]["GBQualifier_value"], "X51700.1:28..330")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][1]["GBQualifier_name"], "note")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][1]["GBQualifier_value"], "bone Gla precursor (100 AA)")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][2]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][2]["GBQualifier_value"], "GOA:P02820")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][3]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][3]["GBQualifier_value"], "InterPro:IPR000294")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][4]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][4]["GBQualifier_value"], "InterPro:IPR002384")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][5]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][5]["GBQualifier_value"], "PDB:1Q3M")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][6]["GBQualifier_name"], "db_xref")
        self.assertEqual(record[0]["GBSeq_feature-table"][3]["GBFeature_quals"][6]["GBQualifier_value"], "UniProtKB/Swiss-Prot:P02820")
        self.assertEqual(record[0]["GBSeq_sequence"], "mrtpmllallalatlclagradakpgdaesgkgaafvskqegsevvkrlrryldhwlgapapypdplepkrevcelnpdcdeladhigfqeayrrfygpv")

    def test_genbank(self):
        '''Test error handling when presented with GenBank non-XML data
        '''
        # Access the nucleotide database using efetch, but return the data
        # in GenBank format.
        # To create the GenBank file, use
        # >>> Bio.Entrez.efetch(db='nucleotide', id='NT_019265', rettype='gb')
        from Bio.Entrez import Parser
        handle = open('GenBank/NT_019265.gb', "rb")
        self.assertRaises(Parser.CorruptedXMLError, Entrez.read, handle)
        handle.close()
        handle = open('GenBank/NT_019265.gb', "rb")
        iterator = Entrez.parse(handle)
        self.assertRaises(Parser.CorruptedXMLError, iterator.next)
        handle.close()

    def test_fasta(self):
        '''Test error handling when presented with Fasta non-XML data
        '''
        from Bio.Entrez import Parser
        handle = open('Fasta/wisteria.nu', "rb")
        self.assertRaises(Parser.CorruptedXMLError, Entrez.read, handle)
        handle.close()
        handle = open('Fasta/wisteria.nu', "rb")
        iterator = Entrez.parse(handle)
        self.assertRaises(Parser.CorruptedXMLError, iterator.next)
        handle.close()


if __name__ == '__main__':
    runner = unittest.TextTestRunner(verbosity = 2)
    unittest.main(testRunner=runner)
