from django.conf import settings
from django.http import HttpResponse
from django.contrib.auth.decorators import permission_required, user_passes_test
from blanc_basic_assets.assets.models import ImageCategory, Image, FileCategory
from .forms import ImageUploadForm, FileUploadForm
from easy_thumbnails.files import get_thumbnailer
import json


UPLOAD_CATEGORY = getattr(settings, 'ASSETS_UPLOAD_CATEGORY', 'Uncategorised')
THUMBNAIL_OPTIONS = getattr(settings, 'ASSETS_THUMBNAIL_OPTIONS', {
    'size': (200, 200),
})


def get_image_list(request):
    image_list = []

    for i in Image.objects.select_related('category').order_by('category__title', 'title'):
        thumbnailer = get_thumbnailer(i.file)
        thumb = thumbnailer.get_thumbnail(THUMBNAIL_OPTIONS)

        image_list.append({
            'title': i.title,
            'folder': i.category.title,
            'image': i.file.url,
            'thumb': thumb.url,
        })

    response = HttpResponse(mimetype='application/json')
    json.dump(image_list, response)
    return response


# Allowing staff to view images is a sane default, but could be changed
image_list = user_passes_test(lambda u: u.is_staff)(get_image_list)


def upload_asset(request, category_class, form_class):
    category, created = category_class.objects.get_or_create(title=UPLOAD_CATEGORY)
    if created:
        category.save()

    form = form_class(request.POST or None, request.FILES or None)

    response = HttpResponse(mimetype='application/json')

    if form.is_valid():
        obj = form.save(commit=False)
        obj.title = obj.file.name
        obj.category = category
        obj.save()

        json.dump({
            'filelink': obj.file.url,
            'filename': obj.title,
        }, response)
        return response

    json.dump({
        'error': 'Upload failed',
    }, response)
    return response


@permission_required('assets.change_image')
def upload_image(request):
    return upload_asset(request, ImageCategory, ImageUploadForm)


@permission_required('assets.change_file')
def upload_file(request):
    return upload_asset(request, FileCategory, FileUploadForm)
