#!/usr/bin/env 3python
"""util: useful auxillary functions and helpers"""

from argparse import ArgumentTypeError
from OpenSSL import crypto
import logging

log = logging.getLogger('blitz-ca.util')

def short_string(length=64):
    def _short_string(s):
        if len(s) > 64:
            raise ArgumentTypeError("Supplied value is too long (Max length: {} Characters)".format(length))
        return s
        
    return _short_string

# Quick Extensions
def generate_ca_extension(is_ca, pathlen=None):
    """Wrapper to spit out the correct x509v3 CA header
    
    :param bool is_ca: indicates if the extension is for a Certificate Authority or not
    :param int pathlen: The maximum number of CAs in the chain underneath this CA
    
    pathlen can be used to limit or terminate the CA chain, setting this value to
    '0' will prevent any certificates being generated by this CA from being a valid 
    Certificate authority while setting this to '1' will allow it to generate a CA 
    Hierarchy with a single intermediate authority
    """
    if is_ca:
        if pathlen is not None:
            return crypto.X509Extension(b'basicConstraints', True, b'CA:TRUE, pathlen:'+bytes(str(pathlen), 'ASCII'))
        else:
            return crypto.X509Extension(b'basicConstraints', True, b'CA:TRUE')
    else:
        return crypto.X509Extension(b'basicConstraints', True, b'CA:FALSE')

def generate_san_extension(domains=[], emails=[], uris=[], rids=[], ips=[], dirnames=[], othernames=[]):
    """Generate the correct data for a Subject alternative name extension"""
    san = ','.join(chain(('DNS:' + name for name in domains),
                         ('email:' + email for email in emails),
                         ('URI:' + uri for uri in uris),
                         ('RID:' + rid for rid in rids),
                         ('IP:' + ip for ip in ips),
                         ('dirName:' + name for name in dirnames),
                         ('otherName:' + name for name in othernames),
                        ))
    san = san.encode('ASCII')
    
    try:
        extension = crypto.X509Extension(b'subjectAltName', False, san)
    except crypto.Error as err:
        raise ValueError(err[0][0][2])
    
    return extension

