__author__="ghermeto"
__date__ ="$27/07/2011 23:23:38$"

import base64
from blitz.api import Curl, ValidationError
from blitz.validation import validate_url, validate

class Request:
    """Represents the request object generated by the sprint. Contains all
        of the headers and POST/PUT data, if any."""
    
    def __init__(self, request):
        """ line: The entire response line (HTTP/1.1 200 Okay, for example)
            method: The method used in the request
            url: The URL, including path, query arguments and hash fragments
            content: All of the response headers (as a Hash of name/value pairs)
            headers: The response content, if any """
        
        self.line = request['line'] if 'line' in request else None
        self.method = request['method'] if 'method' in request else None
        self.url = request['url'] if 'url' in request else None
        if 'content' in request:
            content = bytes(request['content'], "ascii")
            self.content = base64.b64decode(content).decode('UTF-8')
        else:
            self.content = None
        self.headers = request['headers'] if 'headers' in request else None

class Response:
    """ Represents the response object generated by the sprint. Contains all
        of the headers and the response payload, if any."""
        
    def __init__(self, response):
        """ line: The entire response line (HTTP/1.1 200 Okay, for example)
            status: The response status
            message: The message in the response line
            content: All of the response headers (as a Hash of name/value pairs)
            headers: The response content, if any """

        self.line = response['line'] if 'line' in response else None
        self.status = response['status'] if 'status' in response else None
        self.message = response['message'] if 'message' in response else None
        if 'content' in response:
            content = bytes(response['content'], "ascii")
            self.content = base64.b64decode(content).decode('UTF-8')
        else:
            self.content = None
        self.headers = response['headers'] if 'headers' in response else None

class Result:
    """ Contains the result from a successful sprint. """
    
    def __init__(self, result):
        """ region: The region from which this sprint was executed
            duration: The overall response time for the successful hit
            connect: The time it took for the TCP connection
            request: Object with the URL, headers and content,
            response: Object containing the status code, headers and content,"""

        self.region = result['region'] if 'region' in result else None
        self.duration = result['duration'] if 'duration' in result else None
        self.connect = result['connect'] if 'connect' in result else None
        self.request = Request(result['request'])
        self.response = Response(result['response'])

class Sprint(Curl):
    """ Use this to run a sprint against your app. The return values include the
        response time, the region from which the sprint was run along with the 
        full request and response headers and the response body. """
    
    def _validate(self, options):
        """ Raises a ValidationError if validation fails. """
        failed = validate(options)
        if not 'url' in options or not validate_url(options['url']):
            failed.append('url')
        if len(failed) > 0:
            raise ValidationError('Validation error.', failed)
    
    def _format_result(self, result):
        """ Return the sprint result object to be passed to the callback. """
        return Result(result)