__author__="ghermeto"
__date__ ="$27/07/2011 23:23:38$"

import base64
from blitz.api import Curl, ValidationError
from blitz.validation import validate_list, validate

class Request:
    """Represents the request object generated by the sprint. Contains all
        of the headers and POST/PUT data, if any."""
    
    def __init__(self, request):
        """ line: The entire response line (HTTP/1.1 200 Okay, for example)
            method: The method used in the request
            url: The URL, including path, query arguments and hash fragments
            content: All of the response headers (as a Hash of name/value pairs)
            headers: The response content, if any """
        
        self.line = request['line'] if 'line' in request else None
        self.method = request['method'] if 'method' in request else None
        self.url = request['url'] if 'url' in request else None
        if 'content' in request:
            content = bytes(request['content'], "ascii")
            self.content = base64.b64decode(content).decode('UTF-8')
        else:
            self.content = None
        self.headers = request['headers'] if 'headers' in request else None

class Response:
    """ Represents the response object generated by the sprint. Contains all
        of the headers and the response payload, if any."""
        
    def __init__(self, response):
        """ line: The entire response line (HTTP/1.1 200 Okay, for example)
            status: The response status
            message: The message in the response line
            content: All of the response headers (as a Hash of name/value pairs)
            headers: The response content, if any """

        self.line = response['line'] if 'line' in response else None
        self.status = response['status'] if 'status' in response else None
        self.message = response['message'] if 'message' in response else None
        if 'content' in response:
            content = bytes(response['content'], "ascii")
            self.content = base64.b64decode(content).decode('UTF-8')
        else:
            self.content = None
        self.headers = response['headers'] if 'headers' in response else None

class Step:
    """ Represents a step in the transaction (even if there's only one). Each
        step contains the request and response objects as well as the stats
        associated with them. """
        
    def __init__(self, step):
        """ connect: The time it took for the TCP connection 
            duration: The time it took for this step (connect, send and receive)
            request: Object with the URL, headers and content
            response: Object containing the status code, headers and content """
        
        request = step['request']
        response = step['response']
        self.connect = step['connect'] if 'connect' in step else None
        self.duration = step['duration'] if 'duration' in step else None
        self.request = Request(request) if 'request' in step else None
        self.response = Response(response) if 'response' in step else None

class Result:
    """ Contains the result from a successful sprint. """
    
    def __init__(self, result):
        """ region: The region from which this sprint was executed
            duration: The overall response time for the successful hit
            steps: Stats about the individual steps """

        self.region = result['region'] if 'region' in result else None
        self.duration = result['duration'] if 'duration' in result else None
        if 'steps' in result and validate_list(result['steps']):
            def step(s):
                return Step(s)
            self.steps = list(map(step, result['steps']))
        else:
            self.steps = None

class Sprint(Curl):
    """ Use this to run a sprint against your app. The return values include the
        response time, the region from which the sprint was run along with the 
        full request and response headers and the response body. """
    
    def _validate(self, options):
        """ Raises a ValidationError if validation fails. """
        
        failed = validate(options)
        if len(failed) > 0:
            raise ValidationError('Validation error.', failed)
    
    def _format_result(self, result):
        """ Return the sprint result object to be passed to the callback. """
        
        return Result(result)