"""
These system tests are testing the release of fuerte catkin projects.
"""

from __future__ import print_function

import os
import sys

try:
    from vcstools.vcs_abstraction import get_vcs_client
except ImportError:
    print("vcstools was not detected, please install it.", file=sys.stderr)
    sys.exit(1)

from .common import create_release_repo

from ..utils.common import bloom_answer
from ..utils.common import change_directory
from ..utils.common import in_temporary_directory
from ..utils.common import user

from bloom.git import branch_exists
from bloom.git import get_branches
from bloom.git import inbranch

from bloom.util import code


def create_upstream_catkin_fuerte_repository(stack, directory=None):
    upstream_dir = 'upstream_repo_groovy'
    user('mkdir ' + upstream_dir)
    with change_directory(upstream_dir):
        user('git init .')
        user('echo "readme stuff" >> README.md')
        user('git add README.md')
        user('git commit -m "Initial commit" --allow-empty')
        user('git checkout -b fuerte_devel')
        stack_xml = """\
<stack>
  <name>{0}</name>
  <version>0.1.0</version>
  <description>
    ROS Stack named {0}
  </description>
  <author email="foo@bar.com">Foo Baz</author>
  <author>Ping Pong</author>
  <license>BSD</license>
  <copyright>Willow Garage</copyright>
  <url>http://www.ros.org/wiki/{0}</url>
  <build_type>python_distutils</build_type>

  <build_depends>catkin</build_depends>
</stack>
""".format(stack)
        with open('stack.xml', 'w+') as f:
            f.write(stack_xml)
        package_xml = """\
<?xml version="1.0"?>
<package>
  <name>{0}</name>
  <version>1.2.0</version>
  <description>A catkin (groovy) ROS package called '{0}'</description>
  <maintainer email="bar@baz.com">Bar</maintainer>
  <license>BSD</license>

  <url type="bugtracker">https://github.com/ros/this/issues</url>
  <url type="repository">https://github.com/ros/this</url>

  <build_depend>catkin</build_depend>

  <run_depend>catkin</run_depend>
  <!-- required for messages generated by gencpp -->
  <run_depend>roscpp_core</run_depend>
</package>
""".format(stack)
        with open('package.xml', 'w+') as f:
            f.write(package_xml)
        user('git add stack.xml')
        user('git add package.xml')
        user('git commit -m "Releasing version 0.1.0" --allow-empty')
        user('git tag 0.1.0 -m "Releasing version 0.1.0"')
        url = 'file://' + os.getcwd()
    return url


@in_temporary_directory
def test_fuerte_package_repository(directory=None):
    """
    Release a single catkin stack (fuerte) repository.
    """
    directory = directory if directory is not None else os.getcwd()
    # Setup
    upstream_url = create_upstream_catkin_fuerte_repository('foo', directory)
    release_url = create_release_repo(
        upstream_url,
        'git',
        'fuerte_devel',
        'fuerte')
    release_dir = os.path.join(directory, 'foo_release_clone')
    release_client = get_vcs_client('git', release_dir)
    release_client.checkout(release_url)
    with change_directory(release_dir):
        # First run everything
        with bloom_answer(bloom_answer.ASSERT_NO_QUESTION):
            cmd = 'git-bloom-release{0} fuerte'
            if 'BLOOM_VERBOSE' not in os.environ:
                cmd = cmd.format(' --quiet')
            else:
                cmd = cmd.format('')
            user(cmd, silent=False)
        ###
        ### Import upstream tests
        ###
        # does the upstream branch exist?
        assert branch_exists('upstream', local_only=True), "no upstream branch"
        # does the upstrea/0.1.0 tag exist?
        ret, out, err = user('git tag', return_io=True)
        assert out.count('upstream/0.1.0') == 1, "no upstream tag created"
        # Is the package.xml from upstream in the upstream branch now?
        with inbranch('upstream'):
            assert os.path.exists('stack.xml'), \
                "upstream did not import: '" + os.getcwd() + "': " + \
                str(os.listdir(os.getcwd()))
            assert open('stack.xml').read().count('0.1.0'), "not right file"

        ###
        ### Release generator
        ###
        # patch import should have reported OK
        assert ret == code.OK, "actually returned ({0})".format(ret)
        # do the proper branches exist?
        assert branch_exists('release/fuerte/foo'), \
            "no release/foo branch: " + str(get_branches())
        assert branch_exists('patches/release/fuerte/foo'), \
            "no patches/release/foo branch"
        # was the release tag created?
        ret, out, err = user('git tag', return_io=True)
        assert out.count('release/fuerte/foo/0.1.0') == 1, \
            "no release tag created"

        ###
        ### Release generator, again
        ###
        # patch import should have reported OK
        assert ret == code.OK, "actually returned ({0})".format(ret)
        # do the proper branches exist?
        assert branch_exists('release/fuerte/foo'), \
            "no release/foo branch: " + str(get_branches())
        assert branch_exists('patches/release/fuerte/foo'), \
            "no patches/release/fuerte/foo branch"
        # was the release tag created?
        ret, out, err = user('git tag', return_io=True)
        assert out.count('release/fuerte/foo/0.1.0') == 1, \
            "no release tag created"
