import sys
import os
import ConfigParser
import xml.etree.ElementTree as ElementTree
import uuid
import glob
try:
  import cPickle as pickle
except ImportError:
  import pickle

from Boarded.i18n import _
from Boarded.config import DEFAULT_CONFIG, SAVABLE_PROPERTIES, \
                           DEFAULT_PROPERTIES
from Boarded.constants import CONFIG_DIR, CONFIG_FILES, KB_DIRS, \
                              USER_KB_DIR, PROPERTY_FILE


def create_config_if_not_exist():
  if not os.path.isdir(CONFIG_DIR):
    os.makedirs(CONFIG_DIR)
  if not os.path.isdir(USER_KB_DIR):
    os.makedirs(USER_KB_DIR)


class ConfigError(Exception):
  pass


class Keyboard(object):
  def __init__(self, name, keys):
    self.uid = str(uuid.uuid4())
    self.name = name
    self.keys = keys
    self.key_groups = set()
    for key in keys.itervalues():
      if key.group and key.group not in self.key_groups:
        self.key_groups.add(key.group)


class Key(object):
  def __init__(self, size, pos, actions, label, group=None,
               press_after=False):
    self.uid = str(uuid.uuid4())
    self.size = size
    self.pos = pos
    self.actions = actions
    self.label = label
    self.group = group
    self.press_after = press_after


def save_properties(config):
  props = {}
  for propname in SAVABLE_PROPERTIES:
    if propname in config:
      props[propname] = config[propname]
  try:
    data = pickle.dumps(props, pickle.HIGHEST_PROTOCOL)
    f = open(config["property_config_filename"], "wb")
    f.write(data)
    f.close()
  except (IOError):
    return

def load_properties(config):
  for propname, prop in DEFAULT_PROPERTIES.iteritems():
    if not propname in config:
      config[propname] = prop
  try:
    f = open(config["property_config_filename"], "rb")
    data = f.read()
    f.close()
    props = pickle.loads(data)
  except (IOError, pickle.UnpicklingError):
    return
  for propname, prop in props.iteritems():
    config[propname] = prop


def read_config():
  create_config_if_not_exist()

  config = {}

  config["property_config_filename"] = PROPERTY_FILE
  load_properties(config)

  cp = ConfigParser.ConfigParser()
  for section in DEFAULT_CONFIG:
    cp.add_section(section)
    for k, v in DEFAULT_CONFIG[section].iteritems():
      cp.set(section, k, v)
  cp.read(CONFIG_FILES)

  config["colors"] = {}
  config["colors"]["bg"] = cp.get("colors", "bg")
  config["colors"]["key_bg"] = cp.get("colors", "key_bg")
  config["colors"]["key_fg"] = cp.get("colors", "key_fg")
  config["colors"]["key_pressed_bg"] = cp.get("colors", "key_pressed_bg")
  config["colors"]["key_pressed_fg"] = cp.get("colors", "key_pressed_fg")
  config["colors"]["key_locked_bg"] = cp.get("colors", "key_locked_bg")
  config["colors"]["key_locked_fg"] = cp.get("colors", "key_locked_fg")
  config["font"] = cp.get("font", "font") # "Sans Bold"

  config["keyboards"] = {}
  default_keyboard = cp.get("startup", "default_keyboard")
  if default_keyboard == "auto":
    default_keyboard = None
  kbfilenames = []
  for kbdir in KB_DIRS:
    kbfilenames.extend(glob.glob(os.path.join(kbdir, "*")))
  kbfilenames = [kbfilename for kbfilename in kbfilenames \
                 if os.path.isfile(kbfilename)]
  for kbfilename in kbfilenames:
    try:
      kbf = open(kbfilename, "r")
      kbroot = ElementTree.fromstring(kbf.read())
      kbf.close()
    except OSError, err:
      sys.stderr.write("Error loading keyboard file %s: %s\n"
                       % (kbfilename, err))
      continue
    kbs = kbroot.getchildren()
    for _kb in kbs:
      try:
        kbname = _kb.attrib["name"]
      except KeyError:
        raise ConfigError("no name for keyboard")
      kb = {"name":kbname}
      keys = {}
      for _key in _kb:
        try:
          size = _key.attrib["size"].split(",")
          size = (int(size[0]), int(size[1]))
        except KeyError:
          raise ConfigError("no size for key")
        except ValueError:
          raise ConfigError("invalid size for key")
        try:
          pos = _key.attrib["pos"].split(",")
          pos = (int(pos[0]), int(pos[1]))
        except KeyError:
          raise ConfigError("no pos for key")
        except ValueError:
          raise ConfigError("invalid pos for key")
        actions = {}
        attrib = _key.attrib
        if "mod" in attrib:
          actions["mod"] = attrib["mod"]
        elif "action" in attrib:
          actions["action"] = attrib["action"]
        elif "keysym" in attrib:
          try:
            actions["keysym"] = int(attrib["keysym"], 0)
          except ValueError:
            raise ConfigError("invalid hex value")
        elif "unicode" in attrib:
          try:
            actions["ordinal"] = ord(attrib["unicode"])
          except TypeError:
            raise ConfigError("invalid unicode char")
        elif "keycode" in attrib:
          try:
            actions["keycode"] = int(attrib["keycode"])
          except ValueError:
            raise ConfigError("invalid keycode integer")
        try:
          label = attrib["label"].replace("\\n", "\n")
        except KeyError:
          raise ConfigError("no label for key")
        try:
          group = attrib["group"]
          if not group.strip():
            group = None
        except KeyError:
          group = None
        press_after = ("press_after" in attrib and \
                       attrib["press_after"] == "true")
        key = Key(size=size, pos=pos, actions=actions, label=label,
                  group=group, press_after=press_after)
        keys[key.uid] = key
      kb = Keyboard(name=kbname, keys=keys)
      config["keyboards"][kb.uid] = kb
      if default_keyboard is None or default_keyboard == kb.name:
        config["default_keyboard"] = kb.uid

  if "default_keyboard" not in config:
    if config["keyboards"]:
      kb = config["keyboards"][config["keyboards"].iterkeys().next()]
      sys.stderr.write("Warning: default keyboard '%s' does not exist, "
                       "using auto selected one '%s'.\n"
                       % (default_keyboard, kb.name,))
      config["default_keyboard"] = kb.uid
    else:
      config["default_keyboard"] = None

  if "--xembed" in sys.argv[1:]:
    config["xembed"] = True
  else:
    config["xembed"] = False

  return config
