// (C) Copyright 2006 Douglas Gregor <doug.gregor -at- gmail.com>

// Use, modification and distribution is subject to the Boost Software
// License, Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
// http://www.boost.org/LICENSE_1_0.txt)

//  Authors: Douglas Gregor

/** @file serialize.cpp
 *
 *  This file provides Boost.Serialization support for Python objects.
 */
#include <boostmpi/serialize.hpp>
#include <boostmpi/skeleton_and_content.hpp>
#include <boost/mpi.hpp>

namespace boost { namespace python {

struct pickle::data_t {
  object module;
  object dumps;
  object loads;
};


/// Data used for communicating with the Python `pickle' module.
pickle::data_t* pickle::data;

str pickle::dumps(object obj, int protocol)
{
  if (!data) initialize_data();
  return extract<str>((data->dumps)(obj, protocol));
}

object pickle::loads(str s)
{
  if (!data) initialize_data();
  return ((data->loads)(s));
}

void pickle::initialize_data()
{
  data = new data_t;

  handle<> mod = handle<>(allow_null(PyImport_ImportModule("cPickle")));

  if (mod.get() == 0)
  {
    assert(PyErr_Occurred());
    if (!PyErr_GivenExceptionMatches(PyErr_Occurred(), 
          PyExc_ImportError))
      throw error_already_set();

    // The error from the cPickle import was an ImportError.
    // Clear the error and try the regular pickle module instead.

    PyErr_Clear();
    mod = handle<>(PyImport_ImportModule("pickle"));
  }

  data->module = object(mod);
  data->dumps = data->module.attr("dumps");
  data->loads = data->module.attr("loads");
}

} } // end namespace boost::python

BOOST_PYTHON_DIRECT_SERIALIZATION_ARCHIVE_IMPL(
  ::boost::mpi::packed_iarchive,
  ::boost::mpi::packed_oarchive)

namespace boost { namespace mpi { namespace python { namespace detail {

  boost::python::object skeleton_proxy_base_type; 

  // A map from Python type objects to skeleton/content handlers
  typedef std::map<PyTypeObject*, skeleton_content_handler>
    skeleton_content_handlers_type;

  BOOST_MPI_PYTHON_DECL skeleton_content_handlers_type skeleton_content_handlers;

  bool
  skeleton_and_content_handler_registered(PyTypeObject* type)
  {
    return 
      skeleton_content_handlers.find(type) != skeleton_content_handlers.end();
  }

  void 
  register_skeleton_and_content_handler(PyTypeObject* type, 
                                        const skeleton_content_handler& handler)
  {
    skeleton_content_handlers[type] = handler;
  }

} } } } // end namespace boost::mpi::python::detail
