@task
def create_volume(name, size, zone=None):
    """
    Create new EBS with name ``name`` and size ``size`` GB.
    """
    self.log("Creating new EBS volume '%s' (%s GB)..." % (name, size))
    aws = AWS()
    zone = zone or aws.ZONE
    volume = aws.create_volume(size, AWS.ZONE)
    volume.add_tag('Name', name)
    self.log("done.\n")
    return volume


@task
def base(name=None):
    """
    Install base packages plus other tweaks such as setting hostname.

    You will need to ensure that the Whiskey PEM is in your agent or specified
    on the CLI via -i, in order to connect.
    """
    packages = """
        apt-file
        bwm-ng
        git-core
        htop
        traceroute
        tree
    """.split()
    # Update
    with hide('output'):
        sudo("aptitude update")
        sudo("DEBIAN_FRONTEND=noninteractive aptitude -y full-upgrade")
    # Base packages
    for package in packages:
        with hide('output'):
            sudo("aptitude install -y %s" % package)
    # Hostname
    sudo('echo "127.0.0.1 %s" >> /etc/hosts' % name)
    sudo("hostname %s" % name)


def all_instances(aws):
    for reservation in aws.get_all_instances():
        for instance in reservation.instances:
            yield instance


@task
@hosts('')
def list(legend='no'):
    """
    List all Whiskey EC2 instances + metadata.

    Specify ``legend=y`` to print a legend.
    """
    aws = AWS()
    fields = ["S", "T", "Z", "Name", "Public", "Private", "ID"]
    table = PrettyTable(fields)
    for f in ('Name', 'Public', 'Private'):
        table.set_field_align(f, "l")
    for instance in all_instances(aws):
        row = [
            get_size(instance),
            get_state(instance),
            get_az(instance.placement),
            get_hostname(instance.tags.get('Name', BLANK)),
            instance.ip_address or BLANK,
            instance.private_ip_address or BLANK,
            instance.id
        ]
        table.add_row(row)
    if legend in ['y', 'yes']:
        print """
Notes:
    S => Size:
        +: larger than average
        -: smaller than average
    T => sTate:
        =: running
        ^: pending/starting up
        v: shutting down
        x: terminated
        .: stopped
    Z => availability Zone, stripping prefixes:
        * us-east-1 (eg "d")
        * us-east- (eg "2a")
        * us- (eg "west-1b")
        * Anything else is full, eg "europe-2d"
"""
    table.printt(sortby="Name")


def change_instance(name, method, endstate, verb):
    instance = get_instance(name)
    self.log("%s instance ID %s: " % (verb, instance.id))
    # Special-case for renaming terminated instances
    if method == 'terminate':
        name = instance.tags['Name']
        instance.add_tag('Name', 'old-' + name)
    getattr(instance, method)()
    while instance.state != endstate:
        time.sleep(5)
        self.log(".")
        instance.update()
    self.log("done.\n")


@task
def stop(name):
    """
    Stop the instance with given ``name``.
    """
    change_instance(name, "stop", "stopped", "Stopping")

@task
def start(name):
    """
    Start the instance with given ``name``.
    """
    change_instance(name, "start", "running", "Starting")

@task
@hosts('')
def terminate(name):
    """
    Terminate the instance with given ``name``.
    """
    change_instance(name, "terminate", "terminated", "Terminating")


@task
def show(name):
    """
    Display instance's info dict
    """
    aws = AWS()
    reservations = aws.get_all_instances(filters={'tag:Name': [name]})
    pprint.pprint(reservations[0].instances[0].__dict__)


class Volume(object):
    def __init__(self, arg):
        """
        Initialize with volume object or nametag (which will do a lookup).
        """
        if isinstance(arg, types.StringTypes):
            aws = AWS()
            volumes = aws.get_all_volumes(filters={'tag:Name': [arg]})
            if not len(volumes):
                if not arg.startswith('vol-'):
                    arg = 'vol-' + arg
                volumes = aws.get_all_volumes([arg])
            self.v = volumes[0]
        else:
            self.v = arg

    def __getattr__(self, attr):
        return getattr(self.v, attr)

    def __repr__(self):
        return "<Volume %s ('%s')>" % (self._id, self.name)

    @property
    def name(self):
        return self.v.tags.get('Name', BLANK)

    @property
    def status(self):
        return {
            'in-use': '=',
            'available': '.',
            'creating': '^',
            'deleting': 'v',
            'deleted': 'x'
        }.get(self.v.status, self.v.status)

    @property
    def _status(self):
        return self.v.status

    @property
    def id(self):
        return self.v.id[4:]

    @property
    def _id(self):
        return self.v.id

    @property
    def created(self):
        dt = time.strptime(self.v.create_time, "%Y-%m-%dT%H:%M:%S.000Z")
        formatstr = "%Y-%m-%d %H:%M"
        return time.strftime(formatstr, dt)

    def _change(self, method, endstate, verb, wait=False):
        volume = self.v
        self.log("%s volume ID %s: " % (verb, volume.id))
        getattr(volume, method)()
        if wait:
            while volume.status != endstate:
                time.sleep(5)
                self.log(".")
                volume.update()
            self.log("done.\n")
        else:
            self.log("(not waiting)\n")

    def delete(self):
        self._change('delete', 'deleted', "Deleting")


@task
def delete_volume(name):
    """
    Delete EBS volume named ``name`` (an ID will also suffice; vol- may be
    omitted).
    """
    d = ";"
    for n in name.split(d) if d in name else [name]:
        Volume(n).delete()


@task
def list_volumes(legend='no'):
    """
    List all Whiskey EBS volumes. Specify ``legend=y`` for a legend.
    """
    aws = AWS()
    table = PrettyTable(['S', 'Name', 'Z', 'Created', 'ID'])
    for f in ('Name',):
        table.set_field_align(f, "l")
    for volume in aws.get_all_volumes():
        v = Volume(volume)
        table.add_row([
            v.status,
            v.name,
            v.size,
            v.created,
            v.id,
        ])
    if legend in ['y', 'yes']:
        print """
Notes:
    Z => siZe, in GB
    S => State:
        =: in-use (attached)
        .: available (not attached)
        ^: creating
        v: deleting
        x: deleted
"""
    table.printt(sortby='Name')


@task
def attach_volume(volume, instance, device):
    """
    Attach ``volume`` to ``instance`` as ``device``.

    ``volume`` and ``instance`` may be EC2/EBS IDs or names. ``device`` should
    be a Unix device path, eg ``/dev/sda``.
    """
    aws = AWS()
    volume = Volume(volume)
    instance = get_instance(instance)
    self.log("Attaching %s as %s..." % (volume.id, device))
    if not aws.attach_volume(volume._id, instance.id, device):
        print >>sys.stderr, "Attach failed!"
        sys.exit(1)
    self.log("done.\n")


def get_drives():
    """
    Obtain list of drive identifiers, e.g. ['sda', 'sdb', ...].

    Only looks for SATA style identifiers, i.e. sd*.
    """
    with hide('everything'):
        lines = run('egrep "^/dev/sd" /etc/fstab').splitlines()
    identifiers = map(lambda x: x.split()[0].split('/')[-1], lines)
    drives = map(lambda x: re.search(r'(sd.)\d*', x).group(1), identifiers)
    return sorted(drives)


@task
def create_raid1(basename, instance, size, mountpoint, device='/dev/md0'):
    """
    Create ``size`` GB RAID1 mirror on ``instance`` mounted at ``mountpoint``.

    ``basename`` is used as a base for the EBS volume names, so if e.g.
    ``basename='dbslave'``, the volume names may be ``dbslave1`` and
    ``dbslave2``.

    ``instance`` should be an AWS instance ID or nametag.

    To specify a device name for the RAID (eg ``/dev/md1``) override
    ``device``.
    """
    # Install mdadm
    with show('everything'):
        sudo("DEBIAN_FRONTEND=noninteractive aptitude install -y mdadm")
    # Create two EBS volumes of size 'size'
    names = ['%s1' % basename, '%s2' % basename]
    volumes = map(lambda x: new_volume(x, size), names)
    # Obtain new drive letters for these drives
    # (Assumes single-letter drive letters, i.e. sda not sdaa or whever comes
    # after sdz, and also assumes one won't be sitting at alphabet's end)
    highest = get_drives()[-1][-1]
    next_index = alpha.index(highest) + 1
    dev = "/dev/sd%s"
    drives = [dev % alpha[next_index], dev % alpha[next_index + 1]]
    # Attach to instance
    for v, d in zip(volumes, drives):
        attach_volume(v, instance, d)
    with hide('output', 'running', 'warnings'):
        # Wait until new volumes show up for the OS
        cmd = 'fdisk -l | grep "%s"'
        self.log("Waiting for attached drives to appear...")
        with settings(warn_only=True):
            while sudo(cmd % drives[0]).failed or sudo(cmd % drives[1]).failed:
                self.log(".")
                time.sleep(2)
        self.log("done.\n")
        # Run mdadm to create 'device' from the volumes.
        sudo("mdadm --create %s --raid-devices=2 --level=raid1 %s" % (
            device, " ".join(drives)
        ))
        # Persist array
        self.log("Persisting array and mountpoint...")
        array = "ARRAY %s devices=%s level=raid1" % (device, ','.join(drives))
        append('/etc/mdadm/mdadm.conf', array, use_sudo=True)
        # Persist mountpoint
        mount = "%s %s  auto    noatime    0   0" % (device, mountpoint)
        append('/etc/fstab', mount, use_sudo=True)
        self.log("done.\n")
        # Set up filesystem
        self.log("Initializing filesystem...")
        # without -F, we get prompt about targeting disk vs partition
        sudo("mkfs.ext3 -F %s" % device)
        self.log("done.\n")
        # Mount on 'mountpoint'
        self.log("Mounting...")
        sudo("mkdir -p %s" % mountpoint)
        sudo("mount %s" % mountpoint)
        self.log("done.\n")
