!    -*- F90 -*-
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE GUMBEL COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_GUMBEL_F
!   DBV_GUMBEL_F
!   BV_GUMBEL_LOGLIK_F
!   RBV_GUMBEL_F
!   BV_GUMBEL_HFUNC_F
!   BV_GUMBEL_INV_HFUNC_F
!   BV_GUMBEL_MLE


SUBROUTINE PBV_GUMBEL_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: DELTA,POSINF=HUGE(1.0D+00),UMAX,UMIN,B
  INTEGER::N,I

  IF(DELTA.EQ.1D0)THEN
     RES = U1 * U2
  ELSE
     DO I=1,N
        IF((U1(I)==0).OR.(U2(I)==0))THEN
           RES(I) = 0
        ELSE IF(U1(I)==1)THEN
           RES(I) = U2(I)
        ELSE IF(U2(I)==1)THEN
           RES(I) = U1(I)
        ELSE
           UMAX = MAX(U1(I),U2(I))
           UMIN = MIN(U1(I),U2(I))
           IF((LOG(1/UMIN)/LOG(1/UMAX))**DELTA>POSINF)THEN
              RES(I) = UMIN
           ELSE
              B = (1+(LOG(1/UMIN)/LOG(1/UMAX))**DELTA)**(1/DELTA)&
                   *LOG(1/UMAX)
              RES(I) = EXP(-B)
           END IF
        END IF
     END DO
  END IF
END SUBROUTINE PBV_GUMBEL_F


SUBROUTINE DBV_GUMBEL_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),C(N),UD1(N),UD2(N)
  REAL(KIND=8) :: DELTA,POSINF=HUGE(1.0D+00),R
  INTEGER::N,I

  IF(DELTA.EQ.1.0D00)THEN
     RES = 1.0D00
  ELSE
     
     CALL PBV_GUMBEL_F(U1,U2,C,DELTA,N)

     UD1 = LOG(1/U1)
     UD2 = LOG(1/U2)

     DO I = 1,N
        
        R = UD1(I) / UD2(I)
        
        IF(R.LT.1.0D00)THEN
           R = 1.0D00 / R
        END IF

        IF(R**(2.0D00*DELTA).GT.POSINF)THEN
           RES(I) = C(I) / (U1(I)*U2(I)) &
                * ( R**(1-DELTA) &
                + (DELTA-1) * R**(-DELTA) / MIN(UD1(I),UD2(I)))
        ELSE
           RES(I) = C(I) / (U1(I)*U2(I)) &
                * ( R**(DELTA-1) / (1+R**DELTA)**(2-2/DELTA) &
                + (DELTA-1) * R**(DELTA-1) / MIN(UD1(I),UD2(I)) / &
                (1+R**DELTA)**(2-1/DELTA))
        END IF

     END DO

  END IF

END SUBROUTINE DBV_GUMBEL_F

SUBROUTINE BV_GUMBEL_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),C(N),UD1(N),UD2(N)
  REAL(KIND=8) :: DELTA
  INTEGER:: N

  IF(DELTA.EQ.1D0)THEN
     RES = U1
  ELSE

     CALL PBV_GUMBEL_F(U1,U2,C,DELTA,N)

     UD1 = LOG(1.0D00/U1)
     UD2 = LOG(1.0D00/U2)

     RES = C / U2 / ( (UD1/UD2)**DELTA + 1.0D00 )**(1.0D00-1.0D00/DELTA)

  END IF

END SUBROUTINE BV_GUMBEL_HFUNC_F


SUBROUTINE BV_GUMBEL_INV_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),B(N)
  REAL(KIND=8) :: DELTA,Z,LZ,R,POSINF=HUGE(1.00D+00)
  INTEGER::N,I

  IF(DELTA.EQ.1D0)THEN
     RES = U1
     RETURN
  ENDIF

  DO I = 1,N
     Z = 1/(DELTA-1) * EXP(1/(DELTA-1)*LOG(1/U1(I)/U2(I))) * LOG(1/U2(I))
     IF(Z>POSINF)THEN
        LZ = -LOG(DELTA-1)+1/(DELTA-1)*LOG(1/U1(I)/U2(I)) &
             + LOG(LOG(1/U2(I)))
        B(I)=(DELTA-1)*(LZ-LOG(LZ))
     ELSE IF(Z<.01)THEN
        B(I)=(DELTA-1)*LAMBERT_W_TAYLOR(Z)
     ELSE
        B(I)=(DELTA-1)*LAMBERT_W_HALLEY(Z)
     ENDIF
  ENDDO

  DO I = 1,N
     R = B(I)/LOG(1/U2(I))
     IF(R**DELTA>POSINF)THEN
        RES(I) = EXP(-B(I))
     ELSE
        RES(I) = EXP(-LOG(1/U2(I))*(R**DELTA-1)**(1/DELTA))
     ENDIF
  ENDDO

CONTAINS

  FUNCTION LAMBERT_W_HALLEY(Z)RESULT(LAMBERT)
    REAL(KIND=8) :: Z,LAMBERT,W1=1,W2,EPSILON=1.0D-8
    W2 = W1 &
         - (W1*EXP(W1)-Z) &
         / (EXP(W1)*(W1+1)-(W1+2)*(W1*EXP(W1)-Z)/(2*W1+2))
    DO WHILE(ABS(W1-W2)>EPSILON)
       W1 = W2
       W2 = W1 &
            - (W1*EXP(W1)-Z) &
            / (EXP(W1)*(W1+1)-(W1+2)*(W1*EXP(W1)-Z)/(2*W1+2))
    END DO

    LAMBERT = W2
  END FUNCTION LAMBERT_W_HALLEY

  FUNCTION LAMBERT_W_TAYLOR(Z)RESULT(LAMBERT)
    REAL(KIND=8) :: Z,LAMBERT

    LAMBERT = Z-Z**2+1.5*Z**3-8/3*Z**4+125/24*Z**5-7776/720*Z**6

  END FUNCTION LAMBERT_W_TAYLOR
  
END SUBROUTINE BV_GUMBEL_INV_HFUNC_F


SUBROUTINE RBV_GUMBEL_F(U1,U2,DELTA,N)

  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  CALL INIT_RANDOM_SEED()

  CALL RANDOM_NUMBER(RES)
  CALL RANDOM_NUMBER(U2)

  CALL BV_GUMBEL_INV_HFUNC_F(RES,U2,U1,DELTA,N)

END SUBROUTINE RBV_GUMBEL_F



FUNCTION BV_GUMBEL_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_GUMBEL_LOGLIK_F
  INTEGER :: N

  CALL DBV_GUMBEL_F(U1,U2,RES,DELTA,N)

  BV_GUMBEL_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_GUMBEL_LOGLIK_F




SUBROUTINE BV_GUMBEL_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_GUMBEL_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP = 2D0

  LOW_INIT = 1.001D0

  LOGLIK_L = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_GUMBEL_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP, 1.001D0)

END SUBROUTINE BV_GUMBEL_INIT



SUBROUTINE BV_GUMBEL_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  REAL(KIND=8)           :: EPS = 1.5D-08, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_GUMBEL_LOGLIK_F, LOCAL_MIN

  CALL BV_GUMBEL_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_GUMBEL_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_GUMBEL_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_GUMBEL_MLE


