!    -*- F90 -*-
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE CLAYTON COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_CLAYTON_F
!   DBV_CLAYTON_F
!   BV_CLAYTON_LOGLIK_F
!   RBV_CLAYTON_F
!   BV_CLAYTON_HFUNC_F
!   BV_CLAYTON_INV_HFUNC_F
!   BV_CLAYTON_MLE


SUBROUTINE PBV_CLAYTON_F(U1,U2,RES,DELTA,N)
  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N)
  REAL(KIND=8) :: DELTA,V,POSINF=HUGE(0.0D00)
  INTEGER::N,I

  IF(DELTA.EQ.0.0D00)THEN
     RES = U1 * U2
  ELSE
     DO I=1,N
        V = MIN(U1(I),U2(I))
        IF(((1/U1(I))**DELTA+(1/U2(I))**DELTA-1).GT.POSINF)THEN
           RES(I) = V
        ELSE
           RES(I) = ((1/U1(I))**DELTA+(1/U2(I))**DELTA-1)**(-1/DELTA)
        END IF
     END DO
  END IF
  
END SUBROUTINE PBV_CLAYTON_F


SUBROUTINE DBV_CLAYTON_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8) :: U1(N),U2(N),RES(N),C(N),R(N),U1D(N),U2D(N),RD(N)
  REAL(KIND=8) :: DELTA
  INTEGER::N

  IF(DELTA.EQ.0.0D00)THEN
     RES = 1.
  ELSE
     CALL PBV_CLAYTON_F(U1,U2,C,DELTA,N)
     R = U1/U2
     RD = R ** DELTA
     U1D = U1 ** DELTA
     U2D = U2 ** DELTA
     RES = (1+DELTA) * C / (U1*U2) / (RD+1/RD+2-2*U1D-2*U2D+U1D*U2D)
  END IF

END SUBROUTINE DBV_CLAYTON_F


SUBROUTINE BV_CLAYTON_INV_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  ! LOCAL VARIABLE
  REAL(KIND=8) :: POSINF=HUGE(0.0D00),V2,V1
  INTEGER :: I

  IF(DELTA.EQ.0D0)THEN
     RES = U1
     RETURN
  END IF

  DO I=1,N
     V1 = U1(I)**(-DELTA/(DELTA+1.0D00)) - 1.0D00
     V2 = (1.0D00/U2(I))**DELTA
     IF(V2.GE.POSINF)THEN
        RES(I) = V1**(-1.0D00/DELTA) * U2(I) ! OMIT ONE TERM TO HAVE APPROXIMATION
     ELSE
        RES(I) = (V1 * V2 + 1.0D00)**(-1.0D00/DELTA)
     END IF
  END DO

END SUBROUTINE BV_CLAYTON_INV_HFUNC_F

SUBROUTINE BV_CLAYTON_HFUNC_F(U1,U2,RES,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  IF(DELTA.EQ.0D0)THEN
     RES = U1
     RETURN
  END IF

  RES = 1.0D00 / ( (U2/U1)**DELTA + 1.0D00 - U2**DELTA )**(1.0D00+1.0D00/DELTA)

END SUBROUTINE BV_CLAYTON_HFUNC_F


SUBROUTINE RBV_CLAYTON_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  CALL INIT_RANDOM_SEED()

  CALL RANDOM_NUMBER(RES)
  CALL RANDOM_NUMBER(U2)

  CALL BV_CLAYTON_INV_HFUNC_F(RES,U2,U1,DELTA,N)
  

END SUBROUTINE RBV_CLAYTON_F




FUNCTION BV_CLAYTON_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_CLAYTON_LOGLIK_F
  INTEGER :: N

  CALL DBV_CLAYTON_F(U1,U2,RES,DELTA,N)

  BV_CLAYTON_LOGLIK_F = SUM(LOG(RES))

END FUNCTION BV_CLAYTON_LOGLIK_F




SUBROUTINE BV_CLAYTON_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  
  IMPLICIT NONE
  
  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM)
  REAL(KIND=8)          :: LOW_INIT,UPP_INIT
  ! LOCAL
  REAL(KIND=8),EXTERNAL :: BV_CLAYTON_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP = 2D0

  LOW_INIT = 1D-3

  LOGLIK_L = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
  LOGLIK_R = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  DO WHILE(LOGLIK_L.LT.LOGLIK_R)
     LOW_INIT = LOW_INIT + STEP
     
     LOGLIK_L = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_CLAYTON_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
  END DO

  UPP_INIT = LOW_INIT + STEP
  LOW_INIT = MAX(LOW_INIT - STEP, 1D-3)

END SUBROUTINE BV_CLAYTON_INIT



SUBROUTINE BV_CLAYTON_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! EPS IS USED FOR COMPUTING G, WHICH IS GRADIENT OF F, THIS NUMBER
  ! COMES FROM THE _EPSILON USED IN SCIPY.OPTIMIZE.APPROX_FPRIME, AND
  ! _EPSILON = SQRT(NUMPY.FINFO(FLOAT).EPS)
  REAL(KIND=8)           :: EPS = 1.5D-08, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_CLAYTON_LOGLIK_F, LOCAL_MIN

  CALL BV_CLAYTON_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_CLAYTON_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_CLAYTON_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN
  
END SUBROUTINE BV_CLAYTON_MLE

