!    -*- F90 -*-
!
! THIS FILE CONSISTS OF ROUTINES FOR BIVARIATE FRANK COPULA.
!
! LIST OF ROUTINES:
!
!   PBV_FRANK_F
!   DBV_FRANK_F
!   BV_FRANK_LOGLIK_F
!   RBV_FRANK_F
!   BV_FRANK_HFUNC_F
!   BV_FRANK_INV_HFUNC_F
!   BV_FRANK_MLE


SUBROUTINE PBV_FRANK_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8)::U1(N),U2(N),RES(N)
  REAL(KIND=8)::DELTA,THETA
  INTEGER::N

  THETA = -1 * DELTA

  IF(THETA.EQ.0.0D00)THEN
     RES = U1 * U2
  ELSE IF(THETA.GT.0.0D00)THEN
     CALL PBV_FRANK_F_P(U1,U2,THETA,RES,N)
  ELSE
     CALL PBV_FRANK_F_P(1-U1,U2,-THETA,RES,N)
     RES = U2 - RES
  END IF

CONTAINS 

  SUBROUTINE PBV_FRANK_F_P(U1,U2,THETA,RES,N)
    REAL(KIND=8)::U1(N),U2(N),RES(N)
    REAL(KIND=8)::THETA,POSINF=HUGE(1.00D+00)
    REAL(KIND=8)::F1,F2
    INTEGER::N,I

    DO I = 1,N
       F1 = EXP(THETA*(U1(I)+U2(I)-1))
       F2 = EXP(THETA)

       IF((F1.GE.POSINF).AND.(F2.GE.POSINF))THEN
          RES(I) = U1(I) + U2(I) - 1
       ELSE IF((F1.GE.POSINF).AND.(F2.LT.POSINF))THEN
          RES(I) = U1(I) + U2(I) - LOG(F2-1)/THETA
       ELSE IF((F1.LT.POSINF).AND.(F2.GE.POSINF))THEN
          RES(I) = LOG(F1-EXP(THETA*(U1(I)-1))-EXP(THETA*(U2(I)-1))+1)&
               / THETA
       ELSE
          RES(I) = 1 + LOG(F1-EXP(THETA*(U1(I)-1))-EXP(THETA*(U2(I)-1))+1)&
               / THETA - LOG(F2-1)/THETA
       ENDIF
    ENDDO

  END SUBROUTINE PBV_FRANK_F_P

END SUBROUTINE PBV_FRANK_F


SUBROUTINE DBV_FRANK_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8):: U1(N),U2(N),RES(N)
  REAL(KIND=8):: DELTA
  INTEGER::N

  IF(DELTA.EQ.0D0)THEN
     RES = 1D0
     RETURN
  ENDIF

  IF(DELTA.LT.0)THEN
     CALL DBV_FRANK_F_N(U1,U2,RES,DELTA,N)
  ELSE
     CALL DBV_FRANK_F_N(1-U1,U2,RES,-DELTA,N)
  ENDIF

CONTAINS

  SUBROUTINE DBV_FRANK_F_N(U1,U2,RES,DELTA,N)
    REAL(KIND=8):: U1(N),U2(N),RES(N),DENOMINATOR(N)
    REAL(KIND=8):: DELTA
    INTEGER::N

    DENOMINATOR = &
         EXP(DELTA*(U1+U2-1))&
         +EXP(DELTA*(1-U1-U2))&
         +EXP(DELTA*(U2+1-U1))&
         +EXP(DELTA*(U1+1-U2))&
         -2*EXP(DELTA*(1-U1))&
         -2*EXP(DELTA*(1-U2))&
         +2&
         +2*EXP(DELTA)&
         -2*EXP(DELTA*U2)&
         -2*EXP(DELTA*U1)

    RES = -DELTA*(1-EXP(DELTA)) / DENOMINATOR

  END SUBROUTINE DBV_FRANK_F_N

END SUBROUTINE DBV_FRANK_F


SUBROUTINE BV_FRANK_HFUNC_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8)::U1(N),U2(N),RES(N)
  REAL(KIND=8)::DELTA
  INTEGER::N

  IF(DELTA.EQ.0D0)THEN
     RES = U1
     RETURN
  ENDIF

  IF(DELTA.LT.0)THEN
     CALL BV_FRANK_HFUNC_F_N(U1,U2,RES,DELTA,N)
  ELSE
     CALL BV_FRANK_HFUNC_F_N(1-U1,U2,RES,-DELTA,N)
     RES = 1 - RES
  ENDIF

CONTAINS

  SUBROUTINE BV_FRANK_HFUNC_F_N(U1,U2,RES,DELTA,N)
    REAL(KIND=8)::U1(N),U2(N),RES(N)
    REAL(KIND=8)::DELTA,L,R,POSINF=HUGE(1.00D+00)
    REAL(KIND=8)::UD1,UD2,UD12,UDU2,UD
    INTEGER::N,I

    DO I = 1,N

       UD1 = EXP(-DELTA*U1(I))
       UD2 = EXP(-DELTA*U2(I))
       UD12 = EXP(-DELTA*(1-U1(I)-U2(I)))
       UDU2 = EXP(-DELTA*(1-U2(I)))
       UD = EXP(-DELTA)

       IF(UD2.GT.POSINF)THEN
          IF(UD1.GT.POSINF)THEN
             L = UD12
             R = 1
          ELSE
             IF(UDU2.GT.POSINF)THEN
                L = UD12 / (1-1/UD1)
                R = 1
             ELSE
                L = UDU2 / (UD1-1)
                R = 1
             ENDIF
          ENDIF
       ELSE
          IF(UD1.GT.POSINF)THEN
             L = UD12
             R = (UD2-1)/UD2
          ELSE
             IF(UD.GT.POSINF)THEN
                L = UD12 / (1-1/UD1)
                R = (UD2-1)/UD2
             ELSE
                L = (UD-1) / (UD1-1) / UD2
                R = (UD2-1)/UD2
             ENDIF
          ENDIF
       ENDIF
       RES(I) = 1/(L+R)
    ENDDO

  END SUBROUTINE BV_FRANK_HFUNC_F_N

END SUBROUTINE BV_FRANK_HFUNC_F

SUBROUTINE BV_FRANK_INV_HFUNC_F(U1,U2,RES,DELTA,N)
  REAL(KIND=8)::U1(N),U2(N),RES(N)
  REAL(KIND=8)::DELTA
  INTEGER::N

  IF(DELTA.EQ.0D0)THEN
     RES = U1
     RETURN
  END IF

  IF(DELTA<0)THEN
     CALL BV_FRANK_INV_HFUNC_F_N(U1,U2,RES,DELTA,N)
  ELSE
     CALL BV_FRANK_INV_HFUNC_F_N(1-U1,U2,RES,-DELTA,N)
     RES = 1 - RES
  END IF

CONTAINS

  SUBROUTINE BV_FRANK_INV_HFUNC_F_N(U1,U2,RES,DELTA,N)
    REAL(KIND=8)::U1(N),U2(N),RES(N),DELTA,POSINF=HUGE(1.00D+00)
    INTEGER::N,I
    REAL(KIND=8)::UD,UD2,UDU2,XDU2

    UD = EXP(-DELTA)

    DO I = 1,N
       UD2 = EXP(-DELTA*U2(I))
       UDU2 = EXP(-DELTA*(1-U2(I)))
       XDU2 = (UDU2 * U1(I) + 1 - U1(I)) / (1-U1(I)+U1(I)/UD2)
       IF(UD.LE.POSINF)THEN
          RES(I) = -1/DELTA*LOG(&
               (U1(I) * UD + (1-U1(I)) * UD2)&
               / ((1-U1(I))*UD2 + U1(I))&
               )
       ELSE
          IF(XDU2.LE.POSINF)THEN
             RES(I) = -1/DELTA*LOG(XDU2)
          ELSE
             RES(I) = -1/DELTA * LOG(&
                  U1(I)/(1-U1(I)+U1(I)/UD2)&
                  )&
                  + 1 - U2(I)
          END IF
       END IF
    END DO

  END SUBROUTINE BV_FRANK_INV_HFUNC_F_N

END SUBROUTINE BV_FRANK_INV_HFUNC_F


SUBROUTINE RBV_FRANK_F(U1,U2,DELTA,N)

  IMPLICIT NONE
  
  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA
  INTEGER :: N

  CALL INIT_RANDOM_SEED()

  CALL RANDOM_NUMBER(RES)
  CALL RANDOM_NUMBER(U2)

  CALL BV_FRANK_INV_HFUNC_F(RES,U2,U1,DELTA,N)

END SUBROUTINE RBV_FRANK_F




FUNCTION BV_FRANK_LOGLIK_F(U1,U2,DELTA,N)

  IMPLICIT NONE

  REAL(KIND=8) :: U1(N),U2(N),RES(N),DELTA,BV_FRANK_LOGLIK_F
  INTEGER :: N

  CALL DBV_FRANK_F(U1,U2,RES,DELTA,N)
  BV_FRANK_LOGLIK_F = SUM(LOG(RES))
  

END FUNCTION BV_FRANK_LOGLIK_F



! SUBROUTINE BV_FRANK_INIT(U1,U2,NNUM,INIT)
!   IMPLICIT NONE

!   INTEGER      :: NNUM
!   REAL(KIND=8) :: U1(NNUM),U2(NNUM),INIT
!   ! LOCAL
!   REAL(KIND=8) :: M1,M2,UM1(NNUM),UM2(NNUM),R,TAU,ABS_TAU,ABS_ROOT,EPS=1D-8,TOL=1D-2
!   ! REAL(KIND=8),EXTERNAL :: DEBYE1,ZERO
  
!   M1 = SUM(U1) / NNUM
!   M2 = SUM(U2) / NNUM
!   UM1 = U1 - M1
!   UM2 = U2 - M2

!   R = SUM( UM1 * UM2 ) / SQRT( SUM(UM1**2) * SUM(UM2**2) )
!   TAU = ASIN(R) * 2D0 / 3.14159D0

!   IF(TAU.GE.0D0)THEN
!      INIT = 4D0 / (1D0-TAU)
!   ELSE
!      INIT = 4D0 / (TAU-1D0)
!   END IF

!   ! THE RELATIONSHIP BETWEEN KENDALL TAU AND DELTA OF BIVARIATE FRANK
!   ! COPULA IS AS BELOW:
!   ! 
!   !     $\TAU = 1 - 4 / \DELTA ( 1 - 1/\DELTA \INT_0^\DELTA \FRAC{T}{\EXP^T -1})DT$
!   ! 
!   ! USING BRENT ROOT FINDING ALGORITHM FOR THE DELTA INITIAL VALUE.

! !   ABS_TAU = ABS(TAU)
! !   ABS_ROOT = ZERO(1D-2,1D3,EPS,TOL,TARGET_FUN)

! !   IF(TAU.GE.0D0)THEN
! !      INIT = ABS_ROOT
! !   ELSE
! !      INIT = -ABS_ROOT
! !   END IF

! ! CONTAINS

! !   FUNCTION TARGET_FUN(DELTA)
! !     REAL(KIND=8) :: DELTA, TARGET_FUN
! !     TARGET_FUN = 4D0 * DEBYE1(DELTA) / DELTA**2 - 4D0 / DELTA + 1D0 - ABS_TAU
! !   END FUNCTION TARGET_FUN
  
! END SUBROUTINE BV_FRANK_INIT


SUBROUTINE BV_FRANK_INIT(U1,U2,NNUM,LOW_INIT,UPP_INIT)
  IMPLICIT NONE

  INTEGER               :: NNUM
  REAL(KIND=8)          :: U1(NNUM),U2(NNUM),LOW_INIT,UPP_INIT
  REAL(KIND=8),EXTERNAL :: BV_FRANK_LOGLIK_F
  REAL(KIND=8)          :: LOGLIK_L,LOGLIK_R,STEP=2D0
  REAL(KIND=8)          :: M1,M2,UM1(NNUM),UM2(NNUM),R

  M1 = SUM(U1) / NNUM
  M2 = SUM(U2) / NNUM
  UM1 = U1 - M1
  UM2 = U2 - M2

  R = SUM( UM1 * UM2 ) / SQRT( SUM(UM1**2) * SUM(UM2**2) )

  IF(R.GE.0D0)THEN
     
     LOW_INIT = 1D-3
     
     LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
     
     DO WHILE(LOGLIK_L.LT.LOGLIK_R)
        LOW_INIT = LOW_INIT + STEP
        
        LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
        LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT+STEP,NNUM)
     END DO

     UPP_INIT = LOW_INIT + STEP
     LOW_INIT = MAX(LOW_INIT - STEP, 1D-3)

  ELSE
     
     LOW_INIT = -1D-3
     
     LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
     LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT-STEP,NNUM)
     
     DO WHILE(LOGLIK_L.LT.LOGLIK_R)
        LOW_INIT = LOW_INIT - STEP
        
        LOGLIK_L = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT,NNUM)
        LOGLIK_R = BV_FRANK_LOGLIK_F(U1,U2,LOW_INIT-STEP,NNUM)
     END DO

     LOW_INIT = LOW_INIT - STEP
     UPP_INIT = MIN(LOW_INIT + STEP * 2D0, -1D-3)

  END IF


END SUBROUTINE BV_FRANK_INIT


SUBROUTINE BV_FRANK_MLE(U1,U2,NNUM,RES,LL)

  IMPLICIT NONE

  INTEGER                :: NNUM
  REAL(KIND=8)           :: U1(NNUM),U2(NNUM),RES(2),LL
  ! EPS IS USED FOR COMPUTING G, WHICH IS GRADIENT OF F, THIS NUMBER
  ! COMES FROM THE _EPSILON USED IN SCIPY.OPTIMIZE.APPROX_FPRIME, AND
  ! _EPSILON = SQRT(NUMPY.FINFO(FLOAT).EPS)
  REAL(KIND=8)           :: EPS = 1.5D-8, TOL = 1D-3, LOW, UPP, X
  REAL(KIND=8), EXTERNAL :: BV_FRANK_LOGLIK_F, LOCAL_MIN
  
  CALL BV_FRANK_INIT(U1,U2,NNUM,LOW,UPP)

  LL = -LOCAL_MIN(LOW,UPP,EPS,TOL,BV_FRANK_LOGLIK_F,U1,U2,NNUM,X)

  RES(1) = X
  RES(2) = 0D0

! CONTAINS
  
!   FUNCTION TARGET_FUN(DELTA)
!     IMPLICIT NONE
!     REAL(KIND=8) :: DELTA, TARGET_FUN
!     TARGET_FUN = -BV_FRANK_LOGLIK_F(U1,U2,DELTA,NNUM)
!   END FUNCTION TARGET_FUN

END SUBROUTINE BV_FRANK_MLE

