# Copyright 2008 Neil Martinsen-Burrell
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 

"""
Test the updating mechanism.
"""

from cStringIO import StringIO

from bzrlib.plugins.automirror.updater import Updater, NoRemoteBranch

from bzrlib.tests import TestCaseInTempDir
from bzrlib.bzrdir import BzrDir
from bzrlib.delta import TreeDelta


sample_config = ("[DEFAULT]\n"
                 "post_commit_mirror=file:///tmp/test_mirror_on_commit\n")

local_config = ("[DEFAULT]\n"
                 "post_commit_mirror=/tmp/test_mirror_on_commit\n")

relative_config = ("[DEFAULT]\n"
                 "post_commit_mirror=mirror_branch\n")

bogus_config = ("[DEFAULT]\n"
                "post_commit_mirror=qwerty://bogus_url/\n")

ssh_config = ("[DEFAULT]\n"
              "post_commit_mirror=bzr+ssh://remotehost.example.com/path")

sftp_config = ("[DEFAULT]\n"
              "post_commit_mirror=sftp://remotehost.example.com/path")

svn_ssh_config = ("[DEFAULT]\n"
                  "post_commit_mirror=svn+ssh://remotehost.example.com/path")

not_present_config = ("[DEFAULT]")

config_config = ("[DEFAULT]\n"
                 "present = True")

encoded_comma_config = ("[DEFAULT]\n"
                        "post_commit_mirror=file:///tmp/mirror%44branch\n")

multiple_config = ("[DEFAULT]\n"
                   "post_commit_mirror=mirror_branch, mirror2\n")


class TestCaseUpdater(TestCaseInTempDir):

    def get_updater(self, text=sample_config):
        self.branch = BzrDir.create_branch_convenience('from-branch')
        tree = self.branch.bzrdir.open_workingtree()
        tree.commit('foo bar baz\nfuzzy\nwuzzy', rev_id='A',
            allow_pointless=True,
            timestamp=1,
            timezone=0,
            committer="Sample <john@example.com>",
            )
        my_config = self.branch.get_config()
        config_file = StringIO(text)
        (my_config._get_global_config()._get_parser(config_file))
        return Updater(self.branch, 'A', my_config)        


class TestConfigs(TestCaseUpdater):

    """Test framework stolen from Robert Collins's bzr-email plugin."""

    def testExistence(self):
        """Test if our test setup even works."""
        u = self.get_updater()
        self.assertIsInstance(u, Updater)

    def testConfigAdding(self):
        """Is our config getting added?"""
        u = self.get_updater(config_config)
        self.assertEqual(u.config.get_user_option('present'),
                         'True')

    def testTarget(self):
        """Can we get a target branch from our updater?"""
        u = self.get_updater()
        self.assertEqual(['file:///tmp/test_mirror_on_commit'],
                         u.targets())

    def testNoTarget(self):
        u = self.get_updater(not_present_config)
        self.assertEqual([],
                         u.targets())

    def testSSHShouldUpdate(self):
        u = self.get_updater(ssh_config)
        self.assertTrue(u.should_run())

    def testSFTPShouldUpdate(self):
        u = self.get_updater(sftp_config)
        self.assertTrue(u.should_run())

    def testSVNShouldUpdate(self):
        u = self.get_updater(svn_ssh_config)
        self.assertTrue(u.should_run())

    def testShouldUpdateLocalPath(self):
        u = self.get_updater(local_config)
        self.assertTrue(u.should_run())

    def testShouldUpdateRelativePath(self):
        u = self.get_updater(relative_config)
        self.assertTrue(u.should_run())

    def testShouldNotUpdate(self):
        u = self.get_updater(bogus_config)
        self.assertFalse(u.should_run())


class TestErrors(TestCaseUpdater):

    def testNoRemoteBranchMessage(self):
        error = NoRemoteBranch("swedish:///bork/bork/bork")
        self.assertEqualDiff(
            "No branch to mirror to at swedish:///bork/bork/bork.\n"
            "Please create the remote branch to mirror to.", str(error))

    def testBogusBranch(self):
        """updater shouldn't run"""
        u = self.get_updater(bogus_config)
        self.assertIs(u.mirror_to_targets(), None)

    def testNoBranch(self):
        """branch shouldn't be found"""
        u = self.get_updater(relative_config)


class TestMirror(TestCaseUpdater):

    def testLocalUpdate(self):
        mirror_branch = BzrDir.create_branch_convenience("mirror_branch")
        u = self.get_updater(relative_config)
        u.mirror_to_targets()
        # How to check that the remote branch has been updated?
        self.assertEqual(mirror_branch.revision_history(),
                         u.branch.revision_history()
                        )
        # Check the working trees too
        mirror_wt = mirror_branch.bzrdir.open_workingtree()
        from_wt = u.branch.bzrdir.open_workingtree()
        self.assertEqual(mirror_wt.changes_from(from_wt),
                         TreeDelta()
                        )

    def testRemoteUpdate(self):
        """How can we test a remote URL in a unit test?"""
        pass


class TestMultipleLocations(TestCaseUpdater):

    def testEncodedComma(self):
        u = self.get_updater(encoded_comma_config)
        assert len(u.targets()) == 1

    def testMultipleConfiguration(self):
        mirror_branch = BzrDir.create_branch_convenience("mirror_branch")
        second_branch = BzrDir.create_branch_convenience("mirror2")
        u = self.get_updater(multiple_config)
        u.mirror_to_targets()
        # How to check that the remote branch has been updated?
        self.assertEqual(mirror_branch.revision_history(),
                         u.branch.revision_history()
                        )
        self.assertEqual(second_branch.revision_history(),
                         u.branch.revision_history()
                        )
        # Check the working trees too
        mirror_wt = mirror_branch.bzrdir.open_workingtree()
        second_wt = second_branch.bzrdir.open_workingtree()
        from_wt = u.branch.bzrdir.open_workingtree()
        self.assertEqual(mirror_wt.changes_from(from_wt),
                         TreeDelta()
                        )
        self.assertEqual(second_wt.changes_from(from_wt),
                         TreeDelta()
                        )


