#
# Copyright (C) 2012-2013 Craig Hobbs
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#

from ..compat import basestring_, long_, unicode_

import datetime
import decimal


# pyodbc resource mock factory
class PyodbcMockFactory(object):

    # executeCallback(resourceString, query, *args), returns [(columnNames, columnDatas), ...]
    def __init__(self, executeCallback = None, searchescape = '\\'):
        self.executeCallback = executeCallback or SimpleExecuteCallback()
        self._searchescape = searchescape

    def open(self, resourceString):
        return PyodbcConnectionMock(resourceString, self.executeCallback, self._searchescape)

    def close(self, resource):
        resource.close()


# pyodbc module mock
class PyodbcMock(object):

    class Error(Exception):
        pass

    class DatabaseError(Error):
        pass

    class DataError(DatabaseError):
        pass

    class OperationalError(DatabaseError):
        pass

    class IntegrityError(DatabaseError):
        pass

    class InternalError(DatabaseError):
        pass

    class ProgrammingError(DatabaseError):
        pass


# pyodbc connection mock
class PyodbcConnectionMock(object):

    pyodbc = PyodbcMock

    def __init__(self, resourceString, executeCallback, searchescape):

        # Parse resource string for properties
        properties = dict((k.lower(), v) for k, v in (x.split('=') for x in resourceString.split(';') if x))

        self.executeCallback = executeCallback
        self._searchescape = searchescape
        self._autocommit = (properties.get('autocommit', 'false') == 'true')
        self._readonly = (properties.get('readonly', 'false') == 'true')
        self._timeout = int(properties.get('timeout', '0'))
        self._unicode_results = (properties.get('unicode_results', 'false') == 'true')
        self._isClosed = False
        self._cursors = []

    @property
    def autocommit(self):
        return self._autocommit

    @autocommit.setter
    def autocommit(self, value):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to set autocommit on an already-closed connection')

        if not isinstance(value, bool):
            raise PyodbcMock.ProgrammingError('Attempt to set autocommit to a non-bool value')

        self._autocommit = value

    def close(self):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to close an already-closed connection')

        for cursor in self._cursors:
            if not cursor._isClosed:
                raise PyodbcMock.ProgrammingError('Attempt to close a connection with open cursors')

        self._isClosed = True

    def cursor(self):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to get a cursor on an already-closed connection')

        cursor = PyodbcCursorMock(self)
        self._cursors.append(cursor)
        return PyodbcCursorContext(cursor)

    def callproc(self, procname, *args):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to callproc with an already-closed connection')

        cursor = PyodbcCursorMock(self)
        argsOut = cursor.callproc(procname, *args)
        self._cursors.append(cursor)
        return argsOut, PyodbcCursorContext(cursor)

    def execute(self, query, *args):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to execute with an already-closed connection')

        cursor = PyodbcCursorMock(self).execute(query, *args)
        self._cursors.append(cursor)
        return PyodbcCursorContext(cursor)

    def rollback(self):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to rollback an already-closed connection')

    @property
    def searchescape(self):
        return self._searchescape

    @property
    def timeout(self):
        return self._timeout

    @timeout.setter
    def timeout(self, value):

        if self._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to set timeout on an already-closed connection')

        if not isinstance(value, (int, long_)) or isinstance(value, bool):
            raise PyodbcMock.ProgrammingError('Attempt to set timeout to a non-integer value')

        self._timeout = value


# pyodbc row mock
class PyodbcRowMock(object):

    def __init__(self, columnNames, columnData):

        assert len(columnNames) == len(columnData)
        self.columnNames = columnNames
        self.columnData = columnData

    def __getitem__(self, ixColumn):

        try:
            return self.columnData[ixColumn]
        except:
            raise PyodbcMock.ProgrammingError('Attempt to get invalid row column index %r' % (ixColumn,))

    def __getattr__(self, columnName):

        try:
            ixColumn = self.columnNames.index(columnName)
            return self.columnData[ixColumn]
        except:
            raise PyodbcMock.ProgrammingError('Attempt to get invalid row column name %r' % (columnName,))

    def __nonzero__(self):

        return True


# Assert rowSets is properly formed - [(columnNames, columnDatas), ...]
def assertRowSets(rowSets, connection = None):

    assert isinstance(rowSets, (tuple, list)), \
        'Invalid rowset collection %r - list expected' % (rowSets,)
    for rowSet in rowSets:
        assert isinstance(rowSet[0], (tuple)), \
            'Invalid rowset %r - tuple expected' % (rowSet,)
        assert len(rowSet) == 2, \
            'Invalid rowset %r - rowset should have only two elements' % (rowSet,)
        assert isinstance(rowSet[0], (tuple)), \
            'Invalid column names collection %r - tuple expected' % (rowSet[1],)
        for columnName in rowSet[0]:
            assert isinstance(columnName, str), \
                'Invalid column name %r - string expected' % (columnName,)
        assert isinstance(rowSet[1], (tuple, list)), \
            'Invalid rows collection %r - list expected' % (rowSet[1],)
        for columnData in rowSet[1]:
            assert isinstance(columnData, (tuple, list)), \
                'Invalid row %r - tuple expected' % (columnData,)
            assert len(rowSet[0]) == len(columnData), \
                'Column data tuple has different length than column names tuple (%r, %r)' % (rowSet[0], columnData)
            for data in columnData:
                if connection is not None and isinstance(data, basestring_):
                    if connection._unicode_results:
                        assert isinstance(data, unicode_), 'unicode_results connection result set strings must be unicode'
                    else:
                        assert isinstance(data, str), 'non-unicode_results connection result set strings must be non-unicode'
                else:
                    assert isinstance(data, (type(None),
                                             basestring_,
                                             bytearray,
                                             bool,
                                             datetime.date,
                                             datetime.time,
                                             datetime.datetime,
                                             int,
                                             long_,
                                             float,
                                             decimal.Decimal)), \
                                             'Invalid column data %r' % (data,)


# pyodbc cursor mock
class PyodbcCursorMock(object):

    def __init__(self, connection):

        self.connection = connection
        self._reset()

    def _reset(self):

        self._isClosed = False
        self._isCommit = False
        self._rowSets = None
        self._ixRowSet = 0
        self._ixRow = 0

    def close(self):

        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to close an already-closed cursor')

        self._isClosed = True

    def commit(self):

        if self.connection.autocommit:
            raise PyodbcMock.ProgrammingError('Attempt to commit an autocommit cursor')
        if self._rowSets is None:
            raise PyodbcMock.ProgrammingError('Attempt to commit cursor before execute')
        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to commit an already-closed cursor')
        if self._isCommit:
            raise PyodbcMock.ProgrammingError('Attempt to commit an already-committed cursor')

        self._isCommit = True

    def callproc(self, procname, *args):

        assert isinstance(procname, str)

        if self._rowSets is not None:
            raise PyodbcMock.ProgrammingError('Attempt to callproc on an already-executed-on cursor')
        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to callproc on a closed cursor')

        # Call the execute callback
        self._rowSets = self.connection.executeCallback(procname, *args)
        assertRowSets(self._rowSets, connection = self.connection)

        return args

    def execute(self, query, *args):

        assert isinstance(query, str)

        if self._rowSets is not None:
            raise PyodbcMock.ProgrammingError('Attempt to execute on an already-executed-on cursor')
        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to execute on a closed cursor')

        # Call the execute callback
        self._rowSets = self.connection.executeCallback(query, *args)
        assertRowSets(self._rowSets, connection = self.connection)

        return self

    def nextset(self):

        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to call nextset on a closed cursor')
        if self._isCommit:
            raise PyodbcMock.ProgrammingError('Attempt to call nextset on a committed cursor')
        if self._rowSets is None or self._ixRowSet + 1 >= len(self._rowSets):
            raise PyodbcMock.ProgrammingError('Attempt to call nextset when no more row sets available')

        self._ixRowSet += 1
        self._ixRow = 0

    def fetchone(self):
        try:
            return self.__next__()
        except StopIteration:
            return None

    def __iter__(self):
        return self

    def __next__(self):

        if self._isClosed or self.connection._isClosed:
            raise PyodbcMock.ProgrammingError('Attempt to iterate a closed cursor')
        if self._isCommit:
            raise PyodbcMock.ProgrammingError('Attempt to iterate a committed cursor')
        if self._rowSets is None:
            raise PyodbcMock.ProgrammingError('Attempt to iterate a cursor before execute')

        if self._ixRowSet >= len(self._rowSets) or self._ixRow >= len(self._rowSets[self._ixRowSet][1]):
            raise StopIteration

        rowSet = self._rowSets[self._ixRowSet]
        row = PyodbcRowMock(rowSet[0], rowSet[1][self._ixRow])
        self._ixRow += 1
        return row

    def next(self):
        return self.__next__()


# pyodbc cursor context manager
class PyodbcCursorContext(object):

    def __init__(self, cursor):
        self.cursor = cursor

    def __enter__(self):
        return self.cursor

    def __exit__(self, exc_type, exc_value, traceback):
        self.cursor.close()


# Simple execute callback
class SimpleExecuteCallback(object):

    def __init__(self):

        self.executes = {}
        self.executeCount = 0

    def addRowSets(self, query, args, rowSets):

        assert isinstance(args, tuple)
        assertRowSets(rowSets)

        key = (query, args)
        if key not in self.executes:
            self.executes[key] = []
        self.executes[key].append(rowSets)

    def __call__(self, query, *args):

        key = (query, args)
        if key in self.executes and len(self.executes[key]) > 0:
            self.executeCount += 1
            return self.executes[key].pop(0)
        else:
            raise PyodbcMock.DatabaseError('No row sets for %r' % (key,))
