#
# Copyright (C) 2012-2013 Craig Hobbs
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#

import pyodbc


# pyodbc connection resource factory
class PyodbcResourceFactory(object):
    __slots__ = ()

    def open(self, resourceString):
        return PyodbcResource(resourceString)

    def close(self, resource):
        resource.close()


# pyodbc connection resource
#
# This is not just a pyodbc.Connection object for the following reasons:
#
#   - Wrap methods that return cursors to return cursor context manager objects to ensure
#     that cursors are only used within an open/close context.
#
#   - Expose pyodbc
#
# Sadly, pyodbc.Connection not sub-class-able.
#
class PyodbcResource(object):
    __slots__ = ('_connection')

    pyodbc = pyodbc

    def __init__(self, resourceString):

        # Parse resource string for properties
        properties = dict((k.lower(), v) for k, v in (x.split('=', 1) for x in resourceString.split(';') if x))

        # Create the connection
        self._connection = pyodbc.connect(resourceString,
                                          ansi = (properties.get('ansi', 'false').lower() == 'true'),
                                          autocommit = (properties.get('autocommit', 'false') == 'true'),
                                          readonly = (properties.get('readonly', 'false') == 'true'),
                                          timeout = int(properties.get('timeout', '0')),
                                          unicode_results = (properties.get('unicode_results', 'false') == 'true'))

    @property
    def autocommit(self):
        return self._connection.autocommit

    @autocommit.setter
    def autocommit(self, value):
        self._connection.autocommit = value

    def close(self):
        self._connection.close()

    def commit(self):
        self._connection.commit()

    def cursor(self):
        return PyodbcCursorContext(self._connection.cursor())

    def callproc(self, procname, *args):
        cursor = self._connection.cursor()
        try:
            argsOut = cursor.callproc(procname, *args)
            return argsOut, PyodbcCursorContext(cursor)
        except:
            cursor.close()
            raise

    def execute(self, query, *args):
        cursor = self._connection.cursor()
        try:
            cursor.execute(query, *args)
            return PyodbcCursorContext(cursor)
        except:
            cursor.close()
            raise

    def rollback(self):
        self._connection.rollback()

    @property
    def searchescape(self):
        return self._connection.searchescape

    @property
    def timeout(self):
        return self._connection.timeout

    @timeout.setter
    def timeout(self, value):
        self._connection.timeout = value


# pyodbc cursor context manager - cursor is only available within a context
class PyodbcCursorContext(object):
    __slots__ = ('_cursor')

    def __init__(self, cursor):
        self._cursor = cursor

    def __enter__(self):
        return self._cursor

    def __exit__(self, exc_type, exc_value, traceback):
        self._cursor.close()
