import os

from chatty_file import ChattyFile


class Repository(object):
    """
    Representation of the SCM / Dropbox folder that is being used for sharing
    dotfiles between machines. Also provides the functions that translate file
    names and paths from those in the store_dir and the user's home_dir.
    """

    START_OK = 0
    ERROR_REPO_NOT_FOUND = 1
    ERROR_STORE_NOT_FOUND = 2
    ERROR_STORE_EMPTY = 3

    def __init__(self, feedback):
        self.feedback = feedback
        self.home_dir = os.path.expanduser('~')
        self.repository_dir = os.path.join(self.home_dir, 'dotfiles')
        self.store_dir = os.path.join(self.repository_dir, 'store')

    def start(self):
        """
        Look for repository and store

        :returns: START_OK for success, int for error
        """
        if not os.path.isdir(self.store_dir):
            if not os.path.isdir(self.repository_dir):
                # No repository folder
                self.feedback.output(
                    self.feedback.ERROR,
                    'Repository not found at {}'.format(self.repository_dir)
                )
                return self.ERROR_REPO_NOT_FOUND

            # No store folder in repository
            self.feedback.output(
                self.feedback.ERROR,
                'Store not found at {}'.format(self.store_dir)
            )
            return self.ERROR_STORE_NOT_FOUND

        self.feedback.output(
            self.feedback.INFO,
            'Found {}'.format(self.store_dir),
            space_after=True
        )

        return self.START_OK

    def walk_and_talk(self):
        """
        For each file in the store take action

        :returns: Number of files found
        """
        first = True
        count = 0
        for dirpath, dirnames, filenames in os.walk(self.store_dir):

            # Check if there are no files
            if first and len(filenames) == 0 and len(dirnames) == 0:
                self.feedback.output(
                    self.feedback.WARNING,
                    'No files found in store at {}'.format(self.store_dir)
                )
                # TODO when 'add' exists:
                """
                self.feedback.output(
                    self.feedback.INFO,
                    'Add some with --add'
                )
                """
                return 0
            else:
                first = False

            # Call sync on each file in the directory, use its full path
            for single_file in filenames:
                store_filename = os.path.join(dirpath, single_file)
                repo_filename = os.path.relpath(store_filename, self.store_dir)
                home_filename = os.path.join(self.home_dir, repo_filename)

                ChattyFile(store_filename, home_filename).sync()
                count += 1

        return count

    def add(self, filename):
        """
        Create a ChattyFile for the `filename` and call `add`

        :PRE: `filename` exists
        :returns: Result of the `add` call as an int
        """
        filename_rel_home = os.path.relpath(filename, self.home_dir)
        store_filename = os.path.join(self.store_dir, filename_rel_home)
        return ChattyFile(store_filename, filename).add()
