import argparse
import os

from .repository import Repository
from .feedback import Feedback


def main():
    """
    ChattyDotfiles entrypoint
    """
    args = get_args()

    cc = ChattyCommand()

    if args.add is None:
        return cc.sync()
    else:
        return cc.add(args.add)


def get_args():
    """
    Collect args from command line
    """
    epilog = ('By default Chatty will process each file it finds in '
              '~/dotfiles/store/ and ensure that there is a symlink to it '
              'from your home dir. You must still ensure that you manage your '
              'store folder (saving, sharing, etc) yourself.')
    parser = argparse.ArgumentParser(
        description='Chattydotfiles manages dotfiles.',
        epilog=epilog,
    )
    parser.add_argument('--add',
                        help='Add file to store. Will copy file to store and '
                             'then sync it.')
    return parser.parse_args()


class ChattyCommand(object):

    MISSING_FILE = 1

    def __init__(self):
        self.feedback = Feedback()
        self.repository = Repository(self.feedback)

        # Attempt to initialise the repository
        result = self.repository.start()
        if result != Repository.START_OK:
            # TODO needs some work around here to explain error and exit clean
            raise RuntimeError(result)

    def sync(self):
        """
        Sync repo with user's home folder
        """
        sync_results = self.repository.walk_and_talk()

        self.feedback.output(
            Feedback.INFO,
            '{} files processed.'.format(sync_results)
        )

        return 0

    def add(self, new_filename):
        """
        Add requested `new_filename` to the user's store and link it.
        """
        abs_new_filename = os.path.abspath(new_filename)

        if not os.path.isfile(abs_new_filename):
            return self.MISSING_FILE
        else:
            self.repository.add(abs_new_filename)
            return 0
