import filecmp
import os

from feedback import Feedback


class ChattyFile(object):
    """
    Single file in the dotfile manager, can perform basic adding and syncing
    functions

    However, a single file is really a relationship between two endpoints in
    the filesystem:
        * store_filename - The file in the store folder
        * home_filename - The file in or under the user's home folder
    """

    # Status return flags
    # All OK
    SYNC_OK = 0
    # Errors
    STORE_GONE = 1
    HOME_CREATE = 2
    HOME_MATCHES_STORE = 3
    HOME_DIFF_STORE = 4
    HOME_LINK_EXISTS = 5
    HOME_DIR_EXISTS = 6
    USER_SKIP = 7
    # Add codes
    ADD_OK = 10
    # Rnd errors
    ERROR = 99

    def __init__(self, store_filename, home_filename):
        """
        Initialise ChattyFile

        Assumes that passed ``store_filename`` is an existing store file and
        does no checking. Checking occurs when any actions are taken.
        """
        super(ChattyFile, self).__init__()
        self.store_filename = store_filename
        self.home_filename = home_filename
        self.feedback = Feedback()

    def sync(self):
        """
        Creates links to dotfiles

        There are multiple states that the relationship between the store file
        and the home file can be in. This leads to different prompts for the
        user and different possible results, each one returns a status flag:

        * store file = Missing: Something went bad with the store file between
          its discovery and sync. STORE_GONE
        * home file = Missing: There is a store file that hasn't been linked to
          a home dotfile, create as normal. HOME_CREATE
        * home file = Is directory. Do nothing. HOME_DIR_EXISTS
        * home file = Is not link: There is a store file that contains
          content... Check the diff and:
            * contents are the same as store - everything is OK, home file can
              be replaced with a link to the matching store file.
              HOME_MATCHES_STORE
            * contents are different - Skip and message user to solve the
              problem. HOME_DIFF_STORE
        * home file = Is link: to somewhere that's not the store file. Skip and
          message user. HOME_LINK_EXISTS
        * home file = Is link to store file: This is the good status. SYNC_OK

        :returns: Will push back a status flag
        """
        # INFO sync
        self.feedback.output(
            Feedback.INFO,
            'Syncing {}'.format(self.store_filename)
        )

        # TODO test for if there's a file in the home directory that user
        # doesn't have access to changing in the cases where changing is
        # required

        if not os.path.exists(self.store_filename):
            # Store file no longer exists and has gone walkies
            self.feedback.output(
                Feedback.ERROR,
                'Store file {} is gone.'.format(self.store_filename),
                space_after=True
            )
            return self.STORE_GONE

        if os.path.islink(self.home_filename):
            # Home file is a link
            resolved_location = os.path.realpath(self.home_filename)
            if resolved_location == self.store_filename:
                # Link points at store file - all good
                self.feedback.output(
                    Feedback.OK,
                    'Home link exists to store file.',
                    space_after=True
                )
                return self.SYNC_OK
            else:
                # Link points at file that's not the store file as expected
                # Found 'self.home_filename' exists in '~'.
                # However 'self.home_filename' points at
                # os.path.realpath(homefile)

                self.feedback.output(
                    Feedback.ERROR,
                    'Found "{}" but it links to "{}"'.format(
                        self.home_filename,
                        resolved_location
                    )
                )
                self.feedback.output(
                    Feedback.ERROR,
                    'This needs manual resolution.'
                )
                self.feedback.output(
                    Feedback.INFO,
                    'Recommended: Delete "{}" and sync again.'.format(
                        self.home_filename
                    ),
                    space_after=True
                )
                return self.HOME_LINK_EXISTS

        elif os.path.isfile(self.home_filename):
            # Home file is a file
            if filecmp.cmp(self.home_filename, self.store_filename,
                           shallow=False):
                # File contents are same, so migrate file to be link to store
                os.unlink(self.home_filename)
                os.symlink(self.store_filename, self.home_filename)
                self.feedback.output(
                    Feedback.INFO,
                    'Dotfile found in home with same content as store file.'
                )
                self.feedback.output(
                    Feedback.WARNING,
                    'Home dotfile replaced with link to store file.',
                    space_after=True
                )
                return self.HOME_MATCHES_STORE
            else:
                # Different so do nothing and just warn
                self.feedback.output(
                    Feedback.WARNING,
                    ('Dotfile found in home with different content to store '
                     'file.')
                )
                self.feedback.output(
                    Feedback.ERROR,
                    'Skipping syncing this file, need human resolution.',
                    space_after=True
                )
                return self.HOME_DIFF_STORE

        elif os.path.isdir(self.home_filename):
            # Home is a directory
            self.feedback.output(
                Feedback.WARNING,
                'Dotfile found in home is a directory.'
            )
            self.feedback.output(
                Feedback.ERROR,
                'Skipping syncing this file, need human resolution.',
                space_after=True
            )
            return self.HOME_DIR_EXISTS

        else:
            # Home file does not exist - so create it

            # Creating nested links requires that path exists and is a dir
            parent = os.path.dirname(self.home_filename)
            if os.path.exists(parent):
                if not os.path.isdir(parent):
                    self.feedback.output(
                        Feedback.WARNING,
                        'Non-directory found in expected directory location'
                    )
                    pick = self.feedback.input('Delete file or Skip (d/[s]) ?',
                                               ['s', 'd'])
                    # Assume user asks for Delete
                    if pick == 'd':
                        os.remove(parent)
                        self.feedback.output(
                            Feedback.WARNING,
                            'Deleted file at {}'.format(parent)
                        )
                        self.make_dir(parent)
                    else:
                        # Else just return with status 'skipped'
                        self.feedback.output(
                            Feedback.INFO,
                            'Skipped {}'.format(self.store_filename),
                            space_after=True
                        )
                        return self.USER_SKIP
            else:
                self.make_dir(parent)

            # Now create symlink
            os.symlink(self.store_filename, self.home_filename)

            self.feedback.output(
                Feedback.OK,
                'New link created at {} to dotfile'.format(self.home_filename),
                space_after=True
            )

            return self.HOME_CREATE

    def make_dir(self, dir_name):
        """
        Create a directory and message user
        """
        os.makedirs(dir_name)
        self.feedback.output(
            Feedback.WARNING,
            'New folder created at {}'.format(dir_name)
        )

    def add(self):
        """
        Add file at `self.home_filename` to location at `self.store_filename`,
        then create the link from the old `home_filename` location to the new
        `store_filename` and message User.

        NOTE will not work across drives. Needs `shutil.move` for that - later
        version.
        """
        os.rename(self.home_filename, self.store_filename)
        os.symlink(self.store_filename, self.home_filename)
        self.feedback.output(
            Feedback.INFO,
            'New file added to store - remember to update / commit your repo.'
        )
        self.feedback.output(
            Feedback.WARNING,
            'Home dotfile replaced with link to store file.',
            space_after=True
        )
        return self.ADD_OK
