# -*- coding: utf-8 -*-

"""This module is a wrapper to ``argparse`` module. It allow to generate a
command-line from a predefined directory (ie: a YAML, JSON, ... file)."""

import os
import re
import sys
import imp
import argparse
from six import iteritems
from collections import OrderedDict

#
# Constants.
#
# Get types.
BUILTINS = sys.modules['builtins'
                       if sys.version_info.major == 3
                       else '__builtin__']
TYPES = {builtin: getattr(BUILTINS, builtin) for builtin in vars(BUILTINS)}
# Get current module.
SELF = sys.modules[__name__]

# Keywords (argparse and clg).
KEYWORDS = {
    'parsers': {'argparse': ['prog', 'usage', 'description', 'epilog', 'help',
                             'add_help', 'formatter_class', 'argument_default',
                             'conflict_handler'],
                'clg': ['anchors', 'subparsers', 'options', 'args', 'groups',
                        'exclusive_groups', 'execute']},
    'subparsers': {'argparse': ['title', 'description', 'prog', 'help',
                                'metavar'],
                   'clg': ['required', 'parsers']},
    'groups': {'argparse': ['title', 'description'],
               'clg': ['options']},
    'exclusive_groups': {'argparse': ['required'],
                         'clg': ['options']},
    'options': {'argparse': ['action', 'nargs', 'const', 'default', 'choices',
                             'required', 'help', 'metavar', 'type'],
                'clg': ['short'],
                'post': ['match', 'need', 'conflict']},
    'args': {'argparse': ['action', 'nargs', 'const', 'default', 'choices',
                          'required', 'help', 'metavar', 'type'],
             'clg': ['short'],
             'post': ['match', 'need', 'conflict']},
    'execute': {'clg': ['module', 'file', 'function']}}


# Errors messages.
INVALID_SECTION = "this section is not of type '{type}'"
EMPTY_CONF = 'configuration is empty'
UNKNOWN_KEYWORD = "unknown keyword '{keyword}'"
ONE_KEYWORDS = "this section need (only) one of theses keywords: '{keywords}'"
MISSING_KEYWORD = "keyword '{keyword}' is missing"
UNKNOWN_ARG = "unknown {type} '{arg}'"
SHORT_ERR = 'this must be a single letter'
NEED_ERR = "{type} '{arg}' need {need_type} '{need_arg}'"
CONFLICT_ERR = "{type} '{arg}' conflict with {conflict_type} '{conflict_arg}'"
MATCH_ERR = "value '{val}' of {type} '{arg}' does not match pattern '{pattern}'"
FILE_ERR = "Unable to load file: {err}"
LOAD_ERR = "Unable to load module: {err}"

# Bash completion script.
BASH_SCRIPT = """declare -a choices
declare -a options
declare -a subcommands

parse_command () {{
    choices=(${{options[@]}} ${{subcommands[@]}})
    choices=`echo ${{choices[@]}}`
    for index in `seq $2 $COMP_CWORD`; do
        word=${{COMP_WORDS[$index]}}
        for subcommand in ${{subcommands[@]}}; do
            if [[ $subcommand = $word ]]; then
                index=$((index+1))
                "$1_$subcommand" $index
            fi
        done
        COMPREPLY=($(compgen -W "$choices" -- ${{COMP_WORDS[COMP_CWORD]}}))
    done
}}
{functions}

complete -F _{prog} {prog}
"""

# Zsh completion script.
ZSH_SCRIPT = """#compdef {prog}
local state ret=1
local -a options
typeset -A opt_args

parse_command () {{
    choices=($subcommands{ext} $options{ext})

    for index in {{$2..${{#words}}}}; do
        word=$words[$index]
        for subcommand in $subcommands; do
            if [[ $subcommand = $word ]]; then
                ((index=$index+1))
                "$1_$subcommand" $index
            fi
        done
        {command}
    done
}}
{functions}

_{prog}
return ret
"""
ZSH_SIMPLE = '_arguments "*: :($choices)" && ret=0'
ZSH_MENU = "_describe -t desc '$1' choices && ret=0"


#
# Exceptions.
#
class CLGError(Exception):
    """CLG exception."""
    def __init__(self, path, msg):
        Exception.__init__(self, msg)
        self.path = path
        self.msg = msg

    def __str__(self):
        return "/%s: %s" % ('/'.join(self.path), self.msg)


#
# Utils functions.
#
def _gen_parser(parser_conf, subparser=False):
    """Retrieve arguments pass to **argparse.ArgumentParser** from
    **parser_conf**. A subparser can take an extra 'help' keyword."""
    conf = {
        'prog':             parser_conf.get('prog', None),
        'usage':            None,
        'description':      parser_conf.get('description', None),
        'epilog':           parser_conf.get('epilog', None),
        'formatter_class':  getattr(argparse, parser_conf.get('formatter_class',
                                                              'HelpFormatter')),
        'argument_default': parser_conf.get('argument_default', None),
        'conflict_handler': parser_conf.get('conflict_handler', 'error'),
        'add_help':         parser_conf.get('add_help', True)}

    if subparser and 'help' in parser_conf:
        conf['help'] = parser_conf['help']
    return conf


def _get_args(parser_conf):
    """Get options and arguments from a parser configuration."""
    args = OrderedDict()
    for arg_type in ('options', 'args'):
        for arg, arg_conf in iteritems(parser_conf.get(arg_type, {})):
            args[arg] = (arg_type, OrderedDict(arg_conf))
    return args


def _set_builtin(value):
    """Replace configuration values which begin and end by ``__`` by the
    respective builtin function."""
    try:
        return TYPES[re.search('^__([A-Z]*)__$', value).group(1).lower()]
    except (AttributeError, TypeError):
        return (value.replace('__FILE__', sys.path[0])
                if type(value) is str
                else value)


#
# Formatting functions.
#
def _format_usage(prog, usage):
    """Format usage."""
    spaces = re.sub('.', ' ', 'usage: ')
    usage_elts = [prog]
    usage_elts.extend(['%s %s' % (spaces, elt)
                       for elt in usage.split('\n')[:-1]])
    return '\n'.join(usage_elts)


def _format_optname(value):
    """Format the name of an option in the configuration file to a more
    readable option in the command-line."""
    return value.replace('_', '-').replace(' ', '-')


def _format_optdisplay(value, conf):
    """Format the display of an option in error message (short and long option
    with dashe(s) separated by a slash."""
    return ('-%s/--%s' % (conf['short'], _format_optname(value))
            if 'short' in conf
            else '--%s' % _format_optname(value))


#
# Check functions.
#
def _check_empty(path, conf):
    """Check **conf** is not ``None`` or an empty iterable."""
    if conf is None or (hasattr(conf, '__iter__') and not len(conf)):
        raise CLGError(path, EMPTY_CONF)


def _check_type(path, conf, conf_type=dict):
    """Check the **conf** is of **conf_type** type and raise an error if not."""
    if not isinstance(conf, conf_type):
        type_str = str(conf_type).split()[1][1:-2]
        raise CLGError(path, INVALID_SECTION.format(type=type_str))


def _check_keywords(path, conf, section, one=None, need=None):
    """Check items of **conf** from **KEYWORDS[section]**. **one** indicate
    whether a check must be done on the number of elements or not."""
    valid_keywords = [keyword
                      for keywords in KEYWORDS[section].values()
                      for keyword in keywords]

    for keyword in conf:
        if keyword not in valid_keywords:
            raise CLGError(path, UNKNOWN_KEYWORD.format(keyword=keyword))
        _check_empty(path + [keyword], conf[keyword])

    if one and len([arg for arg in conf if arg in one]) != 1:
        keywords_str = "', '".join(one)
        raise CLGError(path, ONE_KEYWORDS.format(keywords=keywords_str))

    if need:
        for keyword in need:
            if keyword not in conf:
                raise CLGError(path, MISSING_KEYWORD.format(keyword=keyword))


def _check_section(path, conf, section, one=None, need=None):
    """Check section is not empty, is a dict and have not extra keywords."""
    _check_empty(path, conf)
    _check_type(path, conf, dict)
    _check_keywords(path, conf, section, one=one, need=need)


#
# Post processing functions.
#
def _has_value(value, conf):
    """The value of an argument not passed in the command is *None*, except:
        * if **nargs** is ``*`` or ``+``: in this case, the value is an empty
          list (this is set by this module),
        * if **action** is ``store_true`` or ``store_false``: in this case, the
          value is respectively ``False`` and ``True``.
    This function take theses cases in consideration and check if an argument
    really has a value.
    """
    if value is None or (isinstance(value, list) and not value):
        return False

    if 'action' in conf:
        action = conf['action']
        store_true = (action == 'store_true' and not value)
        store_false = (action == 'store_false' and value)
        if store_true or store_false:
            return False
    return True


def _print_error(parser, msg):
    """Print parser usage with an error message at the end and exit."""
    parser.print_usage()
    print("%s: error: %s" % (parser.prog, msg))
    sys.exit(1)


def _post_need(parser, parser_args, args_values, arg):
    """Post processing that check all for needing options."""
    arg_type, arg_conf = parser_args[arg]
    for cur_arg in arg_conf['need']:
        cur_arg_type, cur_arg_conf = parser_args[cur_arg]
        if not _has_value(args_values[cur_arg], cur_arg_conf):
            arg_str = (_format_optdisplay(arg, arg_conf)
                       if arg_type == 'options' else arg)
            need_str = (_format_optdisplay(cur_arg, cur_arg_conf)
                        if cur_arg_type == 'options' else cur_arg)
            _print_error(parser, NEED_ERR.format(type=arg_type[:-1],
                                                 arg=arg_str,
                                                 need_type=cur_arg_type[:-1],
                                                 need_arg=need_str))


def _post_conflict(parser, parser_args, args_values, arg):
    """Post processing that check for conflicting options."""
    arg_type, arg_conf = parser_args[arg]
    for cur_arg in arg_conf['conflict']:
        cur_arg_type, cur_arg_conf = parser_args[cur_arg]
        if _has_value(args_values[cur_arg], cur_arg_conf):
            arg_str = (_format_optdisplay(arg, arg_conf)
                       if arg_type == 'options' else arg)
            conflict_str = (_format_optdisplay(cur_arg, cur_arg_conf)
                            if cur_arg_type == 'options' else cur_arg)
            _print_error(parser,
                         CONFLICT_ERR.format(type=arg_type[:-1],
                                             arg=arg_str,
                                             conflict_type=cur_arg_type[:-1],
                                             conflict_arg=conflict_str))


def _post_match(parser, parser_args, args_values, arg):
    """Post processing that check the value."""
    arg_type, arg_conf = parser_args[arg]
    pattern = arg_conf['match']

    msg_elts = {'type': arg_type, 'arg': arg, 'pattern': pattern}
    if arg_conf.get('nargs', None) in ('*', '+'):
        for value in args_values[arg]:
            if not re.match(pattern, value):
                _print_error(parser, MATCH_ERR.format(val=value, **msg_elts))
    elif not re.match(pattern, args_values[arg]):
        _print_error(parser, MATCH_ERR.format(val=args_values[arg], **msg_elts))


def _exec_module(path, exec_conf, args_values):
    """Load and execute a function of a module according to **exec_conf**."""
    mdl_func = exec_conf.get('function', 'main')
    mdl_tree = exec_conf['module'].split('.')
    mdl = None

    for mdl_idx, mdl_name in enumerate(mdl_tree):
        try:
            imp_args = imp.find_module(mdl_name, mdl.__path__ if mdl else None)
            mdl = imp.load_module('.'.join(mdl_tree[:mdl_idx + 1]), *imp_args)
        except (ImportError, AttributeError) as err:
            raise CLGError(path, LOAD_ERR.format(err=err))
    getattr(mdl, mdl_func)(args_values)


def _exec_file(path, exec_conf, args_values):
    """Load and execute a function of a file according to **exec_conf**."""
    mdl_path = os.path.abspath(exec_conf['file'])
    mdl_name = os.path.splitext(os.path.basename(mdl_path))[0]
    mdl_func = exec_conf.get('function', 'main')

    try:
        getattr(imp.load_source(mdl_name, mdl_path), mdl_func)(args_values)
    except (IOError, ImportError, AttributeError) as err:
        raise CLGError(path, FILE_ERR.format(err=err))


#
# Classes.
#
class Namespace(argparse.Namespace):
    """Iterable namespace."""
    def __init__(self, args):
        argparse.Namespace.__init__(self)
        self.__dict__.update(args)

    def __getitem__(self, key):
        return self.__dict__[key]

    def __setitem__(self, key, value):
        if key not in self.__dict__:
            raise KeyError(key)
        self.__dict__[key] = value

    def __iter__(self):
        return ((key, value) for key, value in iteritems(self.__dict__))


class CommandLine(object):
    """CommandLine object that parse a preformatted dictionnary and generate
    ``argparse`` parser."""
    def __init__(self, config, keyword='command'):
        """Initialize the command from **config** which is a dictionnary
        (preferably an OrderedDict). **keyword** is the name use for knowing the
        path of subcommands (ie: 'command0', 'command1', ... in the namespace of
        arguments)."""
        self.config = config
        self.keyword = keyword
        self._parsers = {}
        self.parser = None
        self._add_parser([])


    def _get_config(self, path, ignore=True):
        """Retrieve an element configuration (based on **path**) in the
        configuration."""
        config = self.config
        for idx, elt in enumerate(path):
            config = (config['parsers'][elt]
                      if (not ignore
                          and path[idx-1] == 'subparsers'
                          and 'parsers' in config)
                      else config[elt])
        return config


    def _add_parser(self, path, parser=None):
        """Add a subparser to a parser. If **parser** is ``None``, the subparser
        is in fact the main parser."""
        # Get configuration.
        parser_conf = self._get_config(path)

        # Check parser configuration.
        _check_section(path, parser_conf, 'parsers')
        if 'execute' in parser_conf:
            _check_section(path + ['execute'],
                           parser_conf['execute'],
                           'execute',
                           one=('module', 'file'))

        # Initialize parent parser.
        if parser is None:
            self.parser = argparse.ArgumentParser(**_gen_parser(parser_conf))
            parser = self.parser

        # Index parser (based on path) as it may be necessary to access it
        # later (manage case where subparsers does not have configuration).
        parser_path = [elt
                       for idx, elt in enumerate(path)
                       if not (path[idx-1] == 'subparsers'
                               and elt == 'parsers')]
        self._parsers['/'.join(parser_path)] = parser

        # Add custom usage.
        if 'usage' in parser_conf:
            parser.usage = _format_usage(parser.prog, parser_conf['usage'])

        # Add subparsers.
        if 'subparsers' in parser_conf:
            self._add_subparsers(parser,
                                 path + ['subparsers'],
                                 parser_conf['subparsers'])

        # Add groups and exclusive groups.
        parser_args = _get_args(parser_conf)
        for grp_type in ('groups', 'exclusive_groups'):
            if grp_type in parser_conf:
                self._add_groups(parser,
                                 path + [grp_type],
                                 parser_conf[grp_type],
                                 grp_type,
                                 parser_args)

        # Add options and arguments.
        for arg in parser_args:
            self._add_arg(parser, path, arg, parser_args)


    def _add_subparsers(self, parser, path, subparsers_conf):
        """Add subparsers. Subparsers can have a global configuration or
        directly parsers configuration. This is the keyword **parsers** that
        indicate it."""
        # Get add_subparsers parameters.
        required = True
        subparsers_kwargs = {'dest': '%s%d' % (self.keyword, len(path) / 2)}
        if 'parsers' in subparsers_conf:
            _check_section(path, subparsers_conf, 'subparsers')

            keywords = KEYWORDS['subparsers']['argparse']
            subparsers_kwargs.update({keyword: subparsers_conf[keyword]
                                      for keyword in keywords
                                      if keyword in subparsers_conf})
            required = subparsers_conf.get('required', True)

            subparsers_conf = subparsers_conf['parsers']
            path.append('parsers')

        # Initialize subparsers.
        subparsers = parser.add_subparsers(**subparsers_kwargs)
        subparsers.required = required

        # Add subparsers.
        for parser_name, parser_conf in iteritems(subparsers_conf):
            _check_section(path + [parser_name], parser_conf, 'parsers')
            subparser = subparsers.add_parser(parser_name,
                                              **_gen_parser(parser_conf,
                                                            subparser=True))
            self._add_parser(path + [parser_name], subparser)


    def _add_groups(self, parser, path, groups_conf, grp_type, parser_args):
        """Add groups and mutually exclusive groups."""
        _check_empty(path, groups_conf)
        _check_type(path, groups_conf, list)
        for grp_number, grp_conf in enumerate(groups_conf):
            grp_path = path + ["#%d" % grp_number]
            _check_section(grp_path, grp_conf, grp_type, need=['options'])

            if grp_type == 'groups':
                grp_kwargs = {param: value
                              for param, value in iteritems(grp_conf)
                              if param in KEYWORDS['groups']['argparse']}
                group = parser.add_argument_group(**grp_kwargs)
            elif grp_type == 'exclusive_groups':
                grp_kwargs = {'required': grp_conf.get('required', False)}
                group = parser.add_mutually_exclusive_group(**grp_kwargs)

            for opt in grp_conf['options']:
                if opt not in parser_args or (grp_type == 'exclusive_groups'
                                              and parser_args[opt][0] != 'options'):
                    raise CLGError(grp_path, UNKNOWN_ARG.format(type='option',
                                                                arg=opt))
                self._add_arg(group, path[:-1], opt, parser_args)
                del parser_args[opt]


    def _add_arg(self, parser, path, arg, parser_args):
        """Add an option/argument to **parser**."""
        arg_type, arg_conf = parser_args[arg]
        arg_path = path + [arg_type, arg]

        # Check configuration.
        _check_section(path, arg_conf, arg_type)
        for keyword in ('need', 'conflict'):
            if keyword not in arg_conf:
                continue
            _check_type(path + [keyword], arg_conf[keyword], list)
            for cur_arg in arg_conf[keyword]:
                if cur_arg not in parser_args:
                    cur_arg_type = parser_args[cur_arg][0][:-1]
                    raise CLGError(arg_path + [keyword],
                                   UNKNOWN_ARG.format(type=cur_arg_type,
                                                      arg=cur_arg))

        # Get argument parameters.
        arg_args, arg_kwargs = [], {}
        if arg_type == 'options':
            if 'short' in arg_conf:
                if len(arg_conf['short']) != 1:
                    raise CLGError(arg_path + ['short'], SHORT_ERR)
                arg_args.append('-%s' % arg_conf['short'])
                del arg_conf['short']
            arg_args.append('--%s' % _format_optname(arg))
            arg_kwargs['dest'] = arg
        elif arg_type == 'args':
            arg_args.append(arg)

        default = str(arg_conf.get('default', '?'))
        match = str(arg_conf.get('match', '?'))
        for param, value in sorted(iteritems(arg_conf)):
            if param in KEYWORDS[arg_type]['post']:
                continue

            arg_kwargs[param] = {
                'type': lambda: TYPES[value],
                'help': lambda: value.replace('__DEFAULT__', default)
                                     .replace('__MATCH__', match)
                }.get(param, lambda: _set_builtin(value))()

        # Add argument to parser.
        parser.add_argument(*arg_args, **arg_kwargs)


    def parse(self, args=None):
        """Parse command-line."""
        # Parse command-line.
        args_values = Namespace(self.parser.parse_args(args).__dict__)

        # Get command configuration.
        path = [elt
                for arg, value in sorted(args_values)
                for elt in ('subparsers', value)
                if re.match('^%s[0-9]*$' % self.keyword, arg)]
        parser_conf = self._get_config(path, ignore=False)
        parser = self._parsers['/'.join(path)]

        # Post processing.
        parser_args = _get_args(parser_conf)
        for arg, (arg_type, arg_conf) in iteritems(parser_args):
            if not _has_value(args_values[arg], arg_conf):
                if arg_conf.get('nargs', None) in ('*', '+'):
                    args_values[arg] = []
                continue

            for keyword in KEYWORDS[arg_type]['post']:
                if keyword in arg_conf:
                    getattr(SELF, '_post_%s' % keyword)(parser,
                                                        parser_args,
                                                        args_values,
                                                        arg)

        # Execute.
        if 'execute' in parser_conf:
            for keyword in ('module', 'file'):
                if keyword in parser_conf['execute']:
                    exec_params = (path + ['execute'],
                                   parser_conf['execute'],
                                   args_values)
                    getattr(SELF, '_exec_%s' % keyword)(*exec_params)

        return args_values


    def gen_bash_completion(self, prog, **kwargs):
        """Generate bash completion. **prog** is the name of the program."""
        functions = self._gen_completion([], 'bash', prog, **kwargs)


    def gen_zsh_completion(self, prog, **kwargs):
        pass


    def _gen_completion(self, path, shell, name, **kwargs):
        parser_conf = self._get_config(path)

        functions = ['',
                     '_%s () {' % parser_name]

        # Get subparsers config.
        subparsers = parser_conf.get('subparsers_conf', {})
        path += ['subparsers']
        if 'parsers' in subparsers:
            subparsers = subparsers['parsers']
            path += ['parsers']
        subparsers_desc = ['"%s:%s"' % (parser, parser_conf.get('description',
                                                                'No description'))
                           for parser, parser_conf in iteritems(subparsers)]
        subparsers = subparsers.keys()




#    def gen_bash_completion(self, prog, **kwargs):
#        """Generate bash completion."""
#        self.functions = []
#        self._gen_completion('bash', [], prog, **kwargs)
#        return BASH_SCRIPT.format(prog=prog,
#                                  functions='\n'.join(self.functions))
#
#
#    def gen_zsh_completion(self, prog, **kwargs):
#        """Generate zsh completion."""
#        self.functions = []
#        self._gen_completion('zsh', [], prog, **kwargs)
#        simple = kwargs.get('simple', False)
#        return ZSH_SCRIPT.format(prog=prog,
#                                 functions='\n'.join(self.functions),
#                                 command=ZSH_SIMPLE if simple else ZSH_MENU,
#                                 ext='' if simple else '_desc')
#
#
#
#    def _gen_completion(self, shell, path, parser_name, **kwargs):
#        """Generate completion of current command."""
#        parser_conf = self._get_config(path)
#        self.functions.append('')
#        self.functions.append('_%s () {' % parser_name)
#
#        # Get subparsers config.
#        subparsers_conf = parser_conf.get('subparsers_conf', {})
#        path += ['subparsers']
#        if 'parsers' in subparsers_conf:
#            subparsers_conf = subparsers_conf['parsers']
#            path += ['parsers']
#        subparsers_desc = ['"%s:%s"' % (subparser,
#                                        subparser_conf.get('description',
#                                                           'No description'))
#                           for subparser, subparser_conf in iteritems(subparsers_conf)]
#        subparsers = subparsers_conf.keys()
#
#        # Get options and arguments.
#        opts = ['--%s' % _format_optname(opt)
#                for opt in parser_conf.get('options', {})]
#        parser_opts = iteritems(parser_conf.get('options', {}))
#        opts_desc = ['"--%s:%s"' % (_format_optname(opt),
#                                    opt_conf.get('help', 'No description'))
#                     for opt, opt_conf in parser_opts]
#
#        if parser_conf.get('add_help', True):
#            opts.append('--help')
#            opts_desc.append('"--help:Show this help message and exit."')
#        if kwargs.get('ignore_opts', False) and subparsers_conf:
#            opts = []
#            opts_desc = []
#        args = (_format_optname(arg) for arg in parser_conf.get('args', {}))
#
#        # Generate parser function.
#        self.functions.extend(['    options=(%s)' % ' '.join(opts),
#                               '    args=(%s)' % ' '.join(args),
#                               '    subcommands=(%s)' % ' '.join(subparsers)])
#        if shell == 'zsh' and not kwargs.get('simple', False):
#            self.functions.extend([
#                '    options_desc=(%s)' % '\n'.join(opts_desc),
#                '    subcommands_desc=(%s)' % '\n'.join(subparsers_desc)])
#
#        # Add parse_command in the current function.
#        parser_args = {'bash': 1, 'zsh': 2}[shell] if len(path) == 1 else '$1'
#        self.functions.append('    parse_command _%s %s' % (parser_name,
#                                                            parser_args))
#        self.functions.append('}')
#
#        for subparser in subparsers_conf:
#            self._gen_completion(shell,
#                                 path + [subparser],
#                                 '%s_%s' % (parser_name, subparser),
#                                 **kwargs)
