# -*- coding: UTF-8 -*-
# vim: fileencoding=UTF-8 filetype=python ff=unix et ts=4 sw=4 sts=4 tw=120
# author: Christer Sjöholm -- hcs AT furuvik DOT net

from datetime import datetime
import storm.locals as s
import hcs_utils.path
import hcs_utils.storage
import logging
import os

log = logging.getLogger()

class CLog(object):

    def __init__(self, dir_):
        self._dir = dir_ = hcs_utils.path.expand(dir_)
        db_dir = os.path.join(dir_, 'db_v1')
        if not os.path.isdir(db_dir):
            os.makedirs(db_dir)
        db_path = os.path.join(db_dir, 'clog.sqlite')

        database = s.create_database('sqlite:' + db_path)
        self.store = s.Store(database)
        res = self.store.execute("select name from sqlite_master where type='table';")
        if not res.get_one():
            self._create_tables()

    def _create_tables(self):
        for statement in TodoElement.ddl:
            self.store.execute(statement, noresult=True)
        for statement in LogEntry.ddl:
            self.store.execute(statement, noresult=True)
        self.store.commit()

    ############################
    # Log methods
    def get_logentry(self, id_):
        return self.store.find(LogEntry, LogEntry.id == id_).one()

    def get_logentry_by_time(self, time):
        return self.store.find(LogEntry, LogEntry.time == time).one()

    def get_logentries(self, from_, to_):
        if from_ and to_:
            res = self.store.find(LogEntry, LogEntry.time >= from_, LogEntry.time <= to_)
        elif from_:
            res = self.store.find(LogEntry, LogEntry.time >= from_)
        elif to_:
            res = self.store.find(LogEntry, LogEntry.time <= to_)
        else: # get all
            res = self.store.find(LogEntry)
        return res.order_by(LogEntry.time)

    def create_logentry(self, time=None):
        ent = LogEntry()
        ent.time = time or datetime.now()
        self.store.add(ent)
        return ent

    def update_logentry(self, ent, fields):
        for field in 'time note worked_time todo'.split(' '):
            if field in fields:
                value = fields[field]
                if field == 'time':
                    ent.date = datetime.strptime(value, '%Y-%m-%d %H:%M:%S.%f')
                elif field == 'worked_time':
                    ent.worked_time = int(value)
                elif field == 'note':
                    ent.note = value
                elif field == 'todo':
                    if not value:
                        ent.todo = None
                    elif not ent.todo or value != ent.todo.name:
                        try:
                            ent.todo = self.get_todo(value)
                        except KeyError:
                            ent.todo = self.create_todo(value)
                else:
                    raise Exception('unknown field: ' + field)

    ############################
    # Todo methods
    @property
    def todos(self):
        return self.store.find(TodoElement)

    @property
    def todo_names(self):
        for (name,) in self.store.execute('select name from todos'):
            yield name

    def get_todo(self, name):
        res = self.store.find(TodoElement, TodoElement.name == name).one()
        if not res:
            raise KeyError(u'No todo exists named "{0}"'.format(name))
        return res

    def create_todo(self, name):
        elem = TodoElement(name)
        if '.' in name: # if not a root element
            parent_name = name.rsplit('.', 1)[0]
            try:
                parent = self.get_todo(parent_name)
            except KeyError:
                parent = self.create_todo(parent_name)
            elem.parent = parent
        self.store.add(elem)
        return elem

    def commit(self):
        self.store.commit()

    def rollback(self):
        self.store.rollback()

class LogEntry(object):
    __storm_table__ = 'log'
    ddl = [
        '''CREATE TABLE log (
            id          INTEGER PRIMARY KEY,
            time        TEXT UNIQUE NOT NULL,
            note        TEXT NOT NULL,
            todo_id     INTEGER,
            tags        NULL,
            attachments NULL,
            worked_time INTEGER,
            FOREIGN KEY (todo_id) REFERENCES todo(id)
        );''',
        'CREATE UNIQUE INDEX log_i1 on log(time);'
        ]
    id = s.Int(primary=True)
    time = s.DateTime() # unique
    note = s.Unicode(default=u'')
    todo_id = s.Int()
    #tags = s.List()
    #attachments = s.List()
    #todo = s.Reference(LogEntry.todo_id, TodoElement.id) this is added below, because LogEntry don't exist yet
    worked_time = s.Int(default=0) # seconds

    def to_dict(self):
        res = {}
        for field in 'time note worked_time'.split(' '):
            res[field] = unicode(getattr(self, field))
        if self.todo:
            res['todo'] = unicode(self.todo.name)
        else:
            res['todo'] = u''
        return res

class TodoElement(object):
    __storm_table__ = 'todos'
    ddl = [
        '''CREATE TABLE todos (
            id          INTEGER PRIMARY KEY,
            name        TEXT UNIQUE NOT NULL,
            note        TEXT NOT NULL,
            parent_id   INTEGER,
            time_estimate INTEGER,
            FOREIGN KEY (parent_id) REFERENCES todos(id)
        );''',
        'CREATE UNIQUE INDEX todos_i1 on todos(name);'
        ]
    id = s.Int(primary=True)
    name = s.Unicode() #unique
    note = s.Unicode(default=u'')
    parent_id = s.Int() # None for root element
    time_estimate = s.Int() # Seconds
    parent = s.Reference(parent_id, id)
    children = s.ReferenceSet(id, parent_id)
    log_entries = s.ReferenceSet(id, LogEntry.todo_id)

    def __init__(self, name):
        self.name = name

    def verify(self):
        '''Check that the element is consistent and legal...'''
        #check name
        if self.name != self._build_name():
            raise Exception('Illegal name, is="{0}" should be="{1}"'.format(self.name, self._build_name()))

    def _build_name(self):
        '''look at parents an generate name including parents ex: a.b.c.name,
        this is used to check/update the name field'''
        current = self
        parts = []
        parts.insert(0, current.name.split('.')[-1])
        while current.parent:
            current = current.parent
            parts.insert(0, current.name.split('.')[-1])
        return '.'.join(parts)

LogEntry.todo = s.Reference(LogEntry.todo_id, TodoElement.id)
