# -*- coding: UTF-8 -*-
# vim: fileencoding=UTF-8 filetype=python ff=unix et ts=4 sw=4 sts=4 tw=120
# author: Christer Sjöholm -- hcs AT furuvik DOT net

from clog.dao import CLog
from clog.prompt import Prompt
from datetime import datetime, timedelta
import argparse
import clog.edit
import hcs_utils.path
import hcs_utils.storage
import locale
import logging
import sys
import re

log = logging.getLogger(__name__)

DATE_FORMAT = '%Y-%m-%d %H:%M:%S'
DATE_FORMAT_EXAMPLE = '2006-12-21 00:21:23'

class Application(object):
    CLOG_DIR = "~/.clog/"
    DEFAULT_CONFIG_FILE = 'TODO'
    ENCODING = locale.getpreferredencoding()
    def __init__(self):
        self.args = None
        self.config = None
        self.clog = None

    ##########################################################################

    def main(self):
        parser = argparse.ArgumentParser(description='TODO')
        parser.add_argument('-c', '--config', help='Specify alternative configuration file.', metavar="FILE")
        parser.add_argument('-v', '--verbose', dest="logging_level", action='store_const',
                const=logging.INFO, help='Be verbose about what is going on (stderr).')
        parser.add_argument('-d', '--debug', dest="logging_level", action='store_const',
                const=logging.DEBUG, help='Output debug info (stderr).')
        parser.set_defaults(config=self.DEFAULT_CONFIG_FILE, logging_level=logging.ERROR)

        subparsers = parser.add_subparsers(title='subcommands', description='Default is log without arguments.')

        #### Log subcommands

        sparser = subparsers.add_parser('show',
                description= 'show log')
        sparser.add_argument('interval', nargs='?',
                choices=['all', 'day', 'week', 'month'], default='all',
                metavar='INTERVAL', help='Name to use.')
        sparser.set_defaults(func=self.cmd_show)

        sparser = subparsers.add_parser('log',
                description= 'Create log entry')
        sparser.add_argument('-d', '--time', help="entry date/time on format %s" % DATE_FORMAT_EXAMPLE)
        sparser.add_argument('-t', '--todo', help='Connect entry to TODO')
        sparser.add_argument('-w', '--worked', help='Worked time')
        sparser.add_argument('note', nargs='*', help='use NOTE.')
        sparser.set_defaults(func=self.cmd_log)

        sparser = subparsers.add_parser('edit',
                description= 'Edit log entry')
        sparser.add_argument('id', type=int, help='ID of log entry.')
        sparser.set_defaults(func=self.cmd_edit)

        #### ToDo subcommands

        sparser = subparsers.add_parser('todo',
                description= 'Display todo list')
        sparser.set_defaults(func=self.cmd_todo)

        sparser = subparsers.add_parser('tda',
                description= 'Add a new item, optionally grafting it as a child of the given item.')
        sparser.add_argument('-n', '--note', help='use NOTE.')
        sparser.add_argument('-g', '--graft', help='Specify TODO this element is a child of.')
        sparser.add_argument('name', nargs='*', help='merge name parts into NAME.')
        sparser.set_defaults(func=self.cmd_tda)

        sparser = subparsers.add_parser('tde',
                description= 'Edit todo item')
        sparser.set_defaults(func=self.cmd_tde)

        sparser = subparsers.add_parser('tdr',
                description= 'Remove todo item')
        sparser.set_defaults(func=self.cmd_tdr)

        sparser = subparsers.add_parser('tdd',
                description= 'Mark todo as done')
        sparser.set_defaults(func=self.cmd_tdd)


        #### Parse

        args = [arg.decode(self.ENCODING) for arg in sys.argv[1:]]
        if not args:
            args.append('log') # Use log command as default
        self.args = args = parser.parse_args(args)

        logging.basicConfig(level=args.logging_level)

        self.config = config = hcs_utils.storage.Storage() #TODO read config
        self.clog = CLog(self.CLOG_DIR)

        args.func()

    ##########################################################################
    # Logging subcommands, called from self.main()

    def cmd_log(self):
        '''
        If no arguments prompt for
         - note
         - todo
         - worked time  ex. 1h 5m

        -e --edit Use vim instead

        '''
        args = self.args
        entry = self.clog.create_logentry()
        if args.time:
            entry.time = datetime.strptime(args.time, DATE_FORMAT) #TODO be more flexible about what can be entered
        else:
            inp = self.prompt('When: ', initial_text=entry.time.strftime(DATE_FORMAT))
            entry.time = datetime.strptime(inp, DATE_FORMAT) #TODO be more flexible about what can be entered

        if args.note:
            entry.note = ' '.join(args.note)
        else:
            entry.note = self.note_prompt('Note: ') #TODO add note_history
        if args.worked:
            entry.worked_time = parse_time_amount(args.worked)
        else:
            inp = self.prompt('Worked time: ')
            if inp:
                entry.worked_time = parse_time_amount(inp)
        if args.todo:
            todo_name = args.todo
        else:
            todo_name = self.todo_prompt('Connect to todo (empty for none): ', allow_new=True, allow_empty=True)
        if todo_name:
            try:
                entry.todo = self.clog.get_todo(todo_name)
            except KeyError:
                entry.todo = self.create_todo_wizard(name=todo_name)

#        else:
#            if not edit_entry(entry):
#                print 'entry unchanged, aborting.'
#                self.clog.rollback()
#                return
        self.clog.commit()
#        print 'Wrote entry'

    def cmd_edit(self):
        args = self.args
        ent = self.clog.get_logentry(args.id)
        was_updated = clog.edit.edit_entry(self.clog, ent)
        if was_updated:
            self.clog.commit()
            print 'Entry was updated'
        else:
            print 'No changes'

    def cmd_show(self): #TODO
        args = self.args
        entries = None
        if args.interval == 'all':
            fr_ = None
            to_ = None
        elif args.interval == 'day':
            fr_ = datetime.today()
            to_ = fr_ + timedelta(days=1)
        elif args.interval == 'month':
            fr_ = datetime.today()
            fr_.day = 1 # start of month
            to_ = fr_ + timedelta(months=1)
        else:
            raise AttributeError('Unknown interval: ' + args.interval)

        entries = self.clog.get_logentries(fr_, to_)
        for ent in entries:
            time = ent.time.strftime('%Y-%m-%d %a %H:%M')
            worked = format_time_amount(ent.worked_time)
            todo = ent.todo and ent.todo.name or ''
            note_lines = ent.note.splitlines()
            if len(note_lines) > 1:
                note = note_lines[0] + ' ...'
            else:
                note = note_lines[0]
            print '{0} {1} {2} {3} -- {4}'.format(ent.id, time, worked, todo.encode(self.ENCODING), note.encode(self.ENCODING))

#            print (u'''---- %s -- %s\n%s%s''' % (
#                e.date.strftime('%Y-%m-%d %a %H:%M:%S'),
#                ','.join(e.tags),
#                '',
#                e.body.rstrip())).encode('utf-8')

    ##########################################################################
    # TODO  subcommands, called from self.main()

    def cmd_todo(self):
        args = self.args
        for todo in self.clog.todos:
            print u'{} {}'.format(
                    todo.name,
                    todo.note and u'"{0}"'.format(todo.note) or '').encode(self.ENCODING)

    def cmd_tda(self):
        args = self.args
        name = ' '.join(args.name)
        todo = self.create_todo_wizard(name, args.note)
        self.clog.commit()

    def cmd_tde(self):
        args = self.args

    def cmd_tdr(self):
        args = self.args

    def cmd_tdd(self):
        args = self.args


    ##########################################################################
    # Helpers

    def prompt(self, prompt, initial_text=''):
        pro = Prompt()
        pro.encoding = self.ENCODING
        pro.choices = []
        pro.history = []
        pro.initial_text = initial_text
        pro.allow_empty = True
        pro.exact_choice = False
        pro.require_new = True
        pro.allow_multiline = False
        return pro.prompt(prompt)

    def note_prompt(self, prompt):
        pro = Prompt()
        pro.encoding = self.ENCODING
        pro.choices = []
        pro.history = []
        pro.initial_text = ''
        pro.allow_empty = True
        pro.exact_choice = False
        pro.require_new = False
        pro.allow_multiline = True
        return pro.prompt(prompt)

    def todo_prompt(self, prompt, allow_new=False, allow_empty=False, require_new=False):
        pro = Prompt()
        pro.encoding = self.ENCODING
        pro.choices = [name for name in self.clog.todo_names]
        pro.history = []
        pro.initial_text = ''
        pro.allow_empty = allow_empty
        pro.exact_choice = not (allow_new or require_new)
        pro.require_new = require_new
        pro.allow_multiline = False
        return pro.prompt(prompt)

    def new_todo_prompt(self, prompt, require_new=True):
        pro = Prompt()
        pro.encoding = self.ENCODING
        pro.choices = [name + '.' for name in self.clog.todo_names]
        pro.history = []
        pro.initial_text = ''
        pro.allow_empty = False
        pro.exact_choice = False
        pro.require_new = require_new
        pro.allow_multiline = False
        return pro.prompt(prompt)

    def create_todo_wizard(self, name=None, note=None):
        if not name:
            name = self.todo_prompt('Name for the todo: ', require_new=True)
        entry = self.clog.create_todo(name)
        #TODO prompt for time estimate
        if note:
            entry.note = note
        else:
            entry.note = self.prompt('Todo note: ')
        return entry

def format_time_amount(secs):
    hours = secs // (60 * 60)
    minutes = (secs - hours * 60 * 60) // 60
    return '{0}h {1}m'.format(hours, minutes)


def parse_time_amount(time_str):
    '''
    examples:
     '3h 5m'
     '1.5h'
     '3d 2:15' => 3d 2h 15m
     '2:15:02' => 2h 15m 2s

    returns amount of time in seconds
    '''
    float_expr = r'(\d+(\.\d+)?)'

    days = 0
    hours = 0
    minutes = 0
    seconds = 0

    parts = time_str.split()
    for part in parts:
        part = part.strip()
        if not part:
            continue
        match = re.match(float_expr + 'd$', part)
        if match:
            days += float(match.group(1))
            continue
        match = re.match(float_expr + 'h$', part)
        if match:
            hours += float(match.group(1))
            continue
        match = re.match(float_expr + 'm$', part)
        if match:
            minutes += float(match.group(1))
            continue
        match = re.match(float_expr + 's$', part)
        if match:
            seconds += float(match.group(1))
            continue
        match = re.match(r'(\d+):(\d+)$', part)
        if match:
            hours += int(match.group(1))
            minutes += int(match.group(2))
            continue
        match = re.match(r'(\d+):(\d+):(\d+)$', part)
        if match:
            hours += int(match.group(1))
            minutes += int(match.group(2))
            seconds += int(match.group(3))
            continue
        raise AttributeError('Cant parse time unit: ' + part)

    return ((days*24+hours)*60+minutes)*60+seconds


def main():
    app = Application()
    app.main()

#    def prompt(self, prompt, choices=[], history=[], initial_text='', allow_empty=True, exact_choice=False, require_new=False):
#        initial_text = initial_text.strip()
#
#        @rl.generator
#        @rl.print_exc
#        def complete_string(text):
#            rl.completion.suppress_append = True # don't add a space when matching
#            return [x.encode(self.ENCODING) for x in choices if x.startswith(text)]
#        rl.completer.completer = complete_string
#        rl.history.clear()
#        for line in history:
#            rl.history.add(line)
#        rl.completer.parse_and_bind('TAB: complete')
#        if initial_text:
#            rl.readline.set_startup_hook(lambda: rl.readline.insert_text(initial_text))
#        try:
#            while True:
#                if initial_text.count('\n'):
#
#                res = unicode(raw_input(prompt), self.ENCODING).strip()
#                if res == 'e' or res[-2:] == ' e':
#                    res = edit_in_editor(res)
#                if exact_choice and (res in choices):
#                    break
#                if require_new and res in choices:
#                    print '"{0}" already exists.'
#                    rl.history.add(res)
#                    continue
#                if res or allow_empty:
#                    break
#        finally:
#            rl.readline.set_startup_hook(None)
#            rl.completer.completer = None
#            rl.history.clear()
#        return res

