# -*- coding: UTF-8 -*-
# vim: fileencoding=UTF-8 filetype=python ff=unix et ts=4 sw=4 sts=4 tw=120
# author: Christer Sjöholm -- hcs AT furuvik DOT net

import codecs
import locale
import os
import sys
import rl
import tempfile
import subprocess

class Prompt(object):
    def __init__(self):
        self.encoding = locale.getpreferredencoding()

        self.choices = []
        self.history = []
        self.initial_text = ''
        self.allow_empty = True
        self.exact_choice = True
        self.require_new = True
        self.allow_multiline = False

        self._error_msg = '' # error message to be shown before the next prompt

    def _prompt_once_simple(self, prompt, initial_text=None):
        if not initial_text:
            initial_text = self.initial_text
        try:
            @rl.generator
            @rl.print_exc
            def complete_string(text):
                rl.completion.suppress_append = True # don't add a space when matching
                return [x.encode(self.encoding) for x in self.choices if x.startswith(text)]
            rl.completer.completer = complete_string
            rl.history.clear()
            for line in self.history:
                rl.history.append(line)
            rl.completer.parse_and_bind('TAB: complete')
            if self.initial_text:
                rl.readline.set_startup_hook(lambda: rl.readline.insert_text(self.initial_text))
            if self._error_msg:
                print self._error_msg
            return unicode(raw_input(prompt), self.encoding).strip()
        finally:
            rl.readline.set_startup_hook(None)
            rl.completer.completer = None
            rl.history.clear()

    def _prompt_using_editor(self, prompt, initial_text=None):
        if not initial_text:
            initial_text = self.initial_text
        (fd, path) = tempfile.mkstemp()
        template = initial_text
        with os.fdopen(fd,'w') as f:
            f.write(template.encode(self.encoding))
        try:
            retcode = subprocess.call("vim %s +5" % path, shell=True)
            if retcode < 0:
                print >> sys.stderr, "Child was terminated by signal", -retcode
            else:
                print >> sys.stderr, "Child returned", retcode
        except OSError, e:
            print >> sys.stderr, "Execution failed:", e
        with codecs.open(path, encoding=self.encoding) as inp:
            res = inp.read() #TODO strip template from result
        #TODO rm tempfile
        return res

    def prompt(self, prompt):
        initial_text = self.initial_text
        self._error_msg = ''
        while True:
            if initial_text.count('\n'):
                res = self._prompt_using_editor(prompt, initial_text=initial_text)
            else:
                res = self._prompt_once_simple(prompt, initial_text=initial_text)
                if res == 'e':
                    res = self._prompt_using_editor(prompt, initial_text=initial_text)
                if res[-2:] == ' e':
                    res = self._prompt_using_editor(prompt, initial_text=res[:-2])
            if res.count('\n') and not self.allow_multiline:
                self._error_msg = 'Only single row responses are allowed.'.format(res)
                initial_text = res
                continue
            if self.exact_choice and (res in self.choices):
                break
            if self.require_new and res in self.choices:
                self._error_msg = '"{0}" already exists.'.format(res)
                initial_text = res
                continue
            if res or self.allow_empty:
                break
        return res

