import sys
from collections import OrderedDict
from box.functools import cachedproperty
from ..formatter import Formatter
from ..mixin import mixin as decorate_mixin, convert as convert_mixin
from ..option import Option
from ..parser import Parser, ParseError
from .converter import convert as convert_command
from .decorator import command as decorate_command
from .metaclass import Metaclass


class Command(metaclass=Metaclass):

    # Public

    meta_formatter = Formatter

    def __init__(self, argv=None, *, stack=None):
        if argv is None:
            argv = sys.argv
        if stack is None:
            stack = []
        stack.append(self)
        self.__argv = argv
        self.__stack = stack
        self.__execute()

    def __getattr__(self, name):
        if not name.startswith('_'):
            for call in reversed(self.__stack[:-1]):
                try:
                    return getattr(call, name)
                except AttributeError:
                    pass
        raise AttributeError(name)

    def __call__(self):
        print(self.meta_format('help'))

    def meta_format(self, name, *args, **kwargs):
        call = self.__stack[-1]
        if call is not self:
            return call.meta_format(name, *args, **kwargs)
        function = getattr(self.__formatter, 'format_' + name)
        result = function(*args, **kwargs)
        return result

    # Private

    def __execute(self):
        if self.__subcommand is None:
            try:
                arguments, options = self.__parse()
            except ParseError as exception:
                self.__mixin(exception)
                self.__error(exception)
            try:
                self.__update(options)
                self.__mixin()
                self.__call__(*arguments)
            except Exception as exception:
                self.__mixin(exception)
        else:
            self.__delegate()

    @cachedproperty
    def __subcommand(self):
        try:
            name = self.__argv[len(self.__stack)]
            return self.__subcommands[name]
        except (KeyError, IndexError):
            return None

    def __parse(self):
        parser = Parser(self.__name, options=self.__options)
        argv_slice = self.__argv[len(self.__stack):]
        arguments, options = parser.parse(argv_slice)
        return (arguments, options)

    def __update(self, options):
        for call, cls in self.__classes:
            for name, value in vars(cls).items():
                if isinstance(value, Option):
                    setattr(call, name, options[name])

    def __mixin(self, exception=None):
        for mixin in self.__mixins.values():
            mixin.process(self, exception=exception)

    def __error(self, exception):
        if isinstance(exception, ParseError):
            exit(self.meta_format('error', message=exception.message))

    def __delegate(self):
        self.__subcommand(self.__argv, stack=self.__stack)

    @cachedproperty
    def __attributes(self):
        attributes = OrderedDict()
        for call, cls in reversed(self.__classes):
            for name in getattr(cls, '__order__', vars(cls).keys()):
                value = getattr(cls, name)
                if name.startswith('_'):
                    continue
                attributes[name] = (call, name, value)
        return attributes

    @cachedproperty
    def __classes(self):
        classes = []
        for call in reversed(self.__stack):
            for cls in type(call).mro():
                if cls in Command.mro():
                    continue
                classes.append((call, cls))
        classes.append((None, Command))
        return classes

    @cachedproperty
    def __formatter(self):
        Formatter = self.__attributes['meta_formatter'][2]
        formatter = Formatter(
            name=self.__name,
            command=type(self),
            mixins=self.__mixins,
            options=self.__options,
            subcommands=self.__subcommands)
        return formatter

    @cachedproperty
    def __mixins(self):
        mixins = OrderedDict()
        for call, name, value in self.__attributes.values():
            if getattr(value, decorate_mixin.decorator, False):
                function = getattr(call, name)
                mixins[name] = convert_mixin(function)
        return mixins

    @cachedproperty
    def __name(self):
        names = self.__argv[:len(self.__stack)]
        name = ' '.join(names)
        return name

    @cachedproperty
    def __options(self):
        options = OrderedDict()
        for _, name, value in self.__attributes.values():
            if isinstance(value, Option):
                options[name] = value
        return options

    @cachedproperty
    def __subcommands(self):
        subcommands = OrderedDict()
        for call, name, value in self.__attributes.values():
            if call is not self:
                continue
            if getattr(value, decorate_command.marker, False):
                function = getattr(call, name)
                value = convert_command(function, Command)
            if not isinstance(value, type):
                continue
            if issubclass(value, Command):
                subcommands[name] = value
        return subcommands
