import re
import inspect
from jinja2 import Environment, PackageLoader
from sugarbowl import cachedproperty


class Formatter:

    # Public

    def __init__(self, *, name, command,
                 arguments, mixins, options, subcommands):
        self.__name = name
        self.__command = command
        self.__arguments = arguments
        self.__mixins = mixins
        self.__options = options
        self.__subcommands = subcommands

    # Format

    def format_doc(self, element=None):
        result = inspect.getdoc(self.__command)
        if element is not None:
            lines = result.splitlines()
            if element == 'head':
                lines = lines[:1]
            elif element == 'body':
                lines = lines[2:]
            else:
                lines = []
            result = '\n'.join(lines)
        return result

    def format_help(self):
        return self.render('help')

    def format_error(self, message):
        template = self.render('error')
        error = template.format(message=message)
        return error

    def format_usage(self):
        return self.render('usage')

    # Render

    def render(self, name, *, context=None, loader=None, **params):
        if context is None:
            context = self.context
        if loader is None:
            loader = self.loader
        params.setdefault('trim_blocks', True)
        params.setdefault('lstrip_blocks', True)
        environment = Environment(loader=loader, **params)
        name = environment.get_template(name)
        result = name.render(context)
        return result

    @cachedproperty
    def context(self):
        context = {}
        for name in dir(self):
            match = re.search(r'^context_(?P<key>.*)', name)
            if match:
                context[match.group('key')] = getattr(self, name)
        return context

    @cachedproperty
    def loader(self):
        return PackageLoader('clyde.formatter', 'templates')

    # Context

    @property
    def context_name(self):
        return self.__name

    @property
    def context_command(self):
        return self.__command

    @property
    def context_arguments(self):
        return self.__arguments

    @property
    def context_mixins(self):
        return self.__mixins

    @property
    def context_options(self):
        return self.__options

    @property
    def context_subcommands(self):
        return self.__subcommands
