import sys
import inspect
from collections import OrderedDict
from sugarbowl import cachedproperty
from ..argument import Argument
from ..formatter import Formatter
from ..mixin import Mixin, mixin as decorate_mixin
from ..option import Option
from ..parser import Parser, ParseError
from .decorator import command as decorate_command
from .error import DelegateError
from .metaclass import Metaclass


class Command(metaclass=Metaclass):

    # Public

    meta_formatter = Formatter
    meta_parser = Parser

    def __init__(self, *, name=None, stack=None, execute=None):
        if name is None:
            name = type(self).__name__.lower()
        if stack is None:
            stack = []
        if execute is None:
            execute = self.meta_execute
        if execute.__doc__:
            self.__doc__ = execute.__doc__
        self.__name = name
        self.__stack = stack
        self.__execute = execute

    def __getattr__(self, name):
        if not name.startswith('_'):
            for command in reversed(self.__ancestors):
                try:
                    return getattr(command, name)
                except AttributeError:
                    pass
        raise AttributeError(name)

    def __call__(self, argv=None):
        if argv is None:
            argv = sys.argv
        self.__stack.append(self)
        self.__invoke(argv)
        self.__stack.remove(self)

    def meta_execute(self):
        print(self.meta_format('help'))

    def meta_format(self, name, *args, delegate=None, **kwargs):
        if delegate is None:
            delegate = self in self.__stack[:-1]
        if delegate and self.__stack:
            command = self.__stack[-1]
            return command.meta_format(name, *args, **kwargs)
        function = getattr(self.__formatter, 'format_' + name)
        result = function(*args, **kwargs)
        return result

    def meta_parse(self, argv):
        argv = argv[len(self.__ancestors) + 1:]
        return self.__parser.parse(argv, self.__arguments, self.__options)

    # Private

    def __invoke(self, argv):
        try:
            return self.__delegate(argv)
        except DelegateError:
            pass
        try:
            result = self.meta_parse(argv)
            self.__update(result.options)
            self.__mixin()
            self.__execute(*result.arguments)
        except (ParseError, Exception) as exception:
            self.__mixin(exception)
            self.__error(exception)

    def __delegate(self, argv):
        try:
            name = argv[len(self.__ancestors) + 1]
            subcommand = self.__subcommands[name]
        except (IndexError, KeyError):
            raise DelegateError()
        subcommand(argv)

    def __update(self, options):
        for command, cls in self.__classes:
            for name, value in vars(cls).items():
                if isinstance(value, Option):
                    setattr(command, name, options[name])

    def __mixin(self, exception=None):
        for mixin in self.__mixins.values():
            mixin.process(self, exception=exception)

    def __error(self, exception):
        if isinstance(exception, ParseError):
            exit(self.meta_format('error', message=exception.message))
        raise exception

    @cachedproperty
    def __ancestors(self):
        ancestors = []
        for command in self.__stack:
            if command is self:
                break
            ancestors.append(command)
        return ancestors

    @cachedproperty
    def __arguments(self):
        arguments = OrderedDict()
        signature = inspect.signature(self.__execute)
        for name, item in signature.parameters.items():
            if isinstance(item.annotation, Argument):
                arguments[name] = item.annotation
                continue
            params = {}
            if item.annotation is not item.empty:
                params['type'] = item.annotation
            if item.kind is item.VAR_POSITIONAL:
                params['nargs'] = '*'
                params['default'] = []
            elif item.default is not item.empty:
                params['nargs'] = '?'
                params['default'] = item.default
            arguments[name] = Argument(**params)
        return arguments

    @cachedproperty
    def __attributes(self):
        attributes = OrderedDict()
        for command, cls in reversed(self.__classes):
            for name in getattr(cls, '__order__', vars(cls).keys()):
                value = getattr(cls, name)
                if name.startswith('_'):
                    continue
                attributes[name] = (command, name, value)
        return attributes

    @cachedproperty
    def __classes(self):
        classes = []
        for command in reversed(self.__ancestors + [self]):
            for cls in type(command).mro():
                if cls in Command.mro():
                    continue
                classes.append((command, cls))
        classes.append((None, Command))
        return classes

    @cachedproperty
    def __formatter(self):
        Formatter = self.__attributes['meta_formatter'][2]
        formatter = Formatter(
            name=self.__name,
            command=self,
            arguments=self.__arguments,
            mixins=self.__mixins,
            options=self.__options,
            subcommands=self.__subcommands)
        return formatter

    @cachedproperty
    def __mixins(self):
        mixins = OrderedDict()
        for command, name, value in self.__attributes.values():
            decorator = getattr(value, decorate_mixin.decorator, None)
            if decorator is not None:
                function = getattr(command, name)
                mixins[name] = Mixin(function, require=decorator.require)
        return mixins

    @cachedproperty
    def __options(self):
        options = OrderedDict()
        for _, name, value in self.__attributes.values():
            if isinstance(value, Option):
                options[name] = value
        return options

    @cachedproperty
    def __parser(self):
        Parser = self.__attributes['meta_parser'][2]
        parser = Parser(self.__name)
        return parser

    @cachedproperty
    def __subcommands(self):
        subcommands = OrderedDict()
        for command, name, value in self.__attributes.values():
            execute = None
            if command is not self:
                continue
            if getattr(value, decorate_command.marker, False):
                value = Command
                execute = getattr(command, name)
            if (isinstance(value, type) and
                issubclass(value, Command)):
                value = value(name=name, stack=self.__stack, execute=execute)
            if isinstance(value, Command):
                subcommands[name] = value
        return subcommands
