from cmdr import error
from cmdr import flag
from cmdr import utils


class CLI():
    def __init__(self, name, desc, action, version=None):
        if not utils.validate(name):
            raise error.CmdrError('Name "%s" is not a valid app name' % name)

        self.name = name
        self.desc = desc
        self.version = version
        self.action = action
        self.flags = {}

    def run(self, argv, env=None):
        argv = argv[1:]

        self.flag('help', 'print this help and exit', False)

        if self.version:
            self.flag('version', 'print version and exit', False)

        try:
            opts, args = parse_argv(self, argv, env)
        except error.ParseError:
            self.show_help(env)
            return 1

        if opts['help']:
            self.show_help(env)
            return 0
        elif self.version and opts['version']:
            self.show_version()
            return 0
        else:
            return self.action(opts, args)

    def flag(self, name, desc, val):
        if type(val) not in (int, float, bool, str):
            raise error.CmdrError(
                'Type "%s" not supported as flag value.' % str(type(val).__name__)
            )

        self.flags[name] = flag.Flag(name, desc, val)

    def show_help(self, env=None):
        msg = [
            '%s - %s\n' % (self.name, self.desc.capitalize()),
            '\n',
            'Usage: %s [options] [args]\n' % self.name,
            '\n',
            'Options:\n',
        ]

        flag_strlen = max([len(str(flag)) for _, flag in self.flags.iteritems()])
        for _, flag in self.flags.iteritems():
            padding = ' ' * (flag_strlen - len(str(flag)))
            msg.append('  %s%s    %s\n' % (flag, padding, flag.desc.capitalize()))

        if env is not None:
            msg.append(
                '\nOptions can also be configured via upper-case environment '
                'variables prefixed with "%s_"\n' % utils.canonize(self.name)
            )

        print ''.join(msg)

    def show_version(self):
        print '%s %s' % (self.name, self.version)


def parse_argv(app, argv, env=None):
    opts = {}
    args = []

    for _, fl in app.flags.iteritems():
        env_val = env and utils.resolve(app.name, fl.name, env)
        opts[fl.name] = parse_val(type(fl.val), env_val) if env_val else fl.val

    for idx, arg in enumerate(argv):
        if not arg.startswith('--'):
            args = argv[idx:]
            break

        opt = arg[2:].split('=', 1)
        name, val = opt if len(opt) == 2 else (opt[0], None)

        if name not in app.flags:
            raise error.ParseError('unknown flag "%s"' % name)

        flag = app.flags[name]

        if type(flag.val) != bool and val is None:
            raise error.ParseError('"%s" flags require a value' % type(flag.val).__name__)

        opts[name] = parse_val(type(flag.val), val)

    return opts, args


def parse_val(type, val):
    if type == bool:
        return True if val is None else val == 'True'
    elif type == int:
        return int(val)
    elif type == float:
        return float(val)
    else:
        return val
