#!/usr/bin/env python

from os                    import remove
from cogent.util.unit_test import TestCase, main
from cogent.app.ilm        import ILM, xhlxplot, hlxplot

__author__ = "Shandy Wikman"
__copyright__ = "Copyright 2007, The Cogent Project"
__contributors__ = ["Shandy Wikman"]
__license__ = "GPL"
__version__ = "1.0.1"
__maintainer__ = "Shandy Wikman"
__email__ = "ens01svn@cs.umu.se"
__status__ = "Development"

class xhlxplotTest(TestCase):
    """Tests for xhlxplot application controller"""

    def setUp(self):
        self.input = xhlx_input
        
        
    def test_input_as_lines(self):
        """Test xhlxplot stdout input as lines"""

        x = xhlxplot(InputHandler='_input_as_lines')
        res = x(self.input)

        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

    def test_input_as_string(self):
        """Test xhlxplot input as string"""

        x = xhlxplot()
        f = open('/tmp/ilm.mwm','w')
        f.write('\n'.join(self.input))
        f.close()
        res = x('/tmp/ilm.mwm')

        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()
        remove('/tmp/ilm.mwm')

    def test_get_result_path(self):
        """Tests xhlxplot result path"""

        x = xhlxplot(InputHandler='_input_as_lines')
        res = x(self.input)
        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus'])
        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None

        res.cleanUp()

class IlmTest(TestCase):
    """Tests for ilm application controller"""

    def setUp(self):
        self.input = ilm_input
        
        
    def test_stdout_input_as_lines(self):
        """Test ilm input as lines"""

        i = ILM(InputHandler='_input_as_lines')
        res = i(self.input)
        exp = '%s\n' % '\n'.join([str(i).strip('\n') for i in ilm_stdout])
        obs = res['StdOut'].read()

        self.assertEqual(obs,exp)
        #Exitstatus platform dependent?? Works but gives exitstatus 224
        #self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

    def test_stdout_input_as_string(self):
        """Test ilm input as string"""

        ilm = ILM()
        f = open('/tmp/ilm.mwm','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])        
        f.write(txt)
        f.close()
        res = ilm('/tmp/ilm.mwm')

        exp = '%s\n' % '\n'.join([str(i).strip('\n') for i in ilm_stdout])
        obs = res['StdOut'].read()        
        self.assertEqual(obs,exp)

        #Exitstatus platform dependent?? 
        #self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()
        remove('/tmp/ilm.mwm')

    def test_get_result_path(self):
        """Tests ilm result path"""

        i = ILM(InputHandler='_input_as_lines')
        res = i(self.input)
        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus'])
        #self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

class hlxplotTest(TestCase):
    """Tests for hlxplot application controller"""

    def setUp(self):
        self.input = hlx_input
        
        
    def test_input_as_lines(self):
        """Test hlxplot input as lines"""

        h = hlxplot(InputHandler='_input_as_lines')
        res = h(self.input)

        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

    def test_input_as_string(self):
        """Test hlxplot stdout input as string"""

        h = hlxplot()

        f = open('/tmp/ilm.mwm','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])
        f.write(txt)                      
        f.close()
        res = h('/tmp/ilm.mwm')

        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()
        remove('/tmp/ilm.mwm')

    def test_get_result_path(self):
        """Tests hlxplot result path"""

        h = hlxplot(InputHandler='_input_as_lines')
        res = h(self.input)
        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus'])
        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None

        res.cleanUp()
        
ilm_stdout =['\n', 'Final Matching:\n', '1 72\n', '2 71\n', '3 70\n', '4 69\n', '5 68\n', '6 67\n', '7 66\n', '8 0\n', '9 0\n', '10 25\n', '11 24\n', '12 23\n', '13 22\n', '14 0\n', '15 0\n', '16 0\n', '17 0\n', '18 48\n', '19 47\n', '20 46\n', '21 45\n', '22 13\n', '23 12\n', '24 11\n', '25 10\n', '26 0\n', '27 43\n', '28 42\n', '29 41\n', '30 40\n', '31 39\n', '32 38\n', '33 37\n', '34 56\n', '35 55\n', '36 54\n', '37 33\n', '38 32\n', '39 31\n', '40 30\n', '41 29\n', '42 28\n', '43 27\n', '44 0\n', '45 21\n', '46 20\n', '47 19\n', '48 18\n', '49 65\n', '50 64\n', '51 63\n', '52 62\n', '53 61\n', '54 36\n', '55 35\n', '56 34\n', '57 0\n', '58 0\n', '59 0\n', '60 0\n', '61 53\n', '62 52\n', '63 51\n', '64 50\n', '65 49\n', '66 7\n', '67 6\n', '68 5\n', '69 4\n', '70 3\n', '71 2\n', '72 1\n']

hlx_stdout  = ['\x07\x08d\x96\x04\x00\x00\x00H\x00\x00\x00\xfc\t\x00\x00,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xdc\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xdc\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x04\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffT\x01\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xdc\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00T\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00T\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffT\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffT\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff,\x01\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffT\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00,\x01\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00T\x01\x00\x00d\x00\x00\x00\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00T\x01\x00\x00d\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00,\x01\x00\x00\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xb4\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xb4\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xb4\x00\x00\x00,\x01\x00\x00\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff,\x01\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff,\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\x8c\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x8c\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xb4\x00\x00\x00d\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xffd\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xb4\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\xd8\xff\xff\xff\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00']
xhlx_stdout = ['\x07\x08d\x96\x04\x00\x00\x00H\x00\x00\x00\xfc\t\x00\x00$\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x81\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x8e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x13\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff:\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff$\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff6\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x81\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x87\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x8e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x88\x00\x00\x008\xff\xff\xff\'\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xbe\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x13\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff:\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff$\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff6\x01\x00\x00_\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffO\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x87\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x02\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x80\x00\x00\x00+\x01\x00\x00\xb2\x00\x00\x00\x02\x00\x00\x008\xff\xff\xff8\xff\xff\xffV\x00\x00\x008\xff\xff\xff\xf5\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x8c\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe1\x00\x00\x00\x04\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb9\x00\x00\x002\x01\x00\x00\x93\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff:\x02\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf2\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbf\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x04\x01\x00\x00\x91\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf3\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf7\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd5\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x02\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x00+\x01\x00\x00\x80\x00\x00\x00\x02\x00\x00\x008\xff\xff\xff\x10\x00\x00\x008\xff\xff\xff8\xff\xff\xff\'\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xbe\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe1\x00\x00\x006\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x002\x01\x00\x00\x93\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xce\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x06\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd3\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffs\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf3\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x0b\x02\x00\x008\xff\xff\xff\xcd\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb7\x01\x00\x008\xff\xff\xff\x18\x01\x00\x00\x04\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x02\x00\x00\x008\xff\xff\xff\x07\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x02\x00\x00\x008\xff\xff\xff\x10\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf5\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x8c\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x04\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffa\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcf\x00\x00\x00\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff$\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf1\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x91\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xcf\x00\x00\x00%\x01\x00\x008\xff\xff\xff8\xff\xff\xff)\x02\x00\x008\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xd5\x01\x00\x008\xff\xff\xff6\x01\x00\x006\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff8\xff\xff\xff%\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xcf\x00\x00\x00\xbe\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00c\x01\x00\x00\x9d\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x1b\x01\x00\x008\xff\xff\xff\x06\x02\x00\x008\xff\xff\xffh\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1b\x01\x00\x008\xff\xff\xffs\x01\x00\x008\xff\xff\xff\x1c\x01\x00\x00\x9d\x00\x00\x008\xff\xff\xff\x1b\x01\x00\x008\xff\xff\xff\x0b\x02\x00\x008\xff\xff\xff8\xff\xff\xff\x1b\x01\x00\x008\xff\xff\xff\xb7\x01\x00\x00\x1c\x01\x00\x00\x04\x01\x00\x00\x18\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x00\x1b\x01\x00\x008\xff\xff\xff\x07\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x00\x9d\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x95\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x86\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffN\x01\x00\x008\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff)\x02\x00\x008\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff\xd5\x01\x00\x00N\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x88\x00\x00\x008\xff\xff\xff\x86\x01\x00\x008\xff\xff\xff8\xff\xff\xff*\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x0e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x95\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x86\x01\x00\x00\x80\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1c\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff)\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x80\x00\x00\x00\xf9\x00\x00\x00\x81\x00\x00\x00c\x00\x00\x008\xff\xff\xff8\xff\xff\xffV\x00\x00\x008\xff\xff\xffT\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xf8\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x00\x83\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffc\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffT\x01\x00\x00\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x00\xf9\x00\x00\x00O\x00\x00\x00c\x00\x00\x008\xff\xff\xff\x88\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x86\x01\x00\x008\xff\xff\xff8\xff\xff\xff*\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x00\xb5\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffj\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x95\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x95\x00\x00\x008\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x86\x01\x00\x008\xff\xff\xff8\xff\xff\xff*\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb5\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x00\x00@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffq\x01\x00\x00j\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xea\x01\x00\x00@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xaa\x01\x00\x00*\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcf\x00\x00\x00@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1a\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xda\x00\x00\x008\xff\xff\xff8\xff\xff\xffB\x01\x00\x00q\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x07\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x07\x01\x00\x008\xff\xff\xff\xcf\x00\x00\x00\xea\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff8\xff\xff\xffT\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffH\x01\x00\x00\xaa\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x9d\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xfc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbc\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x10\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe9\x00\x00\x008\xff\xff\xff\x9d\x00\x00\x00\xcc\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x008\xff\xff\xff8\xff\xff\xff6\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffH\x00\x00\x008\xff\xff\xff\xb8\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xa9\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x16\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xfc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc6\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbc\x00\x00\x008\xff\xff\xff8\xff\xff\xff$\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xcc\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff6\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xea\x00\x00\x008\xff\xff\xff8\xff\xff\xff\\\x00\x00\x008\xff\xff\xff\xcc\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xbd\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff*\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe4\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x04\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb8\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffT\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xd6\x00\x00\x008\xff\xff\xff\xf2\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xea\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xdb\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcf\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff6\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa4\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf7\x00\x00\x008\xff\xff\xff\x89\x01\x00\x00\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xea\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x08\x01\x00\x008\xff\xff\xff$\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xea\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xdb\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00i\x00\x00\x00\x9d\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb1\x00\x00\x00\xa8\x00\x00\x00\xcd\x00\x00\x008\xff\xff\xff\x04\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb1\x00\x00\x00i\x00\x00\x00\xcd\x00\x00\x008\xff\xff\xff\x07\x00\x00\x00\x04\x01\x00\x008\xff\xff\xff\xb1\x00\x00\x00O\x00\x00\x00\x86\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xb1\x00\x00\x00O\x00\x00\x00\xd9\x00\x00\x00\xdf\xff\xff\xffW\x01\x00\x00\xcd\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x9d\x00\x00\x00\xcc\x01\x00\x00\xb1\x00\x00\x00\xc8\x00\x00\x00\xcd\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9d\x00\x00\x00\xbc\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf0\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9d\x00\x00\x00}\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x07\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x9d\x00\x00\x00c\x00\x00\x00r\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x9d\x00\x00\x00c\x00\x00\x00\xc5\x00\x00\x00\xdf\xff\xff\xffk\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb1\x00\x00\x00\xcc\x01\x00\x00\x9d\x00\x00\x00\xdc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffV\x00\x00\x008\xff\xff\xff\xff\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x9b\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa8\x00\x00\x008\xff\xff\xff8\xff\xff\xff"\x01\x00\x00\x80\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x07\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa4\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffW\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc8\x00\x00\x00\\\x01\x00\x00\xba\x00\x00\x00\xb9\x00\x00\x008\xff\xff\xff8\xff\xff\xffV\x00\x00\x008\xff\xff\xff\xff\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf0\x00\x00\x00\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffW\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xfa\x00\x00\x00\\\x01\x00\x00\x88\x00\x00\x00\xb9\x00\x00\x008\xff\xff\xff\xf2\x00\x00\x008\xff\xff\xff8\xff\xff\xff1\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00Y\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff`\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x0e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff\x1d\x01\x00\x00\xc9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x89\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xf7\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff$\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\'\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xab\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffB\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x00\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x008\xff\xff\xff\xeb\x00\x00\x00\xab\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffk\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xd9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x96\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xffY\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x97\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xe4\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff.\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe4\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x97\x00\x00\x00\n', "\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc5\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf2\x00\x00\x008\xff\xff\xff\x96\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x81\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd7\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd2\x00\x00\x008\xff\xff\xff8\xff\xff\xff`\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xba\x00\x00\x008\xff\xff\xff\x1d\x01\x00\x00\xf1\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff<\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xf1\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff$\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xffO\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb9\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xe3\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb4\x00\x00\x008\xff\xff\xff8\xff\xff\xff.\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x00\xe8\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x008\xff\xff\xff\xeb\x00\x00\x00\xd3\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1e\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xd3\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x81\x01\x00\x00\x17\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa5\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x15\x01\x00\x008\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa0\x00\x00\x008\xff\xff\xff8\xff\xff\xff.\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1a\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbf\x00\x00\x00V\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbf\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\xe3\x00\x00\x008\xff\xff\xff8\xff\xff\xff+\x00\x00\x008\xff\xff\xff\x91\x00\x00\x008\xff\xff\xff\xb9\x00\x00\x00\xe3\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x94\x00\x00\x008\xff\xff\xff\x05\x00\x00\x008\xff\xff\xff\xa0\x00\x00\x00x\x00\x00\x008\xff\xff\xffB\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x13\x01\x00\x008\xff\xff\xff\xfc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\\\x00\x00\x008\xff\xff\xff8\xff\xff\xffV\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\\\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x15\x01\x00\x00i\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff}\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x15\x01\x00\x008\xff\xff\xff\xf1\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf1\xff\xff\xff8\xff\xff\xff8\xff\xff\xffx\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffE\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffH\x00\x00\x00\xa8\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffH\x00\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x91\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff}\x00\x00\x00O\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x91\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd3\x00\x00\x00O\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf1\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x8c\x00\x00\x00\xed\x00\x00\x00c\x00\x00\x008\xff\xff\xff8\xff\xff\xff'\x01\x00\x00\xdf\x01\x00\x00i\x01\x00\x00\x91\xff\xff\xff\x96\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xa8\x00\x00\x00G\x01\x00\x00\xb8\x00\x00\x00\x96\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xce\x00\x00\x00\x91\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xcd\x00\x00\x00i\x00\x00\x00c\x00\x00\x008\xff\xff\xff\xcd\x00\x00\x00}\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xcd\x00\x00\x00\xbf\x00\x00\x00c\x00\x00\x008\xff\xff\xff\xce\x00\x00\x00\xf1\xff\xff\xff8\xff\xff\xff\xcd\x00\x00\x00x\x00\x00\x00\x01\x01\x00\x00O\x00\x00\x008\xff\xff\xff\xcd\x00\x00\x00\x13\x01\x00\x00\xf3\x01\x00\x00U\x01\x00\x00\x91\xff\xff\xff\xaa\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbc\x00\x00\x00G\x01\x00\x00\xa4\x00\x00\x00\xaa\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00@\x01\x00\x00\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff8\xff\xff\xffq\x01\x00\x00\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff\x1f\x01\x00\x008\xff\xff\xff8\xff\xff\xff\xeb\x00\x00\x008\xff\xff\xff\x11\x02\x00\x00\x87\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffe\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xcf\x00\x00\x00@\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1a\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xda\x00\x00\x008\xff\xff\xff8\xff\xff\xffB\x01\x00\x00q\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x07\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff9\x01\x00\x008\xff\xff\xff\xf2\x00\x00\x00\x11\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x9d\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xfc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbc\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x10\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x1b\x01\x00\x008\xff\xff\xff\xc0\x00\x00\x00\xf3\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe2\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xfc\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc6\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xbc\x00\x00\x008\xff\xff\xff8\xff\xff\xff$\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf3\x01\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xff\xe4\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdf\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x94\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x94\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff'\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffE\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffU\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x85\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x0e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x91\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff}\x00\x00\x00\xb9\x00\x00\x008\xff\xff\xff8\xff\xff\xff}\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff'\x01\x00\x00z\x00\x00\x008\xff\xff\xff8\xff\xff\xffU\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x99\x00\x00\x00\xce\x00\x00\x00\x9c\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf0\x00\x00\x00\xce\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x91\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffi\x00\x00\x00\xcd\x00\x00\x008\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x13\x01\x00\x00\x8e\x00\x00\x008\xff\xff\xff8\xff\xff\xffU\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x85\x00\x00\x00\xe2\x00\x00\x00\x88\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xdc\x00\x00\x00\xe2\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xcd\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x14\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x8e\x00\x00\x008\xff\xff\xff8\xff\xff\xffi\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x81\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x81\x00\x00\x00\x80\x00\x00\x002\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x01\x00\x00O\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffB\x01\x00\x00O\x00\x00\x00\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff$\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb2\x00\x00\x008\xff\xff\xffB\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x94\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x94\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff>\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x9e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x12\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\\\x01\x00\x00\xdd\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x9e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x12\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xab\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x80\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xf4\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xd4\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe4\x00\x00\x008\xff\xff\xff\x12\x02\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xb6\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x7f\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xc6\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\xff\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xb6\x00\x00\x008\xff\xff\xff\x86\x00\x00\x008\xff\xff\xff\x7f\x00\x00\x00\xe8\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff1\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xffr\x00\x00\x008\xff\xff\xff8\xff\xff\xff\x1a\x01\x00\x00\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xff\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x96\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xe8\x00\x00\x002\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x13\x01\x00\x008\xff\xff\xff\x9c\x00\x00\x008\xff\xff\xff\xaa\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x9c\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf9\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xdb\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x0e\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc2\x01\x00\x00\xdc\x00\x00\x008\xff\xff\xff8\xff\xff\xff)\x01\x00\x008\xff\xff\xff\xe2\x00\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\xf4\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xffG\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff\xf4\x01\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\xd6\x01\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff8\xff\xff\xff8\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x008\xff\xff\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"]
ilm_input   = xhlx_stdout
hlx_input   = ['seq1:GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC\n']
xhlx_input  = ['seq1:GGCTAGATAGCTCAGATGGTAGAGCAGAGGATTGAAGATCCTTGTGTCGTCGGTTCGATCCCGGCTCTGGCC\n']


if __name__ == '__main__':
    main()
