#!/usr/bin/env python

from os                    import remove
from cogent.util.unit_test import TestCase, main
from cogent.app.cove       import Coves,Covet

__author__ = "Shandy Wikman"
__copyright__ = "Copyright 2007, The Cogent Project"
__contributors__ = ["Shandy Wikman"]
__license__ = "GPL"
__version__ = "1.0"
__maintainer__ = "Shandy Wikman"
__email__ = "ens01svn@cs.umu.se"
__status__ = "Development"

class CovetTest(TestCase):
    """Tests for Covet application controller"""

    def setUp(self):
        self.input = cove_input
        
        
    def test_stdout_input_as_lines(self):
        """Test covet stdout input as lines"""

        c = Covet(InputHandler='_input_as_lines')
        res = c(self.input,remove_tmp=False)

        #Can't test for stdout since different models are created each run
        self.assertEqual(res['ExitStatus'],0)
        res.cleanUp()

    def test_stdout_input_as_string(self):
        """Test covet stdout input as string"""

        c = Covet()
        f = open('single.fasta','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])
        f.write(txt)
        f.close()

        res = c('single.fasta',remove_tmp=False)
        
        #Can't test for stdout since different models are created each run
        self.assertEqual(res['ExitStatus'],0)
        res.cleanUp()

        remove('single.fasta')

    def test_get_result_path(self):
        """Tests covet result path"""

        c = Covet(InputHandler='_input_as_lines')
        res = c(self.input,remove_tmp=False)
        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus','cm'
                                          ,'_input_filename'])
        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

class CovesTest(TestCase):
    """Tests for Coves application controller"""

    def setUp(self):
        self.input = cove_input
        

    def test_stdout_input_as_string(self):
        """Test coves stdout input as string"""

        c = Coves()
        exp= '%s\n' % '\n'.join([str(i).strip('\n') for i in coves_stdout])
        f = open('single.fasta','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])
        f.write(txt)
        f.close()

        #Create cm file needed for coves
        s = open('single.fasta.cm','w')
        txt = '\n'.join([str(i).strip('\n') for i in cove_cm])
        s.write(txt)
        s.close()

        res = c('single.fasta')
        obs = res['StdOut'].read()
        self.assertEqual(obs,exp) 
        res.cleanUp()
        remove('single.fasta')
        remove('single.fasta.cm')

    def test_get_result_path(self):
        """Tests coves result path"""
        c = Coves()

        f = open('single.fasta','w')
        txt = '\n'.join([str(i).strip('\n') for i in self.input])
        f.write(txt)
        f.close()

        #Create cm file needed for coves
        s = open('single.fasta.cm','w')
        txt = '\n'.join([str(i).strip('\n') for i in cove_cm])
        s.write(txt)
        s.close()

        res = c('single.fasta')
        self.assertEqualItems(res.keys(),['StdOut','StdErr','ExitStatus'])
        self.assertEqual(res['ExitStatus'],0)
        assert res['StdOut'] is not None
        res.cleanUp()

cove_input = ['>seq1\n', 
'GGCUAGAUAGCUCAGAUGGUAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUCCCGGCUCUGGC\n', 
'\n']

coves_stdout = ['coves - scoring and structure prediction of RNA sequences\n', 
'        using a covariance model\n', '     version 2.4.4, January 1996\n', 
'\n', '---------------------------------------------------\n', 
'Database to search/score:      single.fasta\n', 
'Model:                         single.fasta.cm\n', 
'GC% of background model:       50%\n', 
'---------------------------------------------------\n', '\n', 
'-32.55 bits : seq1\n', 
'          seq1 GGCUAGAUAGCUCAGAUGGUAGAGCAGAGGAUUGAAGAUCCUUGUGUCGUCGGUUCGAUC\n', '          seq1 >.>..>.>.>.......>.>.>.>..>.>>...........<<.<..<.<.<.<......\n', '\n', 
'          seq1 CCGGCUCUGGC\n', '          seq1 .<.<.<..<.<\n', '\n']

cove_cm = ['\xb0\xb2\xed\xe3%\x00\x00\x00\x06\x00\x00\x00\x01\x00\x00\x00\xff\xff\xff\xff\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x03\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x04\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x05\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x06\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x07\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x08\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\t\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\n', '\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x0b\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x0c\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\r\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x0e\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x0f\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x10\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x11\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x12\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x13\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x14\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x15\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x16\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x17\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x18\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x19\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1a\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1b\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1c\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1d\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1e\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00\x1f\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00 \x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00!\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00"\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00#\x00\x00\x00\xff\xff\xff\xffUUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x92$I\x92$I\xc2?\x92$I\x92$I\xd2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?\x92$I\x92$I\xc2?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?UUUUUU\xc5?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x01\x00\x00\x00$\x00\x00\x00\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xd9?\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?UUUUUU\xd5?\x00\x00\x00\x00\x00\x00\x00\x00UUUUUU\xd5?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00UUUUUU\xd5?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xbe?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x1e\x1e\x1e\x1e\x1e\x1e\xae?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x02\x00\x00\x00\xff\xff\xff\xff\xff\xff\xff\xff\x00\x00\x00\x00\x00\x00\xe0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00UUUUUU\xe5?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00UUUUUU\xd5?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xe0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\xd0?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x9a\x99\x99\x99\x99\x99\xd9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x9a\x99\x99\x99\x99\x99\xc9?\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00']


if __name__ == '__main__':
    main()
