"""
The idea of this PAS plugin is quite simple. It should check the user profile for the user being logged in
and if user has enabled two-step verification for his account (``enable_two_factor_authentication`` is set to
True), then redirect him further to a another page, where he would enter his Google Authenticator token, after
successful validation of which the user would be definitely logged in.

If user has not enabled the two-step verification for his account (``enable_two_factor_authentication``
is set to False), then do nothing so that Plone continues logging in the user normal way.
"""
import logging

from Globals import InitializeClass
from AccessControl.SecurityInfo import ClassSecurityInfo

from plone import api

from Products.PluggableAuthService.plugins.BasePlugin import BasePlugin
from Products.PluggableAuthService.utils import classImplements
from Products.PluggableAuthService.interfaces.plugins import IAuthenticationPlugin
from Products.PageTemplates.PageTemplateFile import PageTemplateFile
from Products.CMFCore.utils import getToolByName

from collective.googleauthenticator.adapter import ICameFrom
from collective.googleauthenticator.helpers import sign_user_data, is_whitelisted_client

logger = logging.getLogger("collective.googleauthenticator")

manage_addGoogleAuthenticatorPluginForm = PageTemplateFile(
    './www/add_google_authenticator_form',
    globals(),
    __name__ = 'manage_addGoogleAuthenticatorPluginForm'
    )

def addGoogleAuthenticatorPlugin(self, id, title='', REQUEST=None):
    """
    Add a Google Authenticator PAS Plugin to Plone PAS
    """
    o = GoogleAuthenticatorPlugin(id, title)
    self._setObject(o.getId(), o)

    if REQUEST is not None:
        REQUEST['RESPONSE'].redirect(
            '{0}/manage_main?manage_tabs_message=Google+Authentiactor+PAS+Plugin+added.'.format(self.absolute_url())
            )

class GoogleAuthenticatorPlugin(BasePlugin):
    """
    Google Authenticator PAS Plugin
    """
    meta_type = 'Collective Google Authenticator PAS'
    security = ClassSecurityInfo()

    def __init__(self, id, title=None):
        self._setId(id)
        self.title = title

    def authenticateCredentials(self, credentials):
        """
        Place to actually validate the user credentials specified and return a tuple (login, login)
        on success or (None, None) on failure.

        If we find one and two-step verification is not enabled for the account, we consider
        the authentication passed and log the user in. If two-step verification has been enabled
        for the account, the first step of authentication is considered to be passed and we go to
        the next page (having the user and pass remembered), where we check for the token generated by
        the token generator (Google Authenticator). If the token is valid too, we log the user in.
        """
        login = credentials['login']
        password = credentials['password']

        if not login:
            return (None, None)

        user = api.user.get(username=login)

        logger.debug("Found user: {0}".format(user.getProperty('username')))

        two_factor_authentication_enabled = user.getProperty('enable_two_factor_authentication')
        logger.debug("Two-step verification enabled: {0}".format(two_factor_authentication_enabled))

        if two_factor_authentication_enabled:
            # First see, if the password is correct.
            mt = getToolByName(self, 'portal_membership')
            plone_auth_user = None
            try:
                # This is where the password check is made.
                plone_auth_user = mt.authenticate(login, password, self.REQUEST)

            except Exception as e:
                plone_auth_user = None

            if not plone_auth_user:
                # Password check failed.
                return None

            if is_whitelisted_client():
                return (None, None)

            # Setting the data in the session doesn't seem to work. That's why we use the `ska` package.
            # The secret key would be then a combination of username, secret stored in users' profile
            # and the browser version.
            request = self.REQUEST
            response = request['RESPONSE']
            response.setCookie('__ac', '', path='/')

            # Redirect to token thing...
            signed_url = sign_user_data(request=request, user=user, url='@@google-authenticator-token')

            came_from_adapter = ICameFrom(request)
            # Appending possible `came_from`, but give it another name.
            came_from = came_from_adapter.getCameFrom()
            if came_from:
                signed_url = '{0}&next_url={1}'.format(signed_url, came_from)

            response.redirect(signed_url, lock=1)

            return 0

        if credentials.get('extractor') != self.getId():
            return (None, None)

        return (None, None)


classImplements(GoogleAuthenticatorPlugin, IAuthenticationPlugin)
InitializeClass(GoogleAuthenticatorPlugin)
