from zope.interface import implements
from zope.interface import Interface, Attribute
from zope import schema
from zope.schema.vocabulary import SimpleVocabulary, SimpleTerm
from collective.plonetruegallery import PTGMessageFactory as _

class IGalleryAdapter(Interface):
    sizes = Attribute("""image size mappings for the gallery type""")
    schema =  Attribute("""Schema of gallery specific""")
    
    name = Attribute("""Name of the gallery""")
    description = Attribute("Description of gallery type")
    cook_delay = Attribute("Time between updates of gallery images.  "
        "This update of images can be forced by appending refresh on a gallery.")
    cooked_images = Attribute("The images after they've been cooked up.")

    def cook():
        """
        this will make it so the gallery's images are not aggregated every
        single time the gallery is displayed.
        """

    def time_to_cook():
        """
        called to see if it is time to the cook the gallery.
        """

    def get_random_image():
        """
        returns a random image with data.
        returns an empty dict if the gallery is empty.
        """

    def log_error():
        """
        provides an easy way to log errors in gallery adapters.
        we don't want an adapter to prevent a page from loading...
        Who knows what kind of odd behavior some adapters may run into
        when working with picasa or flickr apis...
        """

    def retrieve_images():
        """
        This method retrieves all the images to be cooked
        """
                    
    def number_of_images():
        """"""
        
    def get_all_images():
        """
        returns all the cooked images for the gallery
        """

class IBasicAdapter(IGalleryAdapter):
    """
    Use plone to manage images for the gallery.
    """

    size_map = Attribute("allows us to map specific sizes to plone urls")

class IFlickrAdapter(IGalleryAdapter):
    """
    """
    
    flickr = Attribute("returns a flickrapi object for the api key")
    
    def get_flickr_user_id(username):
        """
        Returns the actual user id of someone given a username.
        if a username is not given, it will use the one in its
        settings
        """
        
    def get_flickr_photoset_id(theset=None, userid=None):
        """
        Returns the photoset id given a set name and user id.
        Uses the set and get_flickr_user_id() if they are
        not specified.
        """

    def get_mini_photo_url(photo):
        """
        takes a photo and creates the thumbnail photo url
        """
        
    def get_photo_link(photo):
        """
        creates the photo link url
        """
        
    def get_large_photo_url(photo):
        """
        create the large photo url
        """
        
class IPicasaAdapter(IGalleryAdapter):
    """
    """
    
    gd_client = Attribute("property for gd_client instance")
        
    def get_album_name(name, user):
        """
        Returns the selected album name and user.
        Uses name and user in settings if not specified.
        """
        
    def feed():
        """
        Returns the picasa feed for the given album.
        """
        
class IDisplayType(Interface):
    
    name = Attribute("""name of display type""")
    description = Attribute("""description of type""")
    schema = Attribute("""Options for this display type""")
    template = Attribute("""The template used to render the gallery""")
    user_warning = Attribute("""A warning to be displayed to to the user if they use this type.""")
    width = Attribute("""The width of the gallery""")
    height = Attribute("""The height of the gallery""")
    start_image_index = Attribute("What image the gallery should start playing at.")
            
class ISlideshowDisplayType(IDisplayType):
    """
    Using Slideshow 2 to display the gallery
    """
    
    image_buffer_ratio = Attribute("ratio an image size can be off with plone sizes")
    burns_zoom = Attribute("how far it zooms so you can calculate it correctly..")
    offset = Attribute("size needed for border offset.")
    
    def image_data():
        """
        assembles the image data to be set for the javascript
        """
    
    def css():
        """
        Returns the dynamic css needed.
        """
        
    def javascript():
        """
        Returns the dynamic javascript needed.
        """
        
    def assemble_image(image):
        """
        Creates image data for a given image
        """
        
    def get_width_and_height():
        """
        calculates a more safe width and height to compensate
        for potential stretching and kenburns effect...
        """
    
class IGallery(Interface):
    """
    marker interface for content types that implement
    the gallery 
    """
    
class IGallerySettings(Interface):
    
    gallery_type = schema.Choice(
        title=_(u"label_gallery_type", default=u"Type"),
        description=_(u"description_gallery_type", 
            default=u"Select the type of gallery you want this to be.  "
                    u"If you select something other than default, you "
                    u"must fill out the information in the corresponding "
                    u"tab for that gallery type."),
        vocabulary="collective.plonetruegallery.GalleryTypeVocabulary",
        default = "basic"
    )
    
    # the specific options for the gallery types will be added
    # dynamcially in the form
    
    size = schema.Choice(
        title=_(u"label_gallery_size", default=u"Size"),
        description=_(u"description_gallery_size",
            default=u"The actual sizes used can vary depending on the "
                    u"gallery type that is used since different services "
                    u"have different size constraints."),
        default='medium',
        vocabulary=SimpleVocabulary([
            SimpleTerm('small', 'small', _(u"label_size_small", default=u'Small')),
            SimpleTerm('medium', 'medium', _(u"label_size_medium", default=u'Medium')),
            SimpleTerm('large', 'large', _(u"label_size_large", default=u'Large'))
        ])
    )
    
    display_type = schema.Choice(
        title=_(u"label_gallery_display_type", default=u"Gallery Display Type"),
        description=_(
            u"label_gallery_display_type_description",
            default=u"Choose the method in which the gallery should be displayed"
        ),
        default="slideshow",
        vocabulary="collective.plonetruegallery.DisplayTypes"
    )
    
    # the options for the display type will also be added dynamically
    
    timed = schema.Bool(
        title=_(u"label_timed", default=u"Timed?"),
        description=_(u"description_timed", 
            default=u"Should this gallery automatically change images for the user?"
        ),
        default=True
    )
    
    delay = schema.Int(
        title=_(u"label_delay", default=u"Delay"),
        description=_(u"description_delay", 
            default=u"If slide show is timed, the delay sets "
                    u"how long before the next image is shown in miliseconds."
        ),
        default=5000,
        required=True
    )
    
    duration = schema.Int(
        title=_(u"label_image_change_duration", default=u"Change Duration"),
        description=_(u"description_fade_in_duration", 
            default=u"The amount of time the change effect should take in miliseconds."
        ),
        default=500,
        required=True
    )
    
    show_subgalleries = schema.Bool(
        title=_(u"label_show_subgalleries", default=u"Show Sub-Galleries?"),
        description=_(u"description_show_subgalleries",
            default=u"If you select this option, previews for all nested galleries "
                    u"will show up below this gallery."
        ),            
        default=True
    )
    
    batch_size = schema.Int(
        title=_(u"label_batch_size", default=u"Batch Size"),
        description=_(u"description_batch_size", 
            default=u"The amount of images shown in one page. "
                    u"This is not used for all display types."
        ),
        default=50,
        required=True
    )
    
class IBaseSettings(Interface):
    pass

class IFancyBoxDisplaySettings(IBaseSettings):
    pass
        
class ISlideShowDisplaySettings(IBaseSettings):
    
    show_slideshow_carousel = schema.Bool(
        title=_(u"label_show_carousel", default=u"Carousel"),
        description=_(u"description_show_carousel",
            default=u"Show Carousel for Slideshow type?"
                    u"This is only used for the Slideshow 2 display type."
        ),
        required=False,
        default=True
    )
    
    show_slideshow_infopane = schema.Bool(
        title=_(u"label_show_info_pane", default=u"Info Pane"),
        description=_(u"description_show_info_pane",
            default=u"Show Info pane for Slideshow type? "
                    u"This is only used for the Slideshow 2 display type."
        ),
        required=False,
        default=True
    )
    
    slideshow_effect = schema.Choice(
        title=_(u"label_slideshowEffect", default=u"Slideshow type effect"),
        description=_(u"description_slideshowEffect", 
            default=u"Select the effect you want to use. "
                    u"This is only used for the Slideshow 2 display type."
        ),
        default='kenburns:Slideshow.KenBurns',
        vocabulary=SimpleVocabulary([
            SimpleTerm('flash:Slideshow.Flash', 'flash:Slideshow.Flash', _(u"label_slideshowEffect_flash", default=u"Flash")),
            SimpleTerm('fold:Slideshow.Fold', 'fold:Slideshow.Fold', _(u"label_slideshowEffect_fold", default=u"Fold")),
            SimpleTerm('kenburns:Slideshow.KenBurns', 'kenburns:Slideshow.KenBurns', _(u"label_slideshowEffect_kenburns", default=u"Ken Burns")),
            SimpleTerm('push:Slideshow.Push', 'push:Slideshow.Push', _(u"label_slideshowEffect_push", default=u"Push")),
            SimpleTerm(':Slideshow', ':Slideshow', _(u"label_slideshowEffect_none", default=u"none"))
        ])
    )
    
class IHighSlideDisplaySettings(IBaseSettings):
    
    highslide_outlineType = schema.Choice(
        title=_(u"label_highslide_outlineType", default=u"Image outline type"),
        description=_(u"description_highslide_outlineType", 
            default=u"The style of the border around the image. "
                    u"This is only used for the Highslide display type."
        ),
        default='drop-shadow',
        vocabulary=SimpleVocabulary([
            SimpleTerm('rounded-white', 'rounded-white', 
                _(u"label_highslide_outlineType_rounded_white", default=u"Rounded White")),
            SimpleTerm('outer-glow', 'outer-glow', 
                _(u"label_highslide_outlineType_outer_glow", default=u"Outer Glow")),
            SimpleTerm('drop-shadow', 'drop-shadow',
                _(u"label_highslide_outlineType_drop_shadow", default=u"Drop Shadow")),
            SimpleTerm('glossy-dark', 'glossy-dark',
                _(u"label_highslide_outlineType_glossy_dark", default=u"Glossy Dark")
            )
        ])
    )

    
class IBasicGallerySettings(IBaseSettings):
    pass
    
class IFlickrGallerySettings(IBaseSettings):
    
    flickr_username = schema.TextLine(
        title=_(u"label_flickr_username", default=u"flickr username"),
        description=_(u"description_flickr_username",
            default=u"The username/id of your flickr account. "
                    u"This is only required if a flickr is selected as the gallery type."
        ),
        required=False
    )
    
    flickr_set = schema.TextLine(
        title=_(u"label_flickr_set", default="Flickr Set"),
        description=_(u"description_flickr_set", 
            default=u"Name/id of your flickr set."
                    u"This is only required if a flickr is selected as the gallery type."
        ),
        required=False
    )
    
class IPicasaGallerySettings(IBaseSettings):
    
    picasa_username = schema.TextLine(
        title=_(u"label_picasa_username", default=u"GMail address"),
        description=_(u"description_picasa_username",
            default=u"GMail address of who this album belongs to(including @gmail.com). "
                    u"This is only required if a Picasa is selected as the gallery type."
        ),
        required=False
    )
    
    picasa_album = schema.TextLine(
        title=_(u"label_picasa_album", default=u"Picasa Album"),
        description=_(u"description_picasa_album", 
            default=u"Name of your picasa web album. "
                    u"This will be the qualified name you'll see in the address bar "
                    u"or the full length name of the album. "
                    u"This is only required if a Picasa is selected as the gallery type."
        ),
        required=False
    )

class IImageInformationRetriever(Interface):
    """
    This interface is interesting for everybody who wants to filter
    the items to be shown in a gallery view
    """
    def getImageInformation(self, size):
        """
        Return a list of Information relevant for gallery display for each
        image.
        Size should be a hint of the image size to use, in string format.
        The standard implementations support the following sizes, which 
        map to the given size of the archetypes Image size:

            small -> mini
            medium -> preview
            large -> large

        This information returned consists of:
        image_url
            The URL to the image itself
        thumb_url
            The URL to a thumbnail version of the image
        link
            The Link to which an image must point to
        title
            The image title
        description
            The image description
        """

