#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Defines unit tests for :mod:`colour.colorimetry.spectrum` module.
"""

from __future__ import division, unicode_literals

import math
import numpy as np
import sys

if sys.version_info[:2] <= (2, 6):
    import unittest2 as unittest
else:
    import unittest

from colour.colorimetry.spectrum import (
    SpectralShape,
    SpectralPowerDistribution,
    TriSpectralPowerDistribution,
    constant_spd,
    zeros_spd,
    ones_spd)

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013 - 2014 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['SAMPLE_SPD_DATA',
           'NON_UNIFORM_SAMPLE_SPD_DATA',
           'ZEROS_SAMPLE_SPD_DATA',
           'INTERPOLATED_SAMPLE_SPD_DATA',
           'INTERPOLATED_NON_UNIFORM_SAMPLE_SPD_DATA',
           'NORMALISED_SAMPLE_SPD_DATA',
           'CIE_1931_2_DEGREE_STANDARD_OBSERVER',
           'CMFS_DATA',
           'TestSpectralShape',
           'TestSpectralPowerDistribution',
           'TestTriSpectralPowerDistribution',
           'TestConstantSpd',
           'TestZerosSpd',
           'TestOnes_spd']

SAMPLE_SPD_DATA = {
    340: 0.0000,
    360: 0.0000,
    380: 0.0000,
    400: 0.0641,
    420: 0.0645,
    440: 0.0562,
    460: 0.0537,
    480: 0.0559,
    500: 0.0651,
    520: 0.0705,
    540: 0.0772,
    560: 0.0870,
    580: 0.1128,
    600: 0.1360,
    620: 0.1511,
    640: 0.1688,
    660: 0.1996,
    680: 0.2397,
    700: 0.2852,
    720: 0.0000,
    740: 0.0000,
    760: 0.0000,
    780: 0.0000,
    800: 0.0000,
    820: 0.0000}

NON_UNIFORM_SAMPLE_SPD_DATA = {
    391.898: 16.33174,
    392.069: 16.333122,
    405.606: 40.197224,
    406.794: 39.923366,
    406.891: 39.924098,
    407.026: 39.925138,
    416.286: 40.064293,
    418.69: 40.00995,
    426.7: 18.045809,
    426.726: 18.045986,
    432.556: 38.435883,
    464.742: 29.534647,
    465.025: 29.534647,
    465.147: 29.534647,
    466.586: 38.22647,
    477.175: 7.487795,
    493.205: 7.684766,
    505.217: 7.684766,
    513.294: 20.701285,
    513.328: 20.704211,
    514.349: 20.704211,
    514.516: 20.709788,
    515.109: 20.709788,
    538.034: 7.684766,
    564.807: 20.704211,
    566.247: 20.709788,
    569.592: 32.103387,
    580.133: 37.54849,
    581.198: 37.54849,
    582.642: 40.197224,
    588.977: 18.045986,
    589.159: 18.045809,
    600.113: 8.64302,
    600.603: 8.647157,
    600.718: 8.640394,
    601.068: 8.640394,
    601.322: 8.647157,
    601.484: 8.64302,
    657.805: 14.448826,
    658.288: 14.448826,
    658.761: 8.537097,
    674.438: 38.22647,
    678.39: 20.709788,
    703.725: 38.435883,
    711.318: 8.647157,
    711.519: 8.640394,
    711.563: 22.532398,
    711.699: 8.647157,
    711.99: 22.536906,
    723.132: 16.33174,
    723.642: 16.333122,
    761.265: 41.342187,
    786.089: 8.850659,
    805.862: 8.850659}

ZEROS_SAMPLE_SPD_DATA = np.array([
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0641,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0645,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0562,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0537,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0559,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0651,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0705,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.0772,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.087,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.1128,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.136,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.1511,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.1688,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.1996,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.2397,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.2852,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.,
    0.])

INTERPOLATED_SAMPLE_SPD_DATA = np.array([
    0.0,
    0.000230709627131,
    0.000384144814593,
    0.000507137093115,
    0.000632114832536,
    0.000778810112328,
    0.000955965592105,
    0.00116304138214,
    0.00139192191388,
    0.00162862281044,
    0.00185499775718,
    0.00205044537212,
    0.00219361607656,
    0.0022641189655,
    0.00224422867823,
    0.0021205922688,
    0.00188593607656,
    0.00154077259663,
    0.00109510735048,
    0.000570145756392,
    0.0,
    -0.000581325882457,
    -0.00118394563098,
    -0.00182033894223,
    -0.00249227866029,
    -0.00319224818472,
    -0.00390485887859,
    -0.00460826747654,
    -0.00527559349282,
    -0.00587633662932,
    -0.00637779418361,
    -0.00674647845703,
    -0.00694953416268,
    -0.00695615583349,
    -0.00673900523026,
    -0.00627562874972,
    -0.00554987483254,
    -0.00455331137139,
    -0.00328664311902,
    -0.00176112909624,
    0.0,
    0.00197983212891,
    0.004221516875,
    0.00676274498047,
    0.00961776,
    0.0127802856445,
    0.016226453125,
    0.0199177284961,
    0.02380384,
    0.0278257054102,
    0.031918359375,
    0.0360138807617,
    0.04004432,
    0.0439446264258,
    0.047655575625,
    0.0511266967773,
    0.0543192,
    0.0572089036914,
    0.059789161875,
    0.062073791543,
    0.0641,
    0.0659089980664,
    0.067477265625,
    0.0687760601367,
    0.06979104,
    0.0705205200195,
    0.070973726875,
    0.0711690545898,
    0.07113232,
    0.0708950182227,
    0.070492578125,
    0.069962617793,
    0.0693432,
    0.0686710876758,
    0.067979999375,
    0.0672988647461,
    0.06665008,
    0.0660477633789,
    0.065496010625,
    0.0649871504492,
    0.0645,
    0.0640078054492,
    0.063510900625,
    0.0630161383789,
    0.06252864,
    0.0620519897461,
    0.061588429375,
    0.0611390526758,
    0.060704,
    0.060282652793,
    0.059873828125,
    0.0594759732227,
    0.05908736,
    0.0587062795898,
    0.058331236875,
    0.0579611450195,
    0.05759552,
    0.0572346751367,
    0.056879915625,
    0.0565337330664,
    0.0562,
    0.0558838117578,
    0.05558951625,
    0.0553200090234,
    0.05507696,
    0.0548609619141,
    0.05467167875,
    0.0545079941797,
    0.05436816,
    0.0542499445703,
    0.05415078125,
    0.0540679168359,
    0.05399856,
    0.0539400297266,
    0.05388990375,
    0.0538461669922,
    0.05380736,
    0.0537727273828,
    0.05374236625,
    0.0537173746484,
    0.0537,
    0.0536924831445,
    0.053693175625,
    0.0537002700586,
    0.05371296,
    0.0537313110352,
    0.053756131875,
    0.0537888454492,
    0.05383136,
    0.0538859401758,
    0.053955078125,
    0.0540413645898,
    0.05414736,
    0.0542754655664,
    0.054427794375,
    0.0546060424805,
    0.05481136,
    0.055044222207,
    0.055304300625,
    0.0555903341211,
    0.0559,
    0.0562314078516,
    0.0565875975,
    0.0569712793359,
    0.05738336,
    0.0578231689453,
    0.058288685,
    0.0587767629297,
    0.05928336,
    0.0598037625391,
    0.0603328125,
    0.0608651340234,
    0.06139536,
    0.0619183586328,
    0.06242946,
    0.0629246826172,
    0.06340096,
    0.0638563672266,
    0.0642903475,
    0.0647039387109,
    0.0651,
    0.0654816472656,
    0.06584678625,
    0.0661929822656,
    0.06651904,
    0.0668248535156,
    0.06711125625,
    0.0673798710156,
    0.06763296,
    0.0678732747656,
    0.06810390625,
    0.0683281347656,
    0.06854928,
    0.0687705510156,
    0.06899489625,
    0.0692248535156,
    0.0694624,
    0.0697088022656,
    0.06996446625,
    0.0702287872656,
    0.0705,
    0.0707762707031,
    0.07105944625,
    0.0713515975781,
    0.07165392,
    0.0719668457031,
    0.07229015625,
    0.0726230950781,
    0.07296448,
    0.0733128157031,
    0.07366640625,
    0.0740234675781,
    0.07438224,
    0.0747411007031,
    0.07509867625,
    0.0754539550781,
    0.0758064,
    0.0761560607031,
    0.07650368625,
    0.0768508375781,
    0.0772,
    0.0775527019922,
    0.0779042625,
    0.0782507620703,
    0.07859088,
    0.0789255615234,
    0.079257685,
    0.0795917291016,
    0.07993344,
    0.0802894985547,
    0.0806671875,
    0.0810740586328,
    0.0815176,
    0.0820049030859,
    0.08254233,
    0.0831351806641,
    0.08378736,
    0.0845010451172,
    0.0852763525,
    0.0861110051953,
    0.087,
    0.0879384532422,
    0.08893089,
    0.0899818330078,
    0.09109296,
    0.0922634521484,
    0.0934903425,
    0.0947688644141,
    0.0960928,
    0.0974548285547,
    0.098846875,
    0.10026045832,
    0.10168704,
    0.103118372461,
    0.1045468475,
    0.105965844727,
    0.10737008,
    0.108755953867,
    0.1101219,
    0.111468733633,
    0.1128,
    0.114120657988,
    0.115431176875,
    0.116730532871,
    0.1180176,
    0.119291174316,
    0.120549998125,
    0.121792784199,
    0.12301824,
    0.124225091895,
    0.125412109375,
    0.126578129277,
    0.12772208,
    0.128843005723,
    0.129940090625,
    0.131012683105,
    0.13206032,
    0.133082750801,
    0.134079961875,
    0.135052200684,
    0.136,
    0.136923531484,
    0.13782092,
    0.138690739766,
    0.13953264,
    0.140347216797,
    0.141135885,
    0.141900750078,
    0.14264448,
    0.143370177109,
    0.14408125,
    0.144781285391,
    0.14547392,
    0.146162712422,
    0.146851015,
    0.147541845703,
    0.14823776,
    0.148940722734,
    0.14965198,
    0.150371931016,
    0.1511,
    0.151834687363,
    0.152574745625,
    0.15331986209,
    0.15407056,
    0.154828088379,
    0.155594311875,
    0.156371600605,
    0.15716272,
    0.157970720645,
    0.158798828125,
    0.159650332871,
    0.16052848,
    0.16143635916,
    0.162376794375,
    0.163352233887,
    0.16436464,
    0.165415378926,
    0.166505110625,
    0.167633678652,
    0.1688,
    0.170002988242,
    0.171244585,
    0.172526722383,
    0.1738504,
    0.175215795898,
    0.1766223775,
    0.178069012539,
    0.17955408,
    0.181075581055,
    0.18263125,
    0.184218665195,
    0.18583536,
    0.187478933711,
    0.1891471625,
    0.190838110352,
    0.19255024,
    0.194282523867,
    0.196034555,
    0.197806658008,
    0.1996,
    0.201405046895,
    0.203174116875,
    0.204868198965,
    0.206468,
    0.207971350098,
    0.209390608125,
    0.210750067168,
    0.21208336,
    0.213430864551,
    0.214837109375,
    0.216348179121,
    0.21800912,
    0.219861345254,
    0.221940040625,
    0.224271569824,
    0.22687088,
    0.229738907207,
    0.232859981875,
    0.236199234277,
    0.2397,
    0.24333728293,
    0.24726205625,
    0.251598942852,
    0.25639424,
    0.261625952148,
    0.26721382375,
    0.27302937207,
    0.27890592,
    0.284648628867,
    0.29004453125,
    0.294872563789,
    0.2989136,
    0.301960483086,
    0.30382805875,
    0.304363208008,
    0.30345488,
    0.301044124805,
    0.29713412625,
    0.291800234727,
    0.2852,
    0.277470396855,
    0.268408756875,
    0.257826504004,
    0.24565104,
    0.23191126709,
    0.216723110625,
    0.200275041738,
    0.1828136,
    0.164628916074,
    0.146040234375,
    0.127381435723,
    0.10898656,
    0.0911753288086,
    0.074238668125,
    0.058424230957,
    0.04392192,
    0.030849410293,
    0.019237671875,
    0.00901649244141,
    0.0,
    -0.00801472138672,
    -0.014901410625,
    -0.0205102174414,
    -0.02476736,
    -0.0276658569336,
    -0.029256259375,
    -0.0296373829883,
    -0.02894704,
    -0.0273527712305,
    -0.025042578125,
    -0.0222156547852,
    -0.01907312,
    -0.0158087492773,
    -0.012599706875,
    -0.00959727783203,
    -0.0069176,
    -0.00463239607422,
    -0.002759705625,
    -0.00125461712891,
    0.0,
    0.00114620320313,
    0.0021657375,
    0.00301036398438,
    0.00365056,
    0.00407329101563,
    0.0042797825,
    0.00428329179688,
    0.00410688,
    0.00378118382813,
    0.0033421875,
    0.00282899460937,
    0.0022816,
    0.00173866164063,
    0.0012352725,
    0.000800732421875,
    0.00045632,
    0.000213064453125,
    6.95175e-05,
    9.52523437499e-06,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0])

INTERPOLATED_NON_UNIFORM_SAMPLE_SPD_DATA = np.array([
    16.329778092653534,
    16.722912720487408,
    17.781520025709202,
    19.389396164994253,
    21.430343339551229,
    23.788163750588797,
    26.346659599315629,
    28.989633086940387,
    31.600886414671749,
    34.064221783718359,
    36.263441395288908,
    38.082347450592067,
    39.404742150836476,
    40.11442769723083,
    40.111243698632961,
    39.925103634593356,
    39.890167182851847,
    39.799324068607753,
    39.680556793202086,
    39.561846087380928,
    39.471172681890323,
    39.436517307476286,
    39.48586069488487,
    39.647183574862154,
    39.948466678154162,
    40.374181466196127,
    40.48094304141032,
    39.587763146291486,
    37.399010976170359,
    34.259517356362792,
    30.574008075395952,
    26.747208921797039,
    23.183845684093217,
    20.288644150811674,
    18.466330110479589,
    18.131455041282106,
    19.61840843219283,
    22.601624594414307,
    26.621530083890939,
    31.218551456567102,
    35.933115268387226,
    40.312773210893994,
    44.121677839586575,
    47.374686359321345,
    50.100647614176424,
    52.328410448229917,
    54.086823705559908,
    55.404736230244509,
    56.310996866361826,
    56.834454457989942,
    57.003957849206984,
    56.848355884091028,
    56.396497406720187,
    55.677231261172572,
    54.719406291526283,
    53.551871341859396,
    52.203475256250044,
    50.703066878776312,
    49.07949505351629,
    47.361608624548111,
    45.578256435949854,
    43.758287331799629,
    41.930550156175542,
    40.12389375315567,
    38.367166966818147,
    36.689218641241062,
    35.118897620502501,
    33.685052748680583,
    32.416532869853405,
    31.342186828099081,
    30.490863467495679,
    29.891411632121336,
    29.572680166054141,
    29.538621483463793,
    33.594913455357137,
    41.048330704481025,
    45.551744551493051,
    47.142069192754008,
    46.264235269424269,
    43.363173422664232,
    38.883814293634217,
    33.271088523494647,
    26.969926753405872,
    20.42525962452827,
    14.08201777802222,
    8.3851318550480816,
    3.7284294254059347,
    0.15696105443034014,
    -2.42990988857985,
    -4.1333077865999499,
    -5.054357022605279,
    -5.2941819795711584,
    -4.9539070404729086,
    -4.1346565882858499,
    -2.937555005985296,
    -1.4637266765465775,
    0.18570401705500184,
    1.9096126918441103,
    3.6068749648454346,
    5.1763664530836513,
    6.5169627735834474,
    7.5275395433694907,
    8.1223798031419818,
    8.3314831440319406,
    8.2374449381432484,
    7.9231247316710824,
    7.471382070810634,
    6.9650765017570722,
    6.4870675707055883,
    6.1202148238513567,
    5.9473778073895645,
    6.0514160675153867,
    6.5151891504240087,
    7.4215566023106092,
    8.833103318324067,
    10.654113908297491,
    12.714472049860778,
    14.843629854563192,
    16.871039433954007,
    18.626152899582493,
    19.938422362997915,
    20.637299935749549,
    20.706835013761975,
    20.713132278871178,
    20.620587315542718,
    20.396819413491482,
    20.054405097489894,
    19.605938071943182,
    19.064012041256547,
    18.4412207098352,
    17.750157782084358,
    17.003416962409233,
    16.21359195521504,
    15.39327646490699,
    14.555064195890296,
    13.711548852570164,
    12.875324139351815,
    12.058983760640459,
    11.275121420841311,
    10.536330824359577,
    9.8552056756004767,
    9.2443396789692205,
    8.7163265388710194,
    8.2837599597110891,
    7.9592336458946358,
    7.7553413018268822,
    7.6846766319130309,
    7.7560843464236555,
    7.9618017238567509,
    8.2894689503223837,
    8.7267260484673717,
    9.2612130409385305,
    9.8805699503826752,
    10.572436799446624,
    11.324453610777194,
    12.1242604070212,
    12.959497210825456,
    13.817804044836787,
    14.686820931702,
    15.554187894067923,
    16.407544954581358,
    17.234532135889136,
    18.022789460638052,
    18.759956951474948,
    19.433674631046632,
    20.031582521999919,
    20.541320646981621,
    20.950529028638556,
    21.246847689617535,
    21.417916652565395,
    21.45137594012893,
    21.334865574954971,
    21.056025579690328,
    20.604750277178088,
    20.494345115826086,
    22.221705466620591,
    25.661450384841267,
    29.775598875670866,
    33.531925166363266,
    36.383260993356132,
    38.399428506226592,
    39.690190938525532,
    40.365311523803889,
    40.53455349561257,
    40.307680087502476,
    39.794454533024528,
    39.104640065729612,
    38.347999919168693,
    37.634297326892622,
    37.374685122294757,
    39.119690201949446,
    40.253206513094959,
    38.299118521375533,
    34.130306341878899,
    28.883500015402014,
    23.695429582741834,
    19.702825084695338,
    18.042370406516945,
    17.94214615027019,
    17.43795204283763,
    16.603615433885011,
    15.527509991951053,
    14.298009385574467,
    13.003487283293955,
    11.732317353648233,
    10.572873265176007,
    9.6135286864159859,
    8.9426572859068862,
    8.6486327321874104,
    8.6382750052781461,
    8.6084617675042612,
    8.468358249678241,
    8.2364858196481414,
    7.9184592502045419,
    7.5198933141380104,
    7.046402784239123,
    6.5036024332984521,
    5.8971070341065701,
    5.2325313594540521,
    4.5154901821314706,
    3.7515982749293966,
    2.9464704106384061,
    2.1057213620490702,
    1.2349659019519637,
    0.33981880313765833,
    -0.57410516160327019,
    -1.5011912194802499,
    -2.435824597702708,
    -3.3723905234800733,
    -4.3052742240217672,
    -5.2288609265372266,
    -6.1375358582358643,
    -7.0256842463271116,
    -7.8876913180204067,
    -8.7179423005251611,
    -9.5108224210508148,
    -10.260716906806783,
    -10.9620109850025,
    -11.609089882847389,
    -12.196338827550875,
    -12.718143046322391,
    -13.168887766371361,
    -13.542958214907209,
    -13.834739619139363,
    -14.038617206277252,
    -14.148976203530301,
    -14.160201838107939,
    -14.066679337219593,
    -13.862793928074687,
    -13.542930837882647,
    -13.1014752938529,
    -12.532812523194881,
    -11.831327753117998,
    -10.991406210831702,
    -10.0074331235454,
    -8.8737937184685336,
    -7.584873222810522,
    -6.1350568637807834,
    -4.5187298685887605,
    -2.7302774644438728,
    -0.76408487855554896,
    1.3854626618667893,
    3.7239799296137082,
    6.2570816974757832,
    8.9903827382435928,
    11.929497824707706,
    14.976162376128192,
    5.0842394268246842,
    -6.729199659031746,
    -14.622346068118178,
    -19.038675241006352,
    -20.42166261826803,
    -19.214783640474934,
    -15.861513748198826,
    -10.805328382011458,
    -4.4897029824845571,
    2.6418870098101221,
    10.145966154300845,
    17.57905901041585,
    24.497690137583401,
    30.458384095231757,
    35.017665442789166,
    37.732058739683879,
    38.169776205018429,
    36.103214684325515,
    31.478432175406848,
    24.247019424727963,
    14.378015306889955,
    2.091468108491096,
    -12.204368693198287,
    -28.096681743592715,
    -45.172657688106732,
    -63.019483172154736,
    -81.224344841151407,
    -99.374429340511085,
    -117.05692331564838,
    -133.85901341197774,
    -149.36788627491376,
    -163.1707285498708,
    -174.85472688226358,
    -184.00706791750642,
    -190.21493830101389,
    -193.06552467820057,
    -192.14601369448084,
    -187.04359199526934,
    -177.3454462259804,
    -162.63876303202883,
    -142.51072905882879,
    -116.54853095179507,
    -84.339355356341926,
    -45.470388917884165,
    0.47118171816398458,
    53.864768804832657,
    111.8945463849911,
    165.94898256049586,
    206.80450977142198,
    225.23756045784464,
    212.02456705983889,
    157.94196201747988,
    53.766177770842909,
    24.17560361919103,
    186.23524781861047,
    339.56114029917143,
    476.46766930775459,
    589.26922309123972,
    670.28018989650786,
    711.81495797043897,
    706.18791555991368,
    645.71345091181195,
    522.70595227301499,
    329.4798078904023,
    58.349406010854921,
    509.1551100147168,
    4467.1261823844488,
    11974.830170191433,
    22711.662586777918,
    36357.018945486168,
    52590.294759658442,
    71090.88554263697,
    91538.18680776401,
    113611.59406838182,
    136990.50283783258,
    161354.3086294588,
    186382.40695660232,
    211754.19333260573,
    237149.06327081125,
    262246.41228456091,
    286725.6358871971,
    310266.12959206227,
    332547.28891249833,
    353248.50936184759,
    372049.18645345268,
    388628.71570065536,
    402666.49261679815,
    413841.91271522321,
    421834.37150927284,
    426323.26451228926,
    426987.98723761475,
    423507.93519859167,
    415562.50390856201,
    402831.08888086828,
    384993.08562885254,
    361727.88966585719,
    332714.89650522423,
    297633.50166029623,
    256163.10064441545,
    207983.08897092377,
    152772.86215316367,
    90211.815704477645,
    19979.345138207736,
    -58110.175210166279,
    -143006.86242516522,
    -232998.00084417849,
    -326364.54862905433,
    -421387.46394164133,
    -516347.70494378969,
    -609526.22979734745,
    -699203.99666416366,
    -783661.96370608767,
    -861181.08908496739,
    -930042.33096265292,
    -988526.6475009931,
    -1034914.9968618358,
    -1067488.3372070307,
    -1084527.6266984274,
    -1084313.8234978735,
    -1065127.8857672184,
    -1025250.7716683119,
    -962963.4393630022,
    -876546.84701313789,
    -764281.95278056816,
    -624449.71482714266,
    -455331.09131470934,
    -255207.04040511837,
    -22358.52026021701,
    243566.4923879448,
    535769.78067795967,
    845123.14010018017,
    1162497.0914997908,
    1478762.1557219757,
    1784788.85361192,
    2071447.7060148111,
    2329609.2337758304,
    2550143.9577401648,
    2723922.398752999,
    2841815.0776595157,
    2894692.5153049049,
    2873425.2325343466,
    2768883.7501930296,
    2571938.5891261348,
    2273460.2701788512,
    1864319.3141963617,
    1335386.2420238478,
    677531.57450650295])

NORMALISED_SAMPLE_SPD_DATA = np.array([
    0.0,
    0.0,
    0.0,
    22.4754558205,
    22.6157082749,
    19.7054698457,
    18.8288920056,
    19.6002805049,
    22.8260869565,
    24.7194950912,
    27.0687237027,
    30.5049088359,
    39.5511921459,
    47.6858345021,
    52.9803646564,
    59.1865357644,
    69.9859747546,
    84.04628331,
    100.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0,
    0.0, ])

CIE_1931_2_DEGREE_STANDARD_OBSERVER = {
    'x_bar': {
        380: 0.001368,
        385: 0.002236,
        390: 0.004243,
        395: 0.007650,
        400: 0.014310,
        405: 0.023190,
        410: 0.043510,
        415: 0.077630,
        420: 0.134380,
        425: 0.214770,
        430: 0.283900,
        435: 0.328500,
        440: 0.348280,
        445: 0.348060,
        450: 0.336200,
        455: 0.318700,
        460: 0.290800,
        465: 0.251100,
        470: 0.195360,
        475: 0.142100,
        480: 0.095640,
        485: 0.057950,
        490: 0.032010,
        495: 0.014700,
        500: 0.004900,
        505: 0.002400,
        510: 0.009300,
        515: 0.029100,
        520: 0.063270,
        525: 0.109600,
        530: 0.165500,
        535: 0.225750,
        540: 0.290400,
        545: 0.359700,
        550: 0.433450,
        555: 0.512050,
        560: 0.594500,
        565: 0.678400,
        570: 0.762100,
        575: 0.842500,
        580: 0.916300,
        585: 0.978600,
        590: 1.026300,
        595: 1.056700,
        600: 1.062200,
        605: 1.045600,
        610: 1.002600,
        615: 0.938400,
        620: 0.854450,
        625: 0.751400,
        630: 0.642400,
        635: 0.541900,
        640: 0.447900,
        645: 0.360800,
        650: 0.283500,
        655: 0.218700,
        660: 0.164900,
        665: 0.121200,
        670: 0.087400,
        675: 0.063600,
        680: 0.046770,
        685: 0.032900,
        690: 0.022700,
        695: 0.015840,
        700: 0.011359,
        705: 0.008111,
        710: 0.005790,
        715: 0.004109,
        720: 0.002899,
        725: 0.002049,
        730: 0.001440,
        735: 0.001000,
        740: 0.000690,
        745: 0.000476,
        750: 0.000332,
        755: 0.000235,
        760: 0.000166,
        765: 0.000117,
        770: 0.000083,
        775: 0.000059,
        780: 0.000042},
    'y_bar': {
        380: 0.000039,
        385: 0.000064,
        390: 0.000120,
        395: 0.000217,
        400: 0.000396,
        405: 0.000640,
        410: 0.001210,
        415: 0.002180,
        420: 0.004000,
        425: 0.007300,
        430: 0.011600,
        435: 0.016840,
        440: 0.023000,
        445: 0.029800,
        450: 0.038000,
        455: 0.048000,
        460: 0.060000,
        465: 0.073900,
        470: 0.090980,
        475: 0.112600,
        480: 0.139020,
        485: 0.169300,
        490: 0.208020,
        495: 0.258600,
        500: 0.323000,
        505: 0.407300,
        510: 0.503000,
        515: 0.608200,
        520: 0.710000,
        525: 0.793200,
        530: 0.862000,
        535: 0.914850,
        540: 0.954000,
        545: 0.980300,
        550: 0.994950,
        555: 1.000000,
        560: 0.995000,
        565: 0.978600,
        570: 0.952000,
        575: 0.915400,
        580: 0.870000,
        585: 0.816300,
        590: 0.757000,
        595: 0.694900,
        600: 0.631000,
        605: 0.566800,
        610: 0.503000,
        615: 0.441200,
        620: 0.381000,
        625: 0.321000,
        630: 0.265000,
        635: 0.217000,
        640: 0.175000,
        645: 0.138200,
        650: 0.107000,
        655: 0.081600,
        660: 0.061000,
        665: 0.044580,
        670: 0.032000,
        675: 0.023200,
        680: 0.017000,
        685: 0.011920,
        690: 0.008210,
        695: 0.005723,
        700: 0.004102,
        705: 0.002929,
        710: 0.002091,
        715: 0.001484,
        720: 0.001047,
        725: 0.000740,
        730: 0.000520,
        735: 0.000361,
        740: 0.000249,
        745: 0.000172,
        750: 0.000120,
        755: 0.000085,
        760: 0.000060,
        765: 0.000042,
        770: 0.000030,
        775: 0.000021,
        780: 0.000015},
    'z_bar': {
        380: 0.006450,
        385: 0.010550,
        390: 0.020050,
        395: 0.036210,
        400: 0.067850,
        405: 0.110200,
        410: 0.207400,
        415: 0.371300,
        420: 0.645600,
        425: 1.039050,
        430: 1.385600,
        435: 1.622960,
        440: 1.747060,
        445: 1.782600,
        450: 1.772110,
        455: 1.744100,
        460: 1.669200,
        465: 1.528100,
        470: 1.287640,
        475: 1.041900,
        480: 0.812950,
        485: 0.616200,
        490: 0.465180,
        495: 0.353300,
        500: 0.272000,
        505: 0.212300,
        510: 0.158200,
        515: 0.111700,
        520: 0.078250,
        525: 0.057250,
        530: 0.042160,
        535: 0.029840,
        540: 0.020300,
        545: 0.013400,
        550: 0.008750,
        555: 0.005750,
        560: 0.003900,
        565: 0.002750,
        570: 0.002100,
        575: 0.001800,
        580: 0.001650,
        585: 0.001400,
        590: 0.001100,
        595: 0.001000,
        600: 0.000800,
        605: 0.000600,
        610: 0.000340,
        615: 0.000240,
        620: 0.000190,
        625: 0.000100,
        630: 0.000050,
        635: 0.000030,
        640: 0.000020,
        645: 0.000010,
        650: 0.000000,
        655: 0.000000,
        660: 0.000000,
        665: 0.000000,
        670: 0.000000,
        675: 0.000000,
        680: 0.000000,
        685: 0.000000,
        690: 0.000000,
        695: 0.000000,
        700: 0.000000,
        705: 0.000000,
        710: 0.000000,
        715: 0.000000,
        720: 0.000000,
        725: 0.000000,
        730: 0.000000,
        735: 0.000000,
        740: 0.000000,
        745: 0.000000,
        750: 0.000000,
        755: 0.000000,
        760: 0.000000,
        765: 0.000000,
        770: 0.000000,
        775: 0.000000,
        780: 0.000000}}

CMFS_DATA = {
    380: (0.001368, 3.9e-05, 0.00645),
    385: (0.002236, 6.4e-05, 0.01055),
    390: (0.004243, 0.00012, 0.02005),
    395: (0.00765, 0.000217, 0.03621),
    400: (0.01431, 0.000396, 0.06785),
    405: (0.02319, 0.00064, 0.1102),
    410: (0.04351, 0.00121, 0.2074),
    415: (0.07763, 0.00218, 0.3713),
    420: (0.13438, 0.004, 0.6456),
    425: (0.21477, 0.0073, 1.03905),
    430: (0.2839, 0.0116, 1.3856),
    435: (0.3285, 0.01684, 1.62296),
    440: (0.34828, 0.023, 1.74706),
    445: (0.34806, 0.0298, 1.7826),
    450: (0.3362, 0.038, 1.77211),
    455: (0.3187, 0.048, 1.7441),
    460: (0.2908, 0.06, 1.6692),
    465: (0.2511, 0.0739, 1.5281),
    470: (0.19536, 0.09098, 1.28764),
    475: (0.1421, 0.1126, 1.0419),
    480: (0.09564, 0.13902, 0.81295),
    485: (0.05795, 0.1693, 0.6162),
    490: (0.03201, 0.20802, 0.46518),
    495: (0.0147, 0.2586, 0.3533),
    500: (0.0049, 0.323, 0.272),
    505: (0.0024, 0.4073, 0.2123),
    510: (0.0093, 0.503, 0.1582),
    515: (0.0291, 0.6082, 0.1117),
    520: (0.06327, 0.71, 0.07825),
    525: (0.1096, 0.7932, 0.05725),
    530: (0.1655, 0.862, 0.04216),
    535: (0.22575, 0.91485, 0.02984),
    540: (0.2904, 0.954, 0.0203),
    545: (0.3597, 0.9803, 0.0134),
    550: (0.43345, 0.99495, 0.00875),
    555: (0.51205, 1.0, 0.00575),
    560: (0.5945, 0.995, 0.0039),
    565: (0.6784, 0.9786, 0.00275),
    570: (0.7621, 0.952, 0.0021),
    575: (0.8425, 0.9154, 0.0018),
    580: (0.9163, 0.87, 0.00165),
    585: (0.9786, 0.8163, 0.0014),
    590: (1.0263, 0.757, 0.0011),
    595: (1.0567, 0.6949, 0.001),
    600: (1.0622, 0.631, 0.0008),
    605: (1.0456, 0.5668, 0.0006),
    610: (1.0026, 0.503, 0.00034),
    615: (0.9384, 0.4412, 0.00024),
    620: (0.85445, 0.381, 0.00019),
    625: (0.7514, 0.321, 0.0001),
    630: (0.6424, 0.265, 5e-05),
    635: (0.5419, 0.217, 3e-05),
    640: (0.4479, 0.175, 2e-05),
    645: (0.3608, 0.1382, 1e-05),
    650: (0.2835, 0.107, 0.0),
    655: (0.2187, 0.0816, 0.0),
    660: (0.1649, 0.061, 0.0),
    665: (0.1212, 0.04458, 0.0),
    670: (0.0874, 0.032, 0.0),
    675: (0.0636, 0.0232, 0.0),
    680: (0.04677, 0.017, 0.0),
    685: (0.0329, 0.01192, 0.0),
    690: (0.0227, 0.00821, 0.0),
    695: (0.01584, 0.005723, 0.0),
    700: (0.011359, 0.004102, 0.0),
    705: (0.008111, 0.002929, 0.0),
    710: (0.00579, 0.002091, 0.0),
    715: (0.004109, 0.001484, 0.0),
    720: (0.002899, 0.001047, 0.0),
    725: (0.002049, 0.00074, 0.0),
    730: (0.00144, 0.00052, 0.0),
    735: (0.001, 0.000361, 0.0),
    740: (0.00069, 0.000249, 0.0),
    745: (0.000476, 0.000172, 0.0),
    750: (0.000332, 0.00012, 0.0),
    755: (0.000235, 8.5e-05, 0.0),
    760: (0.000166, 6e-05, 0.0),
    765: (0.000117, 4.2e-05, 0.0),
    770: (8.3e-05, 3e-05, 0.0),
    775: (5.9e-05, 2.1e-05, 0.0),
    780: (4.2e-05, 1.5e-05, 0.0)}


class TestSpectralShape(unittest.TestCase):
    """
    Defines :class:`colour.colorimetry.spectrum.SpectralShape` class unit tests
    methods.
    """

    def test_required_attributes(self):
        """
        Tests presence of required attributes.
        """

        required_attributes = ('start',
                               'end',
                               'steps')

        for attribute in required_attributes:
            self.assertIn(attribute, dir(SpectralShape))

    def test_required_methods(self):
        """
        Tests presence of required methods.
        """

        required_methods = ('__repr__',
                            '__iter__',
                            '__contains__',
                            '__len__',
                            '__eq__',
                            '__ne__',
                            'range')

        for method in required_methods:
            self.assertIn(method, dir(SpectralShape))

    def test_start(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralShape.start` attribute.
        """

        shape = SpectralShape(360, 830, 1)
        self.assertEqual(shape.start, 360)
        self.assertRaises(AssertionError, lambda: SpectralShape(360, 360, 1))
        self.assertRaises(AssertionError, lambda: SpectralShape(360, 0, 1))

    def test_end(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralShape.end` attribute.
        """

        shape = SpectralShape(360, 830, 1)
        self.assertEqual(shape.end, 830)
        self.assertRaises(AssertionError, lambda: SpectralShape(830, 830, 1))
        self.assertRaises(AssertionError, lambda: SpectralShape(830, 0, 1))

    def test_steps(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralShape.steps` attribute.
        """

        shape = SpectralShape(360, 830, 1)
        self.assertEqual(shape.steps, 1)

    def test__iter__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralShape.__iter__`
        method.
        """

        np.testing.assert_almost_equal(
            [wavelength for wavelength in SpectralShape(0, 10, 0.1)],
            np.arange(0, 10 + 0.1, 0.1))

    def test__contains__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralShape.__contains__`
        method.
        """

        shape = SpectralShape(360, 830, 0.1)
        self.assertIn(360.1, shape)
        self.assertNotIn(360.11, shape)

    def test__len__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralShape.__len__`
        method.
        """

        self.assertEqual(len(SpectralShape(0, 10, 0.1)), 101)

    def test__eq__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralShape.__eq__`
        method.
        """

        self.assertEqual(SpectralShape(0, 10, 0.1), SpectralShape(0, 10, 0.1))

    def test__ne__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralShape.__ne__`
        method.
        """

        self.assertNotEqual(SpectralShape(0, 10, 0.1),
                            SpectralShape(1, 10, 0.1))

    def test_range(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralShape.range` method.
        """

        np.testing.assert_almost_equal(
            [wavelength for wavelength in SpectralShape(0, 10, 0.1)],
            np.arange(0, 10 + 0.1, 0.1))


class TestSpectralPowerDistribution(unittest.TestCase):
    """
    Defines
    :class:`colour.colorimetry.spectrum.SpectralPowerDistribution` class
    unit tests methods.
    """

    def setUp(self):
        """
        Initialises common tests attributes.
        """

        self.__spd = SpectralPowerDistribution('Sample', SAMPLE_SPD_DATA)

        self.__non_uniform_sample_spd = SpectralPowerDistribution(
            'Non Uniform Sample',
            NON_UNIFORM_SAMPLE_SPD_DATA)

        self.__phi = (1 + math.sqrt(5)) / 2

    def test_required_attributes(self):
        """
        Tests presence of required attributes.
        """

        required_attributes = ('name',
                               'data',
                               'wavelengths',
                               'values',
                               'shape')

        for attribute in required_attributes:
            self.assertIn(attribute, dir(SpectralPowerDistribution))

    def test_required_methods(self):
        """
        Tests presence of required methods.
        """

        required_methods = ('__hash__',
                            '__getitem__',
                            '__setitem__',
                            '__iter__',
                            '__contains__',
                            '__len__',
                            '__eq__',
                            '__ne__',
                            '__add__',
                            '__sub__',
                            '__mul__',
                            '__div__',
                            'get',
                            'is_uniform',
                            'extrapolate',
                            'interpolate',
                            'align',
                            'zeros',
                            'normalise',
                            'clone')

        for method in required_methods:
            self.assertIn(method, dir(SpectralPowerDistribution))

    def test_wavelengths(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralPowerDistribution.wavelengths`  # noqa
        attribute.
        """

        np.testing.assert_almost_equal(self.__spd.wavelengths,
                                       sorted(SAMPLE_SPD_DATA))

    def test_values(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralPowerDistribution.values`
        attribute.
        """

        np.testing.assert_almost_equal(
            self.__spd.values,
            [v for k, v in sorted(SAMPLE_SPD_DATA.items())])

    def test_shape(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.SpectralPowerDistribution.shape`
        attribute.
        """

        self.assertEqual(self.__spd.shape, SpectralShape(340, 820, 20))

    def test__getitem__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__getitem__`  # noqa
        method.
        """

        self.assertEqual(self.__spd[340], 0.)
        self.assertEqual(self.__spd[620], 0.1511)
        self.assertEqual(self.__spd[820], 0.)

    def test__iter__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__iter__`
        method.
        """

        self.assertEqual(
            dict([(key, value) for key, value in self.__spd]),
            SAMPLE_SPD_DATA)

    def test__contains__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__contains__`  # noqa
        method.
        """

        self.assertIn(340, self.__spd)
        self.assertIn(460, self.__spd)
        self.assertNotIn(461, self.__spd)

    def test__len__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__len__`
        method.
        """

        self.assertEqual(len(self.__spd), 25)

    def test__eq__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__eq__`
        method.
        """

        self.assertEqual(self.__spd, self.__spd.clone())

    def test__ne__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__ne__`
        method.
        """

        clone_spd = self.__spd.clone()
        clone_spd[500] = 0.

        self.assertNotEqual(self.__spd, clone_spd)

    def test__add__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__add__`
        method.
        """

        spd = self.__spd.clone()
        values = spd.values
        np.testing.assert_almost_equal(
            (spd + self.__phi).values,
            values + self.__phi)

        spd = self.__spd.clone()
        values = spd.values
        random = np.random.random(len(values))
        np.testing.assert_almost_equal((spd + random).values, values + random)

        spd1 = self.__spd.clone()
        spd2 = self.__spd.clone()
        np.testing.assert_almost_equal(
            (spd1 + spd2).values,
            self.__spd.values + self.__spd.values)

    def test__sub__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__sub__`
        method.
        """

        spd = self.__spd.clone()
        values = spd.values
        np.testing.assert_almost_equal(
            (spd - self.__phi).values,
            values - self.__phi)

        spd = self.__spd.clone()
        values = spd.values
        random = np.random.random(len(values))
        np.testing.assert_almost_equal((spd - random).values, values - random)

        spd1 = self.__spd.clone()
        spd2 = self.__spd.clone()
        np.testing.assert_almost_equal(
            (spd1 - spd2).values,
            self.__spd.values - self.__spd.values)

    def test__mul__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__mul__`
        method.
        """

        spd = self.__spd.clone()
        values = spd.values
        np.testing.assert_almost_equal((spd * 2).values, values * 2)

        spd = self.__spd.clone()
        values = spd.values
        random = np.random.random(len(values))
        np.testing.assert_almost_equal((spd * random).values, values * random)

        spd1 = self.__spd.clone()
        spd2 = self.__spd.clone()
        np.testing.assert_almost_equal(
            (spd1 * spd2).values,
            self.__spd.values * self.__spd.values)

    def test__div__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__div__`
        method.
        """

        spd = self.__spd.clone()
        values = spd.values
        np.testing.assert_almost_equal((spd / 2).values, values / 2)

        spd = self.__spd.clone()
        values = spd.values
        random = np.random.random(len(values))
        np.testing.assert_almost_equal((spd / random).values, values / random)

        # Avoiding zero division for testing purpose.
        spd = self.__spd.clone() + 0.001

        spd1 = spd.clone()
        spd2 = spd.clone()
        np.testing.assert_almost_equal(
            (spd1 / spd2).values,
            spd.values / spd.values)

    def test_get(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.get`
        method.
        """

        self.assertEqual(self.__spd.get(340), 0.)
        self.assertEqual(self.__spd.get(620), 0.1511)
        self.assertEqual(self.__spd.get(820), 0.)
        self.assertEqual(self.__spd.get(900, -1), -1)

    def test_is_uniform(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.is_uniform`  # noqa
        method.
        """

        self.assertFalse(self.__non_uniform_sample_spd.is_uniform())
        self.assertTrue(self.__spd.is_uniform())

    def test_extrapolate(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.extrapolate`  # noqa
        method.
        """

        spd = SpectralPowerDistribution(
            '',
            dict(zip(range(25, 35), [0] * 5 + [1] * 5)))
        spd.extrapolate(SpectralShape(10, 50))

        self.assertEqual(spd[10], 0)
        self.assertEqual(spd[50], 1)

    def test_interpolate(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.interpolate`  # noqa
        method.
        """

        np.testing.assert_almost_equal(
            self.__spd.clone().interpolate(
                SpectralShape(steps=1)).values,
            INTERPOLATED_SAMPLE_SPD_DATA,
            decimal=7)

        np.testing.assert_almost_equal(
            self.__non_uniform_sample_spd.clone().interpolate(
                SpectralShape(steps=1)).values,
            INTERPOLATED_NON_UNIFORM_SAMPLE_SPD_DATA,
            # Lower precision for Linux *Travis-ci* tests.
            decimal=0)

    def test_align(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.align`
        method.
        """

        shape = SpectralShape(100, 900, 5)
        self.assertEqual(self.__spd.clone().align(shape).shape, shape)
        shape = SpectralShape(600, 650, 1)
        self.assertEqual(self.__spd.clone().align(shape).shape, shape)

    def test_zeros(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.zeros`
        method.
        """

        np.testing.assert_almost_equal(
            self.__spd.clone().zeros(SpectralShape(steps=1)).values,
            ZEROS_SAMPLE_SPD_DATA)

    def test_normalise(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.normalise`
        method.
        """

        np.testing.assert_almost_equal(
            self.__spd.clone().normalise(100).values,
            NORMALISED_SAMPLE_SPD_DATA)

    def test_clone(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.clone`
        method.
        """

        self.assertFalse(self.__spd is self.__spd.clone())


class TestTriSpectralPowerDistribution(unittest.TestCase):
    """
    Defines :class:`colour.colorimetry.spectrum.TriSpectralPowerDistribution`
    class unit tests methods.
    """

    def setUp(self):
        """
        Initialises common tests attributes.
        """

        self.__mapping = {'x': 'x_bar',
                          'y': 'y_bar',
                          'z': 'z_bar'}

        self.__labels = {'x': 'x_bar',
                         'y': 'y_bar',
                         'z': 'z_bar'}

        self.__tri_spd = TriSpectralPowerDistribution(
            name='Tri Spd',
            data=CIE_1931_2_DEGREE_STANDARD_OBSERVER,
            mapping=self.__mapping,
            labels=self.__labels)

        self.__sample_tri_spd = TriSpectralPowerDistribution(
            name='Sample Tri Spd',
            data={'x_bar': SAMPLE_SPD_DATA,
                  'y_bar': SAMPLE_SPD_DATA,
                  'z_bar': SAMPLE_SPD_DATA},
            mapping=self.__mapping,
            labels=self.__labels)

        self.__non_uniform_sample_tri_spd = TriSpectralPowerDistribution(
            name='Non Uniform Sample Tri spd',
            data={'x_bar': NON_UNIFORM_SAMPLE_SPD_DATA,
                  'y_bar': NON_UNIFORM_SAMPLE_SPD_DATA,
                  'z_bar': NON_UNIFORM_SAMPLE_SPD_DATA},
            mapping=self.__mapping,
            labels=self.__labels)

        self.__phi = (1 + math.sqrt(5)) / 2

    def test_required_attributes(self):
        """
        Tests presence of required attributes.
        """

        required_attributes = ('name',
                               'mapping',
                               'labels',
                               'data',
                               'x',
                               'y',
                               'z',
                               'wavelengths',
                               'values',
                               'shape')

        for attribute in required_attributes:
            self.assertIn(attribute, dir(TriSpectralPowerDistribution))

    def test_required_methods(self):
        """
        Tests presence of required methods.
        """

        required_methods = ('__hash__',
                            '__getitem__',
                            '__setitem__',
                            '__iter__',
                            '__contains__',
                            '__len__',
                            '__eq__',
                            '__ne__',
                            '__add__',
                            '__sub__',
                            '__mul__',
                            '__div__',
                            'get',
                            'is_uniform',
                            'extrapolate',
                            'interpolate',
                            'align',
                            'zeros',
                            'normalise',
                            'clone')

        for method in required_methods:
            self.assertIn(method, dir(TriSpectralPowerDistribution))

    def test_wavelengths(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.wavelengths`  # noqa
        attribute.
        """

        np.testing.assert_almost_equal(
            self.__tri_spd.wavelengths,
            sorted(CIE_1931_2_DEGREE_STANDARD_OBSERVER.get('x_bar')))

    def test_values(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.values`
        attribute.
        """

        np.testing.assert_almost_equal(
            self.__tri_spd.values,
            np.array(tuple(zip(*(
                [v for k, v in sorted(CIE_1931_2_DEGREE_STANDARD_OBSERVER.get(
                    'x_bar').items())],
                [v for k, v in sorted(CIE_1931_2_DEGREE_STANDARD_OBSERVER.get(
                    'y_bar').items())],
                [v for k, v in sorted(CIE_1931_2_DEGREE_STANDARD_OBSERVER.get(
                    'z_bar').items())])))))

    def test_shape(self):
        """
        Tests
        :attr:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.shape`
        attribute.
        """

        self.assertEqual(self.__tri_spd.shape, SpectralShape(380, 780, 5))

    def test__getitem__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__getitem__`  # noqa
        method.
        """

        np.testing.assert_almost_equal(
            self.__tri_spd[380],
            np.array((0.001368, 3.9e-05, 0.00645)))
        np.testing.assert_almost_equal(
            self.__tri_spd[600],
            np.array((1.0622, 0.631, 0.0008)))
        np.testing.assert_almost_equal(
            self.__tri_spd[700],
            np.array((0.011359, 0.004102, 0.)))

    def test__iter__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__iter__`  # noqa
        method.
        """

        self.assertEqual(
            dict([(key, tuple(value)) for key, value in self.__tri_spd]),
            CMFS_DATA)

    def test__contains__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__contains__`  # noqa
        method.
        """

        self.assertIn(380, self.__tri_spd)
        self.assertIn(460, self.__tri_spd)
        self.assertNotIn(461, self.__tri_spd)

    def test__len__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__len__`  # noqa
        method.
        """

        self.assertEqual(len(self.__tri_spd), 81)

    def test__eq__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__eq__`
        method.
        """

        clone_tri_spd = self.__tri_spd.clone()

        self.assertEqual(self.__tri_spd, clone_tri_spd)

    def test__ne__(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.__ne__`
        method.
        """

        clone_tri_spd = self.__tri_spd.clone()
        clone_tri_spd[500] = (0, 0, 0)

        self.assertNotEqual(self.__tri_spd, clone_tri_spd)

    def test__add__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__add__`  # noqa
        method.
        """

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        np.testing.assert_almost_equal(
            (tri_spd + self.__phi).values,
            values + self.__phi)

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        random = np.random.random(values.shape)
        np.testing.assert_almost_equal(
            (tri_spd + random).values,
            values + random)

        tri_spd1 = self.__tri_spd.clone()
        tri_spd2 = self.__tri_spd.clone()
        np.testing.assert_almost_equal(
            (tri_spd1 + tri_spd2).values,
            self.__tri_spd.values + self.__tri_spd.values)

    def test__sub__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__sub__`  # noqa
        method.
        """

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        np.testing.assert_almost_equal(
            (tri_spd - self.__phi).values,
            values - self.__phi)

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        random = np.random.random(values.shape)
        np.testing.assert_almost_equal(
            (tri_spd - random).values,
            values - random)

        tri_spd1 = self.__tri_spd.clone()
        tri_spd2 = self.__tri_spd.clone()
        np.testing.assert_almost_equal(
            (tri_spd1 - tri_spd2).values,
            self.__tri_spd.values - self.__tri_spd.values)

    def test__mul__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__mul__`  # noqa
        method.
        """

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        np.testing.assert_almost_equal(
            (tri_spd * self.__phi).values,
            values * self.__phi)

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        random = np.random.random(values.shape)
        np.testing.assert_almost_equal(
            (tri_spd * random).values,
            values * random)

        tri_spd1 = self.__tri_spd.clone()
        tri_spd2 = self.__tri_spd.clone()
        np.testing.assert_almost_equal(
            (tri_spd1 * tri_spd2).values,
            self.__tri_spd.values * self.__tri_spd.values)

    def test__div__(self):
        """
        Tests :func:`colour.colorimetry.spectrum.SpectralPowerDistribution.__div__`  # noqa
        method.
        """

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        np.testing.assert_almost_equal(
            (tri_spd / self.__phi).values,
            values / self.__phi)

        tri_spd = self.__tri_spd.clone()
        values = tri_spd.values
        random = np.random.random(values.shape)
        np.testing.assert_almost_equal(
            (tri_spd / random).values,
            values / random)

        # Avoiding zero division for testing purpose.
        tri_spd = self.__tri_spd.clone() + 0.001

        tri_spd1 = tri_spd.clone()
        tri_spd2 = tri_spd.clone()
        np.testing.assert_almost_equal(
            (tri_spd1 / tri_spd2).values,
            tri_spd.values / tri_spd.values)

    def test_get(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.get`
        method.
        """

        np.testing.assert_almost_equal(
            self.__tri_spd.get(380),
            np.array((0.001368, 3.9e-05, 0.00645)))
        np.testing.assert_almost_equal(
            self.__tri_spd.get(600),
            np.array((1.0622, 0.631, 0.0008)))
        np.testing.assert_almost_equal(
            self.__tri_spd.get(700),
            np.array((0.011359, 0.004102, 0.)))
        np.testing.assert_almost_equal(
            self.__tri_spd.get(900, (0, 0, 0)),
            np.array((0, 0, 0)))

    def test_is_uniform(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.is_uniform`  # noqa
        method.
        """

        self.assertTrue(self.__tri_spd.is_uniform())

    def test_extrapolate(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.extrapolate`  # noqa
        method.
        """

        spd_data = dict(zip(range(25, 35), [0] * 5 + [1] * 5))
        tri_spd = TriSpectralPowerDistribution(
            name='',
            mapping=self.__mapping,
            data={'x_bar': spd_data,
                  'y_bar': spd_data,
                  'z_bar': spd_data},
            labels=self.__labels)

        tri_spd.extrapolate(SpectralShape(10, 50))

        self.assertEqual(tri_spd.x[10], 0)
        self.assertEqual(tri_spd.y[10], 0)
        self.assertEqual(tri_spd.z[10], 0)

        self.assertEqual(tri_spd.x[50], 1)
        self.assertEqual(tri_spd.y[50], 1)
        self.assertEqual(tri_spd.z[50], 1)

    def test_interpolate(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.interpolate`  # noqa
        method.
        """

        tri_spd = self.__sample_tri_spd.clone()

        tri_spd.interpolate(SpectralShape(steps=1))
        for i in sorted(self.__mapping.keys()):
            np.testing.assert_almost_equal(
                getattr(tri_spd, i).values,
                INTERPOLATED_SAMPLE_SPD_DATA,
                decimal=7)

        tri_spd = self.__non_uniform_sample_tri_spd.clone()

        tri_spd.interpolate(SpectralShape(steps=1))
        for i in sorted(self.__mapping.keys()):
            np.testing.assert_almost_equal(
                getattr(tri_spd, i).values,
                INTERPOLATED_NON_UNIFORM_SAMPLE_SPD_DATA,
                # Lower precision for Linux *Travis-ci* tests.
                decimal=0)

    def test_align(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.align`
        method.
        """

        tri_spd = self.__sample_tri_spd.clone()

        shape = SpectralShape(100, 900, 5)
        self.assertEqual(tri_spd.align(shape).shape, shape)
        shape = SpectralShape(600, 650, 1)
        self.assertEqual(tri_spd.align(shape).shape, shape)

    def test_zeros(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.zeros`
        method.
        """

        tri_spd = TriSpectralPowerDistribution(
            name='',
            mapping=self.__mapping,
            data={'x_bar': SAMPLE_SPD_DATA,
                  'y_bar': SAMPLE_SPD_DATA,
                  'z_bar': SAMPLE_SPD_DATA},
            labels=self.__labels).clone()

        tri_spd.zeros(SpectralShape(steps=1))
        for i in self.__mapping.keys():
            np.testing.assert_almost_equal(
                getattr(tri_spd, i).values,
                ZEROS_SAMPLE_SPD_DATA)

    def test_normalise(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.normalise`  # noqa
        method.
        """

        np.testing.assert_almost_equal(
            self.__sample_tri_spd.clone().normalise(100).values,
            np.array([[x] * 3 for x in NORMALISED_SAMPLE_SPD_DATA]))

    def test_clone(self):
        """
        Tests
        :func:`colour.colorimetry.spectrum.TriSpectralPowerDistribution.clone`
        method.
        """

        self.assertFalse(self.__tri_spd is self.__tri_spd.clone())


class TestConstantSpd(unittest.TestCase):
    """
    Defines :func:`colour.colorimetry.spectrum.constant_spd` definition unit
    tests methods.
    """

    def test_constant_spd(self):
        """
        Tests :func:`colour.colorimetry.spectrum.constant_spd`
        definition.
        """

        k = 3.1415

        spd = constant_spd(k)
        self.assertEqual(spd[360], k)
        self.assertEqual(spd[555], k)
        self.assertEqual(spd[830], k)


class TestZerosSpd(unittest.TestCase):
    """
    Defines :func:`colour.colorimetry.spectrum.zeros_spd` definition unit
    tests methods.
    """

    def test_zeros_spd(self):
        """
        Tests :func:`colour.colorimetry.spectrum.zeros_spd`
        definition.
        """

        spd = zeros_spd()
        self.assertEqual(spd[360], 0.)
        self.assertEqual(spd[555], 0.)
        self.assertEqual(spd[830], 0.)


class TestOnes_spd(unittest.TestCase):
    """
    Defines :func:`colour.colorimetry.spectrum.ones_spd` definition unit
    tests methods.
    """

    def test_ones_spd(self):
        """
        Tests :func:`colour.colorimetry.spectrum.ones_spd`
        definition.
        """

        spd = ones_spd()
        self.assertEqual(spd[360], 1.)
        self.assertEqual(spd[555], 1.)
        self.assertEqual(spd[830], 1.)


if __name__ == '__main__':
    unittest.main()
