#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Light Sources Relative Spectral Power Distributions
===================================================

Defines various light sources relative spectral power distributions.

The light sources data is in the form of a *dict* of
:class:`colour.colorimetry.spectrum.SpectralPowerDistribution` classes as
follows::

    {'name': SpectralPowerDistribution, ..., 'name': SpectralPowerDistribution}

The following light sources are available:

-   *RIT* *PointerData.xls* spreadsheet light sources: Natural,
    Philips TL-84, T8 Luxline Plus White, SA, SC, T8 Polylux 3000,
    T8 Polylux 4000, Thorn Kolor-rite
-   *NIST* *NIST CQS simulation 7.4.xls* spreadsheet traditional light sources:
    Cool White FL, Daylight FL, HPS, Incandescent, LPS, Mercury,
    Metal Halide, Neodimium Incandescent, Super HPS, Triphosphor FL
-   *NIST* *NIST CQS simulation 7.4.xls* spreadsheet LED light sources:
    3-LED-1 (457/540/605), 3-LED-2 (473/545/616), 3-LED-2 Yellow,
    3-LED-3 (465/546/614), 3-LED-4 (455/547/623), 4-LED No Yellow,
    4-LED Yellow, 4-LED-1 (461/526/576/624), 4-LED-2 (447/512/573/627),
    Luxeon WW 2880, PHOS-1, PHOS-2, PHOS-3, PHOS-4,
    Phosphor LED YAG
-   *NIST* *NIST CQS simulation 7.4.xls* spreadsheet Philips light sources:
    60 A/W (Soft White), C100S54 (HPS), C100S54C (HPS),
    F32T8/TL830 (Triphosphor), F32T8/TL835 (Triphosphor),
    F32T8/TL841 (Triphosphor), F32T8/TL850 (Triphosphor),
    F32T8/TL865 /PLUS (Triphosphor), F34/CW/RS/EW (Cool White FL),
    F34T12/LW/RS /EW, F34T12WW/RS /EW (Warm White FL),
    F40/C50 (Broadband FL), F40/C75 (Broadband FL),
    F40/CWX (Broadband FL), F40/DX (Broadband FL), F40/DXTP (Delux FL),
    F40/N (Natural FL), H38HT-100 (Mercury), H38JA-100/DX (Mercury DX),
    MHC100/U/MP /3K, MHC100/U/MP /4K, SDW-T 100W/LV (Super HPS)

See Also
--------
`Light Sources IPython Notebook
<http://nbviewer.ipython.org/github/colour-science/colour-ipython/blob/master/notebooks/colorimetry/light_sources.ipynb>`_  # noqa
"""

from __future__ import division, unicode_literals

from colour.colorimetry.spectrum import SpectralPowerDistribution
from colour.utilities import CaseInsensitiveMapping

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013 - 2014 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA',
           'LIGHT_SOURCES_RIT_RELATIVE_SPDS',
           'LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA',
           'LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS',
           'LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA',
           'LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS',
           'LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA',
           'LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS',
           'LIGHT_SOURCES_RELATIVE_SPDS']

LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA = {
    'Natural': {
        380: 1.88,
        385: 2.24,
        390: 2.65,
        395: 2.95,
        400: 4.06,
        405: 20.6,
        410: 4.66,
        415: 4.23,
        420: 4.54,
        425: 4.87,
        430: 5.36,
        435: 43.4,
        440: 11.07,
        445: 6.21,
        450: 6.52,
        455: 6.82,
        460: 7.09,
        465: 7.32,
        470: 7.5,
        475: 7.66,
        480: 7.76,
        485: 7.83,
        490: 7.98,
        495: 7.96,
        500: 7.94,
        505: 7.97,
        510: 8.01,
        515: 8.09,
        520: 8.23,
        525: 8.48,
        530: 8.84,
        535: 9.38,
        540: 11.16,
        545: 24.5,
        550: 18.11,
        555: 12.62,
        560: 13.46,
        565: 14.3,
        570: 15.22,
        575: 18.15,
        580: 19.54,
        585: 17.02,
        590: 16.74,
        595: 16.78,
        600: 16.71,
        605: 16.52,
        610: 16.23,
        615: 15.88,
        620: 15.46,
        625: 15.03,
        630: 14.5,
        635: 13.94,
        640: 13.32,
        645: 12.7,
        650: 12.09,
        655: 11.51,
        660: 10.88,
        665: 10.28,
        670: 9.65,
        675: 9.01,
        680: 8.33,
        685: 7.74,
        690: 7.17,
        695: 6.56,
        700: 5.99,
        705: 5.52,
        710: 5.03,
        715: 4.61,
        720: 4.2,
        725: 3.81,
        730: 3.46,
        735: 3.15,
        740: 2.82,
        745: 2.53,
        750: 2.28,
        755: 2.04,
        760: 1.84,
        765: 1.65,
        770: 1.47,
        775: 1.3,
        780: 1.2},
    'Philips TL-84': {
        380: 0.784,
        385: 0.553,
        390: 0.416,
        395: 0.363,
        400: 0.88,
        405: 10.82,
        410: 1.554,
        415: 1.8,
        420: 2.51,
        425: 3.5,
        430: 4.76,
        435: 31.7,
        440: 10.24,
        445: 7.34,
        450: 7.63,
        455: 7.58,
        460: 7.17,
        465: 6.56,
        470: 5.89,
        475: 5.24,
        480: 6.35,
        485: 14.53,
        490: 15.07,
        495: 9.5,
        500: 5.23,
        505: 2.72,
        510: 1.716,
        515: 1.303,
        520: 1.07,
        525: 1.012,
        530: 1.395,
        535: 6.19,
        540: 39.3,
        545: 68.8,
        550: 34.9,
        555: 8.89,
        560: 3.24,
        565: 2.23,
        570: 2.01,
        575: 4.5,
        580: 10.48,
        585: 14.27,
        590: 12.96,
        595: 10.08,
        600: 7.51,
        605: 13.86,
        610: 46.3,
        615: 41.7,
        620: 16.49,
        625: 13.14,
        630: 11.42,
        635: 5.72,
        640: 2.35,
        645: 2.55,
        650: 3.34,
        655: 3.09,
        660: 2.49,
        665: 2.15,
        670: 1.67,
        675: 1.39,
        680: 1.49,
        685: 1.82,
        690: 1.81,
        695: 1.3,
        700: 1.63,
        705: 3.55,
        710: 4.68,
        715: 3.11,
        720: 1.06,
        725: 0.32,
        730: 0.24,
        735: 0.22,
        740: 0.22,
        745: 0.23,
        750: 0.19,
        755: 0.14,
        760: 0.1,
        765: 0.1,
        770: 0.11,
        775: 0.11,
        780: 0.1},
    'SA': {
        380: 9.8,
        385: 10.9,
        390: 12.09,
        395: 13.35,
        400: 14.71,
        405: 16.15,
        410: 17.68,
        415: 19.29,
        420: 20.99,
        425: 22.79,
        430: 24.67,
        435: 26.64,
        440: 28.7,
        445: 30.85,
        450: 33.09,
        455: 35.41,
        460: 37.81,
        465: 40.3,
        470: 42.87,
        475: 45.52,
        480: 48.24,
        485: 51.04,
        490: 53.91,
        495: 56.85,
        500: 59.86,
        505: 62.93,
        510: 66.06,
        515: 69.25,
        520: 72.5,
        525: 75.79,
        530: 79.13,
        535: 82.52,
        540: 85.95,
        545: 89.41,
        550: 92.91,
        555: 96.44,
        560: 100.0,
        565: 103.58,
        570: 107.18,
        575: 110.8,
        580: 114.44,
        585: 118.08,
        590: 121.73,
        595: 125.39,
        600: 129.04,
        605: 132.7,
        610: 136.35,
        615: 139.99,
        620: 143.62,
        625: 147.24,
        630: 150.84,
        635: 154.42,
        640: 157.98,
        645: 161.52,
        650: 165.03,
        655: 168.51,
        660: 171.96,
        665: 175.38,
        670: 178.77,
        675: 182.12,
        680: 185.43,
        685: 188.7,
        690: 191.93,
        695: 195.12,
        700: 198.26,
        705: 201.36,
        710: 204.41,
        715: 207.41,
        720: 210.36,
        725: 213.27,
        730: 216.12,
        735: 218.92,
        740: 221.67,
        745: 224.36,
        750: 227.0,
        755: 229.59,
        760: 232.12,
        765: 234.59,
        770: 237.01,
        775: 239.37,
        780: 241.68},
    'SC': {
        380: 33.0,
        385: 39.92,
        390: 47.4,
        395: 55.17,
        400: 63.3,
        405: 71.81,
        410: 80.6,
        415: 89.53,
        420: 98.1,
        425: 105.8,
        430: 112.4,
        435: 117.75,
        440: 121.5,
        445: 123.45,
        450: 124.0,
        455: 123.6,
        460: 123.1,
        465: 123.3,
        470: 123.8,
        475: 124.09,
        480: 123.9,
        485: 122.92,
        490: 120.7,
        495: 116.9,
        500: 112.1,
        505: 106.98,
        510: 102.3,
        515: 98.81,
        520: 96.9,
        525: 96.78,
        530: 98.0,
        535: 99.94,
        540: 102.1,
        545: 103.95,
        550: 105.2,
        555: 105.67,
        560: 105.3,
        565: 104.11,
        570: 102.3,
        575: 100.15,
        580: 97.8,
        585: 95.43,
        590: 93.2,
        595: 91.22,
        600: 89.7,
        605: 88.83,
        610: 88.4,
        615: 88.19,
        620: 88.1,
        625: 88.06,
        630: 88.0,
        635: 87.86,
        640: 87.8,
        645: 87.99,
        650: 88.2,
        655: 88.2,
        660: 87.9,
        665: 87.22,
        670: 86.3,
        675: 85.3,
        680: 84.0,
        685: 82.21,
        690: 80.2,
        695: 78.24,
        700: 76.3,
        705: 74.36,
        710: 72.4,
        715: 70.4,
        720: 68.3,
        725: 66.3,
        730: 64.4,
        735: 62.8,
        740: 61.5,
        745: 60.2,
        750: 59.2,
        755: 58.5,
        760: 58.1,
        765: 58.0,
        770: 58.2,
        775: 58.5,
        780: 59.1},
    'T8 Luxline Plus White': {
        380: 0.833,
        385: 1.04,
        390: 1.296,
        395: 1.511,
        400: 2.32,
        405: 13.56,
        410: 2.85,
        415: 2.65,
        420: 2.95,
        425: 3.26,
        430: 3.69,
        435: 30.9,
        440: 7.52,
        445: 4.46,
        450: 4.73,
        455: 4.96,
        460: 5.15,
        465: 5.34,
        470: 5.49,
        475: 5.57,
        480: 5.59,
        485: 5.6,
        490: 5.71,
        495: 5.73,
        500: 5.81,
        505: 5.98,
        510: 6.34,
        515: 6.82,
        520: 7.35,
        525: 7.9,
        530: 8.58,
        535: 9.41,
        540: 11.14,
        545: 21.6,
        550: 16.75,
        555: 13.47,
        560: 14.5,
        565: 15.51,
        570: 16.3,
        575: 18.58,
        580: 19.93,
        585: 18.37,
        590: 18.21,
        595: 17.54,
        600: 16.14,
        605: 18.29,
        610: 31.3,
        615: 27.1,
        620: 13.64,
        625: 11.14,
        630: 10.77,
        635: 8.2,
        640: 6.03,
        645: 5.33,
        650: 5.01,
        655: 4.38,
        660: 3.76,
        665: 3.25,
        670: 2.68,
        675: 2.23,
        680: 2.01,
        685: 2.03,
        690: 1.97,
        695: 1.68,
        700: 1.76,
        705: 2.53,
        710: 2.9,
        715: 2.08,
        720: 1.11,
        725: 0.74,
        730: 0.66,
        735: 0.6,
        740: 0.55,
        745: 0.52,
        750: 0.47,
        755: 0.42,
        760: 0.39,
        765: 0.36,
        770: 0.32,
        775: 0.3,
        780: 0.3},
    'T8 Polylux 3000': {
        380: 0.552,
        385: 0.381,
        390: 0.289,
        395: 0.231,
        400: 2.424,
        405: 11.475,
        410: 0.976,
        415: 0.511,
        420: 0.512,
        425: 0.602,
        430: 0.807,
        435: 23.309,
        440: 2.105,
        445: 1.1,
        450: 1.145,
        455: 1.159,
        460: 1.111,
        465: 1.083,
        470: 1.028,
        475: 1.0,
        480: 2.003,
        485: 6.847,
        490: 6.437,
        495: 3.446,
        500: 1.673,
        505: 0.838,
        510: 0.575,
        515: 0.48,
        520: 0.463,
        525: 0.541,
        530: 0.919,
        535: 3.46,
        540: 23.729,
        545: 43.7,
        550: 15.651,
        555: 3.975,
        560: 2.458,
        565: 2.276,
        570: 2.264,
        575: 5.44,
        580: 9.046,
        585: 10.028,
        590: 8.862,
        595: 7.267,
        600: 6.146,
        605: 5.466,
        610: 47.27,
        615: 23.714,
        620: 7.706,
        625: 8.606,
        630: 9.498,
        635: 2.824,
        640: 1.752,
        645: 1.715,
        650: 3.009,
        655: 1.886,
        660: 1.771,
        665: 1.536,
        670: 1.022,
        675: 0.846,
        680: 0.91,
        685: 1.431,
        690: 1.31,
        695: 0.805,
        700: 0.486,
        705: 3.212,
        710: 4.339,
        715: 0.993,
        720: 0.207,
        725: 0.183,
        730: 0.145,
        735: 0.113,
        740: 0.181,
        745: 0.161,
        750: 0.104,
        755: 0.13,
        760: 0.219,
        765: 0.064,
        770: 0.093,
        775: 0.051,
        780: 0.027},
    'T8 Polylux 4000': {
        380: 0.498,
        385: 0.384,
        390: 0.394,
        395: 0.267,
        400: 1.244,
        405: 9.703,
        410: 1.102,
        415: 0.99,
        420: 1.263,
        425: 1.681,
        430: 2.322,
        435: 21.723,
        440: 5.96,
        445: 3.618,
        450: 3.831,
        455: 3.815,
        460: 3.714,
        465: 3.459,
        470: 3.162,
        475: 2.873,
        480: 3.291,
        485: 8.191,
        490: 8.611,
        495: 5.122,
        500: 2.854,
        505: 1.573,
        510: 1.111,
        515: 0.916,
        520: 0.811,
        525: 0.823,
        530: 1.096,
        535: 3.291,
        540: 23.51,
        545: 45.698,
        550: 15.958,
        555: 4.176,
        560: 2.21,
        565: 1.818,
        570: 1.689,
        575: 4.109,
        580: 7.743,
        585: 9.196,
        590: 7.473,
        595: 5.804,
        600: 4.814,
        605: 3.62,
        610: 32.462,
        615: 23.618,
        620: 6.969,
        625: 7.643,
        630: 7.51,
        635: 2.564,
        640: 1.345,
        645: 1.347,
        650: 2.352,
        655: 1.681,
        660: 1.441,
        665: 1.336,
        670: 0.91,
        675: 0.786,
        680: 0.858,
        685: 1.216,
        690: 1.037,
        695: 0.669,
        700: 0.351,
        705: 2.372,
        710: 3.317,
        715: 0.886,
        720: 0.173,
        725: 0.159,
        730: 0.132,
        735: 0.113,
        740: 0.151,
        745: 0.145,
        750: 0.086,
        755: 0.093,
        760: 0.319,
        765: 0.064,
        770: 0.093,
        775: 0.051,
        780: 0.055},
    'Thorn Kolor-rite': {
        380: 1.111,
        385: 1.316,
        390: 1.565,
        395: 1.714,
        400: 2.64,
        405: 17.54,
        410: 2.96,
        415: 2.51,
        420: 2.75,
        425: 3.06,
        430: 3.61,
        435: 41.6,
        440: 8.95,
        445: 4.8,
        450: 5.3,
        455: 5.8,
        460: 6.24,
        465: 6.68,
        470: 7.09,
        475: 7.49,
        480: 7.82,
        485: 8.12,
        490: 8.54,
        495: 8.8,
        500: 9.08,
        505: 9.46,
        510: 9.86,
        515: 10.33,
        520: 10.73,
        525: 11.07,
        530: 11.29,
        535: 11.43,
        540: 12.93,
        545: 26.3,
        550: 18.11,
        555: 12.17,
        560: 12.37,
        565: 12.67,
        570: 13.12,
        575: 15.82,
        580: 16.88,
        585: 14.25,
        590: 14.05,
        595: 14.23,
        600: 14.37,
        605: 14.53,
        610: 14.72,
        615: 14.96,
        620: 15.63,
        625: 16.45,
        630: 16.78,
        635: 16.12,
        640: 15.44,
        645: 16.09,
        650: 18.15,
        655: 20.4,
        660: 20.4,
        665: 17.3,
        670: 13.31,
        675: 10.73,
        680: 9.45,
        685: 8.62,
        690: 7.81,
        695: 7.05,
        700: 6.38,
        705: 5.82,
        710: 5.3,
        715: 4.81,
        720: 4.41,
        725: 4.0,
        730: 3.61,
        735: 3.26,
        740: 2.95,
        745: 2.63,
        750: 2.34,
        755: 2.11,
        760: 1.89,
        765: 1.71,
        770: 1.5,
        775: 1.33,
        780: 1.2}}

LIGHT_SOURCES_RIT_RELATIVE_SPDS = CaseInsensitiveMapping(
    {'Natural': SpectralPowerDistribution(
        'Natural',
        LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
            'Natural')),
     'Philips TL-84': SpectralPowerDistribution(
         'Philips TL-84',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'Philips TL-84')),
     'SA': SpectralPowerDistribution(
         'SA',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'SA')),
     'SC': SpectralPowerDistribution(
         'SC',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'SC')),
     'T8 Luxline Plus White': SpectralPowerDistribution(
         'T8 Luxline Plus White',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'T8 Luxline Plus White')),
     'T8 Polylux 3000': SpectralPowerDistribution(
         'T8 Polylux 3000',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'T8 Polylux 3000')),
     'T8 Polylux 4000': SpectralPowerDistribution(
         'T8 Polylux 4000',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'T8 Polylux 4000')),
     'Thorn Kolor-rite': SpectralPowerDistribution(
         'Thorn Kolor-rite',
         LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA.get(
             'Thorn Kolor-rite'))})
"""
Light sources from *RIT* *PointerData.xls* spreadsheet.

Warning
-------
The relative spectral power distributions have been extracted from
*PointerData.xls* spreadsheet that doesn't mention the data source thus the
light source names cannot be accurately verified.

References
----------
.. [1]  Pointer, M. R. (1980). Pointer’s Gamut Data. Retrieved from
        http://www.cis.rit.edu/research/mcsl2/online/PointerData.xls

LIGHT_SOURCES_RIT_RELATIVE_SPDS_DATA : CaseInsensitiveMapping
    {'Natural', 'Philips TL-84', 'T8 Luxline Plus White', 'SA', 'SC',
    'T8 Polylux 3000', 'T8 Polylux 4000', 'Thorn Kolor-rite'}
"""

LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA = {
    'Cool White FL': {
        380: 0.03353465,
        385: 0.04082136,
        390: 0.04995733,
        395: 0.07760631,
        400: 0.16214621,
        405: 0.16590101,
        410: 0.13939696,
        415: 0.09848528,
        420: 0.11529654,
        425: 0.15309344,
        430: 0.36138529,
        435: 1.0,
        440: 0.47411464,
        445: 0.14508605,
        450: 0.1810411,
        455: 0.19408335,
        460: 0.20570331,
        465: 0.21481297,
        470: 0.21981937,
        475: 0.21775708,
        480: 0.2178353,
        485: 0.22045228,
        490: 0.2246551,
        495: 0.22158299,
        500: 0.21617124,
        505: 0.21068127,
        510: 0.21048215,
        515: 0.21334803,
        520: 0.22029583,
        525: 0.23525103,
        530: 0.25610155,
        535: 0.28776845,
        540: 0.33889916,
        545: 0.58557104,
        550: 0.56871711,
        555: 0.46039681,
        560: 0.51555255,
        565: 0.55145783,
        570: 0.58860048,
        575: 0.62758498,
        580: 0.63406343,
        585: 0.61005547,
        590: 0.57386574,
        595: 0.54071256,
        600: 0.49833594,
        605: 0.45226141,
        610: 0.40404637,
        615: 0.35870431,
        620: 0.31634903,
        625: 0.27832456,
        630: 0.24633765,
        635: 0.21325558,
        640: 0.18414166,
        645: 0.15840563,
        650: 0.13439056,
        655: 0.11421562,
        660: 0.09677144,
        665: 0.08196558,
        670: 0.07098848,
        675: 0.06217537,
        680: 0.05450292,
        685: 0.04630778,
        690: 0.04167046,
        695: 0.03630351,
        700: 0.02957687,
        705: 0.02720381,
        710: 0.02373489,
        715: 0.02041104,
        720: 0.01877542,
        725: 0.01705945,
        730: 0.0156962,
        735: 0.01469208,
        740: 0.0137228,
        745: 0.0129249,
        750: 0.01240649,
        755: 0.01231262,
        760: 0.01899801,
        765: 0.01817522,
        770: 0.01183473,
        775: 0.00953278,
        780: 0.00920353},
    'Daylight FL': {
        380: 0.04474924,
        385: 0.05659396,
        390: 0.07026586,
        395: 0.10049547,
        400: 0.18520846,
        405: 0.19377039,
        410: 0.173571,
        415: 0.13960725,
        420: 0.16195166,
        425: 0.20236858,
        430: 0.40082175,
        435: 1.0,
        440: 0.518429,
        445: 0.21892447,
        450: 0.25743807,
        455: 0.27389124,
        460: 0.28964955,
        465: 0.30247734,
        470: 0.30974018,
        475: 0.30807855,
        480: 0.30855589,
        485: 0.3118429,
        490: 0.317571,
        495: 0.3127432,
        500: 0.30381873,
        505: 0.293571,
        510: 0.28662236,
        515: 0.28228399,
        520: 0.27977039,
        525: 0.27754079,
        530: 0.27734743,
        535: 0.28374622,
        540: 0.30420544,
        545: 0.49527492,
        550: 0.45265861,
        555: 0.33021148,
        560: 0.35650151,
        565: 0.37024773,
        570: 0.39148036,
        575: 0.42088822,
        580: 0.4238006,
        585: 0.40008459,
        590: 0.36932326,
        595: 0.34946224,
        600: 0.32535347,
        605: 0.29843505,
        610: 0.2696858,
        615: 0.24326888,
        620: 0.21839879,
        625: 0.19558308,
        630: 0.17557704,
        635: 0.15563746,
        640: 0.13744411,
        645: 0.12087613,
        650: 0.10580665,
        655: 0.09257402,
        660: 0.08107553,
        665: 0.07144411,
        670: 0.06311782,
        675: 0.05713716,
        680: 0.0522719,
        685: 0.04643867,
        690: 0.04323746,
        695: 0.03866042,
        700: 0.03217402,
        705: 0.03034743,
        710: 0.02783807,
        715: 0.02529486,
        720: 0.02349486,
        725: 0.02208278,
        730: 0.02074139,
        735: 0.01944653,
        740: 0.01831541,
        745: 0.01752568,
        750: 0.01690332,
        755: 0.01635831,
        760: 0.01487069,
        765: 0.0138284,
        770: 0.01316616,
        775: 0.01241027,
        780: 0.01204411},
    'HPS': {
        380: 0.0083939426,
        385: 0.0091861681,
        390: 0.0106542846,
        395: 0.0114932882,
        400: 0.0131247036,
        405: 0.0158681257,
        410: 0.0163525586,
        415: 0.0181997788,
        420: 0.02066263,
        425: 0.0225881478,
        430: 0.025273488,
        435: 0.0296422972,
        440: 0.0338150024,
        445: 0.0245030817,
        450: 0.0553529381,
        455: 0.0297319885,
        460: 0.0182223227,
        465: 0.0766598291,
        470: 0.0671995795,
        475: 0.0321898269,
        480: 0.0094290132,
        485: 0.0093821666,
        490: 0.0164154361,
        495: 0.0839860177,
        500: 0.2406839993,
        505: 0.0172085715,
        510: 0.0135166403,
        515: 0.0664860656,
        520: 0.0153956573,
        525: 0.0136191416,
        530: 0.0144758902,
        535: 0.0155227857,
        540: 0.0179781419,
        545: 0.0275498773,
        550: 0.0769794223,
        555: 0.1049451167,
        560: 0.1212042397,
        565: 0.3100762088,
        570: 0.9172816352,
        575: 0.3408922708,
        580: 0.6161317021,
        585: 0.8283991671,
        590: 0.1896168896,
        595: 0.9468979952,
        600: 0.7797498862,
        605: 0.5167643807,
        610: 0.359974244,
        615: 0.4429134241,
        620: 0.2233351421,
        625: 0.1751267514,
        630: 0.1491407729,
        635: 0.130860325,
        640: 0.1161016569,
        645: 0.1058507931,
        650: 0.093576623,
        655: 0.0905438868,
        660: 0.081530368,
        665: 0.0777937264,
        670: 0.0847872834,
        675: 0.0782668774,
        680: 0.0630587114,
        685: 0.0499039257,
        690: 0.0434748752,
        695: 0.0403923575,
        700: 0.0383675279,
        705: 0.0375213958,
        710: 0.0363709696,
        715: 0.0353301907,
        720: 0.0346478936,
        725: 0.033952096,
        730: 0.0334741327,
        735: 0.0336122169,
        740: 0.0340848695,
        745: 0.0337961699,
        750: 0.0377687065,
        755: 0.0340889904,
        760: 0.0333492551,
        765: 0.0802690011,
        770: 0.0540981011,
        775: 0.032648666,
        780: 0.0352925378},
    'Incandescent': {
        380: 0.0221129274,
        385: 0.0257550191,
        390: 0.0303031416,
        395: 0.0402040843,
        400: 0.0502272809,
        405: 0.0596287164,
        410: 0.0685435436,
        415: 0.077222055,
        420: 0.0842087617,
        425: 0.0913199149,
        430: 0.0997370726,
        435: 0.1084944259,
        440: 0.1161691811,
        445: 0.1253379906,
        450: 0.1346917959,
        455: 0.1450470777,
        460: 0.1556685203,
        465: 0.1666161272,
        470: 0.1769997332,
        475: 0.1890140662,
        480: 0.1994984692,
        485: 0.2107503575,
        490: 0.2230259107,
        495: 0.2349042653,
        500: 0.2477963468,
        505: 0.2611051126,
        510: 0.2735574457,
        515: 0.2873463379,
        520: 0.3007511829,
        525: 0.3148017537,
        530: 0.3286265485,
        535: 0.3429364606,
        540: 0.3581597726,
        545: 0.3732534755,
        550: 0.3883420369,
        555: 0.403987105,
        560: 0.4203674722,
        565: 0.4365160648,
        570: 0.4523056904,
        575: 0.4682869076,
        580: 0.4841530664,
        585: 0.5003725619,
        590: 0.5164160981,
        595: 0.5325415882,
        600: 0.5488244682,
        605: 0.5647150329,
        610: 0.5807535747,
        615: 0.5966998519,
        620: 0.6119241813,
        625: 0.627620854,
        630: 0.6424901113,
        635: 0.6578064624,
        640: 0.6733141181,
        645: 0.6882633218,
        650: 0.7030623147,
        655: 0.7181948129,
        660: 0.7331525555,
        665: 0.747833052,
        670: 0.7640991255,
        675: 0.7810516508,
        680: 0.7982514269,
        685: 0.8127518501,
        690: 0.8278508335,
        695: 0.8406662389,
        700: 0.8528341711,
        705: 0.8651898991,
        710: 0.8808045146,
        715: 0.892185268,
        720: 0.9049681735,
        725: 0.9194588124,
        730: 0.9303931443,
        735: 0.9382116765,
        740: 0.9519940858,
        745: 0.9669876832,
        750: 0.9766525623,
        755: 0.9854043089,
        760: 1.0,
        765: 0.9669876832,
        770: 0.9766525623,
        775: 0.9854043089,
        780: 1.0},
    'LPS': {
        380: 0.0,
        385: 0.0,
        390: 0.0,
        395: 0.0,
        400: 0.0,
        405: 0.0,
        410: 0.0,
        415: 0.0,
        420: 0.0,
        425: 0.0,
        430: 0.0,
        435: 0.0,
        440: 0.0,
        445: 0.0,
        450: 0.0,
        455: 0.0,
        460: 0.0,
        465: 0.0,
        470: 0.0,
        475: 0.0,
        480: 0.0,
        485: 0.0,
        490: 0.0,
        495: 0.0,
        500: 0.0,
        505: 0.0,
        510: 0.0,
        515: 0.0,
        520: 0.0,
        525: 0.0,
        530: 0.0,
        535: 0.0,
        540: 0.0,
        545: 0.0,
        550: 0.0,
        555: 0.0,
        560: 0.0,
        565: 0.0,
        570: 0.0,
        575: 0.0,
        580: 0.0,
        585: 0.0,
        590: 1.0,
        595: 0.0,
        600: 0.0,
        605: 0.0,
        610: 0.0,
        615: 0.0,
        620: 0.0,
        625: 0.0,
        630: 0.0,
        635: 0.0,
        640: 0.0,
        645: 0.0,
        650: 0.0,
        655: 0.0,
        660: 0.0,
        665: 0.0,
        670: 0.0,
        675: 0.0,
        680: 0.0,
        685: 0.0,
        690: 0.0,
        695: 0.0,
        700: 0.0,
        705: 0.0,
        710: 0.0,
        715: 0.0,
        720: 0.0,
        725: 0.0,
        730: 0.0,
        735: 0.0,
        740: 0.0,
        745: 0.0,
        750: 0.0,
        755: 0.0,
        760: 0.0,
        765: 0.0,
        770: 0.0,
        775: 0.0,
        780: 0.0},
    'Mercury': {
        380: 0.0104373669,
        385: 0.0078158007,
        390: 0.0200578779,
        395: 0.0076610484,
        400: 0.0110613861,
        405: 0.345639688,
        410: 0.0489706882,
        415: 0.0097542235,
        420: 0.0092107412,
        425: 0.0107885565,
        430: 0.0160358912,
        435: 0.635262009,
        440: 0.0949047348,
        445: 0.0126262844,
        450: 0.0096496965,
        455: 0.0080812689,
        460: 0.0074381603,
        465: 0.0071390793,
        470: 0.0077442532,
        475: 0.0079916618,
        480: 0.0076359546,
        485: 0.0084091185,
        490: 0.0163720623,
        495: 0.0128686374,
        500: 0.0071557219,
        505: 0.0069781037,
        510: 0.0069555478,
        515: 0.0070881258,
        520: 0.0067250903,
        525: 0.0069016358,
        530: 0.0078761268,
        535: 0.0138417249,
        540: 0.0187277695,
        545: 0.7441041872,
        550: 0.1876942249,
        555: 0.0170026604,
        560: 0.0127049957,
        565: 0.0103207523,
        570: 0.0112539716,
        575: 0.3755814786,
        580: 0.7277013187,
        585: 0.0190778829,
        590: 0.0269036873,
        595: 0.0649556356,
        600: 0.0160156414,
        605: 0.0219968756,
        610: 0.062236659,
        615: 0.2658679714,
        620: 0.3328966037,
        625: 0.0507709071,
        630: 0.0281446584,
        635: 0.0161248815,
        640: 0.0121597222,
        645: 0.0136514793,
        650: 0.0222077529,
        655: 0.0166186148,
        660: 0.0108457453,
        665: 0.0100008015,
        670: 0.0108449116,
        675: 0.0091036782,
        680: 0.0090070042,
        685: 0.0095061592,
        690: 0.0289025543,
        695: 0.0500928386,
        700: 0.0829550499,
        705: 0.0670368474,
        710: 0.015473855,
        715: 0.0101121521,
        720: 0.0092409168,
        725: 0.0090681946,
        730: 0.0091454925,
        735: 0.0093017409,
        740: 0.008912659,
        745: 0.0090466381,
        750: 0.0095978931,
        755: 0.0093381677,
        760: 0.0090479193,
        765: 0.0096855504,
        770: 0.0113490129,
        775: 0.009900541,
        780: 0.009495425},
    'Metal Halide': {
        380: 0.0601404826,
        385: 0.0630286148,
        390: 0.2366604181,
        395: 0.1390017215,
        400: 0.2046555993,
        405: 0.3622174434,
        410: 0.1531198649,
        415: 0.1693658196,
        420: 0.0987453494,
        425: 0.1591580003,
        430: 0.1225200054,
        435: 0.2855881537,
        440: 0.1361859185,
        445: 0.0584927417,
        450: 0.0526076706,
        455: 0.0682445933,
        460: 0.0642654817,
        465: 0.0426917758,
        470: 0.1037722376,
        475: 0.3242111432,
        480: 0.0685658127,
        485: 0.096806754,
        490: 0.0623851924,
        495: 0.0821520756,
        500: 0.1059445954,
        505: 0.1723849701,
        510: 0.4831405398,
        515: 0.0444522475,
        520: 0.0604517023,
        525: 0.0596773008,
        530: 0.0535777364,
        535: 0.2086168666,
        540: 0.1103107422,
        545: 0.3744470877,
        550: 0.2598804835,
        555: 0.1164595852,
        560: 0.0622287119,
        565: 0.1337516709,
        570: 0.4662630787,
        575: 0.1720014425,
        580: 0.2009060562,
        585: 0.0638586912,
        590: 1.0,
        595: 0.322894464,
        600: 0.153423957,
        605: 0.0864117251,
        610: 0.0646383054,
        615: 0.0722486842,
        620: 0.13994463,
        625: 0.1979398736,
        630: 0.1662948567,
        635: 0.0546899081,
        640: 0.0693297243,
        645: 0.0414284907,
        650: 0.0280057873,
        655: 0.0322601633,
        660: 0.0385852496,
        665: 0.0211598411,
        670: 0.1344379816,
        675: 0.0380230866,
        680: 0.0390038318,
        685: 0.0288756821,
        690: 0.0252639214,
        695: 0.0197037132,
        700: 0.0222950995,
        705: 0.0157119765,
        710: 0.0149264722,
        715: 0.0219255108,
        720: 0.0183285624,
        725: 0.0135851623,
        730: 0.0126449995,
        735: 0.0123184844,
        740: 0.0142654181,
        745: 0.0124235049,
        750: 0.0165247486,
        755: 0.0202005402,
        760: 0.015806516,
        765: 0.0414979164,
        770: 0.034467834,
        775: 0.0236333145,
        780: 0.0254235315},
    'Neodimium Incandescent': {
        380: 0.0334673674,
        385: 0.0367106969,
        390: 0.0410280255,
        395: 0.0455473384,
        400: 0.0506226629,
        405: 0.0562349826,
        410: 0.0622238391,
        415: 0.0683808247,
        420: 0.0744914499,
        425: 0.0812384335,
        430: 0.083454041,
        435: 0.0945676132,
        440: 0.104529764,
        445: 0.1138775255,
        450: 0.1216878449,
        455: 0.1292710044,
        460: 0.1371191354,
        465: 0.1467502465,
        470: 0.1544362807,
        475: 0.1632724027,
        480: 0.176333928,
        485: 0.1929622196,
        490: 0.2071824116,
        495: 0.2172215495,
        500: 0.2241001123,
        505: 0.2295670418,
        510: 0.225386183,
        515: 0.2300856349,
        520: 0.2481180326,
        525: 0.2314553839,
        530: 0.2361240668,
        535: 0.2746820107,
        540: 0.3230977385,
        545: 0.3534813175,
        550: 0.371877509,
        555: 0.3863895876,
        560: 0.3984217286,
        565: 0.3904282774,
        570: 0.2329025728,
        575: 0.1814030148,
        580: 0.2087447247,
        585: 0.1396931667,
        590: 0.2055993659,
        595: 0.2976530252,
        600: 0.3910370535,
        605: 0.4663146637,
        610: 0.5127735193,
        615: 0.5745016699,
        620: 0.602488237,
        625: 0.616456127,
        630: 0.6332086934,
        635: 0.653120173,
        640: 0.6725252206,
        645: 0.6901662352,
        650: 0.7070018104,
        655: 0.7238974104,
        660: 0.739174282,
        665: 0.7522953851,
        670: 0.7615731228,
        675: 0.7694685235,
        680: 0.7712910882,
        685: 0.77114333,
        690: 0.8019984983,
        695: 0.8329811362,
        700: 0.8569011127,
        705: 0.8750642104,
        710: 0.8908355615,
        715: 0.9026457704,
        720: 0.9086626257,
        725: 0.9090200398,
        730: 0.8761935329,
        735: 0.7126783743,
        740: 0.5643330822,
        745: 0.5915964992,
        750: 0.5915750404,
        755: 0.6939061885,
        760: 0.8146286234,
        765: 0.8999932814,
        770: 0.9622718511,
        775: 0.9966525876,
        780: 1.0},
    'Super HPS': {
        380: 0.0,
        385: 0.003596107,
        390: 0.0045981474,
        395: 0.0050177851,
        400: 0.0057492868,
        405: 0.0076343625,
        410: 0.0082144378,
        415: 0.0099511979,
        420: 0.0122130333,
        425: 0.0137991778,
        430: 0.0165173427,
        435: 0.0217400514,
        440: 0.0203116242,
        445: 0.0212391684,
        450: 0.0309894056,
        455: 0.0259091888,
        460: 0.0201990203,
        465: 0.0308219293,
        470: 0.0346853066,
        475: 0.0219456051,
        480: 0.0126257964,
        485: 0.013564814,
        490: 0.0168973956,
        495: 0.0243983602,
        500: 0.0434304636,
        505: 0.0192060585,
        510: 0.0194871844,
        515: 0.0455673292,
        520: 0.0214293107,
        525: 0.0232814599,
        530: 0.0249044139,
        535: 0.027288095,
        540: 0.0318833455,
        545: 0.0470584913,
        550: 0.1120803531,
        555: 0.1304844572,
        560: 0.1398153826,
        565: 0.1563428339,
        570: 0.1597922384,
        575: 0.1017778764,
        580: 0.027339422,
        585: 0.0050505846,
        590: 0.0062796715,
        595: 0.011816069,
        600: 0.0326627675,
        605: 0.0864624384,
        610: 0.1325538401,
        615: 0.1785194715,
        620: 0.1647322754,
        625: 0.1652710919,
        630: 0.1604288888,
        635: 0.155280408,
        640: 0.1504925888,
        645: 0.1443308799,
        650: 0.1353427208,
        655: 0.1343059341,
        660: 0.1245938386,
        665: 0.1251833569,
        670: 0.1314285899,
        675: 0.1222621215,
        680: 0.1056669972,
        685: 0.0924634081,
        690: 0.0850971124,
        695: 0.0809388186,
        700: 0.0780317816,
        705: 0.0749906848,
        710: 0.0728725021,
        715: 0.0713867551,
        720: 0.0693760096,
        725: 0.0672336597,
        730: 0.0657510427,
        735: 0.0648570879,
        740: 0.0628974098,
        745: 0.0627958506,
        750: 0.0631247959,
        755: 0.0606465187,
        760: 0.0604807146,
        765: 0.0604807146,
        770: 0.0604807146,
        775: 0.0604807146,
        780: 0.0604807146},
    'Triphosphor FL': {
        380: 0.0,
        385: 0.0,
        390: 0.0,
        395: 0.0,
        400: 0.0312165583,
        405: 0.1764182828,
        410: 0.0271236509,
        415: 0.0174611208,
        420: 0.0222175607,
        425: 0.028700239,
        430: 0.0484557598,
        435: 0.4369152872,
        440: 0.1291444626,
        445: 0.0489915809,
        450: 0.0493914708,
        455: 0.0475871896,
        460: 0.0441639176,
        465: 0.0421742718,
        470: 0.0371456337,
        475: 0.0329023313,
        480: 0.0490143379,
        485: 0.162794674,
        490: 0.1751722229,
        495: 0.099797644,
        500: 0.0481894258,
        505: 0.0197095601,
        510: 0.011908578,
        515: 0.0081560051,
        520: 0.0056489845,
        525: 0.0053536776,
        530: 0.0135549811,
        535: 0.0647990169,
        540: 0.5096012278,
        545: 0.9491769908,
        550: 0.4258456058,
        555: 0.0913432846,
        560: 0.027453774,
        565: 0.017759397,
        570: 0.0121038553,
        575: 0.0583116917,
        580: 0.1627650886,
        585: 0.2032754524,
        590: 0.1739997914,
        595: 0.142719745,
        600: 0.1055151599,
        605: 0.085800347,
        610: 0.8499104205,
        615: 0.6957625416,
        620: 0.1748237209,
        625: 0.1824019459,
        630: 0.1965917676,
        635: 0.0733622438,
        640: 0.0282908972,
        645: 0.0289652496,
        650: 0.055266608,
        655: 0.0431298399,
        660: 0.0338880415,
        665: 0.0328566739,
        670: 0.0211993347,
        675: 0.0172926916,
        680: 0.0188396636,
        685: 0.0274920261,
        690: 0.0305164384,
        695: 0.0203802625,
        700: 0.0091290341,
        705: 0.057416353,
        710: 0.1091093084,
        715: 0.0396412038,
        720: 0.0044756182,
        725: 0.0038153808,
        730: 0.0033579794,
        735: 0.0028794702,
        740: 0.0048043101,
        745: 0.0038307798,
        750: 0.0045229902,
        755: 0.002241563,
        760: 0.0028520215,
        765: 0.0079622762,
        770: 0.0024428504,
        775: 0.0029993177,
        780: 0.0005290507}}

LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS = CaseInsensitiveMapping(
    {'Cool White FL': SpectralPowerDistribution(
        'Cool White FL',
        LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
            'Cool White FL')),
     'Daylight FL': SpectralPowerDistribution(
         'Daylight FL',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Daylight FL')),
     'HPS': SpectralPowerDistribution(
         'HPS',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'HPS')),
     'Incandescent': SpectralPowerDistribution(
         'Incandescent',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Incandescent')),
     'LPS': SpectralPowerDistribution(
         'LPS',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'LPS')),
     'Mercury': SpectralPowerDistribution(
         'Mercury',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Mercury')),
     'Metal Halide': SpectralPowerDistribution(
         'Metal Halide',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Metal Halide')),
     'Neodimium Incandescent': SpectralPowerDistribution(
         'Neodimium Incandescent',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Neodimium Incandescent')),
     'Super HPS': SpectralPowerDistribution(
         'Super HPS',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Super HPS')),
     'Triphosphor FL': SpectralPowerDistribution(
         'Triphosphor FL',
         LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS_DATA.get(
             'Triphosphor FL'))})
"""
Traditional light sources from *NIST* *NIST CQS simulation 7.4.xls*
spreadsheet.

References
----------
.. [2]  Ohno, Y., & Davis, W. (2008). NIST CQS simulation 7.4. Retrieved from
        http://cie2.nist.gov/TC1-69/NIST CQS simulation 7.4.xls

LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS : CaseInsensitiveMapping
    {'Cool White FL', 'Daylight FL', 'HPS', 'Incandescent', 'LPS', 'Mercury',
    'Metal Halide', 'Neodimium Incandescent', 'Super HPS', 'Triphosphor FL'}
"""

LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA = {
    '3-LED-1 (457/540/605)': {
        380: 8.47479023841784e-08,
        385: 5.45760813791522e-07,
        390: 3.10162282092079e-06,
        395: 1.55556747080694e-05,
        400: 6.88496856317102e-05,
        405: 0.0002689232,
        410: 0.0009269749,
        415: 0.0028198174,
        420: 0.0075698702,
        425: 0.0179348874,
        430: 0.0375451905,
        435: 0.0702725856,
        440: 0.1251616616,
        445: 0.2385579273,
        450: 0.4567468433,
        455: 0.6711246862,
        460: 0.6416953988,
        465: 0.406801588,
        470: 0.2095939908,
        475: 0.1143320593,
        480: 0.0685449408,
        485: 0.0442682326,
        490: 0.0359704324,
        495: 0.0414173506,
        500: 0.0587096983,
        505: 0.0869279964,
        510: 0.1273252807,
        515: 0.1872008828,
        520: 0.285983535,
        525: 0.4506057455,
        530: 0.6807829422,
        535: 0.9044211557,
        540: 1.0000168848,
        545: 0.9044984933,
        550: 0.6811210385,
        555: 0.4518629436,
        560: 0.2901018361,
        565: 0.1990988954,
        570: 0.1576332566,
        575: 0.1549794926,
        580: 0.1937143069,
        585: 0.2839491007,
        590: 0.4708319143,
        595: 0.8935261511,
        600: 1.573660514,
        605: 1.9603362336,
        610: 1.5689953795,
        615: 0.8826040404,
        620: 0.4503777828,
        625: 0.2489161118,
        630: 0.1373257298,
        635: 0.0687972226,
        640: 0.0305197933,
        645: 0.0119512553,
        650: 0.0041301742,
        655: 0.001259623,
        660: 0.0003390232,
        665: 8.05257762151534e-05,
        670: 1.68794361495293e-05,
        675: 3.1224853634832e-06,
        680: 5.09760495335404e-07,
        685: 7.34450718354606e-08,
        690: 9.33909164024241e-09,
        695: 1e-09,
        700: 1.03844288919885e-10,
        705: 9.08519244775383e-12,
        710: 7.02479050506507e-13,
        715: 4.81028913133838e-14,
        720: 2.93275111371904e-15,
        725: 1.61509556619245e-16,
        730: 8.3451106044055e-18,
        735: 4.41398085232855e-19,
        740: 2.72715756464071e-20,
        745: 2.09488870472631e-21,
        750: 1.84466383851435e-22,
        755: 1.67257464713885e-23,
        760: 1.47593993556198e-24,
        765: 1.24216741363154e-25,
        770: 9.91044375901472e-27,
        775: 7.48346361630974e-28,
        780: 5.34607293395408e-29},
    '3-LED-2 (473/545/616)': {
        380: 8.80700887841404e-11,
        385: 8.64191446543499e-10,
        390: 7.48391162992935e-09,
        395: 5.71969601329458e-08,
        400: 3.85777799611739e-07,
        405: 2.29624731069514e-06,
        410: 1.20619012713231e-05,
        415: 5.59149345801818e-05,
        420: 0.0002287465,
        425: 0.0008258399,
        430: 0.00263119,
        435: 0.0073981747,
        440: 0.0183579,
        445: 0.0402192904,
        450: 0.078214713,
        455: 0.1400875209,
        460: 0.25800504,
        465: 0.5069658632,
        470: 0.8461217331,
        475: 0.9560963501,
        480: 0.6977648778,
        485: 0.3786723063,
        490: 0.2046505176,
        495: 0.1293006169,
        500: 0.0949193874,
        505: 0.0868699571,
        510: 0.1011645711,
        515: 0.135550451,
        520: 0.1945767516,
        525: 0.2964280287,
        530: 0.4664464991,
        535: 0.6996200151,
        540: 0.9177525438,
        545: 0.9994051553,
        550: 0.8903170437,
        555: 0.6620178366,
        560: 0.4355346571,
        565: 0.2772581962,
        570: 0.1845032445,
        575: 0.1324452174,
        580: 0.1073694431,
        585: 0.1101780705,
        590: 0.1488758406,
        595: 0.2331869395,
        600: 0.3917042764,
        605: 0.737921343,
        610: 1.3841944984,
        615: 1.9510279141,
        620: 1.7794979544,
        625: 1.0887186669,
        630: 0.5550635406,
        635: 0.2996143988,
        640: 0.1676834956,
        645: 0.0868592197,
        650: 0.0399750379,
        655: 0.0162446511,
        660: 0.0058258274,
        665: 0.0018438235,
        670: 0.0005149855,
        675: 0.0001269359,
        680: 2.76113347070399e-05,
        685: 5.30034610852958e-06,
        690: 8.97914635951279e-07,
        695: 1.34239493903355e-07,
        700: 1.77109091256319e-08,
        705: 2.062141123221e-09,
        710: 2.11892859379853e-10,
        715: 1.92150692517149e-11,
        720: 1.53784122218474e-12,
        725: 1.08633288257358e-13,
        730: 6.77469739707175e-15,
        735: 3.73195251197816e-16,
        740: 1.81873749533185e-17,
        745: 7.87622127844671e-19,
        750: 3.07135726432428e-20,
        755: 1.12157337263848e-21,
        760: 4.2439069645357e-23,
        765: 1.9652013398463e-24,
        770: 1.19562718661274e-25,
        775: 8.49126403034339e-27,
        780: 6.18707499978848e-28},
    '3-LED-2 Yellow': {
        380: 5.28857670457154e-11,
        385: 5.18882652575306e-10,
        390: 4.49324294120251e-09,
        395: 3.43390526697708e-08,
        400: 2.3160253104842e-07,
        405: 1.37853969558916e-06,
        410: 7.24124676481305e-06,
        415: 3.35678915422407e-05,
        420: 0.0001373254,
        425: 0.0004957853,
        430: 0.0015796223,
        435: 0.0044415178,
        440: 0.0110214731,
        445: 0.0241472595,
        450: 0.0469624827,
        455: 0.0841220129,
        460: 0.1549514882,
        465: 0.3045070284,
        470: 0.5083361463,
        475: 0.574846057,
        480: 0.4207534057,
        485: 0.2310645796,
        490: 0.1299151005,
        495: 0.0902207189,
        500: 0.0782549937,
        505: 0.086171506,
        510: 0.1125227291,
        515: 0.1576335916,
        520: 0.2292700303,
        525: 0.3504034157,
        530: 0.5517353488,
        535: 0.8276341163,
        540: 1.0856978743,
        545: 1.1822953563,
        550: 1.0532430391,
        555: 0.7831575392,
        560: 0.5151957813,
        565: 0.3278356015,
        570: 0.217720047,
        575: 0.1550392886,
        580: 0.1226630873,
        585: 0.1201552523,
        590: 0.1550638348,
        595: 0.2368211431,
        600: 0.3937236292,
        605: 0.7389827858,
        610: 1.3847222802,
        615: 1.9512761616,
        620: 1.77960841,
        625: 1.0887651573,
        630: 0.5550820508,
        635: 0.2996213704,
        640: 0.1676859794,
        645: 0.0868600568,
        650: 0.0399753048,
        655: 0.0162447316,
        660: 0.0058258503,
        665: 0.0018438297,
        670: 0.0005149871,
        675: 0.0001269362,
        680: 2.76114219801591e-05,
        685: 5.30036496786247e-06,
        690: 8.97918491131487e-07,
        695: 1.34240239382623e-07,
        700: 1.77110454893972e-08,
        705: 2.06216471903305e-09,
        710: 2.118967216567e-10,
        715: 1.92156672836855e-11,
        720: 1.53792881690537e-12,
        725: 1.08645425051636e-13,
        730: 6.77628814943881e-15,
        735: 3.73392481524687e-16,
        740: 1.82105071746805e-17,
        745: 7.90188582015684e-19,
        750: 3.0982925890825e-20,
        755: 1.14831472999166e-21,
        760: 4.49504772669227e-23,
        765: 2.18831363146954e-24,
        770: 1.3831276367807e-25,
        775: 9.98183844142717e-27,
        780: 7.30800253280185e-28},
    '3-LED-3 (465/546/614)': {
        380: 4.24356931933744e-09,
        385: 3.35055502496331e-08,
        390: 2.33461667941402e-07,
        395: 1.43558107162236e-06,
        400: 7.79028131957144e-06,
        405: 3.73071287200574e-05,
        410: 0.0001576681,
        415: 0.0005880429,
        420: 0.0019354753,
        425: 0.0056218569,
        430: 0.0144108351,
        435: 0.0326064096,
        440: 0.0653116295,
        445: 0.1186052045,
        450: 0.2141526104,
        455: 0.419111692,
        460: 0.750409624,
        465: 0.9493997063,
        470: 0.7707831225,
        475: 0.4379446959,
        480: 0.225391027,
        485: 0.1284217477,
        490: 0.0786132901,
        495: 0.0534890663,
        500: 0.0482425923,
        505: 0.0593816127,
        510: 0.0841113617,
        515: 0.1219728373,
        520: 0.1784292638,
        525: 0.2708941147,
        530: 0.4264165846,
        535: 0.6507852663,
        540: 0.8815590758,
        545: 0.9984827318,
        550: 0.925301255,
        555: 0.7108364568,
        560: 0.4761832296,
        565: 0.3032184448,
        570: 0.2002061954,
        575: 0.1434331509,
        580: 0.1163388244,
        585: 0.1169248909,
        590: 0.1502378648,
        595: 0.2253626825,
        600: 0.3832280121,
        605: 0.7352864775,
        610: 1.2586014278,
        615: 1.4894514208,
        620: 1.133317893,
        625: 0.6204123938,
        630: 0.3184632339,
        635: 0.177246509,
        640: 0.0969798333,
        645: 0.047908591,
        650: 0.0209391161,
        655: 0.0080777652,
        660: 0.0027500671,
        665: 0.0008262488,
        670: 0.0002190753,
        675: 5.1261418534218e-05,
        680: 1.05852933346746e-05,
        685: 1.929e-06,
        690: 3.10224214617215e-07,
        695: 4.4e-08,
        700: 5.51479030605828e-09,
        705: 6.09610723887443e-10,
        710: 5.94747096451326e-11,
        715: 5.12171902061325e-12,
        720: 3.89415797742087e-13,
        725: 2.6157549054432e-14,
        730: 1.55476773649257e-15,
        735: 8.21313246961633e-17,
        740: 3.903112120718e-18,
        745: 1.72611102996469e-19,
        750: 7.72579136687614e-21,
        755: 4.03701113939912e-22,
        760: 2.68516940390801e-23,
        765: 2.09993710079583e-24,
        770: 1.70971251548166e-25,
        775: 1.35854661437952e-26,
        780: 1.03007323231651e-27},
    '3-LED-4 (455/547/623)': {
        380: 1.48554815599882e-07,
        385: 9.1e-07,
        390: 4.91945989968639e-06,
        395: 2.34694309188581e-05,
        400: 9.881e-05,
        405: 0.0003671237,
        410: 0.0012037523,
        415: 0.0034831741,
        420: 0.008894682,
        425: 0.0200492403,
        430: 0.0400167822,
        435: 0.0725244502,
        440: 0.1312010645,
        445: 0.2570833796,
        450: 0.456931888,
        455: 0.5705517003,
        460: 0.4569964913,
        465: 0.2572699646,
        470: 0.1316591009,
        475: 0.0735745258,
        480: 0.0422903013,
        485: 0.0247046376,
        490: 0.0179119685,
        495: 0.0200054668,
        500: 0.0298438422,
        505: 0.0473568983,
        510: 0.0732535988,
        515: 0.1091225432,
        520: 0.159905858,
        525: 0.2399867425,
        530: 0.3756330202,
        535: 0.5836005961,
        540: 0.8234603813,
        545: 0.9838705345,
        550: 0.9641705295,
        555: 0.7776477851,
        560: 0.5370033829,
        565: 0.3428897428,
        570: 0.2209973019,
        575: 0.1499165163,
        580: 0.1067853989,
        585: 0.0832231171,
        590: 0.0816202917,
        595: 0.1091680792,
        600: 0.1739360904,
        605: 0.2917296769,
        610: 0.5348840896,
        615: 1.0377379062,
        620: 1.6366466718,
        625: 1.7116056209,
        630: 1.1648450523,
        635: 0.6083919924,
        640: 0.3191958702,
        645: 0.1792122278,
        650: 0.0957487311,
        655: 0.0457378312,
        660: 0.0193047537,
        665: 0.007191114,
        670: 0.0023639744,
        675: 0.0006858086,
        680: 0.0001755806,
        685: 3.96701061693615e-05,
        690: 7.9097659889265e-06,
        695: 1.39180147224715e-06,
        700: 2.16124701308646e-07,
        705: 2.96172770638478e-08,
        710: 3.58178693090179e-09,
        715: 3.82268783348228e-10,
        720: 3.60042318489114e-11,
        725: 2.99265299907172e-12,
        730: 2.19524391716436e-13,
        735: 1.42116902184803e-14,
        740: 8.12039530369973e-16,
        745: 4.09600540729487e-17,
        750: 1.8248444830666e-18,
        755: 7.1918957051195e-20,
        760: 2.51922315521053e-21,
        765: 7.96184389637815e-23,
        770: 2.37929410472859e-24,
        775: 7.60786774257376e-26,
        780: 3.1397886480576e-27},
    '4-LED No Yellow': {
        380: 5.00830470941566e-06,
        385: 2.37160686348979e-05,
        390: 9.94095202401805e-05,
        395: 0.0003684067,
        400: 0.0012063431,
        405: 0.0034892167,
        410: 0.0089138207,
        415: 0.0201192199,
        420: 0.0402570188,
        425: 0.0732335592,
        430: 0.132926886,
        435: 0.2605953485,
        440: 0.4638844161,
        445: 0.5855023828,
        450: 0.4874605671,
        455: 0.3109198478,
        460: 0.2160752258,
        465: 0.1989119254,
        470: 0.2267627783,
        475: 0.3058194397,
        480: 0.4594090138,
        485: 0.6842331444,
        490: 0.9056258811,
        495: 1.0004410948,
        500: 0.9048427933,
        505: 0.6820202478,
        510: 0.4545920361,
        515: 0.2975002875,
        520: 0.2165855002,
        525: 0.1934695205,
        530: 0.2186286024,
        535: 0.2964418163,
        540: 0.4693367123,
        545: 0.871511743,
        550: 1.5237206939,
        555: 1.8942663065,
        560: 1.515471306,
        565: 0.8533178434,
        570: 0.4386103796,
        575: 0.2517591342,
        580: 0.1633454618,
        585: 0.1393768576,
        590: 0.1822373617,
        595: 0.2974969095,
        600: 0.5133048718,
        605: 0.9719564749,
        610: 1.8588708015,
        615: 2.7307473319,
        620: 2.6102604402,
        625: 1.6528454396,
        630: 0.8470736203,
        635: 0.4528629836,
        640: 0.2540957124,
        645: 0.1330819356,
        650: 0.062036024,
        655: 0.0255382935,
        660: 0.0092782966,
        665: 0.0029748034,
        670: 0.0008417081,
        675: 0.0002101735,
        680: 4.63134881250587e-05,
        685: 9.00638021417258e-06,
        690: 1.54563254200153e-06,
        695: 2.3408598797697e-07,
        700: 3.12865570429469e-08,
        705: 3.69022900773749e-09,
        710: 3.84115695212995e-10,
        715: 3.52845008045302e-11,
        720: 2.86034943324941e-12,
        725: 2.04629100750059e-13,
        730: 1.29189997394275e-14,
        735: 7.19786294523538e-16,
        740: 3.53908868998459e-17,
        745: 1.53565093768907e-18,
        750: 5.88039855325232e-20,
        755: 1.98716625964952e-21,
        760: 5.92617997896856e-23,
        765: 1.55965542079349e-24,
        770: 3.62239387510338e-26,
        775: 7.42464806370353e-28,
        780: 1.34297913021045e-29},
    '4-LED Yellow': {
        380: 8.34073345191213e-07,
        385: 4.62309206822133e-06,
        390: 2.26138576311572e-05,
        395: 9.76181321599308e-05,
        400: 0.0003718777,
        405: 0.0012502119,
        410: 0.0037092106,
        415: 0.0097117231,
        420: 0.0224430372,
        425: 0.0458645624,
        430: 0.0843479754,
        435: 0.1509992844,
        440: 0.2926113259,
        445: 0.5430706712,
        450: 0.7376612506,
        455: 0.6455081538,
        460: 0.3853224168,
        465: 0.2000431541,
        470: 0.11622805,
        475: 0.077645951,
        480: 0.0620623017,
        485: 0.0653287242,
        490: 0.0850069785,
        495: 0.1197520208,
        500: 0.1734854602,
        505: 0.2619073927,
        510: 0.4116390053,
        515: 0.6318025046,
        520: 0.8661395277,
        525: 0.9961648708,
        530: 0.9382724574,
        535: 0.7325090806,
        540: 0.5005388519,
        545: 0.3334567263,
        550: 0.2492404075,
        555: 0.2295846621,
        560: 0.2608490286,
        565: 0.3646989969,
        570: 0.6327523963,
        575: 1.0793281438,
        580: 1.3339290091,
        585: 1.0720260109,
        590: 0.623012768,
        595: 0.3642920241,
        600: 0.2900835662,
        605: 0.3167957643,
        610: 0.447506532,
        615: 0.7977401989,
        620: 1.4502472884,
        625: 1.9611037213,
        630: 1.712516401,
        635: 1.0169222215,
        640: 0.5173345935,
        645: 0.2816360425,
        650: 0.157068211,
        655: 0.0805061621,
        660: 0.0366130494,
        665: 0.0147005543,
        670: 0.0052089819,
        675: 0.0016288664,
        680: 0.0004495017,
        685: 0.0001094688,
        690: 2.35267847839564e-05,
        695: 4.46218942905036e-06,
        700: 7.468727536897e-07,
        705: 1.10321058401142e-07,
        710: 1.43808121342604e-08,
        715: 1.65432790630949e-09,
        720: 1.67947277647436e-10,
        725: 1.50465707332419e-11,
        730: 1.18963924722871e-12,
        735: 8.30053768968848e-14,
        740: 5.11105269285693e-15,
        745: 2.77733125345012e-16,
        750: 1.33185883339843e-17,
        755: 5.63640124929395e-19,
        760: 2.10503250551692e-20,
        765: 6.93791409316552e-22,
        770: 2.01795866069721e-23,
        775: 5.17975154307456e-25,
        780: 1.17332723412027e-26},
    '4-LED-1 (461/526/576/624)': {
        380: 2.47034736673368e-08,
        385: 1.74863383163897e-07,
        390: 1.09241483462778e-06,
        395: 6.023e-06,
        400: 2.9306e-05,
        405: 0.0001258419,
        410: 0.0004768855,
        415: 0.0015948592,
        420: 0.0047070784,
        425: 0.0122604544,
        430: 0.0281872747,
        435: 0.057323327,
        440: 0.1050910781,
        445: 0.1885157914,
        450: 0.3664358824,
        455: 0.6747520202,
        460: 0.9015268133,
        465: 0.7764380427,
        470: 0.4624165126,
        475: 0.2481247455,
        480: 0.1565677745,
        485: 0.1201464288,
        490: 0.1132302368,
        495: 0.1312440083,
        500: 0.1752511464,
        505: 0.2571440442,
        510: 0.4010423871,
        515: 0.6172340409,
        520: 0.853747073,
        525: 0.9938042128,
        530: 0.948605546,
        535: 0.7514144409,
        540: 0.5239919837,
        545: 0.3638445544,
        550: 0.2943586303,
        555: 0.2992073804,
        560: 0.3798050239,
        565: 0.6135721399,
        570: 1.0635790972,
        575: 1.4215623153,
        580: 1.2456566316,
        585: 0.7545828584,
        590: 0.4166331871,
        595: 0.2946881286,
        600: 0.2912677287,
        605: 0.3772550605,
        610: 0.6105219695,
        615: 1.1586666143,
        620: 1.9585212559,
        625: 2.2763066746,
        630: 1.7027043161,
        635: 0.9243021559,
        640: 0.475718762,
        645: 0.2653176121,
        650: 0.1447206279,
        655: 0.0711464922,
        660: 0.0309366622,
        665: 0.0118732021,
        670: 0.0040214047,
        675: 0.0012019903,
        680: 0.000317057,
        685: 7.38051904658425e-05,
        690: 1.51617616558402e-05,
        695: 2.74868906770862e-06,
        700: 4.39759018807603e-07,
        705: 6.20893458952517e-08,
        710: 7.73628755814796e-09,
        715: 8.50670384604396e-10,
        720: 8.25473646198463e-11,
        725: 7.06900521150489e-12,
        730: 5.34227919448561e-13,
        735: 3.56293624067837e-14,
        740: 2.09702066924617e-15,
        745: 1.0892073507049e-16,
        750: 4.99265562129981e-18,
        755: 2.01960230997846e-19,
        760: 7.20963574573056e-21,
        765: 2.27129761972571e-22,
        770: 6.31463339864241e-24,
        775: 1.5493015588089e-25,
        780: 3.35458329193818e-27},
    '4-LED-2 (447/512/573/627)': {
        380: 3.47088525590474e-06,
        385: 1.73227787723854e-05,
        390: 7.62995752388568e-05,
        395: 0.0002965859,
        400: 0.0010174229,
        405: 0.0030801821,
        410: 0.0082296172,
        415: 0.019406636,
        420: 0.0404442831,
        425: 0.0754508857,
        430: 0.1345142305,
        435: 0.2573666392,
        440: 0.4904398149,
        445: 0.7108377283,
        450: 0.6704591043,
        455: 0.4254812906,
        460: 0.2300455929,
        465: 0.1440595345,
        470: 0.1128724948,
        475: 0.109167769,
        480: 0.1282036266,
        485: 0.171607687,
        490: 0.2512315866,
        495: 0.3912637391,
        500: 0.6041236081,
        505: 0.842077825,
        510: 0.9899997754,
        515: 0.9537807365,
        520: 0.7580812591,
        525: 0.5198871158,
        530: 0.3362122225,
        535: 0.2305410304,
        540: 0.1861122739,
        545: 0.1888163513,
        550: 0.237778663,
        555: 0.3490698559,
        560: 0.6040289428,
        565: 1.1397056922,
        570: 1.7495475627,
        575: 1.7792868525,
        580: 1.1864821619,
        585: 0.6230702032,
        590: 0.3486023593,
        595: 0.2407542116,
        600: 0.2200521394,
        605: 0.276970101,
        610: 0.430931037,
        615: 0.7917849848,
        620: 1.5048761731,
        625: 2.2092877054,
        630: 2.1125014674,
        635: 1.3381309493,
        640: 0.6858253417,
        645: 0.3666131487,
        650: 0.2057056326,
        655: 0.1077508316,
        660: 0.0502351707,
        665: 0.020683273,
        670: 0.0075155225,
        675: 0.0024099761,
        680: 0.0006819923,
        685: 0.0001703176,
        690: 3.75363978233766e-05,
        695: 7.30060620728803e-06,
        700: 1.25307897511291e-06,
        705: 1.89806531416259e-07,
        710: 2.53721372593984e-08,
        715: 3e-09,
        720: 3.11593522066652e-10,
        725: 2.86268715235678e-11,
        730: 2.32098614166006e-12,
        735: 1.6606740353941e-13,
        740: 1.0485989861753e-14,
        745: 5.84315842158581e-16,
        750: 2.87341979093836e-17,
        755: 1.24699205346743e-18,
        760: 4.77574888458134e-20,
        765: 1.61410755281383e-21,
        770: 4.81433876182624e-23,
        775: 1.2672258181104e-24,
        780: 2.94363913832964e-26},
    'Luxeon WW 2880': {
        380: 0.0,
        385: 0.0,
        390: 0.0,
        395: 0.0,
        400: 2.297e-06,
        405: 1.888118e-06,
        410: 4.188269e-06,
        415: 6.39813e-06,
        420: 1.0205089e-05,
        425: 1.6262e-05,
        430: 2.1840675e-05,
        435: 2.550531e-05,
        440: 2.8897053e-05,
        445: 3.2024138e-05,
        450: 3.6703116e-05,
        455: 4.6088e-05,
        460: 6.079116e-05,
        465: 7.6763604e-05,
        470: 8.5592517e-05,
        475: 8.5690075e-05,
        480: 8.8373579e-05,
        485: 9.3201104e-05,
        490: 9.6155462e-05,
        495: 9.72416e-05,
        500: 0.0001027841,
        505: 0.0001101095,
        510: 0.0001201343,
        515: 0.0001313546,
        520: 0.0001443337,
        525: 0.0001567982,
        530: 0.0001679092,
        535: 0.000178065,
        540: 0.0001880002,
        545: 0.0001978362,
        550: 0.0002061661,
        555: 0.0002138138,
        560: 0.0002206354,
        565: 0.0002265298,
        570: 0.0002312203,
        575: 0.0002352487,
        580: 0.0002393245,
        585: 0.000243409,
        590: 0.0002480589,
        595: 0.000252773,
        600: 0.0002589091,
        605: 0.0002670975,
        610: 0.0002777688,
        615: 0.0002878879,
        620: 0.0002990376,
        625: 0.0003091711,
        630: 0.0003174115,
        635: 0.0003227167,
        640: 0.0003212741,
        645: 0.0003154946,
        650: 0.0003036808,
        655: 0.0002880347,
        660: 0.0002668517,
        665: 0.000244548,
        670: 0.0002222251,
        675: 0.000196939,
        680: 0.0001731562,
        685: 0.000150151,
        690: 0.0001298548,
        695: 0.0001102309,
        700: 9.3293572e-05,
        705: 7.8635792e-05,
        710: 6.6303267e-05,
        715: 5.4851252e-05,
        720: 4.5785265e-05,
        725: 3.7650458e-05,
        730: 3.2428674e-05,
        735: 2.7001749e-05,
        740: 2.1683091e-05,
        745: 1.9283911e-05,
        750: 1.6191315e-05,
        755: 1.4043547e-05,
        760: 1.1397697e-05,
        765: 1.0039448e-05,
        770: 8.9298495e-06,
        775: 7.617748e-06,
        780: 7.001642e-06},
    'PHOS-1': {
        380: 0.0111298531,
        385: 0.0181132825,
        390: 0.0278369484,
        395: 0.0403982755,
        400: 0.0553631468,
        405: 0.0716465769,
        410: 0.0875561915,
        415: 0.1010403908,
        420: 0.1101082626,
        425: 0.1221540593,
        430: 0.1313705692,
        435: 0.1409522197,
        440: 0.1508924025,
        445: 0.1611835141,
        450: 0.1718170106,
        455: 0.1827834642,
        460: 0.1940726237,
        465: 0.2056734746,
        470: 0.2175743025,
        475: 0.2297627552,
        480: 0.2422259061,
        485: 0.2549503173,
        490: 0.2679221012,
        495: 0.2811269821,
        500: 0.2945503556,
        505: 0.3081773474,
        510: 0.3219928689,
        515: 0.3359816722,
        520: 0.3501284023,
        525: 0.3644176462,
        530: 0.3788339812,
        535: 0.3933620191,
        540: 0.4079864492,
        545: 0.422692077,
        550: 0.4374638623,
        555: 0.4522869529,
        560: 0.467146717,
        565: 0.4820287721,
        570: 0.4969190117,
        575: 0.5118036301,
        580: 0.5266691438,
        585: 0.5415024112,
        590: 0.5562906503,
        595: 0.5710214533,
        600: 0.5856828003,
        605: 0.60026307,
        610: 0.6147510493,
        615: 0.6291359409,
        620: 0.6434073688,
        625: 0.6575553828,
        630: 0.6715704613,
        635: 0.6854435126,
        640: 0.6991658751,
        645: 0.7127293164,
        650: 0.7261260306,
        655: 0.7393486357,
        660: 0.7523901688,
        665: 0.765244082,
        670: 0.7779042355,
        675: 0.7903648922,
        680: 0.7799535346,
        685: 0.715721128,
        690: 0.6202055995,
        695: 0.5075096054,
        700: 0.3921656833,
        705: 0.2861617924,
        710: 0.1971834428,
        715: 0.128305709,
        720: 0.0788384822,
        725: 0.0457453829,
        730: 0.0250653063,
        735: 0.0129692692,
        740: 0.006336869,
        745: 0.0029238204,
        750: 0.0012739236,
        755: 0.0005241467,
        760: 0.0002036475,
        765: 7.47174315193493e-05,
        770: 2.5887e-05,
        775: 8.46950409085507e-06,
        780: 2.61668300203868e-06},
    'PHOS-2': {
        380: 6.24336458874078e-06,
        385: 1.80201765628625e-05,
        390: 4.91152235508884e-05,
        395: 0.0001264125,
        400: 0.0003072419,
        405: 0.0007051602,
        410: 0.0015283114,
        415: 0.0031278982,
        420: 0.0060451924,
        425: 0.0110327652,
        430: 0.019014082,
        435: 0.0309444728,
        440: 0.0475562447,
        445: 0.0690158363,
        450: 0.0945816085,
        455: 0.1223999877,
        460: 0.1495797457,
        465: 0.172615959,
        470: 0.1881073816,
        475: 0.2063359626,
        480: 0.2195263851,
        485: 0.2331366501,
        490: 0.2471570005,
        495: 0.2615767923,
        500: 0.2763845506,
        505: 0.2915680275,
        510: 0.3071142603,
        515: 0.323009631,
        520: 0.3392399252,
        525: 0.3557903911,
        530: 0.3726457985,
        535: 0.3897904961,
        540: 0.4072084685,
        545: 0.4248833915,
        550: 0.4427986861,
        555: 0.4609375711,
        560: 0.4792831132,
        565: 0.4978182761,
        570: 0.5165259668,
        575: 0.5353890802,
        580: 0.5543905413,
        585: 0.5735133454,
        590: 0.5927405961,
        595: 0.6120555405,
        600: 0.631441603,
        605: 0.6508824158,
        610: 0.6703618481,
        615: 0.6898640329,
        620: 0.7093733909,
        625: 0.7288746533,
        630: 0.727218309,
        635: 0.6673288668,
        640: 0.5782714577,
        645: 0.4731952106,
        650: 0.3656500707,
        655: 0.2668134517,
        660: 0.1838512212,
        665: 0.1196305377,
        670: 0.0735079529,
        675: 0.042652387,
        680: 0.0233705585,
        685: 0.0120923742,
        690: 0.0059084125,
        695: 0.0027261313,
        700: 0.0011877895,
        705: 0.0004887074,
        710: 0.0001898782,
        715: 6.96655400552185e-05,
        720: 2.41366872471511e-05,
        725: 7.89685304340932e-06,
        730: 2.43976045192516e-06,
        735: 7.11798480245962e-07,
        740: 1.96102740877049e-07,
        745: 5.1e-08,
        750: 1.25339250295488e-08,
        755: 2.90779586286737e-09,
        760: 6.37026463166956e-10,
        765: 1.31785543464018e-10,
        770: 2.57451133233467e-11,
        775: 4.74939818412281e-12,
        780: 8.27368713020501e-13},
    'PHOS-3': {
        380: 2.51463414652205e-05,
        385: 6.85384929115295e-05,
        390: 0.0001764051,
        395: 0.0004287501,
        400: 0.0009840433,
        405: 0.0021327547,
        410: 0.0043650008,
        415: 0.0084361541,
        420: 0.0153964789,
        425: 0.0265347535,
        430: 0.0431842561,
        435: 0.0663670579,
        440: 0.0963155539,
        445: 0.1319948618,
        450: 0.1708182873,
        455: 0.2087509393,
        460: 0.24090136,
        465: 0.2625226645,
        470: 0.2701538077,
        475: 0.3017952768,
        480: 0.3181657285,
        485: 0.3348793477,
        490: 0.3519178926,
        495: 0.3692626059,
        500: 0.3868942289,
        505: 0.4047924428,
        510: 0.4229309502,
        515: 0.441246613,
        520: 0.4594906524,
        525: 0.4766862941,
        530: 0.4897060796,
        535: 0.4909188905,
        540: 0.4671077104,
        545: 0.4049706628,
        550: 0.306139489,
        555: 0.2015572822,
        560: 0.1440339718,
        565: 0.1735241155,
        570: 0.284580561,
        575: 0.4309947871,
        580: 0.5623950573,
        585: 0.6544836496,
        590: 0.7102988663,
        595: 0.7440476235,
        600: 0.767857534,
        605: 0.7881667145,
        610: 0.8074348626,
        615: 0.826369227,
        620: 0.845120243,
        625: 0.8393172408,
        630: 0.7702007496,
        635: 0.6674188109,
        640: 0.5461471669,
        645: 0.4220245013,
        650: 0.3079515424,
        655: 0.2121992478,
        660: 0.13807722,
        665: 0.0848431857,
        670: 0.0492298642,
        675: 0.0269747263,
        680: 0.0139573251,
        685: 0.0068196813,
        690: 0.0031466084,
        695: 0.0013710019,
        700: 0.0005640923,
        705: 0.0002191689,
        710: 8.04126871697911e-05,
        715: 2.78603705865815e-05,
        720: 9.1151945127071e-06,
        725: 2.81618856366991e-06,
        730: 8.21626160060091e-07,
        735: 2.26362001084354e-07,
        740: 5.88910813281106e-08,
        745: 1.44681256222285e-08,
        750: 3.35653941463681e-09,
        755: 7.35339625846445e-10,
        760: 1.52125117385059e-10,
        765: 2.97187556394679e-11,
        770: 5.48247997165375e-12,
        775: 9.55081042264917e-13,
        780: 1.57115898612229e-13},
    'PHOS-4': {
        380: 2.65174294164563e-05,
        385: 7.20892236476949e-05,
        390: 0.0001850652,
        395: 0.0004486356,
        400: 0.001027012,
        405: 0.0022200743,
        410: 0.004531744,
        415: 0.0087349869,
        420: 0.0158981906,
        425: 0.0273215926,
        430: 0.0443316913,
        435: 0.0679109808,
        440: 0.0982059987,
        445: 0.1340440575,
        450: 0.1726589283,
        455: 0.2098269134,
        460: 0.2405133288,
        465: 0.2599404435,
        470: 0.2968669942,
        475: 0.3103891625,
        480: 0.3230698,
        485: 0.3346221126,
        490: 0.3447613147,
        495: 0.3532308214,
        500: 0.3598344157,
        505: 0.3644719777,
        510: 0.3671749961,
        515: 0.368136989,
        520: 0.3677334186,
        525: 0.3665259719,
        530: 0.3652473273,
        535: 0.3647647135,
        540: 0.3660234509,
        545: 0.3699748157,
        550: 0.3774954254,
        555: 0.3893073484,
        560: 0.405908814,
        565: 0.4275245273,
        570: 0.4540822183,
        575: 0.4852185591,
        580: 0.5203135462,
        585: 0.5585485887,
        590: 0.5989805223,
        595: 0.6406220815,
        600: 0.6825192092,
        605: 0.7238168836,
        610: 0.7638075399,
        615: 0.8019591443,
        620: 0.8379229785,
        625: 0.8469158152,
        630: 0.788207208,
        635: 0.690634909,
        640: 0.5699908443,
        645: 0.4432968286,
        650: 0.3250191694,
        655: 0.2247345252,
        660: 0.1465926143,
        665: 0.0902291357,
        670: 0.0524158049,
        675: 0.0287426457,
        680: 0.0148796085,
        685: 0.0072726716,
        690: 0.0033563,
        695: 0.0014625486,
        700: 0.0006018039,
        705: 0.0002338315,
        710: 8.57945671424698e-05,
        715: 2.97254474377703e-05,
        720: 9.72547785760084e-06,
        725: 3.0047520635375e-06,
        730: 8.76641820226947e-07,
        735: 2.41519377899335e-07,
        740: 6.28345135673037e-08,
        745: 1.54369371558642e-08,
        750: 3.58129984345344e-09,
        755: 7.84579460690639e-10,
        760: 1.62311729956244e-10,
        765: 3.17087855185187e-11,
        770: 5.84959831217044e-12,
        775: 1.01903527241427e-12,
        780: 1.67636709023523e-13},
    'Phosphor LED YAG': {
        380: 0.0007674402,
        385: 0.0007988052,
        390: 0.000948151,
        395: 0.0015352484,
        400: 0.0021463554,
        405: 0.0034039975,
        410: 0.0055670635,
        415: 0.009578223,
        420: 0.0167538733,
        425: 0.0299676462,
        430: 0.0524114993,
        435: 0.0905289608,
        440: 0.1532631997,
        445: 0.2579687799,
        450: 0.4231985998,
        455: 0.6524548723,
        460: 0.8889410586,
        465: 1.0,
        470: 0.9085683812,
        475: 0.7151250821,
        480: 0.5571559436,
        485: 0.4460579855,
        490: 0.3591474075,
        495: 0.2947898077,
        500: 0.2603399781,
        505: 0.2508517042,
        510: 0.2570335391,
        515: 0.2787399079,
        520: 0.3067370794,
        525: 0.3365625609,
        530: 0.3641840935,
        535: 0.388760403,
        540: 0.4073510497,
        545: 0.4195421493,
        550: 0.4294869591,
        555: 0.4348944502,
        560: 0.4365289504,
        565: 0.4349851996,
        570: 0.4326880095,
        575: 0.4258197301,
        580: 0.4196592679,
        585: 0.4106773297,
        590: 0.3972204766,
        595: 0.382521167,
        600: 0.3694023709,
        605: 0.3517174391,
        610: 0.3339347997,
        615: 0.3136523994,
        620: 0.2949462656,
        625: 0.2740545346,
        630: 0.2560598429,
        635: 0.2373675514,
        640: 0.2182257334,
        645: 0.2000860845,
        650: 0.1812754622,
        655: 0.1643825021,
        660: 0.1476537604,
        665: 0.1315113277,
        670: 0.1154484901,
        675: 0.1030116409,
        680: 0.0918100578,
        685: 0.0804370486,
        690: 0.0710498842,
        695: 0.0632138504,
        700: 0.0555426143,
        705: 0.0494042694,
        710: 0.0435522579,
        715: 0.038346114,
        720: 0.0334050926,
        725: 0.0295323862,
        730: 0.026015539,
        735: 0.0225898129,
        740: 0.0202020262,
        745: 0.0176330719,
        750: 0.015594848,
        755: 0.0135268479,
        760: 0.0117904266,
        765: 0.0102635827,
        770: 0.0091149715,
        775: 0.0079991914,
        780: 0.0070995878}}

LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS = CaseInsensitiveMapping(
    {'3-LED-1 (457/540/605)': SpectralPowerDistribution(
        '3-LED-1 (457/540/605)',
        LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
            '3-LED-1 (457/540/605)')),
     '3-LED-2 (473/545/616)': SpectralPowerDistribution(
         '3-LED-2 (473/545/616)',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '3-LED-2 (473/545/616)')),
     '3-LED-2 Yellow': SpectralPowerDistribution(
         '3-LED-2 Yellow',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '3-LED-2 Yellow')),
     '3-LED-3 (465/546/614)': SpectralPowerDistribution(
         '3-LED-3 (465/546/614)',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '3-LED-3 (465/546/614)')),
     '3-LED-4 (455/547/623)': SpectralPowerDistribution(
         '3-LED-4 (455/547/623)',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '3-LED-4 (455/547/623)')),
     '4-LED No Yellow': SpectralPowerDistribution(
         '4-LED No Yellow',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '4-LED No Yellow')),
     '4-LED Yellow': SpectralPowerDistribution(
         '4-LED Yellow',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '4-LED Yellow')),
     '4-LED-1 (461/526/576/624)': SpectralPowerDistribution(
         '4-LED-1 (461/526/576/624)',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '4-LED-1 (461/526/576/624)')),
     '4-LED-2 (447/512/573/627)': SpectralPowerDistribution(
         '4-LED-2 (447/512/573/627)',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             '4-LED-2 (447/512/573/627)')),
     'Luxeon WW 2880': SpectralPowerDistribution(
         'Luxeon WW 2880',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'Luxeon WW 2880')),
     'PHOS-1': SpectralPowerDistribution(
         'PHOS-1',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'PHOS-1')),
     'PHOS-2': SpectralPowerDistribution(
         'PHOS-2',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'PHOS-2')),
     'PHOS-3': SpectralPowerDistribution(
         'PHOS-3',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'PHOS-3')),
     'PHOS-4': SpectralPowerDistribution(
         'PHOS-4',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'PHOS-4')),
     'Phosphor LED YAG': SpectralPowerDistribution(
         'Phosphor LED YAG',
         LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS_DATA.get(
             'Phosphor LED YAG'))})
"""
LED light sources from *NIST* *NIST CQS simulation 7.4.xls* spreadsheet. [2]_

LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS : CaseInsensitiveMapping
    {'3-LED-1 (457/540/605)', '3-LED-2 (473/545/616)', '3-LED-2 Yellow',
    '3-LED-3 (465/546/614)', '3-LED-4 (455/547/623)', '4-LED No Yellow',
    '4-LED Yellow', '4-LED-1 (461/526/576/624)', '4-LED-2 (447/512/573/627)',
    'Luxeon WW 2880', 'PHOS-1', 'PHOS-2', 'PHOS-3', 'PHOS-4',
    'Phosphor LED YAG'}
"""

LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA = {
    '60 A/W (Soft White)': {
        380: 0.0221129274,
        385: 0.0257550191,
        390: 0.0303031416,
        395: 0.0402040843,
        400: 0.0502272809,
        405: 0.0596287164,
        410: 0.0685435436,
        415: 0.077222055,
        420: 0.0842087617,
        425: 0.0913199149,
        430: 0.0997370726,
        435: 0.1084944259,
        440: 0.1161691811,
        445: 0.1253379906,
        450: 0.1346917959,
        455: 0.1450470777,
        460: 0.1556685203,
        465: 0.1666161272,
        470: 0.1769997332,
        475: 0.1890140662,
        480: 0.1994984692,
        485: 0.2107503575,
        490: 0.2230259107,
        495: 0.2349042653,
        500: 0.2477963468,
        505: 0.2611051126,
        510: 0.2735574457,
        515: 0.2873463379,
        520: 0.3007511829,
        525: 0.3148017537,
        530: 0.3286265485,
        535: 0.3429364606,
        540: 0.3581597726,
        545: 0.3732534755,
        550: 0.3883420369,
        555: 0.403987105,
        560: 0.4203674722,
        565: 0.4365160648,
        570: 0.4523056904,
        575: 0.4682869076,
        580: 0.4841530664,
        585: 0.5003725619,
        590: 0.5164160981,
        595: 0.5325415882,
        600: 0.5488244682,
        605: 0.5647150329,
        610: 0.5807535747,
        615: 0.5966998519,
        620: 0.6119241813,
        625: 0.627620854,
        630: 0.6424901113,
        635: 0.6578064624,
        640: 0.6733141181,
        645: 0.6882633218,
        650: 0.7030623147,
        655: 0.7181948129,
        660: 0.7331525555,
        665: 0.747833052,
        670: 0.7640991255,
        675: 0.7810516508,
        680: 0.7982514269,
        685: 0.8127518501,
        690: 0.8278508335,
        695: 0.8406662389,
        700: 0.8528341711,
        705: 0.8651898991,
        710: 0.8808045146,
        715: 0.892185268,
        720: 0.9049681735,
        725: 0.9194588124,
        730: 0.9303931443,
        735: 0.9382116765,
        740: 0.9519940858,
        745: 0.9669876832,
        750: 0.9766525623,
        755: 0.9854043089,
        760: 1.0,
        765: 0.9669876832,
        770: 0.9766525623,
        775: 0.9854043089,
        780: 1.0},
    'C100S54 (HPS)': {
        380: 0.0090968865,
        385: 0.0099177519,
        390: 0.0106049886,
        395: 0.0118222306,
        400: 0.0130802917,
        405: 0.0162752042,
        410: 0.0154438614,
        415: 0.0171300107,
        420: 0.019300898,
        425: 0.0215427439,
        430: 0.023802917,
        435: 0.0310645301,
        440: 0.0308492469,
        445: 0.0271089809,
        450: 0.0473087634,
        455: 0.0296505586,
        460: 0.0211678708,
        465: 0.0696258418,
        470: 0.0607626697,
        475: 0.0261534996,
        480: 0.0093411354,
        485: 0.0076708609,
        490: 0.0137424517,
        495: 0.1034773639,
        500: 0.1711343827,
        505: 0.0157226389,
        510: 0.0116384986,
        515: 0.0453171888,
        520: 0.0146582087,
        525: 0.0090060877,
        530: 0.0098520672,
        535: 0.0104241916,
        540: 0.0125060396,
        545: 0.0242909072,
        550: 0.0473687103,
        555: 0.0627444986,
        560: 0.0770812304,
        565: 0.3741275185,
        570: 0.772618931,
        575: 0.2356295989,
        580: 0.4404894198,
        585: 0.7918076664,
        590: 0.4382078409,
        595: 1.0,
        600: 0.7325059828,
        605: 0.4736253635,
        610: 0.3283190711,
        615: 0.4094756278,
        620: 0.2398795364,
        625: 0.1612392829,
        630: 0.1372507355,
        635: 0.1200033997,
        640: 0.1074424158,
        645: 0.0978274984,
        650: 0.0879659781,
        655: 0.0844149672,
        660: 0.0764205503,
        665: 0.0729728994,
        670: 0.0819054466,
        675: 0.0772196909,
        680: 0.0593104899,
        685: 0.0444799728,
        690: 0.036451407,
        695: 0.0339297355,
        700: 0.0337682865,
        705: 0.0336553942,
        710: 0.0298482475,
        715: 0.0284354813,
        720: 0.0282247418,
        725: 0.0266439404,
        730: 0.0264402189,
        735: 0.0269007884,
        740: 0.0277447911,
        745: 0.0290665851,
        750: 0.0318508294,
        755: 0.030043758,
        760: 0.0343141879,
        765: 0.0290665851,
        770: 0.0318508294,
        775: 0.030043758,
        780: 0.0343141879},
    'C100S54C (HPS)': {
        380: 0.0,
        385: 0.0091384562,
        390: 0.0210275171,
        395: 0.0231912726,
        400: 0.0253773899,
        405: 0.0261423984,
        410: 0.0285988157,
        415: 0.0341146113,
        420: 0.0397250713,
        425: 0.0427896702,
        430: 0.0510652364,
        435: 0.0618978774,
        440: 0.0628896188,
        445: 0.0682864066,
        450: 0.0903139409,
        455: 0.0699544686,
        460: 0.06677602,
        465: 0.1322128264,
        470: 0.1081492818,
        475: 0.061416055,
        480: 0.0386796956,
        485: 0.0453551033,
        490: 0.0618227042,
        495: 0.2000545826,
        500: 0.2041763817,
        505: 0.0645323922,
        510: 0.0710887037,
        515: 0.1090452304,
        520: 0.0660379682,
        525: 0.0663666564,
        530: 0.0707134625,
        535: 0.0770449812,
        540: 0.0901472313,
        545: 0.1480823084,
        550: 0.3086733458,
        555: 0.3697682599,
        560: 0.4374965183,
        565: 0.8355054906,
        570: 1.0,
        575: 0.7933887682,
        580: 0.6343186245,
        585: 0.1461176703,
        590: 0.0711998311,
        595: 0.3361999098,
        600: 0.7571333272,
        605: 0.8808780535,
        610: 0.8414420095,
        615: 0.8592433929,
        620: 0.6516002626,
        625: 0.57181946,
        630: 0.5123980617,
        635: 0.4649878759,
        640: 0.424047865,
        645: 0.3912030788,
        650: 0.3539753745,
        655: 0.3414936119,
        660: 0.3116194982,
        665: 0.3140716238,
        670: 0.3263098522,
        675: 0.2976349896,
        680: 0.2424455172,
        685: 0.2002338948,
        690: 0.1788421275,
        695: 0.1687568712,
        700: 0.1621922015,
        705: 0.1559086034,
        710: 0.1542434667,
        715: 0.1488947756,
        720: 0.1429461515,
        725: 0.1404206219,
        730: 0.1378455835,
        735: 0.1397081601,
        740: 0.1392581783,
        745: 0.1421969643,
        750: 0.1445350862,
        755: 0.1425608179,
        760: 0.1393291025,
        765: 0.1421969643,
        770: 0.1445350862,
        775: 0.1425608179,
        780: 0.1393291025},
    'F32T8/TL830 (Triphosphor)': {
        380: 0.0039380786,
        385: 0.0021247895,
        390: 0.001092449,
        395: 0.0007445,
        400: 0.0192099017,
        405: 0.1139377991,
        410: 0.0160747568,
        415: 0.0094419469,
        420: 0.0113913934,
        425: 0.0145512605,
        430: 0.017745886,
        435: 0.3622908498,
        440: 0.0992985909,
        445: 0.026489573,
        450: 0.026804218,
        455: 0.0259086428,
        460: 0.0239039496,
        465: 0.0232785752,
        470: 0.0204368091,
        475: 0.017661634,
        480: 0.0297504447,
        485: 0.1370820005,
        490: 0.1766929612,
        495: 0.1213745646,
        500: 0.0577517168,
        505: 0.0216507882,
        510: 0.0227907207,
        515: 0.0242234631,
        520: 0.020243648,
        525: 0.016386678,
        530: 0.0183189837,
        535: 0.0471984967,
        540: 0.4197572764,
        545: 0.9510310959,
        550: 0.3140955063,
        555: 0.0878855654,
        560: 0.0181958459,
        565: 0.0120175611,
        570: 0.0104226055,
        575: 0.0497086487,
        580: 0.1638317443,
        585: 0.231662164,
        590: 0.2135278199,
        595: 0.1364610506,
        600: 0.1042701698,
        605: 0.0869827945,
        610: 1.0,
        615: 0.7612500545,
        620: 0.216521496,
        625: 0.1701831953,
        630: 0.1979345653,
        635: 0.0658488017,
        640: 0.0247309653,
        645: 0.0280871898,
        650: 0.0573097419,
        655: 0.0305788824,
        660: 0.0285740516,
        665: 0.0303813331,
        670: 0.0202105545,
        675: 0.0162600152,
        680: 0.0185130408,
        685: 0.0249383286,
        690: 0.0277838499,
        695: 0.0192753998,
        700: 0.0070923054,
        705: 0.0653809949,
        710: 0.1115066116,
        715: 0.0327332173,
        720: 0.0032297275,
        725: 0.0026943997,
        730: 0.0022714191,
        735: 0.001604085,
        740: 0.0026846055,
        745: 0.0021757603,
        750: 0.001634071,
        755: 0.0007446283,
        760: 0.0,
        765: 0.0021757603,
        770: 0.001634071,
        775: 0.0007446283,
        780: 0.0},
    'F32T8/TL835 (Triphosphor)': {
        380: 0.0044539119,
        385: 0.002724699,
        390: 0.0014395978,
        395: 0.0014605529,
        400: 0.0207581727,
        405: 0.1210709316,
        410: 0.0219292694,
        415: 0.0174448705,
        420: 0.023866461,
        425: 0.0327628639,
        430: 0.0407306763,
        435: 0.4093912201,
        440: 0.137698125,
        445: 0.0630845716,
        450: 0.0643782835,
        455: 0.062649006,
        460: 0.0583216524,
        465: 0.0535988786,
        470: 0.0473229913,
        475: 0.0405633566,
        480: 0.0502169749,
        485: 0.1652071359,
        490: 0.2079782988,
        495: 0.135460999,
        500: 0.0713754439,
        505: 0.0395849992,
        510: 0.0457612606,
        515: 0.0485971774,
        520: 0.0411482338,
        525: 0.0330358554,
        530: 0.0306616327,
        535: 0.0556184344,
        540: 0.4453236174,
        545: 1.0,
        550: 0.3331010969,
        555: 0.0949308279,
        560: 0.0201333024,
        565: 0.0126103069,
        570: 0.0116220609,
        575: 0.0520317311,
        580: 0.1708824084,
        585: 0.2462294879,
        590: 0.2197083915,
        595: 0.1308049775,
        600: 0.0976296501,
        605: 0.0803295312,
        610: 0.8956142751,
        615: 0.684212366,
        620: 0.2132313644,
        625: 0.1613348583,
        630: 0.1756409019,
        635: 0.059616781,
        640: 0.0227255888,
        645: 0.0266411834,
        650: 0.0523557397,
        655: 0.028435581,
        660: 0.0262840653,
        665: 0.0295026461,
        670: 0.0210918604,
        675: 0.0171808329,
        680: 0.0186356851,
        685: 0.0231132761,
        690: 0.0256224661,
        695: 0.0189429063,
        700: 0.0070741647,
        705: 0.059523812,
        710: 0.1035556141,
        715: 0.0314590076,
        720: 0.0034133373,
        725: 0.0030514142,
        730: 0.0032200361,
        735: 0.0027086362,
        740: 0.0030232116,
        745: 0.0028316149,
        750: 0.0020137199,
        755: 0.0020856503,
        760: 0.0015424431,
        765: 0.0028316149,
        770: 0.0020137199,
        775: 0.0020856503,
        780: 0.0015424431},
    'F32T8/TL841 (Triphosphor)': {
        380: 0.0040371862,
        385: 0.0024796605,
        390: 0.0014675082,
        395: 0.0017094705,
        400: 0.0194464146,
        405: 0.1113191989,
        410: 0.0221906642,
        415: 0.0198378149,
        420: 0.0278535356,
        425: 0.0386632307,
        430: 0.0491088467,
        435: 0.4033726577,
        440: 0.1462448079,
        445: 0.0765793134,
        450: 0.0783752768,
        455: 0.0765699536,
        460: 0.0716157566,
        465: 0.0652221269,
        470: 0.0575640594,
        475: 0.0495912359,
        480: 0.0580833912,
        485: 0.1729479226,
        490: 0.214981587,
        495: 0.1414297566,
        500: 0.076677946,
        505: 0.0459631063,
        510: 0.0533211524,
        515: 0.0561922859,
        520: 0.047524495,
        525: 0.0382438791,
        530: 0.0340614125,
        535: 0.0568088778,
        540: 0.4521843869,
        545: 1.0,
        550: 0.3343639992,
        555: 0.0966743072,
        560: 0.0218972364,
        565: 0.0137302425,
        570: 0.0127985536,
        575: 0.0517025421,
        580: 0.1641501786,
        585: 0.2391850676,
        590: 0.2065691065,
        595: 0.1144602882,
        600: 0.0838148826,
        605: 0.0689315799,
        610: 0.7484025422,
        615: 0.5753970025,
        620: 0.2007740113,
        625: 0.1465774558,
        630: 0.1487120061,
        635: 0.0504493139,
        640: 0.0200869947,
        645: 0.0243029892,
        650: 0.0456433333,
        655: 0.0248657706,
        660: 0.0224547425,
        665: 0.026028144,
        670: 0.019646805,
        675: 0.0161739376,
        680: 0.0178745527,
        685: 0.0199037625,
        690: 0.0212176607,
        695: 0.0151921729,
        700: 0.005492622,
        705: 0.0494481851,
        710: 0.084628617,
        715: 0.0248918705,
        720: 0.0024310469,
        725: 0.0021193708,
        730: 0.002230245,
        735: 0.0013854078,
        740: 0.0027224413,
        745: 0.001916888,
        750: 0.0010922856,
        755: 0.0004971904,
        760: 0.0,
        765: 0.001916888,
        770: 0.0010922856,
        775: 0.0004971904,
        780: 0.0},
    'F32T8/TL850 (Triphosphor)': {
        380: 0.0034742373,
        385: 0.0023983574,
        390: 0.001834188,
        395: 0.0023901197,
        400: 0.0220040686,
        405: 0.1180075733,
        410: 0.0291435583,
        415: 0.032835332,
        420: 0.0494204656,
        425: 0.0704090213,
        430: 0.0919757671,
        435: 0.4759819932,
        440: 0.2107111307,
        445: 0.1424846279,
        450: 0.1460627495,
        455: 0.1426440643,
        460: 0.1335116827,
        465: 0.1205321335,
        470: 0.1057579052,
        475: 0.0905118061,
        480: 0.0917282783,
        485: 0.1979628969,
        490: 0.2337421933,
        495: 0.1603472835,
        500: 0.1004299825,
        505: 0.0807447866,
        510: 0.1006438774,
        515: 0.1061748793,
        520: 0.0898937926,
        525: 0.0715927147,
        530: 0.0595809292,
        535: 0.0748589295,
        540: 0.45702242,
        545: 1.0,
        550: 0.3354233844,
        555: 0.0977665583,
        560: 0.0239234294,
        565: 0.0148187466,
        570: 0.0132140928,
        575: 0.0521011578,
        580: 0.1601504544,
        585: 0.2303028023,
        590: 0.1954767556,
        595: 0.1054965252,
        600: 0.0760367817,
        605: 0.063142587,
        610: 0.6735534486,
        615: 0.5155212627,
        620: 0.1919248289,
        625: 0.138014472,
        630: 0.1357638735,
        635: 0.0464654521,
        640: 0.0193124675,
        645: 0.0237533091,
        650: 0.0433201748,
        655: 0.023763367,
        660: 0.0218795456,
        665: 0.0255443739,
        670: 0.0199551028,
        675: 0.0165982045,
        680: 0.0181770644,
        685: 0.0197991664,
        690: 0.020663585,
        695: 0.0152194265,
        700: 0.0062642767,
        705: 0.045876638,
        710: 0.0777608514,
        715: 0.0236979106,
        720: 0.0035880471,
        725: 0.0029197566,
        730: 0.0031527922,
        735: 0.0023984839,
        740: 0.002695943,
        745: 0.0029392159,
        750: 0.0029753244,
        755: 0.0019264764,
        760: 0.0031394812,
        765: 0.0029392159,
        770: 0.0029753244,
        775: 0.0019264764,
        780: 0.0031394812},
    'F32T8/TL865 /PLUS (Triphosphor)': {
        380: 0.0043823207,
        385: 0.0033077391,
        390: 0.0028637083,
        395: 0.0034943154,
        400: 0.0225614013,
        405: 0.1105476556,
        410: 0.0322815394,
        415: 0.0415501393,
        420: 0.063551369,
        425: 0.0909093192,
        430: 0.1201837671,
        435: 0.4825506923,
        440: 0.2421687535,
        445: 0.1828880978,
        450: 0.187122143,
        455: 0.1824460215,
        460: 0.1697975179,
        465: 0.1520867073,
        470: 0.1326169314,
        475: 0.1126467277,
        480: 0.1101510479,
        485: 0.2153209287,
        490: 0.2482022546,
        495: 0.1728467413,
        500: 0.1133590119,
        505: 0.0986272158,
        510: 0.1252563625,
        515: 0.1332386255,
        520: 0.1134850002,
        525: 0.0904252796,
        530: 0.073517476,
        535: 0.0842961431,
        540: 0.4714323093,
        545: 1.0,
        550: 0.3368727011,
        555: 0.0996714846,
        560: 0.0257786896,
        565: 0.0157759505,
        570: 0.0142415938,
        575: 0.0515815998,
        580: 0.1540021672,
        585: 0.2241079847,
        590: 0.184008283,
        595: 0.0923404265,
        600: 0.064830008,
        605: 0.0543302323,
        610: 0.5551932941,
        615: 0.4266333793,
        620: 0.1821410168,
        625: 0.1261687212,
        630: 0.1147738242,
        635: 0.0392044079,
        640: 0.017662259,
        645: 0.0223032575,
        650: 0.0385153629,
        655: 0.0213258032,
        660: 0.0198621019,
        665: 0.0240749219,
        670: 0.0196624734,
        675: 0.0167483225,
        680: 0.0187533176,
        685: 0.0185158469,
        690: 0.018830799,
        695: 0.0142610066,
        700: 0.0069909868,
        705: 0.0407048656,
        710: 0.0661393532,
        715: 0.0217274474,
        720: 0.0051763784,
        725: 0.0054575445,
        730: 0.0054516295,
        735: 0.0064123646,
        740: 0.007013266,
        745: 0.0072031048,
        750: 0.007504532,
        755: 0.0070549007,
        760: 0.0062780959,
        765: 0.0072031048,
        770: 0.007504532,
        775: 0.0070549007,
        780: 0.0062780959},
    'F34/CW/RS/EW (Cool White FL)': {
        380: 0.0296401248,
        385: 0.037356956,
        390: 0.0467012986,
        395: 0.0545847294,
        400: 0.1135736349,
        405: 0.3808968841,
        410: 0.1166414503,
        415: 0.0988773637,
        420: 0.1109995589,
        425: 0.1229772316,
        430: 0.1265901829,
        435: 1.0,
        440: 0.3402739287,
        445: 0.1602563678,
        450: 0.1688941102,
        455: 0.1758410832,
        460: 0.1823124475,
        465: 0.1874629248,
        470: 0.1909802309,
        475: 0.19418079,
        480: 0.1977063586,
        485: 0.1956831996,
        490: 0.1920120467,
        495: 0.1970147162,
        500: 0.1959552584,
        505: 0.1935160051,
        510: 0.1934786916,
        515: 0.1976455665,
        520: 0.2073342723,
        525: 0.2251537829,
        530: 0.252483741,
        535: 0.2788780141,
        540: 0.329824987,
        545: 0.8645561518,
        550: 0.554863976,
        555: 0.4898395315,
        560: 0.5485363882,
        565: 0.5697898165,
        570: 0.5950503867,
        575: 0.6528683853,
        580: 0.6680940942,
        585: 0.5940743859,
        590: 0.5713086179,
        595: 0.5409337618,
        600: 0.4814640889,
        605: 0.4460707931,
        610: 0.4067532871,
        615: 0.3594850161,
        620: 0.3120166941,
        625: 0.2713329441,
        630: 0.2341071175,
        635: 0.2013815866,
        640: 0.1735932489,
        645: 0.1472832487,
        650: 0.1256441874,
        655: 0.1066229766,
        660: 0.0903719665,
        665: 0.0765404446,
        670: 0.0655387796,
        675: 0.0556335046,
        680: 0.0478890878,
        685: 0.0411710487,
        690: 0.037716897,
        695: 0.0305525144,
        700: 0.0258689043,
        705: 0.0238827075,
        710: 0.0214675621,
        715: 0.0181129739,
        720: 0.0171890065,
        725: 0.0165440398,
        730: 0.0131018239,
        735: 0.0123092301,
        740: 0.0109826665,
        745: 0.0114365646,
        750: 0.0102565619,
        755: 0.0107649726,
        760: 0.0787313519,
        765: 0.0114365646,
        770: 0.0102565619,
        775: 0.0107649726,
        780: 0.0787313519},
    'F34T12/LW/RS /EW': {
        380: 0.0138848383,
        385: 0.0171879452,
        390: 0.0227765198,
        395: 0.027505215,
        400: 0.0752379474,
        405: 0.3076328626,
        410: 0.0839002976,
        415: 0.0736568783,
        420: 0.094751759,
        425: 0.1187820125,
        430: 0.1382036806,
        435: 1.0,
        440: 0.3694732556,
        445: 0.1990428182,
        450: 0.2053275681,
        455: 0.2051279183,
        460: 0.2000384871,
        465: 0.1904478584,
        470: 0.1793075582,
        475: 0.1668603797,
        480: 0.1562015196,
        485: 0.149707559,
        490: 0.1435723367,
        495: 0.1322450142,
        500: 0.1233031149,
        505: 0.1204208272,
        510: 0.1234975304,
        515: 0.134318215,
        520: 0.1541719542,
        525: 0.1861260371,
        530: 0.2310164203,
        535: 0.2892904604,
        540: 0.3668337864,
        545: 0.8867757955,
        550: 0.6087014152,
        555: 0.5695756218,
        560: 0.6199534205,
        565: 0.6562808287,
        570: 0.674460303,
        575: 0.7187743351,
        580: 0.7228108672,
        585: 0.6333188074,
        590: 0.5876924651,
        595: 0.5340168668,
        600: 0.4798324565,
        605: 0.4251412723,
        610: 0.3897858071,
        615: 0.3374283272,
        620: 0.2842580341,
        625: 0.2411544413,
        630: 0.2049966527,
        635: 0.1715543773,
        640: 0.1430742195,
        645: 0.1205364592,
        650: 0.1014617091,
        655: 0.0835511732,
        660: 0.0702517604,
        665: 0.0589828511,
        670: 0.0502305102,
        675: 0.0412751645,
        680: 0.0349900959,
        685: 0.029427307,
        690: 0.0280600741,
        695: 0.0215273175,
        700: 0.0179181411,
        705: 0.0166605086,
        710: 0.0143728475,
        715: 0.0121928999,
        720: 0.009649029,
        725: 0.0077380228,
        730: 0.0074911718,
        735: 0.0061265527,
        740: 0.0056868169,
        745: 0.0048038975,
        750: 0.005863369,
        755: 0.0077827634,
        760: 0.0553820445,
        765: 0.0048038975,
        770: 0.005863369,
        775: 0.0077827634,
        780: 0.0553820445},
    'F34T12WW/RS /EW (Warm White FL)': {
        380: 0.0183484283,
        385: 0.0208397838,
        390: 0.0263238473,
        395: 0.0306592764,
        400: 0.0858568428,
        405: 0.3480702794,
        410: 0.0738426812,
        415: 0.0510478385,
        420: 0.0573551487,
        425: 0.0630044835,
        430: 0.066094768,
        435: 1.0,
        440: 0.2808758832,
        445: 0.0845477925,
        450: 0.0886660961,
        455: 0.0925509194,
        460: 0.0960558012,
        465: 0.0988932529,
        470: 0.10103835,
        475: 0.1021411401,
        480: 0.1022015621,
        485: 0.1015821574,
        490: 0.1022912413,
        495: 0.1004712706,
        500: 0.0990989351,
        505: 0.0992901353,
        510: 0.1012042093,
        515: 0.1068412295,
        520: 0.1188400824,
        525: 0.1384844021,
        530: 0.1685760845,
        535: 0.2108284301,
        540: 0.2595929586,
        545: 0.8292231902,
        550: 0.5258526196,
        555: 0.4973028957,
        560: 0.5816710875,
        565: 0.6598568545,
        570: 0.7244036111,
        575: 0.8236679678,
        580: 0.8658776093,
        585: 0.8033235537,
        590: 0.7870124681,
        595: 0.753849179,
        600: 0.7068119021,
        605: 0.6514878532,
        610: 0.5918101178,
        615: 0.5292842525,
        620: 0.4678152456,
        625: 0.4103694433,
        630: 0.3552523607,
        635: 0.3056965652,
        640: 0.2621855754,
        645: 0.2250804317,
        650: 0.1912370778,
        655: 0.1611917424,
        660: 0.1369045554,
        665: 0.1159050285,
        670: 0.098304749,
        675: 0.0828024517,
        680: 0.0695111365,
        685: 0.0589762767,
        690: 0.0527022717,
        695: 0.043108817,
        700: 0.0369743858,
        705: 0.0317352438,
        710: 0.0278560368,
        715: 0.0245954652,
        720: 0.0225711563,
        725: 0.021051978,
        730: 0.0184096802,
        735: 0.0162207543,
        740: 0.0165685445,
        745: 0.0162500215,
        750: 0.0154892424,
        755: 0.0161230441,
        760: 0.0941165395,
        765: 0.0162500215,
        770: 0.0154892424,
        775: 0.0161230441,
        780: 0.0941165395},
    'F40/C50 (Broadband FL)': {
        380: 0.0279518444,
        385: 0.0297109166,
        390: 0.032957064,
        395: 0.0412757705,
        400: 0.0990617866,
        405: 0.3696541019,
        410: 0.0986601613,
        415: 0.0812466833,
        420: 0.0906994303,
        425: 0.1036282061,
        430: 0.1181453284,
        435: 1.0,
        440: 0.3372487627,
        445: 0.1672744623,
        450: 0.184141174,
        455: 0.1989811784,
        460: 0.2170193003,
        465: 0.2312634638,
        470: 0.2426571515,
        475: 0.2548000175,
        480: 0.267144862,
        485: 0.2808903845,
        490: 0.2899766591,
        495: 0.2916870852,
        500: 0.2925433684,
        505: 0.2936518962,
        510: 0.2942380291,
        515: 0.2952253517,
        520: 0.2951815682,
        525: 0.2935205401,
        530: 0.2938378234,
        535: 0.2958383782,
        540: 0.3074362858,
        545: 0.8032006785,
        550: 0.4171194312,
        555: 0.29843591,
        560: 0.2967132089,
        565: 0.2982998528,
        570: 0.3004378841,
        575: 0.362090658,
        580: 0.3926175085,
        585: 0.3200470773,
        590: 0.3223944373,
        595: 0.3228307249,
        600: 0.3237173583,
        605: 0.3227568297,
        610: 0.3846320532,
        615: 0.3653931686,
        620: 0.3233813491,
        625: 0.319519448,
        630: 0.3155369443,
        635: 0.3008717821,
        640: 0.2907922444,
        645: 0.2800489348,
        650: 0.2735352537,
        655: 0.2640275617,
        660: 0.2486541931,
        665: 0.2342873817,
        670: 0.2210471159,
        675: 0.2069917819,
        680: 0.1917764432,
        685: 0.1820940416,
        690: 0.1791407699,
        695: 0.1534225814,
        700: 0.1383813611,
        705: 0.1349911532,
        710: 0.127295556,
        715: 0.1116850944,
        720: 0.0992804049,
        725: 0.087874597,
        730: 0.081617577,
        735: 0.0790542325,
        740: 0.0845333091,
        745: 0.068929827,
        750: 0.0730299188,
        755: 0.0751929522,
        760: 0.0561935817,
        765: 0.068929827,
        770: 0.0730299188,
        775: 0.0751929522,
        780: 0.0561935817},
    'F40/C75 (Broadband FL)': {
        380: 0.0560479743,
        385: 0.0650016588,
        390: 0.0822455418,
        395: 0.0912220116,
        400: 0.1481676916,
        405: 0.4040406936,
        410: 0.1625780294,
        415: 0.1497779866,
        420: 0.1646054506,
        425: 0.1810413062,
        430: 0.1970837661,
        435: 1.0,
        440: 0.3991245238,
        445: 0.2440362408,
        450: 0.2577247631,
        455: 0.2704286378,
        460: 0.2817694292,
        465: 0.2884075779,
        470: 0.2953647929,
        475: 0.2974997254,
        480: 0.2987052941,
        485: 0.2983489285,
        490: 0.2982489019,
        495: 0.2947271703,
        500: 0.2899199532,
        505: 0.2852557114,
        510: 0.2805881884,
        515: 0.2755788516,
        520: 0.2706138226,
        525: 0.2653962104,
        530: 0.2591022872,
        535: 0.2508199748,
        540: 0.2389647599,
        545: 0.661710558,
        550: 0.3290631354,
        555: 0.2267464309,
        560: 0.2244978077,
        565: 0.2219901385,
        570: 0.2206513755,
        575: 0.2768268747,
        580: 0.2981106875,
        585: 0.223437704,
        590: 0.2254188225,
        595: 0.2264308136,
        600: 0.2256007032,
        605: 0.226208846,
        610: 0.2269944373,
        615: 0.2264362381,
        620: 0.223890994,
        625: 0.2232929504,
        630: 0.2177868985,
        635: 0.2138376545,
        640: 0.2094656863,
        645: 0.2019715977,
        650: 0.1929978887,
        655: 0.1879496485,
        660: 0.1763106784,
        665: 0.1739693798,
        670: 0.1633657909,
        675: 0.1534834503,
        680: 0.142576784,
        685: 0.1342658974,
        690: 0.1235162098,
        695: 0.1193914095,
        700: 0.1120845645,
        705: 0.1021450644,
        710: 0.1011805729,
        715: 0.0924209515,
        720: 0.0867792918,
        725: 0.0717824083,
        730: 0.0579288073,
        735: 0.0654143384,
        740: 0.0698613761,
        745: 0.0765039678,
        750: 0.0585353785,
        755: 0.0536685951,
        760: 0.0,
        765: 0.0765039678,
        770: 0.0585353785,
        775: 0.0536685951,
        780: 0.0},
    'F40/CWX (Broadband FL)': {
        380: 0.0273834847,
        385: 0.0335910321,
        390: 0.0415441874,
        395: 0.05036987,
        400: 0.0998079376,
        405: 0.3132570916,
        410: 0.1014011502,
        415: 0.0898287401,
        420: 0.1010674589,
        425: 0.1123265292,
        430: 0.1198708166,
        435: 0.7955161847,
        440: 0.2917612475,
        445: 0.1531697521,
        450: 0.160518783,
        455: 0.1607852358,
        460: 0.1651193669,
        465: 0.1699645097,
        470: 0.1729058002,
        475: 0.1745451014,
        480: 0.174788997,
        485: 0.1749281665,
        490: 0.1769671979,
        495: 0.176492277,
        500: 0.1791247312,
        505: 0.1856632039,
        510: 0.1989325374,
        515: 0.2159402183,
        520: 0.2293640689,
        525: 0.2385025329,
        530: 0.243367257,
        535: 0.2494014735,
        540: 0.2429313931,
        545: 0.6294672343,
        550: 0.3398913497,
        555: 0.2552127876,
        560: 0.2487278428,
        565: 0.2495726621,
        570: 0.2544991846,
        575: 0.3023402807,
        580: 0.3477277617,
        585: 0.3033958502,
        590: 0.3221953519,
        595: 0.31581382,
        600: 0.2889653578,
        605: 0.2607324783,
        610: 1.0,
        615: 0.7383903096,
        620: 0.2138652307,
        625: 0.20924498,
        630: 0.2584532703,
        635: 0.1434636137,
        640: 0.0990622075,
        645: 0.0865978746,
        650: 0.1015560714,
        655: 0.0749556413,
        660: 0.0674415057,
        665: 0.0585000705,
        670: 0.043906886,
        675: 0.0368036403,
        680: 0.032520265,
        685: 0.0412114249,
        690: 0.0456327665,
        695: 0.0362119567,
        700: 0.0243034717,
        705: 0.0750881613,
        710: 0.1125288859,
        715: 0.0419337822,
        720: 0.0145614908,
        725: 0.0136178445,
        730: 0.012804761,
        735: 0.011786784,
        740: 0.0125255388,
        745: 0.0115258604,
        750: 0.0125358374,
        755: 0.0096349159,
        760: 0.0078610249,
        765: 0.0216177183,
        770: 0.0225290016,
        775: 0.0228098117,
        780: 0.0234304973},
    'F40/DX (Broadband FL)': {
        380: 0.0377831211,
        385: 0.0479805195,
        390: 0.0605756851,
        395: 0.0735487881,
        400: 0.1352553909,
        405: 0.3823717837,
        410: 0.1473362473,
        415: 0.1410712093,
        420: 0.1580196201,
        425: 0.1776145172,
        430: 0.1973766662,
        435: 1.0,
        440: 0.3976372095,
        445: 0.246635821,
        450: 0.2613217033,
        455: 0.2743550823,
        460: 0.2846374388,
        465: 0.2931029867,
        470: 0.2997102468,
        475: 0.3025910773,
        480: 0.3024860857,
        485: 0.3012227376,
        490: 0.3025181027,
        495: 0.299469218,
        500: 0.2984138585,
        505: 0.3011559805,
        510: 0.3092951421,
        515: 0.3191799546,
        520: 0.3268236541,
        525: 0.3293390541,
        530: 0.3273837705,
        535: 0.3225724898,
        540: 0.3059897901,
        545: 0.7263356642,
        550: 0.3977448593,
        555: 0.300410982,
        560: 0.2988914607,
        565: 0.3003121311,
        570: 0.2992953857,
        575: 0.3560955961,
        580: 0.379738889,
        585: 0.3065224436,
        590: 0.3048691006,
        595: 0.2930168456,
        600: 0.2701496236,
        605: 0.2469569977,
        610: 0.5272643191,
        615: 0.4070711215,
        620: 0.1913500165,
        625: 0.1793941793,
        630: 0.1885129453,
        635: 0.1315871791,
        640: 0.105174916,
        645: 0.0925197759,
        650: 0.0920957743,
        655: 0.0760583358,
        660: 0.0668542843,
        665: 0.0590817903,
        670: 0.0489776303,
        675: 0.0418666164,
        680: 0.0375829705,
        685: 0.0380351215,
        690: 0.0400026536,
        695: 0.0334473222,
        700: 0.0245181179,
        705: 0.0453157809,
        710: 0.0590618243,
        715: 0.0287504063,
        720: 0.0167978086,
        725: 0.0139945669,
        730: 0.0114092009,
        735: 0.0124492297,
        740: 0.015034471,
        745: 0.0096318955,
        750: 0.0121471762,
        755: 0.0047304152,
        760: 0.0,
        765: 0.0096318955,
        770: 0.0121471762,
        775: 0.0047304152,
        780: 0.0},
    'F40/DXTP (Delux FL)': {
        380: 0.0453299851,
        385: 0.0565911781,
        390: 0.069832964,
        395: 0.0835314162,
        400: 0.1409951105,
        405: 0.3831666918,
        410: 0.1578755823,
        415: 0.148802297,
        420: 0.1673493548,
        425: 0.1861327416,
        430: 0.1997849074,
        435: 1.0,
        440: 0.4094349391,
        445: 0.2544228356,
        450: 0.2681517964,
        455: 0.2797568056,
        460: 0.2902728126,
        465: 0.2981804566,
        470: 0.3037138112,
        475: 0.3069857703,
        480: 0.3081508729,
        485: 0.3051454841,
        490: 0.3031580071,
        495: 0.2971275749,
        500: 0.2893973918,
        505: 0.2821707554,
        510: 0.2747709258,
        515: 0.2681158358,
        520: 0.2637407425,
        525: 0.2631695046,
        530: 0.2665857335,
        535: 0.2746771332,
        540: 0.2816872241,
        545: 0.7051414678,
        550: 0.4150330953,
        555: 0.3435176185,
        560: 0.3615220806,
        565: 0.3759744627,
        570: 0.3848734653,
        575: 0.4389524166,
        580: 0.4538581738,
        585: 0.3731648643,
        590: 0.3554417734,
        595: 0.3337052493,
        600: 0.309097159,
        605: 0.2828144705,
        610: 0.2561397443,
        615: 0.2293607032,
        620: 0.2038602343,
        625: 0.1806218404,
        630: 0.159371775,
        635: 0.1406352884,
        640: 0.1229559298,
        645: 0.1078229302,
        650: 0.0943491271,
        655: 0.0832062939,
        660: 0.072986465,
        665: 0.0640645257,
        670: 0.0563661095,
        675: 0.0498447391,
        680: 0.0446869045,
        685: 0.0410030076,
        690: 0.0389398301,
        695: 0.0335334608,
        700: 0.0306081203,
        705: 0.0289763439,
        710: 0.0267316241,
        715: 0.0240100775,
        720: 0.0231780345,
        725: 0.0211714875,
        730: 0.0204167897,
        735: 0.0188312369,
        740: 0.0179374508,
        745: 0.0178226176,
        750: 0.0166535773,
        755: 0.0162608849,
        760: 0.0131835384,
        765: 0.0178226176,
        770: 0.0166535773,
        775: 0.0162608849,
        780: 0.0131835384},
    'F40/N (Natural FL)': {
        380: 0.0310803197,
        385: 0.0378149459,
        390: 0.0472093878,
        395: 0.0548992972,
        400: 0.1129615307,
        405: 0.3708923984,
        410: 0.1092014789,
        415: 0.0926994044,
        420: 0.1020376843,
        425: 0.1123672662,
        430: 0.1236343854,
        435: 1.0,
        440: 0.3255556857,
        445: 0.149862334,
        450: 0.1583372978,
        455: 0.1667638973,
        460: 0.1733225824,
        465: 0.1798784075,
        470: 0.1850065602,
        475: 0.1884852017,
        480: 0.1915301696,
        485: 0.1952256049,
        490: 0.2002629986,
        495: 0.1995383184,
        500: 0.2003190857,
        505: 0.2048572533,
        510: 0.2118007333,
        515: 0.2203687125,
        520: 0.2285708865,
        525: 0.2351824272,
        530: 0.2390886187,
        535: 0.2426002432,
        540: 0.24796799,
        545: 0.7419130895,
        550: 0.3684371543,
        555: 0.2602929951,
        560: 0.2667421903,
        565: 0.2759699712,
        570: 0.2868946364,
        575: 0.3639225218,
        580: 0.3939530486,
        585: 0.3301830181,
        590: 0.3412332886,
        595: 0.3527184657,
        600: 0.3626632564,
        605: 0.3718106602,
        610: 0.3932417242,
        615: 0.3945641765,
        620: 0.3913177271,
        625: 0.3928678431,
        630: 0.391650004,
        635: 0.3849154568,
        640: 0.3781926329,
        645: 0.3699153639,
        650: 0.3608888885,
        655: 0.3493854068,
        660: 0.3346882348,
        665: 0.3205086429,
        670: 0.3053103596,
        675: 0.2879882185,
        680: 0.2711464487,
        685: 0.2529878591,
        690: 0.2386781451,
        695: 0.2196515756,
        700: 0.2021991109,
        705: 0.1873420456,
        710: 0.1706900821,
        715: 0.1537376739,
        720: 0.1367872264,
        725: 0.1237847657,
        730: 0.113749871,
        735: 0.1045680793,
        740: 0.0925290316,
        745: 0.0816302019,
        750: 0.0794656172,
        755: 0.0624670576,
        760: 0.0538005265,
        765: 0.0816302019,
        770: 0.0794656172,
        775: 0.0624670576,
        780: 0.0538005265},
    'H38HT-100 (Mercury)': {
        380: 0.0,
        385: 0.0116674647,
        390: 0.0303725622,
        395: 0.0259688468,
        400: 0.1433438465,
        405: 0.3487975023,
        410: 0.0529924118,
        415: 0.0250126139,
        420: 0.0265268005,
        425: 0.0399770869,
        430: 0.1394781146,
        435: 0.8115180896,
        440: 0.1198900128,
        445: 0.0318914583,
        450: 0.0231482617,
        455: 0.0202224835,
        460: 0.018123222,
        465: 0.0175848606,
        470: 0.0176403016,
        475: 0.016567826,
        480: 0.0162244063,
        485: 0.0186455924,
        490: 0.0275746264,
        495: 0.0203361856,
        500: 0.014912398,
        505: 0.01459063,
        510: 0.0155189187,
        515: 0.0159060586,
        520: 0.016563821,
        525: 0.0178337875,
        530: 0.021714274,
        535: 0.0319494122,
        540: 0.1094776103,
        545: 1.0,
        550: 0.1912029904,
        555: 0.0341638108,
        560: 0.0280001251,
        565: 0.0281671685,
        570: 0.0485844681,
        575: 0.692966307,
        580: 0.4546454502,
        585: 0.0360948637,
        590: 0.0240272032,
        595: 0.0191764014,
        600: 0.0169633167,
        605: 0.015165037,
        610: 0.0137479525,
        615: 0.0132527319,
        620: 0.0140674828,
        625: 0.0132181268,
        630: 0.0123665772,
        635: 0.0117390118,
        640: 0.0104464888,
        645: 0.0111491975,
        650: 0.0110394743,
        655: 0.0114700945,
        660: 0.0109632795,
        665: 0.0105090841,
        670: 0.0112744755,
        675: 0.0095615242,
        680: 0.0107455314,
        685: 0.0136921704,
        690: 0.021856513,
        695: 0.0096899463,
        700: 0.0103380482,
        705: 0.0131022477,
        710: 0.0117797223,
        715: 0.0091521454,
        720: 0.0097539396,
        725: 0.0089041,
        730: 0.0107792269,
        735: 0.0101288172,
        740: 0.0098136832,
        745: 0.0076255603,
        750: 0.0100327823,
        755: 0.0100743277,
        760: 0.0068657463,
        765: 0.0850852124,
        770: 0.0864842549,
        775: 0.0853029284,
        780: 0.0833691938},
    'H38JA-100/DX (Mercury DX)': {
        380: 0.0,
        385: 0.0142531794,
        390: 0.035510996,
        395: 0.0340505478,
        400: 0.1149315065,
        405: 0.401945863,
        410: 0.0731486494,
        415: 0.0294562448,
        420: 0.0334641067,
        425: 0.0467536223,
        430: 0.0859007702,
        435: 0.8848377912,
        440: 0.21738315,
        445: 0.0396700621,
        450: 0.0310189057,
        455: 0.0238365579,
        460: 0.0213194669,
        465: 0.0207918007,
        470: 0.0210541279,
        475: 0.0191074328,
        480: 0.0185883321,
        485: 0.0219194005,
        490: 0.0289621798,
        495: 0.0257582983,
        500: 0.0182158647,
        505: 0.0185550639,
        510: 0.0183153559,
        515: 0.0194078762,
        520: 0.0209379346,
        525: 0.0211969364,
        530: 0.0267577146,
        535: 0.0429919204,
        540: 0.068156362,
        545: 1.0,
        550: 0.4452179065,
        555: 0.0516026428,
        560: 0.0405264531,
        565: 0.038612226,
        570: 0.051522372,
        575: 0.5434322231,
        580: 0.7645727533,
        585: 0.0840232078,
        590: 0.0799401831,
        595: 0.1283970594,
        600: 0.046728557,
        605: 0.054906118,
        610: 0.1358991153,
        615: 0.4762264071,
        620: 0.5905637157,
        625: 0.1260038434,
        630: 0.0607108596,
        635: 0.0372345875,
        640: 0.0275875935,
        645: 0.0293064162,
        650: 0.0403259082,
        655: 0.0327632449,
        660: 0.0216557115,
        665: 0.0194818001,
        670: 0.0198222222,
        675: 0.0174965232,
        680: 0.0171992097,
        685: 0.0191765713,
        690: 0.0389077992,
        695: 0.1043534182,
        700: 0.1825890986,
        705: 0.1275145389,
        710: 0.033330114,
        715: 0.0188559469,
        720: 0.0194667177,
        725: 0.0175193285,
        730: 0.015485,
        735: 0.0162835133,
        740: 0.0136225521,
        745: 0.0154414394,
        750: 0.0139533505,
        755: 0.0150311372,
        760: 0.0131452521,
        765: 0.113742611,
        770: 0.1156128625,
        775: 0.1140336555,
        780: 0.1114486233},
    'MHC100/U/MP /3K': {
        380: 0.0376147854,
        385: 0.0647066508,
        390: 0.0898482711,
        395: 0.1179908314,
        400: 0.1231236348,
        405: 0.1670751377,
        410: 0.2041234293,
        415: 0.2397149418,
        420: 0.2972040744,
        425: 0.2510951821,
        430: 0.1869314047,
        435: 0.2673806897,
        440: 0.2226429896,
        445: 0.1856918819,
        450: 0.1748492864,
        455: 0.1962595435,
        460: 0.2112985571,
        465: 0.1948964056,
        470: 0.2071706927,
        475: 0.1783784891,
        480: 0.1519824397,
        485: 0.1429624764,
        490: 0.1554462388,
        495: 0.1827124337,
        500: 0.2066282493,
        505: 0.1994095911,
        510: 0.179980979,
        515: 0.1926386417,
        520: 0.2073424963,
        525: 0.1929470473,
        530: 0.2589535877,
        535: 0.8618208655,
        540: 0.8773836648,
        545: 0.4355082793,
        550: 0.2634081808,
        555: 0.2114312034,
        560: 0.2191198704,
        565: 0.3501587856,
        570: 0.4785991902,
        575: 0.3317614276,
        580: 0.3447238006,
        585: 0.5410523908,
        590: 0.5042269513,
        595: 0.9034505191,
        600: 1.0,
        605: 0.8629744987,
        610: 0.7257184831,
        615: 0.6526699809,
        620: 0.5374503302,
        625: 0.4943571367,
        630: 0.4361637077,
        635: 0.3951681808,
        640: 0.3707066792,
        645: 0.3575971318,
        650: 0.3220656175,
        655: 0.3485933584,
        660: 0.3521795178,
        665: 0.3034627621,
        670: 0.3181692007,
        675: 0.3060010916,
        680: 0.2648197974,
        685: 0.2402319214,
        690: 0.1768699724,
        695: 0.1710297327,
        700: 0.1509962127,
        705: 0.1346104345,
        710: 0.1256935694,
        715: 0.1196011911,
        720: 0.1175632151,
        725: 0.115748188,
        730: 0.1096603048,
        735: 0.1074436665,
        740: 0.1093557698,
        745: 0.1081531758,
        750: 0.1143784298,
        755: 0.1376604646,
        760: 0.1168708812,
        765: 0.1081531758,
        770: 0.1143784298,
        775: 0.1376604646,
        780: 0.1168708812},
    'MHC100/U/MP /4K': {
        380: 0.0637674013,
        385: 0.1123934055,
        390: 0.1593879234,
        395: 0.2148287382,
        400: 0.2183549307,
        405: 0.2541677655,
        410: 0.3070083266,
        415: 0.3747787957,
        420: 0.4101332948,
        425: 0.4153295957,
        430: 0.3650745549,
        435: 0.381573482,
        440: 0.3409149933,
        445: 0.3486459178,
        450: 0.3002037858,
        455: 0.3270889625,
        460: 0.3650128105,
        465: 0.3480027257,
        470: 0.3375354257,
        475: 0.3266879254,
        480: 0.316183833,
        485: 0.2910746941,
        490: 0.309523707,
        495: 0.3217119581,
        500: 0.3384589435,
        505: 0.3928866191,
        510: 0.34786773,
        515: 0.3486180975,
        520: 0.3465016981,
        525: 0.3554749465,
        530: 0.4279825108,
        535: 1.0,
        540: 0.8522835071,
        545: 0.5225519999,
        550: 0.3853539405,
        555: 0.3626888531,
        560: 0.4187574389,
        565: 0.4761076135,
        570: 0.4784846361,
        575: 0.4560581042,
        580: 0.4240184578,
        585: 0.4887698591,
        590: 0.7400733059,
        595: 0.8156125621,
        600: 0.6521648225,
        605: 0.5257899371,
        610: 0.5445558904,
        615: 0.5721926378,
        620: 0.473596833,
        625: 0.5184803715,
        630: 0.5279852148,
        635: 0.5274546448,
        640: 0.5478414402,
        645: 0.5490712628,
        650: 0.4320018033,
        655: 0.4164493434,
        660: 0.4316947418,
        665: 0.3774034717,
        670: 0.3493789782,
        675: 0.3372444347,
        680: 0.3386207001,
        685: 0.3666887688,
        690: 0.2644808364,
        695: 0.2755936118,
        700: 0.2371390343,
        705: 0.1958083225,
        710: 0.1765361917,
        715: 0.1772409194,
        720: 0.1706542878,
        725: 0.1579817973,
        730: 0.1456057575,
        735: 0.1403582475,
        740: 0.1458152631,
        745: 0.1418151423,
        750: 0.1594540493,
        755: 0.2387771482,
        760: 0.1694033761,
        765: 0.1418151423,
        770: 0.1594540493,
        775: 0.2387771482,
        780: 0.1694033761},
    'SDW-T 100W/LV (Super HPS)': {
        380: 0.0,
        385: 0.0141961629,
        390: 0.0271220229,
        395: 0.0303775085,
        400: 0.0362696746,
        405: 0.0437712846,
        410: 0.0493281234,
        415: 0.0592142414,
        420: 0.0715570992,
        425: 0.0819754675,
        430: 0.0983933594,
        435: 0.122793082,
        440: 0.1204041762,
        445: 0.126762486,
        450: 0.1673398353,
        455: 0.1511426352,
        460: 0.1264383121,
        465: 0.1914769517,
        470: 0.2005954377,
        475: 0.1164238118,
        480: 0.0753185427,
        485: 0.0793070988,
        490: 0.0972695558,
        495: 0.2124117995,
        500: 0.284236306,
        505: 0.1182060659,
        510: 0.1150698813,
        515: 0.1710991734,
        520: 0.1278038826,
        525: 0.1314923556,
        530: 0.1422584882,
        535: 0.1577590683,
        540: 0.1873125804,
        545: 0.2998456174,
        550: 0.6133527952,
        555: 0.7746796398,
        560: 0.853651898,
        565: 0.9907092798,
        570: 0.9950120886,
        575: 0.6367900135,
        580: 0.2265320086,
        585: 0.0403526987,
        590: 0.0408609349,
        595: 0.0827397798,
        600: 0.2412607036,
        605: 0.5550694147,
        610: 0.8194085692,
        615: 1.0,
        620: 0.9959256484,
        625: 0.9855418078,
        630: 0.9575053362,
        635: 0.9230960248,
        640: 0.8862534822,
        645: 0.8487271043,
        650: 0.8050915362,
        655: 0.7852615081,
        660: 0.7407139373,
        665: 0.7390117239,
        670: 0.7602336888,
        675: 0.7146494177,
        680: 0.6219384705,
        685: 0.5418516212,
        690: 0.4951613123,
        695: 0.4698752382,
        700: 0.4513268189,
        705: 0.435123235,
        710: 0.4215327965,
        715: 0.4104020607,
        720: 0.4002398744,
        725: 0.3891012471,
        730: 0.3777022176,
        735: 0.3720378427,
        740: 0.3643860737,
        745: 0.359417453,
        750: 0.3591964858,
        755: 0.3495216293,
        760: 0.3452301751,
        765: 0.2452291219,
        770: 0.249261385,
        775: 0.2458566141,
        780: 0.2402832833}}

LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS = CaseInsensitiveMapping(
    {'60 A/W (Soft White)': SpectralPowerDistribution(
        '60 A/W (Soft White)',
        LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
            '60 A/W (Soft White)')),
     'C100S54 (HPS)': SpectralPowerDistribution(
         'C100S54 (HPS)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'C100S54 (HPS)')),
     'C100S54C (HPS)': SpectralPowerDistribution(
         'C100S54C (HPS)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'C100S54C (HPS)')),
     'F32T8/TL830 (Triphosphor)': SpectralPowerDistribution(
         'F32T8/TL830 (Triphosphor)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F32T8/TL830 (Triphosphor)')),
     'F32T8/TL835 (Triphosphor)': SpectralPowerDistribution(
         'F32T8/TL835 (Triphosphor)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F32T8/TL835 (Triphosphor)')),
     'F32T8/TL841 (Triphosphor)': SpectralPowerDistribution(
         'F32T8/TL841 (Triphosphor)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F32T8/TL841 (Triphosphor)')),
     'F32T8/TL850 (Triphosphor)': SpectralPowerDistribution(
         'F32T8/TL850 (Triphosphor)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F32T8/TL850 (Triphosphor)')),
     'F32T8/TL865 /PLUS (Triphosphor)': SpectralPowerDistribution(
         'F32T8/TL865 /PLUS (Triphosphor)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F32T8/TL865 /PLUS (Triphosphor)')),
     'F34/CW/RS/EW (Cool White FL)': SpectralPowerDistribution(
         'F34/CW/RS/EW (Cool White FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F34/CW/RS/EW (Cool White FL)')),
     'F34T12/LW/RS /EW': SpectralPowerDistribution(
         'F34T12/LW/RS /EW',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F34T12/LW/RS /EW')),
     'F34T12WW/RS /EW (Warm White FL)': SpectralPowerDistribution(
         'F34T12WW/RS /EW (Warm White FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F34T12WW/RS /EW (Warm White FL)')),
     'F40/C50 (Broadband FL)': SpectralPowerDistribution(
         'F40/C50 (Broadband FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/C50 (Broadband FL)')),
     'F40/C75 (Broadband FL)': SpectralPowerDistribution(
         'F40/C75 (Broadband FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/C75 (Broadband FL)')),
     'F40/CWX (Broadband FL)': SpectralPowerDistribution(
         'F40/CWX (Broadband FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/CWX (Broadband FL)')),
     'F40/DX (Broadband FL)': SpectralPowerDistribution(
         'F40/DX (Broadband FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/DX (Broadband FL)')),
     'F40/DXTP (Delux FL)': SpectralPowerDistribution(
         'F40/DXTP (Delux FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/DXTP (Delux FL)')),
     'F40/N (Natural FL)': SpectralPowerDistribution(
         'F40/N (Natural FL)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'F40/N (Natural FL)')),
     'H38HT-100 (Mercury)': SpectralPowerDistribution(
         'H38HT-100 (Mercury)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'H38HT-100 (Mercury)')),
     'H38JA-100/DX (Mercury DX)': SpectralPowerDistribution(
         'H38JA-100/DX (Mercury DX)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'H38JA-100/DX (Mercury DX)')),
     'MHC100/U/MP /3K': SpectralPowerDistribution(
         'MHC100/U/MP /3K',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'MHC100/U/MP /3K')),
     'MHC100/U/MP /4K': SpectralPowerDistribution(
         'MHC100/U/MP /4K',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'MHC100/U/MP /4K')),
     'SDW-T 100W/LV (Super HPS)': SpectralPowerDistribution(
         'SDW-T 100W/LV (Super HPS)',
         LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS_DATA.get(
             'SDW-T 100W/LV (Super HPS)'))})
"""
Philips light sources from *NIST* *NIST CQS simulation 7.4.xls*
spreadsheet. [2]_

LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS : CaseInsensitiveMapping
    {'60 A/W (Soft White)', 'C100S54 (HPS)', 'C100S54C (HPS)',
    'F32T8/TL830 (Triphosphor)', 'F32T8/TL835 (Triphosphor)',
    'F32T8/TL841 (Triphosphor)', 'F32T8/TL850 (Triphosphor)',
    'F32T8/TL865 /PLUS (Triphosphor)', 'F34/CW/RS/EW (Cool White FL)',
    'F34T12/LW/RS /EW', 'F34T12WW/RS /EW (Warm White FL)',
    'F40/C50 (Broadband FL)', 'F40/C75 (Broadband FL)',
    'F40/CWX (Broadband FL)', 'F40/DX (Broadband FL)', 'F40/DXTP (Delux FL)',
    'F40/N (Natural FL)', 'H38HT-100 (Mercury)', 'H38JA-100/DX (Mercury DX)',
    'MHC100/U/MP /3K', 'MHC100/U/MP /4K', 'SDW-T 100W/LV (Super HPS)'}
"""

LIGHT_SOURCES_RELATIVE_SPDS = CaseInsensitiveMapping(
    LIGHT_SOURCES_RIT_RELATIVE_SPDS)
"""
Aggregated light sources spectral power distributions.

LIGHT_SOURCES_RELATIVE_SPDS : CaseInsensitiveMapping
"""

LIGHT_SOURCES_RELATIVE_SPDS.update(
    LIGHT_SOURCES_NIST_TRADITIONAL_RELATIVE_SPDS)
LIGHT_SOURCES_RELATIVE_SPDS.update(
    LIGHT_SOURCES_NIST_LED_RELATIVE_SPDS)
LIGHT_SOURCES_RELATIVE_SPDS.update(
    LIGHT_SOURCES_NIST_PHILIPS_RELATIVE_SPDS)
