#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Defines unit tests for :mod:`colour.colorimetry.blackbody` module.
"""

from __future__ import division, unicode_literals

import numpy as np
import sys

if sys.version_info[:2] <= (2, 6):
    import unittest2 as unittest
else:
    import unittest

from colour.colorimetry import (
    SpectralShape,
    planck_law,
    blackbody_spd)

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013 - 2014 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['PLANCK_LAW_DATA',
           'BLACKBODY_SPD_DATA',
           'TestPlanckLaw',
           'TestBlackbodySpd']

PLANCK_LAW_DATA = {
    1667: {
        10: 0.0,
        20: 1.4105596687642286e-165,
        30: 5.530067904248269e-104,
        40: 2.2643024143383377e-73,
        50: 4.097537659366942e-55,
        60: 5.144886317514676e-43,
        70: 2.002026097230864e-34,
        80: 5.071435708137485e-28,
        90: 4.5239512888229996e-23,
        100: 3.9052618037311606e-19,
        110: 6.199380452902635e-16,
        120: 2.774108564921223e-13,
        130: 4.700577339170552e-11,
        140: 3.7222321408108333e-09,
        150: 1.606767053049753e-07,
        160: 4.242815847371177e-06,
        170: 7.483554302967325e-05,
        180: 0.0009439877838579169,
        190: 0.008986537508368567,
        200: 0.06739672052016699,
        210: 0.4122636644142145,
        220: 2.115953500760159,
        230: 9.32773792033636,
        240: 36.00993182785636,
        250: 123.74179663748401,
        260: 383.7345544530101,
        270: 1086.5329211269789,
        280: 2837.2351129434164,
        290: 6891.670775622278,
        300: 15687.798375507615,
        310: 33683.44844274908,
        320: 68602.63896731117,
        330: 133194.33701036545,
        340: 247596.97900992312,
        350: 442378.11199124635,
        360: 762281.907435713,
        370: 1270670.6183551583,
        380: 2054593.363812708,
        390: 3230361.862517226,
        400: 4949462.9886321705,
        410: 7404596.883354958,
        420: 10835600.410377722,
        430: 15535001.25153879,
        440: 21852948.726484377,
        450: 30201282.954720583,
        460: 41056532.58435318,
        470: 54961670.46908595,
        480: 72526503.35111238,
        490: 94426622.57829192,
        500: 121400895.01296306,
        510: 154247523.7468478,
        520: 193818754.64929816,
        530: 241014345.31522235,
        540: 296773946.385242,
        550: 362068570.77571,
        560: 437891343.8688596,
        570: 525247737.38369095,
        580: 625145492.0206116,
        590: 738584429.837095,
        600: 866546347.6172808,
        610: 1009985168.2795815,
        620: 1169817509.6764226,
        630: 1346913810.0005624,
        640: 1542090127.3731594,
        650: 1756100708.9020905,
        660: 1989631402.3014297,
        670: 2243293961.6693907,
        680: 2517621278.723494,
        690: 2813063552.061537,
        700: 3129985390.118351,
        710: 3468663828.588369,
        720: 3829287230.2622943,
        730: 4211955024.492659,
        740: 4616678234.80892,
        750: 5043380736.452318,
        760: 5491901180.664997,
        770: 5961995519.29445,
        780: 6453340061.49302,
        790: 6965534993.828248,
        800: 7498108295.795781,
        810: 8050519984.367113,
        820: 8622166623.644691,
        830: 9212386038.779451,
        840: 9820462176.886133,
        850: 10445630061.640814,
        860: 11087080792.442425,
        870: 11743966543.367159,
        880: 12415405521.548801,
        890: 13100486849.00554,
        900: 13798275336.239716,
        910: 14507816120.112028,
        920: 15228139142.490662,
        930: 15958263449.970768,
        940: 16697201298.524448,
        950: 17443962050.26111,
        960: 18197555852.54279,
        970: 18956997092.503765,
        980: 19721307622.57071,
        990: 20489519754.869236,
        1000: 21260679024.44757,
        1010: 22033846723.05022,
        1020: 22808102206.753956,
        1030: 23582544982.139328,
        1040: 24356296576.83462,
        1050: 25128502201.242085,
        1060: 25898332209.061493,
        1070: 26664983364.869366,
        1080: 27427679927.513756,
        1090: 28185674558.457287,
        1100: 28938249064.45347,
        1110: 29684714984.095753,
        1120: 30424414027.83521,
        1130: 31156718381.043312,
        1140: 31881030879.605392,
        1150: 32596785067.378365,
        1160: 33303445144.645103,
        1170: 34000505816.452465,
        1180: 34687492049.439545,
        1190: 35363958745.45504,
        1200: 36029490339.92939,
        1210: 36683700332.62206,
        1220: 37326230758.00247,
        1230: 37956751602.155235,
        1240: 38574960172.731186,
        1250: 39180580428.090706,
        1260: 39773362271.41721,
        1270: 40353080815.21521,
        1280: 40919535621.24541,
        1290: 41472549920.603485,
        1300: 42011969818.30645,
        1310: 42537663486.42292,
        1320: 43049520349.46779,
        1330: 43547450265.47595,
        1340: 44031382705.880585,
        1350: 44501265937.04368,
        1360: 44957066206.02337,
        1370: 45398766932.91232,
        1380: 45826367911.84631,
        1390: 46239884522.55986,
        1400: 46639346954.15597,
        1410: 47024799442.562164,
        1420: 47396299522.962166,
        1430: 47753917298.31928,
        1440: 48097734724.95365,
        1450: 48427844915.98214,
        1460: 48744351463.29889,
        1470: 49047367778.644356,
        1480: 49337016454.19812,
        1490: 49613428643.021576,
        1500: 49876743459.58188,
        1510: 50127107400.49855,
        1520: 50364673785.572845,
        1530: 50589602219.0885,
        1540: 50802058071.30568,
        1550: 51002211980.00954,
        1560: 51190239371.9247,
        1570: 51366320003.756256,
        1580: 51530637522.579056,
        1590: 51683379045.25827,
        1600: 51824734756.5533,
        1610: 51954897525.528915,
        1620: 52074062539.874115,
        1630: 52182426957.7086,
        1640: 52280189576.440544,
        1650: 52367550518.226364,
        1660: 52444710931.57021,
        1670: 52511872708.59633,
        1680: 52569238217.518684,
        1690: 52617010049.82972,
        1700: 52655390781.72925,
        1710: 52684582749.31279,
        1720: 52704787837.04245,
        1730: 52716207279.024796,
        1740: 52719041472.62516,
        1750: 52713489803.95352,
        1760: 52699750484.76343,
        1770: 52678020400.31157,
        1780: 52648494967.73607,
        1790: 52611368004.517494,
        1800: 52566831606.59786,
        1810: 52515076035.7416,
        1820: 52456289615.732574,
        1830: 52390658637.011696,
        1840: 52318367269.371056,
        1850: 52239597482.32978,
        1860: 52154528972.82877,
        1870: 52063339099.89277,
        1880: 51966202825.917854,
        1890: 51863292664.25549,
        1900: 51754778632.772896,
        1910: 51640828213.0829,
        1920: 51521606315.14482,
        1930: 51397275246.95105,
        1940: 51267994689.0221,
        1950: 51133921673.44526,
        1960: 50995210567.20106,
        1970: 50852013059.53231,
        1980: 50704478153.1194,
        1990: 50552752158.83679,
        2000: 50396978693.872505,
        2010: 50237298683.00405,
        2020: 50073850362.83138,
        2030: 49906769288.77661,
        2040: 49736188344.66857,
        2050: 49562237754.73885,
        2060: 49385045097.86284,
        2070: 49204735323.88732,
        2080: 49021430771.89452,
        2090: 48835251190.25805,
        2100: 48646313758.35394,
        2110: 48454733109.796394,
        2120: 48260621357.074646,
        2130: 48064088117.47288,
        2140: 47865240540.16127,
        2150: 47664183334.35247,
        2160: 47461018798.42261,
        2170: 47255846849.90132,
        2180: 47048765056.24102,
        2190: 46839868666.279495,
        2200: 46629250642.315414,
        2210: 46417001692.720345,
        2220: 46203210305.01523,
        2230: 45987962779.34356,
        2240: 45771343262.27702,
        2250: 45553433780.89394,
        2260: 45334314277.07314,
        2270: 45114062641.950584,
        2280: 44892754750.48876,
        2290: 44670464496.11164,
        2300: 44447263825.36195,
        2310: 44223222772.53919,
        2320: 43998409494.28056,
        2330: 43772890304.04888,
        2340: 43546729706.49399,
        2350: 43319990431.65728,
        2360: 43092733468.99001,
        2370: 42865018101.159515,
        2380: 42636901937.61749,
        2390: 42408440947.90929,
        2400: 42179689494.701836,
        2410: 41950700366.51184,
        2420: 41721524810.11636,
        2430: 41492212562.62977,
        2440: 41262811883.23234,
        2450: 41033369584.53776,
        2460: 40803931063.58682,
        2470: 40574540332.457306,
        2480: 40345240048.48027,
        2490: 40116071544.05379,
        2500: 39887074856.04776,
        2510: 39658288754.7923,
        2520: 39429750772.644714,
        2530: 39201497232.13037,
        2540: 38973563273.65309,
        2550: 38745982882.77274,
        2560: 38518788917.04668,
        2570: 38292013132.43421,
        2580: 38065686209.26208,
        2590: 37839837777.75121,
        2600: 37614496443.10401,
        2610: 37389689810.15314,
        2620: 37165444507.57282,
        2630: 36941786211.653656,
        2640: 36718739669.643074,
        2650: 36496328722.65366,
        2660: 36274576328.14166,
        2670: 36053504581.958916,
        2680: 35833134739.98076,
        2690: 35613487239.31411,
        2700: 35394581719.08893,
        2710: 35176437040.837036,
        2720: 34959071308.462654,
        2730: 34742501887.80864,
        2740: 34526745425.8233,
        2750: 34311817869.33225,
        2760: 34097734483.420048,
        2770: 33884509869.426662,
        2780: 33672157982.563717,
        2790: 33460692149.15582,
        2800: 33250125083.511864,
        2810: 33040468904.431908,
        2820: 32831735151.354935,
        2830: 32623934800.15261,
        2840: 32417078278.57489,
        2850: 32211175481.352673,
        2860: 32006235784.963108,
        2870: 31802268062.06314,
        2880: 31599280695.59649,
        2890: 31397281592.58018,
        2900: 31196278197.575497,
        2910: 30996277505.849323,
        2920: 30797286076.231148,
        2930: 30599310043.67133,
        2940: 30402355131.50605,
        2950: 30206426663.434055,
        2960: 30011529575.21113,
        2970: 29817668426.067226,
        2980: 29624847409.851635,
        2990: 29433070365.911415,
        3000: 29242340789.708496,
        3010: 29052661843.180267,
        3020: 28864036364.848946,
        3030: 28676466879.684868,
        3040: 28489955608.728294,
        3050: 28304504478.475124,
        3060: 28120115130.031036,
        3070: 27936788928.038925,
        3080: 27754526969.384502,
        3090: 27573330091.684414,
        3100: 27393198881.56198,
        3110: 27214133682.7145,
        3120: 27036134603.776943,
        3130: 26859201525.986412,
        3140: 26683334110.65162,
        3150: 26508531806.431408,
        3160: 26334793856.42718,
        3170: 26162119305.092438,
        3180: 25990507004.964264,
        3190: 25819955623.220276,
        3200: 25650463648.065193,
        3210: 25482029394.95056,
        3220: 25314651012.631836,
        3230: 25148326489.066196,
        3240: 24983053657.15476,
        3250: 24818830200.332966,
        3260: 24655653658.012356,
        3270: 24493521430.87742,
        3280: 24332430786.040733,
        3290: 24172378862.05984,
        3300: 24013362673.818764,
        3310: 23855379117.277863,
        3320: 23698424974.094685,
        3330: 23542496916.119118,
        3340: 23387591509.765675,
        3350: 23233705220.26598,
        3360: 23080834415.804214,
        3370: 22928975371.538395,
        3380: 22778124273.510197,
        3390: 22628277222.44607,
        3400: 22479430237.45211,
        3410: 22331579259.605606,
        3420: 22184720155.445377,
        3430: 22038848720.363705,
        3440: 21893960681.902065,
        3450: 21750051702.953144,
        3460: 21607117384.871376,
        3470: 21465153270.494354,
        3480: 21324154847.077175,
        3490: 21184117549.142086,
        3500: 21045036761.245327,
        3510: 20906907820.663525,
        3520: 20769726020.00121,
        3530: 20633486609.721977,
        3540: 20498184800.604794,
        3550: 20363815766.12757,
        3560: 20230374644.77971,
        3570: 20097856542.305557,
        3580: 19966256533.880424,
        3590: 19835569666.220882,
        3600: 19705790959.631195,
        3610: 19576915409.987217,
        3620: 19448937990.65966,
        3630: 19321853654.378178,
        3640: 19195657335.03766,
        3650: 19070343949.448544,
        3660: 18945908399.032257,
        3670: 18822345571.46349,
        3680: 18699650342.26053,
        3690: 18577817576.325027,
        3700: 18456842129.43257,
        3710: 18336718849.67542,
        3720: 18217442578.85844,
        3730: 18099008153.84974,
        3740: 17981410407.88698,
        3750: 17864644171.84068,
        3760: 17748704275.435646,
        3770: 17633585548.431595,
        3780: 17519282821.76397,
        3790: 17405790928.6463,
        3800: 17293104705.63485,
        3810: 17181218993.656662,
        3820: 17070128639.002058,
        3830: 16959828494.282494,
        3840: 16850313419.35467,
        3850: 16741578282.211908,
        3860: 16633617959.843605,
        3870: 16526427339.063656,
        3880: 16420001317.308756,
        3890: 16314334803.40733,
        3900: 16209422718.31989,
        3910: 16105259995.851728,
        3920: 16001841583.338558,
        3930: 15899162442.305977,
        3940: 15797217549.103363,
        3950: 15696001895.513037,
        3960: 15595510489.335339,
        3970: 15495738354.950218,
        3980: 15396680533.85614,
        3990: 15298332085.186806,
        4000: 15200688086.206379,
        4010: 15103743632.783978,
        4020: 15007493839.847599,
        4030: 14911933841.818628,
        4040: 14817058793.02697,
        4050: 14722863868.107773,
        4060: 14629344262.379925,
        4070: 14536495192.207136,
        4080: 14444311895.341963,
        4090: 14352789631.253242,
        4100: 14261923681.43762,
        4110: 14171709349.715284,
        4120: 14082141962.510757,
        4130: 13993216869.11899,
        4140: 13904929441.95713,
        4150: 13817275076.802544,
        4160: 13730249193.017466,
        4170: 13643847233.76052,
        4180: 13558064666.185781,
        4190: 13472896981.629543,
        4200: 13388339695.78517,
        4210: 13304388348.866556,
        4220: 13221038505.760393,
        4230: 13138285756.167614,
        4240: 13056125714.734388,
        4250: 12974554021.17303,
        4260: 12893566340.372952,
        4270: 12813158362.502214,
        4280: 12733325803.099806,
        4290: 12654064403.159,
        4300: 12575369929.202068,
        4310: 12497238173.346615,
        4320: 12419664953.36383,
        4330: 12342646112.728895,
        4340: 12266177520.66381,
        4350: 12190255072.17286,
        4360: 12114874688.071022,
        4370: 12040032315.005512,
        4380: 11965723925.470646,
        4390: 11891945517.816345,
        4400: 11818693116.250448,
        4410: 11745962770.834995,
        4420: 11673750557.476772,
        4430: 11602052577.912268,
        4440: 11530864959.687214,
        4450: 11460183856.130997,
        4460: 11390005446.325933,
        4470: 11320325935.071846,
        4480: 11251141552.845882,
        4490: 11182448555.757854,
        4500: 11114243225.501272,
        4510: 11046521869.300163,
        4520: 10979280819.851904,
        4530: 10912516435.26617,
        4540: 10846225099.00016,
        4550: 10780403219.790243,
        4560: 10715047231.580147,
        4570: 10650153593.445868,
        4580: 10585718789.517366,
        4590: 10521739328.89725,
        4600: 10458211745.576466,
        4610: 10395132598.347265,
        4620: 10332498470.713398,
        4630: 10270305970.797773,
        4640: 10208551731.247625,
        4650: 10147232409.137331,
        4660: 10086344685.868944,
        4670: 10025885267.070593,
        4680: 9965850882.492807,
        4690: 9906238285.902851,
        4700: 9847044254.977211,
        4710: 9788265591.192287,
        4720: 9729899119.713387,
        4730: 9671941689.282068,
        4740: 9614390172.101978,
        4750: 9557241463.723253,
        4760: 9500492482.925478,
        4770: 9444140171.599407,
        4780: 9388181494.627407,
        4790: 9332613439.762735,
        4800: 9277433017.507757,
        4810: 9222637260.991117,
        4820: 9168223225.843908,
        4830: 9114187990.075008,
        4840: 9060528653.945503,
        4850: 9007242339.842373,
        4860: 8954326192.15143,
        4870: 8901777377.129566,
        4880: 8849593082.77641,
        4890: 8797770518.705404,
        4900: 8746306916.01434,
        4910: 8695199527.155441,
        4920: 8644445625.805023,
        4930: 8594042506.732754,
        4940: 8543987485.670583,
        4950: 8494277899.181412,
        4960: 8444911104.527425,
        4970: 8395884479.538292,
        4980: 8347195422.479163,
        4990: 8298841351.918492,
        5000: 8250819706.5958,
        5010: 8203127945.289359,
        5020: 8155763546.683814,
        5030: 8108724009.237831,
        5040: 8062006851.05176,
        5050: 8015609609.735345,
        5060: 7969529842.2755375,
        5070: 7923765124.904409,
        5080: 7878313052.967226,
        5090: 7833171240.79065,
        5100: 7788337321.551186,
        5110: 7743808947.143786,
        5120: 7699583788.050758,
        5130: 7655659533.210873,
        5140: 7612033889.888802,
        5150: 7568704583.544822,
        5160: 7525669357.7049055,
        5170: 7482925973.831058,
        5180: 7440472211.19211,
        5190: 7398305866.734847,
        5200: 7356424754.9555235,
        5210: 7314826707.771835,
        5220: 7273509574.395283,
        5230: 7232471221.203981,
        5240: 7191709531.615959,
        5250: 7151222405.962903,
        5260: 7111007761.3643875,
        5270: 7071063531.602623,
        5280: 7031387666.997709,
        5290: 6991978134.283396,
        5300: 6952832916.4833975,
        5310: 6913950012.788244,
        5320: 6875327438.43268,
        5330: 6836963224.573657,
        5340: 6798855418.168842,
        5350: 6761002081.855781,
        5360: 6723401293.8315935,
        5370: 6686051147.733297,
        5380: 6648949752.518734,
        5390: 6612095232.348101,
        5400: 6575485726.466108,
        5410: 6539119389.084758,
        5420: 6502994389.266767,
        5430: 6467108910.809609,
        5440: 6431461152.130217,
        5450: 6396049326.150313,
        5460: 6360871660.182423,
        5470: 6325926395.816512,
        5480: 6291211788.807308,
        5490: 6256726108.96227,
        5500: 6222467640.030232,
        5510: 6188434679.590736,
        5520: 6154625538.943997,
        5530: 6121038543.001583,
        5540: 6087672030.177756,
        5550: 6054524352.281495,
        5560: 6021593874.40921,
        5570: 5988878974.838122,
        5580: 5956378044.920356,
        5590: 5924089488.977687,
        5600: 5892011724.197018,
        5610: 5860143180.526486,
        5620: 5828482300.572332,
        5630: 5797027539.496387,
        5640: 5765777364.914302,
        5650: 5734730256.794444,
        5660: 5703884707.357479,
        5670: 5673239220.976666,
        5680: 5642792314.078818,
        5690: 5612542515.045953,
        5700: 5582488364.117655,
        5710: 5552628413.294097,
        5720: 5522961226.23977,
        5730: 5493485378.187879,
        5740: 5464199455.845434,
        5750: 5435102057.299024,
        5760: 5406191791.9212675,
        5770: 5377467280.277957,
        5780: 5348927154.035841,
        5790: 5320570055.871142,
        5800: 5292394639.378694,
        5810: 5264399568.981788,
        5820: 5236583519.842664,
        5830: 5208945177.773695,
        5840: 5181483239.14921,
        5850: 5154196410.818002,
        5860: 5127083410.016495,
        5870: 5100142964.282542,
        5880: 5073373811.369919,
        5890: 5046774699.163447,
        5900: 5020344385.594769,
        5910: 4994081638.558781,
        5920: 4967985235.8307,
        5930: 4942053964.983788,
        5940: 4916286623.307682,
        5950: 4890682017.727416,
        5960: 4865238964.723016,
        5970: 4839956290.249767,
        5980: 4814832829.659088,
        5990: 4789867427.620043,
        6000: 4765058938.041453,
        6010: 4740406223.994655,
        6020: 4715908157.63684,
        6030: 4691563620.135029,
        6040: 4667371501.59066,
        6050: 4643330700.964742,
        6060: 4619440126.003653,
        6070: 4595698693.165505,
        6080: 4572105327.547136,
        6090: 4548658962.811638,
        6100: 4525358541.116539,
        6110: 4502203013.042508,
        6120: 4479191337.522691,
        6130: 4456322481.772575,
        6140: 4433595421.220465,
        6150: 4411009139.438509,
        6160: 4388562628.07429,
        6170: 4366254886.7829895,
        6180: 4344084923.160104,
        6190: 4322051752.674704,
        6200: 4300154398.603272,
        6210: 4278391891.9640684,
        6220: 4256763271.4520264,
        6230: 4235267583.374232,
        6240: 4213903881.5858974,
        6250: 4192671227.426893,
        6260: 4171568689.658787,
        6270: 4150595344.4024453,
        6280: 4129750275.0761056,
        6290: 4109032572.3340096,
        6300: 4088441334.0055494,
        6310: 4067975665.0348797,
        6320: 4047634677.4210916,
        6330: 4027417490.1588655,
        6340: 4007323229.1796246,
        6350: 3987351027.293179,
        6360: 3967500024.1298876,
        6370: 3947769366.083272,
        6380: 3928158206.253152,
        6390: 3908665704.389244,
        6400: 3889291026.8352327,
        6410: 3870033346.4733367,
        6420: 3850891842.6693296,
        6430: 3831865701.218036,
        6440: 3812954114.2892914,
        6450: 3794156280.37436,
        6460: 3775471404.2328115,
        6470: 3756898696.8398485,
        6480: 3738437375.334096,
        6490: 3720086662.965818,
        6500: 3701845789.0455947,
        6510: 3683713988.8934264,
        6520: 3665690503.7882915,
        6530: 3647774580.918111,
        6540: 3629965473.3301706,
        6550: 3612262439.8819456,
        6560: 3594664745.192371,
        6570: 3577171659.593507,
        6580: 3559782459.082633,
        6590: 3542496425.2747626,
        6600: 3525312845.355553,
        6610: 3508231012.034624,
        6620: 3491250223.4992895,
        6630: 3474369783.368674,
        6640: 3457589000.648234,
        6650: 3440907189.68468,
        6660: 3424323670.121268,
        6670: 3407837766.853492,
        6680: 3391448809.985163,
        6690: 3375156134.7848577,
        6700: 3358959081.642748,
        6710: 3342856996.02781,
        6720: 3326849228.4453974,
        6730: 3310935134.395184,
        6740: 3295114074.32948,
        6750: 3279385413.6118903,
        6760: 3263748522.476351,
        6770: 3248202775.986518,
        6780: 3232747553.9954944,
        6790: 3217382241.1059265,
        6800: 3202106226.6304355,
        6810: 3186918904.552398,
        6820: 3171819673.487058,
        6830: 3156807936.6429987,
        6840: 3141883101.783921,
        6850: 3127044581.1907773,
        6860: 3112291791.6242247,
        6870: 3097624154.287404,
        6880: 3083041094.78905,
        6890: 3068542043.106918,
        6900: 3054126433.551525,
        6910: 3039793704.7302175,
        6920: 3025543299.51154,
        6930: 3011374664.989925,
        6940: 2997287252.4506803,
        6950: 2983280517.3352904,
        6960: 2969353919.207016,
        6970: 2955506921.7167926,
        6980: 2941738992.569427,
        6990: 2928049603.490089,
        7000: 2914438230.191099,
        7010: 2900904352.338997,
        7020: 2887447453.521899,
        7030: 2874067021.2171516,
        7040: 2860762546.7592473,
        7050: 2847533525.3080435,
        7060: 2834379455.817233,
        7070: 2821299841.003111,
        7080: 2808294187.313599,
        7090: 2795362004.897555,
        7100: 2782502807.574328,
        7110: 2769716112.8036,
        7120: 2757001441.6554785,
        7130: 2744358318.780856,
        7140: 2731786272.3820205,
        7150: 2719284834.1835284,
        7160: 2706853539.4033294,
        7170: 2694491926.724141,
        7180: 2682199538.265074,
        7190: 2669975919.5535054,
        7200: 2657820619.497195,
        7210: 2645733190.3566437,
        7220: 2633713187.717698,
        7230: 2621760170.4643836,
        7240: 2609873700.751985,
        7250: 2598053343.980353,
        7260: 2586298668.7674456,
        7270: 2574609246.923101,
        7280: 2562984653.423036,
        7290: 2551424466.3830743,
        7300: 2539928267.0335913,
        7310: 2528495639.6941915,
        7320: 2517126171.748596,
        7330: 2505819453.6197567,
        7340: 2494575078.7451744,
        7350: 2483392643.552453,
        7360: 2472271747.4350348,
        7370: 2461211992.728177,
        7380: 2450212984.6851163,
        7390: 2439274331.4534483,
        7400: 2428395644.0517135,
        7410: 2417576536.3461795,
        7420: 2406816625.0278277,
        7430: 2396115529.5895433,
        7440: 2385472872.3035,
        7450: 2374888278.198735,
        7460: 2364361375.038923,
        7470: 2353891793.3003516,
        7480: 2343479166.1500645,
        7490: 2333123129.42422,
        7500: 2322823321.6066175,
        7510: 2312579383.8074145,
        7520: 2302390959.742034,
        7530: 2292257695.710249,
        7540: 2282179240.5754433,
        7550: 2272155245.744057,
        7560: 2262185365.145211,
        7570: 2252269255.210504,
        7580: 2242406574.8539805,
        7590: 2232596985.452279,
        7600: 2222840150.824943,
        7610: 2213135737.214912,
        7620: 2203483413.2691684,
        7630: 2193882850.01956,
        7640: 2184333720.8637815,
        7650: 2174835701.546527,
        7660: 2165388470.140799,
        7670: 2155991707.029373,
        7680: 2146645094.8864355,
        7690: 2137348318.6593654,
        7700: 2128101065.5506806,
        7710: 2118903025.0001376,
        7720: 2109753888.6669805,
        7730: 2100653350.4123495,
        7740: 2091601106.2818317,
        7750: 2082596854.4881697,
        7760: 2073640295.39411,
        7770: 2064731131.4954088,
        7780: 2055869067.4039705,
        7790: 2047053809.831138,
        7800: 2038285067.5711288,
        7810: 2029562551.484602,
        7820: 2020885974.482379,
        7830: 2012255051.5092869,
        7840: 2003669499.5281606,
        7850: 1995129037.5039594,
        7860: 1986633386.3880334,
        7870: 1978182269.1025212,
        7880: 1969775410.524878,
        7890: 1961412537.4725332,
        7900: 1953093378.6876884,
        7910: 1944817664.822232,
        7920: 1936585128.4227922,
        7930: 1928395503.91591,
        7940: 1920248527.593343,
        7950: 1912143937.5974925,
        7960: 1904081473.9069514,
        7970: 1896060878.3221788,
        7980: 1888081894.451292,
        7990: 1880144267.695984,
        8000: 1872247745.2375557,
        8010: 1864392076.0230708,
        8020: 1856577010.7516174,
        8030: 1848802301.8607001,
        8040: 1841067703.5127382,
        8050: 1833372971.5816817,
        8060: 1825717863.6397338,
        8070: 1818102138.9441948,
        8080: 1810525558.4244092,
        8090: 1802987884.6688309,
        8100: 1795488881.912185,
        8110: 1788028316.0227504,
        8120: 1780605954.4897437,
        8130: 1773221566.410806,
        8140: 1765874922.4796042,
        8150: 1758565794.9735253,
        8160: 1751293957.7414834,
        8170: 1744059186.1918213,
        8180: 1736861257.2803211,
        8190: 1729699949.498313,
        8200: 1722575042.8608754,
        8210: 1715486318.895146,
        8220: 1708433560.6287253,
        8230: 1701416552.5781753,
        8240: 1694435080.73762,
        8250: 1687488932.567433,
        8260: 1680577896.9830265,
        8270: 1673701764.3437316,
        8280: 1666860326.4417727,
        8290: 1660053376.4913275,
        8300: 1653280709.1176896,
        8310: 1646542120.3465087,
        8320: 1639837407.593129,
        8330: 1633166369.652017,
        8340: 1626528806.6862679,
        8350: 1619924520.2172098,
        8360: 1613353313.1140862,
        8370: 1606814989.5838344,
        8380: 1600309355.160931,
        8390: 1593836216.6973429,
        8400: 1587395382.352544,
        8410: 1580986661.5836267,
        8420: 1574609865.1354856,
        8430: 1568264805.0310926,
        8440: 1561951294.5618398,
        8450: 1555669148.277977,
        8460: 1549418181.9791143,
        8470: 1543198212.7048109,
        8480: 1537009058.7252402,
        8490: 1530850539.531933,
        8500: 1524722475.8285925,
        8510: 1518624689.5219884,
        8520: 1512557003.7129245,
        8530: 1506519242.6872847,
        8540: 1500511231.9071462,
        8550: 1494532798.001968,
        8560: 1488583768.759856,
        8570: 1482663973.1188912,
        8580: 1476773241.1585383,
        8590: 1470911404.091118,
        8600: 1465078294.253352,
        8610: 1459273745.0979764,
        8620: 1453497591.1854227,
        8630: 1447749668.1755724,
        8640: 1442029812.819568,
        8650: 1436337862.9517019,
        8660: 1430673657.4813666,
        8670: 1425037036.385069,
        8680: 1419427840.6985118,
        8690: 1413845912.5087411,
        8700: 1408291094.9463544,
        8710: 1402763232.1777725,
        8720: 1397262169.3975794,
        8730: 1391787752.820915,
        8740: 1386339829.6759386,
        8750: 1380918248.1963468,
        8760: 1375522857.613959,
        8770: 1370153508.1513536,
        8780: 1364810051.014575,
        8790: 1359492338.3858883,
        8800: 1354200223.4166043,
        8810: 1348933560.2199485,
        8820: 1343692203.8640027,
        8830: 1338476010.3646917,
        8840: 1333284836.678834,
        8850: 1328118540.6972442,
        8860: 1322976981.2378917,
        8870: 1317860018.0391185,
        8880: 1312767511.7529042,
        8890: 1307699323.9381938,
        8900: 1302655317.05427,
        8910: 1297635354.454185,
        8920: 1292639300.3782473,
        8930: 1287667019.9475496,
        8940: 1282718379.1575615,
        8950: 1277793244.8717644,
        8960: 1272891484.8153434,
        8970: 1268012967.5689256,
        8980: 1263157562.5623684,
        8990: 1258325140.0686018,
        9000: 1253515571.1975148,
        9010: 1248728727.8898933,
        9020: 1243964482.9114063,
        9030: 1239222709.8466387,
        9040: 1234503283.093172,
        9050: 1229806077.8557148,
        9060: 1225130970.1402736,
        9070: 1220477836.7483785,
        9080: 1215846555.2713478,
        9090: 1211237004.0846004,
        9100: 1206649062.3420136,
        9110: 1202082609.9703267,
        9120: 1197537527.663587,
        9130: 1193013696.87764,
        9140: 1188510999.8246655,
        9150: 1184029319.4677522,
        9160: 1179568539.5155199,
        9170: 1175128544.416784,
        9180: 1170709219.3552573,
        9190: 1166310450.2442973,
        9200: 1161932123.7216964,
        9210: 1157574127.144509,
        9220: 1153236348.5839226,
        9230: 1148918676.8201668,
        9240: 1144621001.3374648,
        9250: 1140343212.3190222,
        9260: 1136085200.6420567,
        9270: 1131846857.8728657,
        9280: 1127628076.2619343,
        9290: 1123428748.7390795,
        9300: 1119248768.9086337,
        9310: 1115088031.0446646,
        9320: 1110946430.086236,
        9330: 1106823861.6326993,
        9340: 1102720221.93903,
        9350: 1098635407.9111915,
        9360: 1094569317.1015427,
        9370: 1090521847.7042768,
        9380: 1086492898.5508993,
        9390: 1082482369.1057355,
        9400: 1078490159.4614797,
        9410: 1074516170.3347733,
        9420: 1070560303.0618236,
        9430: 1066622459.5940468,
        9440: 1062702542.4937555,
        9450: 1058800454.9298736,
        9460: 1054916100.673683,
        9470: 1051049384.0946108,
        9480: 1047200210.1560388,
        9490: 1043368484.4111552,
        9500: 1039554112.9988303,
        9510: 1035757002.6395295,
        9520: 1031977060.6312561,
        9530: 1028214194.845524,
        9540: 1024468313.7233624,
        9550: 1020739326.2713544,
        9560: 1017027142.057698,
        9570: 1013331671.2083074,
        9580: 1009652824.402937,
        9590: 1005990512.8713367,
        9600: 1002344648.3894413,
        9610: 998715143.27558,
        9620: 995101910.3867251,
        9630: 991504863.1147605,
        9640: 987923915.3827869,
        9650: 984358981.6414477,
        9660: 980809976.8652887,
        9670: 977276816.5491424,
        9680: 973759416.7045414,
        9690: 970257693.8561577,
        9700: 966771565.0382711,
        9710: 963300947.791263,
        9720: 959845760.1581377,
        9730: 956405920.6810682,
        9740: 952981348.3979728,
        9750: 949571962.8391118,
        9760: 946177684.0237142,
        9770: 942798432.45663,
        9780: 939434129.1250037,
        9790: 936084695.4949789,
        9800: 932750053.5084231,
        9810: 929430125.5796785,
        9820: 926124834.592341,
        9830: 922834103.8960564,
        9840: 919557857.3033476,
        9850: 916296019.0864621,
        9860: 913048513.9742442,
        9870: 909815267.1490294,
        9880: 906596204.2435669,
        9890: 903391251.3379576,
        9900: 900200334.9566219,
        9910: 897023382.0652865,
        9920: 893860320.0679966,
        9930: 890711076.8041475,
        9940: 887575580.5455389,
        9950: 884453759.9934568,
        9960: 881345544.2757663,
        9970: 878250862.9440376,
        9980: 875169645.9706867,
        9990: 872101823.7461381},
    5000: {
        10: 1.2686657620556471e-101,
        20: 1.2147510901468522e-40,
        30: 1.0784359438119513e-20,
        40: 6.644806632888718e-11,
        50: 3.8597687701104664e-05,
        60: 0.22719968663713835,
        70: 99.35995854036662,
        80: 8687.701348124925,
        90: 262344.8542223218,
        100: 3790262.780185453,
        110: 32197359.716756683,
        120: 184348422.5644144,
        130: 781511147.7798123,
        140: 2622251176.8197403,
        150: 7310813379.301067,
        160: 17560673506.94481,
        170: 37355495596.31305,
        180: 71885904020.19241,
        190: 127249248362.55402,
        200: 209965815507.5334,
        210: 326404472526.7038,
        220: 482217548459.91327,
        230: 681866434701.7426,
        240: 928288522250.5859,
        250: 1222724003873.0156,
        260: 1564694926272.392,
        270: 1952111619606.5203,
        280: 2381473138872.0713,
        290: 2848126857131.9497,
        300: 3346555559183.1343,
        310: 3870666158905.362,
        320: 4414060841196.303,
        330: 4970277890013.342,
        340: 5532995066240.421,
        350: 6096192864406.664,
        360: 6654278270641.816,
        370: 7202171873625.862,
        380: 7735362522399.224,
        390: 8249934375727.928,
        400: 8742571332991.204,
        410: 9210543636689.166,
        420: 9651681021262.902,
        430: 10064336251055.02,
        440: 10447342313778.72,
        450: 10799965964060.963,
        460: 11121859775912.46,
        470: 11413014381374.893,
        480: 11673712153465.965,
        490: 11904483236613.303,
        500: 12106064534472.861,
        510: 12279362028244.258,
        520: 12425416611891.379,
        530: 12545373487161.81,
        540: 12640455054190.627,
        550: 12711937156432.8,
        560: 12761128485989.89,
        570: 12789352922088.682,
        580: 12797934557260.469,
        590: 12788185159078.791,
        600: 12761393817184.904,
        610: 12718818533367.135,
        620: 12661679524746.469,
        630: 12591154025150.178,
        640: 12508372386361.768,
        650: 12414415298238.336,
        660: 12310311964027.225,
        670: 12197039084123.23,
        680: 12075520517657.367,
        690: 11946627506484.299,
        700: 11811179360210.418,
        710: 11669944513816.889,
        720: 11523641881168.229,
        730: 11372942438278.482,
        740: 11218470979680.795,
        750: 11060807999673.422,
        760: 10900491657668.412,
        770: 10738019793425.453,
        780: 10573851963691.01,
        790: 10408411476759.814,
        800: 10242087405806.607,
        810: 10075236565571.938,
        820: 9908185440192.857,
        830: 9741232052709.107,
        840: 9574647769103.143,
        850: 9408679031700.016,
        860: 9243549018405.45,
        870: 9079459225639.514,
        880: 8916590973964.725,
        890: 8755106836345.099,
        900: 8595151989733.47,
        910: 8436855491295.577,
        920: 8280331481061.894,
        930: 8125680313171.294,
        940: 7972989618152.377,
        950: 7822335298891.998,
        960: 7673782463079.693,
        970: 7527386295002.167,
        980: 7383192869602.831,
        990: 7241239911726.184,
        1000: 7101557503441.967,
        1010: 6964168742295.949,
        1020: 6829090353267.201,
        1030: 6696333257130.915,
        1040: 6565903097833.766,
        1050: 6437800731389.404,
        1060: 6312022678696.728,
        1070: 6188561544575.374,
        1080: 6067406405203.46,
        1090: 5948543166032.418,
        1100: 5831954892145.251,
        1110: 5717622112917.179,
        1120: 5605523102733.317,
        1130: 5495634139416.433,
        1140: 5387929741919.919,
        1150: 5282382888746.81,
        1160: 5178965218465.419,
        1170: 5077647213605.944,
        1180: 4978398369140.238,
        1190: 4881187346669.046,
        1200: 4785982115367.064,
        1210: 4692750080666.377,
        1220: 4601458201592.762,
        1230: 4512073097607.392,
        1240: 4424561145747.846,
        1250: 4338888568807.477,
        1260: 4255021515240.621,
        1270: 4172926131432.8413,
        1280: 4092568626930.067,
        1290: 4013915333178.263,
        1300: 3936932756285.639,
        1310: 3861587624282.4478,
        1320: 3787846929318.995,
        1330: 3715677965210.2603,
        1340: 3645048360705.616,
        1350: 3575926108834.145,
        1360: 3508279592650.1562,
        1370: 3442077607679.249,
        1380: 3377289381342.781,
        1390: 3313884589617.787,
        1400: 3251833371169.8276,
        1410: 3191106339178.3506,
        1420: 3131674591057.2827,
        1430: 3073509716258.0884,
        1440: 3016583802328.0996,
        1450: 2960869439383.516,
        1460: 2906339723144.103,
        1470: 2852968256665.0913,
        1480: 2800729150891.1646,
        1490: 2749597024147.482,
        1500: 2699547000673.642,
        1510: 2650554708297.928,
        1520: 2602596275341.4136,
        1530: 2555648326834.2305,
        1540: 2509687980119.586,
        1550: 2464692839914.93,
        1560: 2420640992893.9395,
        1570: 2377511001847.645,
        1580: 2335281899478.197,
        1590: 2293933181874.167,
        1600: 2253444801712.1387,
        1610: 2213797161225.487,
        1620: 2174971104977.6362,
        1630: 2136947912473.8599,
        1640: 2099709290642.587,
        1650: 2063237366214.4482,
        1660: 2027514678024.622,
        1670: 1992524169261.78,
        1680: 1958249179684.6357,
        1690: 1924673437825.1328,
        1700: 1891781053195.4666,
        1710: 1859556508514.3577,
        1720: 1827984651966.524,
        1730: 1797050689507.7334,
        1740: 1766740177226.6138,
        1750: 1737039013773.0713,
        1760: 1707933432862.1335,
        1770: 1679409995860.9387,
        1780: 1651455584465.7104,
        1790: 1624057393474.6475,
        1800: 1597202923661.9087,
        1810: 1570879974757.1235,
        1820: 1545076638534.2153,
        1830: 1519781292012.7205,
        1840: 1494982590774.2278,
        1850: 1470669462396.0896,
        1860: 1446831100004.0623,
        1870: 1423456955945.166,
        1880: 1400536735581.6335,
        1890: 1378060391206.5276,
        1900: 1356018116081.24,
        1910: 1334400338594.88,
        1920: 1313197716545.2458,
        1930: 1292401131540.889,
        1940: 1272001683523.5566,
        1950: 1251990685410.1516,
        1960: 1232359657853.1377,
        1970: 1213100324118.2393,
        1980: 1194204605078.0886,
        1990: 1175664614320.4429,
        2000: 1157472653369.4233,
        2010: 1139621207018.2,
        2020: 1122102938771.4343,
        2030: 1104910686395.7659,
        2040: 1088037457576.5522,
        2050: 1071476425679.037,
        2060: 1055220925612.0895,
        2070: 1039264449792.6287,
        2080: 1023600644208.824,
        2090: 1008223304580.1523,
        2100: 993126372612.3845,
        2110: 978303932345.5569,
        2120: 963750206593.0055,
        2130: 949459553469.5254,
        2140: 935426463006.7385,
        2150: 921645553853.7559,
        2160: 908111570061.2391,
        2170: 894819377946.984,
        2180: 881763963041.1619,
        2190: 868940427109.386,
        2200: 856343985251.7738,
        2210: 843969963076.2278,
        2220: 831813793944.1587,
        2230: 819871016286.9135,
        2240: 808137270991.1938,
        2250: 796608298851.7905,
        2260: 785279938089.9662,
        2270: 774148121935.8751,
        2280: 763208876273.4172,
        2290: 752458317345.967,
        2300: 741892649521.4453,
        2310: 731508163115.2296,
        2320: 721301232269.4397,
        2330: 711268312887.1467,
        2340: 701405940620.1116,
        2350: 691710728908.6665,
        2360: 682179367072.3943,
        2370: 672808618450.2952,
        2380: 663595318589.142,
        2390: 654536373478.7831,
        2400: 645628757833.1561,
        2410: 636869513415.8188,
        2420: 628255747408.8237,
        2430: 619784630823.8049,
        2440: 611453396954.155,
        2450: 603259339867.2135,
        2460: 595199812935.4044,
        2470: 587272227405.2936,
        2480: 579474051003.5607,
        2490: 571802806578.9021,
        2500: 564256070778.9153,
        2510: 556831472761.0271,
        2520: 549526692936.5657,
        2530: 542339461747.0902,
        2540: 535267558472.11646,
        2550: 528308810067.40466,
        2560: 521461090032.9903,
        2570: 514722317310.1672,
        2580: 508090455206.6467,
        2590: 501563510349.1434,
        2600: 495139531662.6506,
        2610: 488816609375.69855,
        2620: 482592874050.8956,
        2630: 476466495640.0836,
        2640: 470435682563.44073,
        2650: 464498680811.9054,
        2660: 458653773072.28595,
        2670: 452899277874.46295,
        2680: 447233548760.0843,
        2690: 441654973472.18823,
        2700: 436161973165.1944,
        2710: 430753001634.71954,
        2720: 425426544566.6931,
        2730: 420181118805.2557,
        2740: 415015271638.9474,
        2750: 409927580104.6956,
        2760: 404916650309.13214,
        2770: 399981116766.7811,
        2780: 395119641754.66956,
        2790: 390330914682.9283,
        2800: 385613651480.9595,
        2810: 380966593998.75977,
        2820: 376388509422.9995,
        2830: 371878189707.4722,
        2840: 367434451017.5299,
        2850: 363056133188.14465,
        2860: 358742099195.2324,
        2870: 354491234639.8949,
        2880: 350302447245.23926,
        2890: 346174666365.4502,
        2900: 342106842506.7883,
        2910: 338097946860.2099,
        2920: 334146970845.30255,
        2930: 330252925665.24115,
        2940: 326414841872.4821,
        2950: 322631768944.9105,
        2960: 318902774872.1773,
        2970: 315226945751.9562,
        2980: 311603385395.86914,
        2990: 308031214944.82666,
        3000: 304509572493.5437,
        3010: 301037612723.99365,
        3020: 297614506547.5716,
        3030: 294239440755.743,
        3040: 290911617678.96014,
        3050: 287630254853.6374,
        3060: 284394584696.9755,
        3070: 281203854189.4401,
        3080: 278057324564.6945,
        3090: 274954271006.801,
        3100: 271893982354.50558,
        3110: 268875760812.4266,
        3120: 265898921668.97076,
        3130: 262962793020.8118,
        3140: 260066715503.76102,
        3150: 257210042029.87204,
        3160: 254392137530.6237,
        3170: 251612378706.02545,
        3180: 248870153779.50146,
        3190: 246164862258.40512,
        3200: 243495914700.02747,
        3210: 240862732482.95795,
        3220: 238264747583.66956,
        3230: 235701402358.19556,
        3240: 233172149328.77216,
        3250: 230676450975.32635,
        3260: 228213779531.68597,
        3270: 225783616786.39868,
        3280: 223385453888.0458,
        3290: 221018791154.9399,
        3300: 218683137889.09836,
        3310: 216378012194.39075,
        3320: 214102940798.75565,
        3330: 211857458880.389,
        3340: 209641109897.8069,
        3350: 207453445423.689,
        3360: 205294024982.41098,
        3370: 203162415891.17676,
        3380: 201058193104.66406,
        3390: 198980939063.09628,
        3400: 196930243543.66254,
        3410: 194905703515.20123,
        3420: 192906922996.07217,
        3430: 190933512915.138,
        3440: 188985090975.78333,
        3450: 187061281522.898,
        3460: 185161715412.75366,
        3470: 183286029885.70557,
        3480: 181433868441.65216,
        3490: 179604880718.18805,
        3500: 177798722371.38562,
        3510: 176015054959.14432,
        3520: 174253545827.04602,
        3530: 172513867996.6598,
        3540: 170795700056.23572,
        3550: 169098726053.7347,
        3560: 167422635392.13788,
        3570: 165767122726.98288,
        3580: 164131887866.0759,
        3590: 162516635671.3286,
        3600: 160921075962.66983,
        3610: 159344923423.9856,
        3620: 157787897511.03925,
        3630: 156249722361.3274,
        3640: 154730126705.82568,
        3650: 153228843782.58246,
        3660: 151745611252.11664,
        3670: 150280171114.57993,
        3680: 148832269628.64175,
        3690: 147401657232.05865,
        3700: 145988088463.88965,
        3710: 144591321888.31934,
        3720: 143211120020.05386,
        3730: 141847249251.25342,
        3740: 140499479779.9655,
        3750: 139167585540.02707,
        3760: 137851344132.40192,
        3770: 136550536757.9202,
        3780: 135264948151.38937,
        3790: 133994366517.04663,
        3800: 132738583465.32109,
        3810: 131497393950.87877,
        3820: 130270596211.91989,
        3830: 129057991710.70149,
        3840: 127859385075.25932,
        3850: 126674584042.30023,
        3860: 125503399401.24103,
        3870: 124345644939.36835,
        3880: 123201137388.0938,
        3890: 122069696370.2821,
        3900: 120951144348.62692,
        3910: 119845306575.05255,
        3920: 118752011041.11945,
        3930: 117671088429.41063,
        3940: 116602372065.87828,
        3950: 115545697873.12955,
        3960: 114500904324.632,
        3970: 113467832399.81737,
        3980: 112446325540.06548,
        3990: 111436229605.54878,
        4000: 110437392832.91953,
        4010: 109449665793.82085,
        4020: 108472901354.2045,
        4030: 107506954634.43825,
        4040: 106551682970.18565,
        4050: 105606945874.04169,
        4060: 104672604997.90916,
        4070: 103748524096.09862,
        4080: 102834568989.13812,
        4090: 101930607528.2763,
        4100: 101036509560.66588,
        4110: 100152146895.2109,
        4120: 99277393269.06654,
        4130: 98412124314.77563,
        4140: 97556217528.02962,
        4150: 96709552236.04074,
        4160: 95872009566.51228,
        4170: 95043472417.19525,
        4180: 94223825426.01828,
        4190: 93412954941.78036,
        4200: 92610748995.3926,
        4210: 91817097271.65967,
        4220: 91031891081.58914,
        4230: 90255023335.2173,
        4240: 89486388514.9414,
        4250: 88725882649.34814,
        4260: 87973403287.52753,
        4270: 87228849473.86292,
        4280: 86492121723.28716,
        4290: 85763121996.99503,
        4300: 85041753678.60371,
        4310: 84327921550.75113,
        4320: 83621531772.12389,
        4330: 82922491854.906,
        4340: 82230710642.63997,
        4350: 81546098288.49165,
        4360: 80868566233.91116,
        4370: 80198027187.68208,
        4380: 79534395105.34982,
        4390: 78877585169.02364,
        4400: 78227513767.54327,
        4410: 77584098477.00354,
        4420: 76947258041.62976,
        4430: 76316912354.99683,
        4440: 75692982441.58543,
        4450: 75075390438.66876,
        4460: 74464059578.52257,
        4470: 73858914170.95303,
        4480: 73259879586.13588,
        4490: 72666882237.76039,
        4500: 72079849566.47264,
        4510: 71498710023.6118,
        4520: 70923393055.23444,
        4530: 70353829086.42033,
        4540: 69789949505.8549,
        4550: 69231686650.68289,
        4560: 68678973791.62749,
        4570: 68131745118.37061,
        4580: 67589935725.18856,
        4590: 67053481596.838524,
        4600: 66522319594.69096,
        4610: 65996387443.103134,
        4620: 65475623716.029305,
        4630: 64959967823.86305,
        4640: 64449360000.5067,
        4650: 63943741290.66477,
        4660: 63443053537.35577,
        4670: 62947239369.639244,
        4680: 62456242190.55349,
        4690: 61970006165.259926,
        4700: 61488476209.39056,
        4710: 61011597977.59426,
        4720: 60539317852.2787,
        4730: 60071582932.54338,
        4740: 59608341023.301186,
        4750: 59149540624.584114,
        4760: 58695130921.030365,
        4770: 58245061771.548706,
        4780: 57799283699.15755,
        4790: 57357747880.994965,
        4800: 56920406138.49664,
        4810: 56487210927.738846,
        4820: 56058115329.94276,
        4830: 55633073042.138016,
        4840: 55212038367.981926,
        4850: 54794966208.73153,
        4860: 54381812054.366104,
        4870: 53972531974.85687,
        4880: 53567082611.58146,
        4890: 53165421168.880554,
        4900: 52767505405.753784,
        4910: 52373293627.69282,
        4920: 51982744678.648544,
        4930: 51595817933.130684,
        4940: 51212473288.436554,
        4950: 50832671157.00749,
        4960: 50456372458.90984,
        4970: 50083538614.43902,
        4980: 49714131536.84381,
        4990: 49348113625.16898,
        5000: 48985447757.21427,
        5010: 48626097282.60742,
        5020: 48270026015.98913,
        5030: 47917198230.30836,
        5040: 47567578650.22543,
        5050: 47221132445.62167,
        5060: 46877825225.21295,
        5070: 46537623030.266045,
        5080: 46200492328.41531,
        5090: 45866400007.57851,
        5100: 45535313369.96948,
        5110: 45207200126.206375,
        5120: 44882028389.513565,
        5130: 44559766670.015564,
        5140: 44240383869.1215,
        5150: 43923849273.99839,
        5160: 43610132552.13191,
        5170: 43299203745.97253,
        5180: 42991033267.66651,
        5190: 42685591893.869286,
        5200: 42382850760.640526,
        5210: 42082781358.41902,
        5220: 41785355527.076294,
        5230: 41490545451.047264,
        5240: 41198323654.53716,
        5250: 40908662996.802605,
        5260: 40621536667.50639,
        5270: 40336918182.14405,
        5280: 40054781377.54157,
        5290: 39775100407.422356,
        5300: 39497849738.04281,
        5310: 39223004143.895294,
        5320: 38950538703.476845,
        5330: 38680428795.12345,
        5340: 38412650092.907455,
        5350: 38147178562.59851,
        5360: 37883990457.68579,
        5370: 37623062315.461105,
        5380: 37364370953.161644,
        5390: 37107893464.1714,
        5400: 36853607214.280266,
        5410: 36601489837.99988,
        5420: 36351519234.9352,
        5430: 36103673566.21095,
        5440: 35857931250.9519,
        5450: 35614270962.816345,
        5460: 35372671626.58151,
        5470: 35133112414.78037,
        5480: 34895572744.38887,
        5490: 34660032273.56262,
        5500: 34426470898.422516,
        5510: 34194868749.88825,
        5520: 33965206190.558884,
        5530: 33737463811.64001,
        5540: 33511622429.916405,
        5550: 33287663084.76958,
        5560: 33065567035.23951,
        5570: 32845315757.12979,
        5580: 32626890940.15548,
        5590: 32410274485.132885,
        5600: 32195448501.21094,
        5610: 31982395303.14282,
        5620: 31771097408.59804,
        5630: 31561537535.51344,
        5640: 31353698599.48328,
        5650: 31147563711.187176,
        5660: 30943116173.855545,
        5670: 30740339480.77204,
        5680: 30539217312.812202,
        5690: 30339733536.017624,
        5700: 30141872199.205563,
        5710: 29945617531.61292,
        5720: 29750953940.574196,
        5730: 29557866009.23305,
        5740: 29366338494.286667,
        5750: 29176356323.762573,
        5760: 28987904594.82738,
        5770: 28800968571.626926,
        5780: 28615533683.157146,
        5790: 28431585521.165615,
        5800: 28249109838.08271,
        5810: 28068092544.982513,
        5820: 27888519709.572456,
        5830: 27710377554.211685,
        5840: 27533652453.957344,
        5850: 27358330934.638615,
        5860: 27184399670.95796,
        5870: 27011845484.61909,
        5880: 26840655342.48128,
        5890: 26670816354.739784,
        5900: 26502315773.13165,
        5910: 26335140989.166763,
        5920: 26169279532.383606,
        5930: 26004719068.629498,
        5940: 25841447398.364693,
        5950: 25679452454.990303,
        5960: 25518722303.19925,
        5970: 25359245137.350346,
        5980: 25201009279.86481,
        5990: 25044003179.64506,
        6000: 24888215410.515327,
        6010: 24733634669.683846,
        6020: 24580249776.226246,
        6030: 24428049669.589764,
        6040: 24277023408.118217,
        6050: 24127160167.59693,
        6060: 23978449239.817883,
        6070: 23830880031.164345,
        6080: 23684442061.214924,
        6090: 23539124961.36667,
        6100: 23394918473.476845,
        6110: 23251812448.523354,
        6120: 23109796845.283222,
        6130: 22968861729.02907,
        6140: 22828997270.243256,
        6150: 22690193743.34938,
        6160: 22552441525.460987,
        6170: 22415731095.14706,
        6180: 22280053031.214237,
        6190: 22145398011.505363,
        6200: 22011756811.71411,
        6210: 21879120304.215694,
        6220: 21747479456.91299,
        6230: 21616825332.09829,
        6240: 21487149085.33015,
        6250: 21358441964.325264,
        6260: 21230695307.865047,
        6270: 21103900544.716824,
        6280: 20978049192.569237,
        6290: 20853132856.981873,
        6300: 20729143230.3488,
        6310: 20606072090.875645,
        6320: 20483911301.57037,
        6330: 20362652809.247257,
        6340: 20242288643.544006,
        6350: 20122810915.951782,
        6360: 20004211818.857975,
        6370: 19886483624.601467,
        6380: 19769618684.540356,
        6390: 19653609428.13182,
        6400: 19538448362.023926,
        6410: 19424128069.159405,
        6420: 19310641207.890938,
        6430: 19197980511.1081,
        6440: 19086138785.375534,
        6450: 18975108910.08229,
        6460: 18864883836.602226,
        6470: 18755456587.465206,
        6480: 18646820255.5391,
        6490: 18538968003.222134,
        6500: 18431893061.645832,
        6510: 18325588729.888092,
        6520: 18220048374.19636,
        6530: 18115265427.220837,
        6540: 18011233387.257416,
        6550: 17907945817.500298,
        6560: 17805396345.304287,
        6570: 17703578661.456223,
        6580: 17602486519.455936,
        6590: 17502113734.806225,
        6600: 17402454184.311867,
        6610: 17303501805.387497,
        6620: 17205250595.374268,
        6630: 17107694610.865137,
        6640: 17010827967.038687,
        6650: 16914644837.001316,
        6660: 16819139451.137745,
        6670: 16724306096.469625,
        6680: 16630139116.022264,
        6690: 16536632908.199291,
        6700: 16443781926.165087,
        6710: 16351580677.234982,
        6720: 16260023722.273083,
        6730: 16169105675.09755,
        6740: 16078821201.893366,
        6750: 15989165020.632305,
        6760: 15900131900.50017,
        6770: 15811716661.331139,
        6780: 15723914173.04906,
        6790: 15636719355.11573,
        6800: 15550127175.985933,
        6810: 15464132652.56921,
        6820: 15378730849.69827,
        6830: 15293916879.603968,
        6840: 15209685901.396633,
        6850: 15126033120.55392,
        6860: 15042953788.414762,
        6870: 14960443201.679636,
        6880: 14878496701.916914,
        6890: 14797109675.075182,
        6900: 14716277551.001572,
        6910: 14635995802.96592,
        6920: 14556259947.19071,
        6930: 14477065542.38673,
        6940: 14398408189.294363,
        6950: 14320283530.23042,
        6960: 14242687248.64045,
        6970: 14165615068.656477,
        6980: 14089062754.66006,
        6990: 14013026110.850576,
        7000: 13937500980.818802,
        7010: 13862483247.125475,
        7020: 13787968830.885017,
        7030: 13713953691.354204,
        7040: 13640433825.525787,
        7050: 13567405267.726915,
        7060: 13494864089.222393,
        7070: 13422806397.822674,
        7080: 13351228337.496468,
        7090: 13280126087.988024,
        7100: 13209495864.4389,
        7110: 13139333917.014244,
        7120: 13069636530.533468,
        7130: 13000400024.105392,
        7140: 12931620750.767529,
        7150: 12863295097.12977,
        7160: 12795419483.022211,
        7170: 12727990361.147121,
        7180: 12661004216.734993,
        7190: 12594457567.204674,
        7200: 12528346961.827414,
        7210: 12462668981.394894,
        7220: 12397420237.891138,
        7230: 12332597374.16822,
        7240: 12268197063.625782,
        7250: 12204216009.894276,
        7260: 12140650946.521881,
        7270: 12077498636.665075,
        7280: 12014755872.782787,
        7290: 11952419476.334084,
        7300: 11890486297.479362,
        7310: 11828953214.785019,
        7320: 11767817134.93149,
        7330: 11707074992.424673,
        7340: 11646723749.310667,
        7350: 11586760394.893837,
        7360: 11527181945.458025,
        7370: 11467985443.991049,
        7380: 11409167959.91231,
        7390: 11350726588.803518,
        7400: 11292658452.14252,
        7410: 11234960697.040155,
        7420: 11177630495.98008,
        7430: 11120665046.561586,
        7440: 11064061571.24539,
        7450: 11007817317.102167,
        7460: 10951929555.564102,
        7470: 10896395582.17915,
        7480: 10841212716.368103,
        7490: 10786378301.18442,
        7500: 10731889703.076754,
        7510: 10677744311.65416,
        7520: 10623939539.453909,
        7530: 10570472821.712017,
        7540: 10517341616.136217,
        7550: 10464543402.681568,
        7560: 10412075683.328552,
        7570: 10359935981.86366,
        7580: 10308121843.662422,
        7590: 10256630835.474844,
        7600: 10205460545.213299,
        7610: 10154608581.74268,
        7620: 10104072574.673,
        7630: 10053850174.154211,
        7640: 10003939050.6733,
        7650: 9954336894.85372,
        7660: 9905041417.256962,
        7670: 9856050348.186316,
        7680: 9807361437.49285,
        7690: 9758972454.383488,
        7700: 9710881187.231241,
        7710: 9663085443.387472,
        7720: 9615583048.996273,
        7730: 9568371848.81085,
        7740: 9521449706.011946,
        7750: 9474814502.028217,
        7760: 9428464136.358603,
        7770: 9382396526.396633,
        7780: 9336609607.25663,
        7790: 9291101331.601843,
        7800: 9245869669.474398,
        7810: 9200912608.127157,
        7820: 9156228151.857363,
        7830: 9111814321.84209,
        7840: 9067669155.975517,
        7850: 9023790708.707895,
        7860: 8980177050.886307,
        7870: 8936826269.597143,
        7880: 8893736468.010225,
        7890: 8850905765.224688,
        7900: 8808332296.116455,
        7910: 8766014211.187365,
        7920: 8723949676.415956,
        7930: 8682136873.109787,
        7940: 8640573997.759382,
        7950: 8599259261.893757,
        7960: 8558190891.937402,
        7970: 8517367129.068894,
        7980: 8476786229.080955,
        7990: 8436446462.242002,
        8000: 8396346113.159188,
        8010: 8356483480.642905,
        8020: 8316856877.572674,
        8030: 8277464630.764513,
        8040: 8238305080.839688,
        8050: 8199376582.094843,
        8060: 8160677502.373482,
        8070: 8122206222.93888,
        8080: 8083961138.348247,
        8090: 8045940656.328253,
        8100: 8008143197.651885,
        8110: 7970567196.01654,
        8120: 7933211097.923448,
        8130: 7896073362.55833,
        8140: 7859152461.673321,
        8150: 7822446879.470094,
        8160: 7785955112.48426,
        8170: 7749675669.470907,
        8180: 7713607071.29138,
        8190: 7677747850.801237,
        8200: 7642096552.739325,
        8210: 7606651733.618052,
        8220: 7571411961.61477,
        8230: 7536375816.464317,
        8240: 7501541889.352614,
        8250: 7466908782.81142,
        8260: 7432475110.614135,
        8270: 7398239497.672684,
        8280: 7364200579.9354725,
        8290: 7330357004.28639,
        8300: 7296707428.444834,
        8310: 7263250520.866789,
        8320: 7229984960.6468725,
        8330: 7196909437.421459,
        8340: 7164022651.272709,
        8350: 7131323312.633634,
        8360: 7098810142.194112,
        8370: 7066481870.807879,
        8380: 7034337239.40042,
        8390: 7002374998.877866,
        8400: 6970593910.036729,
        8410: 6938992743.474656,
        8420: 6907570279.502005,
        8430: 6876325308.05434,
        8440: 6845256628.605839,
        8450: 6814363050.08353,
        8460: 6783643390.782424,
        8470: 6753096478.281498,
        8480: 6722721149.360525,
        8490: 6692516249.91772,
        8500: 6662480634.88825,
        8510: 6632613168.163517,
        8520: 6602912722.5113,
        8530: 6573378179.496649,
        8540: 6544008429.403609,
        8550: 6514802371.157696,
        8560: 6485758912.249165,
        8570: 6456876968.657033,
        8580: 6428155464.773866,
        8590: 6399593333.331313,
        8600: 6371189515.326377,
        8610: 6342942959.948405,
        8620: 6314852624.506841,
        8630: 6286917474.359644,
        8640: 6259136482.842451,
        8650: 6231508631.198427,
        8660: 6204032908.508806,
        8670: 6176708311.624117,
        8680: 6149533845.096093,
        8690: 6122508521.110249,
        8700: 6095631359.419105,
        8710: 6068901387.276098,
        8720: 6042317639.37012,
        8730: 6015879157.760693,
        8740: 5989584991.813781,
        8750: 5963434198.13826,
        8760: 5937425840.522955,
        8770: 5911558989.874333,
        8780: 5885832724.154794,
        8790: 5860246128.321541,
        8800: 5834798294.266084,
        8810: 5809488320.75429,
        8820: 5784315313.367036,
        8830: 5759278384.441451,
        8840: 5734376653.012693,
        8850: 5709609244.756312,
        8860: 5684975291.931168,
        8870: 5660473933.322893,
        8880: 5636104314.187895,
        8890: 5611865586.197937,
        8900: 5587756907.385184,
        8910: 5563777442.087823,
        8920: 5539926360.896215,
        8930: 5516202840.599522,
        8940: 5492606064.132888,
        8950: 5469135220.525092,
        8960: 5445789504.846707,
        8970: 5422568118.158787,
        8980: 5399470267.461996,
        8990: 5376495165.646263,
        9000: 5353642031.440869,
        9010: 5330910089.36507,
        9020: 5308298569.679136,
        9030: 5285806708.335879,
        9040: 5263433746.932641,
        9050: 5241178932.663722,
        9060: 5219041518.273275,
        9070: 5197020762.00865,
        9080: 5175115927.5741415,
        9090: 5153326284.0852165,
        9100: 5131651106.02316,
        9110: 5110089673.1901245,
        9120: 5088641270.664635,
        9130: 5067305188.757497,
        9140: 5046080722.968109,
        9150: 5024967173.941204,
        9160: 5003963847.423977,
        9170: 4983070054.223636,
        9180: 4962285110.165319,
        9190: 4941608336.050437,
        9200: 4921039057.615379,
        9210: 4900576605.49061,
        9220: 4880220315.160169,
        9230: 4859969526.9215,
        9240: 4839823585.845712,
        9250: 4819781841.738148,
        9260: 4799843649.099373,
        9270: 4780008367.086485,
        9280: 4760275359.474805,
        9290: 4740643994.61991,
        9300: 4721113645.420031,
        9310: 4701683689.278769,
        9320: 4682353508.068188,
        9330: 4663122488.092229,
        9340: 4643990020.050466,
        9350: 4624955499.002175,
        9360: 4606018324.330781,
        9370: 4587177899.708575,
        9380: 4568433633.061782,
        9390: 4549784936.535959,
        9400: 4531231226.461684,
        9410: 4512771923.320572,
        9420: 4494406451.711605,
        9430: 4476134240.317753,
        9440: 4457954721.87291,
        9450: 4439867333.129134,
        9460: 4421871514.82418,
        9470: 4403966711.649315,
        9480: 4386152372.217459,
        9490: 4368427949.031562,
        9500: 4350792898.453322,
        9510: 4333246680.672157,
        9520: 4315788759.674445,
        9530: 4298418603.213075,
        9540: 4281135682.777252,
        9550: 4263939473.562565,
        9560: 4246829454.4413495,
        9570: 4229805107.9332924,
        9580: 4212865920.176319,
        9590: 4196011380.897724,
        9600: 4179240983.3855844,
        9610: 4162554224.460393,
        9620: 4145950604.44698,
        9630: 4129429627.1466727,
        9640: 4112990799.8096914,
        9650: 4096633633.107815,
        9660: 4080357641.107264,
        9670: 4064162341.2418523,
        9680: 4048047254.286355,
        9690: 4032011904.330116,
        9700: 4016055818.7509093,
        9710: 4000178528.1890006,
        9720: 3984379566.5214653,
        9730: 3968658470.836722,
        9740: 3953014781.4092946,
        9750: 3937448041.674784,
        9760: 3921957798.205082,
        9770: 3906543600.683795,
        9780: 3891205001.881875,
        9790: 3875941557.633469,
        9800: 3860752826.811989,
        9810: 3845638371.306396,
        9820: 3830597755.9976554,
        9830: 3815630548.735455,
        9840: 3800736320.3150773,
        9850: 3785914644.454505,
        9860: 3771165097.771703,
        9870: 3756487259.762127,
        9880: 3741880712.776395,
        9890: 3727345041.998192,
        9900: 3712879835.422315,
        9910: 3698484683.832967,
        9920: 3684159180.7821965,
        9930: 3669902922.5685453,
        9940: 3655715508.215864,
        9950: 3641596539.4523425,
        9960: 3627545620.6896772,
        9970: 3613562359.0024605,
        9980: 3599646364.1077194,
        9990: 3585797248.3446465},
    10000: {
        10: 3.887203488469927e-39,
        20: 2.1263381225243896e-09,
        30: 7.270389972388427,
        40: 278006.4431399976,
        50: 121288408.9659345,
        60: 5899149522.061261,
        70: 83912037658.23169,
        80: 561941552222.2339,
        90: 2300348928646.157,
        100: 6718906096241.068,
        110: 15430961550717.225,
        120: 29705232712018.75,
        130: 50070237640716.55,
        140: 76207140443906.28,
        150: 107089777893860.3,
        160: 141249946918281.7,
        170: 177055842119693.47,
        180: 212936904660538.12,
        190: 247531600716775.16,
        200: 279762282655718.53,
        210: 308853792680412.9,
        220: 334314954040919.06,
        230: 355899512829198.75,
        240: 373558788910910.9,
        250: 387394065103131.56,
        260: 397613331580524.1,
        270: 404494561734100.06,
        280: 408356111551676.5,
        290: 409533905832335.0,
        300: 408364602149916.94,
        310: 405173744791887.0,
        320: 400267916363576.56,
        330: 393929982848871.2,
        340: 386416656565035.75,
        350: 377957739526733.4,
        360: 368756540197383.3,
        370: 358991071349785.44,
        380: 348815733045389.2,
        390: 338363262838112.3,
        400: 327746796985811.44,
        410: 317061934086171.44,
        420: 306388728611300.56,
        430: 295793568588974.4,
        440: 285330911166871.2,
        450: 275044863671471.03,
        460: 264970607393980.6,
        470: 255135667780876.06,
        480: 245561038818550.2,
        490: 236262171827290.6,
        500: 227249840110142.94,
        510: 218530891304550.44,
        520: 210108899130680.5,
        530: 201984725718295.28,
        540: 194157004966510.72,
        550: 186622556548648.03,
        560: 179376739287466.16,
        570: 172413751741437.4,
        580: 165726886990893.4,
        590: 159308747812487.62,
        600: 153151427691746.06,
        610: 147246662450968.38,
        620: 141585956663931.06,
        630: 136160688487699.88,
        640: 130962196061762.14,
        650: 125981848201140.45,
        660: 121211101738207.84,
        670: 116641547542579.72,
        680: 112264946964805.0,
        690: 108073260202968.98,
        700: 104058667877434.48,
        710: 100213586913833.05,
        720: 96530681674499.19,
        730: 93002871140611.3,
        740: 89623332828517.27,
        750: 86385504021556.55,
        760: 83283080810925.23,
        770: 80310015363826.75,
        780: 77460511772606.06,
        790: 74729020783302.3,
        800: 72110233654788.44,
        810: 69599075359284.36,
        820: 67190697300562.78,
        830: 64880469696787.91,
        840: 62663973749908.34,
        850: 60536993702254.93,
        860: 58495508862928.766,
        870: 56535685671251.64,
        880: 54653869851588.27,
        890: 52846578702902.734,
        900: 51110493557181.08,
        910: 49442452433094.89,
        920: 47839442904775.29,
        930: 46298595200129.99,
        940: 44817175538612.27,
        950: 43392579714599.68,
        960: 42022326929447.87,
        970: 40704053872753.81,
        980: 39435509051300.41,
        990: 38214547362498.836,
        1000: 37039124907821.55,
        1010: 35907294040685.9,
        1020: 34817198642449.67,
        1030: 33767069619586.863,
        1040: 32755220614682.367,
        1050: 31780043923596.305,
        1060: 30840006610976.176,
        1070: 29933646816215.633,
        1080: 29059570241959.652,
        1090: 28216446817317.18,
        1100: 27403007528056.76,
        1110: 26618041406213.08,
        1120: 25860392671718.203,
        1130: 25128958018878.918,
        1140: 24422684040749.35,
        1150: 23740564784686.25,
        1160: 23081639432622.07,
        1170: 22444990099843.57,
        1180: 21829739746316.617,
        1190: 21235050194852.543,
        1200: 20660120250660.996,
        1210: 20104183917082.36,
        1220: 19566508702532.96,
        1230: 19046394013932.94,
        1240: 18543169632113.94,
        1250: 18056194264925.29,
        1260: 17584854173970.102,
        1270: 17128561871107.727,
        1280: 16686754881055.69,
        1290: 16258894566612.695,
        1300: 15844465013204.818,
        1310: 15442971969628.66,
        1320: 15053941842030.104,
        1330: 14676920738313.15,
        1340: 14311473560322.697,
        1350: 13957183141286.22,
        1360: 13613649426134.18,
        1370: 13280488692446.316,
        1380: 12957332809892.229,
        1390: 12643828536149.426,
        1400: 12339636847390.703,
        1410: 12044432301535.984,
        1420: 11757902432560.957,
        1430: 11479747174247.438,
        1440: 11209678311847.656,
        1450: 10947418960217.227,
        1460: 10692703067049.682,
        1470: 10445274939919.428,
        1480: 10204888795909.674,
        1490: 9971308332667.812,
        1500: 9744306319793.398,
        1510: 9523664209522.291,
        1520: 9309171765726.725,
        1530: 9100626710303.217,
        1540: 8897834386070.225,
        1550: 8700607435344.179,
        1560: 8508765493407.064,
        1570: 8322134896120.353,
        1580: 8140548400979.958,
        1590: 7963844920944.047,
        1600: 7791869270400.879,
        1610: 7624471922677.426,
        1620: 7461508778520.709,
        1630: 7302840945013.951,
        1640: 7148334524417.466,
        1650: 6997860412451.147,
        1660: 6851294105560.181,
        1670: 6708515516729.82,
        1680: 6569408799437.151,
        1690: 6433862179349.25,
        1700: 6301767793397.201,
        1710: 6173021535874.31,
        1720: 6047522911225.066,
        1730: 5925174893208.117,
        1740: 5805883790132.869,
        1750: 5689559115884.34,
        1760: 5576113466465.473,
        1770: 5465462401799.543,
        1780: 5357524332548.412,
        1790: 5252220411714.429,
        1800: 5149474430805.435,
        1810: 5049212720353.256,
        1820: 4951364054586.422,
        1830: 4855859560067.732,
        1840: 4762632628116.536,
        1850: 4671618830844.469,
        1860: 4582755840641.724,
        1870: 4495983352958.896,
        1880: 4411243012236.861,
        1890: 4328478340844.46,
        1900: 4247634670890.275,
        1910: 4168659078781.4365,
        1920: 4091500322408.2983,
        1930: 4016108780839.713,
        1940: 3942436396419.0015,
        1950: 3870436619156.0615,
        1960: 3800064353315.8223,
        1970: 3731275906108.105,
        1980: 3664028938388.224,
        1990: 3598282417282.095,
        2000: 3533996570653.425,
        2010: 3471132843334.544,
        2020: 3409653855045.941,
        2030: 3349523359933.0933,
        2040: 3290706207652.42,
        2050: 3233168305941.308,
        2060: 3176876584610.1294,
        2070: 3121798960896.9478,
        2080: 3067904306128.393,
        2090: 3015162413632.585,
        2100: 2963543967852.563,
        2110: 2913020514610.8525,
        2120: 2863564432478.125,
        2130: 2815148905200.881,
        2140: 2767747895145.189,
        2150: 2721336117715.319,
        2160: 2675889016707.9644,
        2170: 2631382740564.479,
        2180: 2587794119485.157,
        2190: 2545100643371.1943,
        2200: 2503280440561.385,
        2210: 2462312257332.1523,
        2220: 2422175438130.734,
        2230: 2382849906512.722,
        2240: 2344316146756.348,
        2250: 2306555186127.1245,
        2260: 2269548577767.502,
        2270: 2233278384187.357,
        2280: 2197727161332.0796,
        2290: 2162877943206.034,
        2300: 2128714227030.1108,
        2310: 2095219958912.9377,
        2320: 2062379520016.2144,
        2330: 2030177713195.3845,
        2340: 1998599750097.7117,
        2350: 1967631238700.4978,
        2360: 1937258171272.9185,
        2370: 1907466912745.638,
        2380: 1878244189472.9717,
        2390: 1849577078373.0415,
        2400: 1821452996431.8926,
        2410: 1793859690558.1682,
        2420: 1766785227775.4216,
        2430: 1740217985739.7153,
        2440: 1714146643570.6067,
        2450: 1688560172984.121,
        2460: 1663447829716.7534,
        2470: 1638799145229.9697,
        2480: 1614603918685.115,
        2490: 1590852209179.002,
        2500: 1567534328230.8567,
        2510: 1544640832511.652,
        2520: 1522162516807.2139,
        2530: 1500090407206.8145,
        2540: 1478415754509.29,
        2550: 1457130027839.0234,
        2560: 1436224908464.434,
        2570: 1415692283811.888,
        2580: 1395524241668.2212,
        2590: 1375713064565.3284,
        2600: 1356251224340.4968,
        2610: 1337131376866.4414,
        2620: 1318346356945.1892,
        2630: 1299889173360.2046,
        2640: 1281753004081.3303,
        2650: 1263931191617.3699,
        2660: 1246417238511.259,
        2670: 1229204802973.0386,
        2680: 1212287694645.9453,
        2690: 1195659870501.1726,
        2700: 1179315430856.9685,
        2710: 1163248615517.9265,
        2720: 1147453800030.4607,
        2730: 1131925492050.6084,
        2740: 1116658327820.4438,
        2750: 1101647068749.5205,
        2760: 1086886598097.8843,
        2770: 1072371917757.325,
        2780: 1058098145127.6643,
        2790: 1044060510084.9753,
        2800: 1030254352038.7501,
        2810: 1016675117075.1373,
        2820: 1003318355183.465,
        2830: 990179717563.3774,
        2840: 977254954009.9905,
        2850: 964539910374.578,
        2860: 952030526098.3743,
        2870: 939722831817.1733,
        2880: 927612947034.4762,
        2890: 915697077861.0287,
        2900: 903971514818.6467,
        2910: 892432630706.3186,
        2920: 881076878526.635,
        2930: 869900789470.6547,
        2940: 858900970959.401,
        2950: 848074104740.2128,
        2960: 837416945036.2754,
        2970: 826926316747.6702,
        2980: 816599113702.376,
        2990: 806432296955.674,
        3000: 796422893136.4905,
        3010: 786567992839.2399,
        3020: 776864749059.783,
        3030: 767310375674.1722,
        3040: 757902145958.8776,
        3050: 748637391151.259,
        3060: 739513499049.0631,
        3070: 730527912647.7842,
        3080: 721678128814.7516,
        3090: 712961696998.8556,
        3100: 704376217974.8516,
        3110: 695919342621.2157,
        3120: 687588770730.5648,
        3130: 679382249851.6815,
        3140: 671297574162.2156,
        3150: 663332583371.1616,
        3160: 655485161650.2518,
        3170: 647753236593.4082,
        3180: 640134778203.4552,
        3190: 632627797905.2914,
        3200: 625230347584.7656,
        3210: 617940518652.51,
        3220: 610756441132.0171,
        3230: 603676282771.2712,
        3240: 596698248177.2512,
        3250: 589820577972.6666,
        3260: 583041547974.2836,
        3270: 576359468392.2373,
        3280: 569772683049.7372,
        3290: 563279568622.59,
        3300: 556878533897.9797,
        3310: 550568019051.9766,
        3320: 544346494945.238,
        3330: 538212462436.40784,
        3340: 532164451712.71246,
        3350: 526201021637.2828,
        3360: 520320759112.73865,
        3370: 514522278460.5877,
        3380: 508804220816.00543,
        3390: 503165253537.56995,
        3400: 497604069631.54987,
        3410: 492119387190.34467,
        3420: 486709948844.69226,
        3430: 481374521229.2724,
        3440: 476111894461.34125,
        3450: 470920881632.0503,
        3460: 465800318310.1027,
        3470: 460749062057.4196,
        3480: 455765991956.4944,
        3490: 450850008149.12195,
        3500: 446000031386.20166,
        3510: 441215002588.32056,
        3520: 436493882416.8252,
        3530: 431835650855.1166,
        3540: 427239306799.88696,
        3550: 422703867662.0448,
        3560: 418228368977.071,
        3570: 413811864024.5609,
        3580: 409453423456.71075,
        3590: 405152134935.5198,
        3600: 400907102778.47723,
        3610: 396717447612.5164,
        3620: 392582306036.025,
        3630: 388500830288.7002,
        3640: 384472187929.0492,
        3650: 380495561519.3396,
        3660: 376570148317.8077,
        3670: 372695159977.94135,
        3680: 368869822254.6568,
        3690: 365093374717.1939,
        3700: 361365070468.56067,
        3710: 357684175871.36255,
        3720: 354049970279.8525,
        3730: 350461745778.0513,
        3740: 346918806923.7793,
        3750: 343420470498.4561,
        3760: 339966065262.5251,
        3770: 336554931716.35895,
        3780: 333186421866.5137,
        3790: 329859898997.1975,
        3800: 326574737446.82556,
        3810: 323330322389.536,
        3820: 320126049621.5456,
        3830: 316961325352.2228,
        3840: 313835565999.7712,
        3850: 310748197991.3997,
        3860: 307698657567.88074,
        3870: 304686390592.38226,
        3880: 301710852363.4753,
        3890: 298771507432.21216,
        3900: 295867829423.1807,
        3910: 292999300859.4356,
        3920: 290165412991.21655,
        3930: 287365665628.3618,
        3940: 284599566976.3272,
        3950: 281866633475.72656,
        3960: 279166389645.3106,
        3970: 276498367928.30023,
        3980: 273862108541.99686,
        3990: 271257159330.59055,
        4000: 268683075621.09402,
        4010: 266139420082.32556,
        4020: 263625762586.87003,
        4030: 261141680075.95142,
        4040: 258686756427.1439,
        4050: 256260582324.86032,
        4060: 253862755133.55032,
        4070: 251492878773.54626,
        4080: 249150563599.4963,
        4090: 246835426281.32416,
        4100: 244547089687.6584,
        4110: 242285182771.67264,
        4120: 240049340459.28366,
        4130: 237839203539.6523,
        4140: 235654418557.93512,
        4150: 233494637710.23468,
        4160: 231359518740.69992,
        4170: 229248724840.72668,
        4180: 227161924550.21158,
        4190: 225098791660.81345,
        4200: 223059005121.17532,
        4210: 221042248944.06415,
        4220: 219048212115.38684,
        4230: 217076588505.03757,
        4240: 215127076779.5368,
        4250: 213199380316.424,
        4260: 211293207120.3616,
        4270: 209408269740.9155,
        4280: 207544285191.97327,
        4290: 205700974872.7637,
        4300: 203878064490.44406,
        4310: 202075283984.2189,
        4320: 200292367450.95844,
        4330: 198529053072.2818,
        4340: 196785083043.07498,
        4350: 195060203501.41135,
        4360: 193354164459.84384,
        4370: 191666719738.041,
        4380: 189997626896.73456,
        4390: 188346647172.9534,
        4400: 186713545416.51468,
        4410: 185098090027.74515,
        4420: 183500052896.4062,
        4430: 181919209341.79736,
        4440: 180355338054.01263,
        4450: 178808221036.3259,
        4460: 177277643548.6789,
        4470: 175763394052.2524,
        4480: 174265264155.09464,
        4490: 172783048558.78387,
        4500: 171316545006.10782,
        4510: 169865554229.73236,
        4520: 168429879901.8445,
        4530: 167009328584.7448,
        4540: 165603709682.37253,
        4550: 164212835392.74112,
        4560: 162836520661.2683,
        4570: 161474583134.9795,
        4580: 160126843117.56726,
        4590: 158793123525.29022,
        4600: 157473249843.69214,
        4610: 156167050085.1254,
        4620: 154874354747.06277,
        4630: 153594996771.17993,
        4640: 152328811503.19376,
        4650: 151075636653.44098,
        4660: 149835312258.182,
        4670: 148607680641.6149,
        4680: 147392586378.585,
        4690: 146189876257.97702,
        4700: 144999399246.7739,
        4710: 143821006454.77136,
        4720: 142654551099.9331,
        4730: 141499888474.37375,
        4740: 140356875910.95868,
        4750: 139225372750.50632,
        4760: 138105240309.58224,
        4770: 136996341848.87207,
        4780: 135898542542.12318,
        4790: 134811709445.64221,
        4800: 133735711468.3387,
        4810: 132670419342.30336,
        4820: 131615705593.91013,
        4830: 130571444515.43245,
        4840: 129537512137.16336,
        4850: 128513786200.0291,
        4860: 127500146128.68689,
        4870: 126496473005.09743,
        4880: 125502649542.56262,
        4890: 124518560060.22,
        4900: 123544090457.98364,
        4910: 122579128191.92497,
        4920: 121623562250.08247,
        4930: 120677283128.69449,
        4940: 119740182808.84464,
        4950: 118812154733.51408,
        4960: 117893093785.03029,
        4970: 116982896262.90765,
        4980: 116081459862.06967,
        4990: 115188683651.44702,
        5000: 114304468052.94405,
        5010: 113428714820.76639,
        5020: 112561327021.1031,
        5030: 111702209012.156,
        5040: 110851266424.51071,
        5050: 110008406141.842,
        5060: 109173536281.9472,
        5070: 108346566178.10214,
        5080: 107527406360.73296,
        5090: 106715968539.39763,
        5100: 105912165585.0727,
        5110: 105115911512.7375,
        5120: 104327121464.25215,
        5130: 103545711691.52306,
        5140: 102771599539.95009,
        5150: 102004703432.15112,
        5160: 101244942851.95879,
        5170: 100492238328.68286,
        5180: 99746511421.6358,
        5190: 99007684704.91426,
        5200: 98275681752.43373,
        5210: 97550427123.21039,
        5220: 96831846346.88611,
        5230: 96119865909.4921,
        5240: 95414413239.44753,
        5250: 94715416693.78673,
        5260: 94022805544.61386,
        5270: 93336509965.77814,
        5280: 92656461019.7678,
        5290: 91982590644.81654,
        5300: 91314831642.22116,
        5310: 90653117663.86476,
        5320: 89997383199.94186,
        5330: 89347563566.88364,
        5340: 88703594895.47685,
        5350: 88065414119.17537,
        5360: 87432958962.59982,
        5370: 86806167930.22217,
        5380: 86184980295.23163,
        5390: 85569336088.5795,
        5400: 84959176088.19858,
        5410: 84354441808.39522,
        5420: 83755075489.41048,
        5430: 83161020087.14655,
        5440: 82572219263.05756,
        5450: 81988617374.19968,
        5460: 81410159463.43915,
        5470: 80836791249.81488,
        5480: 80268459119.05313,
        5490: 79705110114.23149,
        5500: 79146691926.58928,
        5510: 78593152886.48299,
        5520: 78044441954.48228,
        5530: 77500508712.60603,
        5540: 76961303355.69478,
        5550: 76426776682.91757,
        5560: 75896880089.41133,
        5570: 75371565558.04951,
        5580: 74850785651.33876,
        5590: 74334493503.4406,
        5600: 73822642812.31688,
        5610: 73315187831.99586,
        5620: 72812083364.95775,
        5630: 72313284754.6367,
        5640: 71818747878.03879,
        5650: 71328429138.47217,
        5660: 70842285458.38896,
        5670: 70360274272.3361,
        5680: 69882353520.01385,
        5690: 69408481639.43973,
        5700: 68938617560.21645,
        5710: 68472720696.90173,
        5720: 68010750942.47879,
        5730: 67552668661.92525,
        5740: 67098434685.87926,
        5750: 66648010304.40065,
        5760: 66201357260.82667,
        5770: 65758437745.71905,
        5780: 65319214390.902534,
        5790: 64883650263.59221,
        5800: 64451708860.60851,
        5810: 64023354102.67904,
        5820: 63598550328.824486,
        5830: 63177262290.82861,
        5840: 62759455147.79003,
        5850: 62345094460.7546,
        5860: 61934146187.427605,
        5870: 61526576676.963585,
        5880: 61122352664.83326,
        5890: 60721441267.76604,
        5900: 60323809978.76653,
        5910: 59929426662.20465,
        5920: 59538259548.9773,
        5930: 59150277231.74091,
        5940: 58765448660.21358,
        5950: 58383743136.54579,
        5960: 58005130310.75833,
        5970: 57629580176.24676,
        5980: 57257063065.350746,
        5990: 56887549644.98788,
        6000: 56521010912.35056,
        6010: 56157418190.66498,
        6020: 55796743125.01118,
        6030: 55438957678.20324,
        6040: 55084034126.72911,
        6050: 54731945056.7478,
        6060: 54382663360.14508,
        6070: 54036162230.64428,
        6080: 53692415159.97393,
        6090: 53351395934.08896,
        6100: 53013078629.446335,
        6110: 52677437609.33335,
        6120: 52344447520.248146,
        6130: 52014083288.33144,
        6140: 51686320115.848595,
        6150: 51361133477.72184,
        6160: 51038499118.11059,
        6170: 50718393047.040924,
        6180: 50400791537.08144,
        6190: 50085671120.06651,
        6200: 49773008583.865036,
        6210: 49462780969.194756,
        6220: 49154965566.4805,
        6230: 48849539912.756966,
        6240: 48546481788.61403,
        6250: 48245769215.18505,
        6260: 47947380451.17647,
        6270: 47651293989.939095,
        6280: 47357488556.57937,
        6290: 47065943105.11122,
        6300: 46776636815.64672,
        6310: 46489549091.625534,
        6320: 46204659557.082565,
        6330: 45921948053.95327,
        6340: 45641394639.41566,
        6350: 45362979583.268776,
        6360: 45086683365.34728,
        6370: 44812486672.9709,
        6380: 44540370398.42897,
        6390: 44270315636.49925,
        6400: 44002303682.00012,
        6410: 43736316027.37641,
        6420: 43472334360.317566,
        6430: 43210340561.40835,
        6440: 42950316701.8113,
        6450: 42692245040.98005,
        6460: 42436108024.40383,
        6470: 42181888281.382286,
        6480: 41929568622.830025,
        6490: 41679132039.11067,
        6500: 41430561697.89993,
        6510: 41183840942.07725,
        6520: 40938953287.64561,
        6530: 40695882421.679016,
        6540: 40454612200.29741,
        6550: 40215126646.66825,
        6560: 39977409949.03512,
        6570: 39741446458.771835,
        6580: 39507220688.46271,
        6590: 39274717310.0082,
        6600: 39043921152.75535,
        6610: 38814817201.65283,
        6620: 38587390595.43071,
        6630: 38361626624.80342,
        6640: 38137510730.69704,
        6650: 37915028502.49917,
        6660: 37694165676.33221,
        6670: 37474908133.34874,
        6680: 37257241898.04953,
        6690: 37041153136.62318,
        6700: 36826628155.30753,
        6710: 36613653398.772194,
        6720: 36402215448.5222,
        6730: 36192301021.322266,
        6740: 35983896967.641464,
        6750: 35776990270.11788,
        6760: 35571568042.04306,
        6770: 35367617525.86626,
        6780: 35165126091.717545,
        6790: 34964081235.950096,
        6800: 34764470579.701126,
        6810: 34566281867.47115,
        6820: 34369502965.72152,
        6830: 34174121861.489944,
        6840: 33980126661.023388,
        6850: 33787505588.428726,
        6860: 33596246984.340107,
        6870: 33406339304.60375,
        6880: 33217771118.979008,
        6890: 33030531109.85608,
        6900: 32844608070.98982,
        6910: 32659990906.2497,
        6920: 32476668628.38527,
        6930: 32294630357.80747,
        6940: 32113865321.384937,
        6950: 31934362851.255714,
        6960: 31756112383.653675,
        6970: 31579103457.749695,
        6980: 31403325714.50738,
        6990: 31228768895.552856,
        7000: 31055422842.05897,
        7010: 30883277493.643147,
        7020: 30712322887.278984,
        7030: 30542549156.221478,
        7040: 30373946528.945415,
        7050: 30206505328.097076,
        7060: 30040215969.458633,
        7070: 29875068960.925846,
        7080: 29711054901.497585,
        7090: 29548164480.278698,
        7100: 29386388475.49436,
        7110: 29225717753.517006,
        7120: 29066143267.905006,
        7130: 28907656058.45312,
        7140: 28750247250.25473,
        7150: 28593908052.775223,
        7160: 28438629758.93695,
        7170: 28284403744.215263,
        7180: 28131221465.745422,
        7190: 27979074461.440567,
        7200: 27827954349.120094,
        7210: 27677852825.648987,
        7220: 27528761666.087208,
        7230: 27380672722.849545,
        7240: 27233577924.875492,
        7250: 27087469276.80936,
        7260: 26942338858.189964,
        7270: 26798178822.65014,
        7280: 26654981397.126106,
        7290: 26512738881.075874,
        7300: 26371443645.707447,
        7310: 26231088133.21594,
        7320: 26091664856.02987,
        7330: 25953166396.066685,
        7340: 25815585403.996696,
        7350: 25678914598.51623,
        7360: 25543146765.629112,
        7370: 25408274757.93671,
        7380: 25274291493.936584,
        7390: 25141189957.329197,
        7400: 25008963196.33294,
        7410: 24877604323.007202,
        7420: 24747106512.58348,
        7430: 24617463002.80417,
        7440: 24488667093.269485,
        7450: 24360712144.791622,
        7460: 24233591578.756794,
        7470: 24107298876.494667,
        7480: 23981827578.655006,
        7490: 23857171284.591957,
        7500: 23733323651.75505,
        7510: 23610278395.087677,
        7520: 23488029286.4324,
        7530: 23366570153.94321,
        7540: 23245894881.50448,
        7550: 23125997408.156883,
        7560: 23006871727.52974,
        7570: 22888511887.280117,
        7580: 22770911988.53817,
        7590: 22654066185.359097,
        7600: 22537968684.181282,
        7610: 22422613743.29073,
        7620: 22307995672.29159,
        7630: 22194108831.5828,
        7640: 22080947631.840702,
        7650: 21968506533.507614,
        7660: 21856780046.286186,
        7670: 21745762728.63957,
        7680: 21635449187.297287,
        7690: 21525834076.766697,
        7700: 21416912098.85003,
        7710: 21308678002.166878,
        7720: 21201126581.682213,
        7730: 21094252678.23956,
        7740: 20988051178.099644,
        7750: 20882517012.48412,
        7760: 20777645157.12443,
        7770: 20673430631.81598,
        7780: 20569868499.976936,
        7790: 20466953868.21245,
        7800: 20364681885.88328,
        7810: 20263047744.679703,
        7820: 20162046678.199917,
        7830: 20061673961.533226,
        7840: 19961924910.84793,
        7850: 19862794882.98377,
        7860: 19764279275.048866,
        7870: 19666373524.02133,
        7880: 19569073106.355015,
        7890: 19472373537.58994,
        7900: 19376270371.966846,
        7910: 19280759202.045994,
        7920: 19185835658.33045,
        7930: 19091495408.893116,
        7940: 18997734159.008266,
        7950: 18904547650.786945,
        7960: 18811931662.81636,
        7970: 18719882009.803337,
        7980: 18628394542.22162,
        7990: 18537465145.963123,
        8000: 18447089741.992844,
        8010: 18357264286.007812,
        8020: 18267984768.099472,
        8030: 18179247212.419884,
        8040: 18091047676.851665,
        8050: 18003382252.68139,
        8060: 17916247064.276493,
        8070: 17829638268.76592,
        8080: 17743552055.723915,
        8090: 17657984646.85762,
        8100: 17572932295.697628,
        8110: 17488391287.29226,
        8120: 17404357937.904903,
        8130: 17320828594.7147,
        8140: 17237799635.520485,
        8150: 17155267468.447874,
        8160: 17073228531.659462,
        8170: 16991679293.06818,
        8180: 16910616250.053774,
        8190: 16830035929.182213,
        8200: 16749934885.928112,
        8210: 16670309704.400278,
        8220: 16591156997.069937,
        8230: 16512473404.502028,
        8240: 16434255595.089334,
        8250: 16356500264.789387,
        8260: 16279204136.864246,
        8270: 16202363961.622906,
        8280: 16125976516.166597,
        8290: 16050038604.136644,
        8300: 15974547055.465088,
        8310: 15899498726.127872,
        8320: 15824890497.900648,
        8330: 15750719278.117184,
        8340: 15676981999.43031,
        8350: 15603675619.575298,
        8360: 15530797121.135885,
        8370: 15458343511.312525,
        8380: 15386311821.693295,
        8390: 15314699108.027079,
        8400: 15243502449.999077,
        8410: 15172718951.008781,
        8420: 15102345737.950182,
        8430: 15032379960.9943,
        8440: 14962818793.373896,
        8450: 14893659431.170612,
        8460: 14824899093.10404,
        8470: 14756535020.32326,
        8480: 14688564476.200317,
        8490: 14620984746.126007,
        8500: 14553793137.307585,
        8510: 14486986978.568735,
        8520: 14420563620.151533,
        8530: 14354520433.520355,
        8540: 14288854811.168041,
        8550: 14223564166.423725,
        8560: 14158645933.262972,
        8570: 14094097566.119549,
        8580: 14029916539.69941,
        8590: 13966100348.796322,
        8600: 13902646508.109571,
        8610: 13839552552.063461,
        8620: 13776816034.628536,
        8630: 13714434529.144876,
        8640: 13652405628.146952,
        8650: 13590726943.190327,
        8660: 13529396104.680225,
        8670: 13468410761.701677,
        8680: 13407768581.851439,
        8690: 13347467251.071726,
        8700: 13287504473.485435,
        8710: 13227877971.233128,
        8720: 13168585484.31169,
        8730: 13109624770.414572,
        8740: 13050993604.773561,
        8750: 12992689780.002317,
        8760: 12934711105.941305,
        8770: 12877055409.50438,
        8780: 12819720534.526873,
        8790: 12762704341.61525,
        8800: 12706004707.99815,
        8810: 12649619527.378994,
        8820: 12593546709.79013,
        8830: 12537784181.448254,
        8840: 12482329884.611446,
        8850: 12427181777.437487,
        8860: 12372337833.843683,
        8870: 12317796043.367983,
        8880: 12263554411.031563,
        8890: 12209610957.20278,
        8900: 12155963717.462286,
        8910: 12102610742.469727,
        8920: 12049550097.83155,
        8930: 11996779863.970285,
        8940: 11944298135.99491,
        8950: 11892103023.572706,
        8960: 11840192650.802204,
        8970: 11788565156.087467,
        8980: 11737218692.013527,
        8990: 11686151425.223177,
        9000: 11635361536.294767,
        9010: 11584847219.621408,
        9020: 11534606683.291145,
        9030: 11484638148.968494,
        9040: 11434939851.776972,
        9050: 11385510040.182846,
        9060: 11336346975.880033,
        9070: 11287448933.676048,
        9080: 11238814201.379063,
        9090: 11190441079.686155,
        9100: 11142327882.072502,
        9110: 11094472934.681725,
        9120: 11046874576.21729,
        9130: 10999531157.834946,
        9140: 10952441043.036144,
        9150: 10905602607.562569,
        9160: 10859014239.29161,
        9170: 10812674338.13294,
        9180: 10766581315.925888,
        9190: 10720733596.338032,
        9200: 10675129614.764597,
        9210: 10629767818.22887,
        9220: 10584646665.283548,
        9230: 10539764625.913074,
        9240: 10495120181.436811,
        9250: 10450711824.413275,
        9260: 10406538058.545101,
        9270: 10362597398.585047,
        9280: 10318888370.242897,
        9290: 10275409510.093105,
        9300: 10232159365.48347,
        9310: 10189136494.444609,
        9320: 10146339465.600206,
        9330: 10103766858.078302,
        9340: 10061417261.42322,
        9350: 10019289275.50843,
        9360: 9977381510.450212,
        9370: 9935692586.522121,
        9380: 9894221134.07022,
        9390: 9852965793.429253,
        9400: 9811925214.839365,
        9410: 9771098058.363811,
        9420: 9730482993.80733,
        9430: 9690078700.635307,
        9440: 9649883867.893711,
        9450: 9609897194.129696,
        9460: 9570117387.313078,
        9470: 9530543164.758385,
        9480: 9491173253.047764,
        9490: 9452006387.954533,
        9500: 9413041314.367455,
        9510: 9374276786.215712,
        9520: 9335711566.394573,
        9530: 9297344426.691776,
        9540: 9259174147.714525,
        9550: 9221199518.817226,
        9560: 9183419338.029861,
        9570: 9145832411.98704,
        9580: 9108437555.857624,
        9590: 9071233593.27513,
        9600: 9034219356.268673,
        9610: 8997393685.194557,
        9620: 8960755428.668531,
        9630: 8924303443.498617,
        9640: 8888036594.618572,
        9650: 8851953755.021992,
        9660: 8816053805.69696,
        9670: 8780335635.561289,
        9680: 8744798141.398413,
        9690: 8709440227.793814,
        9700: 8674260807.072035,
        9710: 8639258799.234299,
        9720: 8604433131.896572,
        9730: 8569782740.22838,
        9740: 8535306566.892041,
        9750: 8501003561.982425,
        9760: 8466872682.967408,
        9770: 8432912894.628675,
        9780: 8399123169.003227,
        9790: 8365502485.3253,
        9800: 8332049829.96881,
        9810: 8298764196.39044,
        9820: 8265644585.073038,
        9830: 8232690003.469712,
        9840: 8199899465.948283,
        9850: 8167271993.736356,
        9860: 8134806614.866777,
        9870: 8102502364.123612,
        9880: 8070358282.988652,
        9890: 8038373419.588365,
        9900: 8006546828.641256,
        9910: 7974877571.4058,
        9920: 7943364715.628804,
        9930: 7912007335.494175,
        9940: 7880804511.572181,
        9950: 7849755330.769227,
        9960: 7818858886.277912,
        9970: 7788114277.527709,
        9980: 7757520610.135957,
        9990: 7727076995.859316},
    100000: {
        10: 6.71890609624107e+17,
        20: 2.797622826557186e+19,
        30: 4.083646021499168e+19,
        40: 3.2774679698581156e+19,
        50: 2.2724984011014296e+19,
        60: 1.5315142769174602e+19,
        70: 1.0405866787743447e+19,
        80: 7.211023365478845e+18,
        90: 5.111049355718106e+18,
        100: 3.703912490782156e+18,
        110: 2.7403007528056755e+18,
        120: 2.0660120250660992e+18,
        130: 1.5844465013204826e+18,
        140: 1.2339636847390702e+18,
        150: 9.744306319793395e+17,
        160: 7.79186927040088e+17,
        170: 6.3017677933972e+17,
        180: 5.1494744308054336e+17,
        190: 4.2476346708902746e+17,
        200: 3.533996570653426e+17,
        210: 2.963543967852565e+17,
        220: 2.503280440561384e+17,
        230: 2.1287142270301098e+17,
        240: 1.8214529964318922e+17,
        250: 1.567534328230856e+17,
        260: 1.3562512243404973e+17,
        270: 1.1793154308569685e+17,
        280: 1.03025435203875e+17,
        290: 9.039715148186467e+16,
        300: 7.964228931364902e+16,
        310: 7.0437621797485176e+16,
        320: 6.2523034758476584e+16,
        330: 5.5687853389797976e+16,
        340: 4.976040696315497e+16,
        350: 4.460000313862017e+16,
        360: 4.009071027784771e+16,
        370: 3.613650704685607e+16,
        380: 3.2657473744682548e+16,
        390: 2.9586782942318064e+16,
        400: 2.6868307562109412e+16,
        410: 2.4454708968765844e+16,
        420: 2.2305900512117548e+16,
        430: 2.038780644904441e+16,
        440: 1.8671354541651464e+16,
        450: 1.7131654500610778e+16,
        460: 1.5747324984369202e+16,
        470: 1.4499939924677388e+16,
        480: 1.3373571146833868e+16,
        490: 1.2354409045798368e+16,
        500: 1.14304468052944e+16,
        510: 1.0591216558507274e+16,
        520: 9827568175243378.0,
        530: 9131483164222116.0,
        540: 8495917608819857.0,
        550: 7914669192658926.0,
        560: 7382264281231687.0,
        570: 6893861756021644.0,
        580: 6445170886060851.0,
        590: 6032380997876651.0,
        600: 5652101091235055.0,
        610: 5301307862944630.0,
        620: 4977300858386507.0,
        630: 4677663681564674.0,
        640: 4400230368200013.5,
        650: 4143056169789993.5,
        660: 3904392115275535.0,
        670: 3682662815530752.5,
        680: 3476447057970112.0,
        690: 3284460807098982.5,
        700: 3105542284205897.5,
        710: 2938638847549436.5,
        720: 2782795434912008.5,
        730: 2637144364570745.5,
        740: 2500896319633294.5,
        750: 2373332365175505.0,
        760: 2253796868418128.0,
        770: 2141691209885001.5,
        780: 2036468188588327.5,
        790: 1937627037196684.5,
        800: 1844708974199285.0,
        810: 1757293229569762.2,
        820: 1674993488592811.5,
        830: 1597454705546508.2,
        840: 1524350244999908.8,
        850: 1455379313730758.2,
        860: 1390264650810957.2,
        870: 1328750447348543.2,
        880: 1270600470799814.8,
        890: 1215596371746229.0,
        900: 1163536153629476.5,
        910: 1114232788207250.2,
        920: 1067512961476458.9,
        930: 1023215936548347.0,
        940: 981192521483935.9,
        950: 941304131436745.6,
        960: 903421935626867.1,
        970: 867426080707203.5,
        980: 833204982996880.9,
        990: 800654682864125.5,
        1000: 769678255254570.9,
        1010: 740185270992819.2,
        1020: 712091304044972.0,
        1030: 685317480426837.8,
        1040: 659790064883927.9,
        1050: 635440081862038.8,
        1060: 612202967636844.8,
        1070: 590018250782637.4,
        1080: 568829258438471.1,
        1090: 548582846078490.7,
        1100: 529229148715441.8,
        1110: 510721351665287.3,
        1120: 493015479179173.7,
        1130: 476070199408861.94,
        1140: 459846644315378.1,
        1150: 444308243259650.4,
        1160: 429420569130013.06,
        1170: 415151195966009.8,
        1180: 401469567132094.94,
        1190: 388346873179877.9,
        1200: 375755938614218.25,
        1210: 363671116847872.06,
        1220: 352068192692043.3,
        1230: 340924291786989.0,
        1240: 330217796428213.94,
        1250: 319928267290413.4,
        1260: 310036370593628.94,
        1270: 300523810294453.2,
        1280: 291373264920060.0,
        1290: 282568328694544.4,
        1300: 274093456635951.84,
        1310: 265933913328697.12,
        1320: 258075725100025.44,
        1330: 250505635351068.78,
        1340: 243211062812984.28,
        1350: 236180062516913.9,
        1360: 229401289283154.2,
        1370: 222863963550173.22,
        1380: 216557839378051.38,
        1390: 210473174473705.56,
        1400: 204600702096948.84,
        1410: 198931604717174.3,
        1420: 193457489300308.62,
        1430: 188170364114697.2,
        1440: 183062616952917.62,
        1450: 178126994674123.97,
        1460: 173356583978561.66,
        1470: 168744793332348.12,
        1480: 164285335966562.47,
        1490: 159972213880169.88,
        1500: 155799702781354.78,
        1510: 151762337906502.34,
        1520: 147854900660362.1,
        1530: 144072406024899.22,
        1540: 140410090688010.02,
        1550: 136863401846659.27,
        1560: 133427986642146.11,
        1570: 130099682188085.9,
        1580: 126874506154399.22,
        1590: 123748647873075.97,
        1600: 120718459933782.02,
        1610: 117780450239525.53,
        1620: 114931274494564.6,
        1630: 112167729098590.0,
        1640: 109486744422913.47,
        1650: 106885378445982.08,
        1660: 104360810727006.11,
        1670: 101910336697867.83,
        1680: 99531362254735.45,
        1690: 97221398632005.8,
        1700: 94978057542294.84,
        1710: 92799046567218.81,
        1720: 90682164784674.86,
        1730: 88625298619210.62,
        1740: 86626417902912.67,
        1750: 84683572135008.53,
        1760: 82794886929115.22,
        1770: 80958560637723.77,
        1780: 79172861144158.47,
        1790: 77436122812823.78,
        1800: 75746743589112.81,
        1810: 74103182240857.9,
        1820: 72503955733693.56,
        1830: 70947636733144.14,
        1840: 69432851226677.516,
        1850: 67958276259354.38,
        1860: 66522637777079.52,
        1870: 65124708571798.99,
        1880: 63763306323322.69,
        1890: 62437291732745.805,
        1900: 61145566742734.914,
        1910: 59887072840211.58,
        1920: 58660789437213.766,
        1930: 57465732325957.93,
        1940: 56300952204339.914,
        1950: 55165533268327.42,
        1960: 54058591867886.57,
        1970: 52979275223273.52,
        1980: 51926760198695.38,
        1990: 50900252130501.71,
        2000: 49898983707231.64,
        2010: 48922213898973.98,
        2020: 47969226933642.46,
        2030: 47039331317893.484,
        2040: 46131858900531.85,
        2050: 45246163976367.16,
        2060: 44381622428588.45,
        2070: 43537630907826.914,
        2080: 42713606046171.055,
        2090: 41908983704487.055,
        2100: 41123218251487.2,
        2110: 40355781873061.016,
        2120: 39606163910468.414,
        2130: 38873870226058.0,
        2140: 38158422595245.76,
        2150: 37459358123551.375,
        2160: 36776228687550.4,
        2170: 36108600398656.42,
        2180: 35456053088702.92,
        2190: 34818179816343.46,
        2200: 34194586393339.23,
        2210: 33584890929847.594,
        2220: 32988723397867.97,
        2230: 32405725212043.5,
        2240: 31835548827055.09,
        2250: 31277857350881.332,
        2260: 30732324173233.14,
        2270: 30198632608502.992,
        2280: 29676475552603.67,
        2290: 29165555153096.8,
        2300: 28665582492042.46,
        2310: 28176277281028.16,
        2320: 27697367567857.785,
        2330: 27228589454408.883,
        2340: 26769686825187.336,
        2350: 26320411086130.0,
        2360: 25880520913228.37,
        2370: 25449782010563.703,
        2380: 25027966877364.523,
        2390: 24614854583713.492,
        2400: 24210230554549.098,
        2410: 23813886361621.605,
        2420: 23425619523080.68,
        2430: 23045233310383.168,
        2440: 22672536562226.797,
        2450: 22307343505225.53,
        2460: 21949473581057.37,
        2470: 21598751279824.895,
        2480: 21255005979383.035,
        2490: 20918071790395.664,
        2500: 20587787406896.773,
        2510: 20263995962138.527,
        2520: 19946544889519.617,
        2530: 19635285788396.28,
        2540: 19330074294585.21,
        2550: 19030769955377.64,
        2560: 18737236108890.098,
        2570: 18449339767586.34,
        2580: 18166951505808.754,
        2590: 17889945351169.266,
        2600: 17618198679650.586,
        2610: 17351592114278.848,
        2620: 17090009427233.361,
        2630: 16833337445262.887,
        2640: 16581465958286.596,
        2650: 16334287631059.766,
        2660: 16091697917790.965,
        2670: 15853594979601.48,
        2680: 15619879604722.28,
        2690: 15390455131327.828,
        2700: 15165227372910.803,
        2710: 14944104546104.139,
        2720: 14726997200862.775,
        2730: 14513818152918.56,
        2740: 14304482418427.209,
        2750: 14098907150728.072,
        2760: 13897011579141.205,
        2770: 13698716949729.547,
        2780: 13503946467955.09,
        2790: 13312625243163.746,
        2800: 13124680234832.918,
        2810: 12940040200520.383,
        2820: 12758635645454.87,
        2830: 12580398773710.945,
        2840: 12405263440913.037,
        2850: 12233165108415.678,
        2860: 12064040798908.715,
        2870: 11897829053398.727,
        2880: 11734469889519.129,
        2890: 11573904761123.846,
        2900: 11416076519120.523,
        2910: 11260929373500.97,
        2920: 11108408856529.13,
        2930: 10958461787046.506,
        2940: 10811036235858.014,
        2950: 10666081492162.441,
        2960: 10523548030991.246,
        2970: 10383387481623.852,
        2980: 10245552596945.959,
        2990: 10109997223719.629,
        3000: 9976676273735.705,
        3010: 9845545695819.326,
        3020: 9716562448660.236,
        3030: 9589684474441.564,
        3040: 9464870673240.574,
        3050: 9342080878176.78,
        3060: 9221275831282.85,
        3070: 9102417160075.428,
        3080: 8985467354803.008,
        3090: 8870389746349.592,
        3100: 8757148484772.743,
        3110: 8645708518456.251,
        3120: 8536035573857.646,
        3130: 8428096135832.14,
        3140: 8321857428514.057,
        3150: 8217287396739.326,
        3160: 8114354687991.163,
        3170: 8013028634853.089,
        3180: 7913279237953.237,
        3190: 7815077149385.13,
        3200: 7718393656589.585,
        3210: 7623200666684.126,
        3220: 7529470691225.601,
        3230: 7437176831393.412,
        3240: 7346292763579.575,
        3250: 7256792725374.212,
        3260: 7168651501933.775,
        3270: 7081844412720.704,
        3280: 6996347298603.18,
        3290: 6912136509304.378,
        3300: 6829188891190.4375,
        3310: 6747481775387.058,
        3320: 6666992966215.409,
        3330: 6587700729936.908,
        3340: 6509583783798.747,
        3350: 6432621285370.585,
        3360: 6356792822163.951,
        3370: 6282078401526.433,
        3380: 6208458440802.092,
        3390: 6135913757750.542,
        3400: 6064425561217.242,
        3410: 5993975442047.647,
        3420: 5924545364238.199,
        3430: 5856117656316.932,
        3440: 5788675002947.848,
        3450: 5722200436751.911,
        3460: 5656677330338.943,
        3470: 5592089388543.884,
        3480: 5528420640862.425,
        3490: 5465655434079.36,
        3500: 5403778425085.023,
        3510: 5342774573874.018,
        3520: 5282629136721.349,
        3530: 5223327659530.821,
        3540: 5164855971351.129,
        3550: 5107200178054.552,
        3560: 5050346656174.055,
        3570: 4994282046894.508,
        3580: 4938993250193.272,
        3590: 4884467419126.502,
        3600: 4830691954256.869,
        3610: 4777654498219.097,
        3620: 4725342930419.064,
        3630: 4673745361863.556,
        3640: 4622850130116.29,
        3650: 4572645794377.416,
        3660: 4523121130682.81,
        3670: 4474265127220.076,
        3680: 4426066979758.171,
        3690: 4378516087187.197,
        3700: 4331602047166.0625,
        3710: 4285314651874.602,
        3720: 4239643883867.4946,
        3730: 4194579912027.525,
        3740: 4150113087615.1924,
        3750: 4106233940412.318,
        3760: 4062933174957.2803,
        3770: 4020201666869.1294,
        3780: 3978030459258.5747,
        3790: 3936410759223.488,
        3800: 3895333934426.615,
        3810: 3854791509753.3784,
        3820: 3814775164047.902,
        3830: 3775276726924.8184,
        3840: 3736288175655.5376,
        3850: 3697801632126.229,
        3860: 3659809359866.3984,
        3870: 3622303761146.024,
        3880: 3585277374139.207,
        3890: 3548722870153.0674,
        3900: 3512633050920.0366,
        3910: 3477000845951.869,
        3920: 3441819309953.8896,
        3930: 3407081620298.1562,
        3940: 3372781074553.4653,
        3950: 3338911088071.614,
        3960: 3305465191627.6924,
        3970: 3272437029113.641,
        3980: 3239820355283.478,
        3990: 3207609033548.876,
        4000: 3175797033824.0825,
        4010: 3144378430418.6167,
        4020: 3113347399977.013,
        4030: 3082698219463.9043,
        4040: 3052425264193.8467,
        4050: 3022523005904.4507,
        4060: 2992986010871.9854,
        4070: 2963808938068.1055,
        4080: 2934986537357.1665,
        4090: 2906513647732.756,
        4100: 2878385195592.525,
        4110: 2850596193050.724,
        4120: 2823141736287.0996,
        4130: 2796017003931.6074,
        4140: 2769217255483.8506,
        4150: 2742737829766.679,
        4160: 2716574143412.7847,
        4170: 2690721689383.833,
        4180: 2665176035521.2383,
        4190: 2639932823127.7026,
        4200: 2614987765579.017,
        4210: 2590336646965.295,
        4220: 2565975320760.8755,
        4230: 2541899708522.4116,
        4240: 2518105798614.3164,
        4250: 2494589644960.9663,
        4260: 2471347365825.1445,
        4270: 2448375142611.9565,
        4280: 2425669218697.741,
        4290: 2403225898283.3237,
        4300: 2381041545271.119,
        4310: 2359112582165.5244,
        4320: 2337435488995.964,
        4330: 2316006802262.227,
        4340: 2294823113901.4663,
        4350: 2273881070276.4233,
        4360: 2253177371184.39,
        4370: 2232708768886.348,
        4380: 2212472067156.0845,
        4390: 2192464120348.4316,
        4400: 2172681832486.568,
        4410: 2153122156367.7686,
        4420: 2133782092687.2695,
        4430: 2114658689179.6765,
        4440: 2095749039777.7551,
        4450: 2077050283788.138,
        4460: 2058559605083.3933,
        4470: 2040274231310.416,
        4480: 2022191433114.404,
        4490: 2004308523378.5461,
        4500: 1986622856478.5042,
        4510: 1969131827551.8745,
        4520: 1951832871782.029,
        4530: 1934723463696.0164,
        4540: 1917801116476.4087,
        4550: 1901063381286.5327,
        4560: 1884507846608.9795,
        4570: 1868132137597.0935,
        4580: 1851933915439.008,
        4590: 1835910876734.126,
        4600: 1820060752881.735,
        4610: 1804381309481.4226,
        4620: 1788870345745.0815,
        4630: 1773525693920.3213,
        4640: 1758345218724.9421,
        4650: 1743326816792.2583,
        4660: 1728468416127.122,
        4670: 1713767975572.2915,
        4680: 1699223484285.0105,
        4690: 1684832961223.5886,
        4700: 1670594454643.7065,
        4710: 1656506041604.286,
        4720: 1642565827482.7593,
        4730: 1628771945499.3826,
        4740: 1615122556250.6946,
        4750: 1601615847251.562,
        4760: 1588250032485.9958,
        4770: 1575023351966.2256,
        4780: 1561934071300.1697,
        4790: 1548980481266.8562,
        4800: 1536160897399.8162,
        4810: 1523473659578.1877,
        4820: 1510917131625.423,
        4830: 1498489700915.4001,
        4840: 1486189777985.852,
        4850: 1474015796158.8215,
        4860: 1461966211168.2622,
        4870: 1450039500794.284,
        4880: 1438234164504.2488,
        4890: 1426548723100.3413,
        4900: 1414981718373.626,
        4910: 1403531712764.352,
        4920: 1392197289028.423,
        4930: 1380977049910.033,
        4940: 1369869617820.0374,
        4950: 1358873634520.2827,
        4960: 1347987760813.5928,
        4970: 1337210676239.2769,
        4980: 1326541078774.1477,
        4990: 1315977684538.9148,
        5000: 1305519227509.7815,
        5010: 1295164459235.205,
        5020: 1284912148557.7336,
        5030: 1274761081340.7432,
        5040: 1264710060200.0144,
        5050: 1254757904240.1487,
        5060: 1244903448795.5132,
        5070: 1235145545175.88,
        5080: 1225483060416.4197,
        5090: 1215914877032.2175,
        5100: 1206439892776.9695,
        5110: 1197057020405.999,
        5120: 1187765187443.352,
        5130: 1178563335952.9712,
        5140: 1169450422313.8413,
        5150: 1160425416999.0703,
        5160: 1151487304358.7625,
        5170: 1142635082406.6785,
        5180: 1133867762610.526,
        5190: 1125184369685.9434,
        5200: 1116583941393.9238,
        5210: 1108065528341.856,
        5220: 1099628193787.8003,
        5230: 1091271013448.2942,
        5240: 1082993075309.2667,
        5250: 1074793479440.3116,
        5260: 1066671337812.0234,
        5270: 1058625774116.5092,
        5280: 1050655923590.8562,
        5290: 1042760932843.6814,
        5300: 1034939959684.5034,
        5310: 1027192172956.0835,
        5320: 1019516752369.5392,
        5330: 1011912888342.207,
        5340: 1004379781838.2548,
        5350: 996916644211.955,
        5360: 989522697053.5531,
        5370: 982197172037.7372,
        5380: 974939310774.5638,
        5390: 967748364662.9548,
        5400: 960623594746.5071,
        5410: 953564271571.7982,
        5420: 946569675048.9287,
        5430: 939639094314.4648,
        5440: 932771827596.5293,
        5450: 925967182082.1647,
        5460: 919224473786.8965,
        5470: 912543027426.3756,
        5480: 905922176290.1104,
        5490: 899361262117.3301,
        5500: 892859634974.7788,
        5510: 886416653136.5468,
        5520: 880031682965.8585,
        5530: 873704098798.7098,
        5540: 867433282829.4485,
        5550: 861218624998.1802,
        5560: 855059522879.9738,
        5570: 848955381575.8392,
        5580: 842905613605.5132,
        5590: 836909638801.8594,
        5600: 830966884207.0535,
        5610: 825076783970.3597,
        5620: 819238779247.5779,
        5630: 813452318102.0471,
        5640: 807716855407.2653,
        5650: 802031852750.9854,
        5660: 796396778340.922,
        5670: 790811106911.8524,
        5680: 785274319634.2522,
        5690: 779785904024.2954,
        5700: 774345353855.3517,
        5710: 768952169070.8043,
        5720: 763605855698.2571,
        5730: 758305925765.0981,
        5740: 753051897215.3187,
        5750: 747843293827.7258,
        5760: 742679645135.3091,
        5770: 737560486345.9745,
        5780: 732485358264.398,
        5790: 727453807215.1644,
        5800: 722465384967.0712,
        5810: 717519648658.5706,
        5820: 712616160724.3851,
        5830: 707754488823.252,
        5840: 702934205766.7695,
        5850: 698154889449.3145,
        5860: 693416122779.0521,
        5870: 688717493609.9824,
        5880: 684058594675.0568,
        5890: 679439023520.2299,
        5900: 674858382439.6288,
        5910: 670316278411.5775,
        5920: 665812323035.7026,
        5930: 661346132470.8939,
        5940: 656917327374.2812,
        5950: 652525532841.0446,
        5960: 648170378345.2207,
        5970: 643851497681.3223,
        5980: 639568528906.8608,
        5990: 635321114285.7306,
        6000: 631108900232.4326,
        6010: 626931537257.1333,
        6020: 622788679911.5234,
        6030: 618679986735.5045,
        6040: 614605120204.6539,
        6050: 610563746678.4557,
        6060: 606555536349.3281,
        6070: 602580163192.3584,
        6080: 598637304915.823,
        6090: 594726642912.3918,
        6100: 590847862211.0939,
        6110: 587000651429.9434,
        6120: 583184702729.298,
        6130: 579399711765.8593,
        6140: 575645377647.3625,
        6150: 571921402887.9585,
        6160: 568227493364.1606,
        6170: 564563358271.5327,
        6180: 560928710081.9141,
        6190: 557323264501.3353,
        6200: 553746740428.4817,
        6210: 550198859913.8053,
        6220: 546679348119.18427,
        6230: 543187933278.19653,
        6240: 539724346656.9305,
        6250: 536288322515.40356,
        6260: 532879598069.4681,
        6270: 529497913453.31604,
        6280: 526143011682.5014,
        6290: 522814638617.47345,
        6300: 519512542927.6403,
        6310: 516236476055.96906,
        6320: 512986192184.0247,
        6330: 509761448197.5526,
        6340: 506562003652.5615,
        6350: 503387620741.80493,
        6360: 500238064261.83685,
        6370: 497113101580.45337,
        6380: 494012502604.618,
        6390: 490936039748.8657,
        6400: 487883487904.07117,
        6410: 484854624406.7478,
        6420: 481849229008.7025,
        6430: 478867083847.15753,
        6440: 475907973415.2536,
        6450: 472971684532.9882,
        6460: 470058006318.55225,
        6470: 467166730160.0346,
        6480: 464297649687.5811,
        6490: 461450560745.86926,
        6500: 458625261366.98956,
        6510: 455821551743.7201,
        6520: 453039234203.138,
        6530: 450278113180.60095,
        6540: 447537995194.09985,
        6550: 444818688818.93726,
        6560: 442120004662.7777,
        6570: 439441755341.0156,
        6580: 436783755452.486,
        6590: 434145821555.52203,
        6600: 431527772144.301,
        6610: 428929427625.55145,
        6620: 426350610295.5433,
        6630: 423791144317.4105,
        6640: 421250855698.76636,
        6650: 418729572269.62103,
        6660: 416227123660.6094,
        6670: 413743341281.4925,
        6680: 411278058299.95447,
        6690: 408831109620.7031,
        6700: 406402331864.7988,
        6710: 403991563349.3309,
        6720: 401598644067.29614,
        6730: 399223415667.7825,
        6740: 396865721436.4204,
        6750: 394525406276.0693,
        6760: 392202316687.7782,
        6770: 389896300751.98474,
        6780: 387607208109.98065,
        6790: 385334889945.6019,
        6800: 383079198967.1831,
        6810: 380839989389.715,
        6820: 378617116917.2766,
        6830: 376410438725.6653,
        6840: 374219813445.2664,
        6850: 372045101144.1651,
        6860: 369886163311.4356,
        6870: 367742862840.68774,
        6880: 365615064013.811,
        6890: 363502632484.943,
        6900: 361405435264.6154,
        6910: 359323340704.1387,
        6920: 357256218480.17053,
        6930: 355203939579.4969,
        6940: 353166376283.99414,
        6950: 351143402155.7893,
        6960: 349134892022.6241,
        6970: 347140721963.3948,
        6980: 345160769293.8799,
        6990: 343194912552.6347,
        7000: 341243031487.1208,
        7010: 339305007039.9289,
        7020: 337380721335.26587,
        7030: 335470057665.5314,
        7040: 333572900478.1436,
        7050: 331689135362.46295,
        7060: 329818649036.9293,
        7070: 327961329336.3518,
        7080: 326117065199.32367,
        7090: 324285746655.8646,
        7100: 322467264815.1513,
        7110: 320661511853.4449,
        7120: 318868381002.15063,
        7130: 317087766536.0439,
        7140: 315319563761.61725,
        7150: 313563669005.6041,
        7160: 311819979603.6328,
        7170: 310088393889.0082,
        7180: 308368811181.655,
        7190: 306661131777.2007,
        7200: 304965256936.1651,
        7210: 303281088873.3107,
        7220: 301608530747.12604,
        7230: 299947486649.41724,
        7240: 298297861595.0548,
        7250: 296659561511.82666,
        7260: 295032493230.42664,
        7270: 293416564474.564,
        7280: 291811683851.192,
        7290: 290217760840.8693,
        7300: 288634705788.2106,
        7310: 287062429892.4927,
        7320: 285500845198.3438,
        7330: 283949864586.5712,
        7340: 282409401765.0661,
        7350: 280879371259.8707,
        7360: 279359688406.3015,
        7370: 277850269340.21655,
        7380: 276351030989.37213,
        7390: 274861891064.8902,
        7400: 273382768052.82962,
        7410: 271913581205.8511,
        7420: 270454250534.9884,
        7430: 269004696801.53046,
        7440: 267564841508.9743,
        7450: 266134606895.10367,
        7460: 264713915924.14493,
        7470: 263302692279.0116,
        7480: 261900860353.67242,
        7490: 260508345245.57382,
        7500: 259125072748.17145,
        7510: 257750969343.55508,
        7520: 256385962195.1474,
        7530: 255029979140.50665,
        7540: 253682948684.19623,
        7550: 252344799990.7478,
        7560: 251015462877.71957,
        7570: 249694867808.80902,
        7580: 248382945887.0739,
        7590: 247079628848.2287,
        7600: 245784849053.98935,
        7610: 244498539485.55206,
        7620: 243220633737.0956,
        7630: 241951066009.39136,
        7640: 240689771103.47284,
        7650: 239436684414.39792,
        7660: 238191741925.05316,
        7670: 236954880200.07034,
        7680: 235726036379.77396,
        7690: 234505148174.23032,
        7700: 233292153857.34448,
        7710: 232086992261.04077,
        7720: 230889602769.50864,
        7730: 229699925313.49994,
        7740: 228517900364.71463,
        7750: 227343468930.23093,
        7760: 226176572547.02142,
        7770: 225017153276.51,
        7780: 223865153699.205,
        7790: 222720516909.39767,
        7800: 221583186509.90405,
        7810: 220453106606.88052,
        7820: 219330221804.7083,
        7830: 218214477200.9046,
        7840: 217105818381.13138,
        7850: 216004191414.2245,
        7860: 214909542847.31235,
        7870: 213821819700.96033,
        7880: 212740969464.39493,
        7890: 211666940090.76736,
        7900: 210599679992.4742,
        7910: 209539138036.53156,
        7920: 208485263540.0051,
        7930: 207438006265.49194,
        7940: 206397316416.64197,
        7950: 205363144633.74268,
        7960: 204335441989.34433,
        7970: 203314159983.95496,
        7980: 202299250541.73953,
        7990: 201290666006.33,
        8000: 200288359136.61334,
        8010: 199292283102.63022,
        8020: 198302391481.47336,
        8030: 197318638253.25455,
        8040: 196340977797.11057,
        8050: 195369364887.25497,
        8060: 194403754689.069,
        8070: 193444102755.2442,
        8080: 192490365021.96295,
        8090: 191542497805.11542,
        8100: 190600457796.5795,
        8110: 189664202060.50287,
        8120: 188733688029.6804,
        8130: 187808873501.91092,
        8140: 186889716636.44672,
        8150: 185976175950.45346,
        8160: 185068210315.51062,
        8170: 184165778954.16174,
        8180: 183268841436.48676,
        8190: 182377357676.7341,
        8200: 181491287929.96954,
        8210: 180610592788.7599,
        8220: 179735233179.92136,
        8230: 178865170361.26593,
        8240: 178000365918.406,
        8250: 177140781761.59402,
        8260: 176286380122.58374,
        8270: 175437123551.53436,
        8280: 174592974913.94925,
        8290: 173753897387.64096,
        8300: 172919854459.73853,
        8310: 172090809923.71442,
        8320: 171266727876.45435,
        8330: 170447572715.35757,
        8340: 169633309135.46158,
        8350: 168823902126.60062,
        8360: 168019316970.60107,
        8370: 167219519238.49762,
        8380: 166424474787.78842,
        8390: 165634149759.70255,
        8400: 164848510576.52142,
        8410: 164067523938.91068,
        8420: 163291156823.28455,
        8430: 162519376479.20108,
        8440: 161752150426.78098,
        8450: 160989446454.15643,
        8460: 160231232614.95108,
        8470: 159477477225.7773,
        8480: 158728148863.76593,
        8490: 157983216364.1204,
        8500: 157242648817.70145,
        8510: 156506415568.6289,
        8520: 155774486211.91153,
        8530: 155046830591.10788,
        8540: 154323418796.00436,
        8550: 153604221160.31897,
        8560: 152889208259.4327,
        8570: 152178350908.144,
        8580: 151471620158.4394,
        8590: 150768987297.29776,
        8600: 150070423844.51126,
        8610: 149375901550.5281,
        8620: 148685392394.325,
        8630: 147998868581.29306,
        8640: 147316302541.15298,
        8650: 146637666925.88092,
        8660: 145962934607.67148,
        8670: 145292078676.91214,
        8680: 144625072440.1697,
        8690: 143961889418.22614,
        8700: 143302503344.097,
        8710: 142646888161.10425,
        8720: 141995018020.94568,
        8730: 141346867281.79776,
        8740: 140702410506.42462,
        8750: 140061622460.32663,
        8760: 139424478109.8859,
        8770: 138790952620.54755,
        8780: 138161021355.00537,
        8790: 137534659871.4204,
        8800: 136911843921.64748,
        8810: 136292549449.48303,
        8820: 135676752588.93163,
        8830: 135064429662.48808,
        8840: 134455557179.43721,
        8850: 133850111834.17313,
        8860: 133248070504.53537,
        8870: 132649410250.15875,
        8880: 132054108310.83987,
        8890: 131462142104.92636,
        8900: 130873489227.71214,
        8910: 130288127449.86124,
        8920: 129706034715.83461,
        8930: 129127189142.34335,
        8940: 128551569016.81108,
        8950: 127979152795.85304,
        8960: 127409919103.7749,
        8970: 126843846731.07376,
        8980: 126280914632.97566,
        8990: 125721101927.96347,
        9000: 125164387896.33864,
        9010: 124610751978.78694,
        9020: 124060173774.95737,
        9030: 123512633042.06848,
        9040: 122968109693.51207,
        9050: 122426583797.4757,
        9060: 121888035575.58597,
        9070: 121352445401.5622,
        9080: 120819793799.87114,
        9090: 120290061444.41579,
        9100: 119763229157.21886,
        9110: 119239277907.13426,
        9120: 118718188808.55466,
        9130: 118199943120.14711,
        9140: 117684522243.59444,
        9150: 117171907722.34898,
        9160: 116662081240.39424,
        9170: 116155024621.03296,
        9180: 115650719825.66463,
        9190: 115149148952.59961,
        9200: 114650294235.86678,
        9210: 114154138044.04117,
        9220: 113660662879.07535,
        9230: 113169851375.15779,
        9240: 112681686297.5593,
        9250: 112196150541.51665,
        9260: 111713227131.10152,
        9270: 111232899218.12318,
        9280: 110755150081.02371,
        9290: 110279963123.79134,
        9300: 109807321874.89426,
        9310: 109337209986.20094,
        9320: 108869611231.93327,
        9330: 108404509507.62003,
        9340: 107941888829.0611,
        9350: 107481733331.29971,
        9360: 107024027267.608,
        9370: 106568755008.48541,
        9380: 106115901040.65602,
        9390: 105665449966.08484,
        9400: 105217386501.00201,
        9410: 104771695474.93022,
        9420: 104328361829.73412,
        9430: 103887370618.65692,
        9440: 103448707005.39705,
        9450: 103012356263.16054,
        9460: 102578303773.75063,
        9470: 102146535026.64262,
        9480: 101717035618.08817,
        9490: 101289791250.21182,
        9500: 100864787730.12686,
        9510: 100442010969.04875,
        9520: 100021446981.43231,
        9530: 99603081884.10187,
        9540: 99186901895.39465,
        9550: 98772893334.32066,
        9560: 98361042619.71613,
        9570: 97951336269.4137,
        9580: 97543760899.41917,
        9590: 97138303223.09851,
        9600: 96734950050.36034,
        9610: 96333688286.86812,
        9620: 95934504933.23207,
        9630: 95537387084.23524,
        9640: 95142321928.05011,
        9650: 94749296745.461,
        9660: 94358298909.11217,
        9670: 93969315882.74048,
        9680: 93582335220.42659,
        9690: 93197344565.85986,
        9700: 92814331651.58675,
        9710: 92433284298.29672,
        9720: 92054190414.08836,
        9730: 91677037993.75917,
        9740: 91301815118.09486,
        9750: 90928509953.16125,
        9760: 90557110749.61377,
        9770: 90187605842.00795,
        9780: 89819983648.10767,
        9790: 89454232668.2142,
        9800: 89090341484.49277,
        9810: 88728298760.3068,
        9820: 88368093239.55977,
        9830: 88009713746.0432,
        9840: 87653149182.78104,
        9850: 87298388531.40172,
        9860: 86945420851.49179,
        9870: 86594235279.97044,
        9880: 86244821030.46413,
        9890: 85897167392.691,
        9900: 85551263731.84332,
        9910: 85207099487.98438,
        9920: 84864664175.4432,
        9930: 84523947382.22423,
        9940: 84184938769.4087,
        9950: 83847628070.57414,
        9960: 83512005091.21025,
        9970: 83178059708.1488,
        9980: 82845781868.98933,
        9990: 82515161591.5363}}

BLACKBODY_SPD_DATA = np.array([
    6654278270641.8164,
    6709605279251.8242,
    6764825121520.0215,
    6819933078643.3184,
    6874924489829.6582,
    6929794752622.3604,
    6984539323198.4238,
    7039153716641.1973,
    7093633507187.9199,
    7147974328452.79,
    7202171873625.8623,
    7256221895648.418,
    7310120207365.2773,
    7363862681654.5146,
    7417445251535.1982,
    7470863910253.4277,
    7524114711347.416,
    7577193768691.916,
    7630097256522.6279,
    7682821409440.8594,
    7735362522399.2236,
    7787716950668.5537,
    7839881109786.6299,
    7891851475489.291,
    7943624583624.1924,
    7995197030047.749,
    8046565470505.8623,
    8097726620498.582,
    8148677255129.4268,
    8199414208939.6592,
    8249934375727.9277,
    8300234708355.8584,
    8350312218539.7832,
    8400163976629.2471,
    8449787111372.5605,
    8499178809669.8564,
    8548336316314.084,
    8597256933720.2305,
    8645938021643.2539,
    8694376996885.1318,
    8742571332991.2041,
    8790518559936.5361,
    8838216263802.2441,
    8885662086442.5137,
    8932853725142.3965,
    8979788932266.9473,
    9026465514901.832,
    9072881334485.918,
    9119034306436.0684,
    9164922399764.5254,
    9210543636689.166,
    9255896092236.9062,
    9300977893840.7051,
    9345787220930.2656,
    9390322304516.8867,
    9434581426772.6758,
    9478562920604.4199,
    9522265169222.4082,
    9565686605704.5176,
    9608825712555.6133,
    9651681021262.9023,
    9694251111847.0938,
    9736534612409.9258,
    9778530198678.0938,
    9820236593543.9746,
    9861652566603.2188,
    9902776933689.582,
    9943608556407.1777,
    9984146341660.2715,
    10024389241180.996,
    10064336251055.02,
    10103986411245.57,
    10143338805115.729,
    10182392558949.537,
    10221146841471.77,
    10259600863366.846,
    10297753876796.676,
    10335605174918.225,
    10373154091400.221,
    10410399999939.768,
    10447342313778.721,
    10483980485220.086,
    10520314005144.52,
    10556342402527.148,
    10592065243954.838,
    10627482133143.963,
    10662592710459.016,
    10697396652431.82,
    10731893671281.994,
    10766083514438.223,
    10799965964060.963,
    10833540836566.229,
    10866807982151.029,
    10899767284320.166,
    10932418659414.785,
    10964762056142.584,
    10996797455109.844,
    11028524868355.518,
    11059944338887.141,
    11091055940219.014,
    11121859775912.461,
    11152355979118.445,
    11182544712122.428,
    11212426165891.754,
    11242000559625.406,
    11271268140306.48,
    11300229182257.111,
    11328883986696.256,
    11357232881300.062,
    11385276219765.18,
    11413014381374.893,
    11440447770568.145,
    11467576816511.492,
    11494401972674.252,
    11520923716406.49,
    11547142548520.236,
    11573058992873.834,
    11598673595959.461,
    11623986926493.828,
    11648999575012.27,
    11673712153465.965,
    11698125294822.561,
    11722239652670.162,
    11746055900824.623,
    11769574732940.322,
    11792796862124.297,
    11815723020553.906,
    11838353959097.84,
    11860690446940.725,
    11882733271211.16,
    11904483236613.303,
    11925941165062.002,
    11947107895321.389,
    11967984282647.137,
    11988571198432.213,
    12008869529856.135,
    12028880179537.986,
    12048604065192.846,
    12068042119291.859,
    12087195288725.926,
    12106064534472.861,
    12124650831268.346,
    12142955167280.287,
    12160978543786.838,
    12178721974857.982,
    12196186487040.689,
    12213373119047.705,
    12230282921449.855,
    12246916956371.861,
    12263276297191.846,
    12279362028244.258,
    12295175244526.406,
    12310717051408.449,
    12325988564347.025,
    12340990908602.223,
    12355725218958.162,
    12370192639447.006,
    12384394323076.43,
    12398331431560.562,
    12412005135054.371,
    12425416611891.379,
    12438567048324.936,
    12451457638272.658,
    12464089583064.539,
    12476464091194.037,
    12488582378072.854,
    12500445665788.748,
    12512055182866.818,
    12523412164033.971,
    12534517849986.645,
    12545373487161.811,
    12555980327511.164,
    12566339628278.518,
    12576452651780.256,
    12586320665189.135,
    12595944940321.068,
    12605326753425.016,
    12614467384975.994,
    12623368119471.16,
    12632030245228.828,
    12640455054190.627,
    12648643841726.588,
    12656597906443.111,
    12664318549994.041,
    12671807076894.48,
    12679064794337.594,
    12686093012014.309,
    12692893041935.688,
    12699466198258.367,
    12705813797112.535,
    12711937156432.801,
    12717837595791.857,
    12723516436236.711,
    12728975000127.754,
    12734214610980.357,
    12739236593309.203,
    12744042272475.174,
    12748632974534.93,
    12753010026092.844,
    12757174754155.734,
    12761128485989.891,
    12764872548980.688,
    12768408270494.654,
    12771736977743.973,
    12774859997653.355,
    12777778656729.365,
    12780494280932.047,
    12783008195548.846,
    12785321725071.0,
    12787436193072.057,
    12789352922088.682,
    12791073233503.762,
    12792598447431.654,
    12793929882605.611,
    12795068856267.463,
    12796016684059.291,
    12796774679917.309,
    12797344155967.82,
    12797726422425.18,
    12797922787491.859,
    12797934557260.469,
    12797763035617.822,
    12797409524150.9,
    12796875322054.816,
    12796161726042.639,
    12795270030257.166,
    12794201526184.549,
    12792957502569.717,
    12791539245333.676,
    12789948037492.641,
    12788185159078.791,
    12786251887063.008,
    12784149495279.119,
    12781879254350.061,
    12779442431615.486,
    12776840291061.33,
    12774074093250.713,
    12771145095256.709,
    12768054550596.617,
    12764803709167.744,
    12761393817184.904,
    12757826117119.26,
    12754101847638.871,
    12750222243550.609,
    12746188535743.562,
    12742001951133.975,
    12737663712611.49,
    12733175038986.91,
    12728537144941.33,
    12723751240976.582,
    12718818533367.135,
    12713740224113.178,
    12708517510895.238,
    12703151587029.873,
    12697643641426.754,
    12691994858547.029,
    12686206418362.818,
    12680279496318.066,
    12674215263290.443,
    12668014885554.613,
    12661679524746.469,
    12655210337828.74,
    12648608477057.459,
    12641875089949.822,
    12635011319252.975,
    12628018302913.916,
    12620897174050.449,
    12613649060923.275,
    12606275086908.967,
    12598776370474.07,
    12591154025150.178,
    12583409159509.971,
    12575542877144.205,
    12567556276639.73,
    12559450451558.285,
    12551226490416.475,
    12542885476666.35,
    12534428488677.109,
    12525856599717.566,
    12517170877939.469,
    12508372386361.768,
    12499462182855.59,
    12490441320130.119,
    12481310845719.219,
    12472071801968.932,
    12462725226025.639,
    12453272149825.078,
    12443713600082.1,
    12434050598281.088,
    12424284160667.234,
    12414415298238.336,
    12404445016737.533,
    12394374316646.484,
    12384204193179.453,
    12373935636277.811,
    12363569630605.447,
    12353107155544.566,
    12342549185192.32,
    12331896688357.947,
    12321150628560.488,
    12310311964027.225,
    12299381647692.541,
    12288360627197.451,
    12277249844889.68,
    12266050237824.26,
    12254762737764.674,
    12243388271184.57,
    12231927759269.893,
    12220382117921.643,
    12208752257759.043,
    12197039084123.23,
    12185243497081.41,
    12173366391431.496,
    12161408656707.199,
    12149371177183.518,
    12137254831882.795,
    12125060494581.031,
    12112789033814.807,
    12100441312888.402,
    12088018189881.537,
    12075520517657.367,
    12062949143870.912,
    12050304910977.828,
    12037588656243.65,
    12024801211753.225,
    12011943404420.666,
    11999016055999.574,
    11986019983093.574,
    11972955997167.262,
    11959824904557.355,
    11946627506484.299,
    11933364599064.09,
    11920036973320.385,
    11906645415196.973,
    11893190705570.496,
    11879673620263.432,
    11866094930057.385,
    11852455400706.646,
    11838755792951.932,
    11824996862534.492,
    11811179360210.418,
    11797304031765.102,
    11783371618028.107,
    11769382854888.08,
    11755338473308.053,
    11741239199340.797,
    11727085754144.553,
    11712878853998.805,
    11698619210320.354,
    11684307529679.598,
    11669944513816.889,
    11655530859659.211,
    11641067259336.898,
    11626554400200.639,
    11611992964838.582,
    11597383631093.629,
    11582727072080.848,
    11568023956205.076,
    11553274947178.664,
    11538480704039.338,
    11523641881168.229,
    11508759128307.998,
    11493833090581.143,
    11478864408508.326,
    11463853718026.965,
    11448801650509.822,
    11433708832783.754,
    11418575887148.539,
    11403403431395.852,
    11388192078828.281,
    11372942438278.482,
    11357655114128.41,
    11342330706328.627,
    11326969810417.693,
    11311573017541.658,
    11296140914473.592,
    11280674083633.223,
    11265173103106.639,
    11249638546666.002,
    11234070983789.426,
    11218470979680.795,
    11202839095289.715,
    11187175887331.518,
    11171481908307.24,
    11155757706523.762,
    11140003826113.857,
    11124220807056.41,
    11108409185196.574,
    11092569492266.006,
    11076702255903.129,
    11060807999673.422,
    11044887243089.707,
    11028940501632.523,
    11012968286770.445,
    10996971105980.475,
    10980949462768.436,
    10964903856689.342,
    10948834783367.83,
    10932742734518.596,
    10916628197966.777,
    10900491657668.412,
    10884333593730.85,
    10868154482433.188,
    10851954796246.691,
    10835735003855.213,
    10819495570175.609,
    10803236956378.148,
    10786959619906.879,
    10770664014500.018,
    10754350590210.328,
    10738019793425.453,
    10721672066888.24,
    10705307849717.057,
    10688927577426.07,
    10672531681945.504,
    10656120591641.879,
    10639694731338.24,
    10623254522334.295,
    10606800382426.604,
    10590332725928.654,
    10573851963691.01,
    10557358503121.295,
    10540852748204.236,
    10524335099521.66,
    10507805954272.381,
    10491265706292.148,
    10474714746073.469,
    10458153460785.447,
    10441582234293.516,
    10425001447179.211,
    10408411476759.814,
    10391812697108.0,
    10375205479071.404,
    10358590190292.176,
    10341967195226.443,
    10325336855163.762,
    10308699528246.461,
    10292055569489.031,
    10275405330797.311,
    10258749160987.756,
    10242087405806.607,
    10225420407948.955,
    10208748507077.795,
    10192072039843.021,
    10175391339900.34,
    10158706737930.152,
    10142018561656.332,
    10125327135864.982,
    10108632782423.104,
    10091935820297.221,
    10075236565571.938,
    10058535331468.398,
    10041832428362.729,
    10025128163804.373,
    10008422842534.434,
    9991716766503.8105,
    9975010234891.4004,
    9958303544122.207,
    9941596987885.3184,
    9924890857151.8496,
    9908185440192.8574,
    9891481022597.1055,
    9874777887288.8418,
    9858076314545.4297,
    9841376582014.9648,
    9824678964733.791,
    9807983735143.9473,
    9791291163110.5352,
    9774601515939.0684,
    9757915058392.6328,
    9741232052709.1074])


class TestPlanckLaw(unittest.TestCase):
    """
    Defines :func:`colour.colorimetry.blackbody.planck_law` definition units
    tests methods.
    """

    def test_planck_law(self):
        """
        Tests :func:`colour.colorimetry.blackbody.planck_law` definition.
        """

        for temperature, wavelengths in sorted(PLANCK_LAW_DATA.items()):
            for wavelength, radiance in sorted(wavelengths.items()):
                np.testing.assert_allclose(
                    planck_law(wavelength * 1e-9, temperature),
                    radiance,
                    rtol=0.00000001,
                    verbose=False)


class TestBlackbodySpd(unittest.TestCase):
    """
    Defines
    :func:`colour.colorimetry.blackbody.blackbody_spd`
    definition unit tests methods.
    """

    def test_blackbody_spd(self):
        """
        Tests
        :func:`colour.colorimetry.blackbody.blackbody_spd`
        definition.
        """

        np.testing.assert_almost_equal(
            blackbody_spd(5000, SpectralShape(360, 830, 1)).values,
            BLACKBODY_SPD_DATA,
            # Lower precision for Linux *Travis-ci* tests.
            decimal=0)


if __name__ == '__main__':
    unittest.main()
