#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
Defines unit tests for :mod:`colour.notation.munsell` module.
"""

from __future__ import division, unicode_literals

import numpy as np
import sys

if sys.version_info[:2] <= (2, 6):
    import unittest2 as unittest
else:
    import unittest

from colour.notation.munsell import (
    parse_munsell_colour,
    is_grey_munsell_colour,
    normalize_munsell_specification)
from colour.notation.munsell import (
    munsell_colour_to_munsell_specification,
    munsell_specification_to_munsell_colour)
from colour.notation.munsell import (
    xyY_from_renotation,
    is_specification_in_renotation)
from colour.notation.munsell import bounding_hues_from_renotation
from colour.notation.munsell import hue_to_hue_angle, hue_angle_to_hue
from colour.notation.munsell import hue_to_ASTM_hue
from colour.notation.munsell import (
    interpolation_method_from_renotation_ovoid,
    xy_from_renotation_ovoid)
from colour.notation.munsell import LCHab_to_munsell_specification
from colour.notation.munsell import maximum_chroma_from_renotation
from colour.notation.munsell import munsell_specification_to_xy
from colour.notation.munsell import (
    munsell_specification_to_xyY,
    xyY_to_munsell_specification)
from colour.notation import (
    munsell_value_Priest1920,
    munsell_value_Munsell1933,
    munsell_value_Moon1943,
    munsell_value_Saunderson1944,
    munsell_value_Ladd1955,
    munsell_value_McCamy1987,
    munsell_value_ASTMD153508)

__author__ = 'Colour Developers'
__copyright__ = 'Copyright (C) 2013 - 2014 - Colour Developers'
__license__ = 'New BSD License - http://opensource.org/licenses/BSD-3-Clause'
__maintainer__ = 'Colour Developers'
__email__ = 'colour-science@googlegroups.com'
__status__ = 'Production'

__all__ = ['MUNSELL_SPECIFICATIONS',
           'MUNSELL_GREYS_SPECIFICATIONS',
           'MUNSELL_EVEN_SPECIFICATIONS',
           'MUNSELL_BOUNDING_HUES',
           'MUNSELL_HUE_TO_ANGLE',
           'MUNSELL_HUE_TO_ASTM_HUE',
           'MUNSELL_INTERPOLATION_METHODS',
           'MUNSELL_XY_FROM_RENOTATION_OVOID',
           'MUNSELL_SPECIFICATIONS_TO_XY',
           'MUNSELL_COLOURS_TO_XYY',
           'MUNSELL_GREYS_TO_XYY',
           'XYY_TO_MUNSELL_SPECIFICATIONS',
           'XYY_TO_MUNSELL_GREYS_SPECIFICATIONS',
           'NON_CONVERGING_XYY',
           'TestMunsellValuePriest1920',
           'TestMunsellValueMunsell1933',
           'TestMunsellValueMoon1943',
           'TestMunsellValueSaunderson1944',
           'TestMunsellValueLadd1955',
           'TestMunsellValueMcCamy1992',
           'TestMunsellValueASTMD153508',
           'TestMunsellSpecification_to_xyY',
           'TestMunsellColour_to_xyY',
           'TestxyY_to_munsell_specification',
           'TestxyY_to_munsell_colour',
           'TestParseMunsellColour',
           'TestIsGreyMunsellColour',
           'TestNormalizeMunsellSpecification',
           'TestMunsellColourToMunsellSpecification',
           'TestMunsellSpecificationToMunsellColour',
           'Test_xyY_fromRenotation',
           'TestIsSpecificationInRenotation',
           'TestBoundingHuesFromRenotation',
           'TestHueToHueAngle',
           'TestHueAngleToHue',
           'TestHueTo_ASTM_hue',
           'TestInterpolationMethodFromRenotationOvoid',
           'Test_xy_fromRenotationOvoid',
           'TestLCHabToMunsellSpecification',
           'TestMaximumChromaFromRenotation',
           'TestMunsellSpecification_to_xy']

# TODO: Investigate if tests can be simplified by using a common valid set of
# specifications.
MUNSELL_SPECIFICATIONS = (
    (2.5, 7.9653798470827155, 11.928546308350969, 4),
    (2.5, 6.197794822090879, 6.923610826208884, 4),
    (2.5, 5.311956978256753, 2.0, 4),
    (5.613007062442384, 8.402756538070792, 18.56590894044391, 4),
    (5.845640071004907, 8.062638664520136, 5.782325614552295, 4),
    (5.780794121059599, 3.174804081025836, 3.3492086825591487, 4),
    (5.483684299639117, 3.8994120994080133, 5.761459062506715, 4),
    (5.809580308813496, 5.816975143899512, 6.662613753958899, 4),
    (5.209252955662903, 2.9770364483569107, 5.141472643810014, 4),
    (7.706105853911573, 2.789942201654241, 11.396648897274897, 4),
    (7.5675942867463615, 9.569378264154928, 16.714918860774414, 4),
    (8.117640564564343, 2.7489429651492028, 3.1653563832640272, 4),
    (7.8731203012311255, 2.6438472620092806, 13.241107969297714, 4),
    (8.04983322214289, 2.4630649870973422, 7.501924679081063, 4),
    (8.355307569391062, 2.703242274198649, 11.925441344336392, 4),
    (8.342795760577609, 1.0627446691234035, 6.298818145909256, 4),
    (7.5947244020062845, 1.5750745121803325, 4.626613135331287, 4),
    (8.19517786608579, 8.732504313513864, 23.571122010181508, 4),
    (7.754763634912469, 8.437206137825585, 21.00944901061068, 4),
    (9.010231962978862, 6.1312711883866395, 6.803370568930175, 4),
    (9.041566851651622, 6.4540531985593965, 17.010037203566448, 4),
    (9.915652169827913, 8.56438797679146, 11.13108215988432, 4),
    (10.0, 8.651470349341308, 27.322046186799103, 4),
    (9.961336111598143, 8.039682739223524, 13.20009863344056, 4),
    (9.887406551063181, 8.321342653987184, 2.0660963235598375, 4),
    (10.0, 3.400787121787084, 2.5700932200974145, 4),
    (10.0, 3.063915609453643, 13.514066607169514, 4),
    (10.0, 5.461465491798149, 12.753899774963989, 4),
    (10.0, 5.90081409486059, 15.244598276849418, 4),
    (10.0, 5.4222087054147545, 27.929001019877095, 4),
    (9.757039645743053, 5.653647411872443, 3.4112871270786895, 4),
    (10.0, 5.790357134071424, 24.86360130658431, 4),
    (9.862075817629322, 4.487864213671867, 7.67196809500038, 4),
    (3.2140937198013564, 9.345163595199718, 3.4367939376082868, 3),
    (3.484005759599379, 9.572118958552942, 14.905079424139613, 3),
    (3.1967035260607033, 9.059573376604588, 24.78003138905329, 3),
    (2.5, 9.479129956842218, 27.736581704977635, 3),
    (2.7908763449337677, 8.166099921946278, 20.868304564027603, 3),
    (3.221499566897477, 5.507741920664265, 5.467726257137659, 3),
    (2.622512070432247, 5.989380652373817, 19.364472252973304, 3),
    (3.2873061024849806, 5.439892524933965, 19.855724192587914, 3),
    (5.727612405003367, 3.013295327457818, 10.746642552166502, 3),
    (5.347955701149093, 3.003537709503816, 18.900471815194905, 3),
    (5.7385751713204325, 3.987559993529851, 4.223160837759656, 3),
    (5.720824103581511, 1.804037523043165, 4.878068159363519, 3),
    (5.316780024484356, 1.0305080135789524, 8.043957606541364, 3),
    (5.7623230008312385, 1.6541934959363132, 9.507411716255689, 3),
    (5.985579505387595, 2.2109765673980277, 14.803434527189347, 3),
    (5.461619603420755, 2.805568235937479, 6.6471547360970025, 3),
    (7.838277926195208, 2.8050500161595604, 6.238528025218592, 3),
    (8.2830613968175, 2.716343821673611, 10.350825174769154, 3),
    (7.603155032355272, 6.1394212951580345, 29.139541165198704, 3),
    (8.324115039527976, 6.971801555303874, 23.515778973195257, 3),
    (8.44424273124686, 6.657492305333222, 2.4130843113046656, 3),
    (8.309061774521076, 6.371190719454564, 17.507252134514488, 3),
    (8.14037117068092, 2.6868573867536836, 14.649933295354042, 3),
    (8.484903553213694, 2.2057045177976002, 11.879562262633948, 3),
    (8.454109029623016, 2.3630506284708144, 4.606317173304252, 3),
    (8.305262429168986, 5.460535517182709, 3.9045072719017924, 3),
    (8.189730004579287, 5.069933398792441, 28.126992759236863, 3),
    (7.54028778107475, 5.779995612547662, 6.635319193935916, 3),
    (7.9629991342362985, 5.233597701388516, 20.293354805626866, 3),
    (8.432959559038371, 5.797128354507666, 26.469970873757067, 3),
    (10.0, 9.005161484782885, 6.0469956581432704, 3),
    (9.771353946056914, 9.383759836829901, 20.82975271547889, 3),
    (9.376380796522223, 9.46044820450894, 13.348522394106682, 3),
    (9.912704179532229, 4.057804958576875, 25.778231770351923, 3),
    (10.0, 4.853695964045051, 13.712247643370837, 3),
    (10.0, 4.221211292509457, 28.587923360931033, 3),
    (9.287535146732925, 4.404206868704275, 6.997389565284625, 3),
    (10.0, 5.717897422867529, 30.932435068478792, 3),
    (10.0, 5.121046242854478, 7.946854746461393, 3),
    (10.0, 5.631186501571907, 26.172410297895773, 3),
    (2.5, 6.822278767379375, 12.643410557057086, 2),
    (2.5, 3.3435596434006034, 19.167537762557394, 2),
    (3.284581774573411, 3.7457477655465423, 10.316761862277126, 2),
    (3.0814075494281132, 3.302789020993419, 4.031683724514751, 2),
    (2.5, 9.595267222759654, 9.136435041220121, 2),
    (2.5899169115530087, 9.55055785508054, 8.263133397233354, 2),
    (2.5342634625499727, 9.494299074607266, 14.863663104253218, 2),
    (5.275920564662094, 9.02282018751374, 12.879135949769728, 2),
    (5.522856449128964, 9.387711396347438, 17.412586595686815, 2),
    (5.885914939777947, 9.191119089368966, 17.388086814072437, 2),
    (5.4717401116974616, 9.868862187868638, 11.646848538821667, 2),
    (5.956560321967156, 4.186123335197883, 4.31169020481439, 2),
    (5.6279111948942635, 4.547202429787774, 16.56681914443115, 2),
    (5.8534547245334565, 4.592599799739227, 18.83506980508535, 2),
    (5.144720369630256, 5.318575486426688, 18.979172966805407, 2),
    (5.2907074463880175, 6.000990946276877, 13.598520998056053, 2),
    (5.415844403197766, 6.398031110922737, 15.178617464461626, 2),
    (8.204144852288245, 5.902107978077237, 4.020177691372295, 2),
    (9.366069953403018, 3.3728653869498273, 15.422766182794579, 2),
    (10.0, 3.949081763597084, 9.192387616705815, 2),
    (10.0, 3.187455579956449, 15.954247893607032, 2),
    (9.260586271537607, 3.4545177339210404, 10.59517579170162, 2),
    (9.571675864670619, 3.149737124891618, 17.398847531397934, 2),
    (3.2387393821759787, 4.827650915864795, 3.7435106940988625, 1),
    (2.5, 4.30220435408426, 7.399343614420917, 1),
    (2.5, 4.329470943798639, 8.860840417367838, 1),
    (2.5, 7.620094327678255, 10.887265616829124, 1),
    (2.5, 7.1449996531857725, 10.10233537418591, 1),
    (2.6104349455855846, 7.700939489093993, 4.236171515065992, 1),
    (2.5, 8.524455647347406, 5.3613636980274295, 1),
    (3.1731014606584806, 8.133658146416419, 15.199536235308303, 1),
    (2.5, 7.129372162253073, 5.4202608625739925, 1),
    (2.5, 7.70850985024877, 9.619364938403443, 1),
    (3.252581509053177, 7.081532543557421, 6.2224060204343745, 1),
    (2.5, 7.67557944940156, 12.261808397585057, 1),
    (2.5, 3.4825807865537914, 7.768505546917617, 1),
    (2.5, 3.020783157962588, 6.998840911724095, 1),
    (3.020562119690717, 3.1223174909201346, 5.203087539105082, 1),
    (5.2190911687613255, 3.0070655951585925, 13.573887550967275, 1),
    (5.5962506280473505, 2.15728255216339, 5.165106850365733, 1),
    (5.078574838897358, 2.9637552645895053, 6.8599427244043705, 1),
    (5.1756171558445825, 2.772951703906637, 4.56080038214103, 1),
    (5.497353020782844, 5.410551418942688, 2.0, 1),
    (5.841773513544001, 5.686667624085427, 13.28936566781855, 1),
    (5.580549185463668, 6.964187735662777, 16.1803201492634, 1),
    (5.287772726922527, 6.865396694853934, 14.098946461580404, 1),
    (8.358221285614269, 4.591594256415192, 17.271563597297103, 1),
    (7.87724479635977, 4.744438140664897, 5.598934346859475, 1),
    (8.323336953587479, 4.566800376285041, 7.0881523668119195, 1),
    (7.845486096299681, 4.586270737017715, 16.23379517928239, 1),
    (8.382569502344943, 4.562211644069123, 13.97512411087629, 1),
    (7.855593749782354, 3.238350356301548, 5.360435825061775, 1),
    (7.655501153733914, 3.903923881082662, 9.769593047963392, 1),
    (7.653019158008493, 6.348396270933699, 11.704589766625281, 1),
    (10.0, 2.7176353295329094, 5.415846167802247, 1),
    (9.196648156004963, 8.15078293499349, 5.069223366759241, 1),
    (10.0, 6.040694822625091, 7.76280231640685, 1),
    (10.0, 6.719017792521678, 18.37437538640251, 1),
    (2.8739501345809, 3.5100389001084373, 4.494521106912674, 10),
    (2.979763831715893, 8.5642374861117, 6.426710793964199, 10),
    (2.5, 8.924876646785982, 2.491252841450378, 10),
    (2.5, 8.121352187119456, 8.82337986403619, 10),
    (2.5, 4.643160393937538, 18.83933997786449, 10),
    (2.5, 4.925443059836121, 5.417711811598947, 10),
    (2.5, 8.509385882792433, 8.04535672534691, 10),
    (2.5, 2.709647356385667, 16.195810159806815, 10),
    (5.6678871626197305, 1.8444622064585485, 18.226010811743183, 10),
    (5.759673840199206, 1.960972599684376, 30.42873152741525, 10),
    (5.783634661463273, 1.5360819708237339, 21.480194214511137, 10),
    (5.118173248862928, 1.5400563354602976, 41.86847335857883, 10),
    (5.757349724389667, 1.6383453350505301, 13.609604267804956, 10),
    (5.279304061296045, 4.900840641360432, 22.876127528048663, 10),
    (5.715709801059808, 4.570357108788123, 30.360213488022158, 10),
    (5.947947304520848, 4.273422536180247, 4.8966439066197935, 10),
    (5.09899322481724, 4.947505227279317, 26.26875042475258, 10),
    (5.53222949762985, 4.629910893964432, 7.756449262721482, 10),
    (5.923584541768192, 4.593239396795306, 19.567605030849386, 10),
    (5.950156387030171, 2.42463499343633, 4.953666946161412, 10),
    (5.614158136535322, 2.4812727587161407, 20.644953904366893, 10),
    (5.435908140730638, 2.7884847594702746, 21.585064332200393, 10),
    (5.539908561343329, 2.9864344023506266, 44.90369903995316, 10),
    (5.3792514320991325, 2.137036038265424, 25.88907455882873, 10),
    (5.632909830682246, 5.9349482115124506, 21.384042506861697, 10),
    (5.20332651493292, 5.825367195549048, 15.514467427422431, 10),
    (5.927793692134072, 5.448079050348612, 3.7766395197414253, 10),
    (5.817322396187511, 5.292185862716667, 11.31804158090752, 10),
    (7.949960633591607, 2.873765731226449, 25.621368902089333, 10),
    (8.382592436810759, 2.461570417216745, 40.54127195292601, 10),
    (7.96379736332257, 2.200134671312228, 36.70731870996695, 10),
    (8.373924456610474, 2.3066883154384743, 8.623846064990166, 10),
    (8.151990686473388, 2.2622251239305577, 42.229127196458144, 10),
    (8.25502764532606, 9.609182815192318, 7.080986046028279, 10),
    (8.488384085232076, 8.098523111957578, 9.779628072315807, 10),
    (8.438357068876163, 2.6893452283620705, 26.873452492074044, 10),
    (8.309434906530441, 2.4623229011742396, 48.49966399344499, 10),
    (7.7115794149655015, 2.724728645017314, 5.729859843354196, 10),
    (7.6273740879401934, 2.2251923932068416, 26.724973070776922, 10),
    (7.693923337226084, 2.6579274123978887, 48.407897505690485, 10),
    (10.0, 6.197418391023862, 10.97195381591066, 10),
    (9.113097274740381, 6.270996638245157, 2.7564645951736484, 10),
    (10.0, 9.235232580795238, 6.003388325186025, 10),
    (10.0, 5.050367446997329, 19.170756721559698, 10),
    (9.380110088755156, 5.5356649305105154, 18.817507743754415, 10),
    (9.001795946577033, 7.786061808916703, 4.453854563212078, 10),
    (10.0, 7.692030956316567, 3.653159723688856, 10),
    (9.046182896421445, 3.0439259875156295, 22.300946806849847, 10),
    (9.459420796383784, 3.0372188559586917, 10.552556949414955, 10),
    (10.0, 3.3229506269252425, 31.2476220198124, 10),
    (10.0, 3.1004893435032645, 29.2734347311525, 10),
    (2.5, 7.990213836555715, 8.375074375178261, 9),
    (2.5, 7.298301069157875, 9.502846862649331, 9),
    (2.8619005171223564, 7.275426002317967, 7.466126134628901, 9),
    (3.0874221941355513, 8.485000561300847, 2.493857829360787, 9),
    (2.5, 3.690667859366627, 19.77471678075617, 9),
    (3.220553507003754, 3.281507210559706, 37.05938066272616, 9),
    (2.5, 3.8989428412499203, 39.166418500944374, 9),
    (2.7654037016841957, 3.1169069187360945, 29.726535569137937, 9),
    (2.5, 3.703448940191029, 12.087654687250128, 9),
    (2.5, 3.433194385943258, 3.382852759577178, 9),
    (2.836612137080781, 3.9924265837199604, 2.0, 9),
    (2.8888545547050946, 3.2474346036095905, 14.618307037832857, 9),
    (5.164399331990519, 6.2346627424063925, 9.111465383743912, 9),
    (5.500356903003388, 6.736841239972426, 13.154128131968298, 9),
    (5.535810057742433, 6.970342536034459, 8.892716664134475, 9),
    (5.590040966343994, 3.5668609688847175, 22.75661278689855, 9),
    (5.282620261743346, 3.2367340323019573, 18.732823688754383, 9),
    (5.172895640160181, 3.0043051231231956, 6.2292543458148515, 9),
    (5.259721854731981, 3.3004333429874864, 35.890872110681414, 9),
    (5.5536463415959245, 3.4948508349893386, 10.076683709549055, 9),
    (5.730003972159145, 2.488034141173207, 15.985698390269977, 9),
    (5.782381516990652, 2.4812045413951833, 28.774618518379302, 9),
    (5.069379781665461, 6.741533325352479, 2.2194841714206595, 9),
    (5.1346796709796605, 6.103139133682482, 27.726398643923417, 9),
    (5.383260687864624, 5.56099784134289, 18.302295934127923, 9),
    (5.869792088464701, 5.233311379347905, 32.55343216796663, 9),
    (5.462451143540612, 5.746471808899983, 30.948864634440213, 9),
    (5.357445269639698, 5.68852667194441, 5.261434469006405, 9),
    (5.626373453003034, 5.771003693827525, 25.170846666445236, 9),
    (8.284200895164993, 2.466049819474928, 17.238899804160177, 9),
    (8.318102784124019, 2.2658035726687236, 22.596147383535918, 9),
    (7.851936866242713, 7.45229335345878, 20.962374407911458, 9),
    (8.146081336032703, 7.714405906472637, 13.533962918469337, 9),
    (8.09720864316275, 7.247339841946607, 17.33899155052454, 9),
    (7.830256291991797, 6.872416994269415, 10.706822163825924, 9),
    (7.80065897068848, 6.330678323824742, 6.211375680877805, 9),
    (8.044863647118635, 6.808226317611471, 15.557155261544228, 9),
    (8.461774802909071, 4.745965252820717, 36.03729693977732, 9),
    (7.612382882207284, 4.372367470892327, 14.168690780706225, 9),
    (8.169633927695997, 4.48833473800357, 27.23584610386441, 9),
    (9.602031136015775, 5.527970638413552, 20.5806356758181, 9),
    (9.663686030178818, 5.516978463101205, 29.047658472982956, 9),
    (9.75292854736471, 5.461162553197844, 34.11493160528129, 9),
    (10.0, 5.650424904167431, 4.216215730437086, 9),
    (10.0, 5.73654367766597, 34.72852675583916, 9),
    (10.0, 5.4360854849263855, 14.779627294882367, 9),
    (10.0, 5.79544155795279, 2.0, 9),
    (9.49705091394873, 5.914105479148815, 10.80885478009873, 9),
    (9.826635163465532, 1.9759992882300867, 7.06711443184985, 9),
    (9.382502350301259, 4.709738717837755, 19.999476877446362, 9),
    (9.115530591819274, 4.986025386567032, 5.883436488694818, 9),
    (10.0, 4.813033015882831, 24.745870232952445, 9),
    (9.378359588580793, 4.574376802251692, 26.295787257422923, 9),
    (10.0, 2.1709322459501545, 21.57257635660235, 9),
    (10.0, 2.5713046143569223, 26.039872491235577, 9),
    (2.5, 2.6357605512712707, 4.712138166253982, 8),
    (2.8874578666829285, 2.0337681540970594, 13.994896052145748, 8),
    (3.435419560439465, 2.2299190864211247, 6.718989113532732, 8),
    (2.9925336062737173, 1.928933557645075, 7.198014339866309, 8),
    (2.5, 1.3726890604845965, 14.156726710024465, 8),
    (2.6104579288975813, 1.2137704813997643, 3.3458156268951917, 8),
    (5.1670653045538115, 7.761502840367845, 2.1409481568506346, 8),
    (5.054434114346951, 7.011456904063963, 6.442157332603133, 8),
    (5.803735682450612, 8.51299345440391, 10.443841773523394, 8),
    (5.044877539779968, 6.342036003669621, 18.424428701407553, 8),
    (5.484832402621484, 6.739510598555563, 5.474777491295647, 8),
    (5.162300427200289, 6.57672216934989, 24.999056248525125, 8),
    (5.877256360743413, 6.789776791182118, 15.450444143259661, 8),
    (8.197449080109873, 2.2092984979309276, 2.0, 8),
    (7.997237265754237, 2.060313094466323, 11.655829335806517, 8),
    (7.973192560907184, 8.67128307488709, 4.272886886879181, 8),
    (7.836498646186221, 8.168701526186094, 13.596658717999025, 8),
    (7.782186965908517, 9.202193528464585, 13.902105524067945, 8),
    (9.531795266771761, 5.037755377967032, 2.0, 8),
    (10.0, 5.41661210331397, 11.055624912778937, 8),
    (9.312270837393163, 7.466203120412419, 11.185222099189973, 8),
    (10.0, 7.097905887270363, 13.895455902446677, 8),
    (9.925669940032272, 4.692192166283825, 7.2040789887667955, 8),
    (9.416740882402403, 4.697368796121149, 8.720116348180492, 8),
    (10.0, 4.338509514756336, 16.469698910991372, 8),
    (10.0, 6.402201264456283, 6.599237233947309, 8),
    (10.0, 5.182208073338139, 4.550269784467781, 8),
    (9.970332530519679, 5.903209540812212, 10.837022722087644, 8),
    (2.962707587174585, 9.2513521634857, 9.999116931630539, 7),
    (3.1672052728994915, 9.141134617154027, 7.383624729892915, 7),
    (2.5, 5.049858089466979, 17.881593853007615, 7),
    (2.7415018638966284, 5.680976628228491, 18.00290873780138, 7),
    (2.5, 5.8481154189353175, 10.232668996271492, 7),
    (2.877902226185231, 5.567414385297515, 3.5582034231201787, 7),
    (2.5, 5.8534450733346, 27.77999592691697, 7),
    (5.412821771284458, 2.5214549204115335, 7.258040020605607, 7),
    (5.83754747605084, 2.530273181625722, 11.998261380615471, 7),
    (5.9693975439749885, 4.3487706338488, 14.397906420283302, 7),
    (5.004079000563381, 4.657273345320005, 22.736677614468775, 7),
    (5.168438425945292, 4.24641271720769, 4.844860547907693, 7),
    (5.863284315202094, 4.359153796629064, 23.489710023246513, 7),
    (5.756333389411959, 8.952011225713635, 7.301135618422141, 7),
    (5.108337403014788, 8.31154202432518, 11.359771531491097, 7),
    (8.314898437378535, 9.185953513281046, 4.238233636005843, 7),
    (8.201460399608226, 4.230965415446139, 11.589840844520428, 7),
    (7.595604919273442, 4.88445113865134, 6.798265747221928, 7),
    (8.378186361828917, 9.484819582257831, 8.022357890675561, 7),
    (8.028236284464779, 9.757701617444052, 11.574198271062086, 7),
    (8.229270762113973, 8.691786353579515, 6.350022396927342, 7),
    (10.0, 3.3059509658558612, 3.1152259635487924, 7),
    (9.756267998308681, 3.1863606517354883, 14.803384721914584, 7),
    (10.0, 3.5046891678155427, 13.90160960971739, 7),
    (10.0, 8.784136629159212, 6.218490965882184, 7),
    (10.0, 8.37434528326138, 13.887493044276624, 7),
    (10.0, 4.6140458786417, 14.68907159946693, 7),
    (10.0, 8.03303730091703, 13.518172354943417, 7),
    (2.7455640547144746, 1.6521001852026693, 5.569110673549164, 6),
    (3.1452880891491906, 5.155515834056653, 8.595832717291, 6),
    (2.5, 4.389047661368727, 4.950679151608691, 6),
    (2.5, 4.394863837189541, 4.383231249423155, 6),
    (2.5, 1.5580252510526358, 3.307282274836235, 6),
    (5.045583268005572, 8.635334543903529, 9.59194524860244, 6),
    (5.594284526041456, 8.6320252698003, 10.197201238166286, 6),
    (5.988802467213943, 8.132531816914582, 12.30595195616923, 6),
    (5.425850947396252, 5.185445600639579, 8.046156862703112, 6),
    (5.369364240119585, 5.088077743168478, 7.340573827339962, 6),
    (5.702045821590509, 5.271793984998375, 10.325652051724541, 6),
    (5.411096326958829, 5.545898372969883, 5.292034843095026, 6),
    (8.242968536635763, 9.082400742895011, 4.90020586532881, 6),
    (8.050426422258862, 9.780537958506372, 18.978339720751418, 6),
    (8.238754570485817, 8.602489911338367, 5.94133011037865, 6),
    (8.39568424389748, 4.506736427736353, 9.461515968715135, 6),
    (10.0, 5.138757136469953, 12.704963485646498, 6),
    (10.0, 5.159912610631281, 15.6753707607594, 6),
    (10.0, 5.549472965121217, 3.506573388368494, 6),
    (10.0, 5.795090421330749, 14.063922879568509, 6),
    (10.0, 6.983123234599715, 3.128443413944953, 6),
    (10.0, 6.680204754366847, 11.632405914314647, 6),
    (9.050263182466011, 6.721800647918977, 17.08367694275979, 6),
    (10.0, 6.0634616201345715, 4.736966947326921, 6),
    (9.409402543801862, 6.94420363069249, 6.28766021168659, 6),
    (9.633394604006961, 7.505827554006868, 4.623044001702525, 6),
    (9.020770192275748, 7.3138794160617016, 13.422245014577644, 6),
    (9.26317609686154, 7.357994930871833, 15.233295182477667, 6),
    (3.332782026387723, 7.225679089752617, 16.113419977677538, 5),
    (2.5, 5.428663116358418, 6.5436496028361315, 5),
    (2.5, 2.829072524106358, 2.0, 5),
    (2.8285591842433737, 8.730390823623916, 21.473258817290873, 5),
    (2.5, 8.17012010036135, 12.020108658634838, 5),
    (2.5, 8.74354045618398, 14.42790441415372, 5),
    (2.5, 4.638913962811717, 8.380243803410817, 5),
    (3.363079416671538, 4.670651645625486, 2.7755096642090313, 5),
    (5.339079962653624, 8.064094823108675, 16.611574939424255, 5),
    (5.347356764781598, 8.43641762101464, 15.41216519823205, 5),
    (5.368950609634622, 7.371653807185894, 7.038165919924306, 5),
    (5.929552854535908, 6.895926920816455, 7.57281344704806, 5),
    (5.72794655950891, 6.581660847859535, 10.668172633934036, 5),
    (5.641782139668679, 6.458019104693064, 9.549016885745186, 5),
    (5.344359642058747, 2.871097758194079, 5.430489560972486, 5),
    (7.749909297802317, 4.328832721055091, 4.268933751175051, 5),
    (8.145409228909998, 4.865021714408405, 7.545633529064384, 5),
    (7.907253670159305, 5.688395096546548, 10.770986229289623, 5),
    (7.592508492261312, 5.098997604455221, 4.933568344499713, 5),
    (7.674872690410821, 5.441049019888879, 3.5502452884794837, 5),
    (7.991979987062054, 6.616295483614106, 3.2837012487472252, 5),
    (9.345599185286883, 7.224736586735167, 17.48852175788182, 5),
    (9.659595218511388, 7.899577776723924, 3.3572177484844636, 5))

MUNSELL_GREYS_SPECIFICATIONS = np.linspace(0, 10, 25)

MUNSELL_EVEN_SPECIFICATIONS = (
    (2.5, 5.0, 12.0, 4),
    (2.5, 5.0, 32.0, 4),
    (2.5, 5.0, 22.0, 4),
    (2.5, 5.0, 32.0, 4),
    (2.5, 6.0, 18.0, 4),
    (2.5, 6.0, 32.0, 4),
    (2.5, 6.0, 6.0, 4),
    (2.5, 5.0, 42.0, 4),
    (2.5, 5.0, 26.0, 4),
    (2.5, 5.0, 48.0, 4),
    (2.5, 2.0, 14.0, 4),
    (2.5, 2.0, 14.0, 4),
    (2.5, 0.0, 14.0, 4),
    (2.5, 0.0, 2.0, 4),
    (5.0, 1.0, 46.0, 4),
    (5.0, 1.0, 38.0, 4),
    (5.0, 1.0, 12.0, 4),
    (5.0, 1.0, 10.0, 4),
    (5.0, 4.0, 16.0, 4),
    (5.0, 2.0, 44.0, 4),
    (5.0, 7.0, 2.0, 4),
    (5.0, 7.0, 8.0, 4),
    (5.0, 7.0, 32.0, 4),
    (7.5, 2.0, 28.0, 4),
    (7.5, 2.0, 12.0, 4),
    (7.5, 2.0, 34.0, 4),
    (7.5, 4.0, 24.0, 4),
    (7.5, 4.0, 10.0, 4),
    (7.5, 4.0, 18.0, 4),
    (7.5, 9.0, 44.0, 4),
    (7.5, 5.0, 12.0, 4),
    (7.5, 5.0, 40.0, 4),
    (7.5, 5.0, 30.0, 4),
    (7.5, 5.0, 12.0, 4),
    (10.0, 3.0, 38.0, 4),
    (10.0, 3.0, 16.0, 4),
    (10.0, 3.0, 32.0, 4),
    (10.0, 3.0, 44.0, 4),
    (10.0, 3.0, 42.0, 4),
    (10.0, 3.0, 34.0, 4),
    (10.0, 3.0, 18.0, 4),
    (10.0, 7.0, 10.0, 4),
    (10.0, 7.0, 40.0, 4),
    (10.0, 7.0, 12.0, 4),
    (10.0, 6.0, 42.0, 4),
    (10.0, 6.0, 6.0, 4),
    (10.0, 4.0, 40.0, 4),
    (2.5, 7.0, 28.0, 3),
    (2.5, 7.0, 26.0, 3),
    (2.5, 9.0, 44.0, 3),
    (2.5, 9.0, 26.0, 3),
    (2.5, 0.0, 32.0, 3),
    (2.5, 0.0, 26.0, 3),
    (2.5, 8.0, 30.0, 3),
    (2.5, 8.0, 30.0, 3),
    (2.5, 8.0, 6.0, 3),
    (2.5, 6.0, 32.0, 3),
    (2.5, 6.0, 12.0, 3),
    (5.0, 7.0, 28.0, 3),
    (5.0, 7.0, 26.0, 3),
    (5.0, 7.0, 46.0, 3),
    (5.0, 7.0, 10.0, 3),
    (5.0, 6.0, 10.0, 3),
    (5.0, 6.0, 44.0, 3),
    (5.0, 1.0, 2.0, 3),
    (5.0, 9.0, 34.0, 3),
    (5.0, 9.0, 30.0, 3),
    (7.5, 3.0, 12.0, 3),
    (7.5, 7.0, 26.0, 3),
    (7.5, 7.0, 18.0, 3),
    (7.5, 7.0, 42.0, 3),
    (7.5, 7.0, 20.0, 3),
    (7.5, 7.0, 16.0, 3),
    (7.5, 3.0, 36.0, 3),
    (7.5, 3.0, 38.0, 3),
    (7.5, 3.0, 14.0, 3),
    (7.5, 2.0, 30.0, 3),
    (7.5, 2.0, 12.0, 3),
    (7.5, 2.0, 8.0, 3),
    (7.5, 2.0, 6.0, 3),
    (7.5, 6.0, 34.0, 3),
    (7.5, 6.0, 12.0, 3),
    (10.0, 4.0, 14.0, 3),
    (10.0, 4.0, 40.0, 3),
    (10.0, 5.0, 2.0, 3),
    (10.0, 5.0, 26.0, 3),
    (10.0, 6.0, 40.0, 3),
    (10.0, 6.0, 46.0, 3),
    (10.0, 6.0, 18.0, 3),
    (10.0, 6.0, 38.0, 3),
    (10.0, 3.0, 16.0, 3),
    (10.0, 3.0, 32.0, 3),
    (10.0, 3.0, 26.0, 3),
    (10.0, 3.0, 22.0, 3),
    (10.0, 8.0, 2.0, 3),
    (10.0, 8.0, 10.0, 3),
    (10.0, 8.0, 12.0, 3),
    (10.0, 8.0, 18.0, 3),
    (10.0, 8.0, 44.0, 3),
    (2.5, 8.0, 2.0, 2),
    (2.5, 8.0, 42.0, 2),
    (2.5, 7.0, 34.0, 2),
    (2.5, 4.0, 36.0, 2),
    (2.5, 4.0, 34.0, 2),
    (2.5, 4.0, 22.0, 2),
    (2.5, 0.0, 42.0, 2),
    (2.5, 0.0, 32.0, 2),
    (2.5, 1.0, 28.0, 2),
    (2.5, 1.0, 2.0, 2),
    (2.5, 1.0, 24.0, 2),
    (2.5, 1.0, 12.0, 2),
    (5.0, 5.0, 22.0, 2),
    (5.0, 5.0, 46.0, 2),
    (5.0, 5.0, 24.0, 2),
    (5.0, 1.0, 48.0, 2),
    (5.0, 1.0, 12.0, 2),
    (5.0, 1.0, 16.0, 2),
    (5.0, 1.0, 2.0, 2),
    (5.0, 1.0, 18.0, 2),
    (5.0, 8.0, 28.0, 2),
    (5.0, 8.0, 32.0, 2),
    (5.0, 8.0, 24.0, 2),
    (5.0, 8.0, 38.0, 2),
    (5.0, 2.0, 24.0, 2),
    (5.0, 2.0, 4.0, 2),
    (5.0, 2.0, 32.0, 2),
    (5.0, 2.0, 38.0, 2),
    (5.0, 9.0, 36.0, 2),
    (5.0, 9.0, 34.0, 2),
    (5.0, 9.0, 4.0, 2),
    (7.5, 7.0, 28.0, 2),
    (7.5, 7.0, 10.0, 2),
    (7.5, 7.0, 48.0, 2),
    (7.5, 9.0, 48.0, 2),
    (7.5, 9.0, 48.0, 2),
    (7.5, 9.0, 30.0, 2),
    (7.5, 5.0, 42.0, 2),
    (7.5, 5.0, 46.0, 2),
    (7.5, 6.0, 26.0, 2),
    (7.5, 6.0, 28.0, 2),
    (7.5, 6.0, 22.0, 2),
    (7.5, 6.0, 10.0, 2),
    (7.5, 6.0, 32.0, 2),
    (7.5, 6.0, 32.0, 2),
    (10.0, 7.0, 10.0, 2),
    (10.0, 7.0, 30.0, 2),
    (10.0, 7.0, 30.0, 2),
    (10.0, 7.0, 14.0, 2),
    (10.0, 7.0, 10.0, 2),
    (10.0, 7.0, 12.0, 2),
    (10.0, 8.0, 12.0, 2),
    (10.0, 8.0, 28.0, 2),
    (10.0, 8.0, 42.0, 2),
    (10.0, 8.0, 4.0, 2),
    (10.0, 8.0, 10.0, 2),
    (10.0, 8.0, 22.0, 2),
    (10.0, 9.0, 6.0, 2),
    (10.0, 9.0, 38.0, 2),
    (2.5, 2.0, 18.0, 1),
    (2.5, 2.0, 24.0, 1),
    (2.5, 9.0, 18.0, 1),
    (2.5, 9.0, 28.0, 1),
    (2.5, 9.0, 20.0, 1),
    (2.5, 4.0, 14.0, 1),
    (2.5, 4.0, 36.0, 1),
    (2.5, 4.0, 26.0, 1),
    (2.5, 3.0, 22.0, 1),
    (2.5, 3.0, 42.0, 1),
    (2.5, 3.0, 32.0, 1),
    (2.5, 3.0, 16.0, 1),
    (2.5, 3.0, 38.0, 1),
    (2.5, 9.0, 2.0, 1),
    (2.5, 9.0, 8.0, 1),
    (2.5, 9.0, 26.0, 1),
    (2.5, 9.0, 42.0, 1),
    (5.0, 2.0, 2.0, 1),
    (5.0, 2.0, 14.0, 1),
    (5.0, 2.0, 8.0, 1),
    (5.0, 1.0, 20.0, 1),
    (5.0, 1.0, 32.0, 1),
    (5.0, 3.0, 48.0, 1),
    (5.0, 0.0, 42.0, 1),
    (7.5, 3.0, 2.0, 1),
    (7.5, 3.0, 36.0, 1),
    (7.5, 5.0, 32.0, 1),
    (7.5, 5.0, 20.0, 1),
    (7.5, 5.0, 34.0, 1),
    (7.5, 0.0, 6.0, 1),
    (7.5, 0.0, 12.0, 1),
    (7.5, 8.0, 48.0, 1),
    (7.5, 8.0, 32.0, 1),
    (7.5, 8.0, 4.0, 1),
    (10.0, 5.0, 22.0, 1),
    (10.0, 5.0, 18.0, 1),
    (10.0, 5.0, 46.0, 1),
    (10.0, 5.0, 12.0, 1),
    (10.0, 5.0, 30.0, 1),
    (10.0, 7.0, 36.0, 1),
    (10.0, 7.0, 30.0, 1),
    (10.0, 7.0, 20.0, 1),
    (10.0, 7.0, 38.0, 1),
    (10.0, 7.0, 20.0, 1),
    (10.0, 1.0, 18.0, 1),
    (10.0, 1.0, 10.0, 1),
    (10.0, 1.0, 18.0, 1),
    (10.0, 1.0, 20.0, 1),
    (10.0, 0.0, 12.0, 1),
    (10.0, 0.0, 46.0, 1),
    (10.0, 0.0, 38.0, 1),
    (2.5, 7.0, 40.0, 10),
    (2.5, 7.0, 22.0, 10),
    (2.5, 4.0, 12.0, 10),
    (2.5, 4.0, 32.0, 10),
    (2.5, 4.0, 36.0, 10),
    (2.5, 0.0, 20.0, 10),
    (2.5, 0.0, 30.0, 10),
    (2.5, 3.0, 40.0, 10),
    (2.5, 3.0, 10.0, 10),
    (2.5, 8.0, 42.0, 10),
    (2.5, 8.0, 4.0, 10),
    (2.5, 8.0, 44.0, 10),
    (2.5, 8.0, 32.0, 10),
    (2.5, 8.0, 24.0, 10),
    (5.0, 9.0, 42.0, 10),
    (5.0, 9.0, 18.0, 10),
    (5.0, 9.0, 2.0, 10),
    (5.0, 7.0, 46.0, 10),
    (5.0, 7.0, 42.0, 10),
    (5.0, 7.0, 34.0, 10),
    (5.0, 0.0, 46.0, 10),
    (5.0, 0.0, 8.0, 10),
    (5.0, 5.0, 28.0, 10),
    (5.0, 1.0, 4.0, 10),
    (5.0, 1.0, 10.0, 10),
    (5.0, 1.0, 26.0, 10),
    (7.5, 3.0, 26.0, 10),
    (7.5, 3.0, 42.0, 10),
    (7.5, 3.0, 36.0, 10),
    (7.5, 0.0, 16.0, 10),
    (7.5, 0.0, 40.0, 10),
    (7.5, 2.0, 4.0, 10),
    (7.5, 2.0, 14.0, 10),
    (7.5, 2.0, 46.0, 10),
    (7.5, 8.0, 38.0, 10),
    (7.5, 8.0, 6.0, 10),
    (7.5, 8.0, 24.0, 10),
    (7.5, 8.0, 20.0, 10),
    (7.5, 0.0, 48.0, 10),
    (7.5, 0.0, 20.0, 10),
    (7.5, 0.0, 46.0, 10),
    (7.5, 0.0, 38.0, 10),
    (10.0, 2.0, 32.0, 10),
    (10.0, 2.0, 10.0, 10),
    (10.0, 2.0, 30.0, 10),
    (10.0, 8.0, 14.0, 10),
    (10.0, 8.0, 24.0, 10),
    (10.0, 8.0, 44.0, 10),
    (10.0, 9.0, 28.0, 10),
    (10.0, 9.0, 36.0, 10),
    (10.0, 9.0, 12.0, 10),
    (10.0, 6.0, 20.0, 10),
    (10.0, 6.0, 46.0, 10),
    (10.0, 6.0, 20.0, 10),
    (10.0, 6.0, 28.0, 10),
    (10.0, 6.0, 16.0, 10),
    (10.0, 6.0, 44.0, 10),
    (10.0, 6.0, 28.0, 10),
    (2.5, 6.0, 6.0, 9),
    (2.5, 5.0, 24.0, 9),
    (2.5, 5.0, 6.0, 9),
    (2.5, 2.0, 42.0, 9),
    (2.5, 2.0, 24.0, 9),
    (2.5, 2.0, 36.0, 9),
    (2.5, 2.0, 42.0, 9),
    (2.5, 2.0, 16.0, 9),
    (2.5, 2.0, 22.0, 9),
    (2.5, 2.0, 26.0, 9),
    (2.5, 2.0, 36.0, 9),
    (2.5, 8.0, 30.0, 9),
    (2.5, 8.0, 6.0, 9),
    (5.0, 9.0, 6.0, 9),
    (5.0, 9.0, 22.0, 9),
    (5.0, 9.0, 42.0, 9),
    (5.0, 1.0, 10.0, 9),
    (5.0, 2.0, 32.0, 9),
    (5.0, 2.0, 28.0, 9),
    (5.0, 0.0, 34.0, 9),
    (5.0, 0.0, 22.0, 9),
    (5.0, 4.0, 2.0, 9),
    (5.0, 4.0, 2.0, 9),
    (5.0, 4.0, 4.0, 9),
    (7.5, 5.0, 6.0, 9),
    (7.5, 5.0, 28.0, 9),
    (7.5, 3.0, 2.0, 9),
    (7.5, 3.0, 34.0, 9),
    (7.5, 3.0, 8.0, 9),
    (7.5, 7.0, 46.0, 9),
    (7.5, 9.0, 34.0, 9),
    (7.5, 9.0, 44.0, 9),
    (7.5, 4.0, 10.0, 9),
    (7.5, 4.0, 10.0, 9),
    (10.0, 4.0, 16.0, 9),
    (10.0, 4.0, 4.0, 9),
    (10.0, 1.0, 44.0, 9),
    (10.0, 1.0, 16.0, 9),
    (10.0, 1.0, 30.0, 9),
    (10.0, 1.0, 44.0, 9),
    (10.0, 3.0, 4.0, 9),
    (10.0, 3.0, 46.0, 9),
    (10.0, 0.0, 14.0, 9),
    (2.5, 1.0, 4.0, 8),
    (2.5, 1.0, 18.0, 8),
    (2.5, 1.0, 8.0, 8),
    (2.5, 3.0, 32.0, 8),
    (2.5, 3.0, 28.0, 8),
    (2.5, 3.0, 46.0, 8),
    (2.5, 3.0, 12.0, 8),
    (2.5, 3.0, 18.0, 8),
    (2.5, 0.0, 34.0, 8),
    (2.5, 0.0, 22.0, 8),
    (2.5, 2.0, 22.0, 8),
    (2.5, 2.0, 14.0, 8),
    (2.5, 2.0, 42.0, 8),
    (2.5, 0.0, 16.0, 8),
    (5.0, 4.0, 24.0, 8),
    (5.0, 4.0, 26.0, 8),
    (5.0, 0.0, 26.0, 8),
    (5.0, 2.0, 44.0, 8),
    (5.0, 2.0, 38.0, 8),
    (5.0, 2.0, 48.0, 8),
    (5.0, 2.0, 26.0, 8),
    (5.0, 2.0, 6.0, 8),
    (5.0, 4.0, 12.0, 8),
    (5.0, 9.0, 36.0, 8),
    (5.0, 9.0, 48.0, 8),
    (5.0, 9.0, 16.0, 8),
    (5.0, 9.0, 6.0, 8),
    (7.5, 0.0, 34.0, 8),
    (7.5, 6.0, 10.0, 8),
    (7.5, 6.0, 12.0, 8),
    (7.5, 6.0, 10.0, 8),
    (7.5, 9.0, 26.0, 8),
    (7.5, 9.0, 6.0, 8),
    (7.5, 9.0, 40.0, 8),
    (7.5, 9.0, 18.0, 8),
    (7.5, 4.0, 20.0, 8),
    (7.5, 4.0, 42.0, 8),
    (7.5, 4.0, 16.0, 8),
    (10.0, 5.0, 24.0, 8),
    (10.0, 5.0, 44.0, 8),
    (10.0, 5.0, 20.0, 8),
    (10.0, 5.0, 22.0, 8),
    (10.0, 3.0, 16.0, 8),
    (10.0, 3.0, 14.0, 8),
    (10.0, 3.0, 6.0, 8),
    (10.0, 2.0, 24.0, 8),
    (10.0, 2.0, 36.0, 8),
    (10.0, 2.0, 46.0, 8),
    (10.0, 0.0, 42.0, 8),
    (10.0, 0.0, 26.0, 8),
    (10.0, 0.0, 42.0, 8),
    (10.0, 0.0, 10.0, 8),
    (10.0, 9.0, 12.0, 8),
    (10.0, 9.0, 8.0, 8),
    (2.5, 4.0, 42.0, 7),
    (2.5, 4.0, 14.0, 7),
    (2.5, 4.0, 46.0, 7),
    (2.5, 4.0, 18.0, 7),
    (2.5, 2.0, 2.0, 7),
    (2.5, 2.0, 38.0, 7),
    (2.5, 2.0, 14.0, 7),
    (2.5, 8.0, 26.0, 7),
    (2.5, 7.0, 12.0, 7),
    (2.5, 7.0, 46.0, 7),
    (2.5, 1.0, 44.0, 7),
    (5.0, 8.0, 26.0, 7),
    (5.0, 0.0, 46.0, 7),
    (5.0, 9.0, 44.0, 7),
    (5.0, 9.0, 16.0, 7),
    (5.0, 9.0, 40.0, 7),
    (5.0, 5.0, 14.0, 7),
    (5.0, 7.0, 6.0, 7),
    (5.0, 7.0, 30.0, 7),
    (7.5, 1.0, 16.0, 7),
    (7.5, 1.0, 18.0, 7),
    (7.5, 1.0, 30.0, 7),
    (7.5, 1.0, 4.0, 7),
    (7.5, 1.0, 10.0, 7),
    (7.5, 1.0, 40.0, 7),
    (7.5, 1.0, 18.0, 7),
    (7.5, 3.0, 14.0, 7),
    (7.5, 3.0, 48.0, 7),
    (7.5, 3.0, 48.0, 7),
    (7.5, 3.0, 6.0, 7),
    (7.5, 0.0, 36.0, 7),
    (7.5, 0.0, 42.0, 7),
    (7.5, 0.0, 22.0, 7),
    (10.0, 9.0, 8.0, 7),
    (10.0, 9.0, 8.0, 7),
    (10.0, 7.0, 36.0, 7),
    (10.0, 7.0, 46.0, 7),
    (10.0, 7.0, 20.0, 7),
    (10.0, 7.0, 14.0, 7),
    (10.0, 4.0, 32.0, 7),
    (10.0, 4.0, 16.0, 7),
    (10.0, 1.0, 24.0, 7),
    (10.0, 1.0, 40.0, 7),
    (10.0, 1.0, 18.0, 7),
    (10.0, 1.0, 24.0, 7),
    (10.0, 7.0, 30.0, 7),
    (10.0, 7.0, 30.0, 7),
    (2.5, 5.0, 24.0, 6),
    (2.5, 5.0, 42.0, 6),
    (2.5, 8.0, 34.0, 6),
    (2.5, 8.0, 34.0, 6),
    (2.5, 8.0, 48.0, 6),
    (2.5, 8.0, 26.0, 6),
    (2.5, 3.0, 28.0, 6),
    (5.0, 4.0, 44.0, 6),
    (5.0, 4.0, 46.0, 6),
    (5.0, 6.0, 28.0, 6),
    (5.0, 6.0, 46.0, 6),
    (5.0, 2.0, 10.0, 6),
    (5.0, 2.0, 4.0, 6),
    (5.0, 2.0, 34.0, 6),
    (5.0, 4.0, 46.0, 6),
    (5.0, 4.0, 20.0, 6),
    (7.5, 7.0, 2.0, 6),
    (7.5, 6.0, 46.0, 6),
    (7.5, 6.0, 48.0, 6),
    (7.5, 6.0, 36.0, 6),
    (7.5, 6.0, 42.0, 6),
    (10.0, 7.0, 8.0, 6),
    (10.0, 8.0, 18.0, 6),
    (10.0, 8.0, 42.0, 6),
    (10.0, 8.0, 32.0, 6),
    (10.0, 2.0, 22.0, 6),
    (10.0, 2.0, 28.0, 6),
    (10.0, 2.0, 6.0, 6),
    (10.0, 2.0, 36.0, 6),
    (10.0, 2.0, 48.0, 6),
    (10.0, 2.0, 28.0, 6),
    (10.0, 2.0, 36.0, 6),
    (10.0, 2.0, 6.0, 6),
    (10.0, 9.0, 24.0, 6),
    (2.5, 6.0, 42.0, 5),
    (2.5, 6.0, 8.0, 5),
    (2.5, 7.0, 20.0, 5),
    (2.5, 7.0, 4.0, 5),
    (2.5, 7.0, 16.0, 5),
    (2.5, 5.0, 34.0, 5),
    (2.5, 5.0, 22.0, 5),
    (2.5, 2.0, 26.0, 5),
    (2.5, 2.0, 38.0, 5),
    (2.5, 2.0, 30.0, 5),
    (5.0, 9.0, 42.0, 5),
    (5.0, 9.0, 20.0, 5),
    (5.0, 9.0, 32.0, 5),
    (5.0, 6.0, 46.0, 5),
    (5.0, 6.0, 14.0, 5),
    (5.0, 0.0, 38.0, 5),
    (5.0, 0.0, 16.0, 5),
    (5.0, 4.0, 16.0, 5),
    (5.0, 4.0, 42.0, 5),
    (5.0, 4.0, 8.0, 5),
    (5.0, 9.0, 2.0, 5),
    (5.0, 9.0, 26.0, 5),
    (7.5, 4.0, 20.0, 5),
    (7.5, 4.0, 6.0, 5),
    (7.5, 9.0, 6.0, 5),
    (7.5, 9.0, 48.0, 5),
    (7.5, 9.0, 46.0, 5),
    (7.5, 9.0, 36.0, 5),
    (7.5, 6.0, 40.0, 5),
    (7.5, 3.0, 28.0, 5),
    (7.5, 3.0, 24.0, 5),
    (7.5, 7.0, 14.0, 5),
    (7.5, 7.0, 26.0, 5),
    (7.5, 7.0, 48.0, 5),
    (7.5, 7.0, 16.0, 5),
    (10.0, 4.0, 42.0, 5),
    (10.0, 9.0, 42.0, 5),
    (10.0, 9.0, 6.0, 5),
    (10.0, 9.0, 12.0, 5),
    (10.0, 0.0, 16.0, 5),
    (10.0, 0.0, 14.0, 5),
    (10.0, 8.0, 28.0, 5),
    (10.0, 8.0, 12.0, 5),
    (10.0, 8.0, 34.0, 5),
    (10.0, 6.0, 22.0, 5),
    (10.0, 6.0, 44.0, 5),
    (10.0, 6.0, 32.0, 5))

MUNSELL_BOUNDING_HUES = (
    ((2.5, 4), (2.5, 4)),
    ((2.5, 4), (2.5, 4)),
    ((2.5, 4), (2.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((5.0, 4), (7.5, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((10.0, 4), (10.0, 4)),
    ((7.5, 4), (10, 4)),
    ((7.5, 4), (10, 4)),
    ((10.0, 4), (10.0, 4)),
    ((10.0, 4), (10.0, 4)),
    ((10.0, 4), (10.0, 4)),
    ((10.0, 4), (10.0, 4)),
    ((10.0, 4), (10.0, 4)),
    ((7.5, 4), (10, 4)),
    ((10.0, 4), (10.0, 4)),
    ((7.5, 4), (10, 4)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (2.5, 3)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (5.0, 3)),
    ((2.5, 3), (5.0, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((5.0, 3), (7.5, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((10.0, 3), (10.0, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((7.5, 3), (10, 3)),
    ((10.0, 3), (10.0, 3)),
    ((10.0, 3), (10.0, 3)),
    ((7.5, 3), (10, 3)),
    ((10.0, 3), (10.0, 3)),
    ((10.0, 3), (10.0, 3)),
    ((10.0, 3), (10.0, 3)),
    ((2.5, 2), (2.5, 2)),
    ((2.5, 2), (2.5, 2)),
    ((2.5, 2), (5.0, 2)),
    ((2.5, 2), (5.0, 2)),
    ((2.5, 2), (2.5, 2)),
    ((2.5, 2), (5.0, 2)),
    ((2.5, 2), (5.0, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((5.0, 2), (7.5, 2)),
    ((7.5, 2), (10, 2)),
    ((7.5, 2), (10, 2)),
    ((10.0, 2), (10.0, 2)),
    ((10.0, 2), (10.0, 2)),
    ((7.5, 2), (10, 2)),
    ((7.5, 2), (10, 2)),
    ((2.5, 1), (5.0, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (5.0, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (5.0, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (5.0, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (2.5, 1)),
    ((2.5, 1), (5.0, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((5.0, 1), (7.5, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((7.5, 1), (10, 1)),
    ((10.0, 1), (10.0, 1)),
    ((7.5, 1), (10, 1)),
    ((10.0, 1), (10.0, 1)),
    ((10.0, 1), (10.0, 1)),
    ((2.5, 10), (5.0, 10)),
    ((2.5, 10), (5.0, 10)),
    ((2.5, 10), (2.5, 10)),
    ((2.5, 10), (2.5, 10)),
    ((2.5, 10), (2.5, 10)),
    ((2.5, 10), (2.5, 10)),
    ((2.5, 10), (2.5, 10)),
    ((2.5, 10), (2.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((5.0, 10), (7.5, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((10.0, 10), (10.0, 10)),
    ((7.5, 10), (10, 10)),
    ((10.0, 10), (10.0, 10)),
    ((10.0, 10), (10.0, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((10.0, 10), (10.0, 10)),
    ((7.5, 10), (10, 10)),
    ((7.5, 10), (10, 10)),
    ((10.0, 10), (10.0, 10)),
    ((10.0, 10), (10.0, 10)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (5.0, 9)),
    ((2.5, 9), (5.0, 9)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (5.0, 9)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (5.0, 9)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (2.5, 9)),
    ((2.5, 9), (5.0, 9)),
    ((2.5, 9), (5.0, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((5.0, 9), (7.5, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((10.0, 9), (10.0, 9)),
    ((10.0, 9), (10.0, 9)),
    ((10.0, 9), (10.0, 9)),
    ((10.0, 9), (10.0, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((7.5, 9), (10, 9)),
    ((10.0, 9), (10.0, 9)),
    ((7.5, 9), (10, 9)),
    ((10.0, 9), (10.0, 9)),
    ((10.0, 9), (10.0, 9)),
    ((2.5, 8), (2.5, 8)),
    ((2.5, 8), (5.0, 8)),
    ((2.5, 8), (5.0, 8)),
    ((2.5, 8), (5.0, 8)),
    ((2.5, 8), (2.5, 8)),
    ((2.5, 8), (5.0, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((5.0, 8), (7.5, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((10.0, 8), (10.0, 8)),
    ((7.5, 8), (10, 8)),
    ((10.0, 8), (10.0, 8)),
    ((7.5, 8), (10, 8)),
    ((7.5, 8), (10, 8)),
    ((10.0, 8), (10.0, 8)),
    ((10.0, 8), (10.0, 8)),
    ((10.0, 8), (10.0, 8)),
    ((7.5, 8), (10, 8)),
    ((2.5, 7), (5.0, 7)),
    ((2.5, 7), (5.0, 7)),
    ((2.5, 7), (2.5, 7)),
    ((2.5, 7), (5.0, 7)),
    ((2.5, 7), (2.5, 7)),
    ((2.5, 7), (5.0, 7)),
    ((2.5, 7), (2.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((5.0, 7), (7.5, 7)),
    ((7.5, 7), (10, 7)),
    ((7.5, 7), (10, 7)),
    ((7.5, 7), (10, 7)),
    ((7.5, 7), (10, 7)),
    ((7.5, 7), (10, 7)),
    ((7.5, 7), (10, 7)),
    ((10.0, 7), (10.0, 7)),
    ((7.5, 7), (10, 7)),
    ((10.0, 7), (10.0, 7)),
    ((10.0, 7), (10.0, 7)),
    ((10.0, 7), (10.0, 7)),
    ((10.0, 7), (10.0, 7)),
    ((10.0, 7), (10.0, 7)),
    ((2.5, 6), (5.0, 6)),
    ((2.5, 6), (5.0, 6)),
    ((2.5, 6), (2.5, 6)),
    ((2.5, 6), (2.5, 6)),
    ((2.5, 6), (2.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((5.0, 6), (7.5, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((10.0, 6), (10.0, 6)),
    ((10.0, 6), (10.0, 6)),
    ((10.0, 6), (10.0, 6)),
    ((10.0, 6), (10.0, 6)),
    ((10.0, 6), (10.0, 6)),
    ((10.0, 6), (10.0, 6)),
    ((7.5, 6), (10, 6)),
    ((10.0, 6), (10.0, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((7.5, 6), (10, 6)),
    ((2.5, 5), (5.0, 5)),
    ((2.5, 5), (2.5, 5)),
    ((2.5, 5), (2.5, 5)),
    ((2.5, 5), (5.0, 5)),
    ((2.5, 5), (2.5, 5)),
    ((2.5, 5), (2.5, 5)),
    ((2.5, 5), (2.5, 5)),
    ((2.5, 5), (5.0, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((5.0, 5), (7.5, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)),
    ((7.5, 5), (10, 5)))

MUNSELL_HUE_TO_ANGLE = (
    (2.5, 1, 208.75),
    (2.5, 2, 153.75),
    (2.5, 3, 118.75),
    (2.5, 4, 63.75),
    (2.5, 5, 39.375),
    (2.5, 6, 16.875),
    (2.5, 7, 348.75),
    (2.5, 8, 300.0),
    (2.5, 9, 251.25),
    (2.5, 10, 236.25),
    (5.0, 1, 225.0),
    (5.0, 2, 160.0),
    (5.0, 3, 135.0),
    (5.0, 4, 70.0),
    (5.0, 5, 45.0),
    (5.0, 6, 22.5),
    (5.0, 7, 0.0),
    (5.0, 8, 315.0),
    (5.0, 9, 255.0),
    (5.0, 10, 240.0),
    (7.5, 1, 228.75),
    (7.5, 2, 176.25),
    (7.5, 3, 141.25),
    (7.5, 4, 86.25),
    (7.5, 5, 51.25),
    (7.5, 6, 28.125),
    (7.5, 7, 5.625),
    (7.5, 8, 326.25),
    (7.5, 9, 270.0),
    (7.5, 10, 243.75),
    (10.0, 1, 232.5),
    (10.0, 2, 192.5),
    (10.0, 3, 147.5),
    (10.0, 4, 102.5),
    (10.0, 5, 57.5),
    (10.0, 6, 33.75),
    (10.0, 7, 11.25),
    (10.0, 8, 337.5),
    (10.0, 9, 285.0),
    (10.0, 10, 247.5))

MUNSELL_HUE_TO_ASTM_HUE = (
    (2.5, 0, 72.5),
    (2.5, 1, 62.5),
    (2.5, 2, 52.5),
    (2.5, 3, 42.5),
    (2.5, 4, 32.5),
    (2.5, 5, 22.5),
    (2.5, 6, 12.5),
    (2.5, 7, 2.5),
    (2.5, 8, 92.5),
    (2.5, 9, 82.5),
    (2.5, 10, 72.5),
    (5.0, 0, 75.0),
    (5.0, 1, 65.0),
    (5.0, 2, 55.0),
    (5.0, 3, 45.0),
    (5.0, 4, 35.0),
    (5.0, 5, 25.0),
    (5.0, 6, 15.0),
    (5.0, 7, 5.0),
    (5.0, 8, 95.0),
    (5.0, 9, 85.0),
    (5.0, 10, 75.0),
    (7.5, 0, 77.5),
    (7.5, 1, 67.5),
    (7.5, 2, 57.5),
    (7.5, 3, 47.5),
    (7.5, 4, 37.5),
    (7.5, 5, 27.5),
    (7.5, 6, 17.5),
    (7.5, 7, 7.5),
    (7.5, 8, 97.5),
    (7.5, 9, 87.5),
    (7.5, 10, 77.5),
    (10.0, 0, 80.0),
    (10.0, 1, 70.0),
    (10.0, 2, 60.0),
    (10.0, 3, 50.0),
    (10.0, 4, 40.0),
    (10.0, 5, 30.0),
    (10.0, 6, 20.0),
    (10.0, 7, 10.0),
    (10.0, 8, 100.0),
    (10.0, 9, 90.0),
    (10.0, 10, 80.0))

MUNSELL_INTERPOLATION_METHODS = (
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    None,
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    None,
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    None,
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    None,
    None,
    'Radial',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Radial',
    'Radial',
    'Linear',
    'Linear',
    'Linear',
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    None,
    None,
    'Linear',
    'Radial',
    'Linear',
    'Radial',
    'Radial',
    'Radial')

MUNSELL_XY_FROM_RENOTATION_OVOID = (
    (0.4333, 0.5602),
    None,
    None,
    None,
    None,
    None,
    (0.3799, 0.447),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.3284, 0.3559),
    (0.3722, 0.4669),
    None,
    None,
    (0.274, 0.879),
    None,
    None,
    (0.3395, 0.5913),
    (0.303, 0.809),
    None,
    (0.345, 0.5949),
    None,
    None,
    (0.345, 0.5949),
    None,
    (0.202, 0.807),
    None,
    None,
    None,
    None,
    (0.168, 0.88),
    (0.3123, 0.4732),
    None,
    (0.3092, 0.5095),
    None,
    (0.3128, 0.4175),
    None,
    (0.149, 0.681),
    (0.1689, 0.6549),
    None,
    (0.206, 0.619),
    None,
    None,
    (0.163, 0.67),
    (0.163, 0.67),
    (0.2952, 0.3851),
    (0.069, 0.764),
    (0.2574, 0.4814),
    (0.123, 0.546),
    (0.1397, 0.5312),
    None,
    (0.2554, 0.4087),
    (0.2466, 0.4181),
    None,
    (0.2833, 0.3564),
    None,
    None,
    (0.1516, 0.4505),
    (0.1303, 0.4858),
    (0.1841, 0.4448),
    None,
    (0.1688, 0.457),
    (0.1982, 0.433),
    None,
    None,
    (0.1262, 0.4667),
    None,
    (0.1022, 0.4759),
    (0.1842, 0.4244),
    (0.22, 0.3983),
    (0.034, 0.546),
    (0.2171, 0.4138),
    (0.1398, 0.4168),
    None,
    (0.291, 0.331),
    (0.069, 0.4542),
    None,
    None,
    (0.1551, 0.4208),
    None,
    (0.0925, 0.4275),
    None,
    None,
    (0.0333, 0.4444),
    (0.2957, 0.3293),
    (0.243, 0.371),
    (0.2282, 0.3811),
    (0.1866, 0.4086),
    None,
    (0.294, 0.3268),
    None,
    None,
    None,
    None,
    (0.0636, 0.3788),
    None,
    None,
    None,
    (0.26, 0.3289),
    None,
    None,
    (0.0781, 0.3211),
    None,
    (0.067, 0.32),
    None,
    None,
    None,
    (0.25, 0.3141),
    None,
    None,
    None,
    (0.122, 0.351),
    None,
    None,
    (0.2234, 0.315),
    None,
    None,
    None,
    None,
    (0.2768, 0.3287),
    None,
    (0.2094, 0.3165),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.068, 0.283),
    None,
    (0.092, 0.29),
    (0.1961, 0.311),
    None,
    None,
    (0.2035, 0.2956),
    None,
    None,
    (0.1671, 0.2832),
    (0.2035, 0.2956),
    (0.1841, 0.2892),
    (0.1937, 0.2978),
    None,
    None,
    (0.2686, 0.313),
    (0.212, 0.3025),
    (0.118, 0.273),
    (0.2501, 0.3118),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1027, 0.2057),
    None,
    None,
    None,
    None,
    None,
    (0.065, 0.17),
    None,
    (0.2909, 0.3125),
    (0.228, 0.296),
    None,
    None,
    (0.2559, 0.2874),
    None,
    (0.1245, 0.1827),
    None,
    None,
    None,
    None,
    (0.2616, 0.2857),
    None,
    None,
    (0.098, 0.146),
    None,
    None,
    None,
    None,
    None,
    (0.2688, 0.2956),
    (0.096, 0.126),
    (0.1203, 0.1505),
    None,
    (0.1666, 0.1964),
    None,
    None,
    None,
    (0.128, 0.162),
    None,
    (0.128, 0.162),
    None,
    (0.084, 0.094),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1634, 0.1698),
    None,
    None,
    None,
    None,
    None,
    (0.1576, 0.16),
    None,
    (0.2758, 0.2879),
    None,
    None,
    None,
    None,
    None,
    (0.2991, 0.3057),
    None,
    None,
    None,
    None,
    None,
    (0.109, 0.079),
    (0.2012, 0.1867),
    (0.1285, 0.087),
    (0.095, 0.027),
    (0.1642, 0.0655),
    (0.157, 0.034),
    (0.159, 0.044),
    None,
    None,
    (0.242, 0.2148),
    (0.1762, 0.0955),
    (0.161, 0.016),
    None,
    (0.2702, 0.2648),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.1918, 0.0379),
    (0.22, 0.133),
    (0.1925, 0.042),
    (0.24, 0.196),
    None,
    None,
    None,
    None,
    None,
    (0.214, 0.143),
    None,
    (0.214, 0.143),
    (0.194, 0.101),
    (0.2265, 0.1671),
    None,
    (0.194, 0.101),
    (0.2842, 0.255),
    (0.2372, 0.1223),
    (0.2806, 0.2444),
    (0.218, 0.022),
    (0.2277, 0.0621),
    (0.22, 0.031),
    (0.218, 0.022),
    (0.2372, 0.098),
    (0.2298, 0.0696),
    (0.226, 0.0555),
    (0.22, 0.031),
    None,
    (0.2881, 0.2671),
    (0.296, 0.271),
    None,
    None,
    (0.2701, 0.1178),
    (0.254, 0.039),
    (0.2559, 0.0525),
    None,
    None,
    (0.3022, 0.2825),
    (0.3022, 0.2825),
    (0.2958, 0.2565),
    (0.3093, 0.2555),
    (0.3018, 0.1253),
    (0.3088, 0.274),
    (0.291, 0.06),
    (0.3037, 0.1981),
    None,
    None,
    None,
    (0.3056, 0.206),
    (0.3056, 0.206),
    (0.337, 0.1756),
    (0.321, 0.2686),
    None,
    (0.3078, 0.0839),
    None,
    None,
    (0.3214, 0.2517),
    None,
    None,
    (0.329, 0.2095),
    (0.337, 0.08),
    (0.3342, 0.1551),
    None,
    (0.414, 0.102),
    None,
    (0.3754, 0.1898),
    (0.3929, 0.1506),
    None,
    None,
    (0.383, 0.096),
    (0.3711, 0.1449),
    None,
    None,
    (0.473, 0.172),
    (0.482, 0.162),
    None,
    None,
    None,
    None,
    None,
    (0.3708, 0.238),
    (0.4104, 0.2361),
    None,
    None,
    (0.4, 0.263),
    (0.3431, 0.2988),
    None,
    (0.396, 0.286),
    (0.4125, 0.2784),
    (0.396, 0.286),
    None,
    (0.3512, 0.3052),
    None,
    None,
    (0.513, 0.2101),
    None,
    (0.4799, 0.2329),
    (0.57, 0.24),
    None,
    (0.5396, 0.2535),
    (0.554, 0.246),
    (0.5628, 0.2241),
    (0.538, 0.2369),
    (0.4218, 0.2864),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.414, 0.302),
    (0.376, 0.31),
    None,
    (0.5369, 0.281),
    None,
    (0.5898, 0.2622),
    (0.3614, 0.3033),
    None,
    (0.5734, 0.2083),
    None,
    (0.4435, 0.3119),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.5341, 0.3158),
    (0.3805, 0.3244),
    None,
    None,
    None,
    None,
    (0.466, 0.2888),
    (0.6111, 0.229),
    None,
    None,
    (0.6492, 0.3012),
    None,
    None,
    (0.4738, 0.3316),
    None,
    None,
    None,
    (0.416, 0.35),
    (0.416, 0.35),
    None,
    None,
    (0.592, 0.374),
    (0.5234, 0.37),
    None,
    (0.6409, 0.3533),
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    None,
    (0.602, 0.405),
    None,
    None,
    None,
    None,
    None,
    (0.684, 0.415),
    (0.4674, 0.3738),
    None,
    None,
    None,
    (0.3437, 0.3397),
    None,
    None,
    None,
    None,
    (0.4399, 0.4164),
    (0.5179, 0.467),
    None,
    None,
    None,
    None,
    (0.545, 0.458),
    None,
    None,
    None,
    None,
    (0.545, 0.458),
    (0.532, 0.478),
    None,
    (0.4517, 0.4421),
    (0.516, 0.492),
    (0.3761, 0.38),
    (0.5049, 0.4843),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.483, 0.5092),
    None,
    None,
    (0.4905, 0.5038),
    None,
    None,
    None,
    None,
    (0.4745, 0.481),
    (0.3378, 0.3504),
    None,
    None,
    (0.4331, 0.4688),
    (0.3811, 0.4123),
    None,
    None,
    None,
    None,
    None,
    None,
    (0.4652, 0.5128),
    None,
    None,
    (0.4728, 0.5215),
    None,
    None,
    (0.3761, 0.4155),
    (0.4271, 0.492),
    None,
    None,
    None,
    (0.4341, 0.502),
    None,
    None,
    None,
    None)

MUNSELL_SPECIFICATIONS_TO_XY = (
    ((2.5, 8.0, 11.928546308350969, 4),
     (0.41492483295053395, 0.5123568112702328)),
    ((2.5, 6.0, 6.923610826208884, 4),
     (0.38945937205126197, 0.46616492464383436)),
    ((3.0588107073010358, 6.0, 14.50196667770457, 4),
     (0.42971427832243203, 0.5665812104155267)),
    ((2.5, 5.0, 2.0, 4), (0.3352, 0.3636)),
    ((5.613007062442384, 8.0, 18.56590894044391, 4),
     (0.39909107942315, 0.5888839244592698)),
    ((5.845640071004907, 8.0, 5.782325614552295, 4),
     (0.3476544320849577, 0.415451228906664)),
    ((5.780794121059599, 3.0, 3.3492086825591487, 4),
     (0.3397413408138117, 0.4168925891450931)),
    ((5.483684299639117, 4.0, 5.761459062506715, 4),
     (0.3624285934444264, 0.4768182836607108)),
    ((5.809580308813496, 6.0, 6.662613753958899, 4),
     (0.35692510294957597, 0.4554043675955618)),
    ((5.209252955662903, 3.0, 5.141472643810014, 4),
     (0.36331762907025356, 0.4808425142327246)),
    ((7.706105853911573, 3.0, 11.396648897274897, 4),
     (0.3131610795605752, 0.693666965393307)),
    ((8.117640564564343, 3.0, 3.1653563832640272, 4),
     (0.3191717580108967, 0.3988687571291857)),
    ((7.8731203012311255, 3.0, 13.241107969297714, 4),
     (0.2945431122633574, 0.7584911897992045)),
    ((8.04983322214289, 2.0, 7.501924679081063, 4),
     (0.3074690943666493, 0.6094306766573039)),
    ((8.355307569391062, 3.0, 11.925441344336392, 4),
     (0.2925864520380243, 0.7013043294887223)),
    ((8.342795760577609, 1.0, 6.298818145909256, 4),
     (0.2563315496261501, 0.7193319941337727)),
    ((7.5947244020062845, 2.0, 4.626613135331287, 4),
     (0.32400993792893495, 0.47353910662213033)),
    ((8.19517786608579, 9.0, 23.571122010181508, 4),
     (0.3393737554129363, 0.6530759138839242)),
    ((7.754763634912469, 8.0, 21.00944901061068, 4),
     (0.3524522658493676, 0.6357582251092715)),
    ((9.010231962978862, 6.0, 6.803370568930175, 4),
     (0.3215881688765898, 0.4404084623233154)),
    ((9.041566851651622, 6.0, 17.010037203566448, 4),
     (0.3077087669482304, 0.6419582684061792)),
    ((9.915652169827913, 9.0, 11.13108215988432, 4),
     (0.3156394420758696, 0.47190649633818943)),
    ((10.0, 9.0, 27.322046186799103, 4),
     (0.27105079215940403, 0.6981866004283883)),
    ((9.961336111598143, 8.0, 13.20009863344056, 4),
     (0.3110050190589885, 0.512285067166623)),
    ((9.887406551063181, 8.0, 2.0660963235598375, 4),
     (0.3124556963039024, 0.3470190461737042)),
    ((10.0, 3.0, 2.5700932200974145, 4),
     (0.30780233686482955, 0.37333504024765457)),
    ((10.0, 3.0, 13.514066607169514, 4),
     (0.23432557407109803, 0.7247335078491779)),
    ((10.0, 5.0, 12.753899774963989, 4),
     (0.29015511114768366, 0.5923266098579271)),
    ((10.0, 6.0, 15.244598276849418, 4),
     (0.29059930775417764, 0.6049052757954609)),
    ((10.0, 5.0, 27.929001019877095, 4),
     (0.1548874872515363, 0.9471895260068659)),
    ((9.757039645743053, 6.0, 3.4112871270786895, 4),
     (0.31331400537253434, 0.37344423041130265)),
    ((10.0, 6.0, 24.86360130658431, 4),
     (0.23238659150720198, 0.7984084248251019)),
    ((9.862075817629322, 4.0, 7.67196809500038, 4),
     (0.3036210954390051, 0.5020744140554851)),
    ((3.2140937198013564, 9.0, 3.4367939376082868, 3),
     (0.30085241356011383, 0.3526678514951116)),
    ((3.1967035260607033, 9.0, 24.78003138905329, 3),
     (0.2002744068603639, 0.577845271609722)),
    ((2.5, 9.0, 27.736581704977635, 3),
     (0.19384392806515655, 0.6398389804597316)),
    ((2.7908763449337677, 8.0, 20.868304564027603, 3),
     (0.22423490392017478, 0.5571230146385515)),
    ((3.221499566897477, 6.0, 5.467726257137659, 3),
     (0.2873858207778558, 0.3848413427300203)),
    ((2.622512070432247, 6.0, 19.364472252973304, 3),
     (0.19637966981582017, 0.5896085735184569)),
    ((3.2873061024849806, 5.0, 19.855724192587914, 3),
     (0.15107222663217493, 0.6036632631469063)),
    ((5.727612405003367, 3.0, 10.746642552166502, 3),
     (0.17920676997174168, 0.4665320350934196)),
    ((5.347955701149093, 3.0, 18.900471815194905, 3),
     (0.07543521561825134, 0.5598764933852344)),
    ((5.7385751713204325, 4.0, 4.223160837759656, 3),
     (0.27341911934324464, 0.3703759727313385)),
    ((5.720824103581511, 2.0, 4.878068159363519, 3),
     (0.2467519504501329, 0.396042965876929)),
    ((5.316780024484356, 1.0, 8.043957606541364, 3),
     (0.05387242044052574, 0.5623155400405534)),
    ((5.7623230008312385, 2.0, 9.507411716255689, 3),
     (0.16257886565663734, 0.4748117643951303)),
    ((5.569555328848931, 2.0, 17.594491934810442, 3),
     (-0.005438870615513399, 0.5989078963083921)),
    ((5.985579505387595, 2.0, 14.803434527189347, 3),
     (0.05088199721408976, 0.5508235640922985)),
    ((5.461619603420755, 3.0, 6.6471547360970025, 3),
     (0.2368393501455289, 0.4149076913619224)),
    ((8.479050960745253, 3.0, 19.932170607445244, 3),
     (0.055960429731629305, 0.4808828600574318)),
    ((7.838277926195208, 3.0, 6.238528025218592, 3),
     (0.23006932476384195, 0.3896583956403046)),
    ((8.2830613968175, 3.0, 10.350825174769154, 3),
     (0.17154862793173067, 0.42348961815670966)),
    ((7.603155032355272, 6.0, 29.139541165198704, 3),
     (0.07552585363483849, 0.5167802829864803)),
    ((8.324115039527976, 7.0, 23.515778973195257, 3),
     (0.1410408872001719, 0.4624726907273586)),
    ((8.44424273124686, 7.0, 2.4130843113046656, 3),
     (0.2935043828079583, 0.3347759364340525)),
    ((8.309061774521076, 6.0, 17.507252134514488, 3),
     (0.16639630860705243, 0.44095922688344563)),
    ((7.698664797242625, 3.0, 18.828802660207376, 3),
     (0.06976476518639417, 0.495578148736382)),
    ((8.14037117068092, 3.0, 14.649933295354042, 3),
     (0.11587116950368091, 0.45887391411549944)),
    ((8.484903553213694, 2.0, 11.879562262633948, 3),
     (0.10118404307076279, 0.4519866737136369)),
    ((8.454109029623016, 2.0, 4.606317173304252, 3),
     (0.23983680211973246, 0.37222196630379506)),
    ((8.305262429168986, 5.0, 3.9045072719017924, 3),
     (0.27629481645975607, 0.3507630813736354)),
    ((8.189730004579287, 5.0, 28.126992759236863, 3),
     (0.05705292734056282, 0.5054724651556731)),
    ((7.54028778107475, 6.0, 6.635319193935916, 3),
     (0.2612474304996075, 0.3719851145582656)),
    ((7.9629991342362985, 5.0, 20.293354805626866, 3),
     (0.11711998926043284, 0.47473573591944374)),
    ((8.432959559038371, 6.0, 26.469970873757067, 3),
     (0.0950374077805397, 0.48534709230317313)),
    ((10.0, 9.0, 6.0469956581432704, 3),
     (0.2699968780049759, 0.35154672720525215)),
    ((9.771353946056914, 9.0, 20.82975271547889, 3),
     (0.1703271759874031, 0.4176338874276043)),
    ((9.376380796522223, 9.0, 13.348522394106682, 3),
     (0.22483106117061774, 0.3904997531995366)),
    ((9.912704179532229, 4.0, 25.778231770351923, 3),
     (0.041682531090741895, 0.45638108279644746)),
    ((10.0, 5.0, 13.712247643370837, 3),
     (0.16970415882749393, 0.4075044010703485)),
    ((10.0, 4.0, 28.587923360931033, 3),
     (0.018590574793017255, 0.4614698084023276)),
    ((9.287535146732925, 4.0, 6.997389565284625, 3),
     (0.22779618119117986, 0.3782363477013876)),
    ((10.0, 6.0, 30.932435068478792, 3),
     (0.059472954520648456, 0.4677973052054364)),
    ((10.0, 5.0, 7.946854746461393, 3),
     (0.23028991231427853, 0.372620011437199)),
    ((10.0, 6.0, 26.172410297895773, 3),
     (0.09297071254878268, 0.45251723089368734)),
    ((2.5, 7.0, 12.643410557057086, 2),
     (0.20473101026501478, 0.3654658906154655)),
    ((2.5, 3.0, 19.167537762557394, 2),
     (0.05510943657077363, 0.3689588995456623)),
    ((3.284581774573411, 4.0, 10.316761862277126, 2),
     (0.16617263015780218, 0.34847110450044866)),
    ((3.0814075494281132, 3.0, 4.031683724514751, 2),
     (0.24102409919891465, 0.3343695993283068)),
    ((2.5342634625499727, 9.0, 14.863663104253218, 2),
     (0.20177805681461625, 0.3722383864015372)),
    ((5.275920564662094, 9.0, 12.879135949769728, 2),
     (0.20397359836905862, 0.342464374531538)),
    ((5.522856449128964, 9.0, 17.412586595686815, 2),
     (0.16690331344259282, 0.3431822968019505)),
    ((5.885914939777947, 9.0, 17.388086814072437, 2),
     (0.16584353646682692, 0.3393430032707385)),
    ((5.956560321967156, 4.0, 4.31169020481439, 2),
     (0.24129709871769092, 0.3181985042671494)),
    ((5.6279111948942635, 5.0, 16.56681914443115, 2),
     (0.11616623038399936, 0.3161181105644866)),
    ((5.8534547245334565, 5.0, 18.83506980508535, 2),
     (0.09538175760437877, 0.3095289993435369)),
    ((5.445581146699364, 5.0, 25.690737024023207, 2),
     (0.05808395140333093, 0.3095707855049915)),
    ((5.144720369630256, 5.0, 18.979172966805407, 2),
     (0.09701692047501353, 0.32135192146478)),
    ((5.2907074463880175, 6.0, 13.598520998056053, 2),
     (0.16894640868927147, 0.3308774206782637)),
    ((5.415844403197766, 6.0, 15.178617464461626, 2),
     (0.15478320533987108, 0.32921474300090464)),
    ((8.204144852288245, 6.0, 4.020177691372295, 2),
     (0.259276270369745, 0.3144147576406639)),
    ((9.366069953403018, 3.0, 15.422766182794579, 2),
     (0.06961435752241517, 0.22151452459065538)),
    ((10.0, 4.0, 9.192387616705815, 2),
     (0.1593065733661186, 0.2652494804914122)),
    ((10.0, 3.0, 15.954247893607032, 2),
     (0.06533856558730797, 0.20620817208408798)),
    ((9.260586271537607, 3.0, 10.59517579170162, 2),
     (0.1194483796728095, 0.2491723584774583)),
    ((9.571675864670619, 3.0, 17.398847531397934, 2),
     (0.056120220665006354, 0.20987427459193636)),
    ((3.2387393821759787, 5.0, 3.7435106940988625, 1),
     (0.2529831434272229, 0.29484575887521186)),
    ((2.5, 4.0, 7.399343614420917, 1),
     (0.18304020679575475, 0.25792603874732756)),
    ((2.5, 4.0, 8.860840417367838, 1),
     (0.1619064862820606, 0.24508285645237338)),
    ((2.5, 8.0, 10.887265616829124, 1),
     (0.1982153399209648, 0.2800403945667933)),
    ((2.5, 7.0, 10.10233537418591, 1),
     (0.19839199656426879, 0.2769729728229426)),
    ((2.6104349455855846, 8.0, 4.236171515065992, 1),
     (0.26435154821318096, 0.30558752020931723)),
    ((2.5, 9.0, 5.3613636980274295, 1),
     (0.253705681170712, 0.3036923862002273)),
    ((3.1731014606584806, 8.0, 15.199536235308303, 1),
     (0.1580590485886202, 0.25147475798106617)),
    ((2.5, 7.0, 5.4202608625739925, 1),
     (0.24791624789984437, 0.2982609826359614)),
    ((2.5, 8.0, 9.619364938403443, 1),
     (0.21036828710980593, 0.28549866725870554)),
    ((3.252581509053177, 7.0, 6.2224060204343745, 1),
     (0.2390630935153159, 0.29177058812605583)),
    ((2.5, 8.0, 12.261808397585057, 1),
     (0.1856448040789573, 0.27399568137110875)),
    ((2.5, 3.0, 7.768505546917617, 1),
     (0.15474603763604755, 0.23547281814409443)),
    ((2.5, 3.0, 6.998840911724095, 1),
     (0.16686825564034552, 0.24336188065482803)),
    ((3.020562119690717, 3.0, 5.203087539105082, 1),
     (0.19673848562818472, 0.2591147093725996)),
    ((5.2190911687613255, 3.0, 13.573887550967275, 1),
     (0.0906282824426043, 0.15261288582990243)),
    ((5.5962506280473505, 2.0, 5.165106850365733, 1),
     (0.18029987042327314, 0.22800579756695252)),
    ((5.078574838897358, 3.0, 6.8599427244043705, 1),
     (0.17038770476932538, 0.22610259207324904)),
    ((5.1756171558445825, 3.0, 4.56080038214103, 1),
     (0.20820572971121643, 0.25527794433698137)),
    ((5.497353020782844, 5.0, 2.0, 1),
     (0.279552235480715, 0.30256072076105706)),
    ((5.841773513544001, 6.0, 13.28936566781855, 1),
     (0.15805095852659043, 0.219578147164172)),
    ((5.580549185463668, 7.0, 16.1803201492634, 1),
     (0.14439615742800296, 0.21383831068232395)),
    ((5.287772726922527, 7.0, 14.098946461580404, 1),
     (0.16108942091896006, 0.22841596148263876)),
    ((8.358221285614269, 5.0, 17.271563597297103, 1),
     (0.11773949049199511, 0.15965371392065625)),
    ((7.87724479635977, 5.0, 5.598934346859475, 1),
     (0.23076678125893227, 0.2641771899347817)),
    ((8.323336953587479, 5.0, 7.0881523668119195, 1),
     (0.21347348637221167, 0.24827181533645154)),
    ((7.845486096299681, 5.0, 16.23379517928239, 1),
     (0.122698773315744, 0.168346491930928)),
    ((8.020564429287921, 5.0, 18.390260797283936, 1),
     (0.10913570150779599, 0.15334234592862114)),
    ((8.382569502344943, 5.0, 13.97512411087629, 1),
     (0.14368321354779087, 0.18508081572797264)),
    ((7.855593749782354, 3.0, 5.360435825061775, 1),
     (0.19862762162502848, 0.23353318934066486)),
    ((7.655501153733914, 4.0, 9.769593047963392, 1),
     (0.1631153136408173, 0.20467585514331496)),
    ((7.653019158008493, 6.0, 11.704589766625281, 1),
     (0.17675687521357686, 0.22227909812729865)),
    ((9.843286146335094, 3.0, 15.473317400474043, 1),
     (0.09727706414766314, 0.11902807291267123)),
    ((10.0, 3.0, 5.415846167802247, 1),
     (0.20244200747389485, 0.22588706133330697)),
    ((9.474422358368296, 3.0, 15.178161395592507, 1),
     (0.09722523557671268, 0.12212902007311457)),
    ((9.196648156004963, 8.0, 5.069223366759241, 1),
     (0.259590110606016, 0.2847030494945458)),
    ((9.59661128432634, 8.0, 11.180193797198104, 1),
     (0.19725272984671793, 0.23273775140666494)),
    ((10.0, 6.0, 7.76280231640685, 1),
     (0.22139057567772807, 0.24895849892069766)),
    ((10.0, 7.0, 18.37437538640251, 1),
     (0.1385665599883837, 0.17337937229518244)),
    ((2.8739501345809, 4.0, 4.494521106912674, 10),
     (0.24363463440447483, 0.2527864036169822)),
    ((2.979763831715893, 9.0, 6.426710793964199, 10),
     (0.2536062796096411, 0.2679911028334137)),
    ((2.5, 9.0, 2.491252841450378, 10),
     (0.29221903195440846, 0.3020506629214542)),
    ((2.5, 8.0, 8.82337986403619, 10),
     (0.2291778912916562, 0.2455895812236743)),
    ((2.5, 5.0, 18.83933997786449, 10),
     (0.13113805913613338, 0.1355442901438808)),
    ((2.5, 5.0, 5.417711811598947, 10),
     (0.24334188621371236, 0.2555545429854522)),
    ((2.5, 9.0, 8.04535672534691, 10),
     (0.23865982455989818, 0.25668250292257166)),
    ((2.5, 3.0, 16.195810159806815, 10),
     (0.11192304412106252, 0.10562932888135229)),
    ((5.6678871626197305, 2.0, 18.226010811743183, 10),
     (0.12710395272558006, 0.06481848713426154)),
    ((5.759673840199206, 2.0, 30.42873152741525, 10),
     (0.11709221638289552, 0.036741995906091535)),
    ((5.783634661463273, 2.0, 21.480194214511137, 10),
     (0.12525468287616728, 0.054567458290390045)),
    ((5.118173248862928, 2.0, 41.86847335857883, 10),
     (0.09079570140926681, 0.021796051692063684)),
    ((5.757349724389667, 2.0, 13.609604267804956, 10),
     (0.14301420657653752, 0.09199523693485535)),
    ((5.279304061296045, 5.0, 22.876127528048663, 10),
     (0.13458758534346954, 0.10713223523452234)),
    ((5.715709801059808, 5.0, 30.360213488022158, 10),
     (0.11886049487302211, 0.06981616463630416)),
    ((5.947947304520848, 4.0, 4.8966439066197935, 10),
     (0.2500490236976596, 0.24288778972563252)),
    ((5.09899322481724, 5.0, 26.26875042475258, 10),
     (0.11709613281836206, 0.08769066848517765)),
    ((5.53222949762985, 5.0, 7.756449262721482, 10),
     (0.2311862502859707, 0.22551387883678542)),
    ((5.923584541768192, 5.0, 19.567605030849386, 10),
     (0.15795418059599556, 0.1254715070218464)),
    ((5.950156387030171, 2.0, 4.953666946161412, 10),
     (0.218650721667177, 0.1994067474023979)),
    ((5.614158136535322, 2.0, 20.644953904366893, 10),
     (0.12233990218571343, 0.05689997242979489)),
    ((5.435908140730638, 3.0, 21.585064332200393, 10),
     (0.12301577743313093, 0.07397033989544444)),
    ((5.539908561343329, 3.0, 44.90369903995316, 10),
     (0.1037616438284486, 0.027945682306968186)),
    ((5.3792514320991325, 2.0, 25.88907455882873, 10),
     (0.10873545894297332, 0.04236234760922594)),
    ((5.632909830682246, 6.0, 21.384042506861697, 10),
     (0.15459252917395186, 0.1315791146519223)),
    ((5.20332651493292, 6.0, 15.514467427422431, 10),
     (0.17721170525528537, 0.168150476585649)),
    ((5.927793692134072, 5.0, 3.7766395197414253, 10),
     (0.27134805483093305, 0.27045711680975104)),
    ((5.817322396187511, 5.0, 11.31804158090752, 10),
     (0.20484725163482398, 0.1908000868612774)),
    ((7.887786042250045, 1.0, 12.574240714561657, 10),
     (0.18031303615086874, 0.07804201835995273)),
    ((7.949960633591607, 3.0, 25.621368902089333, 10),
     (0.17044491341806606, 0.06725474122671778)),
    ((8.382592436810759, 2.0, 40.54127195292601, 10),
     (0.17178949078589845, 0.024188092070610984)),
    ((7.96379736332257, 2.0, 36.70731870996695, 10),
     (0.16770209089072902, 0.02993902193504958)),
    ((8.373924456610474, 2.0, 8.623846064990166, 10),
     (0.20728444801820906, 0.14388370971167386)),
    ((8.151990686473388, 2.0, 42.229127196458144, 10),
     (0.16904149941391258, 0.021656309205312783)),
    ((8.488384085232076, 8.0, 9.779628072315807, 10),
     (0.2488450360455433, 0.22627788517996875)),
    ((8.438357068876163, 3.0, 26.873452492074044, 10),
     (0.17599559283280108, 0.0648395414079383)),
    ((8.309434906530441, 2.0, 48.49966399344499, 10),
     (0.16957295408937242, 0.012250504009832514)),
    ((7.7115794149655015, 3.0, 5.729859843354196, 10),
     (0.23559517277851352, 0.20522005774679514)),
    ((7.6273740879401934, 2.0, 26.724973070776922, 10),
     (0.16657953614593823, 0.047841689338012895)),
    ((7.693923337226084, 3.0, 48.407897505690485, 10),
     (0.15852007773506502, 0.024265876491785094)),
    ((10.0, 6.0, 10.97195381591066, 10),
     (0.24914023092044668, 0.2089496110383951)),
    ((9.113097274740381, 6.0, 2.7564645951736484, 10),
     (0.2924932485533538, 0.28804004921141624)),
    ((10.0, 9.0, 6.003388325186025, 10),
     (0.2779813642114769, 0.26596442258554676)),
    ((10.0, 5.0, 19.170756721559698, 10),
     (0.2142974946878668, 0.13766814885103174)),
    ((9.380110088755156, 6.0, 18.817507743754415, 10),
     (0.21117041381027712, 0.14914374281863527)),
    ((9.001795946577033, 8.0, 4.453854563212078, 10),
     (0.2859321611242305, 0.28009650987573115)),
    ((10.0, 8.0, 3.653159723688856, 10),
     (0.29311167360260465, 0.2880429565835092)),
    ((9.046182896421445, 3.0, 22.300946806849847, 10),
     (0.1878970718888041, 0.08063250752355927)),
    ((9.459420796383784, 3.0, 10.552556949414955, 10),
     (0.21991607033340996, 0.1509982577104353)),
    ((10.0, 3.0, 31.2476220198124, 10),
     (0.19305142678811255, 0.056344277243534656)),
    ((10.0, 3.0, 29.2734347311525, 10),
     (0.1942359391613085, 0.06175274143556113)),
    ((2.5, 8.0, 8.375074375178261, 9),
     (0.27887477687446527, 0.24546183806091346)),
    ((2.5, 7.0, 9.502846862649331, 9),
     (0.27464003598072734, 0.23312580166748068)),
    ((2.8619005171223564, 7.0, 7.466126134628901, 9),
     (0.28349638659529314, 0.2511139999686381)),
    ((3.0874221941355513, 8.0, 2.493857829360787, 9),
     (0.30329446073589955, 0.29950346826291635)),
    ((2.5, 4.0, 19.77471678075617, 9),
     (0.2398055097946389, 0.12335032186680327)),
    ((2.5, 3.0, 45.299844868071496, 9),
     (0.21835007756596425, 0.029400310263857008)),
    ((3.220553507003754, 3.0, 37.05938066272616, 9),
     (0.23069339455828514, 0.046593234792596326)),
    ((2.5, 4.0, 39.166418500944374, 9),
     (0.21883358149905563, 0.05350074449716688)),
    ((2.7654037016841957, 3.0, 29.726535569137937, 9),
     (0.22876621648675485, 0.06517243299208932)),
    ((2.5, 4.0, 12.087654687250128, 9),
     (0.25568086328187467, 0.17236450351743657)),
    ((2.5, 3.0, 3.382852759577178, 9),
     (0.28321145706274836, 0.2446297883631457)),
    ((2.836612137080781, 4.0, 2.0, 9),
     (0.29704964788098587, 0.2808823856404053)),
    ((2.8888545547050946, 3.0, 14.618307037832857, 9),
     (0.2477649838426708, 0.12970326868762677)),
    ((5.164399331990519, 6.0, 9.111465383743912, 9),
     (0.28951764964116206, 0.23371824784650205)),
    ((5.954180129965368, 6.0, 34.844915916827865, 9),
     (0.2714161513182527, 0.09857200508129484)),
    ((5.500356903003388, 7.0, 13.154128131968298, 9),
     (0.28721219988452035, 0.2147291790311368)),
    ((5.777207914079591, 6.0, 29.94398353538339, 9),
     (0.27203705163012554, 0.11592402886317606)),
    ((5.535810057742433, 7.0, 8.892716664134475, 9),
     (0.2942708344242646, 0.2451099871340891)),
    ((5.590040966343994, 4.0, 22.75661278689855, 9),
     (0.27287420887848585, 0.12066815247186011)),
    ((5.282620261743346, 3.0, 18.732823688754383, 9),
     (0.26860898171122366, 0.11390224239719735)),
    ((5.172895640160181, 3.0, 6.2292543458148515, 9),
     (0.2877332357942354, 0.2114241583957755)),
    ((5.259721854731981, 3.0, 35.890872110681414, 9),
     (0.2577945586528, 0.05159015452853822)),
    ((5.5536463415959245, 3.0, 10.076683709549055, 9),
     (0.28257922939755636, 0.1719757532037015)),
    ((5.730003972159145, 2.0, 15.985698390269977, 9),
     (0.273099534393889, 0.10636788604049591)),
    ((5.782381516990652, 2.0, 28.774618518379302, 9),
     (0.26483796474476695, 0.050195747909618664)),
    ((5.069379781665461, 7.0, 2.2194841714206595, 9),
     (0.3055058860092272, 0.2991172470460963)),
    ((5.903716333756614, 6.0, 35.50557429199497, 9),
     (0.27002853761437307, 0.09626706166005734)),
    ((5.1346796709796605, 6.0, 27.726398643923417, 9),
     (0.2627660248271106, 0.12221521725840637)),
    ((5.383260687864624, 6.0, 18.302295934127923, 9),
     (0.2779886327084519, 0.17403664791365248)),
    ((5.869792088464701, 5.0, 32.55343216796663, 9),
     (0.27112789885723526, 0.09796127335276403)),
    ((5.462451143540612, 6.0, 30.948864634440213, 9),
     (0.2654630166563432, 0.10997924832348281)),
    ((5.357445269639698, 6.0, 5.261434469006405, 9),
     (0.2990173572087526, 0.26625184527715584)),
    ((5.626373453003034, 6.0, 25.170846666445236, 9),
     (0.27422758103121975, 0.13772616835371584)),
    ((8.284200895164993, 2.0, 17.238899804160177, 9),
     (0.3015541927190563, 0.10594069228544409)),
    ((8.318102784124019, 2.0, 22.596147383535918, 9),
     (0.29986823051730344, 0.07933946724011191)),
    ((7.851936866242713, 7.0, 20.962374407911458, 9),
     (0.31278104464771295, 0.18168350176147405)),
    ((8.146081336032703, 8.0, 13.533962918469337, 9),
     (0.3169359353603149, 0.2297919580402022)),
    ((8.09720864316275, 7.0, 17.33899155052454, 9),
     (0.31638806878498915, 0.20284841728512604)),
    ((7.830256291991797, 7.0, 10.706822163825924, 9),
     (0.3131765403497294, 0.24112945829089102)),
    ((7.80065897068848, 6.0, 6.211375680877805, 9),
     (0.31162664368160353, 0.2642498962824071)),
    ((8.044863647118635, 7.0, 15.557155261544228, 9),
     (0.31566706911724673, 0.21257488877131692)),
    ((8.461774802909071, 5.0, 36.03729693977732, 9),
     (0.3193853258545449, 0.09829269654684747)),
    ((7.612382882207284, 4.0, 14.168690780706225, 9),
     (0.30487133599504435, 0.1749148969907045)),
    ((8.169633927695997, 4.0, 27.23584610386441, 9),
     (0.31018864589103556, 0.11196625320524731)),
    ((9.602031136015775, 6.0, 20.5806356758181, 9),
     (0.3358702478125188, 0.18315683261358986)),
    ((9.663686030178818, 6.0, 29.047658472982956, 9),
     (0.3420516528840192, 0.14510390621759314)),
    ((9.75292854736471, 5.0, 34.11493160528129, 9),
     (0.3460058496414126, 0.11306352412679725)),
    ((10.0, 6.0, 4.216215730437086, 9),
     (0.31858648539348344, 0.2854243280891126)),
    ((10.0, 6.0, 34.72852675583916, 9),
     (0.35136426337791954, 0.1268144197324825)),
    ((10.0, 5.0, 14.779627294882367, 9),
     (0.3368575900243706, 0.20211714305442638)),
    ((10.0, 6.0, 2.0, 9), (0.3146, 0.3018)),
    ((9.49705091394873, 6.0, 10.80885478009873, 9),
     (0.32614459837684673, 0.23803767844806178)),
    ((9.826635163465532, 2.0, 7.06711443184985, 9),
     (0.31994127256104854, 0.19825489523721926)),
    ((9.382502350301259, 5.0, 19.999476877446362, 9),
     (0.3328136867226051, 0.17031645852096106)),
    ((9.115530591819274, 5.0, 5.883436488694818, 9),
     (0.3188381363847329, 0.2614152937200581)),
    ((10.0, 5.0, 24.745870232952445, 9),
     (0.3456712832096572, 0.1519571163934759)),
    ((9.378359588580793, 5.0, 26.295787257422923, 9),
     (0.33580634416152166, 0.14170543655286671)),
    ((10.0, 2.0, 21.57257635660235, 9),
     (0.3230213711821699, 0.08825848939915812)),
    ((10.0, 3.0, 26.039872491235577, 9),
     (0.3343139553719324, 0.09760462479294567)),
    ((2.5, 3.0, 4.712138166253982, 8),
     (0.3435962977395826, 0.2553142252457729)),
    ((2.8874578666829285, 2.0, 13.994896052145748, 8),
     (0.37835548166727323, 0.14725225859091387)),
    ((3.435419560439465, 2.0, 6.718989113532732, 8),
     (0.35983522137260177, 0.22143958809308129)),
    ((2.9925336062737173, 2.0, 7.198014339866309, 8),
     (0.3575475618962941, 0.21313809904749426)),
    ((2.5, 1.0, 14.156726710024465, 8), (0.3368, 0.10107531241085566)),
    ((2.6104579288975813, 1.0, 3.3458156268951917, 8),
     (0.3281972508449899, 0.2218263920769044)),
    ((5.1670653045538115, 8.0, 2.1409481568506346, 8),
     (0.31905079087740745, 0.31163900088406254)),
    ((5.054434114346951, 7.0, 6.442157332603133, 8),
     (0.35015366076172083, 0.2933004270294575)),
    ((5.803735682450612, 9.0, 10.443841773523394, 8),
     (0.37537800934530957, 0.2859037624969863)),
    ((5.044877539779968, 6.0, 18.424428701407553, 8),
     (0.4277251999841766, 0.23640419314222352)),
    ((5.484832402621484, 7.0, 5.474777491295647, 8),
     (0.3449818965080228, 0.2983629686835587)),
    ((5.162300427200289, 7.0, 24.999056248525125, 8),
     (0.4499951490629939, 0.22267274868897197)),
    ((5.877256360743413, 7.0, 15.450444143259661, 8),
     (0.4135138022364607, 0.2615018632684154)),
    ((8.197449080109873, 2.0, 2.0, 8),
     (0.3479365513139208, 0.2910133676082857)),
    ((7.997237265754237, 2.0, 11.655829335806517, 8),
     (0.45364925931660116, 0.19649613209426764)),
    ((7.973192560907184, 9.0, 4.272886886879181, 8),
     (0.3382290806106464, 0.3100993156381419)),
    ((7.510355740108461, 8.0, 14.320141317950995, 8),
     (0.41602573142890514, 0.2791113211989992)),
    ((7.836498646186221, 8.0, 13.596658717999025, 8),
     (0.41449506838703515, 0.28339932777466953)),
    ((7.782186965908517, 9.0, 13.902105524067945, 8),
     (0.4117119567839591, 0.28581945828988925)),
    ((9.531795266771761, 5.0, 2.0, 8),
     (0.3325257851841513, 0.3124819697521387)),
    ((10.0, 5.0, 11.055624912778937, 8),
     (0.44623696767281984, 0.2877358440858011)),
    ((9.312270837393163, 7.0, 11.185222099189973, 8),
     (0.4115569799140528, 0.2962671589127124)),
    ((10.0, 7.0, 13.895455902446677, 8),
     (0.44457546784397745, 0.29335613303900565)),
    ((9.925669940032272, 5.0, 7.2040789887667955, 8),
     (0.39993423356459634, 0.3000028935899999)),
    ((9.416740882402403, 5.0, 8.720116348180492, 8),
     (0.41420875274374974, 0.2926385477337663)),
    ((10.0, 4.0, 16.469698910991372, 8),
     (0.5288485073674999, 0.2505105957717457)),
    ((10.0, 6.0, 6.599237233947309, 8),
     (0.37969275372249944, 0.30632137297889483)),
    ((10.0, 5.0, 4.550269784467781, 8),
     (0.36647096673041096, 0.30759681204960715)),
    ((9.970332530519679, 6.0, 10.837022722087644, 8),
     (0.4235409167024141, 0.2965173827647608)),
    ((2.962707587174585, 9.0, 9.999116931630539, 7),
     (0.4082112009458871, 0.317851549976094)),
    ((3.1672052728994915, 9.0, 7.383624729892915, 7),
     (0.38120519554647453, 0.3201699383275524)),
    ((2.5, 5.0, 17.881593853007615, 7),
     (0.5525791262360914, 0.28084994335857105)),
    ((2.7415018638966284, 6.0, 18.00290873780138, 7),
     (0.5290289042496541, 0.2948216346860084)),
    ((2.5, 6.0, 10.232668996271492, 7),
     (0.4348850955537665, 0.3113811958067113)),
    ((2.877902226185231, 6.0, 3.5582034231201787, 7),
     (0.3519353725942859, 0.31712947263253355)),
    ((2.5, 6.0, 27.77999592691697, 7),
     (0.5960099816711264, 0.2656600122192491)),
    ((5.412821771284458, 3.0, 7.258040020605607, 7),
     (0.49172648760450915, 0.31621073924712606)),
    ((5.83754747605084, 3.0, 11.998261380615471, 7),
     (0.5975448574102102, 0.29794809688903484)),
    ((5.9693975439749885, 4.0, 14.397906420283302, 7),
     (0.5881617924445709, 0.31236418999305104)),
    ((5.004079000563381, 5.0, 22.736677614468775, 7),
     (0.63920549388473, 0.28783077224442344)),
    ((5.168438425945292, 4.0, 4.844860547907693, 7),
     (0.40839719509713, 0.32305082659991396)),
    ((5.863284315202094, 4.0, 23.489710023246513, 7),
     (0.7027892157150165, 0.2695458294720114)),
    ((5.756333389411959, 9.0, 7.301135618422141, 7),
     (0.39078475628981396, 0.32893785909586065)),
    ((5.108337403014788, 8.0, 11.359771531491097, 7),
     (0.44539182306302255, 0.32701253286287824)),
    ((8.314898437378535, 9.0, 4.238233636005843, 7),
     (0.3599697419936892, 0.33117366775661333)),
    ((7.729982986777109, 5.0, 24.923686571499648, 7),
     (0.6880520401649296, 0.31336218352271883)),
    ((8.201460399608226, 4.0, 11.589840844520428, 7),
     (0.5584840865313621, 0.33959721735842735)),
    ((7.595604919273442, 5.0, 6.798265747221928, 7),
     (0.4338251159286378, 0.3369357659148767)),
    ((8.378186361828917, 9.0, 8.022357890675561, 7),
     (0.4105838750780343, 0.34204796349468947)),
    ((8.300135000740797, 8.0, 14.433553547681656, 7),
     (0.5019186699595175, 0.35172447128424394)),
    ((8.229270762113973, 9.0, 6.350022396927342, 7),
     (0.38792000675351795, 0.33801569665443554)),
    ((10.0, 3.0, 3.1152259635487924, 7),
     (0.405141552942915, 0.33686460722138906)),
    ((9.756267998308681, 3.0, 14.803384721914584, 7),
     (0.6814476281796187, 0.3179618099654896)),
    ((10.0, 4.0, 13.90160960971739, 7),
     (0.6136634096115119, 0.3568983903902826)),
    ((10.0, 8.0, 19.365358380679876, 7), (0.5632882254261189, 0.374)),
    ((10.0, 9.0, 6.218490965882184, 7),
     (0.3910588735223506, 0.34456639744594064)),
    ((10.0, 8.0, 13.887493044276624, 7),
     (0.5055374095755961, 0.36877498608855325)),
    ((10.0, 5.0, 14.68907159946693, 7),
     (0.5862646522729101, 0.3661588249401866)),
    ((10.0, 5.0, 24.263442351912005, 7),
     (0.704502702343164, 0.353473115296176)),
    ((10.0, 8.0, 13.518172354943417, 7),
     (0.5007362406142645, 0.36803634470988683)),
    ((2.7455640547144746, 2.0, 5.569110673549164, 6),
     (0.5147965927948814, 0.35947427079052985)),
    ((3.1452880891491906, 5.0, 8.595832717291, 6),
     (0.49156559782999765, 0.3836139842529673)),
    ((2.5, 4.0, 4.950679151608691, 6),
     (0.4364884940203847, 0.3603170842733587)),
    ((2.5, 4.0, 4.383231249423155, 6),
     (0.42312509592391534, 0.3564868109336063)),
    ((2.5, 2.0, 3.307282274836235, 6),
     (0.43396162885139156, 0.3458470682650791)),
    ((5.045583268005572, 9.0, 9.59194524860244, 6),
     (0.4426976823901471, 0.3880493030502878)),
    ((5.594284526041456, 9.0, 10.197201238166286, 6),
     (0.45008651645515263, 0.39500104997859575)),
    ((5.988802467213943, 8.0, 12.30595195616923, 6),
     (0.48725749804679613, 0.4136999258156572)),
    ((5.425850947396252, 5.0, 8.046156862703112, 6),
     (0.48370601248767936, 0.39930129085649035)),
    ((5.405852543210212, 6.0, 16.635714109554605, 6),
     (0.5613340460279886, 0.4289299103499902)),
    ((5.369364240119585, 5.0, 7.340573827339962, 6),
     (0.46958736593496786, 0.39345497811572305)),
    ((5.702045821590509, 5.0, 10.325652051724541, 6),
     (0.5189311698950891, 0.41373924250477145)),
    ((5.411096326958829, 6.0, 5.292034843095026, 6),
     (0.40946256871366055, 0.37022550255078585)),
    ((8.242968536635763, 9.0, 4.90020586532881, 6),
     (0.38006673083868486, 0.3693561101855342)),
    ((8.238754570485817, 9.0, 5.94133011037865, 6),
     (0.3940851904797918, 0.379080970224506)),
    ((8.39568424389748, 5.0, 9.461515968715135, 6),
     (0.5006508183439433, 0.43147844246085765)),
    ((10.0, 5.0, 12.704963485646498, 6),
     (0.5274094231965362, 0.462819853942586)),
    ((10.0, 5.0, 15.6753707607594, 6),
     (0.5498899099449361, 0.47553916842341726)),
    ((10.0, 6.0, 3.506573388368494, 6),
     (0.37697160768481713, 0.3696933421148326)),
    ((10.0, 6.0, 14.063922879568509, 6),
     (0.5203515758376268, 0.46251414164655447)),
    ((10.0, 7.0, 3.128443413944953, 6),
     (0.36320142718357795, 0.36035187523477064)),
    ((10.0, 7.0, 11.632405914314647, 6),
     (0.4857175289017656, 0.4453349428787812)),
    ((9.050263182466011, 7.0, 17.08367694275979, 6),
     (0.5287506410501914, 0.459929219239207)),
    ((10.0, 6.0, 4.736966947326921, 6),
     (0.40006552365184517, 0.386391115357349)),
    ((9.409402543801862, 7.0, 6.28766021168659, 6),
     (0.41478835014788323, 0.39548485637022385)),
    ((9.633394604006961, 8.0, 4.623044001702525, 6),
     (0.37931316473145726, 0.3728258686141236)),
    ((9.020770192275748, 7.0, 13.422245014577644, 6),
     (0.5060230048016112, 0.44738008068068885)),
    ((9.26317609686154, 7.0, 15.233295182477667, 6),
     (0.517940485402123, 0.4564421027270388)),
    ((3.332782026387723, 7.0, 16.113419977677538, 5),
     (0.49943843353399675, 0.4913166627882885)),
    ((2.5, 5.0, 6.5436496028361315, 5),
     (0.446290656443251, 0.4355063353928991)),
    ((2.5, 6.0, 15.572129740854304, 5),
     (0.5138820422172288, 0.4876949957096056)),
    ((2.5, 3.0, 2.0, 5), (0.3703, 0.37)),
    ((2.8285591842433737, 9.0, 21.473258817290873, 5),
     (0.5043337880565358, 0.4951865962256489)),
    ((2.5, 8.0, 12.020108658634838, 5),
     (0.4679648910008057, 0.4590236682506042)),
    ((2.5, 9.0, 14.42790441415372, 5),
     (0.47578137869199916, 0.46735347427784546)),
    ((2.5, 5.0, 8.380243803410817, 5), (0.472682681837519, 0.455422938237116)),
    ((3.363079416671538, 5.0, 2.7755096642090313, 5),
     (0.36889260512263344, 0.3743534718948429)),
    ((5.9271524261020545, 9.0, 20.603131952471927, 5),
     (0.4802472251615271, 0.5159774928137729)),
    ((5.339079962653624, 8.0, 16.611574939424255, 5),
     (0.4789877671483087, 0.5049439528400183)),
    ((5.347356764781598, 8.0, 15.41216519823205, 5),
     (0.4745806920664243, 0.5005081883465945)),
    ((5.368950609634622, 7.0, 7.038165919924306, 5),
     (0.41341154716192496, 0.4348136096758073)),
    ((5.063316239211655, 7.0, 16.01331933482103, 5),
     (0.487164109418344, 0.5051543240966131)),
    ((5.929552854535908, 7.0, 7.57281344704806, 5),
     (0.41853653124143764, 0.444243976557503)),
    ((5.72794655950891, 7.0, 10.668172633934036, 5),
     (0.45318357111848423, 0.47904872268084375)),
    ((5.641782139668679, 6.0, 9.549016885745186, 5),
     (0.4561067615040472, 0.4783274892995489)),
    ((5.344359642058747, 3.0, 5.430489560972486, 5),
     (0.4516333592896905, 0.461109580193912)),
    ((7.749909297802317, 4.0, 4.268933751175051, 5),
     (0.40175883449950806, 0.4334105720840665)),
    ((8.145409228909998, 5.0, 7.545633529064384, 5),
     (0.435789245569801, 0.4810623452292749)),
    ((7.907253670159305, 6.0, 10.770986229289623, 5),
     (0.4538016350466874, 0.5021167554370949)),
    ((7.592508492261312, 5.0, 4.933568344499713, 5),
     (0.4009033326671016, 0.4323309706149007)),
    ((7.674872690410821, 5.0, 3.5502452884794837, 5),
     (0.37590596292111755, 0.4014473524868083)),
    ((7.991979987062054, 7.0, 3.2837012487472252, 5),
     (0.35678303301647424, 0.37978502351744886)),
    ((9.345599185286883, 7.0, 17.48852175788182, 5),
     (0.46492781928598614, 0.537405269620011)),
    ((9.659595218511388, 8.0, 3.3572177484844636, 5),
     (0.35143609296322403, 0.377417525766746)))

MUNSELL_COLOURS_TO_XYY = (
    np.array([0.41515095, 0.51288165, 0.5702441]),
    np.array([0.38804358, 0.46299149, 0.31592072]),
    np.array([0.33491518, 0.36277402, 0.22128409]),
    np.array([0.39936353, 0.58547238, 0.64852094]),
    np.array([0.34767896, 0.4152922, 0.58706989]),
    np.array([0.33966055, 0.41527226, 0.07167165]),
    np.array([0.36265912, 0.47966922, 0.11068168]),
    np.array([0.35748002, 0.45915987, 0.2727359]),
    np.array([0.36348032, 0.48213512, 0.06293782]),
    np.array([0.30330033, 0.73038471, 0.05538644]),
    np.array([0.33159302, 0.43388935, 0.89380734]),
    np.array([0.31838794, 0.40167814, 0.05382145]),
    np.array([0.27202005, 0.83522048, 0.04995375]),
    np.array([0.31425413, 0.58372544, 0.04377268]),
    np.array([0.27634942, 0.75063178, 0.05211431]),
    np.array([0.258837, 0.71096717, 0.01266934]),
    np.array([0.31405111, 0.53120144, 0.02111891]),
    np.array([0.33914454, 0.6563647, 0.71217401]),
    np.array([0.35328989, 0.63157007, 0.65497851]),
    np.array([0.32167873, 0.43862617, 0.3080991]),
    np.array([0.31168045, 0.6270064, 0.34717087]),
    np.array([0.31496017, 0.47530248, 0.67920304]),
    np.array([0.26882355, 0.70549119, 0.69614462]),
    np.array([0.31107787, 0.51188895, 0.58306925]),
    np.array([0.31254722, 0.34686238, 0.6334334]),
    np.array([0.30880402, 0.37157402, 0.08263161]),
    np.array([0.23582365, 0.72197618, 0.06667783]),
    np.array([0.29476305, 0.57521949, 0.23583791]),
    np.array([0.28891056, 0.61005165, 0.28191444]),
    np.array([0.17590584, 0.91365, 0.23196178]),
    np.array([0.31292041, 0.3752074, 0.25538037]),
    np.array([0.22307972, 0.8153644, 0.2698602]),
    np.array([0.30648167, 0.48754769, 0.15098549]),
    np.array([0.30382174, 0.34089453, 0.84210967]),
    np.array([0.28517207, 0.38369148, 0.89445395]),
    np.array([0.20621151, 0.56369357, 0.77955867]),
    np.array([0.2465848, 0.49294784, 0.87271533]),
    np.array([0.22538285, 0.5564611, 0.60532773]),
    np.array([0.28500017, 0.38833563, 0.24045742]),
    np.array([0.19598037, 0.59002914, 0.29181101]),
    np.array([0.16437784, 0.59069112, 0.23370301]),
    np.array([0.17940333, 0.4663929, 0.06448045]),
    np.array([0.07553293, 0.55981543, 0.06406275]),
    np.array([0.27330162, 0.37048932, 0.11621278]),
    np.array([0.23251367, 0.40832841, 0.02585745]),
    np.array([0.05704598, 0.55990299, 0.01221862]),
    np.array([0.09405428, 0.51916421, 0.02268015]),
    np.array([0.06306305, 0.54336526, 0.0361037]),
    np.array([0.23250342, 0.41833342, 0.0559913]),
    np.array([0.22630523, 0.39163204, 0.05597116]),
    np.array([0.15858055, 0.42916814, 0.05259972]),
    np.array([0.07933408, 0.51474312, 0.30905098]),
    np.array([0.14028772, 0.46282023, 0.41589047]),
    np.array([0.29271668, 0.33531051, 0.37326792]),
    np.array([0.17253811, 0.43786778, 0.33686994]),
    np.array([0.09180367, 0.46823752, 0.05151176]),
    np.array([0.10903846, 0.44893518, 0.03595462]),
    np.array([0.2428693, 0.37094376, 0.04060119]),
    np.array([0.27771166, 0.34994832, 0.23574564]),
    np.array([0.05867972, 0.50502648, 0.19891229]),
    np.array([0.25930387, 0.37349411, 0.26874577]),
    np.array([0.12284826, 0.47211684, 0.21388094]),
    np.array([0.0890682, 0.48703791, 0.27058998]),
    np.array([0.27018357, 0.35138182, 0.76804186]),
    np.array([0.22062535, 0.38110738, 0.85084234]),
    np.array([0.26193025, 0.3581405, 0.86839733]),
    np.array([0.0431053, 0.45634623, 0.12074655]),
    np.array([0.16522669, 0.40881359, 0.18014875]),
    np.array([0.02517244, 0.46138968, 0.1317301]),
    np.array([0.23349872, 0.37536989, 0.14476492]),
    np.array([0.05119965, 0.46839242, 0.26212526]),
    np.array([0.2315995, 0.37207726, 0.20351563]),
    np.array([0.08301372, 0.45335265, 0.25304755]),
    np.array([0.20183026, 0.36561544, 0.39526058]),
    np.array([0.06340759, 0.37121187, 0.07975536]),
    np.array([0.16044634, 0.34707426, 0.10145605]),
    np.array([0.24416648, 0.33434737, 0.07774819]),
    np.array([0.28155768, 0.33248001, 0.89992977]),
    np.array([0.28105936, 0.3327088, 0.88937678]),
    np.array([0.25255297, 0.34594245, 0.87623351]),
    np.array([0.20616318, 0.34192146, 0.77176579]),
    np.array([0.21898553, 0.33335124, 0.85174026]),
    np.array([0.19119679, 0.33526743, 0.80792502]),
    np.array([0.29624596, 0.31950269, 0.96665647]),
    np.array([0.24328961, 0.31868567, 0.12931978]),
    np.array([0.10471116, 0.30938022, 0.15549815]),
    np.array([0.0862452, 0.30268915, 0.15900713]),
    np.array([0.10497041, 0.32451898, 0.22191645]),
    np.array([0.16894641, 0.33087742, 0.29312371]),
    np.array([0.16144965, 0.33133829, 0.34018592]),
    np.array([0.25864013, 0.31415379, 0.28205753]),
    np.array([0.07732853, 0.22846579, 0.08121964]),
    np.array([0.15795868, 0.26417318, 0.11377678]),
    np.array([0.06907834, 0.20994435, 0.0722573]),
    np.array([0.12862477, 0.25616557, 0.08539517]),
    np.array([0.05881481, 0.21256736, 0.07052095]),
    np.array([0.25058288, 0.29329096, 0.17796585]),
    np.array([0.18830894, 0.26192867, 0.13740285]),
    np.array([0.1684076, 0.25029878, 0.13934697]),
    np.array([0.1951648, 0.27716957, 0.51306785]),
    np.array([0.19935306, 0.27783329, 0.44060477]),
    np.array([0.26308512, 0.3046212, 0.52610451]),
    np.array([0.2532416, 0.30291555, 0.67153139]),
    np.array([0.15890128, 0.2532598, 0.59956247]),
    np.array([0.24841933, 0.2986962, 0.43833832]),
    np.array([0.2082133, 0.28356991, 0.52733609]),
    np.array([0.23939654, 0.2920611, 0.43144538]),
    np.array([0.18279859, 0.27122662, 0.52199238]),
    np.array([0.16449512, 0.24371038, 0.08686299]),
    np.array([0.16724393, 0.24366794, 0.06480227]),
    np.array([0.19881487, 0.26071106, 0.06927689]),
    np.array([0.09076654, 0.15277497, 0.06421355]),
    np.array([0.18253778, 0.23018215, 0.03460635]),
    np.array([0.16926303, 0.22496873, 0.06237928]),
    np.array([0.20398493, 0.2513471, 0.05473403]),
    np.array([0.28140041, 0.30378091, 0.23081828]),
    np.array([0.15231331, 0.21384066, 0.25883348]),
    np.array([0.14386953, 0.21327677, 0.41482428]),
    np.array([0.1593506, 0.22670722, 0.40114326]),
    np.array([0.10949743, 0.15034868, 0.15892888]),
    np.array([0.22674934, 0.26033997, 0.17110185]),
    np.array([0.20569472, 0.2404847, 0.15700695]),
    np.array([0.11359218, 0.15851929, 0.15851498]),
    np.array([0.13446868, 0.17456223, 0.15665285]),
    np.array([0.20295637, 0.23758918, 0.07464645]),
    np.array([0.16020908, 0.20160833, 0.11096053]),
    np.array([0.17946292, 0.22546056, 0.3340693]),
    np.array([0.19584886, 0.21874231, 0.05264774]),
    np.array([0.25950493, 0.28494406, 0.60260113]),
    np.array([0.22170777, 0.24928491, 0.29763974]),
    np.array([0.13564759, 0.16991066, 0.38138893]),
    np.array([0.23373145, 0.24171207, 0.08831548]),
    np.array([0.25339824, 0.26720506, 0.67917402]),
    np.array([0.29210338, 0.30192924, 0.75127547]),
    np.array([0.22958296, 0.2462168, 0.59738522]),
    np.array([0.1258535, 0.12764109, 0.16297312]),
    np.array([0.24227309, 0.25436998, 0.18624748]),
    np.array([0.23758242, 0.25457444, 0.66865194]),
    np.array([0.10476265, 0.09497701, 0.05235122]),
    np.array([0.12612865, 0.06066443, 0.02676646]),
    np.array([0.11705747, 0.03587748, 0.02951591]),
    np.array([0.1232905, 0.0441543, 0.02037758]),
    np.array([0.09139852, 0.01529466, 0.02045231]),
    np.array([0.13833192, 0.07953813, 0.02236117]),
    np.array([0.13361693, 0.10504399, 0.18414205]),
    np.array([0.1210474, 0.06862453, 0.15728175]),
    np.array([0.25249867, 0.24628189, 0.1353695]),
    np.array([0.11706407, 0.08706468, 0.18814811]),
    np.array([0.22549284, 0.2180621, 0.16192792]),
    np.array([0.1534495, 0.11674072, 0.15905692]),
    np.array([0.2235872, 0.20668864, 0.04253357]),
    np.array([0.12515256, 0.06568452, 0.04436879]),
    np.array([0.12125722, 0.0687482, 0.05533026]),
    np.array([0.10373316, 0.0277414, 0.06333516]),
    np.array([0.10925991, 0.04419045, 0.03405371]),
    np.array([0.15402461, 0.13042053, 0.28570417]),
    np.array([0.17573216, 0.16578146, 0.27364637]),
    np.array([0.27401103, 0.27401935, 0.23451177]),
    np.array([0.2075913, 0.19464274, 0.21940166]),
    np.array([0.17049737, 0.06465369, 0.05868583]),
    np.array([0.17064728, 0.0288915, 0.04372401]),
    np.array([0.1672038, 0.03196773, 0.03579761]),
    np.array([0.21031018, 0.15034168, 0.03888934]),
    np.array([0.16827351, 0.02413193, 0.03757647]),
    np.array([0.29178046, 0.29061931, 0.90323404]),
    np.array([0.24910224, 0.22648966, 0.59336016]),
    np.array([0.17601554, 0.0587606, 0.05160293]),
    np.array([0.16834537, 0.01686511, 0.04374851]),
    np.array([0.23182863, 0.19825806, 0.05291206]),
    np.array([0.16638758, 0.05075245, 0.03650792]),
    np.array([0.16028497, 0.01948654, 0.05046003]),
    np.array([0.24957235, 0.21006823, 0.31587613]),
    np.array([0.29306654, 0.28917618, 0.32466527]),
    np.array([0.28495343, 0.27687408, 0.81760638]),
    np.array([0.21441304, 0.13814375, 0.19716723]),
    np.array([0.20941829, 0.14321541, 0.24327119]),
    np.array([0.28541299, 0.27913907, 0.54006024]),
    np.array([0.29230469, 0.28656219, 0.52465762]),
    np.array([0.18804124, 0.08137467, 0.06580398]),
    np.array([0.22025958, 0.15180899, 0.06551257]),
    np.array([0.19309397, 0.06115047, 0.07873642]),
    np.array([0.19437258, 0.06326427, 0.06829742]),
    np.array([0.27887167, 0.24543217, 0.57450962]),
    np.array([0.27487624, 0.23376357, 0.46322748]),
    np.array([0.28356864, 0.2519005, 0.45980664]),
    np.array([0.30333596, 0.30005216, 0.66401066]),
    np.array([0.23835467, 0.11558036, 0.09827669]),
    np.array([0.23067198, 0.05028062, 0.07671426]),
    np.array([0.21902307, 0.05208443, 0.11065271]),
    np.array([0.22907253, 0.06719948, 0.06903321]),
    np.array([0.2536145, 0.16387485, 0.0990085]),
    np.array([0.28535713, 0.25114971, 0.08429109]),
    np.array([0.29701504, 0.28076672, 0.11652327]),
    np.array([0.24894294, 0.13513311, 0.0750785]),
    np.array([0.28976435, 0.23551078, 0.3203068]),
    np.array([0.28699217, 0.2122739, 0.38376156]),
    np.array([0.2942318, 0.24483482, 0.41568603]),
    np.array([0.27112866, 0.10892559, 0.09137276]),
    np.array([0.26932562, 0.11871922, 0.07456975]),
    np.array([0.28774446, 0.21149857, 0.06409553]),
    np.array([0.25815891, 0.05632389, 0.07763328]),
    np.array([0.28438514, 0.18361032, 0.08751006]),
    np.array([0.27466364, 0.11623324, 0.04459164]),
    np.array([0.26635689, 0.0603288, 0.04436654]),
    np.array([0.30526917, 0.29787617, 0.38438766]),
    np.array([0.26275899, 0.12295408, 0.3048271]),
    np.array([0.27733084, 0.16764806, 0.24584118]),
    np.array([0.27121622, 0.0996767, 0.21385417]),
    np.array([0.26547923, 0.10802713, 0.26515926]),
    np.array([0.29841781, 0.26325636, 0.25902873]),
    np.array([0.27412192, 0.13541072, 0.26778091]),
    np.array([0.3042953, 0.11611832, 0.04387]),
    np.array([0.30157505, 0.08506396, 0.03768091]),
    np.array([0.31391169, 0.1856442, 0.48667459]),
    np.array([0.3167079, 0.22835511, 0.52829657]),
    np.array([0.31664956, 0.20454265, 0.45562827]),
    np.array([0.31300137, 0.23982828, 0.40210613]),
    np.array([0.31187872, 0.26667157, 0.33190218]),
    np.array([0.31537904, 0.21052765, 0.39335492]),
    np.array([0.31803143, 0.09273886, 0.1712263]),
    np.array([0.30594132, 0.18152717, 0.14244072]),
    np.array([0.31195968, 0.12089229, 0.15102095]),
    np.array([0.33618672, 0.17589268, 0.24249386]),
    np.array([0.34207627, 0.13875616, 0.24138597]),
    np.array([0.34605075, 0.11899797, 0.23580785]),
    np.array([0.31923003, 0.28291153, 0.25504488]),
    np.array([0.35136426, 0.12256902, 0.2641027]),
    np.array([0.33639641, 0.20777481, 0.23332748]),
    np.array([0.31464507, 0.3010788, 0.27040807]),
    np.array([0.32622786, 0.23679153, 0.28338647]),
    np.array([0.31964789, 0.19702337, 0.02988488]),
    np.array([0.33202416, 0.16293316, 0.16828902]),
    np.array([0.3188341, 0.26119414, 0.19149517]),
    np.array([0.34497302, 0.14740581, 0.17674791]),
    np.array([0.33396066, 0.13204228, 0.15759269]),
    np.array([0.32447663, 0.09207588, 0.03498261]),
    np.array([0.32823298, 0.08288658, 0.04740281]),
    np.array([0.34263192, 0.2492826, 0.04966462]),
    np.array([0.37863885, 0.1480557, 0.03133476]),
    np.array([0.36067287, 0.22508694, 0.03664306]),
    np.array([0.35583972, 0.20890369, 0.0287403]),
    np.array([0.34728299, 0.11402692, 0.01746108]),
    np.array([0.32940771, 0.22789278, 0.01489395]),
    np.array([0.31972567, 0.31122932, 0.53600948]),
    np.array([0.35012172, 0.29333067, 0.42147094]),
    np.array([0.37589661, 0.2850717, 0.66934047]),
    np.array([0.42549932, 0.23904177, 0.33329037]),
    np.array([0.34641765, 0.2972505, 0.38411768]),
    np.array([0.45441652, 0.21797623, 0.36276856]),
    np.array([0.41521602, 0.25989123, 0.39086156]),
    np.array([0.34780042, 0.2928404, 0.0360562]),
    np.array([0.4544551, 0.19822245, 0.03201793]),
    np.array([0.33858745, 0.3098545, 0.70004006]),
    np.array([0.41381262, 0.2839371, 0.60579167]),
    np.array([0.39278492, 0.2914687, 0.81034741]),
    np.array([0.33239612, 0.31251827, 0.19604738]),
    np.array([0.43846181, 0.29096381, 0.23141236]),
    np.array([0.40958022, 0.29719222, 0.48882871]),
    np.array([0.44399899, 0.29369509, 0.43379687]),
    np.array([0.40554919, 0.29723013, 0.16687769]),
    np.array([0.42007003, 0.28930815, 0.1672933]),
    np.array([0.52108329, 0.25574146, 0.13999526]),
    np.array([0.3763801, 0.30728007, 0.34070289]),
    np.array([0.36495307, 0.30801481, 0.20910915]),
    np.array([0.42566912, 0.29564012, 0.28217939]),
    np.array([0.38537971, 0.31745807, 0.82116554]),
    np.array([0.37201534, 0.31965197, 0.79705828]),
    np.array([0.55136347, 0.28138892, 0.19712193]),
    np.array([0.53899416, 0.29048788, 0.25823634]),
    np.array([0.43854811, 0.3103317, 0.27612362]),
    np.array([0.35589069, 0.3165537, 0.24649473]),
    np.array([0.6015019, 0.26287828, 0.27670596]),
    np.array([0.49631592, 0.30111191, 0.04570504]),
    np.array([0.60338354, 0.2746834, 0.04600213]),
    np.array([0.57619776, 0.31554717, 0.14073356]),
    np.array([0.65681487, 0.27970869, 0.16409107]),
    np.array([0.40414547, 0.32310724, 0.13347887]),
    np.array([0.68743116, 0.27762719, 0.14148314]),
    np.array([0.39097754, 0.32893313, 0.75691217]),
    np.array([0.44274163, 0.32668916, 0.63163346]),
    np.array([0.3514372, 0.32860694, 0.80679695]),
    np.array([0.55200335, 0.34090583, 0.13240521]),
    np.array([0.43719237, 0.33673056, 0.18274766]),
    np.array([0.36439573, 0.33015311, 0.87403366]),
    np.array([0.34753957, 0.32343836, 0.93909237]),
    np.array([0.38880059, 0.33783693, 0.7040874]),
    np.array([0.40006019, 0.33663147, 0.07790261]),
    np.array([0.67248369, 0.32330365, 0.07220649]),
    np.array([0.64354918, 0.33973639, 0.08803122]),
    np.array([0.39181364, 0.3446948, 0.72252254]),
    np.array([0.49958346, 0.36703778, 0.64322822]),
    np.array([0.60231065, 0.36168845, 0.16068191]),
    np.array([0.50023387, 0.36789369, 0.5819145]),
    np.array([0.5479115, 0.34892913, 0.02263783]),
    np.array([0.48794187, 0.38293202, 0.2066566]),
    np.array([0.42836733, 0.35891726, 0.14365536]),
    np.array([0.4151889, 0.35495306, 0.14408043]),
    np.array([0.46542405, 0.34082576, 0.02079253]),
    np.array([0.44637245, 0.38945422, 0.69298338]),
    np.array([0.45404134, 0.39647154, 0.69233629]),
    np.array([0.48546253, 0.4130592, 0.59936297]),
    np.array([0.47978428, 0.39825981, 0.20940783]),
    np.array([0.46785973, 0.39297126, 0.20053901]),
    np.array([0.51331905, 0.41265964, 0.21747039]),
    np.array([0.41770486, 0.37303747, 0.24430738]),
    np.array([0.37481816, 0.36536787, 0.78442772]),
    np.array([0.35992953, 0.34813048, 0.94470358]),
    np.array([0.39620152, 0.37951049, 0.68657974]),
    np.array([0.51299224, 0.43528921, 0.15241166]),
    np.array([0.52529122, 0.46205863, 0.20512581]),
    np.array([0.54693353, 0.47444863, 0.20705936]),
    np.array([0.38249101, 0.37271251, 0.24466996]),
    np.array([0.52465528, 0.46381752, 0.27037022]),
    np.array([0.36332827, 0.36042625, 0.417479]),
    np.array([0.49103786, 0.44741782, 0.37625337]),
    np.array([0.53260672, 0.46085797, 0.38175879]),
    np.array([0.39947636, 0.38607101, 0.30024818]),
    np.array([0.41566751, 0.39590836, 0.41203391]),
    np.array([0.38379951, 0.37533829, 0.49499836]),
    np.array([0.50184813, 0.44587964, 0.46556628]),
    np.array([0.51381974, 0.45512612, 0.47222976]),
    np.array([0.49758967, 0.49049612, 0.45242214]),
    np.array([0.43935016, 0.43103452, 0.23259639]),
    np.array([0.37285524, 0.37178029, 0.05690988]),
    np.array([0.50597795, 0.49533702, 0.71175266]),
    np.array([0.46625888, 0.45805354, 0.60604474]),
    np.array([0.47877996, 0.46893263, 0.71437706]),
    np.array([0.48009865, 0.45955664, 0.16263768]),
    np.array([0.37334852, 0.37797455, 0.16515524]),
    np.array([0.47856178, 0.5047532, 0.58732427]),
    np.array([0.47119689, 0.49876821, 0.65483021]),
    np.array([0.40983957, 0.43186708, 0.474305]),
    np.array([0.42018308, 0.44575131, 0.405341]),
    np.array([0.45841191, 0.4830471, 0.3634053]),
    np.array([0.45040867, 0.47371048, 0.34766863]),
    np.array([0.45990386, 0.46807933, 0.05857873]),
    np.array([0.39793427, 0.428948, 0.13930127]),
    np.array([0.43789903, 0.48337683, 0.18110313]),
    np.array([0.4578588, 0.50623924, 0.25901491]),
    np.array([0.39985008, 0.43116457, 0.20152195]),
    np.array([0.37226961, 0.3973812, 0.23381716]),
    np.array([0.3589998, 0.38250951, 0.36788985]),
    np.array([0.46379107, 0.5360814, 0.45228296]),
    np.array([0.35180708, 0.37798088, 0.55904288]))

MUNSELL_GREYS_TO_XYY = (
    np.array([0.31006, 0.31616, 0.]),
    np.array([0.31006, 0.31616, 0.00473582]),
    np.array([0.31006, 0.31616, 0.00961944]),
    np.array([0.31006, 0.31616, 0.01545756]),
    np.array([0.31006, 0.31616, 0.02293343]),
    np.array([0.31006, 0.31616, 0.03261914]),
    np.array([0.31006, 0.31616, 0.044988]),
    np.array([0.31006, 0.31616, 0.0604269]),
    np.array([0.31006, 0.31616, 0.07924864]),
    np.array([0.31006, 0.31616, 0.10170428]),
    np.array([0.31006, 0.31616, 0.12799549]),
    np.array([0.31006, 0.31616, 0.15828689]),
    np.array([0.31006, 0.31616, 0.19271844]),
    np.array([0.31006, 0.31616, 0.23141772]),
    np.array([0.31006, 0.31616, 0.27451233]),
    np.array([0.31006, 0.31616, 0.32214224]),
    np.array([0.31006, 0.31616, 0.3744721]),
    np.array([0.31006, 0.31616, 0.43170362]),
    np.array([0.31006, 0.31616, 0.4940879]),
    np.array([0.31006, 0.31616, 0.56193781]),
    np.array([0.31006, 0.31616, 0.6356403]),
    np.array([0.31006, 0.31616, 0.71566876]),
    np.array([0.31006, 0.31616, 0.80259539]),
    np.array([0.31006, 0.31616, 0.89710353]),
    np.array([0.31006, 0.31616, 1.]))

XYY_TO_MUNSELL_SPECIFICATIONS = (
    (np.array([0.41515095, 0.51288165, 0.5702441]),
     (2.4974254984450397, 7.9653798278107182, 11.928549858473941, 4)),
    (np.array([0.38804358, 0.46299149, 0.31592072]),
     (2.5006439954892556, 6.1977947932238182, 6.9236106970679092, 4)),
    (np.array([0.33491518, 0.36277402, 0.22128409]),
     (2.4903091620270921, 5.3119569245226996, 1.9986380341015466, 4)),
    (np.array([0.39936353, 0.58547238, 0.64852094]),
     (5.611784843667591, 8.4027565252844596, 18.567138539062917, 4)),
    (np.array([0.34767896, 0.4152922, 0.58706989]),
     (5.8509402259358456, 8.0626386670645687, 5.7841425045791395, 4)),
    (np.array([0.33966055, 0.41527226, 0.07167165]),
     (5.7816682069236824, 3.174803944794176, 3.349385637554362, 4)),
    (np.array([0.36265912, 0.47966922, 0.11068168]),
     (5.4883266507471262, 3.899412030844517, 5.7627095549416296, 4)),
    (np.array([0.35748002, 0.45915987, 0.2727359]),
     (5.8091607027319458, 5.8169751521608619, 6.6660161915407334, 4)),
    (np.array([0.36348032, 0.48213512, 0.06293782]),
     (5.2106889846935189, 2.9770363655668297, 5.1418366196999559, 4)),
    (np.array([0.31838794, 0.40167814, 0.05382145]),
     (8.112670582764574, 2.7489429215052237, 3.1644657849632849, 4)),
    (np.array([0.31425413, 0.58372544, 0.04377268]),
     (8.0454960027236311, 2.4630649861246052, 7.5000612073563522, 4)),
    (np.array([0.31405111, 0.53120144, 0.02111891]),
     (7.5930163631603786, 1.5750743888437724, 4.6261748431094443, 4)),
    (np.array([0.32167873, 0.43862617, 0.3080991]),
     (9.0152103567385531, 6.1312711912840276, 6.8052936649435196, 4)),
    (np.array([0.31168045, 0.6270064, 0.34717087]),
     (9.0429810192754712, 6.4540531798090557, 17.010229922329884, 4)),
    (np.array([0.31496017, 0.47530248, 0.67920304]),
     (9.9147922785103972, 8.564387948380265, 11.130516599293573, 4)),
    (np.array([0.31107787, 0.51188895, 0.58306925]),
     (9.9600028417318498, 8.0396827436851854, 13.199185802941964, 4)),
    (np.array([0.31254722, 0.34686238, 0.6334334]),
     (9.8852069946025658, 8.3213426295629134, 2.0708474520194438, 4)),
    (np.array([0.30880402, 0.37157402, 0.08263161]),
     (9.9880916197181513, 3.4007871110002572, 2.5684536428873437, 4)),
    (np.array([0.23582365, 0.72197618, 0.06667783]),
     (9.9984874163608151, 3.063915653085683, 13.513388270694881, 4)),
    (np.array([0.29476305, 0.57521949, 0.23583791]),
     (9.9976375658465031, 5.461465412699126, 12.753028500364776, 4)),
    (np.array([0.28891056, 0.61005165, 0.28191444]),
     (9.9978537677920656, 5.9008140943398812, 15.244070244743728, 4)),
    (np.array([0.31292041, 0.3752074, 0.25538037]),
     (9.7548278524788188, 5.6536473977778643, 3.4107844965142462, 4)),
    (np.array([0.30648167, 0.48754769, 0.15098549]),
     (9.8616202679251828, 4.4878641370082102, 7.6757055354477961, 4)),
    (np.array([0.30382174, 0.34089453, 0.84210967]),
     (3.2303908627800304, 9.3451636022528515, 3.4403440367216769, 3)),
    (np.array([0.22538285, 0.5564611, 0.60532773]),
     (2.7893896011049968, 8.1660999142444801, 20.865361708231859, 3)),
    (np.array([0.28500017, 0.38833563, 0.24045742]),
     (3.2217895159709276, 5.5077419322120686, 5.4630728253870444, 3)),
    (np.array([0.19598037, 0.59002914, 0.29181101]),
     (2.6212805877659342, 5.9893806530207669, 19.362394859916822, 3)),
    (np.array([0.16437784, 0.59069112, 0.23370301]),
     (3.2866319103339858, 5.4398925329719443, 19.854410847201201, 3)),
    (np.array([0.17940333, 0.4663929, 0.06448045]),
     (5.7282498562431972, 3.0132952817775913, 10.742779986622125, 3)),
    (np.array([0.07553293, 0.55981543, 0.06406275]),
     (5.3447124424617432, 3.003537623832734, 18.896804785539025, 3)),
    (np.array([0.27330162, 0.37048932, 0.11621278]),
     (5.7450040846850658, 3.9875599357896836, 4.2238304339640553, 3)),
    (np.array([0.23251367, 0.40832841, 0.02585745]),
     (5.7167484363405752, 1.8040377069834592, 4.8775349629391567, 3)),
    (np.array([0.05704598, 0.55990299, 0.01221862]),
     (5.3134014009438513, 1.0305079836154281, 8.0423751566497952, 3)),
    (np.array([0.09405428, 0.51916421, 0.02268015]),
     (5.7582243025990465, 1.6541936507188186, 9.5055870640895321, 3)),
    (np.array([0.06306305, 0.54336526, 0.0361037]),
     (5.982440185520268, 2.2109766905487875, 14.80076750240695, 3)),
    (np.array([0.23250342, 0.41833342, 0.0559913]),
     (5.4561315302616542, 2.8055682870311611, 6.6458910422254247, 3)),
    (np.array([0.22630523, 0.39163204, 0.05597116]),
     (7.8355832971785766, 2.8050499614961404, 6.2381161041005031, 3)),
    (np.array([0.15858055, 0.42916814, 0.05259972]),
     (8.2773608986748037, 2.7163438063692538, 10.349070921928153, 3)),
    (np.array([0.14028772, 0.46282023, 0.41589047]),
     (8.319398373380178, 6.971801563527217, 23.511305166143138, 3)),
    (np.array([0.29271668, 0.33531051, 0.37326792]),
     (8.4446073967845603, 6.6574922739334328, 2.4100016954494836, 3)),
    (np.array([0.17253811, 0.43786778, 0.33686994]),
     (8.3062536616909277, 6.3711907135696917, 17.50573930025228, 3)),
    (np.array([0.09180367, 0.46823752, 0.05151176]),
     (8.1366760619167078, 2.6868572712647723, 14.647241441839769, 3)),
    (np.array([0.10903846, 0.44893518, 0.03595462]),
     (8.4804731414469181, 2.2057044721731964, 11.877741549341193, 3)),
    (np.array([0.2428693, 0.37094376, 0.04060119]),
     (8.4528136469348514, 2.3630505444499383, 4.6062410883921121, 3)),
    (np.array([0.27771166, 0.34994832, 0.23574564]),
     (8.3053924744903362, 5.4605354972582942, 3.9020883808495852, 3)),
    (np.array([0.05867972, 0.50502648, 0.19891229]),
     (8.1854332748360292, 5.0699333724644822, 28.120377018384666, 3)),
    (np.array([0.25930387, 0.37349411, 0.26874577]),
     (7.5395384410319366, 5.7799955833061949, 6.6412975765309845, 3)),
    (np.array([0.12284826, 0.47211684, 0.21388094]),
     (7.9597279338124061, 5.2335977016179758, 20.290421956393594, 3)),
    (np.array([0.0890682, 0.48703791, 0.27058998]),
     (8.429197168007434, 5.7971282967167381, 26.464916551909962, 3)),
    (np.array([0.27018357, 0.35138182, 0.76804186]),
     (0.0029116335013412709, 9.0051614807808651, 6.0521479959935158, 2)),
    (np.array([0.0431053, 0.45634623, 0.12074655]),
     (9.9090940657999482, 4.0578050043303753, 25.772345770419662, 3)),
    (np.array([0.16522669, 0.40881359, 0.18014875]),
     (9.9976284265517368, 4.8536959511881248, 13.711432917514353, 3)),
    (np.array([0.23349872, 0.37536989, 0.14476492]),
     (9.2805000536689342, 4.4042068716055978, 6.9966797139394705, 3)),
    (np.array([0.2315995, 0.37207726, 0.20351563]),
     (9.9968489688298856, 5.1210462040554292, 7.9466011879104901, 3)),
    (np.array([0.08301372, 0.45335265, 0.25304755]),
     (9.9995652694576354, 5.6311864544796251, 26.172410781131642, 3)),
    (np.array([0.20183026, 0.36561544, 0.39526058]),
     (2.4927553349648868, 6.8222787267061182, 12.637144808932389, 2)),
    (np.array([0.06340759, 0.37121187, 0.07975536]),
     (2.4983577898662457, 3.3435595770060069, 19.16688155401506, 2)),
    (np.array([0.16044634, 0.34707426, 0.10145605]),
     (3.2889502832179485, 3.7457477699374353, 10.316221955381685, 2)),
    (np.array([0.24416648, 0.33434737, 0.07774819]),
     (3.0834483697271065, 3.3027889692997099, 4.0314789302131331, 2)),
    (np.array([0.28105936, 0.3327088, 0.88937678]),
     (2.5569047038313286, 9.5505578383904854, 8.2592949619273135, 2)),
    (np.array([0.24328961, 0.31868567, 0.12931978]),
     (5.9613749867002497, 4.1861233500163211, 4.3110624347372388, 2)),
    (np.array([0.10471116, 0.30938022, 0.15549815]),
     (5.6283551180725855, 4.5472024424749877, 16.561283650587431, 2)),
    (np.array([0.0862452, 0.30268915, 0.15900713]),
     (5.8537976763848487, 4.5925997102794662, 18.829597868809778, 2)),
    (np.array([0.10497041, 0.32451898, 0.22191645]),
     (5.145089894686512, 5.3185755017128393, 18.972215318595008, 2)),
    (np.array([0.16894641, 0.33087742, 0.29312371]),
     (5.2908544585019168, 6.0, 13.596495394774379, 2)),
    (np.array([0.16144965, 0.33133829, 0.34018592]),
     (5.4160474347058862, 6.3980311319848546, 15.175835231610055, 2)),
    (np.array([0.25864013, 0.31415379, 0.28205753]),
     (8.2025559167724538, 5.9021079413020505, 4.0260977652628069, 2)),
    (np.array([0.15795868, 0.26417318, 0.11377678]),
     (0.012536235321443101, 3.9490817702416789, 9.1888977386924644, 1)),
    (np.array([0.12862477, 0.25616557, 0.08539517]),
     (9.2612367834642395, 3.4545177235566826, 10.589381636411931, 2)),
    (np.array([0.25058288, 0.29329096, 0.17796585]),
     (3.241079303134331, 4.827650836237364, 3.743365882969607, 1)),
    (np.array([0.18830894, 0.26192867, 0.13740285]),
     (2.5158920949660768, 4.3022043533741527, 7.3971111785249475, 1)),
    (np.array([0.1684076, 0.25029878, 0.13934697]),
     (2.5131427837361731, 4.329470861478244, 8.858704782948033, 1)),
    (np.array([0.1951648, 0.27716957, 0.51306785]),
     (2.5013789364411032, 7.6200943072565934, 10.879242856664604, 1)),
    (np.array([0.19935306, 0.27783329, 0.44060477]),
     (2.5013473966576427, 7.1449996623226202, 10.096343120964026, 1)),
    (np.array([0.26308512, 0.3046212, 0.52610451]),
     (2.6086704821311635, 7.7009394844971908, 4.2409228690438132, 1)),
    (np.array([0.2532416, 0.30291555, 0.67153139]),
     (2.4970799555123779, 8.524455640257985, 5.3618112705721508, 1)),
    (np.array([0.24841933, 0.2986962, 0.43833832]),
     (2.5045629814751091, 7.1293721373614156, 5.4196706970655724, 1)),
    (np.array([0.2082133, 0.28356991, 0.52733609]),
     (2.5131771577412465, 7.7085098474609053, 9.6149141336841648, 1)),
    (np.array([0.23939654, 0.2920611, 0.43144538]),
     (3.2651896190197771, 7.0815325065875827, 6.2204042252646667, 1)),
    (np.array([0.18279859, 0.27122662, 0.52199238]),
     (2.5035978550267846, 7.6755794342432786, 12.259534701163053, 1)),
    (np.array([0.16449512, 0.24371038, 0.08686299]),
     (2.5134247452410996, 3.4825807614595732, 7.7661876537009666, 1)),
    (np.array([0.16724393, 0.24366794, 0.06480227]),
     (2.5011108303709673, 3.0207831644481167, 6.9952041725206957, 1)),
    (np.array([0.19881487, 0.26071106, 0.06927689]),
     (3.0402893156428767, 3.1223174742573971, 5.2006395603475859, 1)),
    (np.array([0.18253778, 0.23018215, 0.03460635]),
     (5.597657625685617, 2.1572823771224261, 5.1627644612166348, 1)),
    (np.array([0.16926303, 0.22496873, 0.06237928]),
     (5.0963427942773887, 2.9637551635524759, 6.8587937787896989, 1)),
    (np.array([0.20398493, 0.2513471, 0.05473403]),
     (5.1969692716096816, 2.7729516729961743, 4.5594222945860832, 1)),
    (np.array([0.28140041, 0.30378091, 0.23081828]),
     (5.496561269157576, 5.4105513508464149, 2.0008136754401193, 1)),
    (np.array([0.15231331, 0.21384066, 0.25883348]),
     (5.8564961151865536, 5.6866675850563144, 13.28811457163558, 1)),
    (np.array([0.1593506, 0.22670722, 0.40114326]),
     (5.2738491300453916, 6.8653966593629132, 14.101257686332218, 1)),
    (np.array([0.22674934, 0.26033997, 0.17110185]),
     (7.8734843119339804, 4.7444381618873681, 5.5988944477630973, 1)),
    (np.array([0.20569472, 0.2404847, 0.15700695]),
     (8.3151048023527618, 4.5668004089848395, 7.0880114815101756, 1)),
    (np.array([0.11359218, 0.15851929, 0.15851498]),
     (7.8472605398133766, 4.5862706767212797, 16.22461017448477, 1)),
    (np.array([0.13446868, 0.17456223, 0.15665285]),
     (8.3830368679291833, 4.5622116144353626, 13.972942153711159, 1)),
    (np.array([0.20295637, 0.23758918, 0.07464645]),
     (7.856435533177164, 3.2383503193149088, 5.3584916284061572, 1)),
    (np.array([0.16020908, 0.20160833, 0.11096053]),
     (7.6347078681738623, 3.9039238069235207, 9.7687367916502765, 1)),
    (np.array([0.17946292, 0.22546056, 0.3340693]),
     (7.6477212875072365, 6.3483962769765867, 11.704570140652267, 1)),
    (np.array([0.19584886, 0.21874231, 0.05264774]),
     (9.9898352731700779, 2.7176352217817406, 5.4160154998040086, 1)),
    (np.array([0.25950493, 0.28494406, 0.60260113]),
     (9.1958881934475922, 8.1507829370037523, 5.0725149953944122, 1)),
    (np.array([0.22170777, 0.24928491, 0.29763974]),
     (9.9993390910882241, 6.0406947699165405, 7.7621615159154409, 1)),
    (np.array([0.23373145, 0.24171207, 0.08831548]),
     (2.8725008274335195, 3.5100389169220145, 4.4944374759441068, 10)),
    (np.array([0.29210338, 0.30192924, 0.75127547]),
     (2.5573566809324166, 8.9248766535028565, 2.4932404586928576, 10)),
    (np.array([0.1258535, 0.12764109, 0.16297312]),
     (2.5007360838933224, 4.6431603215843316, 18.831399277234546, 10)),
    (np.array([0.24227309, 0.25436998, 0.18624748]),
     (2.493358398116623, 4.9254430182491289, 5.4174887497658339, 10)),
    (np.array([0.25249867, 0.24628189, 0.1353695]),
     (5.9478312821937074, 4.2734224989493885, 4.8966308718341258, 10)),
    (np.array([0.22549284, 0.2180621, 0.16192792]),
     (5.5322841596577277, 4.6299109349866532, 7.7522529293268203, 10)),
    (np.array([0.1534495, 0.11674072, 0.15905692]),
     (5.9220754173118184, 4.5932393336579818, 19.565654953046174, 10)),
    (np.array([0.2235872, 0.20668864, 0.04253357]),
     (5.9517759003047477, 2.4246350396441594, 4.9539113586948025, 10)),
    (np.array([0.17573216, 0.16578146, 0.27364637]),
     (5.2008216960854803, 5.8253671682452843, 15.513073935981227, 10)),
    (np.array([0.27401103, 0.27401935, 0.23451177]),
     (5.8906638458196703, 5.4480790029813448, 3.7740354308969817, 10)),
    (np.array([0.2075913, 0.19464274, 0.21940166]),
     (5.8176239302630783, 5.2921858536043249, 11.318160969256796, 10)),
    (np.array([0.21031018, 0.15034168, 0.03888934]),
     (8.37336964793856, 2.3066883689858764, 8.6310377953949686, 10)),
    (np.array([0.23182863, 0.19825806, 0.05291206]),
     (7.7121590219520009, 2.7247287040518557, 5.7300160185960349, 10)),
    (np.array([0.24957235, 0.21006823, 0.31587613]),
     (9.9948213055847646, 6.1974183872008091, 10.969746297783992, 10)),
    (np.array([0.29306654, 0.28917618, 0.32466527]),
     (9.1126758970844079, 6.2709966101935857, 2.7508294514108931, 10)),
    (np.array([0.21441304, 0.13814375, 0.19716723]),
     (9.998713332858582, 5.050367418553515, 19.168834467026883, 10)),
    (np.array([0.20941829, 0.14321541, 0.24327119]),
     (9.3829287452992816, 5.535664936769356, 18.821027597932279, 10)),
    (np.array([0.28541299, 0.27913907, 0.54006024]),
     (9.002080407348938, 7.7860618086454938, 4.4538886914437397, 10)),
    (np.array([0.29230469, 0.28656219, 0.52465762]),
     (9.991235613354732, 7.6920309739694908, 3.652324068875179, 10)),
    (np.array([0.18804124, 0.08137467, 0.06580398]),
     (9.0425654223037455, 3.0439259368754286, 22.292781603620266, 10)),
    (np.array([0.22025958, 0.15180899, 0.06551257]),
     (9.4618219859335095, 3.0372188057325871, 10.554387963579108, 10)),
    (np.array([0.27887167, 0.24543217, 0.57450962]),
     (2.5015728390355108, 7.9902138330733292, 8.3830608857310125, 9)),
    (np.array([0.27487624, 0.23376357, 0.46322748]),
     (2.4996651744668252, 7.2983010268318242, 9.5028467299290504, 9)),
    (np.array([0.28356864, 0.2519005, 0.45980664]),
     (2.8618699795852098, 7.2754260032575946, 7.46611567875307, 9)),
    (np.array([0.30333596, 0.30005216, 0.66401066]),
     (3.0821290942108526, 8.4850005681408049, 2.4935002898711431, 9)),
    (np.array([0.23835467, 0.11558036, 0.09827669]),
     (2.5037499210886072, 3.6906677571198614, 19.779266412450269, 9)),
    (np.array([0.2536145, 0.16387485, 0.0990085]),
     (2.4992857563119486, 3.703448918491747, 12.087654980113633, 9)),
    (np.array([0.28535713, 0.25114971, 0.08429109]),
     (2.4890318095296315, 3.4331943344980584, 3.3814929201024877, 9)),
    (np.array([0.29701504, 0.28076672, 0.11652327]),
     (2.8365261300922739, 3.992426600758435, 2.0008049439042344, 9)),
    (np.array([0.24894294, 0.13513311, 0.0750785]),
     (2.8801166239961695, 3.2474344484707478, 14.610390101910371, 9)),
    (np.array([0.28976435, 0.23551078, 0.3203068]),
     (5.1644159196302653, 6.2346627202329179, 9.1133260523497661, 9)),
    (np.array([0.28699217, 0.2122739, 0.38376156]),
     (5.5001367381271749, 6.7368412342989172, 13.153893654633693, 9)),
    (np.array([0.2942318, 0.24483482, 0.41568603]),
     (5.5355820929681876, 6.97034254644461, 8.8925671354071021, 9)),
    (np.array([0.27112866, 0.10892559, 0.09137276]),
     (5.5896397325517455, 3.5668608875999919, 22.756075736806583, 9)),
    (np.array([0.26932562, 0.11871922, 0.07456975]),
     (5.2816804395644468, 3.2367339016487255, 18.731686204369552, 9)),
    (np.array([0.28774446, 0.21149857, 0.06409553]),
     (5.1732976532712982, 3.0043049868755096, 6.2294092968441577, 9)),
    (np.array([0.28438514, 0.18361032, 0.08751006]),
     (5.5527847263028818, 3.4948508627605213, 10.076128044498653, 9)),
    (np.array([0.27466364, 0.11623324, 0.04459164]),
     (5.7295438817735267, 2.4880342757163407, 15.986078477134843, 9)),
    (np.array([0.30526917, 0.29787617, 0.38438766]),
     (5.0697134888906703, 6.7415332853801191, 2.219542488732086, 9)),
    (np.array([0.27733084, 0.16764806, 0.24584118]),
     (5.3830683593123041, 5.5609978860251674, 18.302040299981691, 9)),
    (np.array([0.29841781, 0.26325636, 0.25902873]),
     (5.3573489710687028, 5.6885266519601112, 5.2553309577287299, 9)),
    (np.array([0.3042953, 0.11611832, 0.04387]),
     (8.2848827562071747, 2.4660497719128411, 17.239324057538756, 9)),
    (np.array([0.3167079, 0.22835511, 0.52829657]),
     (8.1437019218335216, 7.7144058793057653, 13.532589823660205, 9)),
    (np.array([0.31664956, 0.20454265, 0.45562827]),
     (8.0950673003541418, 7.2473398239288285, 17.337282390950396, 9)),
    (np.array([0.31300137, 0.23982828, 0.40210613]),
     (7.8286696950237911, 6.872416979415612, 10.705903656469591, 9)),
    (np.array([0.31187872, 0.26667157, 0.33190218]),
     (7.7997180017353251, 6.3306783023663034, 6.2110658750956116, 9)),
    (np.array([0.31537904, 0.21052765, 0.39335492]),
     (8.044182832558544, 6.8082262964881988, 15.55667343253385, 9)),
    (np.array([0.30594132, 0.18152717, 0.14244072]),
     (7.61253523767909, 4.3723674448096084, 14.16881179143731, 9)),
    (np.array([0.31195968, 0.12089229, 0.15102095]),
     (8.1692313405838135, 4.4883347140084675, 27.235344441294185, 9)),
    (np.array([0.33618672, 0.17589268, 0.24249386]),
     (9.6020077678142357, 5.527970670885737, 20.580420190036016, 9)),
    (np.array([0.34207627, 0.13875616, 0.24138597]),
     (9.6633397378583119, 5.51697844390592, 29.047422373609326, 9)),
    (np.array([0.31923003, 0.28291153, 0.25504488]),
     (9.9976373285878672, 5.6504248984821048, 4.2159521394352462, 9)),
    (np.array([0.33639641, 0.20777481, 0.23332748]),
     (9.9973290603118059, 5.4360854689605906, 14.778819186976174, 9)),
    (np.array([0.31464507, 0.3010788, 0.27040807]),
     (9.9981812895483202, 5.7954414845493361, 1.989415064944841, 9)),
    (np.array([0.32622786, 0.23679153, 0.28338647]),
     (9.4957863462098757, 5.9141054746428159, 10.808393081380505, 9)),
    (np.array([0.31964789, 0.19702337, 0.02988488]),
     (9.8255286116794061, 1.9759993085792666, 7.0669157653548789, 9)),
    (np.array([0.33202416, 0.16293316, 0.16828902]),
     (9.3833430362731995, 4.7097386903129328, 20.001567691188573, 9)),
    (np.array([0.3188341, 0.26119414, 0.19149517]),
     (9.1151365491745828, 4.9860254232389654, 5.8833516462242006, 9)),
    (np.array([0.34497302, 0.14740581, 0.17674791]),
     (9.9993661799172529, 4.8130329637008931, 24.745869339266417, 9)),
    (np.array([0.33396066, 0.13204228, 0.15759269]),
     (9.3788217006559904, 4.5743767927679624, 26.296103195549776, 9)),
    (np.array([0.34263192, 0.2492826, 0.04966462]),
     (2.5000378111202792, 2.6357603664122133, 4.7116011462627716, 8)),
    (np.array([0.37863885, 0.1480557, 0.03133476]),
     (2.8899890484960267, 2.0337680336115653, 13.996029826620704, 8)),
    (np.array([0.36067287, 0.22508694, 0.03664306]),
     (3.4356425026035353, 2.2299191499236484, 6.7219022055024089, 8)),
    (np.array([0.35583972, 0.20890369, 0.0287403]),
     (2.9923928706346814, 1.9289333829363795, 7.1957475402702213, 8)),
    (np.array([0.34728299, 0.11402692, 0.01746108]),
     (2.500584368220391, 1.3726888338846817, 14.170766991414974, 8)),
    (np.array([0.32940771, 0.22789278, 0.01489395]),
     (2.6107190169889272, 1.2137703916290237, 3.3488976554537588, 8)),
    (np.array([0.31972567, 0.31122932, 0.53600948]),
     (5.1656963312103521, 7.7615027982277569, 2.1410423395557343, 8)),
    (np.array([0.35012172, 0.29333067, 0.42147094]),
     (5.0476033618933336, 7.0114568961277142, 6.451158237816637, 8)),
    (np.array([0.37589661, 0.2850717, 0.66934047]),
     (5.8078232482982983, 8.512993428798092, 10.441995222524335, 8)),
    (np.array([0.42549932, 0.23904177, 0.33329037]),
     (5.0441061851453028, 6.3420359675892799, 18.425239270239359, 8)),
    (np.array([0.34641765, 0.2972505, 0.38411768]),
     (5.4855084167238033, 6.7395106003336087, 5.4746289633709742, 8)),
    (np.array([0.41521602, 0.25989123, 0.39086156]),
     (5.8766492669350967, 6.7897767720171638, 15.4400719961405, 8)),
    (np.array([0.34780042, 0.2928404, 0.0360562]),
     (8.1980634796784635, 2.2092985443242368, 1.9999688315566178, 8)),
    (np.array([0.4544551, 0.19822245, 0.03201793]),
     (7.9964595957086892, 2.0603132172126144, 11.656390753035632, 8)),
    (np.array([0.33858745, 0.3098545, 0.70004006]),
     (7.9849282958267054, 8.6712830581905536, 4.2705189024026593, 8)),
    (np.array([0.33239612, 0.31251827, 0.19604738]),
     (9.5354740556904183, 5.0377553290689132, 1.9992885406730441, 8)),
    (np.array([0.43846181, 0.29096381, 0.23141236]),
     (9.9958803631122173, 5.4166120275490517, 11.058404052106438, 8)),
    (np.array([0.40958022, 0.29719222, 0.48882871]),
     (9.3095533384450668, 7.4662030728919255, 11.186885768515552, 8)),
    (np.array([0.44399899, 0.29369509, 0.43379687]),
     (9.9949745495327491, 7.0979059067934074, 13.900038696652489, 8)),
    (np.array([0.40554919, 0.29723013, 0.16687769]),
     (9.9247647943179373, 4.6921921147267351, 7.1970576398528765, 8)),
    (np.array([0.42007003, 0.28930815, 0.1672933]),
     (9.41497757686939, 4.6973688173207524, 8.7211281744258624, 8)),
    (np.array([0.52108329, 0.25574146, 0.13999526]),
     (0.0019220181363621691, 4.3385094379603402, 16.467729969190529, 7)),
    (np.array([0.3763801, 0.30728007, 0.34070289]),
     (0.013622323425028782, 6.4022012583755865, 6.5944639522864303, 7)),
    (np.array([0.36495307, 0.30801481, 0.20910915]),
     (0.01085925621531203, 5.1822080765947414, 4.5480549505954455, 7)),
    (np.array([0.42566912, 0.29564012, 0.28217939]),
     (9.9664422617946791, 5.9032094955646706, 10.839641908347604, 8)),
    (np.array([0.55136347, 0.28138892, 0.19712193]),
     (2.497658061438166, 5.0498581073710413, 17.884020184861189, 7)),
    (np.array([0.53899416, 0.29048788, 0.25823634]),
     (2.7485546629604585, 5.6809766685439209, 17.995724599746378, 7)),
    (np.array([0.43854811, 0.3103317, 0.27612362]),
     (2.4943078709206112, 5.8481154233099364, 10.235754640176427, 7)),
    (np.array([0.35589069, 0.3165537, 0.24649473]),
     (2.8983979131373161, 5.5674143643020697, 3.5545236839116567, 7)),
    (np.array([0.49631592, 0.30111191, 0.04570504]),
     (5.4103686174523524, 2.5214548835517885, 7.258854844117816, 7)),
    (np.array([0.60338354, 0.2746834, 0.04600213]),
     (5.8342027901144569, 2.5302731702838765, 12.000615206905717, 7)),
    (np.array([0.57619776, 0.31554717, 0.14073356]),
     (5.9788280212108607, 4.3487706615362089, 14.392102528163205, 7)),
    (np.array([0.40414547, 0.32310724, 0.13347887]),
     (5.1783763049447264, 4.2464126669886673, 4.8428138961508438, 7)),
    (np.array([0.3514372, 0.32860694, 0.80679695]),
     (8.3238088961026691, 9.1859534871888915, 4.2364443377740981, 7)),
    (np.array([0.55200335, 0.34090583, 0.13240521]),
     (8.1957437659127912, 4.2309654368964633, 11.592666113548891, 7)),
    (np.array([0.43719237, 0.33673056, 0.18274766]),
     (7.6095498405871478, 4.8844510705007274, 6.7936327467178108, 7)),
    (np.array([0.38880059, 0.33783693, 0.7040874]),
     (8.254754870479271, 8.6917863577621315, 6.3425646390065618, 7)),
    (np.array([0.40006019, 0.33663147, 0.07790261]),
     (9.9982747461203623, 3.3059508982469912, 3.1154069559449877, 7)),
    (np.array([0.67248369, 0.32330365, 0.07220649]),
     (9.7627074021020093, 3.1863606829924529, 14.800047578330393, 7)),
    (np.array([0.64354918, 0.33973639, 0.08803122]),
     (9.9982818667900677, 3.5046891565568421, 13.900125103004392, 7)),
    (np.array([0.39181364, 0.3446948, 0.72252254]),
     (0.012146453319250128, 8.7841366100324851, 6.2155473412367117, 6)),
    (np.array([0.60231065, 0.36168845, 0.16068191]),
     (9.9987262089904458, 4.6140459204878024, 14.686472574413189, 7)),
    (np.array([0.5479115, 0.34892913, 0.02263783]),
     (2.7484465676051997, 1.6521000438004581, 5.5680030289264844, 6)),
    (np.array([0.48794187, 0.38293202, 0.2066566]),
     (3.1414206652130154, 5.15551583761708, 8.5966568435236024, 6)),
    (np.array([0.42836733, 0.35891726, 0.14365536]),
     (2.4978474308952627, 4.389047620589019, 4.9510475190736321, 6)),
    (np.array([0.4151889, 0.35495306, 0.14408043]),
     (2.4977653679963083, 4.3948638180430448, 4.383585180612334, 6)),
    (np.array([0.46542405, 0.34082576, 0.02079253]),
     (2.4915569441282575, 1.5580251224422779, 3.3089610305380077, 6)),
    (np.array([0.44637245, 0.38945422, 0.69298338]),
     (5.0392323757189894, 8.6353345266290926, 9.5932959267609448, 6)),
    (np.array([0.45404134, 0.39647154, 0.69233629]),
     (5.5898251806057857, 8.6320252640015713, 10.198248046129891, 6)),
    (np.array([0.48546253, 0.4130592, 0.59936297]),
     (5.9863485388822646, 8.1325317705551239, 12.306429102395787, 6)),
    (np.array([0.47978428, 0.39825981, 0.20940783]),
     (5.423024282295728, 5.1854456161550386, 8.0466534894368298, 6)),
    (np.array([0.46785973, 0.39297126, 0.20053901]),
     (5.3628014385081624, 5.0880776998143249, 7.3415299026055951, 6)),
    (np.array([0.51331905, 0.41265964, 0.21747039]),
     (5.6984660234683009, 5.2717939258175175, 10.326189160215149, 6)),
    (np.array([0.41770486, 0.37303747, 0.24430738]),
     (5.4077385973960945, 5.5458983510218864, 5.29247139332697, 6)),
    (np.array([0.37481816, 0.36536787, 0.78442772]),
     (8.2404764347274213, 9.0824007127074626, 4.9005464383176545, 6)),
    (np.array([0.39620152, 0.37951049, 0.68657974]),
     (8.2335502892855814, 8.602489889558802, 5.9421358761305845, 6)),
    (np.array([0.51299224, 0.43528921, 0.15241166]),
     (8.3941734779753023, 4.5067363905693325, 9.4616609022002027, 6)),
    (np.array([0.52529122, 0.46205863, 0.20512581]),
     (9.9976788857335226, 5.1387570613223055, 12.705101930365435, 6)),
    (np.array([0.38249101, 0.37271251, 0.24466996]),
     (9.9984253107088783, 5.5494729401082514, 3.5067321598335401, 6)),
    (np.array([0.52465528, 0.46381752, 0.27037022]),
     (9.9982936130158038, 5.7950904465362001, 14.064036258689498, 6)),
    (np.array([0.36332827, 0.36042625, 0.417479]),
     (9.998229887302589, 6.9831232007774773, 3.1286280407833535, 6)),
    (np.array([0.49103786, 0.44741782, 0.37625337]),
     (9.9996397801166204, 6.6802047042100119, 11.62960087994983, 6)),
    (np.array([0.53260672, 0.46085797, 0.38175879]),
     (9.0504958615431423, 6.7218006441915881, 17.087646825588862, 6)),
    (np.array([0.39947636, 0.38607101, 0.30024818]),
     (9.9937932662019193, 6.0634615534921332, 4.7377750435233841, 6)),
    (np.array([0.41566751, 0.39590836, 0.41203391]),
     (9.4034002686824678, 6.9442036094507777, 6.2887155865660711, 6)),
    (np.array([0.38379951, 0.37533829, 0.49499836]),
     (9.6285300256909068, 7.5058275345570138, 4.6237130906881818, 6)),
    (np.array([0.50184813, 0.44587964, 0.46556628]),
     (9.0181705389692919, 7.3138793763343148, 13.422606463167975, 6)),
    (np.array([0.51381974, 0.45512612, 0.47222976]),
     (9.2621126809280128, 7.35799496270426, 15.233390464742534, 6)),
    (np.array([0.49758967, 0.49049612, 0.45242214]),
     (3.3345190350913629, 7.225679043279789, 16.113161767033557, 5)),
    (np.array([0.43935016, 0.43103452, 0.23259639]),
     (2.4976979465295202, 5.428663062814679, 6.5439469780374466, 5)),
    (np.array([0.37285524, 0.37178029, 0.05690988]),
     (2.5014987253563152, 2.8290724121722768, 2.0022392543702865, 5)),
    (np.array([0.46625888, 0.45805354, 0.60604474]),
     (2.5003392833265892, 8.1701201056514439, 12.020108503258744, 5)),
    (np.array([0.47877996, 0.46893263, 0.71437706]),
     (2.4988626911278722, 8.7435404322303185, 14.428234223724067, 5)),
    (np.array([0.48009865, 0.45955664, 0.16263768]),
     (2.4965520790199704, 4.6389139549495999, 8.3806574469277688, 5)),
    (np.array([0.37334852, 0.37797455, 0.16515524]),
     (3.3620190337138762, 4.6706516252868591, 2.7755349063599879, 5)),
    (np.array([0.47856178, 0.5047532, 0.58732427]),
     (5.3384876152333405, 8.0640948096578331, 16.61169450964346, 5)),
    (np.array([0.47119689, 0.49876821, 0.65483021]),
     (5.3460599386324459, 8.4364175866715456, 15.412331133566335, 5)),
    (np.array([0.40983957, 0.43186708, 0.474305]),
     (5.366812938492127, 7.3716537562112343, 7.0382511111700721, 5)),
    (np.array([0.42018308, 0.44575131, 0.405341]),
     (5.9280307732079107, 6.8959269327086279, 7.5728873383657325, 5)),
    (np.array([0.45841191, 0.4830471, 0.3634053]),
     (5.7279389031851657, 6.5816608432402566, 10.668259627347156, 5)),
    (np.array([0.45040867, 0.47371048, 0.34766863]),
     (5.6432925986164273, 6.4580191382703749, 9.5488985904423309, 5)),
    (np.array([0.45990386, 0.46807933, 0.05857873]),
     (5.3436234094102453, 2.8710977062882246, 5.4305157214898205, 5)),
    (np.array([0.39793427, 0.428948, 0.13930127]),
     (7.7529577552993034, 4.3288327150737675, 4.2690288393490228, 5)),
    (np.array([0.43789903, 0.48337683, 0.18110313]),
     (8.1477743423848548, 4.8650216831032598, 7.5455998129262518, 5)),
    (np.array([0.4578588, 0.50623924, 0.25901491]),
     (7.9076312988106556, 5.6883950942156059, 10.778720186943522, 5)),
    (np.array([0.39985008, 0.43116457, 0.20152195]),
     (7.5927891641099432, 5.0989975862006771, 4.9370032698930322, 5)),
    (np.array([0.37226961, 0.3973812, 0.23381716]),
     (7.6750849187559878, 5.4410490370532214, 3.5531579567884455, 5)),
    (np.array([0.3589998, 0.38250951, 0.36788985]),
     (7.9921807359007735, 6.6162954611683462, 3.2858365447090425, 5)),
    (np.array([0.35180708, 0.37798088, 0.55904288]),
     (9.7114794283318631, 7.8995777724463387, 3.3577087155476923, 5)))

XYY_TO_MUNSELL_GREYS_SPECIFICATIONS = (
    (np.array([0.31006, 0.31616, 0.]), 0.0),
    (np.array([0.31006, 0.31616, 0.00473582]), 0.41666654855514407),
    (np.array([0.31006, 0.31616, 0.00961944]), 0.83333338141854485),
    (np.array([0.31006, 0.31616, 0.01545756]), 1.2499997206399589),
    (np.array([0.31006, 0.31616, 0.02293343]), 1.6666666076278116),
    (np.array([0.31006, 0.31616, 0.03261914]), 2.0833334496813687),
    (np.array([0.31006, 0.31616, 0.044988]), 2.5000000587596318),
    (np.array([0.31006, 0.31616, 0.0604269]), 2.9166665833170446),
    (np.array([0.31006, 0.31616, 0.07924864]), 3.3333332462968395),
    (np.array([0.31006, 0.31616, 0.10170428]), 3.7499999984535592),
    (np.array([0.31006, 0.31616, 0.12799549]), 4.1666666773491983),
    (np.array([0.31006, 0.31616, 0.15828689]), 4.5833332635744588),
    (np.array([0.31006, 0.31616, 0.19271844]), 5.0),
    (np.array([0.31006, 0.31616, 0.23141772]), 5.4166666679388724),
    (np.array([0.31006, 0.31616, 0.27451233]), 5.8333332789177978),
    (np.array([0.31006, 0.31616, 0.32214224]), 6.2499999924435414),
    (np.array([0.31006, 0.31616, 0.3744721]), 6.666666652206092),
    (np.array([0.31006, 0.31616, 0.43170362]), 7.0833333403693537),
    (np.array([0.31006, 0.31616, 0.4940879]), 7.4999999974980192),
    (np.array([0.31006, 0.31616, 0.56193781]), 7.9166666465175153),
    (np.array([0.31006, 0.31616, 0.6356403]), 8.3333333273024248),
    (np.array([0.31006, 0.31616, 0.71566876]), 8.7499999935449377),
    (np.array([0.31006, 0.31616, 0.80259539]), 9.1666666220972619),
    (np.array([0.31006, 0.31616, 0.89710353]), 9.5833332999710645),
    (np.array([0.31006, 0.31616, 1.]), 10.0))

NON_CONVERGING_XYY = (
    np.array([0.33159302, 0.43388935, 0.89380734]),
    np.array([0.258837, 0.71096717, 0.01266934]),
    np.array([0.28517207, 0.38369148, 0.89445395]),
    np.array([0.28155768, 0.33248001, 0.89992977]),
    np.array([0.29624596, 0.31950269, 0.96665647]),
    np.array([0.13833192, 0.07953813, 0.02236117]),
    np.array([0.22907253, 0.06719948, 0.06903321]))


class TestMunsellValuePriest1920(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Priest1920` definition
    unit tests methods.
    """

    def test_munsell_value_Priest1920(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Priest1920`
        definition.
        """

        self.assertAlmostEqual(munsell_value_Priest1920(10.08), 3.17490157328,
                               places=7)
        self.assertAlmostEqual(munsell_value_Priest1920(56.76), 7.53392328073,
                               places=7)
        self.assertAlmostEqual(munsell_value_Priest1920(98.32), 9.91564420499,
                               places=7)


class TestMunsellValueMunsell1933(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Munsell1933`
    definition unit tests methods.
    """

    def test_munsell_value_Munsell1933(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Munsell1933`
        definition.
        """

        self.assertAlmostEqual(munsell_value_Munsell1933(10.08), 3.79183555086,
                               places=7)
        self.assertAlmostEqual(munsell_value_Munsell1933(56.76), 8.27013181776,
                               places=7)
        self.assertAlmostEqual(munsell_value_Munsell1933(98.32), 9.95457710587,
                               places=7)


class TestMunsellValueMoon1943(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Moon1943` definition
    unit tests methods.
    """

    def test_munsell_value_Moon1943(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Moon1943`
        definition.
        """

        self.assertAlmostEqual(munsell_value_Moon1943(10.08), 3.74629715382,
                               places=7)
        self.assertAlmostEqual(munsell_value_Moon1943(56.76), 7.8225814259,
                               places=7)
        self.assertAlmostEqual(munsell_value_Moon1943(98.32), 9.88538236116,
                               places=7)


class TestMunsellValueSaunderson1944(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Saunderson1944`
    definition unit tests methods.
    """

    def test_munsell_value_Saunderson1944(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Saunderson1944`
        definition.
        """

        self.assertAlmostEqual(munsell_value_Saunderson1944(10.08),
                               3.68650805994, places=7)
        self.assertAlmostEqual(munsell_value_Saunderson1944(56.76),
                               7.89881184275, places=7)
        self.assertAlmostEqual(munsell_value_Saunderson1944(98.32),
                               9.85197100995, places=7)


class TestMunsellValueLadd1955(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_Ladd1955` definition
    unit tests methods.
    """

    def test_munsell_value_Ladd1955(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_Ladd1955`
        definition.
        """

        self.assertAlmostEqual(munsell_value_Ladd1955(10.08), 3.69528622419,
                               places=7)
        self.assertAlmostEqual(munsell_value_Ladd1955(56.76), 7.84875137062,
                               places=7)
        self.assertAlmostEqual(munsell_value_Ladd1955(98.32), 9.75492813681,
                               places=7)


class TestMunsellValueMcCamy1992(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_McCamy1987` definition
    unit tests methods.
    """

    def test_munsell_value_McCamy1987(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_McCamy1987`
        definition.
        """

        self.assertAlmostEqual(munsell_value_McCamy1987(10.08),
                               3.7347235258546316, places=7)
        self.assertAlmostEqual(munsell_value_McCamy1987(56.76),
                               7.951693123402992, places=7)
        self.assertAlmostEqual(munsell_value_McCamy1987(98.32),
                               9.935265378231314, places=7)


class TestMunsellValueASTMD153508(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_value_ASTMD153508`
    definition unit tests methods.
    """

    def test_munsell_value_ASTMD153508(self):
        """
        Tests :func:`colour.notation.munsell.munsell_value_ASTMD153508`
        definition.
        """

        self.assertAlmostEqual(munsell_value_ASTMD153508(10.08),
                               3.7344764769311354, places=7)
        self.assertAlmostEqual(munsell_value_ASTMD153508(56.76),
                               7.94992421224809, places=7)
        self.assertAlmostEqual(munsell_value_ASTMD153508(98.32),
                               9.934372853115544, places=7)


class TestMunsellSpecification_to_xyY(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_specification_to_xyY`
    definition unit tests methods.
    """

    def test_munsell_specification_to_xyY(self):
        """
        Tests :func:`colour.notation.munsell.munsell_specification_to_xyY`
        definition.
        """

        for i, specification in enumerate(MUNSELL_SPECIFICATIONS):
            np.testing.assert_almost_equal(
                munsell_specification_to_xyY(specification),
                np.array(MUNSELL_COLOURS_TO_XYY[i]),
                decimal=7)

        for i, specification in enumerate(MUNSELL_GREYS_SPECIFICATIONS):
            np.testing.assert_almost_equal(
                munsell_specification_to_xyY(specification),
                np.array(MUNSELL_GREYS_TO_XYY[i]),
                decimal=7)


class TestMunsellColour_to_xyY(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_colour_to_xyY` definition
    unit tests methods.
    """

    def test_munsell_colour_to_xyY(self):
        """
        Tests :func:`colour.notation.munsell.munsell_colour_to_xyY` definition.
        """

        # TODO: This test is covered by the previous class,
        # do we need a dedicated one?

        pass


class TestxyY_to_munsell_specification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_to_munsell_specification`
    definition unit tests methods.
    """

    def test_xyY_to_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.xyY_to_munsell_specification`
        definition.
        """

        for xyY, specification in XYY_TO_MUNSELL_SPECIFICATIONS:
            np.testing.assert_almost_equal(
                xyY_to_munsell_specification(xyY),
                specification,
                decimal=7)

        for xyY in NON_CONVERGING_XYY:
            self.assertRaises((RuntimeError, ValueError),
                              xyY_to_munsell_specification, xyY)

        for xyY, specification in XYY_TO_MUNSELL_GREYS_SPECIFICATIONS:
            np.testing.assert_almost_equal(
                xyY_to_munsell_specification(xyY),
                specification,
                decimal=7)


class TestxyY_to_munsell_colour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_to_munsell_colour` definition
    unit tests methods.
    """

    def test_xyY_to_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.xyY_to_munsell_colour` definition.
        """

        # TODO: This test is covered by the previous class,
        # do we need a dedicated one?

        pass


class TestParseMunsellColour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.parse_munsell_colour` definition
    unit tests methods.
    """

    def test_parse_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.is_grey_munsell_colour`
        definition.
        """

        self.assertEqual(parse_munsell_colour('N5.2'), 5.2)
        self.assertTupleEqual(parse_munsell_colour('0YR 2.0/4.0'),
                              (0.0, 2.0, 4.0, 6))
        self.assertTupleEqual(parse_munsell_colour('4.2YR 8.1/5.3'),
                              (4.2, 8.1, 5.3, 6))


class TestIsGreyMunsellColour(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.is_grey_munsell_colour` definition
    unit tests methods.
    """

    def test_is_grey_munsell_colour(self):
        """
        Tests :func:`colour.notation.munsell.is_grey_munsell_colour`
        definition.
        """

        self.assertTrue(is_grey_munsell_colour(5.2))
        self.assertFalse(is_grey_munsell_colour((0.0, 2.0, 4.0, 6)))
        self.assertFalse(is_grey_munsell_colour((4.2, 8.1, 5.3, 6)))


class TestNormalizeMunsellSpecification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.normalize_munsell_specification`
    definition unit tests methods.
    """

    def test_normalize_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.normalize_munsell_specification`
        definition.
        """

        self.assertTupleEqual(
            normalize_munsell_specification((0.0, 2.0, 4.0, 6)),
            (10.0, 2.0, 4.0, 7))
        self.assertTupleEqual(
            normalize_munsell_specification((0.0, 2.0, 4.0, 8)),
            (10.0, 2.0, 4.0, 9))
        self.assertTupleEqual(
            normalize_munsell_specification((0, 2.0, 4.0, 10)),
            (10.0, 2.0, 4.0, 1))
        self.assertEqual(
            normalize_munsell_specification((0, 2.0, 0, 10)),
            2)


class TestMunsellColourToMunsellSpecification(unittest.TestCase):
    """
    Defines
    :func:`colour.notation.munsell.munsell_colour_to_munsell_specification`
    definition unit tests methods.
    """

    def test_munsell_colour_to_munsell_specification(self):
        """
        Tests
        :func:`colour.notation.munsell.munsell_colour_to_munsell_specification`
        definition.
        """

        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('0.0YR 2.0/4.0'),
            (10., 2.0, 4.0, 7))
        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('0.0RP 2.0/4.0'),
            (10.0, 2.0, 4.0, 9))
        self.assertTupleEqual(
            munsell_colour_to_munsell_specification('10.0B 2.0/4.0'),
            (10.0, 2.0, 4.0, 1))
        self.assertEqual(munsell_colour_to_munsell_specification('N5.2'),
                         5.2)
        self.assertEqual(
            munsell_colour_to_munsell_specification('0.0YR 2.0/0.0'),
            2)


class TestMunsellSpecificationToMunsellColour(unittest.TestCase):
    """
    Defines
    :func:`colour.notation.munsell.munsell_specification_to_munsell_colour`
    definition unit tests methods.
    """

    def test_munsell_specification_to_munsell_colour(self):
        """
        Tests
        :func:`colour.notation.munsell.munsell_specification_to_munsell_colour`
        definition.
        """

        self.assertEqual(
            munsell_specification_to_munsell_colour((10, 2.0, 4.0, 7)),
            '10.0R 2.0/4.0')
        self.assertEqual(
            munsell_specification_to_munsell_colour((10.0, 2.0, 4.0, 9)),
            '10.0P 2.0/4.0')
        self.assertEqual(
            munsell_specification_to_munsell_colour((10.0, 2.0, 4.0, 1)),
            '10.0B 2.0/4.0')
        self.assertEqual(munsell_specification_to_munsell_colour(5.2),
                         'N5.2')


class Test_xyY_fromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xyY_from_renotation` definition
    unit tests methods.
    """

    def test_xyY_from_renotation(self):
        """
        Tests :func:`colour.notation.munsell.xyY_from_renotation`
        definition.
        """

        self.assertTupleEqual(xyY_from_renotation((2.5, 0.2, 2.0, 4)),
                              (0.713, 1.414, 0.237))
        self.assertTupleEqual(xyY_from_renotation((5.0, 0.2, 2.0, 4)),
                              (0.449, 1.145, 0.237))
        self.assertTupleEqual(xyY_from_renotation((7.5, 0.2, 2.0, 4)),
                              (0.262, 0.837, 0.237))


class TestIsSpecificationInRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.is_specification_in_renotation`
    definition unit tests methods.
    """

    def test_is_specification_in_renotation(self):
        """
        Tests :func:`colour.notation.munsell.is_specification_in_renotation`
        definition.
        """

        self.assertTrue(is_specification_in_renotation((2.5, 0.2, 2.0, 4)))
        self.assertTrue(is_specification_in_renotation((5.0, 0.2, 2.0, 4)))
        self.assertFalse(is_specification_in_renotation((25.0, 0.2, 2.0, 4)))


class TestBoundingHuesFromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.bounding_hues_from_renotation`
    definition unit tests methods.
    """

    def test_bounding_hues_from_renotation(self):
        """
        Tests :func:`colour.notation.munsell.bounding_hues_from_renotation`
        definition.
        """

        for i, specification in enumerate(MUNSELL_SPECIFICATIONS):
            hue, value, chroma, code = specification
            self.assertTupleEqual(bounding_hues_from_renotation(hue, code),
                                  MUNSELL_BOUNDING_HUES[i])


class TestHueToHueAngle(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_to_hue_angle` definition units
    tests methods.
    """

    def test_hue_to_hue_angle(self):
        """
        Tests :func:`colour.notation.munsell.hue_to_hue_angle` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ANGLE:
            self.assertEqual(hue_to_hue_angle(hue, code), angle)


class TestHueAngleToHue(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_angle_to_hue` definition units
    tests methods.
    """

    def test_hue_angle_to_hue(self):
        """
        Tests :func:`colour.notation.munsell.hue_angle_to_hue` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ANGLE:
            self.assertEqual(hue_angle_to_hue(angle), (hue, code))


class TestHueTo_ASTM_hue(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.hue_to_ASTM_hue` definition units
    tests methods.
    """

    def test_hue_to_ASTM_hue(self):
        """
        Tests :func:`colour.notation.munsell.hue_to_ASTM_hue` definition.
        """

        for hue, code, angle in MUNSELL_HUE_TO_ASTM_HUE:
            self.assertEqual(hue_to_ASTM_hue(hue, code), angle)


class TestInterpolationMethodFromRenotationOvoid(unittest.TestCase):
    """
    Defines
    :func:`colour.notation.munsell.interpolation_method_from_renotation_ovoid`
    definition unit tests methods.
    """

    def test_interpolation_method_from_renotation_ovoid(self):
        """
        Tests
        :func:`colour.notation.munsell.interpolation_method_from_renotation_ovoid`  # noqa
        definition.
        """

        for i, specification in enumerate(MUNSELL_EVEN_SPECIFICATIONS):
            self.assertEqual(
                interpolation_method_from_renotation_ovoid(specification),
                MUNSELL_INTERPOLATION_METHODS[i])


class Test_xy_fromRenotationOvoid(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.xy_from_renotation_ovoid`
    definition unit tests methods.
    """

    def test_xy_from_renotation_ovoid(self):
        """
        Tests :func:`colour.notation.munsell.xy_from_renotation_ovoid`
        definition.
        """

        for i, specification in enumerate(MUNSELL_EVEN_SPECIFICATIONS):
            if is_specification_in_renotation(specification):
                self.assertEqual(xy_from_renotation_ovoid(specification),
                                 MUNSELL_XY_FROM_RENOTATION_OVOID[i])


class TestLCHabToMunsellSpecification(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.LCHab_to_munsell_specification`
    definition unit tests methods.
    """

    def test_LCHab_to_munsell_specification(self):
        """
        Tests :func:`colour.notation.munsell.LCHab_to_munsell_specification`
        definition.
        """

        self.assertTupleEqual(
            LCHab_to_munsell_specification(
                np.array([100, 21.57210357, 272.2281935])),
            (5.6189426388888819, 10.0, 4.3144207139999997, 10))

        self.assertTupleEqual(
            LCHab_to_munsell_specification(
                np.array([100, 426.67945353, 72.39590835])),
            (0.10997454166666643, 10.0, 85.335890706000001, 5))

        self.assertTupleEqual(
            LCHab_to_munsell_specification(
                np.array([100, 74.05216981, 276.45318193])),
            (6.7925505361111194, 10.0, 14.810433961999999, 10))


class TestMaximumChromaFromRenotation(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.maximum_chroma_from_renotation`
    definition unit tests methods.
    """

    def test_maximum_chroma_from_renotation(self):
        """
        Tests
        :func:`colour.notation.munsell.maximum_chroma_from_renotation`
        definition.
        """

        self.assertEqual(maximum_chroma_from_renotation(2.5, 5, 5),
                         14.0)
        self.assertEqual(maximum_chroma_from_renotation(8.675, 1.225, 10),
                         48.0)
        self.assertEqual(maximum_chroma_from_renotation(6.875, 3.425, 1),
                         16.0)


class TestMunsellSpecification_to_xy(unittest.TestCase):
    """
    Defines :func:`colour.notation.munsell.munsell_specification_to_xy`
    definition unit tests methods.
    """

    def test_munsell_specification_to_xy(self):
        """
        Tests :func:`colour.notation.munsell.munsell_specification_to_xy`
        definition.
        """

        for specification, xy in MUNSELL_SPECIFICATIONS_TO_XY:
            np.testing.assert_almost_equal(
                munsell_specification_to_xy(specification),
                xy,
                decimal=7)

        for specification in MUNSELL_GREYS_SPECIFICATIONS:
            self.assertEqual(munsell_specification_to_xy(specification),
                             (0.31006, 0.31616))


if __name__ == '__main__':
    unittest.main()
