import json
import functools
from datetime import datetime, timedelta
from time import time

from django.contrib.contenttypes import generic
from django.contrib.contenttypes.models import ContentType
from django.db import models
from django.dispatch import Signal
from django.utils import timezone

from controller.utils.time import heartbeat_expires
from nodes.models import Node
from slices.models import Sliver

from . import settings


class State(models.Model):
    UNKNOWN = 'unknown'
    OFFLINE = 'offline'
    NODATA = 'nodata'
    FAILURE = 'failure'
    CRASHED = 'crashed'
    BASE_STATES = (
        (UNKNOWN, 'UNKNOWN'),
        (OFFLINE, 'OFFLINE'),
        (NODATA, 'NODATA'),
    )
    NODE_STATES = BASE_STATES + (
        ('production', 'PRODUCTION'),
        ('safe', 'SAFE'),
        ('debug', 'DEBUG'),
        (FAILURE, 'FAILURE'),
        (CRASHED, 'CRASHED'),
    )
    SLIVER_STATES = BASE_STATES + (
        ('started', 'STARTED'),
        ('deployed', 'DEPLOYED'),
        ('registered', 'REGISTERED'),
        ('fail_start', 'FAIL_START'),
        ('fail_deploy', 'FAIL_DEPLOY'),
        ('fail_alloc', 'FAIL_ALLOC'),
    )
    STATES = tuple(set(BASE_STATES+NODE_STATES+SLIVER_STATES))
    
    content_type = models.ForeignKey(ContentType)
    object_id = models.PositiveIntegerField()
    last_seen_on = models.DateTimeField(null=True,
            help_text='Last time the state retrieval was successfull')
    last_try_on = models.DateTimeField(null=True,
            help_text='Last time the state retrieval operation has been executed')
    last_contact_on = models.DateTimeField(null=True,
            help_text='Last API pull of this resource received from the node.')
    value = models.CharField(max_length=32, choices=STATES)
    metadata = models.TextField()
    data = models.TextField()
    add_date = models.DateTimeField(auto_now_add=True)
    
    content_object = generic.GenericForeignKey()
    
    class Meta:
        unique_together = ('content_type', 'object_id')
    
    def __unicode__(self):
        return self.value
    
    @property
    def last_change_on(self):
        last_state = self.last
        return last_state.start if last_state else None
    
    @property
    def heartbeat_expires(self):
        model = self.content_type.model_class()
        schedule = State.get_setting(model, 'SCHEDULE')
        window = State.get_setting(model, 'EXPIRE_WINDOW')
        return functools.partial(heartbeat_expires, freq=schedule, expire_window=window)
    
    @property
    def last(self):
        history = self.history.all().order_by('-start')
        return history[0] if history else None
    
    @property
    def current(self):
        if not self.last_try_on or time() > self.heartbeat_expires(self.last_try_on):
            return self.NODATA
        return self.value
    
    @property
    def next_retry_on(self):
        model = self.content_type.model_class()
        freq = State.get_setting(model, 'SCHEDULE')
        return self.last_try_on + timedelta(seconds=freq)
    
    @classmethod
    def store_glet(cls, obj, glet, get_data=lambda g: g.value):
        state = obj.state
        now = timezone.now()
        state.last_try_on = now
        metadata = {
            'exception': str(glet._exception) if glet._exception else None
        }
        response = get_data(glet)
        if response is not None:
            state.last_seen_on = now
            if response.status_code != 304:
                state.data = response.content
                if isinstance(obj, Node):
                    state._store_soft_version()
            metadata.update({
                'url': response.url,
                'headers': response.headers,
                'status_code': response.status_code
            })
            state._compute_current()
        else:
            # Exception, we assume OFFLINE state
            state.data = ''
            state.value = State.OFFLINE
        state.metadata = json.dumps(metadata, indent=4)
        state.history.store()
        state.save()
        return state
    
    @classmethod
    def register_heartbeat(cls, obj):
        state = obj.state
        state.last_seen_on = timezone.now()
        state.last_contact_on = state.last_seen_on
        if state.value in (cls.CRASHED, cls.OFFLINE):
            state._compute_current()
            state.history.store()
        state.save()
        node_heartbeat.send(sender=cls, node=obj.state.get_node())
    
    def _compute_current(self):
        if (not self.last_seen_on and time() > self.heartbeat_expires(self.add_date) or
                self.last_seen_on and time() > self.heartbeat_expires(self.last_seen_on)):
            self.value = State.OFFLINE
        else:
            try:
                self.value = json.loads(self.data).get('state', self.UNKNOWN)
            except ValueError:
                self.value = self.UNKNOWN
            if self.value != State.FAILURE:
                # check if CRASHED
                timeout_expire = timezone.now()-settings.STATE_NODE_PULL_TIMEOUT
                if self.add_date < timeout_expire:
                    if not self.last_contact_on or self.last_contact_on < timeout_expire:
                        self.value = self.CRASHED
    
    def _store_soft_version(self):
            try:
                version = json.loads(self.data).get('soft_version')
            except ValueError:
                pass
            else:
                NodeSoftwareVersion.objects.store(self.content_object, version)
    
    @classmethod
    def get_setting(cls, model, setting):
        name = model.__name__.upper()
        return getattr(settings, "STATE_%s_%s" % (name, setting))
    
    def get_url(self):
        model = self.content_type.model_class()
        URI = State.get_setting(model, 'URI')
        context = {
            'mgmt_addr': self.get_node().mgmt_net.addr,
            'object_id': self.object_id
        }
        return URI % context
    
    def get_current_display(self):
        current = self.current
        for name,verbose in State.STATES:
            if name == current:
                return verbose
        return current
    
    def get_node(self):
        return getattr(self.content_object, 'node', self.content_object)


class StateHistoryManager(models.Manager):
    def store(self, **kwargs):
        state = self.instance
        last_state = state.last
        now = timezone.now()
        if not last_state:
            last_state = state.history.create(value=state.value, start=now, end=now)
        else:
            expiration = state.heartbeat_expires(last_state.end)
            if time() > expiration:
                last_state.end = datetime.fromtimestamp(expiration)
                last_state.save()
                state.history.create(value=State.NODATA, start=last_state.end, end=now)
                last_state = state.history.create(value=state.value, start=now, end=now)
            else:
                last_state.end = now
                last_state.save()
                if last_state.value != state.value:
                    last_state = state.history.create(value=state.value, start=now, end=now)
        return last_state


class StateHistory(models.Model):
    state = models.ForeignKey(State, related_name='history')
    value = models.CharField(max_length=32, choices=State.STATES)
    start = models.DateTimeField()
    end = models.DateTimeField()
    
    objects = StateHistoryManager()
    
    class Meta:
        ordering = ['-start']
    
    def __unicode__(self):
        return self.value


class NodeSoftwareVersionManager(models.Manager):
    def store(self, node, version):
        if version:
            soft_version, __ = self.get_or_create(node=node)
            if soft_version.value != version:
                soft_version.value = version
                soft_version.save()


class NodeSoftwareVersion(models.Model):
    node = models.OneToOneField('nodes.Node', related_name='soft_version')
    value = models.CharField(max_length=256)
    
    objects = NodeSoftwareVersionManager()
    
    def __unicode__(self):
        return self.value


node_heartbeat = Signal(providing_args=["instance", "node"])


@property
def state(self):
    ct = ContentType.objects.get_for_model(type(self))
    return self.state_set.get_or_create(object_id=self.id, content_type=ct)[0]

for model in [Node, Sliver]:
    model.add_to_class('state_set', generic.GenericRelation('state.State'))
    model.state = state
